/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/**
 * ggv-control.c
 *
 * Authors:  Jaka Mocnik  <jaka@gnu.org>
 *
 * Copyright (c) 2001, 2002 Free Software Foundation
 */

#include <config.h>
#include <stdio.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkmarshal.h>
#include <gtk/gtktypeutils.h>

#include <gnome.h>

#include <ggv-control.h>
#include <gsdefaults.h>
#include <ggvutils.h>

struct _GgvControlPrivate {
	GgvPostscriptView *ps_view;

	BonoboZoomable *zoomable;
	float zoom_level;
	gboolean has_zoomable_frame;

	gboolean pane_auto_jump;

	GtkWidget *root;

	BonoboUIComponent *uic;
};

struct _GgvControlClassPrivate {
	BonoboUINode *zoom_level_menus;
};

static BonoboControlClass *ggv_control_parent_class;

static void
ggv_control_prefs_changed(GConfClient *client, guint cnxn_id,
						  GConfEntry *entry, gpointer user_data)
{
	GgvControl *control = GGV_CONTROL(user_data);

	if(!strcmp(entry->key, "/apps/ggv/control/autojump"))
		control->priv->pane_auto_jump = gconf_value_get_bool(entry->value);
}

static void
ggv_control_destroy (BonoboObject *object)
{
	GgvControl *control;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GGV_IS_CONTROL (object));

	control = GGV_CONTROL (object);

	if (control->priv->root)
		gtk_widget_unref (control->priv->root);
	control->priv->root = NULL;

	if(BONOBO_OBJECT_CLASS (ggv_control_parent_class)->destroy)
		BONOBO_OBJECT_CLASS (ggv_control_parent_class)->destroy (object);
}

static void
ggv_control_finalize (GObject *object)
{
	GgvControl *control;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GGV_IS_CONTROL (object));

	control = GGV_CONTROL (object);

	g_free (control->priv);

	G_OBJECT_CLASS (ggv_control_parent_class)->finalize (object);
}

static void
zoomable_set_frame_cb (BonoboZoomable *zoomable, GgvControl *control)
{
	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));

	control->priv->has_zoomable_frame = TRUE;
}

static void
zoomable_set_zoom_level_cb (BonoboZoomable *zoomable, float new_zoom_level,
			    GgvControl *control)
{
	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));

	ggv_postscript_view_set_zoom_factor
		(control->priv->ps_view, new_zoom_level);
	control->priv->zoom_level = ggv_postscript_view_get_zoom_factor
		(control->priv->ps_view);

	bonobo_zoomable_report_zoom_level_changed
		(zoomable, control->priv->zoom_level, NULL);
}

static void
zoomable_zoom_in_cb (BonoboZoomable *zoomable, GgvControl *control)
{
	float new_zoom_level;
	int index;

	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));

	index = ggv_zoom_index_from_float (control->priv->zoom_level);
	if (index == ggv_max_zoom_levels)
		return;

	index++;
	new_zoom_level = ggv_zoom_level_from_index (index);

	g_signal_emit_by_name (G_OBJECT (zoomable), "set_zoom_level",
						   new_zoom_level);
}

static void
zoomable_zoom_out_cb (BonoboZoomable *zoomable, GgvControl *control)
{
	float new_zoom_level;
	int index;

	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));

	index = ggv_zoom_index_from_float (control->priv->zoom_level);
	if (index == 0)
		return;

	index--;
	new_zoom_level = ggv_zoom_level_from_index (index);

	g_signal_emit_by_name (G_OBJECT (zoomable), "set_zoom_level",
						   new_zoom_level);
}

static void
zoomable_zoom_to_fit_cb (BonoboZoomable *zoomable, GgvControl *control)
{
	float new_zoom_level;

	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));

	new_zoom_level =
		ggv_postscript_view_zoom_to_fit (control->priv->ps_view, FALSE);

	g_signal_emit_by_name (G_OBJECT (zoomable), "set_zoom_level",
						   new_zoom_level);
}

static void
zoomable_zoom_to_default_cb (BonoboZoomable *zoomable, GgvControl *control)
{
	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));

	g_signal_emit_by_name (G_OBJECT (zoomable), "set_zoom_level", 1.0);
}

static gboolean
scrollbar_button_press_event(GtkWidget *widget, GdkEventButton *event,
							 gpointer data)
{
	GtkGS *gs = GTK_GS(data);

	if(event->button == 1)
		gtk_gs_start_scroll(gs);

	return FALSE;
}

static gboolean
scrollbar_button_release_event(GtkWidget *widget, GdkEventButton *event,
							   gpointer data)
{
	GtkGS *gs = GTK_GS(data);

	if(event->button == 1)
		gtk_gs_end_scroll(gs);

	return FALSE;
}

static gboolean
ggv_control_key_press_event(GtkWidget *widget, GdkEventKey *event,
							gpointer data)
{
	GtkGS *gs = GTK_GS(widget);
	gint key = event->keyval;
	GgvControl *control = GGV_CONTROL(data);
	GtkGSOrientation orientation = gtk_gs_get_orientation(gs);
	GgvPostscriptView *ps_view = control->priv->ps_view;

	/*ugh. the possibilities! */
	switch (key) {
	case GDK_space:
		switch (orientation) {
		case GTK_GS_ORIENTATION_PORTRAIT:
			if(!gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_DOWN, TRUE)) {
                                /* Since we are doing wrapping, go to top of the page */
				gtk_gs_scroll_to_edge(gs, GTK_POS_BOTTOM,  GTK_POS_BOTTOM);
				ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) + 1);
			}
			break;
		case GTK_GS_ORIENTATION_LANDSCAPE:
			if(!gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_LEFT, TRUE)) {
				ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) + 1);
			}
			break;
		case GTK_GS_ORIENTATION_SEASCAPE:
			if(!gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_RIGHT, TRUE)) {
				ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) + 1);
			}
			break;
		case GTK_GS_ORIENTATION_UPSIDEDOWN:
			if(!gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_UP, TRUE)) {
				ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) + 1);
			}
			break;
		}
		break;
	case GDK_BackSpace:
	case GDK_Delete:
		switch (orientation) {
		case GTK_GS_ORIENTATION_PORTRAIT:
			if(!gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_UP, TRUE)) {
				ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) - 1);
			}
			break;
		case GTK_GS_ORIENTATION_LANDSCAPE:
			if(!gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_RIGHT, TRUE)) {
				ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) - 1);
			}
			break;
		case GTK_GS_ORIENTATION_SEASCAPE:
			if(!gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_LEFT, TRUE)) {
				ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) - 1);
			}
			break;
		case GTK_GS_ORIENTATION_UPSIDEDOWN:
			if(!gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_DOWN, TRUE)) {
				ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) - 1);
			}
			break;
		}
		break;
	case GDK_Left:
		gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_LEFT, FALSE);
		break;
	case GDK_Right:
		gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_RIGHT, FALSE);
		break;
	case GDK_Up:
		gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_UP, FALSE);
		break;
	case GDK_Down:
		gtk_gs_scroll_step(gs, GTK_SCROLL_STEP_DOWN, FALSE);
		break;
	case GDK_Page_Up:
		ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) - 1);
		if(control->priv->pane_auto_jump)
			gtk_gs_scroll_to_edge(gs, GTK_POS_BOTTOM, GTK_POS_BOTTOM);
		break;
	case GDK_Page_Down:
		ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) + 1);
		if(control->priv->pane_auto_jump)
			gtk_gs_scroll_to_edge(gs, GTK_POS_TOP, GTK_POS_TOP);
		break;
	case GDK_plus:
		g_signal_emit_by_name(G_OBJECT (control->priv->zoomable),
							  "zoom_in");
		break;
	case GDK_minus:
		g_signal_emit_by_name(G_OBJECT(control->priv->zoomable),
							  "zoom_out");
		break;
	default:
                return FALSE;
	}

	return TRUE;
}

static void
verb_ZoomIn_cb (BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	GgvControl *control;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_CONTROL (user_data));

	control = GGV_CONTROL (user_data);

	g_signal_emit_by_name (G_OBJECT (control->priv->zoomable),
						   "zoom_in");
}

static void
verb_ZoomOut_cb (BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	GgvControl *control;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_CONTROL (user_data));

	control = GGV_CONTROL (user_data);

	g_signal_emit_by_name (G_OBJECT (control->priv->zoomable),
						   "zoom_out");
}

static void
verb_ZoomToDefault_cb (BonoboUIComponent *uic, gpointer user_data,
					   const char *cname)
{
	GgvControl *control;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_CONTROL (user_data));

	control = GGV_CONTROL (user_data);

	g_signal_emit_by_name (G_OBJECT (control->priv->zoomable),
						   "zoom_to_default");
}

static void
verb_ZoomToFit_cb (BonoboUIComponent *uic, gpointer user_data,
				   const char *cname)
{
	GgvControl *control;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_CONTROL (user_data));

	control = GGV_CONTROL (user_data);

	g_signal_emit_by_name (G_OBJECT (control->priv->zoomable),
						   "zoom_to_fit");
}

static BonoboUIVerb ggv_control_verbs[] = {
		BONOBO_UI_VERB ("ZoomIn",        verb_ZoomIn_cb),
		BONOBO_UI_VERB ("ZoomOut",       verb_ZoomOut_cb),
		BONOBO_UI_VERB ("ZoomToDefault", verb_ZoomToDefault_cb),
		BONOBO_UI_VERB ("ZoomToFit",     verb_ZoomToFit_cb),
		BONOBO_UI_VERB_END
};

static void
ggv_control_create_ui (GgvControl *control)
{
	BonoboUINode *zoom_level_menus;
	GgvControlClass *klass;

	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));

	klass = GGV_CONTROL_CLASS(G_OBJECT_GET_CLASS(control));


	bonobo_ui_util_set_ui(control->priv->uic, NULL,
						  "ggv-control-ui.xml", "GGV", NULL);

	zoom_level_menus = bonobo_ui_node_copy(klass->priv->zoom_level_menus,
										   TRUE);
	bonobo_ui_component_set_tree(control->priv->uic,
								 "/menu/ViewPlaceholder/View/ZoomMenu/Zoom/ZoomLevels",
								 zoom_level_menus, NULL);

	bonobo_ui_component_add_verb_list_with_data
		(control->priv->uic, ggv_control_verbs,
		 control);
}

static void
ggv_control_set_ui_container (GgvControl *control,
							  Bonobo_UIContainer ui_container)
{
	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));
	g_return_if_fail (ui_container != CORBA_OBJECT_NIL);

	ggv_postscript_view_set_ui_container (control->priv->ps_view,
										  ui_container);

	bonobo_ui_component_set_container (control->priv->uic, ui_container, NULL);

	if(!control->priv->has_zoomable_frame) {
		ggv_control_create_ui(control);
	}
}

static void
ggv_control_unset_ui_container (GgvControl *control)
{
	g_return_if_fail (control != NULL);
	g_return_if_fail (GGV_IS_CONTROL (control));

	ggv_postscript_view_unset_ui_container (control->priv->ps_view);

	bonobo_ui_component_unset_container (control->priv->uic, NULL);
}

static void
ggv_control_activate (BonoboControl *object, gboolean state)
{
	GgvControl *control;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GGV_IS_CONTROL (object));

	control = GGV_CONTROL (object);

	if (state) {
		Bonobo_UIContainer ui_container;
			
		ui_container = bonobo_control_get_remote_ui_container (BONOBO_CONTROL (control), NULL);
		if (ui_container != CORBA_OBJECT_NIL) {
			ggv_control_set_ui_container (control, ui_container);
			bonobo_object_release_unref (ui_container, NULL);
		}
	} else
		ggv_control_unset_ui_container (control);

	if (BONOBO_CONTROL_CLASS (ggv_control_parent_class)->activate)
		BONOBO_CONTROL_CLASS (ggv_control_parent_class)->activate (object, state);
}

static void
ggv_control_class_init (GgvControlClass *klass)
{
	BonoboUINode *placeholder, *item;
	GObjectClass *object_class = (GObjectClass *)klass;
	BonoboObjectClass *bonobo_object_class = (BonoboObjectClass *)klass;
	BonoboControlClass *control_class = (BonoboControlClass *)klass;
	int i;

	ggv_control_parent_class = gtk_type_class (bonobo_control_get_type ());

	bonobo_object_class->destroy = ggv_control_destroy;
	object_class->finalize = ggv_control_finalize;

	control_class->activate = ggv_control_activate;

	klass->priv = g_new0(GgvControlClassPrivate, 1);

	placeholder = bonobo_ui_node_new("placeholder");
	bonobo_ui_node_set_attr(placeholder, "name", "ZoomLevels");
	for(i = 0; i <= ggv_max_zoom_levels; i++) {
		item = bonobo_ui_node_new("menuitem");
		bonobo_ui_node_set_attr(item, "name", ggv_zoom_level_names[i]);
		bonobo_ui_node_set_attr(item, "label", ggv_zoom_level_names[i]);
		bonobo_ui_node_set_attr(item, "type", "radio");
		bonobo_ui_node_set_attr(item, "group", "ZoomLevel");
		bonobo_ui_node_set_attr(item, "verb", "");		
		bonobo_ui_node_add_child(placeholder, item);
	}
	klass->priv->zoom_level_menus = placeholder;
}

static void
ggv_control_init (GgvControl *control)
{
	control->priv = g_new0 (GgvControlPrivate, 1);
}

BONOBO_TYPE_FUNC (GgvControl, BONOBO_TYPE_CONTROL, ggv_control);

GgvControl *
ggv_control_construct (GgvControl *control, GgvPostscriptView *ps_view)
{
	BonoboControl         *retval;
	BonoboPropertyBag     *property_bag;
	BonoboPropertyControl *property_control;
	GtkWidget *view;

	g_return_val_if_fail (ps_view != NULL, NULL);
	g_return_val_if_fail (control != NULL, NULL);
	g_return_val_if_fail (GGV_IS_POSTSCRIPT_VIEW (ps_view), NULL);
	g_return_val_if_fail (GGV_IS_CONTROL (control), NULL);

	control->priv->ps_view = ps_view;
	bonobo_object_ref (BONOBO_OBJECT (ps_view));

	if (!ggv_postscript_view_add_interfaces (ps_view, BONOBO_OBJECT (control))) {
		g_message("control: can't add interfaces");
		return NULL;
	}

	view = ggv_postscript_view_get_widget (control->priv->ps_view);
	g_signal_connect(G_OBJECT(view), "key_press_event",
					 G_CALLBACK(ggv_control_key_press_event), control);

	control->priv->root = gtk_scrolled_window_new(ggv_postscript_view_get_hadj(ps_view),
												  ggv_postscript_view_get_vadj(ps_view));
	
	gtk_widget_show(control->priv->root);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(control->priv->root),
										GTK_SHADOW_IN);
	g_signal_connect(G_OBJECT(GTK_SCROLLED_WINDOW(control->priv->root)->hscrollbar),
					 "button_press_event",
					 G_CALLBACK(scrollbar_button_press_event),
					 view);
	g_signal_connect(G_OBJECT(GTK_SCROLLED_WINDOW(control->priv->root)->vscrollbar),
					 "button_press_event",
					 G_CALLBACK(scrollbar_button_press_event),
					 view);
	g_signal_connect(G_OBJECT(GTK_SCROLLED_WINDOW(control->priv->root)->hscrollbar),
					 "button_release_event",
					 G_CALLBACK(scrollbar_button_release_event),
					 view);
	g_signal_connect(G_OBJECT(GTK_SCROLLED_WINDOW(control->priv->root)->vscrollbar),
					 "button_release_event",
					 G_CALLBACK(scrollbar_button_release_event),
					 view);
	gtk_container_add(GTK_CONTAINER(control->priv->root), view);
	
	bonobo_control_construct (BONOBO_CONTROL (control), control->priv->root);

	bonobo_object_add_interface (BONOBO_OBJECT (control),
								 BONOBO_OBJECT (control->priv->ps_view));

	/* Interface Bonobo::Zoomable */
	control->priv->zoomable = bonobo_zoomable_new ();

	g_signal_connect (G_OBJECT (control->priv->zoomable),
					  "set_frame",
					  G_CALLBACK (zoomable_set_frame_cb),
					  control);
	g_signal_connect (G_OBJECT (control->priv->zoomable),
					  "set_zoom_level",
					  G_CALLBACK (zoomable_set_zoom_level_cb),
					  control);
	g_signal_connect (G_OBJECT (control->priv->zoomable),
					  "zoom_in",
					  G_CALLBACK (zoomable_zoom_in_cb),
					  control);
	g_signal_connect (G_OBJECT (control->priv->zoomable),
					  "zoom_out",
					  G_CALLBACK (zoomable_zoom_out_cb),
					  control);
	g_signal_connect (G_OBJECT (control->priv->zoomable),
					  "zoom_to_fit",
					  G_CALLBACK (zoomable_zoom_to_fit_cb),
					  control);
	g_signal_connect (G_OBJECT (control->priv->zoomable),
					  "zoom_to_default",
					  G_CALLBACK (zoomable_zoom_to_default_cb),
					  control);

	control->priv->zoom_level = 1.0;
	bonobo_zoomable_set_parameters_full (control->priv->zoomable,
										 control->priv->zoom_level,
										 ggv_zoom_levels [0],
										 ggv_zoom_levels [ggv_max_zoom_levels],
										 TRUE, TRUE, TRUE,
										 ggv_zoom_levels,
										 ggv_zoom_level_names,
										 ggv_max_zoom_levels + 1);

	bonobo_object_add_interface (BONOBO_OBJECT (control),
								 BONOBO_OBJECT (control->priv->zoomable));

	property_bag =
		ggv_postscript_view_get_property_bag (control->priv->ps_view);
	bonobo_control_set_properties (BONOBO_CONTROL (control),
								   BONOBO_OBJREF(property_bag), NULL);
	bonobo_object_unref (BONOBO_OBJECT (property_bag));

	property_control =
		ggv_postscript_view_get_property_control (control->priv->ps_view);

	bonobo_object_add_interface (BONOBO_OBJECT (control),
								 BONOBO_OBJECT (property_control));

	control->priv->uic =
		bonobo_control_get_ui_component (BONOBO_CONTROL (control));

	control->priv->pane_auto_jump = gconf_client_get_bool(gtk_gs_defaults_gconf_client(),
														  "/apps/ggv/control/autojump",
														  NULL);
	gconf_client_notify_add(gtk_gs_defaults_gconf_client(),
							"/apps/ggv/control",
							(GConfClientNotifyFunc)ggv_control_prefs_changed,
							control, NULL, NULL);

	return control;
}

GgvControl *
ggv_control_new (GgvPostscriptView *ps_view)
{
	GgvControl *control;
	
	g_return_val_if_fail (ps_view != NULL, NULL);
	g_return_val_if_fail (GGV_IS_POSTSCRIPT_VIEW (ps_view), NULL);

	control = g_object_new(GGV_CONTROL_TYPE, NULL);

	return ggv_control_construct (control, ps_view);
}
