/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * main.c: the ggv shell
 *
 * Copyright (C) 2002 the Free Software Foundation
 *
 * Author: Jaka Mocnik  <jaka@gnu.org>
 */

#include <config.h>

#include <gnome.h>
#include <bonobo-activation/bonobo-activation.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gconf/gconf-client.h>
#include <bonobo.h>
#include <bonobo/bonobo-ui-main.h>

#include <math.h>
#include <ctype.h>

#include "gtkgs.h"
#include "gtkchecklist.h"
#include "ggv-prefs.h"
#include "ggv-window.h"

#define GGV_POSTSCRIPT_VIEWER_CONTROL_IID "OAFIID:GNOME_GGV_Control"

static BonoboWindowClass *parent_class;

static GList *window_list;

/* what can be dragged in us... */
enum {
        TARGET_URI_LIST,
};

static void
control_property_changed_handler(BonoboListener    *listener,
                                 char              *event_name, 
                                 CORBA_any         *any,
                                 CORBA_Environment *ev,
                                 gpointer           data)
{
        GgvWindow *win = GGV_WINDOW(data);

        if(!g_strcasecmp(event_name, "Bonobo/Property:change:title")) {
                gchar *title = g_strconcat(_("GGV: "), BONOBO_ARG_GET_STRING(any), NULL);
                gtk_window_set_title(GTK_WINDOW(win), title);
                g_free(title);
        }
        else if(!g_strcasecmp(event_name, "Bonobo/Property:change:status")) {
                gnome_appbar_set_status(GNOME_APPBAR(win->statusbar),
                                        BONOBO_ARG_GET_STRING(any));
        }
        else if(!g_strcasecmp(event_name, "Bonobo/Property:change:page")) {
                GtkTreeSelection *sel;
                gchar path_str[16];
                GtkTreePath *path;

                g_message("Page change: %d.", BONOBO_ARG_GET_INT(any));

                sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(win->checklist));
                win->current_page = BONOBO_ARG_GET_INT(any);
                g_snprintf(path_str, 15, "%d", BONOBO_ARG_GET_INT(any));
                path = gtk_tree_path_new_from_string(path_str);
                if(!gtk_tree_selection_path_is_selected(sel, path))
                        gtk_tree_selection_select_path(sel, path);
                gtk_tree_path_free(path);
        }
}

static void
get_window_size(GtkWidget *widget, gint *width, gint *height)
{
        *width = widget->allocation.width;
        *height = widget->allocation.height;
}

static void
file_sel_unmap(GtkWidget *widget)
{
        get_window_size(widget, &file_sel_width, &file_sel_height);
}

static gboolean
file_sel_delete_event(GtkWidget *widget, GdkEventAny *e, gpointer data)
{
        gtk_widget_hide(widget);
        return FALSE;
}

static void
file_sel_ok_clicked(GtkWidget *widget, gpointer data)
{
        GgvWindow *win = GGV_WINDOW(data);
        const gchar *fname;

        gtk_widget_hide(win->file_sel);
        fname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(win->file_sel));
        if(!ggv_window_load(win, fname)) {
                /* TODO: report an error */
        }
}

static void
file_sel_cancel_clicked(GtkWidget *widget, gpointer data)
{
        GgvWindow *win = GGV_WINDOW(data);

        gtk_widget_hide(win->file_sel);
}

static void
verb_FileOpen(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GgvWindow *win = GGV_WINDOW(data);

        if(!win->file_sel) {
                win->file_sel = gtk_file_selection_new(_("Select a PostScript document"));
                gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(win->file_sel));
                gtk_window_set_resizable(GTK_WINDOW(win->file_sel), TRUE);
                g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(win->file_sel)->ok_button),
                                 "clicked", G_CALLBACK(file_sel_ok_clicked), win);
                g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(win->file_sel)->cancel_button),
                                 "clicked", G_CALLBACK(file_sel_cancel_clicked), win);
                g_signal_connect(G_OBJECT(win->file_sel), "delete_event",
                                 G_CALLBACK(file_sel_cancel_clicked), win);
                g_signal_connect(G_OBJECT(win->file_sel), "unmap",
                                 G_CALLBACK(file_sel_unmap), NULL);
        }
        if(!GTK_WIDGET_VISIBLE(win->file_sel)) {
                gtk_widget_set_usize(GTK_WIDGET(win->file_sel),
                                     file_sel_width, file_sel_height);
                gtk_widget_show(win->file_sel);
        }
}

static void
verb_FileClose(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GgvWindow *win = GGV_WINDOW(data);

        ggv_window_close(win);
}

static void
verb_FileNew(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GtkWidget *win = ggv_window_new();

        gtk_widget_show(win);
}

static void
verb_FileExit(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GList *l;

        for(l = window_list; l != NULL; l = l->next)
                gtk_widget_destroy(GTK_WIDGET(l->data));
        g_list_free(window_list);
        window_list = NULL;
        bonobo_main_quit();
}

static void
verb_SettingsPreferences(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        static GtkWidget *dlg = NULL;

        if(dlg == NULL) {
                dlg = ggv_prefs_dialog_new();
        }
        ggv_prefs_dialog_show(GGV_PREFS_DIALOG(dlg));
}

static void
listener_SettingsShow (BonoboUIComponent *uic, const char *path,
                       Bonobo_UIComponent_EventType type, const char *state,
                       gpointer user_data)
{
	GgvWindow *window;
        gboolean state_f;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_WINDOW (user_data));

	if(type != Bonobo_UIComponent_STATE_CHANGED)
		return;

	if(!state)
		return;

        window = GGV_WINDOW(user_data);
        state_f = atoi(state);

        if(!strcmp(path, "SettingsShowMenus")) {
                if(window->show_menus != state_f) {
                        window->show_menus = state_f;
                        bonobo_ui_component_set_prop(uic, "/menu", "hidden",
                                                     state_f?"0":"1", NULL);
                }
        }
	else if(!strcmp(path, "SettingsShowSidebar")) {
                if(window->show_sidebar != state_f) {
                        window->show_sidebar = state_f;
                        if(state_f)
                                gtk_widget_show(window->sidebar);
                        else
                                gtk_widget_hide(window->sidebar);
                }
        }
	else if(!strcmp(path, "SettingsShowToolbar")) {
                if(window->show_toolbar != state_f) {
                        window->show_toolbar = state_f;
                        bonobo_ui_component_set_prop(uic, "/Toolbar", "hidden",
                                                     state_f?"0":"1", NULL);
                }
        }
	else if(!strcmp(path, "SettingsShowStatusbar")) {
                if(window->show_statusbar != state_f) {
                        window->show_statusbar = state_f;
                        if(state_f)
                                gtk_widget_show(window->statusbar);
                        else
                                gtk_widget_hide(window->statusbar);
                }
        }
}

static void
verb_HelpAbout(BonoboUIComponent *uic, gpointer data, const char *cname)
{
	static GtkWidget *about = NULL;
	static const char *authors[] = {
                "Jaka Mocnik (current maintainer)",
                "Jonathan Blandford",
                "Daniel M. German",
                "Satyajit Kanungo",
                "Dan E. Kelley",
                "Werner Koerner",
                "Tuomas J. Lukka",
                "Johannes Plass",
		"Istvan Szekeres",
                "Tim Theisen",
                "And many more...",
 		NULL
	};
        static GdkPixbuf *logo = NULL;

        if(!logo)
                logo = gdk_pixbuf_new_from_file(GNOMEICONDIR
                                                "/ggv/ggv-splash.png",
                                                NULL);
	if (!about) {
		about = gnome_about_new (
			_("Gnome Ghostview"),
			VERSION,
			_("Copyright (C) 1998-2002 The Free Software Foundation"),
			_("The GNOME PostScript document previewer"),
			authors,
			NULL, /* char **documentors */
			NULL, /* char *translators */
			logo);
		g_signal_connect (G_OBJECT (about), "destroy",
                                  G_CALLBACK (gtk_widget_destroyed),
                                  &about);
	}

	gtk_widget_show_now (about);
	ggv_raise_and_focus_widget (about);
}

/* our verb list */
static BonoboUIVerb ggv_app_verbs[] = {
        BONOBO_UI_VERB("FileNew", verb_FileNew),
        BONOBO_UI_VERB("FileClose", verb_FileClose),
        BONOBO_UI_VERB("FileOpen", verb_FileOpen),
        BONOBO_UI_VERB("FileExit", verb_FileExit),
        BONOBO_UI_VERB("SettingsPreferences", verb_SettingsPreferences),
        BONOBO_UI_VERB("HelpAbout", verb_HelpAbout),
        BONOBO_UI_VERB_END
};

static void
control_frame_activate_uri(BonoboControlFrame *control_frame, const char *uri,
                           gboolean relative, gpointer data)
{
	GgvWindow *win;
	char *path;

	g_return_if_fail(uri != NULL);

	win = GGV_WINDOW(ggv_window_new());
	
	if(g_strncasecmp("file:", uri, 5) == 0) {
                ggv_window_load(win, uri + 5);
                gtk_widget_show(GTK_WIDGET(win));
        }
}

static gboolean
cl_toggle_all(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
        gboolean val;

        gtk_tree_model_get(model, iter, 0, &val, -1);
        gtk_list_store_set(GTK_LIST_STORE(model), iter, 0, !val, -1);

        return FALSE;
}

static gboolean
cl_toggle_even(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
        gint *n = (gint *)data;
        if(*n%2 == 0) {
                gboolean val;
                gtk_tree_model_get(model, iter, 0, &val, -1);
                gtk_list_store_set(GTK_LIST_STORE(model), iter, 0, !val, -1);
        }
        (*n)++;
        return FALSE;
}

static gboolean
cl_toggle_odd(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
        gint *n = (gint *)data;
        if(*n%2 != 0) {
                gboolean val;
                gtk_tree_model_get(model, iter, 0, &val, -1);
                gtk_list_store_set(GTK_LIST_STORE(model), iter, 0, !val, -1);
        }
        (*n)++;
        return FALSE;
}

static gboolean
cl_clear_all(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
        gtk_list_store_set(GTK_LIST_STORE(model), iter, 0, FALSE, -1);
        return FALSE;
}

static void
toggle_all_clicked(GtkWidget *widget, gpointer data)
{
        GgvWindow *win = GGV_WINDOW(data);
        gtk_tree_model_foreach(GTK_TREE_MODEL(GTK_CHECK_LIST(win->checklist)->list_model),
                               cl_toggle_all, NULL);
}

static void
toggle_odd_clicked(GtkWidget *widget, gpointer data)
{
        gint n = 1;
        GgvWindow *win = GGV_WINDOW(data);
        gtk_tree_model_foreach(GTK_TREE_MODEL(GTK_CHECK_LIST(win->checklist)->list_model),
                               cl_toggle_odd, &n);
}

static void
toggle_even_clicked(GtkWidget *widget, gpointer data)
{
        GgvWindow *win = GGV_WINDOW(data);
        gint n = 1;
        gtk_tree_model_foreach(GTK_TREE_MODEL(GTK_CHECK_LIST(win->checklist)->list_model),
                               cl_toggle_even, &n);
}

static void
clear_all_clicked(GtkWidget *widget, gpointer data)
{
        GgvWindow *win = GGV_WINDOW(data);
        gtk_tree_model_foreach(GTK_TREE_MODEL(GTK_CHECK_LIST(win->checklist)->list_model),
                               cl_clear_all, NULL);
}

static void
page_list_selection_changed(GtkTreeSelection *sel,
                            gpointer user_data)
{
        GgvWindow *win = GGV_WINDOW(user_data);
        GtkTreeIter iter;
        GtkTreeModel *model;

        if(gtk_tree_selection_get_selected(sel, &model, &iter)) {
                GtkTreePath *path;
                gint page;
                gchar *path_str;

                path = gtk_tree_model_get_path(model,
                                               &iter);
                path_str = gtk_tree_path_to_string(path);
                page = atoi(path_str);
                g_free(path_str);
                if(page >= 0)
                        bonobo_pbclient_set_long(win->pb, "page", page, NULL);
        }
}

static void
ggv_window_create_page_list(GgvWindow *win)
{
        CORBA_long page_count;
        GNOME_GGV_PageNameList *page_names;
        BonoboArg *arg;
        int i;

        page_count = bonobo_pbclient_get_long(win->pb, "page_count", NULL);
        arg = bonobo_pbclient_get_value(win->pb, "page_names",
                                        TC_GNOME_GGV_PageNameList,
                                        NULL);
        if(arg == NULL) {
                g_warning("Unable to fetch page names from control.");
                return;
        }
        page_names = (GNOME_GGV_PageNameList *)arg->_value;
        for(i = 0; i < page_names->_length; i++) {
                gtk_check_list_append_row(GTK_CHECK_LIST(win->checklist),
                                          page_names->_buffer[i]);                                                  
        }
        CORBA_sequence_set_release(page_names, CORBA_TRUE);
        bonobo_arg_release(arg);
}

static void
ggv_window_destroy(GtkObject *object)
{
        GgvWindow *win;

        g_return_if_fail(object != NULL);
        g_return_if_fail(GGV_IS_WINDOW(object));

        win = GGV_WINDOW(object);

        if(win->ctlframe != NULL) {
                bonobo_object_unref(BONOBO_OBJECT(win->ctlframe));
                win->ctlframe = NULL;
        }

        get_window_size(GTK_WIDGET(object),
                        &ggv_default_width, &ggv_default_height);

        if(GTK_OBJECT_CLASS(parent_class)->destroy)
                GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

static gboolean
ggv_window_delete_event(GtkWidget *widget, GdkEventAny *e)
{
        ggv_window_close(GGV_WINDOW(widget));
        return TRUE;
}

static void
ggv_window_class_init(GgvWindowClass *class)
{
	GObjectClass   *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	gobject_class = (GObjectClass *) class;
	object_class = (GtkObjectClass *) class;
	widget_class = (GtkWidgetClass *) class;

	parent_class = g_type_class_peek_parent (class);

	object_class->destroy = ggv_window_destroy;

	widget_class->delete_event = ggv_window_delete_event;
#if 0
	widget_class->key_press_event = ggv_window_key_press_event;
	widget_class->drag_data_received = ggv_window_drag_data_received;
#endif
}

static void
ggv_window_init (GgvWindow *window)
{
        window_list = g_list_prepend(window_list, window);
}

GType
ggv_window_get_type (void) 
{
	static GType ggv_window_type = 0;
	
	if(!ggv_window_type) {
		static const GTypeInfo ggv_window_info =
		{
			sizeof(GgvWindowClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) ggv_window_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof(GgvWindow),
			0,		/* n_preallocs */
			(GInstanceInitFunc) ggv_window_init,
		};
		
		ggv_window_type = g_type_register_static(BONOBO_TYPE_WINDOW, 
                                                         "GgvWindow", 
                                                         &ggv_window_info, 0);
	}

	return ggv_window_type;
}

GtkWidget *
ggv_window_new(void)
{
        GgvWindow *win;
	BonoboUIContainer *ui_container;
	BonoboUIComponent *uic;
	GdkGeometry geometry;
        CORBA_Environment ev;
        Bonobo_Control control;
        GtkWidget *widget, *sw, *hbox, *image;
        Bonobo_PropertyBag pb;
        gchar *mask;
        GtkTreeSelection *sel;

        /* get the control */
        CORBA_exception_init(&ev);
        control = bonobo_get_object(GGV_POSTSCRIPT_VIEWER_CONTROL_IID,
                                    "Bonobo/Control", &ev);
        if(BONOBO_EX(&ev) || control == CORBA_OBJECT_NIL) {
                CORBA_exception_free(&ev);
                return NULL;
        }
        CORBA_exception_free(&ev);

	win = GGV_WINDOW(g_object_new(GGV_TYPE_WINDOW, NULL));
        win->control = control;

        win->current_page = 0;
        win->show_toolbar = ggv_toolbar;
        win->show_menus = ggv_menubar;
        win->show_statusbar = ggv_statusbar;
        win->show_sidebar = ggv_panel;

        /* a vbox */
        win->vbox = gtk_vbox_new(FALSE, 0);
        gtk_container_set_border_width(GTK_CONTAINER(win->vbox), 2);
        gtk_widget_show(win->vbox);

        /* a hbox at its top*/
        win->hbox = gtk_hbox_new(FALSE, 2);
        gtk_widget_show(win->hbox);
        gtk_box_pack_start(GTK_BOX(win->vbox), win->hbox,
                           TRUE, TRUE, 2);


        /* with a sidebar on the left */
        win->sidebar = gtk_vbox_new(FALSE, 0);
        if(ggv_panel)
                gtk_widget_show(win->sidebar);
        gtk_box_pack_start(GTK_BOX(win->hbox), win->sidebar,
                           FALSE, TRUE, 0);

        hbox = gtk_hbox_new(TRUE, 2);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(win->sidebar), hbox, FALSE, TRUE, 0);

        image = gtk_image_new_from_file(GNOMEICONDIR "/ggv/toggleall.xpm");
        gtk_widget_show(image);
        win->toggle_all = gtk_button_new();
        gtk_button_set_relief(GTK_BUTTON(win->toggle_all), GTK_RELIEF_NONE);
        gtk_widget_show(win->toggle_all);
        gtk_box_pack_start(GTK_BOX(hbox), win->toggle_all, FALSE, TRUE, 0);
        gtk_container_add(GTK_CONTAINER(win->toggle_all), image);
        g_signal_connect(G_OBJECT(win->toggle_all), "clicked",
                         G_CALLBACK(toggle_all_clicked), win);
        image = gtk_image_new_from_file(GNOMEICONDIR "/ggv/toggleodd.xpm");
        gtk_widget_show(image);
        win->toggle_odd = gtk_button_new();
        gtk_button_set_relief(GTK_BUTTON(win->toggle_odd), GTK_RELIEF_NONE);
        gtk_widget_show(win->toggle_odd);
        gtk_box_pack_start(GTK_BOX(hbox), win->toggle_odd, FALSE, TRUE, 0);
        gtk_container_add(GTK_CONTAINER(win->toggle_odd), image);
        g_signal_connect(G_OBJECT(win->toggle_odd), "clicked",
                         G_CALLBACK(toggle_odd_clicked), win);
        image = gtk_image_new_from_file(GNOMEICONDIR "/ggv/toggleeven.xpm");
        gtk_widget_show(image);
        win->toggle_even = gtk_button_new();
        gtk_button_set_relief(GTK_BUTTON(win->toggle_even), GTK_RELIEF_NONE);
        gtk_widget_show(win->toggle_even);
        gtk_box_pack_start(GTK_BOX(hbox), win->toggle_even, FALSE, TRUE, 0);
        gtk_container_add(GTK_CONTAINER(win->toggle_even), image);
        g_signal_connect(G_OBJECT(win->toggle_even), "clicked",
                         G_CALLBACK(toggle_even_clicked), win);
        image = gtk_image_new_from_file(GNOMEICONDIR "/ggv/clearall.xpm");
        gtk_widget_show(image);
        win->clear_all = gtk_button_new();
        gtk_button_set_relief(GTK_BUTTON(win->clear_all), GTK_RELIEF_NONE);
        gtk_widget_show(win->clear_all);
        gtk_box_pack_start(GTK_BOX(hbox), win->clear_all, FALSE, TRUE, 0);
        gtk_container_add(GTK_CONTAINER(win->clear_all), image);
        g_signal_connect(G_OBJECT(win->clear_all), "clicked",
                         G_CALLBACK(clear_all_clicked), win);

        /* a checklist */
        win->checklist = gtk_check_list_new();
        sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(win->checklist));
        g_signal_connect(G_OBJECT(sel), "changed",
                         G_CALLBACK(page_list_selection_changed), win);
        gtk_widget_show(win->checklist);

        sw = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
                                            GTK_SHADOW_IN);
        gtk_widget_show(sw);
        gtk_container_add(GTK_CONTAINER(sw), win->checklist);
        gtk_box_pack_start(GTK_BOX(win->sidebar), sw, TRUE, TRUE, 0);
        
	/* add statusbar */
	win->statusbar = gnome_appbar_new(FALSE, TRUE,
                                          GNOME_PREFERENCES_NEVER);
        if(ggv_statusbar)
                gtk_widget_show(GTK_WIDGET(win->statusbar));
	gtk_box_pack_end(GTK_BOX(win->vbox), GTK_WIDGET(win->statusbar),
                         FALSE, FALSE, 0);

	ui_container = bonobo_ui_container_new();
	bonobo_window_construct(BONOBO_WINDOW(win), ui_container,
                                "ggv", _("Gnome Ghostview"));
        bonobo_window_set_contents(BONOBO_WINDOW(win), win->vbox);

	/* add menu and toolbar */
	uic = bonobo_ui_component_new("ggv");
	bonobo_ui_component_set_container(uic, BONOBO_OBJREF(ui_container),
                                          NULL);
	bonobo_ui_util_set_ui(uic, NULL, "ggv-ui.xml", "GGV", NULL);
	bonobo_ui_component_add_verb_list_with_data(uic, ggv_app_verbs, win);
	bonobo_ui_component_add_listener(uic, "SettingsShowMenus",
                                         listener_SettingsShow, win);
	bonobo_ui_component_add_listener(uic, "SettingsShowSidebar",
                                         listener_SettingsShow, win);
	bonobo_ui_component_add_listener(uic, "SettingsShowToolbar",
                                         listener_SettingsShow, win);
	bonobo_ui_component_add_listener(uic, "SettingsShowStatusbar",
                                         listener_SettingsShow, win);
        bonobo_ui_component_set_prop(uic, "/commands/SettingsShowMenus", "state",
                                     win->show_menus?"1":"0", NULL);
        bonobo_ui_component_set_prop(uic, "/commands/SettingsShowToolbar", "state",
                                     win->show_toolbar?"1":"0", NULL);
        bonobo_ui_component_set_prop(uic, "/commands/SettingsShowSidebar", "state",
                                     win->show_sidebar?"1":"0", NULL);
        bonobo_ui_component_set_prop(uic, "/commands/SettingsShowStatusbar", "state",
                                     win->show_statusbar?"1":"0", NULL);
        if(!win->show_toolbar)
                bonobo_ui_component_set_prop(uic, "/Toolbar", "hidden", "1", NULL);
        if(!win->show_menus)
                bonobo_ui_component_set_prop(uic, "/menu", "hidden", "1", NULL);

	/* add control frame interface and bind it to ggv control */
	win->ctlframe = bonobo_control_frame_new(BONOBO_OBJREF(ui_container));
	bonobo_control_frame_set_autoactivate(win->ctlframe, FALSE);
	g_signal_connect(G_OBJECT(win->ctlframe), "activate_uri",
                         (GtkSignalFunc)control_frame_activate_uri, NULL);
	bonobo_control_frame_bind_to_control(win->ctlframe, control, NULL);
	widget = bonobo_control_frame_get_widget(win->ctlframe);
	gtk_widget_show(widget);
        gtk_box_pack_start(GTK_BOX(win->hbox), widget,
                           TRUE, TRUE, 0);
	bonobo_control_frame_control_activate(win->ctlframe);

        /* now get the control's property bag */
        pb = bonobo_control_frame_get_control_property_bag(win->ctlframe, NULL);
        if(pb == CORBA_OBJECT_NIL) {
                g_warning("Control does not have any properties.");
        }
        else {
                /* TODO: set initial status & title */
                mask =  "Bonobo/Property:change:page,"
                        "Bonobo/Property:change:page_count,"
                        "Bonobo/Property:change:width,"
                        "Bonobo/Property:change:height,"
                        "Bonobo/Property:change:title,"
                        "Bonobo/Property:change:status";
                bonobo_event_source_client_add_listener(pb,
                                                        (BonoboListenerCallbackFn)control_property_changed_handler,
                                                        mask, NULL, win);
                win->pb = pb;
        }

	/* set default geometry */
        gtk_widget_set_usize(GTK_WIDGET(win),
                             ggv_default_width, ggv_default_height);
	gtk_window_set_resizable(GTK_WINDOW(win), TRUE);

	return GTK_WIDGET(win);
}

gboolean
ggv_window_load(GgvWindow *win, const gchar *filename)
{
        CORBA_Environment ev;
        Bonobo_PersistFile pf;

        g_return_val_if_fail(win != NULL, FALSE);
        g_return_val_if_fail(GGV_IS_WINDOW(win), FALSE);
        g_return_val_if_fail(filename != NULL, FALSE);

        /* TODO: add handling of stdin: should probably be solved by 
           copying stdin to a stream. */
        g_assert(strcmp(filename, "-"));

        CORBA_exception_init(&ev);
        pf = Bonobo_Unknown_queryInterface(win->control,
                                           "IDL:Bonobo/PersistFile:1.0", &ev);
        if(BONOBO_EX(&ev) || pf == CORBA_OBJECT_NIL) {
                CORBA_exception_free(&ev);
                return FALSE;
        }
        Bonobo_PersistFile_load(pf, filename, &ev);
        bonobo_object_release_unref(pf, NULL);
        if(BONOBO_EX(&ev)) {
                CORBA_exception_free(&ev);
                return FALSE;
        }
        CORBA_exception_free(&ev);

        gtk_check_list_clear(GTK_CHECK_LIST(win->checklist));

        ggv_window_create_page_list(win);

        return TRUE;
}

void
ggv_window_close(GgvWindow *win)
{
        g_return_if_fail(win != NULL);
        g_return_if_fail(GGV_IS_WINDOW(win));

        window_list = g_list_remove(window_list, win);

        gtk_widget_destroy(GTK_WIDGET(win));

        if(!window_list)
                bonobo_main_quit();
}

void
ggv_window_apply_prefs(GgvWindow *win)
{
        if(win) {
                /* TODO: set control properties */
        }
        else {
                GList *node = window_list;
                while(node) {
                        ggv_window_apply_prefs(GGV_WINDOW(node->data));
                        node = node->next;
                }
        }
}
