/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/**
 * ggv-postscript-view.c
 *
 * Author:  Jaka Mocnik  <jaka@gnu.org>
 *
 * Copyright (c) 2001, 2002 Free Software Foundation
 */

#include <config.h>
#include <stdio.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkmarshal.h>
#include <gtk/gtktypeutils.h>
#include <gconf/gconf-client.h>

#include <gnome.h>

#include <ggv-postscript-view.h>
#include <ggv-control.h>

#include <gtkgs.h>
#include <ps.h>
#include <gsdefaults.h>
#include <ggvutils.h>
#include <ggv-msg-window.h>
#include <ggv-prefs.h>
#include <ggv-prefs-ui.h>
#include <ggv-sidebar.h>

#include <bonobo.h>

struct _GgvPostScriptViewPrivate {
	GtkWidget *gs;
	GtkObject *hadj, *vadj;

	GNOME_GGV_Size def_size;
	GNOME_GGV_Orientation def_orientation;

	gint magstep;
	gboolean pan;
	gdouble prev_x, prev_y;
	gchar *tmp_name;

	GgvMsgWindow *msg_win;
	GgvSidebar *sidebar;

	GtkWidget *save_file_sel;

	BonoboPropertyBag     *property_bag;
	BonoboPropertyControl *property_control;

	BonoboUIComponent     *uic, *popup_uic;

	gboolean               zoom_fit;
};

struct _GgvPostScriptViewClassPrivate {
	int dummy;
};

enum {
	PROP_ANTIALIASING,
	PROP_DEFAULT_ORIENTATION,
	PROP_OVERRIDE_ORIENTATION,
	PROP_DEFAULT_SIZE,
	PROP_OVERRIDE_SIZE,
	PROP_RESPECT_EOF,
	PROP_WATCH_FILE,
	PROP_WIDTH,
	PROP_HEIGHT,
	PROP_ORIENTATION,
	PROP_PAGE,
	PROP_PAGE_COUNT,
	PROP_PAGE_NAMES,
	PROP_TITLE,
	PROP_STATUS
};

enum {
	PROP_CONTROL_TITLE,
};

static GdkCursor *pan_cursor = NULL;

static BonoboObjectClass *ggv_postscript_view_parent_class;

static const gchar *orientation_paths[] = {
	"/commands/OverrideOrientation",
	"/commands/OrientationPortrait",
	"/commands/OrientationLandscape",
	"/commands/OrientationUpsideDown",
	"/commands/OrientationSeascape",
	NULL
};

static const gchar *size_paths[] = {
	"/commands/OverrideSize",
	"/commands/PaperSizeBBox",
	"/commands/PaperSizeLetter",
	"/commands/PaperSizeTabloid",
	"/commands/PaperSizeLedger",
	"/commands/PaperSizeLegal",
	"/commands/PaperSizeStatement",
	"/commands/PaperSizeExecutive",
	"/commands/PaperSizeA3",
	"/commands/PaperSizeA4",
	"/commands/PaperSizeA5",
	"/commands/PaperSizeB4",
	"/commands/PaperSizeB5",
	"/commands/PaperSizeFolio",
	"/commands/PaperSizeQuarto",
	"/commands/PaperSize10x14",
	NULL
};

void
get_status(GgvPostScriptView *ps_view, BonoboArg *arg)
{
	GtkGS *gs = GTK_GS(ps_view->priv->gs);
	gchar *status;

	g_assert(arg->_type == TC_CORBA_string);
	
	status = "GGV control status...";

	*(CORBA_string *)arg->_value = CORBA_string_dup(status);
}

static void
prefs_changed_handler(gpointer data)
{
	GgvPostScriptView *ps_view = GGV_POSTSCRIPT_VIEW(data);

	/* TODO: set sidebar properties - coordinate units */
}

static void
get_title(GgvPostScriptView *ps_view, BonoboArg *arg)
{
	GtkGS *gs = GTK_GS(ps_view->priv->gs);
	gchar *title;

	g_assert(arg->_type == TC_CORBA_string);
	
	if(!gs->loaded)
		title = "No document loaded.";
	else if(!strcmp(gs->gs_filename, "-"))
		title = "(stdin)";
	else
		title = gs->gs_filename;

	*(CORBA_string *)arg->_value = CORBA_string_dup(title);
}

static void
_set_page_item_sensitivity(GgvPostScriptView *ps_view, BonoboUIComponent *uic)
{
	GtkGS *gs = GTK_GS(ps_view->priv->gs);
	gchar *prop_val;

	prop_val = (gtk_gs_get_current_page(gs) >= gtk_gs_get_page_count(gs) - 1 ||
				gtk_gs_get_current_page(gs) < 0)?
		"0":"1";
	bonobo_ui_component_set_prop(uic, "/commands/NextPage",
								 "sensitive", prop_val, NULL);
	prop_val = (gtk_gs_get_current_page(gs) <= 0 || !gs->structured_doc)?
		"0":"1";
	bonobo_ui_component_set_prop(uic, "/commands/PrevPage",
								 "sensitive", prop_val, NULL);
	bonobo_ui_component_set_prop(uic, "/commands/FirstPage",
								 "sensitive", prop_val, NULL);
	prop_val = (gtk_gs_get_current_page(gs) >= gtk_gs_get_page_count(gs) - 1 ||
				!gs->structured_doc || gtk_gs_get_current_page(gs) < 0)?
		"0":"1";
	bonobo_ui_component_set_prop(uic, "/commands/LastPage",
								 "sensitive", prop_val, NULL);
}

static void
set_page_item_sensitivity(GgvPostScriptView *ps_view)
{
	_set_page_item_sensitivity(ps_view, ps_view->priv->uic);
	if(ps_view->priv->popup_uic != NULL)
		_set_page_item_sensitivity(ps_view, ps_view->priv->popup_uic);
}

static void
notify_page_count_change(GgvPostScriptView *ps_view)
{
	BonoboArg *arg;

	set_page_item_sensitivity(ps_view);

	arg = bonobo_arg_new(TC_CORBA_long);
	BONOBO_ARG_SET_LONG(arg, gtk_gs_get_page_count(GTK_GS(ps_view->priv->gs)));
	bonobo_event_source_notify_listeners(ps_view->priv->property_bag->es,
										 "Bonobo/Property:change:page_count",
										 arg, NULL);
	bonobo_arg_release(arg);
}

static void
notify_orientation_change(GgvPostScriptView *ps_view)
{
	BonoboArg *arg;

	arg = bonobo_arg_new(TC_GNOME_GGV_Orientation);
	*(GNOME_GGV_Orientation *)arg->_value = gtk_gs_get_orientation(GTK_GS(ps_view->priv->gs));
	bonobo_event_source_notify_listeners(ps_view->priv->property_bag->es,
										 "Bonobo/Property:change:orientation",
										 arg, NULL);
	bonobo_arg_release(arg);
}

static void
notify_page_change(GgvPostScriptView *ps_view)
{
	BonoboArg *arg;

	set_page_item_sensitivity(ps_view);

	arg = bonobo_arg_new(TC_CORBA_long);
	*(CORBA_long *)arg->_value = gtk_gs_get_current_page(GTK_GS(ps_view->priv->gs));
	bonobo_event_source_notify_listeners(ps_view->priv->property_bag->es,
										 "Bonobo/Property:change:page",
										 arg, NULL);
	bonobo_arg_release(arg);
}

static void
notify_title_change(GgvPostScriptView *ps_view)
{
	BonoboArg *arg;
	arg = bonobo_arg_new(TC_CORBA_string);
	get_title(ps_view, arg);
	bonobo_event_source_notify_listeners(ps_view->priv->property_bag->es,
										 "Bonobo/Property:change:title",
										 arg, NULL);
	bonobo_arg_release(arg);
}

void
ggv_postscript_view_goto_page(GgvPostScriptView *ps_view, gint page)
{
	gint old_page = gtk_gs_get_current_page(GTK_GS(ps_view->priv->gs));

	gtk_gs_goto_page(GTK_GS(ps_view->priv->gs), page);
	if(gtk_gs_get_current_page(GTK_GS(ps_view->priv->gs)) != old_page) {
		notify_page_change(ps_view);
	}
}

gint
ggv_postscript_view_get_current_page(GgvPostScriptView *ps_view)
{
	return gtk_gs_get_current_page(GTK_GS(ps_view->priv->gs));
}

gint
ggv_postscript_view_get_page_count(GgvPostScriptView *ps_view)
{
	return gtk_gs_get_page_count(GTK_GS(ps_view->priv->gs));
}

static void
load_ps(GgvPostScriptView *ps_view, const gchar *fname)
{
	gtk_gs_load(GTK_GS(ps_view->priv->gs), fname);
	ggv_sidebar_create_page_list(ps_view->priv->sidebar);
	notify_title_change(ps_view);
	notify_page_count_change(ps_view);
	notify_orientation_change(ps_view);
	ggv_postscript_view_goto_page(ps_view, 0);
	ggv_sidebar_update_coordinates(ps_view->priv->sidebar, 0.0, 0.0);
}

/*
 * Loads an postscript document from a Bonobo_Stream
 */
static void
load_ps_from_stream (BonoboPersistStream *ps,
					 Bonobo_Stream stream,
					 Bonobo_Persist_ContentType type,
					 void *data,
					 CORBA_Environment *ev)
{
	GgvPostScriptView *ps_view;
	Bonobo_Stream_iobuf *buffer;
	CORBA_long len_read;
	FILE *tmpfile;
	int fd;

	g_return_if_fail (data != NULL);
	g_return_if_fail (GGV_IS_POSTSCRIPT_VIEW (data));

	ps_view = GGV_POSTSCRIPT_VIEW(data);

	/* copy stream to a tmp file */
	if(ps_view->priv->tmp_name != NULL) {
		unlink(ps_view->priv->tmp_name);
		g_free(ps_view->priv->tmp_name);
		ps_view->priv->tmp_name = NULL;
	}
	ps_view->priv->tmp_name = g_strconcat(g_get_tmp_dir(), "/ggvXXXXXX", NULL);
	if((fd = mkstemp(ps_view->priv->tmp_name)) < 0) {
		g_free(ps_view->priv->tmp_name),
		ps_view->priv->tmp_name = NULL;
		return;
	}
	tmpfile = fdopen(fd, "w");
	if(!tmpfile) {
		close(fd);
		return;
	}

	do {
		Bonobo_Stream_read (stream, 32768, &buffer, ev);
		if (ev->_major != CORBA_NO_EXCEPTION)
			goto exit_clean;

		len_read = buffer->_length;

		if (buffer->_buffer && len_read)
			if(fwrite(buffer->_buffer, 1, len_read, tmpfile) != len_read) {
				CORBA_free (buffer);
				goto exit_clean;
			}

		CORBA_free (buffer);
	} while (len_read > 0);

	fclose(tmpfile);
	load_ps(ps_view, ps_view->priv->tmp_name);
	return;

 exit_clean:
	fclose (tmpfile);
	return;
}

/* 
 * handlers for mouse actions
 */
static gboolean
view_button_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GgvPostScriptView *ps_view;

	g_return_if_fail(data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(data));

	ps_view = GGV_POSTSCRIPT_VIEW(data);

	if(event->button == 1 && !ps_view->priv->pan) {
		gint wx = 0, wy = 0;
			
		gtk_widget_grab_focus(ps_view->priv->gs);

		gdk_window_get_pointer(widget->window, &wx, &wy, NULL);
			
		gtk_gs_start_scroll(GTK_GS(ps_view->priv->gs));

		ps_view->priv->pan = TRUE;
		if(pan_cursor == NULL)
			pan_cursor = gdk_cursor_new(GDK_FLEUR);
			
		gtk_grab_add(widget);
		gdk_pointer_grab(widget->window, FALSE,
						 GDK_POINTER_MOTION_MASK |
						 GDK_BUTTON_RELEASE_MASK, NULL,
						 pan_cursor, GDK_CURRENT_TIME);
		ps_view->priv->prev_x = wx;
		ps_view->priv->prev_y = wy;

		return TRUE;
	}

	return FALSE;
}

static gboolean
view_button_release_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GgvPostScriptView *ps_view;

	g_return_if_fail(data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(data));

	ps_view = GGV_POSTSCRIPT_VIEW(data);

	if(event->button == 1 && ps_view->priv->pan) {
		ps_view->priv->pan = FALSE;
		gdk_pointer_ungrab(GDK_CURRENT_TIME);
		gtk_grab_remove(widget);
		gtk_gs_end_scroll(GTK_GS(ps_view->priv->gs));

		return TRUE;
	}

	return FALSE;
}

static gboolean
view_motion_cb(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
	GgvPostScriptView *ps_view;

	g_return_if_fail(data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(data));

	ps_view = GGV_POSTSCRIPT_VIEW(data);
	if(ps_view->priv->pan) {
		gtk_gs_scroll(GTK_GS(ps_view->priv->gs),
					  -(gint)event->x + ps_view->priv->prev_x,
					  -(gint)event->y + ps_view->priv->prev_y);
		ps_view->priv->prev_x = (gint)event->x;
		ps_view->priv->prev_y = (gint)event->y;

		return TRUE;
	}
	else {
		GtkGS *gs;

		gs = GTK_GS(ps_view->priv->gs);
		if(event->window == gs->pstarget && gs->doc != NULL) {
			gfloat xcoord, ycoord;
		
			xcoord = event->x/gs->xdpi/gs->zoom_factor;
			ycoord = event->y/gs->ydpi/gs->zoom_factor;
			
			ggv_sidebar_update_coordinates(ps_view->priv->sidebar, xcoord, ycoord);
		}
	}

	return FALSE;
}


/*
 * Loads an postscript document from a Bonobo_File
 */
static gint
load_ps_from_file (BonoboPersistFile *pf, const CORBA_char *filename,
				   CORBA_Environment *ev, void *data)
{
	GgvPostScriptView *ps_view;
	BonoboArg *arg;

	g_return_val_if_fail (data != NULL, -1);
	g_return_val_if_fail (GGV_IS_POSTSCRIPT_VIEW (data), -1);

	ps_view = GGV_POSTSCRIPT_VIEW (data);

	load_ps(ps_view, filename);

	return 0;
}

static void
listener_Orientation_cb (BonoboUIComponent *uic, const char *path,
						 Bonobo_UIComponent_EventType type, const char *state,
						 gpointer user_data)
{
	BonoboArg *arg;
	GgvPostScriptView *ps_view;
	GNOME_GGV_Orientation orientation;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW (user_data));

	if(type != Bonobo_UIComponent_STATE_CHANGED)
		return;

	if(!state || !atoi(state))
		return;

	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	if(!strcmp(path, "OrientationPortrait"))
		orientation = GNOME_GGV_ORIENTATION_PORTRAIT;
	else if(!strcmp(path, "OrientationLandscape"))
		orientation = GNOME_GGV_ORIENTATION_LANDSCAPE;
	else if(!strcmp(path, "OrientationUpsideDown"))
		orientation = GNOME_GGV_ORIENTATION_UPSIDEDOWN;
	else if(!strcmp(path, "OrientationSeascape"))
		orientation = GNOME_GGV_ORIENTATION_SEASCAPE;
	else {
		g_warning("Unknown orientation `%s'", path);
		return;
	}

	arg = bonobo_arg_new(TC_GNOME_GGV_Orientation);
	BONOBO_ARG_SET_GENERAL(arg, orientation, TC_GNOME_GGV_Orientation,
						   GNOME_GGV_Orientation, NULL);

	bonobo_pbclient_set_value(BONOBO_OBJREF(ps_view->priv->property_bag),
							  "default_orientation", arg, NULL);

	bonobo_arg_release(arg);
}

static void
listener_Toggle_cb (BonoboUIComponent *uic, const char *path,
					Bonobo_UIComponent_EventType type, const char *state,
					gpointer user_data)
{
	BonoboArg *arg;
	GgvPostScriptView *ps_view;
	CORBA_boolean flag;
	gchar *prop;
	int i;

	static gchar *properties[][2] = {
		{ "OverrideOrientation", "override_orientation" },
		{ "OverrideSize", "override_size" },
		{ "RespectEOF", "respect_eof" },
		{ "Antialiasing", "antialiasing" },
		{ "WatchFile", "watch_file" },
		{ NULL, NULL }
	};

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW (user_data));

	if(type != Bonobo_UIComponent_STATE_CHANGED)
		return;

	if(!state)
		return;
	
	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	if(atoi(state))
		flag = CORBA_TRUE;
	else
		flag = CORBA_FALSE;

	prop = NULL;
	for(i = 0; properties[i][0]; i++) {
		if(!strcmp(path, properties[i][0])) {
			prop = properties[i][1];
			break;
		}
	}

	g_assert(prop != NULL);

	arg = bonobo_arg_new(TC_CORBA_boolean);
	BONOBO_ARG_SET_GENERAL(arg, flag, TC_CORBA_boolean,
						   CORBA_boolean, NULL);

	bonobo_pbclient_set_value(BONOBO_OBJREF(ps_view->priv->property_bag),
							  prop, arg, NULL);

	bonobo_arg_release(arg);
}

static void
listener_Size_cb(BonoboUIComponent *uic, const char *path,
				  Bonobo_UIComponent_EventType type, const char *state,
				  gpointer user_data)
{
	GgvPostScriptView *ps_view;
	GNOME_GGV_Size size;
	BonoboArg *arg;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(user_data));

	if(type != Bonobo_UIComponent_STATE_CHANGED)
		return;

	if(!state || !atoi(state))
		return;

	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	size = CORBA_string_dup(path + strlen("PaperSize"));

	arg = bonobo_arg_new(TC_GNOME_GGV_Size);
	BONOBO_ARG_SET_GENERAL(arg, size, TC_GNOME_GGV_Size,
						   GNOME_GGV_Size, NULL);

	bonobo_pbclient_set_value(BONOBO_OBJREF(ps_view->priv->property_bag),
							  "default_size", arg, NULL);

	bonobo_arg_release(arg);
}


static gchar *
ggv_postscript_view_get_ps(GgvPostScriptView *ps_view, gint *active_rows)
{
	gchar *ps;

	if(active_rows == NULL)
		ps = gtk_gs_get_postscript(GTK_GS(ps_view->priv->gs), NULL);
	else {
		gint num, i;
		GtkGS *gs;
		gint *page_mask;

		gs = GTK_GS(ps_view->priv->gs);
		num = gtk_gs_get_page_count(gs);
		page_mask = g_new0(gint, num);

		for(i = 0; active_rows[i] != -1; i++) {
			if(active_rows[i] < num)
				page_mask[active_rows[i]] = TRUE;
		}
		ps = gtk_gs_get_postscript(gs, page_mask);
	}
	return ps;
}

static void
ggv_postscript_view_print(GgvPostScriptView *ps_view, gchar *ps)
{
	gchar **argv;
	gint print_in;

	argv = g_strsplit(ggv_print_cmd, " ", 0);
	if(!g_spawn_async_with_pipes(NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
								 NULL, NULL, NULL, &print_in, NULL, NULL,
								 NULL)) {
		GtkWidget *dlg;
		dlg = gtk_message_dialog_new(GTK_WINDOW(gtk_widget_get_toplevel(ps_view->priv->gs)),
									 GTK_DIALOG_MODAL,
									 GTK_MESSAGE_ERROR,
									 GTK_BUTTONS_OK,
									 _("Unable to execute print command:\n%s"),
									 ggv_print_cmd);
		gtk_widget_show(dlg);
		gtk_dialog_run(GTK_DIALOG(dlg));
		gtk_widget_destroy(dlg);
	}
	else {
		write(print_in, ps, strlen(ps));
		close(print_in);
	}
	g_strfreev(argv);
}

static void
save_file_sel_unmap(GtkWidget *widget)
{
	ggv_get_window_size(widget, &file_sel_width, &file_sel_height);
}

static gboolean
save_file_sel_delete_event(GtkWidget *widget, GdkEventAny *e, gpointer data)
{
	gtk_widget_hide(widget);
	return FALSE;
}

static void
save_file_sel_ok_clicked(GtkWidget *widget, gpointer data)
{
	GgvPostScriptView *ps_view = GGV_POSTSCRIPT_VIEW(data);
	gint *active_rows;
	const gchar *fname;
	gchar *doc;
	FILE *f;

	active_rows = ggv_sidebar_get_active_list(ps_view->priv->sidebar);
	doc = ggv_postscript_view_get_ps(ps_view, active_rows);
	g_free(active_rows);

	if(doc != NULL) {
		/* get file name */
		gtk_widget_hide(ps_view->priv->save_file_sel);
		fname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(ps_view->priv->save_file_sel));
		
		if(fname && (f = fopen(fname, "w")) != NULL) {
			fwrite(doc, 1, strlen(doc), f);
			fclose(f);
		}
		g_free(doc);
	}
}

static void
save_file_sel_cancel_clicked(GtkWidget *widget, gpointer data)
{
	GgvPostScriptView *ps_view = GGV_POSTSCRIPT_VIEW(data);

	gtk_widget_hide(ps_view->priv->save_file_sel);
}

static void
verb_FileSaveMarked(BonoboUIComponent *uic, gpointer data, const char *cname)
{
	GgvPostScriptView *ps_view = GGV_POSTSCRIPT_VIEW(data);
	gint *active_rows;
	
	active_rows = ggv_sidebar_get_active_list(ps_view->priv->sidebar);
	if(active_rows[0] == -1) {
		GtkWidget *dlg;
		
		dlg = gtk_message_dialog_new(GTK_WINDOW(gtk_widget_get_toplevel(ps_view->priv->gs)),
									 GTK_DIALOG_MODAL,
									 GTK_MESSAGE_QUESTION,
									 GTK_BUTTONS_YES_NO,
									 _("No pages have been marked.\n"
									   "Do you want to save the whole document?"));
		gtk_widget_show(dlg);
		switch(gtk_dialog_run(GTK_DIALOG(dlg))) {
		case GTK_RESPONSE_NO:
		case GTK_RESPONSE_DELETE_EVENT:
			gtk_widget_destroy(dlg);
			return;
		default:
			break;
		}
		gtk_widget_destroy(dlg);
	}
	g_free(active_rows);
	
	if(!ps_view->priv->save_file_sel) {
		gchar *home_path = g_strconcat(g_get_home_dir(), "/", NULL);
		ps_view->priv->save_file_sel = gtk_file_selection_new(_("Select a file to save pages as"));
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(ps_view->priv->save_file_sel),
										home_path);
		g_free(home_path);
		gtk_window_set_transient_for(GTK_WINDOW(ps_view->priv->save_file_sel),
									 GTK_WINDOW(gtk_widget_get_toplevel(ps_view->priv->gs)));
		gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(ps_view->priv->save_file_sel));
		gtk_window_set_resizable(GTK_WINDOW(ps_view->priv->save_file_sel), TRUE);
		g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(ps_view->priv->save_file_sel)->ok_button),
						 "clicked", G_CALLBACK(save_file_sel_ok_clicked), ps_view);
		g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(ps_view->priv->save_file_sel)->cancel_button),
						 "clicked", G_CALLBACK(save_file_sel_cancel_clicked), ps_view);
		g_signal_connect(G_OBJECT(ps_view->priv->save_file_sel), "delete_event",
						 G_CALLBACK(save_file_sel_delete_event), ps_view);
		g_signal_connect(G_OBJECT(ps_view->priv->save_file_sel), "unmap",
						 G_CALLBACK(save_file_sel_unmap), NULL);
	}
	if(!GTK_WIDGET_VISIBLE(ps_view->priv->save_file_sel)) {
		gtk_widget_set_usize(GTK_WIDGET(ps_view->priv->save_file_sel),
							 file_sel_width, file_sel_height);
		gtk_widget_show(ps_view->priv->save_file_sel);
		ggv_raise_and_focus_widget(ps_view->priv->save_file_sel);
	}
}

static void
verb_FilePrintMarked(BonoboUIComponent *uic, gpointer data, const char *cname)
{
	GgvPostScriptView *ps_view = GGV_POSTSCRIPT_VIEW(data);
	gchar *ps;
	gint *active_rows;
	
	active_rows = ggv_sidebar_get_active_list(ps_view->priv->sidebar);
	if(active_rows[0] == -1) {
		GtkWidget *dlg;
		
		dlg = gtk_message_dialog_new(GTK_WINDOW(gtk_widget_get_toplevel(ps_view->priv->gs)),
									 GTK_DIALOG_MODAL,
									 GTK_MESSAGE_QUESTION,
									 GTK_BUTTONS_YES_NO,
									 _("No pages have been marked.\n"
									   "Do you want to print the whole document?"));
		gtk_widget_show(dlg);
		switch(gtk_dialog_run(GTK_DIALOG(dlg))) {
		case GTK_RESPONSE_NO:
		case GTK_RESPONSE_DELETE_EVENT:
			gtk_widget_destroy(dlg);
			return;
		default:
			break;
		}
		gtk_widget_destroy(dlg);
	}
	if((ps = ggv_postscript_view_get_ps(ps_view, active_rows)) != NULL) {
		ggv_postscript_view_print(ps_view, ps);
		g_free(ps);
	}
	g_free(active_rows);
}

static void
verb_FilePrintAll(BonoboUIComponent *uic, gpointer data, const char *cname)
{
	GgvPostScriptView *ps_view = GGV_POSTSCRIPT_VIEW(data);
	gchar *ps;

	if((ps = ggv_postscript_view_get_ps(ps_view, NULL)) != NULL) {
		ggv_postscript_view_print(ps_view, ps);
		g_free(ps);
	}
}

static void
verb_NextPage_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	GgvPostScriptView *ps_view;
	GtkGS *gs;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_POSTSCRIPT_VIEW (user_data));

	ps_view = GGV_POSTSCRIPT_VIEW (user_data);
	gs = GTK_GS(ps_view->priv->gs);

	ggv_postscript_view_goto_page(ps_view, gtk_gs_get_current_page(gs) + 1);
}

static void
verb_PrevPage_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	GgvPostScriptView *ps_view;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_POSTSCRIPT_VIEW (user_data));

	ps_view = GGV_POSTSCRIPT_VIEW (user_data);

	ggv_postscript_view_goto_page(ps_view,
								  GTK_GS(ps_view->priv->gs)->current_page - 1);

}

static void
verb_LastPage_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	GgvPostScriptView *ps_view;
	GtkGS *gs;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_POSTSCRIPT_VIEW (user_data));

	ps_view = GGV_POSTSCRIPT_VIEW (user_data);
	gs = GTK_GS(ps_view->priv->gs);

	ggv_postscript_view_goto_page(ps_view, gtk_gs_get_page_count(gs) - 1);
}

static void
verb_FirstPage_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	GgvPostScriptView *ps_view;
	GtkGS *gs;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_POSTSCRIPT_VIEW (user_data));

	ps_view = GGV_POSTSCRIPT_VIEW (user_data);
	gs = GTK_GS(ps_view->priv->gs);

	ggv_postscript_view_goto_page(ps_view, 0);
}

static void
verb_SettingsPreferences(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        static GtkWidget *dlg = NULL;

        if(dlg == NULL) {
                dlg = ggv_prefs_dialog_new();
        }
        ggv_prefs_dialog_show(GGV_PREFS_DIALOG(dlg));
		ggv_raise_and_focus_widget(dlg);
}

static void
sync_orientation_items(GgvPostScriptView *ps_view)
{
	gint i;
	gboolean val, orient_state;

	val = gtk_gs_get_override_orientation(GTK_GS(ps_view->priv->gs));
	bonobo_ui_component_set_prop(ps_view->priv->uic, orientation_paths[0],
								 "state", val?"1":"0", NULL);
	for(i = 1; orientation_paths[i] != NULL; i++) {
		bonobo_ui_component_set_prop(ps_view->priv->uic, orientation_paths[i],
									 "sensitive", val?"1":"0", NULL);
		orient_state = (i - 1 == gtk_gs_get_default_orientation(GTK_GS(ps_view->priv->gs)));
		bonobo_ui_component_set_prop(ps_view->priv->uic, orientation_paths[i],
									 "state", orient_state?"1":"0", NULL);
	}
}

static void
sync_size_items(GgvPostScriptView *ps_view)
{
	gint i;
	gboolean val, size_state;
	const gchar **paths;

	paths = size_paths;

	val = gtk_gs_get_override_size(GTK_GS(ps_view->priv->gs));
	bonobo_ui_component_set_prop(ps_view->priv->uic, paths[0],
								 "state", val?"1":"0", NULL);
	for(i = 1; paths[i] != NULL; i++) {
		bonobo_ui_component_set_prop(ps_view->priv->uic, paths[i],
									 "sensitive", val?"1":"0", NULL);
		size_state = i - 1 == gtk_gs_get_default_size(GTK_GS(ps_view->priv->gs));
		bonobo_ui_component_set_prop(ps_view->priv->uic, paths[i],
									 "state", size_state?"1":"0", NULL);
	}
}

BonoboUIVerb ggv_postscript_view_verbs[] = {
	BONOBO_UI_VERB("SettingsPreferences", verb_SettingsPreferences),
	BONOBO_UI_VERB("FileSaveMarked", verb_FileSaveMarked),
	BONOBO_UI_VERB("FilePrintMarked", verb_FilePrintMarked),
	BONOBO_UI_VERB("FilePrintAll", verb_FilePrintAll),
	BONOBO_UI_VERB("NextPage", verb_NextPage_cb),
	BONOBO_UI_VERB("PrevPage", verb_PrevPage_cb),
	BONOBO_UI_VERB("LastPage", verb_LastPage_cb),
	BONOBO_UI_VERB("FirstPage", verb_FirstPage_cb),
	BONOBO_UI_VERB_END
};

static void
ggv_postscript_view_create_ui(GgvPostScriptView *ps_view)
{
	BonoboUINode *ui_root;
	GgvPostScriptViewClass *klass = GGV_POSTSCRIPT_VIEW_CLASS(G_OBJECT_GET_CLASS(ps_view));
	gint i;

	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	/* Set up the UI from an XML file. */
	bonobo_ui_util_set_ui(ps_view->priv->uic, DATADIR,
						  "ggv-postscript-view-ui.xml", 
						  "GGV", NULL);
	bonobo_ui_component_object_set(ps_view->priv->uic, "/Sidebar/GgvSidebar",
								   BONOBO_OBJREF(ps_view->priv->sidebar), NULL);
	bonobo_ui_component_set_prop(ps_view->priv->uic, "/Sidebar",
								 "hidden", ggv_panel?"0":"1", NULL);
	bonobo_ui_component_add_verb_list_with_data(ps_view->priv->uic,
												ggv_postscript_view_verbs,
												ps_view);

	for(i = 1; size_paths[i]; i++) 
		bonobo_ui_component_add_listener(ps_view->priv->uic,
										 size_paths[i] + strlen("/commands/"),
										 listener_Size_cb, ps_view);

	for(i = 1; orientation_paths[i]; i++) 
		bonobo_ui_component_add_listener(ps_view->priv->uic,
										 orientation_paths[i] + strlen("/commands/"),
										 listener_Orientation_cb, ps_view);
	bonobo_ui_component_add_listener(ps_view->priv->uic, "OverrideSize",
									 listener_Toggle_cb, ps_view);
	bonobo_ui_component_add_listener(ps_view->priv->uic, "OverrideOrientation",
									 listener_Toggle_cb, ps_view);

	set_page_item_sensitivity(ps_view);
	sync_orientation_items(ps_view);
	sync_size_items(ps_view);
}

gchar **
ggv_postscript_view_get_page_names(GgvPostScriptView *ps_view)
{
	gchar **names = NULL;
	GtkGS *gs;
	gint i;

	gs = GTK_GS(ps_view->priv->gs);
	if(gs->loaded && gs->doc && gs->structured_doc) {
		names = g_new0(gchar *, gs->doc->numpages + 1);
		for(i = 0; i < gs->doc->numpages; i++) {
			names[i] = g_strdup(gs->doc->pages[i].label);
		}
	}
	return names;
}

static void
ggv_postscript_view_get_prop(BonoboPropertyBag *bag,
							 BonoboArg         *arg,
							 guint              arg_id,
							 CORBA_Environment *ev,
							 gpointer           user_data)
{
	GgvPostScriptView *ps_view;
	gchar *size;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(user_data));

	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	switch(arg_id) {
	case PROP_PAGE: {
		g_assert(arg->_type == TC_CORBA_long);

		*(CORBA_long *)arg->_value = gtk_gs_get_current_page(GTK_GS(ps_view->priv->gs));
		break;
	}
	case PROP_PAGE_COUNT: {
		g_assert(arg->_type == TC_CORBA_long);

		*(CORBA_long *)arg->_value = gtk_gs_get_page_count(GTK_GS(ps_view->priv->gs));
		break;
	}
	case PROP_PAGE_NAMES: {
		GNOME_GGV_PageNameList *names;
		GtkGS *gs;
		int i;

		g_assert(arg->_type == TC_GNOME_GGV_PageNameList);

		names = GNOME_GGV_PageNameList__alloc();
		names->_length = 0;
		names->_buffer = NULL;
		gs = GTK_GS(ps_view->priv->gs);
		if(gs->loaded && gs->doc && gs->structured_doc) {
			names->_length = gs->doc->numpages;
			names->_buffer = CORBA_sequence_CORBA_string_allocbuf(names->_length);
			for(i = 0; i < gs->doc->numpages; i++) {
				names->_buffer[i] = CORBA_string_dup(gs->doc->pages[i].label);
			}
		}
		CORBA_sequence_set_release(names, CORBA_TRUE);
		arg->_value = names;
		break;
	}
	case PROP_TITLE: {
		g_assert(arg->_type == TC_CORBA_string);
		get_title(ps_view, arg);
		break;
	}
	case PROP_STATUS: {
		g_assert(arg->_type == TC_CORBA_string);
		get_status(ps_view, arg);
		break;
	}
	case PROP_ORIENTATION: {
		GNOME_GGV_Orientation orient;

		g_assert(arg->_type == TC_GNOME_GGV_Orientation);

		switch(gtk_gs_get_orientation(GTK_GS(ps_view->priv->gs))) {
		case GTK_GS_ORIENTATION_PORTRAIT:
			orient = GNOME_GGV_ORIENTATION_PORTRAIT;
			break;
		case GTK_GS_ORIENTATION_LANDSCAPE:
			orient = GNOME_GGV_ORIENTATION_LANDSCAPE;
			break;
		case GTK_GS_ORIENTATION_UPSIDEDOWN:
			orient = GNOME_GGV_ORIENTATION_UPSIDEDOWN;
			break;
		case GTK_GS_ORIENTATION_SEASCAPE:
			orient = GNOME_GGV_ORIENTATION_SEASCAPE;
			break;
		default:
			orient = GNOME_GGV_ORIENTATION_PORTRAIT;
			break;
		}
		*(GNOME_GGV_Orientation *)arg->_value = orient;
		break;
	}
	case PROP_WIDTH: {
		CORBA_float w;
		GtkGSOrientation orient;
		GtkGS *gs = GTK_GS(ps_view->priv->gs);

		g_assert(arg->_type == TC_CORBA_float);

        orient = gtk_gs_get_orientation(gs);
		switch(orient) {
		case GTK_GS_ORIENTATION_PORTRAIT:
        case GTK_GS_ORIENTATION_UPSIDEDOWN:
			w = gs->urx - gs->llx;
			break;
        case GTK_GS_ORIENTATION_LANDSCAPE:
        case GTK_GS_ORIENTATION_SEASCAPE:
			w = gs->ury - gs->lly;
			break;
		}
		w = MAX(w, 0);
		*(CORBA_long *)arg->_value = w;
		break;
	}
	case PROP_HEIGHT: {
		CORBA_float h;
		GtkGSOrientation orient;
		GtkGS *gs = GTK_GS(ps_view->priv->gs);

		g_assert(arg->_type == TC_CORBA_float);

        orient = gtk_gs_get_orientation(gs);
		switch(orient) {
		case GTK_GS_ORIENTATION_PORTRAIT:
        case GTK_GS_ORIENTATION_UPSIDEDOWN:
			h = gs->ury - gs->lly;
			break;
        case GTK_GS_ORIENTATION_LANDSCAPE:
        case GTK_GS_ORIENTATION_SEASCAPE:
			h = gs->urx - gs->llx;
			break;
		}
		h = MAX(h, 0);
		*(CORBA_long *)arg->_value = h;
		break;
	}
	case PROP_DEFAULT_ORIENTATION: {
		g_assert(arg->_type == TC_GNOME_GGV_Orientation);

		*(GNOME_GGV_Orientation *)arg->_value = ps_view->priv->def_orientation;
		break;
	}
	case PROP_OVERRIDE_ORIENTATION: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_override_orientation(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	case PROP_DEFAULT_SIZE: {
		g_assert(arg->_type == TC_GNOME_GGV_Size);

		size = gtk_gs_defaults_get_paper_sizes()
			[gtk_gs_get_default_size(GTK_GS(ps_view->priv->gs))].name;

		*(GNOME_GGV_Size *)arg->_value =
			CORBA_string_dup(size);
		break;
	}
	case PROP_OVERRIDE_SIZE: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_override_size(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	case PROP_RESPECT_EOF: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_respect_eof(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	case PROP_ANTIALIASING: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_antialiasing(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	case PROP_WATCH_FILE: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_watch_file(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	default:
		g_assert_not_reached();
	}
}

static void
ggv_postscript_view_set_prop(BonoboPropertyBag *bag,
							 const BonoboArg   *arg,
							 guint              arg_id,
							 CORBA_Environment *ev,
							 gpointer           user_data)
{
	GgvPostScriptView *ps_view;
	GgvPostScriptViewClass *klass;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(user_data));

	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	klass = GGV_POSTSCRIPT_VIEW_CLASS(G_OBJECT_GET_CLASS(ps_view));

	switch(arg_id) {
	case PROP_PAGE: {
		g_assert(arg->_type == TC_CORBA_long);
		ggv_postscript_view_goto_page(ps_view, *(CORBA_long *)arg->_value);
		break;
	}
	case PROP_DEFAULT_ORIENTATION: {
		GtkGSOrientation orient = GTK_GS_ORIENTATION_PORTRAIT;

		g_assert(arg->_type == TC_GNOME_GGV_Orientation);

		ps_view->priv->def_orientation = *(GNOME_GGV_Orientation *) arg->_value;

		switch(ps_view->priv->def_orientation) {
		case GNOME_GGV_ORIENTATION_PORTRAIT:
			orient = GTK_GS_ORIENTATION_PORTRAIT;
			break;
		case GNOME_GGV_ORIENTATION_LANDSCAPE:
			orient = GTK_GS_ORIENTATION_LANDSCAPE;
			break;
		case GNOME_GGV_ORIENTATION_UPSIDEDOWN:
			orient = GTK_GS_ORIENTATION_UPSIDEDOWN;
			break;
		case GNOME_GGV_ORIENTATION_SEASCAPE:
			orient = GTK_GS_ORIENTATION_SEASCAPE;
			break;
		default:
			g_assert_not_reached();
		}

		gtk_gs_set_default_orientation(GTK_GS(ps_view->priv->gs), orient);
		notify_orientation_change(ps_view);
		/*
		sync_orientation_items(ps_view);
		*/

		break;
	}
	case PROP_OVERRIDE_ORIENTATION: {
		 gtk_gs_set_override_orientation(GTK_GS(ps_view->priv->gs),
										 *(CORBA_boolean *)arg->_value);
		 notify_orientation_change(ps_view);
		 sync_orientation_items(ps_view);
		 break;
	}
	case PROP_DEFAULT_SIZE: {
		gint size;
		gchar *size_name;

		g_assert(arg->_type == TC_GNOME_GGV_Size);
		
		size_name = *(GNOME_GGV_Size *)arg->_value;

		size = gtk_gs_get_size_index(size_name,
									   gtk_gs_defaults_get_paper_sizes());

		gtk_gs_set_default_size(GTK_GS(ps_view->priv->gs), size);
		sync_size_items(ps_view);

		break;
	}
	case PROP_OVERRIDE_SIZE: {
		 gtk_gs_set_override_size(GTK_GS(ps_view->priv->gs),
								   *(CORBA_boolean *)arg->_value);
		 sync_size_items(ps_view);
		 break;
	}
	case PROP_ANTIALIASING: {
		 gtk_gs_set_override_size(GTK_GS(ps_view->priv->gs),
								   *(CORBA_boolean *)arg->_value);
		 break;
	}
	case PROP_RESPECT_EOF: {
		 gtk_gs_set_respect_eof(GTK_GS(ps_view->priv->gs),
								*(CORBA_boolean *)arg->_value);
		 break;
	}
	case PROP_WATCH_FILE: {
		 gtk_gs_set_watch_file(GTK_GS(ps_view->priv->gs),
							   *(CORBA_boolean *)arg->_value);
		 break;
	}
	default:
		g_assert_not_reached();
	}
}
	
static Bonobo_Unknown
ggv_postscript_view_get_object(BonoboItemContainer *item_container,
							   CORBA_char          *item_name,
							   CORBA_boolean       only_if_exists,
							   CORBA_Environment   *ev,
							   GgvPostScriptView   *ps_view)
{
	Bonobo_Unknown corba_object;
	BonoboObject *object = NULL;
	GSList *params, *c;

	g_return_val_if_fail(ps_view != NULL, CORBA_OBJECT_NIL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), CORBA_OBJECT_NIL);

	g_message ("ggv_postscript_view_get_object: %d - %s",
			   only_if_exists, item_name);

	params = ggv_split_string (item_name, "!");
	for (c = params; c; c = c->next) {
		gchar *name = c->data;

		if ((!strcmp (name, "control") || !strcmp (name, "embeddable"))
		    && (object != NULL)) {
			g_message ("ggv_postscript_view_get_object: "
					   "can only return one kind of an Object");
			continue;
		}

		if (!strcmp (name, "control"))
			object = (BonoboObject *) ggv_control_new (ps_view);
#if 0
		else if (!strcmp (item_name, "embeddable"))
			object = (BonoboObject *) ggv_embeddable_new (image);
#endif
		else
			g_message ("ggv_postscript_view_get_object: "
					   "unknown parameter `%s'",
					   name);
	}

	g_slist_foreach (params, (GFunc) g_free, NULL);
	g_slist_free (params);

	if (object == NULL)
		return NULL;

	corba_object = bonobo_object_corba_objref (object);

	return bonobo_object_dup_ref (corba_object, ev);
}

BonoboObject *
ggv_postscript_view_add_interfaces (GgvPostScriptView *ps_view,
									BonoboObject *to_aggregate)
{
	BonoboPersistFile   *file;
	BonoboPersistStream *stream;
	BonoboItemContainer *item_container;
	
	g_return_val_if_fail (GGV_IS_POSTSCRIPT_VIEW (ps_view), NULL);
	g_return_val_if_fail (BONOBO_IS_OBJECT (to_aggregate), NULL);

	/* Interface Bonobo::PersistStream */
	stream = bonobo_persist_stream_new (load_ps_from_stream, 
										NULL, NULL, NULL, ps_view);
	if (!stream) {
		bonobo_object_unref (BONOBO_OBJECT (to_aggregate));
		return NULL;
	}

	bonobo_object_add_interface (BONOBO_OBJECT (to_aggregate),
								 BONOBO_OBJECT (stream));

	/* Interface Bonobo::PersistFile */
	file = bonobo_persist_file_new (load_ps_from_file, NULL,
									"OAFIID:GNOME_GGV_PostScriptView", ps_view);
	if (!file) {
		bonobo_object_unref (BONOBO_OBJECT (to_aggregate));
		return NULL;
	}

	bonobo_object_add_interface (BONOBO_OBJECT (to_aggregate),
								 BONOBO_OBJECT (file));

	/* BonoboItemContainer */
	item_container = bonobo_item_container_new ();

	gtk_signal_connect (GTK_OBJECT (item_container),
						"get_object",
						GTK_SIGNAL_FUNC (ggv_postscript_view_get_object),
						ps_view);

	bonobo_object_add_interface (BONOBO_OBJECT (to_aggregate),
								 BONOBO_OBJECT (item_container));

	return to_aggregate;
}

BonoboPropertyBag *
ggv_postscript_view_get_property_bag(GgvPostScriptView *ps_view)
{
	g_return_val_if_fail(ps_view != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), NULL);

	bonobo_object_ref(BONOBO_OBJECT(ps_view->priv->property_bag));

	return ps_view->priv->property_bag;
}

BonoboPropertyControl *
ggv_postscript_view_get_property_control(GgvPostScriptView *ps_view)
{
	g_return_val_if_fail(ps_view != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), NULL);

	bonobo_object_ref(BONOBO_OBJECT(ps_view->priv->property_control));

	return ps_view->priv->property_control;
}

void
ggv_postscript_view_set_ui_container(GgvPostScriptView *ps_view,
				 Bonobo_UIContainer ui_container)
{
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));
	g_return_if_fail(ui_container != CORBA_OBJECT_NIL);

	bonobo_ui_component_set_container(ps_view->priv->uic, ui_container, NULL);

	ggv_postscript_view_create_ui(ps_view);
}

void
ggv_postscript_view_unset_ui_container(GgvPostScriptView *ps_view)
{
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	bonobo_ui_component_unset_container(ps_view->priv->uic, NULL);
}

GtkWidget *
ggv_postscript_view_get_widget(GgvPostScriptView *ps_view)
{
	g_return_val_if_fail(ps_view != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), NULL);

	gtk_widget_ref(ps_view->priv->gs);

	return ps_view->priv->gs;
}

float
ggv_postscript_view_get_zoom_factor(GgvPostScriptView *ps_view)
{
	g_return_val_if_fail(ps_view != NULL, 0.0);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), 0.0);

	return gtk_gs_get_zoom(GTK_GS(ps_view->priv->gs));
}

void
ggv_postscript_view_set_zoom_factor(GgvPostScriptView *ps_view,
									float zoom_factor)
{
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));
	g_return_if_fail(zoom_factor > 0.0);

	gtk_gs_set_zoom(GTK_GS(ps_view->priv->gs), zoom_factor);
}

void
ggv_postscript_view_set_zoom(GgvPostScriptView *ps_view,
							 double zoomx, double zoomy)
{
	g_return_if_fail(zoomx > 0.0);
	g_return_if_fail(zoomy > 0.0);
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	gtk_gs_set_zoom(GTK_GS(ps_view->priv->gs), zoomx);
}

gfloat
ggv_postscript_view_zoom_to_fit(GgvPostScriptView *ps_view,
								gboolean fit_width)
{
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	/* we don't know how not to keep the aspect ratio */
	return gtk_gs_zoom_to_fit(GTK_GS(ps_view->priv->gs), fit_width);
}

void
ggv_postscript_view_set_default_orientation(GgvPostScriptView *ps_view,
											GNOME_GGV_Orientation orientation)
{
	BonoboArg *arg;

	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	arg = bonobo_arg_new(TC_GNOME_GGV_Orientation);
	BONOBO_ARG_SET_GENERAL(arg, orientation, TC_GNOME_GGV_Orientation,
						   GNOME_GGV_Orientation, NULL);

	bonobo_pbclient_set_value(BONOBO_OBJREF(ps_view->priv->property_bag),
							  "default_orientation", arg, NULL);
}

GNOME_GGV_Orientation
ggv_postscript_view_get_default_orientation(GgvPostScriptView *ps_view)
{
	BonoboArg *arg;

	g_return_val_if_fail(ps_view != NULL, 0);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), 0);

	arg = bonobo_pbclient_get_value(BONOBO_OBJREF(ps_view->priv->property_bag),
									"default_orientation", NULL, NULL);
	if(arg == NULL)
		return 0;

	g_assert(arg->_type == TC_GNOME_GGV_Orientation);
	return *(GNOME_GGV_Orientation *) arg->_value;
}

static void
ggv_postscript_view_destroy(BonoboObject *object)
{
	GgvPostScriptView *ps_view;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(object));

	ps_view = GGV_POSTSCRIPT_VIEW(object);

	if(ps_view->priv->property_bag) {
		bonobo_object_unref(BONOBO_OBJECT(ps_view->priv->property_bag));
		ps_view->priv->property_bag = NULL;
	}

	if(ps_view->priv->property_control) {
		bonobo_object_unref(BONOBO_OBJECT(ps_view->priv->property_control));
		ps_view->priv->property_control = NULL;
	}

	if(ps_view->priv->gs) {
		gtk_widget_destroy(ps_view->priv->gs);
		ps_view->priv->gs = NULL;
	}

	if(ps_view->priv->msg_win) {
		ggv_msg_window_free(ps_view->priv->msg_win);
		ps_view->priv->msg_win = NULL;
	}

	if(ps_view->priv->uic) {
		bonobo_object_unref(BONOBO_OBJECT(ps_view->priv->uic));
		ps_view->priv->uic = NULL;
	}

	ggv_prefs_remove_callback(ps_view);

	if (BONOBO_OBJECT_CLASS (ggv_postscript_view_parent_class)->destroy)
		BONOBO_OBJECT_CLASS(ggv_postscript_view_parent_class)->destroy(object);
}

static void
ggv_postscript_view_finalize(GObject *object)
{
	GgvPostScriptView *ps_view;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(object));

	ps_view = GGV_POSTSCRIPT_VIEW(object);

	g_free(ps_view->priv);

	G_OBJECT_CLASS(ggv_postscript_view_parent_class)->finalize(object);
}

static CORBA_string
impl_GNOME_GGV_PostScriptView_getDocument(PortableServer_Servant servant,
										  CORBA_Environment *ev)
{
	GgvPostScriptView *ps_view;
	gchar *doc;
	CORBA_string retval;

	ps_view = GGV_POSTSCRIPT_VIEW(bonobo_object_from_servant(servant));
	doc = gtk_gs_get_postscript(GTK_GS(ps_view->priv->gs), NULL);
	retval = CORBA_string_dup(doc);
	g_free(doc);
	return retval;
}

static CORBA_string
impl_GNOME_GGV_PostScriptView_getPages(PortableServer_Servant servant,
									   GNOME_GGV_PageList *pages,
									   CORBA_Environment *ev)
{
	gint *page_mask;
	GgvPostScriptView *ps_view;
	gchar *doc;
	CORBA_string retval;
	GtkGS *gs;
	int i, num;

	ps_view = GGV_POSTSCRIPT_VIEW(bonobo_object_from_servant(servant));
	gs = GTK_GS(ps_view->priv->gs);
	num = gtk_gs_get_page_count(gs);
	page_mask = g_new0(gint, num);
	for(i = 0; i < pages->_length; i++) {
		if(pages->_buffer[i] < num)
			page_mask[pages->_buffer[i]] = TRUE;
	}
	doc = gtk_gs_get_postscript(gs, page_mask);
	retval = CORBA_string_dup(doc);
	g_free(doc);
	return retval;
}

static void
ggv_postscript_view_class_init(GgvPostScriptViewClass *klass)
{
	GObjectClass *object_class = (GObjectClass *)klass;
	BonoboObjectClass *bonobo_object_class = (BonoboObjectClass *)klass;
	gint n = gtk_gs_defaults_get_paper_count(), i;
	BonoboUINode *placeholder;
	BonoboUINode *item;
	gchar *enc_str;
	GtkGSPaperSize *papersizes;
	POA_GNOME_GGV_PostScriptView__epv *epv;

	klass->priv = g_new0(GgvPostScriptViewClassPrivate, 1);

	ggv_postscript_view_parent_class = g_type_class_peek_parent(klass);

	bonobo_object_class->destroy = ggv_postscript_view_destroy;
	object_class->finalize = ggv_postscript_view_finalize;
	
	epv = &klass->epv;

	epv->getDocument = impl_GNOME_GGV_PostScriptView_getDocument;
	epv->getPages = impl_GNOME_GGV_PostScriptView_getPages;

	/* this seems as a nice place to load the prefs */
	ggv_prefs_load();
}

static void
ggv_postscript_view_init(GgvPostScriptView *ps_view)
{
	ps_view->priv = g_new0(GgvPostScriptViewPrivate, 1);
}

BONOBO_TYPE_FUNC_FULL(GgvPostScriptView, 
					  GNOME_GGV_PostScriptView,
					  BONOBO_TYPE_OBJECT,
					  ggv_postscript_view);

static void
property_control_get_prop(BonoboPropertyBag *bag,
						  BonoboArg         *arg,
						  guint              arg_id,
						  CORBA_Environment *ev,
						  gpointer           user_data)
{
	switch(arg_id) {
	case PROP_CONTROL_TITLE:
		g_assert(arg->_type == BONOBO_ARG_STRING);
		BONOBO_ARG_SET_STRING(arg, _("Display"));
		break;
	default:
		g_assert_not_reached();
	}
}

static BonoboControl *
property_control_get_cb(BonoboPropertyControl *property_control,
						int page_number, void *closure)
{
	GgvPostScriptView *ps_view;
	GtkWidget *container;
	BonoboControl *control;
	BonoboPropertyBag *property_bag;

	g_return_val_if_fail(closure != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(closure), NULL);
	g_return_val_if_fail(page_number == 0, NULL);

	ps_view = GGV_POSTSCRIPT_VIEW(closure);

#if 0
	container = eog_create_preferences_page(ps_view, page_number);

	gtk_widget_show_all(container);

	control = bonobo_control_new(container);

	/* Property Bag */
	property_bag = bonobo_property_bag_new(property_control_get_prop,
										   NULL, control);

	bonobo_property_bag_add(property_bag, "bonobo:title",
							PROP_CONTROL_TITLE, BONOBO_ARG_STRING,
							NULL, NULL, BONOBO_PROPERTY_READABLE);

	bonobo_object_add_interface(BONOBO_OBJECT(control),
								BONOBO_OBJECT(property_bag));

	return control;
#else
	return NULL;
#endif
}

static void
interpreter_message_cb(GtkGS *gs, gchar *msg, gpointer data)
{
	GgvPostScriptView *ps_view = GGV_POSTSCRIPT_VIEW(data);

	if(ps_view->priv->msg_win)
		ggv_msg_window_add_text(ps_view->priv->msg_win, msg, TRUE);

#ifdef BONOBO_DEBUG
	g_warning(msg);
#endif /* BONOBO_DEBUG */
}

static void
interpreter_error_cb(GtkGS *gs, gint status, gpointer data)
{
	notify_title_change(GGV_POSTSCRIPT_VIEW(data));
	notify_page_change(GGV_POSTSCRIPT_VIEW(data));
	notify_page_count_change(GGV_POSTSCRIPT_VIEW(data));
}

GgvPostScriptView *
ggv_postscript_view_construct(GgvPostScriptView *ps_view,
							  GtkGS *gs, gboolean zoom_fit)
{
	g_return_val_if_fail(ps_view != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), NULL);
	g_return_val_if_fail(gs != NULL, NULL);
	g_return_val_if_fail(GTK_IS_GS(gs), NULL);
	g_return_val_if_fail(!GTK_WIDGET_REALIZED(gs), NULL);

	/* Make sure GConf is initialized */
	if(!gconf_is_initialized())
		gconf_init(0, NULL, NULL);
	
	ps_view->priv->msg_win = ggv_msg_window_new(_("GhostScript output"));

	ps_view->priv->gs = GTK_WIDGET(gs);
	g_object_set(gs, "can-focus", TRUE, NULL);
	gtk_widget_set_events(GTK_WIDGET(gs), 
						  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
						  GDK_POINTER_MOTION_MASK | GDK_KEY_PRESS_MASK);
	g_signal_connect(G_OBJECT(gs), "button_press_event",
					 G_CALLBACK(view_button_press_cb), ps_view);
	g_signal_connect(G_OBJECT(gs), "button_release_event",
					 G_CALLBACK(view_button_release_cb), ps_view);
	g_signal_connect(G_OBJECT(gs), "motion_notify_event",
					 G_CALLBACK(view_motion_cb), ps_view);
	g_signal_connect(G_OBJECT(gs), "interpreter_message",
					 G_CALLBACK(interpreter_message_cb), ps_view);
	g_signal_connect(G_OBJECT(gs), "interpreter_error",
					 G_CALLBACK(interpreter_error_cb), ps_view);

	gtk_widget_show(GTK_WIDGET(gs));
	ps_view->priv->zoom_fit = zoom_fit;

	/* Property Bag */
	ps_view->priv->property_bag = bonobo_property_bag_new(ggv_postscript_view_get_prop,
														  ggv_postscript_view_set_prop,
														  ps_view);


	bonobo_property_bag_add(ps_view->priv->property_bag, "title",
							PROP_TITLE, TC_CORBA_string, NULL,
							_("Document title"), 
							BONOBO_PROPERTY_READABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "status",
							PROP_STATUS, TC_CORBA_string, NULL,
							_("GGV control status"), 
							BONOBO_PROPERTY_READABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "page_count",
							PROP_PAGE_COUNT, TC_CORBA_long, NULL,
							_("Number of pages"), 
							BONOBO_PROPERTY_READABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "page_names",
							PROP_PAGE_NAMES, TC_GNOME_GGV_PageNameList, NULL,
							_("Page names"), 
							BONOBO_PROPERTY_READABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "page",
							PROP_PAGE, TC_CORBA_long, NULL,
							_("Current page number"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "width",
							PROP_WIDTH, TC_CORBA_float, NULL,
							_("Document width"), 
							BONOBO_PROPERTY_READABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "height",
							PROP_HEIGHT, TC_CORBA_float, NULL,
							_("Document height"), 
							BONOBO_PROPERTY_READABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "orientation",
							PROP_ORIENTATION, TC_GNOME_GGV_Orientation, NULL,
							_("Document orientation"), 
							BONOBO_PROPERTY_READABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "default_orientation",
							PROP_DEFAULT_ORIENTATION, TC_GNOME_GGV_Orientation, NULL,
							_("Default orientation"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "default_size",
							PROP_DEFAULT_SIZE, TC_GNOME_GGV_Size, NULL,
							_("Default size"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "override_orientation",
							PROP_OVERRIDE_ORIENTATION, TC_CORBA_boolean, NULL,
							_("Override document orientation"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "override_size",
							PROP_OVERRIDE_SIZE, TC_CORBA_boolean, NULL,
							_("Override document size"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "respect_eof",
							PROP_RESPECT_EOF, TC_CORBA_boolean, NULL,
							_("Respect EOF comment"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "watch_file",
							PROP_WATCH_FILE, TC_CORBA_boolean, NULL,
							_("Watch displayed file for changes"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "antialiasing",
							PROP_ANTIALIASING, TC_CORBA_boolean, NULL,
							_("Antialiasing"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);

	/* Property Control */
	ps_view->priv->property_control = bonobo_property_control_new
		(property_control_get_cb, 1, ps_view);

	/* UI Component */
	ps_view->priv->uic = bonobo_ui_component_new("GgvPostScriptView");

	ggv_postscript_view_add_interfaces (ps_view, BONOBO_OBJECT (ps_view));

	ps_view->priv->sidebar = ggv_sidebar_new(ps_view);

	ggv_prefs_add_callback(prefs_changed_handler, ps_view);

	return ps_view;
}

GgvPostScriptView *
ggv_postscript_view_new(GtkGS *gs, gboolean zoom_fit)
{
	GgvPostScriptView *ps_view;
	
	g_return_val_if_fail(gs != NULL, NULL);
	g_return_val_if_fail(GTK_IS_GS(gs), NULL);

	ps_view = g_object_new(GGV_POSTSCRIPT_VIEW_TYPE, NULL);

	return ggv_postscript_view_construct(ps_view, gs, zoom_fit);
}

GtkAdjustment *
ggv_postscript_view_get_hadj(GgvPostScriptView *ps_view)
{
	return GTK_GS(ps_view->priv->gs)->hadj;
}

GtkAdjustment *
ggv_postscript_view_get_vadj(GgvPostScriptView *ps_view)
{
	return GTK_GS(ps_view->priv->gs)->vadj;
}

void
ggv_postscript_view_set_popup_ui_component(GgvPostScriptView *ps_view,
										   BonoboUIComponent *uic)
{
	ps_view->priv->popup_uic = uic;
	_set_page_item_sensitivity(ps_view, uic);
}
