/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * main.c: the ggv shell
 *
 * Copyright (C) 2002 the Free Software Foundation
 *
 * Author: Jaka Mocnik  <jaka@gnu.org>
 */

#include <config.h>

#include <gnome.h>
#include <bonobo-activation/bonobo-activation.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gconf/gconf-client.h>
#include <bonobo.h>
#include <bonobo/bonobo-ui-main.h>

#include <math.h>
#include <ctype.h>

#include "ggv-window.h"
#include "ggv-prefs.h"

#define BONOBO_DEBUG

static gint win_count = 0;

static GgvWindow *
create_window(const gchar *filename)
{
        GgvWindow *win = GGV_WINDOW(ggv_window_new());

        if(filename)
                ggv_window_load(win, filename);

        gtk_widget_show(GTK_WIDGET(win));

        return win;
}

static gboolean
create_windows_from_cmdline(gpointer data)
{
        const gchar **files = NULL;
        poptContext *ctx;
        int i;

        ctx = (poptContext *)data;
        if(*ctx)
                files = poptGetArgs(*ctx);

        if(files)
                for(i = 0; files[i]; i++)
                        create_window(files[i]);
        else if(win_count == 0)
                win_count = 1;
        while(win_count > 0) {
                create_window(NULL);
                win_count--;
        }

        return FALSE;
}

static void
client_die(GnomeClient *client, gpointer data)
{
        gtk_main_quit ();
}

static gint
save_session(GnomeClient        *client,
             gint                phase,
             GnomeRestartStyle   save_style,
             gint                shutdown,
             GnomeInteractStyle  interact_style,
             gint                fast,
             gpointer            client_data)
{
	gchar *argv[128];
	gint wnc = 0, argc;
        const GList *node;
        gchar *win_opt = NULL;
        GgvWindow *win;

	argv[0] = (gchar *)client_data;
        argc = 1;
        node = ggv_get_window_list();
        while(node) {
                win = GGV_WINDOW(node->data);
                if(win->filename)
                        argv[argc++] = win->filename;
                else
                        wnc++;
                node = node->next;
        }
	if(wnc > 0) {
                win_opt = g_strdup_printf("--windows=%d", wnc);
                argv[argc++] = win_opt;
        }
        gnome_client_set_clone_command(client, argc, argv);
        gnome_client_set_restart_command(client, argc, argv);
        if(win_opt)
                g_free(win_opt);

	return TRUE;
}

int
main(int argc, char **argv)
{
	CORBA_Environment ev;
	CORBA_ORB orb;
	GError *error;
	poptContext ctx;
	GValue value = { 0, };
	GnomeProgram *program;
        GnomeClient *client;

	static struct poptOption options[] = {
		{ "windows", 'w', POPT_ARG_INT, &win_count, 0, 
		  N_("Specify the number of empty windows to open."), 
		  N_("Number of empty windows") },
		{ NULL, '\0', 0, NULL, 0 }
	};

	bindtextdomain(GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	program = gnome_program_init("ggv", VERSION,
                                     LIBGNOMEUI_MODULE,
                                     argc, argv,
                                     GNOME_PARAM_POPT_TABLE, options,
                                     GNOME_PARAM_APP_DATADIR, DATADIR, NULL);

	CORBA_exception_init(&ev);

        g_value_init(&value, G_TYPE_POINTER);
	g_object_get_property(G_OBJECT(program), GNOME_PARAM_POPT_CONTEXT,
                              &value);
	ctx = g_value_get_pointer(&value);
	g_value_unset(&value);

	error = NULL;
	if (gconf_init(argc, argv, &error) == FALSE) {
		g_assert(error != NULL);
		g_message("GConf init failed: %s", error->message);
		g_error_free (error);
		exit (EXIT_FAILURE);
	}

	if (bonobo_ui_init ("Gnome Ghostview", VERSION, &argc, argv) == FALSE)
		g_error (_("Could not initialize Bonobo!\n"));

        client = gnome_master_client();

	g_signal_connect (client, "save_yourself", G_CALLBACK (save_session), argv[0]);
	g_signal_connect (client, "die", G_CALLBACK (client_die), NULL);

        gtk_idle_add (create_windows_from_cmdline, &ctx);

        ggv_prefs_add_callback((GgvPrefsCb)ggv_window_apply_prefs, NULL);

	gnome_window_icon_set_default_from_file (GNOMEICONDIR "/gnome-ghostview.png");

        ggv_prefs_load();

	bonobo_main ();

#ifdef BONOBO_DEBUG
        bonobo_debug_shutdown ();
#endif /* BONOBO_DEBUG */

        g_warning("GGV exiting...");

        ggv_prefs_save();

	CORBA_exception_free (&ev);
	
        return 0;
}
