/* GIO - GLib Input, Output and Streaming Library
 * 
 * Copyright (C) 2006-2007 Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Alexander Larsson <alexl@redhat.com>
 */

#include <config.h>
#include <glib/gi18n-lib.h>


#include "gdatainputstream.h"

/* TODO: Real P_() */
#define P_(_x) (_x)

struct _GDataInputStreamPrivate {
  GDataStreamByteOrder byte_order;
  GDataStreamNewlineType newline_type;
};

enum {
  PROP_0,
};

static void g_data_input_stream_set_property (GObject      *object,
					      guint         prop_id,
					      const GValue *value,
					      GParamSpec   *pspec);
static void g_data_input_stream_get_property (GObject      *object,
					      guint         prop_id,
					      GValue       *value,
					      GParamSpec   *pspec);

G_DEFINE_TYPE (GDataInputStream,
               g_data_input_stream,
               G_TYPE_BUFFERED_INPUT_STREAM)


static void
g_data_input_stream_class_init (GDataInputStreamClass *klass)
{
  GObjectClass *object_class;

  g_type_class_add_private (klass, sizeof (GDataInputStreamPrivate));

  object_class = G_OBJECT_CLASS (klass);
  object_class->get_property = g_data_input_stream_get_property;
  object_class->set_property = g_data_input_stream_set_property;
}

static void
g_data_input_stream_set_property (GObject         *object,
				  guint            prop_id,
				  const GValue    *value,
				  GParamSpec      *pspec)
{
  GDataInputStreamPrivate *priv;
  GDataInputStream        *dstream;

  dstream = G_DATA_INPUT_STREAM (object);
  priv = dstream->priv;

  switch (prop_id) 
    {

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }

}

static void
g_data_input_stream_get_property (GObject    *object,
                                      guint       prop_id,
                                      GValue     *value,
                                      GParamSpec *pspec)
{
  GDataInputStreamPrivate *priv;
  GDataInputStream        *dstream;

  dstream = G_DATA_INPUT_STREAM (object);
  priv = dstream->priv;

  switch (prop_id)
    { 
  
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }

}
static void
g_data_input_stream_init (GDataInputStream *stream)
{
  stream->priv = G_TYPE_INSTANCE_GET_PRIVATE (stream,
                                              G_TYPE_DATA_INPUT_STREAM,
                                              GDataInputStreamPrivate);

  stream->priv->byte_order = G_DATA_STREAM_BYTE_ORDER_BIG_ENDIAN;
  stream->priv->newline_type = G_DATA_STREAM_NEWLINE_TYPE_LF;
}

/**
 * g_data_input_stream_new:
 * @base_stream:
 * 
 * Returns: 
 **/
GDataInputStream *
g_data_input_stream_new (GInputStream *base_stream)
{
  GDataInputStream *stream;

  g_assert (base_stream != NULL);

  stream = g_object_new (G_TYPE_DATA_INPUT_STREAM,
                         "base-stream", base_stream,
                         NULL);

  return stream;
}

/**
 * g_data_input_stream_set_byte_order:
 * @stream:
 * @order:
 *  
 **/
void
g_data_input_stream_set_byte_order (GDataInputStream *stream,
				    GDataStreamByteOrder order)
{
  stream->priv->byte_order = order;
}

/**
 * g_data_input_stream_get_byte_order:
 * @stream:
 * 
 * Returns: GDataStreamByteOrder. 
 **/
GDataStreamByteOrder
g_data_input_stream_get_byte_order (GDataInputStream *stream)
{
  return stream->priv->byte_order;
}

/**
 * g_data_input_stream_newline_type:
 * @stream:
 * @type:
 * 
 * Returns: 
 **/
void
g_data_input_stream_set_newline_type (GDataInputStream        *stream,
				      GDataStreamNewlineType   type)
{
  stream->priv->newline_type = type;
}

/**
 * g_data_input_stream_get_newline_type:
 * @stream:
 * 
 * Returns: 
 **/
GDataStreamNewlineType
g_data_input_stream_get_newline_type (GDataInputStream *stream)
{
  return stream->priv->newline_type;
}

static gboolean
read_data (GDataInputStream *stream,
	  void *buffer,
	  gsize size,
	  GCancellable *cancellable,
	  GError **error)
{
  gsize availible;
  gssize res;

  while ((availible = g_buffered_input_stream_get_availible (G_BUFFERED_INPUT_STREAM (stream))) < size)
    {
      res = g_buffered_input_stream_fill (G_BUFFERED_INPUT_STREAM (stream),
					  size - availible,
					  cancellable, error);
      if (res < 0)
	return FALSE;
      if (res == 0)
	{
	  g_set_error (error, G_IO_ERROR, G_IO_ERROR_FAILED,
		       _("Unexpected early end-of-stream"));
	  return FALSE;
	}
    }
  
  /* This should always succeed, since its in the buffer */
  res = g_input_stream_read (G_INPUT_STREAM (stream),
			     buffer, size,
			     NULL, NULL);
  g_assert (res == size);
  return TRUE;
}

/**
 * g_data_input_stream_read_byte:
 * @stream:
 * @cancellable:
 * @error:
 * 
 * Returns: 
 **/
guchar
g_data_input_stream_read_byte (GDataInputStream        *stream,
			      GCancellable            *cancellable,
			      GError                 **error)
{
  guchar c;
  
  if (read_data (stream, &c, 1, cancellable, error))
      return c;
  
  return 0;
}

/**
 * g_data_input_stream_read_int16:
 * @stream:
 * @cancellable:
 * @error:
 * 
 * Returns: 
 **/
gint16
g_data_input_stream_read_int16 (GDataInputStream        *stream,
			       GCancellable            *cancellable,
			       GError                 **error)
{
  gint16 v;
  
  if (read_data (stream, &v, 2, cancellable, error))
    {
      switch (stream->priv->byte_order)
	{
	case G_DATA_STREAM_BYTE_ORDER_BIG_ENDIAN:
	  v = GINT16_FROM_BE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_LITTLE_ENDIAN:
	  v = GINT16_FROM_LE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_HOST_ENDIAN:
	default:
	  break;
	}
      return v;
    }
  
  return 0;
}

/**
 * g_data_input_stream_read_uint16:
 * @stream:
 * @cancellable:
 * @error:
 * 
 * Returns: 
 **/
guint16
g_data_input_stream_read_uint16 (GDataInputStream        *stream,
				GCancellable            *cancellable,
				GError                 **error)
{
  guint16 v;
  
  if (read_data (stream, &v, 2, cancellable, error))
    {
      switch (stream->priv->byte_order)
	{
	case G_DATA_STREAM_BYTE_ORDER_BIG_ENDIAN:
	  v = GUINT16_FROM_BE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_LITTLE_ENDIAN:
	  v = GUINT16_FROM_LE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_HOST_ENDIAN:
	default:
	  break;
	}
      return v;
    }
  
  return 0;
}

/**
 * g_data_input_stream_read_int32:
 * @stream:
 * @cancellable:
 * @error:
 * 
 * Returns: 
 **/
gint32
g_data_input_stream_read_int32 (GDataInputStream        *stream,
			       GCancellable            *cancellable,
			       GError                 **error)
{
  gint32 v;
  
  if (read_data (stream, &v, 4, cancellable, error))
    {
      switch (stream->priv->byte_order)
	{
	case G_DATA_STREAM_BYTE_ORDER_BIG_ENDIAN:
	  v = GINT32_FROM_BE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_LITTLE_ENDIAN:
	  v = GINT32_FROM_LE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_HOST_ENDIAN:
	default:
	  break;
	}
      return v;
    }
  
  return 0;
}

/**
 * g_data_input_stream_read_uint32:
 * @stream:
 * @cancellable:
 * @error:
 * 
 * Returns: 
 **/
guint32
g_data_input_stream_read_uint32 (GDataInputStream        *stream,
				GCancellable            *cancellable,
				GError                 **error)
{
  guint32 v;
  
  if (read_data (stream, &v, 4, cancellable, error))
    {
      switch (stream->priv->byte_order)
	{
	case G_DATA_STREAM_BYTE_ORDER_BIG_ENDIAN:
	  v = GUINT32_FROM_BE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_LITTLE_ENDIAN:
	  v = GUINT32_FROM_LE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_HOST_ENDIAN:
	default:
	  break;
	}
      return v;
    }
  
  return 0;
}

/**
 * g_data_input_stream_read_int64:
 * @stream:
 * @cancellable:
 * @error:
 * 
 * Returns: #gint64 read from @stream. 
 **/
gint64
g_data_input_stream_read_int64 (GDataInputStream        *stream,
			       GCancellable            *cancellable,
			       GError                 **error)
{
  gint64 v;
  
  if (read_data (stream, &v, 8, cancellable, error))
    {
      switch (stream->priv->byte_order)
	{
	case G_DATA_STREAM_BYTE_ORDER_BIG_ENDIAN:
	  v = GINT64_FROM_BE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_LITTLE_ENDIAN:
	  v = GINT64_FROM_LE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_HOST_ENDIAN:
	default:
	  break;
	}
      return v;
    }
  
  return 0;
}

/**
 * g_data_input_stream_read_uint64:
 * @stream:
 * @cancellable:
 * @error:
 * 
 * Returns: guint64 read from @stream:
 **/
guint64
g_data_input_stream_read_uint64 (GDataInputStream        *stream,
				GCancellable            *cancellable,
				GError                 **error)
{
  guint64 v;
  
  if (read_data (stream, &v, 8, cancellable, error))
    {
      switch (stream->priv->byte_order)
	{
	case G_DATA_STREAM_BYTE_ORDER_BIG_ENDIAN:
	  v = GUINT64_FROM_BE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_LITTLE_ENDIAN:
	  v = GUINT64_FROM_LE (v);
	  break;
	case G_DATA_STREAM_BYTE_ORDER_HOST_ENDIAN:
	default:
	  break;
	}
      return v;
    }
  
  return 0;
}

static gssize
scan_for_newline (GDataInputStream *stream,
		  gsize *checked_out,
		  gboolean *last_saw_cr_out,
		  int *newline_len_out)
{
  GBufferedInputStream *bstream;
  GDataInputStreamPrivate *priv;
  char buffer[100];
  gsize start, end, peeked;
  int i;
  gssize found_pos;
  int newline_len;
  gsize availible, checked;
  gboolean last_saw_cr;

  priv = stream->priv;
  
  bstream = G_BUFFERED_INPUT_STREAM (stream);
  
  availible = g_buffered_input_stream_get_availible (bstream);

  checked = *checked_out;
  last_saw_cr = *last_saw_cr_out;
  found_pos = -1;
  newline_len = 0;
  
  while (checked < availible)
    {
      start = checked;
      end = MIN (start + sizeof(buffer), availible);
      peeked = g_buffered_input_stream_peek (bstream, buffer, start, end - start);
      end = start + peeked;

      for (i = 0; i < peeked; i++)
	{
	  switch (priv->newline_type)
	    {
	    case G_DATA_STREAM_NEWLINE_TYPE_LF:
	      if (buffer[i] == 10)
		{
		  found_pos = start + i;
		  newline_len = 1;
		}
	      break;
	    case G_DATA_STREAM_NEWLINE_TYPE_CR:
	      if (buffer[i] == 13)
		{
		  found_pos = start + i;
		  newline_len = 1;
		}
	      break;
	    case G_DATA_STREAM_NEWLINE_TYPE_CR_LF:
	      if (last_saw_cr && buffer[i] == 10)
		{
		  found_pos = start + i - 1;
		  newline_len = 2;
		}
	      break;
	    default:
	    case G_DATA_STREAM_NEWLINE_TYPE_ANY:
	      if (buffer[i] == 10) /* LF */
		{
		  if (last_saw_cr)
		    {
		      /* CR LF */
		      found_pos = start + i - 1;
		      newline_len = 2;
		    }
		  else
		    {
		      /* LF */
		      found_pos = start + i;
		      newline_len = 1;
		    }
		}
	      else if (last_saw_cr)
		{
		  /* Last was cr, this is not LF, end is CR */
		  found_pos = start + i - 1;
		  newline_len = 1;
		}
	      /* Don't check for CR here, instead look at last_saw_cr on next byte */
	      break;
	    }
	  
	  last_saw_cr = (buffer[i] == 13);

	  if (found_pos != -1)
	    {
	      *newline_len_out = newline_len;
	      return found_pos;
	    }
	}
      checked = end;
    }

  *checked_out = checked;
  *last_saw_cr_out = last_saw_cr;
  return -1;
}
		  
/**
 * g_data_input_stream_read_line:
 * @stream:
 * @length:
 * @cancellable:
 * @error:
 * 
 * Returns: 
 **/
char *
g_data_input_stream_read_line (GDataInputStream        *stream,
			      gsize                   *length,
			      GCancellable            *cancellable,
			      GError                 **error)
{
  GBufferedInputStream *bstream;
  gsize checked;
  gboolean last_saw_cr;
  gssize found_pos;
  gssize res;
  int newline_len;
  char *line;

  bstream = G_BUFFERED_INPUT_STREAM (stream);

  newline_len = 0;
  checked = 0;
  last_saw_cr = FALSE;

  while ((found_pos = scan_for_newline (stream, &checked, &last_saw_cr, &newline_len)) == -1)
    {
      if (g_buffered_input_stream_get_availible (bstream) ==
	  g_buffered_input_stream_get_buffer_size (bstream))
	g_buffered_input_stream_set_buffer_size (bstream,
						 2 * g_buffered_input_stream_get_buffer_size (bstream));

      res = g_buffered_input_stream_fill (bstream, -1, cancellable, error);
      if (res < 0)
	return NULL;
      if (res == 0)
	{
	  /* End of stream */
	  if (g_buffered_input_stream_get_availible (bstream) == 0)
	    {
	      if (length)
		*length = 0;
	      return NULL;
	    }
	  else
	    {
	      found_pos = checked;
	      newline_len = 0;
	      break;
	    }
	}
    }

  line = g_malloc (found_pos + newline_len + 1);

  res = g_input_stream_read (G_INPUT_STREAM (stream),
			     line,
			     found_pos + newline_len,
			     NULL, NULL);
  if (length)
    *length = (gsize)found_pos;
  g_assert (res == found_pos + newline_len);
  line[found_pos] = 0;
  
  return line;
}

/**
 * g_data_input_stream_read_until:
 * @stream:
 * @stop_char:
 * @length:
 * @cancellable:
 * @error:
 * 
 * Returns: 
 **/
char *
g_data_input_stream_read_until (GDataInputStream        *stream,
			       gchar                    stop_char,
			       gsize                   *length,
			       GCancellable            *cancellable,
			       GError                 **error)
{
  g_assert_not_reached ();
  return NULL;
}
