/*
 * Copyright 2011 Collabora Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * See the included COPYING file for more information.
 */

#undef G_DISABLE_ASSERT
#undef G_LOG_DOMAIN

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "glib.h"

static const gchar *NYAN = "nyannyan";
static const gsize N_NYAN = 8;

static void
test_new (void)
{
  const gchar *data;
  GBytes *bytes;

  data = "test";
  bytes = g_bytes_new (data, 4);
  g_assert (bytes != NULL);
  g_assert (g_bytes_get_data (bytes) != data);
  g_assert_cmpuint (g_bytes_get_size (bytes), ==, 4);
  g_assert (memcmp (data, g_bytes_get_data (bytes), g_bytes_get_size (bytes)) == 0);

  g_bytes_unref (bytes);
}

static void
test_new_take (void)
{
  gchar *data;
  GBytes *bytes;

  data = g_strdup ("test");
  bytes = g_bytes_new_take (data, 4);
  g_assert (bytes != NULL);
  g_assert (g_bytes_get_data (bytes) == data);
  g_assert_cmpuint (g_bytes_get_size (bytes), ==, 4);

  g_bytes_unref (bytes);
}

static void
test_new_static (void)
{
  const gchar *data;
  GBytes *bytes;

  data = "test";
  bytes = g_bytes_new_static (data, 4);
  g_assert (bytes != NULL);
  g_assert (g_bytes_get_data (bytes) == data);
  g_assert_cmpuint (g_bytes_get_size (bytes), ==, 4);

  g_bytes_unref (bytes);
}

static void
test_new_from_bytes (void)
{
  const gchar *data = "smile and wave";
  GBytes *bytes;
  GBytes *sub;

  bytes = g_bytes_new (data, 14);
  sub = g_bytes_new_from_bytes (bytes, 10, 4);

  g_assert (sub != NULL);
  g_assert (g_bytes_get_data (sub) == ((gchar *)g_bytes_get_data (bytes)) + 10);
  g_assert (g_bytes_get_size (sub) == 4);
  g_bytes_unref (bytes);

  g_assert (memcmp (g_bytes_get_data (sub), "wave", 4) == 0);
  g_bytes_unref (sub);
}

static void
on_destroy_increment (gpointer data)
{
  gint *count = data;
  g_assert (count != NULL);
  (*count)++;
}

static void
test_new_with_free_func (void)
{
  GBytes *bytes;
  gchar *data;
  gint count = 0;

  data = "test";
  bytes = g_bytes_new_with_free_func (data, 4, on_destroy_increment, &count);
  g_assert (bytes != NULL);
  g_assert_cmpint (count, ==, 0);
  g_assert (g_bytes_get_data (bytes) == data);
  g_assert_cmpuint (g_bytes_get_size (bytes), ==, 4);

  g_bytes_unref (bytes);
  g_assert_cmpuint (count, ==, 1);
}

static void
test_hash (void)
{
  GBytes *bytes1;
  GBytes *bytes2;
  guint hash1;
  guint hash2;

  bytes1 = g_bytes_new ("blah", 4);
  bytes2 = g_bytes_new ("blah", 4);

  hash1 = g_bytes_hash (bytes1);
  hash2 = g_bytes_hash (bytes2);
  g_assert (hash1 == hash2);

  g_bytes_unref (bytes1);
  g_bytes_unref (bytes2);
}

static void
test_equal (void)
{
  GBytes *bytes;
  GBytes *bytes2;

  bytes = g_bytes_new ("blah", 4);

  bytes2 = g_bytes_new ("blah", 4);
  g_assert (g_bytes_equal (bytes, bytes2));
  g_assert (g_bytes_equal (bytes2, bytes));
  g_bytes_unref (bytes2);

  bytes2 = g_bytes_new ("bla", 3);
  g_assert (!g_bytes_equal (bytes, bytes2));
  g_assert (!g_bytes_equal (bytes2, bytes));
  g_bytes_unref (bytes2);

  bytes2 = g_bytes_new ("true", 4);
  g_assert (!g_bytes_equal (bytes, bytes2));
  g_assert (!g_bytes_equal (bytes2, bytes));
  g_bytes_unref (bytes2);

  g_bytes_unref (bytes);
}

static void
test_compare (void)
{
  GBytes *bytes;
  GBytes *bytes2;

  bytes = g_bytes_new ("blah", 4);

  bytes2 = g_bytes_new ("blah", 4);
  g_assert_cmpint (g_bytes_compare (bytes, bytes2), ==, 0);
  g_bytes_unref (bytes2);

  bytes2 = g_bytes_new ("bla", 3);
  g_assert_cmpint (g_bytes_compare (bytes, bytes2), >, 0);
  g_bytes_unref (bytes2);

  bytes2 = g_bytes_new ("abcd", 4);
  g_assert_cmpint (g_bytes_compare (bytes, bytes2), >, 0);
  g_bytes_unref (bytes2);

  bytes2 = g_bytes_new ("blahblah", 8);
  g_assert_cmpint (g_bytes_compare (bytes, bytes2), <, 0);
  g_bytes_unref (bytes2);

  bytes2 = g_bytes_new ("zyx", 3);
  g_assert_cmpint (g_bytes_compare (bytes, bytes2), <, 0);
  g_bytes_unref (bytes2);

  bytes2 = g_bytes_new ("zyxw", 4);
  g_assert_cmpint (g_bytes_compare (bytes, bytes2), <, 0);
  g_bytes_unref (bytes2);

  g_bytes_unref (bytes);
}

static void
test_to_data_transferred (void)
{
  gconstpointer memory;
  gpointer data;
  gsize size;
  GBytes *bytes;

  /* Memory transferred: one reference, and allocated with g_malloc */
  bytes = g_bytes_new (NYAN, N_NYAN);
  memory = g_bytes_get_data (bytes);
  data = g_bytes_unref_to_data (bytes, &size);
  g_assert (data == memory);
  g_assert_cmpuint (size, ==, N_NYAN);
  g_assert (memcmp (data, NYAN, N_NYAN) == 0);
  g_free (data);
}

static void
test_to_data_two_refs (void)
{
  gconstpointer memory;
  gpointer data;
  gsize size;
  GBytes *bytes;

  /* Memory copied: two references */
  bytes = g_bytes_new (NYAN, N_NYAN);
  bytes = g_bytes_ref (bytes);
  memory = g_bytes_get_data (bytes);
  data = g_bytes_unref_to_data (bytes, &size);
  g_assert (data != memory);
  g_assert_cmpuint (size, ==, N_NYAN);
  g_assert (memcmp (data, NYAN, N_NYAN) == 0);
  g_free (data);
  g_assert (g_bytes_get_data (bytes) == memory);
  g_assert_cmpuint (g_bytes_get_size (bytes), ==, N_NYAN);
  g_bytes_unref (bytes);
}

static void
test_to_data_non_malloc (void)
{
  gpointer data;
  gsize size;
  GBytes *bytes;

  /* Memory copied: non malloc memory */
  bytes = g_bytes_new_static (NYAN, N_NYAN);
  g_assert (g_bytes_get_data (bytes) == NYAN);
  data = g_bytes_unref_to_data (bytes, &size);
  g_assert (data != (gpointer)NYAN);
  g_assert_cmpuint (size, ==, N_NYAN);
  g_assert (memcmp (data, NYAN, N_NYAN) == 0);
  g_free (data);
}

static void
test_to_array_transferred (void)
{
  gconstpointer memory;
  GByteArray *array;
  GBytes *bytes;

  /* Memory transferred: one reference, and allocated with g_malloc */
  bytes = g_bytes_new (NYAN, N_NYAN);
  memory = g_bytes_get_data (bytes);
  array = g_bytes_unref_to_array (bytes);
  g_assert (array != NULL);
  g_assert (array->data == memory);
  g_assert_cmpuint (array->len, ==, N_NYAN);
  g_assert (memcmp (array->data, NYAN, N_NYAN) == 0);
  g_byte_array_unref (array);
}

static void
test_to_array_two_refs (void)
{
  gconstpointer memory;
  GByteArray *array;
  GBytes *bytes;

  /* Memory copied: two references */
  bytes = g_bytes_new (NYAN, N_NYAN);
  bytes = g_bytes_ref (bytes);
  memory = g_bytes_get_data (bytes);
  array = g_bytes_unref_to_array (bytes);
  g_assert (array != NULL);
  g_assert (array->data != memory);
  g_assert_cmpuint (array->len, ==, N_NYAN);
  g_assert (memcmp (array->data, NYAN, N_NYAN) == 0);
  g_byte_array_unref (array);
  g_assert (g_bytes_get_data (bytes) == memory);
  g_assert_cmpuint (g_bytes_get_size (bytes), ==, N_NYAN);
  g_bytes_unref (bytes);
}

static void
test_to_array_non_malloc (void)
{
  GByteArray *array;
  GBytes *bytes;

  /* Memory copied: non malloc memory */
  bytes = g_bytes_new_static (NYAN, N_NYAN);
  g_assert (g_bytes_get_data (bytes) == NYAN);
  array = g_bytes_unref_to_array (bytes);
  g_assert (array != NULL);
  g_assert (array->data != (gpointer)NYAN);
  g_assert_cmpuint (array->len, ==, N_NYAN);
  g_assert (memcmp (array->data, NYAN, N_NYAN) == 0);
  g_byte_array_unref (array);
}

int
main (int argc, char *argv[])
{
  g_test_init (&argc, &argv, NULL);

  g_test_bug_base ("http://bugs.gnome.org/");

  g_test_add_func ("/bytes/new", test_new);
  g_test_add_func ("/bytes/new-take", test_new_take);
  g_test_add_func ("/bytes/new-static", test_new_static);
  g_test_add_func ("/bytes/new-with-free-func", test_new_with_free_func);
  g_test_add_func ("/bytes/new-from-bytes", test_new_from_bytes);
  g_test_add_func ("/bytes/hash", test_hash);
  g_test_add_func ("/bytes/equal", test_equal);
  g_test_add_func ("/bytes/compare", test_compare);
  g_test_add_func ("/bytes/to-data/transfered", test_to_data_transferred);
  g_test_add_func ("/bytes/to-data/two-refs", test_to_data_two_refs);
  g_test_add_func ("/bytes/to-data/non-malloc", test_to_data_non_malloc);
  g_test_add_func ("/bytes/to-array/transfered", test_to_array_transferred);
  g_test_add_func ("/bytes/to-array/two-refs", test_to_array_two_refs);
  g_test_add_func ("/bytes/to-array/non-malloc", test_to_array_non_malloc);

  return g_test_run ();
}
