# -*- Mode: Python -*-

# GDBus - GLib D-Bus Library
#
# Copyright (C) 2008-2011 Red Hat, Inc.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General
# Public License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place, Suite 330,
# Boston, MA 02111-1307, USA.
#
# Author: David Zeuthen <davidz@redhat.com>

import sys

from . import config
from . import utils
from . import dbustypes

# ----------------------------------------------------------------------------------------------------

class CodeGenerator:
    def __init__(self, ifaces, namespace, interface_prefix, generate_objmanager, docbook_gen, h, c):
        self.docbook_gen = docbook_gen
        self.generate_objmanager = generate_objmanager
        self.ifaces = ifaces
        self.h = h
        self.c = c
        self.namespace = namespace
        if len(namespace) > 0:
            if utils.is_ugly_case(namespace):
                self.namespace = namespace.replace('_', '')
                self.ns_upper = namespace.upper() + '_'
                self.ns_lower = namespace.lower() + '_'
            else:
                self.ns_upper = utils.camel_case_to_uscore(namespace).upper() + '_'
                self.ns_lower = utils.camel_case_to_uscore(namespace).lower() + '_'
        else:
            self.ns_upper = ''
            self.ns_lower = ''
        self.interface_prefix = interface_prefix
        self.header_guard = self.h.name.upper().replace('.', '_').replace('-', '_').replace('/', '_')

    # ----------------------------------------------------------------------------------------------------

    def generate_intro(self):
        self.c.write('/*\n'
                     ' * Generated by gdbus-codegen %s. DO NOT EDIT.\n'
                     ' *\n'
                     ' * The license of this code is the same as for the source it was derived from.\n'
                     ' */\n'
                     '\n'
                     %(config.VERSION))
        self.c.write('#ifdef HAVE_CONFIG_H\n'
                     '#  include "config.h"\n'
                     '#endif\n'
                     '\n'
                     '#include "%s"\n'
                     '\n'
                     '#include <string.h>\n'
                     %(self.h.name))

        self.c.write('#ifdef G_OS_UNIX\n'
                     '#  include <gio/gunixfdlist.h>\n'
                     '#endif\n'
                     '\n')

        self.c.write('typedef struct\n'
                     '{\n'
                     '  GDBusArgInfo parent_struct;\n'
                     '  gboolean use_gvariant;\n'
                     '} _ExtendedGDBusArgInfo;\n'
                     '\n')

        self.c.write('typedef struct\n'
                     '{\n'
                     '  GDBusMethodInfo parent_struct;\n'
                     '  const gchar *signal_name;\n'
                     '  gboolean pass_fdlist;\n'
                     '} _ExtendedGDBusMethodInfo;\n'
                     '\n')

        self.c.write('typedef struct\n'
                     '{\n'
                     '  GDBusSignalInfo parent_struct;\n'
                     '  const gchar *signal_name;\n'
                     '} _ExtendedGDBusSignalInfo;\n'
                     '\n')

        self.c.write('typedef struct\n'
                     '{\n'
                     '  GDBusPropertyInfo parent_struct;\n'
                     '  const gchar *hyphen_name;\n'
                     '  gboolean use_gvariant;\n'
                     '} _ExtendedGDBusPropertyInfo;\n'
                     '\n')

        self.c.write('typedef struct\n'
                     '{\n'
                     '  GDBusInterfaceInfo parent_struct;\n'
                     '  const gchar *hyphen_name;\n'
                     '} _ExtendedGDBusInterfaceInfo;\n'
                     '\n')

        self.c.write('typedef struct\n'
                     '{\n'
                     '  const _ExtendedGDBusPropertyInfo *info;\n'
                     '  guint prop_id;\n'
                     '  GValue orig_value; /* the value before the change */\n'
                     '} ChangedProperty;\n'
                     '\n'
                     'static void\n'
                     '_changed_property_free (ChangedProperty *data)\n'
                     '{\n'
                     '  g_value_unset (&data->orig_value);\n'
                     '  g_free (data);\n'
                     '}\n'
                     '\n')

        self.c.write('static gboolean\n'
                     '_g_strv_equal0 (gchar **a, gchar **b)\n'
                     '{\n'
                     '  gboolean ret = FALSE;\n'
                     '  guint n;\n'
                     '  if (a == NULL && b == NULL)\n'
                     '    {\n'
                     '      ret = TRUE;\n'
                     '      goto out;\n'
                     '    }\n'
                     '  if (a == NULL || b == NULL)\n'
                     '    goto out;\n'
                     '  if (g_strv_length (a) != g_strv_length (b))\n'
                     '    goto out;\n'
                     '  for (n = 0; a[n] != NULL; n++)\n'
                     '    if (g_strcmp0 (a[n], b[n]) != 0)\n'
                     '      goto out;\n'
                     '  ret = TRUE;\n'
                     'out:\n'
                     '  return ret;\n'
                     '}\n'
                     '\n')

        self.c.write('static gboolean\n'
                     '_g_variant_equal0 (GVariant *a, GVariant *b)\n'
                     '{\n'
                     '  gboolean ret = FALSE;\n'
                     '  if (a == NULL && b == NULL)\n'
                     '    {\n'
                     '      ret = TRUE;\n'
                     '      goto out;\n'
                     '    }\n'
                     '  if (a == NULL || b == NULL)\n'
                     '    goto out;\n'
                     '  ret = g_variant_equal (a, b);\n'
                     'out:\n'
                     '  return ret;\n'
                     '}\n'
                     '\n')

        # simplified - only supports the types we use
        self.c.write('G_GNUC_UNUSED static gboolean\n'
                     '_g_value_equal (const GValue *a, const GValue *b)\n'
                     '{\n'
                     '  gboolean ret = FALSE;\n'
                     '  g_assert (G_VALUE_TYPE (a) == G_VALUE_TYPE (b));\n'
                     '  switch (G_VALUE_TYPE (a))\n'
                     '    {\n'
                     '      case G_TYPE_BOOLEAN:\n'
                     '        ret = (g_value_get_boolean (a) == g_value_get_boolean (b));\n'
                     '        break;\n'
                     '      case G_TYPE_UCHAR:\n'
                     '        ret = (g_value_get_uchar (a) == g_value_get_uchar (b));\n'
                     '        break;\n'
                     '      case G_TYPE_INT:\n'
                     '        ret = (g_value_get_int (a) == g_value_get_int (b));\n'
                     '        break;\n'
                     '      case G_TYPE_UINT:\n'
                     '        ret = (g_value_get_uint (a) == g_value_get_uint (b));\n'
                     '        break;\n'
                     '      case G_TYPE_INT64:\n'
                     '        ret = (g_value_get_int64 (a) == g_value_get_int64 (b));\n'
                     '        break;\n'
                     '      case G_TYPE_UINT64:\n'
                     '        ret = (g_value_get_uint64 (a) == g_value_get_uint64 (b));\n'
                     '        break;\n'
                     '      case G_TYPE_DOUBLE:\n'
                     '        {\n'
                     '          /* Avoid -Wfloat-equal warnings by doing a direct bit compare */\n'
                     '          gdouble da = g_value_get_double (a);\n'
                     '          gdouble db = g_value_get_double (b);\n'
                     '          ret = memcmp (&da, &db, sizeof (gdouble)) == 0;\n'
                     '        }\n'
                     '        break;\n'
                     '      case G_TYPE_STRING:\n'
                     '        ret = (g_strcmp0 (g_value_get_string (a), g_value_get_string (b)) == 0);\n'
                     '        break;\n'
                     '      case G_TYPE_VARIANT:\n'
                     '        ret = _g_variant_equal0 (g_value_get_variant (a), g_value_get_variant (b));\n'
                     '        break;\n'
                     '      default:\n'
                     '        if (G_VALUE_TYPE (a) == G_TYPE_STRV)\n'
                     '          ret = _g_strv_equal0 (g_value_get_boxed (a), g_value_get_boxed (b));\n'
                     '        else\n'
                     '          g_critical ("_g_value_equal() does not handle type %s", g_type_name (G_VALUE_TYPE (a)));\n'
                     '        break;\n'
                     '    }\n'
                     '  return ret;\n'
                     '}\n'
                     '\n')

        self.h.write('/*\n'
                     ' * Generated by gdbus-codegen %s. DO NOT EDIT.\n'
                     ' *\n'
                     ' * The license of this code is the same as for the source it was derived from.\n'
                     ' */\n'
                     '\n'
                     '#ifndef __%s__\n'
                     '#define __%s__\n'
                     '\n'%(config.VERSION, self.header_guard, self.header_guard))
        self.h.write('#include <gio/gio.h>\n'
                     '\n'
                     'G_BEGIN_DECLS\n'
                     '\n')

    # ----------------------------------------------------------------------------------------------------

    def declare_types(self):
        for i in self.ifaces:
            self.h.write('\n')
            self.h.write('/* ------------------------------------------------------------------------ */\n')
            self.h.write('/* Declarations for %s */\n'%i.name)
            self.h.write('\n')

            # First the GInterface
            self.h.write('#define %sTYPE_%s (%s_get_type ())\n'%(i.ns_upper, i.name_upper, i.name_lower))
            self.h.write('#define %s%s(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), %sTYPE_%s, %s))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper, i.camel_name))
            self.h.write('#define %sIS_%s(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), %sTYPE_%s))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper))
            self.h.write('#define %s%s_GET_IFACE(o) (G_TYPE_INSTANCE_GET_INTERFACE ((o), %sTYPE_%s, %sIface))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper, i.camel_name))
            self.h.write('\n')
            self.h.write('struct _%s;\n'%(i.camel_name))
            self.h.write('typedef struct _%s %s;\n'%(i.camel_name, i.camel_name))
            self.h.write('typedef struct _%sIface %sIface;\n'%(i.camel_name, i.camel_name))
            self.h.write('\n')
            self.h.write('struct _%sIface\n'%(i.camel_name))
            self.h.write('{\n')
            self.h.write('  GTypeInterface parent_iface;\n')

            function_pointers = {}

            # vfuncs for methods
            if len(i.methods) > 0:
                self.h.write('\n')
                for m in i.methods:
                    unix_fd = False
                    if utils.lookup_annotation(m.annotations, 'org.gtk.GDBus.C.UnixFD'):
                        unix_fd = True
                    key = (m.since, '_method_%s'%m.name_lower)
                    value  = '  gboolean (*handle_%s) (\n'%(m.name_lower)
                    value += '    %s *object,\n'%(i.camel_name)
                    value += '    GDBusMethodInvocation *invocation'%()
                    if unix_fd:
                        value += ',\n    GUnixFDList *fd_list'
                    for a in m.in_args:
                        value += ',\n    %sarg_%s'%(a.ctype_in, a.name)
                    value += ');\n\n'
                    function_pointers[key] = value

            # vfuncs for signals
            if len(i.signals) > 0:
                self.h.write('\n')
                for s in i.signals:
                    key = (s.since, '_signal_%s'%s.name_lower)
                    value  = '  void (*%s) (\n'%(s.name_lower)
                    value += '    %s *object'%(i.camel_name)
                    for a in s.args:
                        value += ',\n    %sarg_%s'%(a.ctype_in, a.name)
                    value += ');\n\n'
                    function_pointers[key] = value

            # vfuncs for properties
            if len(i.properties) > 0:
                self.h.write('\n')
                for p in i.properties:
                    key = (p.since, '_prop_get_%s'%p.name_lower)
                    value = '  %s (*get_%s) (%s *object);\n\n'%(p.arg.ctype_in, p.name_lower, i.camel_name)
                    function_pointers[key] = value

            # Sort according to @since tag, then name.. this ensures
            # that the function pointers don't change order assuming
            # judicious use of @since
            #
            # Also use a proper version comparison function so e.g.
            # 10.0 comes after 2.0.
            #
            # See https://bugzilla.gnome.org/show_bug.cgi?id=647577#c5
            # for discussion
            for key in sorted(function_pointers.keys(), key=utils.version_cmp_key):
                self.h.write('%s'%function_pointers[key])

            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('GType %s_get_type (void) G_GNUC_CONST;\n'%(i.name_lower))
            self.h.write('\n')
            self.h.write('GDBusInterfaceInfo *%s_interface_info (void);\n'%(i.name_lower))
            self.h.write('guint %s_override_properties (GObjectClass *klass, guint property_id_begin);\n'%(i.name_lower))
            self.h.write('\n')

            # Then method call completion functions
            if len(i.methods) > 0:
                self.h.write('\n')
                self.h.write('/* D-Bus method call completion functions: */\n')
                for m in i.methods:
                    unix_fd = False
                    if utils.lookup_annotation(m.annotations, 'org.gtk.GDBus.C.UnixFD'):
                        unix_fd = True
                    if m.deprecated:
                        self.h.write('G_GNUC_DEPRECATED ')
                    self.h.write('void %s_complete_%s (\n'
                                 '    %s *object,\n'
                                 '    GDBusMethodInvocation *invocation'%(i.name_lower, m.name_lower, i.camel_name))
                    if unix_fd:
                        self.h.write(',\n    GUnixFDList *fd_list')
                    for a in m.out_args:
                        self.h.write(',\n    %s%s'%(a.ctype_in, a.name))
                    self.h.write(');\n')
                    self.h.write('\n')
                self.h.write('\n')

            # Then signal emission functions
            if len(i.signals) > 0:
                self.h.write('\n')
                self.h.write('/* D-Bus signal emissions functions: */\n')
                for s in i.signals:
                    if s.deprecated:
                        self.h.write('G_GNUC_DEPRECATED ')
                    self.h.write('void %s_emit_%s (\n'
                                 '    %s *object'%(i.name_lower, s.name_lower, i.camel_name))
                    for a in s.args:
                        self.h.write(',\n    %sarg_%s'%(a.ctype_in, a.name))
                    self.h.write(');\n')
                    self.h.write('\n')
                self.h.write('\n')

            # Then method call declarations
            if len(i.methods) > 0:
                self.h.write('\n')
                self.h.write('/* D-Bus method calls: */\n')
                for m in i.methods:
                    unix_fd = False
                    if utils.lookup_annotation(m.annotations, 'org.gtk.GDBus.C.UnixFD'):
                        unix_fd = True
                    # async begin
                    if m.deprecated:
                        self.h.write('G_GNUC_DEPRECATED ')
                    self.h.write('void %s_call_%s (\n'
                                 '    %s *proxy'%(i.name_lower, m.name_lower, i.camel_name))
                    for a in m.in_args:
                        self.h.write(',\n    %sarg_%s'%(a.ctype_in, a.name))
                    if unix_fd:
                        self.h.write(',\n    GUnixFDList *fd_list')
                    self.h.write(',\n'
                                 '    GCancellable *cancellable,\n'
                                 '    GAsyncReadyCallback callback,\n'
                                 '    gpointer user_data);\n')
                    self.h.write('\n')
                    # async finish
                    if m.deprecated:
                        self.h.write('G_GNUC_DEPRECATED ')
                    self.h.write('gboolean %s_call_%s_finish (\n'
                                 '    %s *proxy'%(i.name_lower, m.name_lower, i.camel_name))
                    for a in m.out_args:
                        self.h.write(',\n    %sout_%s'%(a.ctype_out, a.name))
                    if unix_fd:
                        self.h.write(',\n    GUnixFDList **out_fd_list')
                    self.h.write(',\n'
                                 '    GAsyncResult *res,\n'
                                 '    GError **error);\n')
                    self.h.write('\n')
                    # sync
                    if m.deprecated:
                        self.h.write('G_GNUC_DEPRECATED ')
                    self.h.write('gboolean %s_call_%s_sync (\n'
                                 '    %s *proxy'%(i.name_lower, m.name_lower, i.camel_name))
                    for a in m.in_args:
                        self.h.write(',\n    %sarg_%s'%(a.ctype_in, a.name))
                    if unix_fd:
                        self.h.write(',\n    GUnixFDList  *fd_list')
                    for a in m.out_args:
                        self.h.write(',\n    %sout_%s'%(a.ctype_out, a.name))
                    if unix_fd:
                        self.h.write(',\n    GUnixFDList **out_fd_list')
                    self.h.write(',\n'
                                 '    GCancellable *cancellable,\n'
                                 '    GError **error);\n')
                    self.h.write('\n')
                self.h.write('\n')

            # Then the property accessor declarations
            if len(i.properties) > 0:
                self.h.write('\n')
                self.h.write('/* D-Bus property accessors: */\n')
                for p in i.properties:
                    # getter
                    if p.deprecated:
                        self.h.write('G_GNUC_DEPRECATED ')
                    self.h.write('%s%s_get_%s (%s *object);\n'%(p.arg.ctype_in, i.name_lower, p.name_lower, i.camel_name))
                    if p.arg.free_func != None:
                        if p.deprecated:
                            self.h.write('G_GNUC_DEPRECATED ')
                        self.h.write('%s%s_dup_%s (%s *object);\n'%(p.arg.ctype_in_dup, i.name_lower, p.name_lower, i.camel_name))
                    # setter
                    if p.deprecated:
                        self.h.write('G_GNUC_DEPRECATED ')
                    self.h.write('void %s_set_%s (%s *object, %svalue);\n'%(i.name_lower, p.name_lower, i.camel_name, p.arg.ctype_in, ))
                    self.h.write('\n')

            # Then the proxy
            self.h.write('\n')
            self.h.write('/* ---- */\n')
            self.h.write('\n')
            self.h.write('#define %sTYPE_%s_PROXY (%s_proxy_get_type ())\n'%(i.ns_upper, i.name_upper, i.name_lower))
            self.h.write('#define %s%s_PROXY(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), %sTYPE_%s_PROXY, %sProxy))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper, i.camel_name))
            self.h.write('#define %s%s_PROXY_CLASS(k) (G_TYPE_CHECK_CLASS_CAST ((k), %sTYPE_%s_PROXY, %sProxyClass))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper, i.camel_name))
            self.h.write('#define %s%s_PROXY_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), %sTYPE_%s_PROXY, %sProxyClass))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper, i.camel_name))
            self.h.write('#define %sIS_%s_PROXY(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), %sTYPE_%s_PROXY))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper))
            self.h.write('#define %sIS_%s_PROXY_CLASS(k) (G_TYPE_CHECK_CLASS_TYPE ((k), %sTYPE_%s_PROXY))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper))
            self.h.write('\n')
            self.h.write('typedef struct _%sProxy %sProxy;\n'%(i.camel_name, i.camel_name))
            self.h.write('typedef struct _%sProxyClass %sProxyClass;\n'%(i.camel_name, i.camel_name))
            self.h.write('typedef struct _%sProxyPrivate %sProxyPrivate;\n'%(i.camel_name, i.camel_name))
            self.h.write('\n')
            self.h.write('struct _%sProxy\n'%(i.camel_name))
            self.h.write('{\n')
            self.h.write('  /*< private >*/\n')
            self.h.write('  GDBusProxy parent_instance;\n')
            self.h.write('  %sProxyPrivate *priv;\n'%(i.camel_name))
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('struct _%sProxyClass\n'%(i.camel_name))
            self.h.write('{\n')
            self.h.write('  GDBusProxyClass parent_class;\n')
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('GType %s_proxy_get_type (void) G_GNUC_CONST;\n'%(i.name_lower))

            self.h.write('\n')
            if i.deprecated:
                self.h.write('G_GNUC_DEPRECATED ')
            self.h.write('void %s_proxy_new (\n'
                         '    GDBusConnection     *connection,\n'
                         '    GDBusProxyFlags      flags,\n'
                         '    const gchar         *name,\n'
                         '    const gchar         *object_path,\n'
                         '    GCancellable        *cancellable,\n'
                         '    GAsyncReadyCallback  callback,\n'
                         '    gpointer             user_data);\n'
                         %(i.name_lower))
            if i.deprecated:
                self.h.write('G_GNUC_DEPRECATED ')
            self.h.write('%s *%s_proxy_new_finish (\n'
                         '    GAsyncResult        *res,\n'
                         '    GError             **error);\n'
                         %(i.camel_name, i.name_lower))
            if i.deprecated:
                self.h.write('G_GNUC_DEPRECATED ')
            self.h.write('%s *%s_proxy_new_sync (\n'
                         '    GDBusConnection     *connection,\n'
                         '    GDBusProxyFlags      flags,\n'
                         '    const gchar         *name,\n'
                         '    const gchar         *object_path,\n'
                         '    GCancellable        *cancellable,\n'
                         '    GError             **error);\n'
                         %(i.camel_name, i.name_lower))
            self.h.write('\n')
            if i.deprecated:
                self.h.write('G_GNUC_DEPRECATED ')
            self.h.write('void %s_proxy_new_for_bus (\n'
                         '    GBusType             bus_type,\n'
                         '    GDBusProxyFlags      flags,\n'
                         '    const gchar         *name,\n'
                         '    const gchar         *object_path,\n'
                         '    GCancellable        *cancellable,\n'
                         '    GAsyncReadyCallback  callback,\n'
                         '    gpointer             user_data);\n'
                         %(i.name_lower))
            if i.deprecated:
                self.h.write('G_GNUC_DEPRECATED ')
            self.h.write('%s *%s_proxy_new_for_bus_finish (\n'
                         '    GAsyncResult        *res,\n'
                         '    GError             **error);\n'
                         %(i.camel_name, i.name_lower))
            if i.deprecated:
                self.h.write('G_GNUC_DEPRECATED ')
            self.h.write('%s *%s_proxy_new_for_bus_sync (\n'
                         '    GBusType             bus_type,\n'
                         '    GDBusProxyFlags      flags,\n'
                         '    const gchar         *name,\n'
                         '    const gchar         *object_path,\n'
                         '    GCancellable        *cancellable,\n'
                         '    GError             **error);\n'
                         %(i.camel_name, i.name_lower))
            self.h.write('\n')

            # Then the skeleton
            self.h.write('\n')
            self.h.write('/* ---- */\n')
            self.h.write('\n')
            self.h.write('#define %sTYPE_%s_SKELETON (%s_skeleton_get_type ())\n'%(i.ns_upper, i.name_upper, i.name_lower))
            self.h.write('#define %s%s_SKELETON(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), %sTYPE_%s_SKELETON, %sSkeleton))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper, i.camel_name))
            self.h.write('#define %s%s_SKELETON_CLASS(k) (G_TYPE_CHECK_CLASS_CAST ((k), %sTYPE_%s_SKELETON, %sSkeletonClass))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper, i.camel_name))
            self.h.write('#define %s%s_SKELETON_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), %sTYPE_%s_SKELETON, %sSkeletonClass))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper, i.camel_name))
            self.h.write('#define %sIS_%s_SKELETON(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), %sTYPE_%s_SKELETON))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper))
            self.h.write('#define %sIS_%s_SKELETON_CLASS(k) (G_TYPE_CHECK_CLASS_TYPE ((k), %sTYPE_%s_SKELETON))\n'%(i.ns_upper, i.name_upper, i.ns_upper, i.name_upper))
            self.h.write('\n')
            self.h.write('typedef struct _%sSkeleton %sSkeleton;\n'%(i.camel_name, i.camel_name))
            self.h.write('typedef struct _%sSkeletonClass %sSkeletonClass;\n'%(i.camel_name, i.camel_name))
            self.h.write('typedef struct _%sSkeletonPrivate %sSkeletonPrivate;\n'%(i.camel_name, i.camel_name))
            self.h.write('\n')
            self.h.write('struct _%sSkeleton\n'%(i.camel_name))
            self.h.write('{\n')
            self.h.write('  /*< private >*/\n')
            self.h.write('  GDBusInterfaceSkeleton parent_instance;\n')
            self.h.write('  %sSkeletonPrivate *priv;\n'%(i.camel_name))
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('struct _%sSkeletonClass\n'%(i.camel_name))
            self.h.write('{\n')
            self.h.write('  GDBusInterfaceSkeletonClass parent_class;\n')
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('GType %s_skeleton_get_type (void) G_GNUC_CONST;\n'%(i.name_lower))
            self.h.write('\n')
            if i.deprecated:
                self.h.write('G_GNUC_DEPRECATED ')
            self.h.write('%s *%s_skeleton_new (void);\n'%(i.camel_name, i.name_lower))

            self.h.write('\n')

        # Finally, the Object, ObjectProxy, ObjectSkeleton and ObjectManagerClient
        if self.generate_objmanager:
            self.h.write('\n')
            self.h.write('/* ---- */\n')
            self.h.write('\n')
            self.h.write('#define %sTYPE_OBJECT (%sobject_get_type ())\n'%(self.ns_upper, self.ns_lower))
            self.h.write('#define %sOBJECT(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), %sTYPE_OBJECT, %sObject))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sIS_OBJECT(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), %sTYPE_OBJECT))\n'%(self.ns_upper, self.ns_upper))
            self.h.write('#define %sOBJECT_GET_IFACE(o) (G_TYPE_INSTANCE_GET_INTERFACE ((o), %sTYPE_OBJECT, %sObject))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('\n')
            self.h.write('struct _%sObject;\n'%(self.namespace))
            self.h.write('typedef struct _%sObject %sObject;\n'%(self.namespace, self.namespace))
            self.h.write('typedef struct _%sObjectIface %sObjectIface;\n'%(self.namespace, self.namespace))
            self.h.write('\n')
            self.h.write('struct _%sObjectIface\n'%(self.namespace))
            self.h.write('{\n'
                         '  GTypeInterface parent_iface;\n'
                         '};\n'
                         '\n')
            self.h.write('GType %sobject_get_type (void) G_GNUC_CONST;\n'
                         '\n'
                         %(self.ns_lower))
            for i in self.ifaces:
                if i.deprecated:
                    self.h.write('G_GNUC_DEPRECATED ')
                self.h.write ('%s *%sobject_get_%s (%sObject *object);\n'
                              %(i.camel_name, self.ns_lower, i.name_upper.lower(), self.namespace))
            for i in self.ifaces:
                if i.deprecated:
                    self.h.write('G_GNUC_DEPRECATED ')
                self.h.write ('%s *%sobject_peek_%s (%sObject *object);\n'
                              %(i.camel_name, self.ns_lower, i.name_upper.lower(), self.namespace))
            self.h.write('\n')
            self.h.write('#define %sTYPE_OBJECT_PROXY (%sobject_proxy_get_type ())\n'%(self.ns_upper, self.ns_lower))
            self.h.write('#define %sOBJECT_PROXY(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), %sTYPE_OBJECT_PROXY, %sObjectProxy))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sOBJECT_PROXY_CLASS(k) (G_TYPE_CHECK_CLASS_CAST ((k), %sTYPE_OBJECT_PROXY, %sObjectProxyClass))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sOBJECT_PROXY_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), %sTYPE_OBJECT_PROXY, %sObjectProxyClass))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sIS_OBJECT_PROXY(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), %sTYPE_OBJECT_PROXY))\n'%(self.ns_upper, self.ns_upper))
            self.h.write('#define %sIS_OBJECT_PROXY_CLASS(k) (G_TYPE_CHECK_CLASS_TYPE ((k), %sTYPE_OBJECT_PROXY))\n'%(self.ns_upper, self.ns_upper))
            self.h.write('\n')
            self.h.write('typedef struct _%sObjectProxy %sObjectProxy;\n'%(self.namespace, self.namespace))
            self.h.write('typedef struct _%sObjectProxyClass %sObjectProxyClass;\n'%(self.namespace, self.namespace))
            self.h.write('typedef struct _%sObjectProxyPrivate %sObjectProxyPrivate;\n'%(self.namespace, self.namespace))
            self.h.write('\n')
            self.h.write('struct _%sObjectProxy\n'%(self.namespace))
            self.h.write('{\n')
            self.h.write('  /*< private >*/\n')
            self.h.write('  GDBusObjectProxy parent_instance;\n')
            self.h.write('  %sObjectProxyPrivate *priv;\n'%(self.namespace))
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('struct _%sObjectProxyClass\n'%(self.namespace))
            self.h.write('{\n')
            self.h.write('  GDBusObjectProxyClass parent_class;\n')
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('GType %sobject_proxy_get_type (void) G_GNUC_CONST;\n'%(self.ns_lower))
            self.h.write('%sObjectProxy *%sobject_proxy_new (GDBusConnection *connection, const gchar *object_path);\n'%(self.namespace, self.ns_lower))
            self.h.write('\n')
            self.h.write('#define %sTYPE_OBJECT_SKELETON (%sobject_skeleton_get_type ())\n'%(self.ns_upper, self.ns_lower))
            self.h.write('#define %sOBJECT_SKELETON(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), %sTYPE_OBJECT_SKELETON, %sObjectSkeleton))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sOBJECT_SKELETON_CLASS(k) (G_TYPE_CHECK_CLASS_CAST ((k), %sTYPE_OBJECT_SKELETON, %sObjectSkeletonClass))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sOBJECT_SKELETON_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), %sTYPE_OBJECT_SKELETON, %sObjectSkeletonClass))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sIS_OBJECT_SKELETON(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), %sTYPE_OBJECT_SKELETON))\n'%(self.ns_upper, self.ns_upper))
            self.h.write('#define %sIS_OBJECT_SKELETON_CLASS(k) (G_TYPE_CHECK_CLASS_TYPE ((k), %sTYPE_OBJECT_SKELETON))\n'%(self.ns_upper, self.ns_upper))
            self.h.write('\n')
            self.h.write('typedef struct _%sObjectSkeleton %sObjectSkeleton;\n'%(self.namespace, self.namespace))
            self.h.write('typedef struct _%sObjectSkeletonClass %sObjectSkeletonClass;\n'%(self.namespace, self.namespace))
            self.h.write('typedef struct _%sObjectSkeletonPrivate %sObjectSkeletonPrivate;\n'%(self.namespace, self.namespace))
            self.h.write('\n')
            self.h.write('struct _%sObjectSkeleton\n'%(self.namespace))
            self.h.write('{\n')
            self.h.write('  /*< private >*/\n')
            self.h.write('  GDBusObjectSkeleton parent_instance;\n')
            self.h.write('  %sObjectSkeletonPrivate *priv;\n'%(self.namespace))
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('struct _%sObjectSkeletonClass\n'%(self.namespace))
            self.h.write('{\n')
            self.h.write('  GDBusObjectSkeletonClass parent_class;\n')
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('GType %sobject_skeleton_get_type (void) G_GNUC_CONST;\n'%(self.ns_lower))
            self.h.write('%sObjectSkeleton *%sobject_skeleton_new (const gchar *object_path);\n'
                         %(self.namespace, self.ns_lower))
            for i in self.ifaces:
                if i.deprecated:
                    self.h.write('G_GNUC_DEPRECATED ')
                self.h.write ('void %sobject_skeleton_set_%s (%sObjectSkeleton *object, %s *interface_);\n'
                              %(self.ns_lower, i.name_upper.lower(), self.namespace, i.camel_name))
            self.h.write('\n')

            self.h.write('/* ---- */\n')
            self.h.write('\n')
            self.h.write('#define %sTYPE_OBJECT_MANAGER_CLIENT (%sobject_manager_client_get_type ())\n'%(self.ns_upper, self.ns_lower))
            self.h.write('#define %sOBJECT_MANAGER_CLIENT(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), %sTYPE_OBJECT_MANAGER_CLIENT, %sObjectManagerClient))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sOBJECT_MANAGER_CLIENT_CLASS(k) (G_TYPE_CHECK_CLASS_CAST ((k), %sTYPE_OBJECT_MANAGER_CLIENT, %sObjectManagerClientClass))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sOBJECT_MANAGER_CLIENT_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), %sTYPE_OBJECT_MANAGER_CLIENT, %sObjectManagerClientClass))\n'%(self.ns_upper, self.ns_upper, self.namespace))
            self.h.write('#define %sIS_OBJECT_MANAGER_CLIENT(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), %sTYPE_OBJECT_MANAGER_CLIENT))\n'%(self.ns_upper, self.ns_upper))
            self.h.write('#define %sIS_OBJECT_MANAGER_CLIENT_CLASS(k) (G_TYPE_CHECK_CLASS_TYPE ((k), %sTYPE_OBJECT_MANAGER_CLIENT))\n'%(self.ns_upper, self.ns_upper))
            self.h.write('\n')
            self.h.write('typedef struct _%sObjectManagerClient %sObjectManagerClient;\n'%(self.namespace, self.namespace))
            self.h.write('typedef struct _%sObjectManagerClientClass %sObjectManagerClientClass;\n'%(self.namespace, self.namespace))
            self.h.write('typedef struct _%sObjectManagerClientPrivate %sObjectManagerClientPrivate;\n'%(self.namespace, self.namespace))
            self.h.write('\n')
            self.h.write('struct _%sObjectManagerClient\n'%(self.namespace))
            self.h.write('{\n')
            self.h.write('  /*< private >*/\n')
            self.h.write('  GDBusObjectManagerClient parent_instance;\n')
            self.h.write('  %sObjectManagerClientPrivate *priv;\n'%(self.namespace))
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('struct _%sObjectManagerClientClass\n'%(self.namespace))
            self.h.write('{\n')
            self.h.write('  GDBusObjectManagerClientClass parent_class;\n')
            self.h.write('};\n')
            self.h.write('\n')
            self.h.write('GType %sobject_manager_client_get_type (void) G_GNUC_CONST;\n'%(self.ns_lower))
            self.h.write('\n')
            self.h.write('GType %sobject_manager_client_get_proxy_type (GDBusObjectManagerClient *manager, const gchar *object_path, const gchar *interface_name, gpointer user_data);\n'%(self.ns_lower))
            self.h.write('\n')
            self.h.write('void %sobject_manager_client_new (\n'
                         '    GDBusConnection        *connection,\n'
                         '    GDBusObjectManagerClientFlags  flags,\n'
                         '    const gchar            *name,\n'
                         '    const gchar            *object_path,\n'
                         '    GCancellable           *cancellable,\n'
                         '    GAsyncReadyCallback     callback,\n'
                         '    gpointer                user_data);\n'
                         %(self.ns_lower))
            self.h.write('GDBusObjectManager *%sobject_manager_client_new_finish (\n'
                         '    GAsyncResult        *res,\n'
                         '    GError             **error);\n'
                         %(self.ns_lower))
            self.h.write('GDBusObjectManager *%sobject_manager_client_new_sync (\n'
                         '    GDBusConnection        *connection,\n'
                         '    GDBusObjectManagerClientFlags  flags,\n'
                         '    const gchar            *name,\n'
                         '    const gchar            *object_path,\n'
                         '    GCancellable           *cancellable,\n'
                         '    GError                **error);\n'
                         %(self.ns_lower))
            self.h.write('\n')
            self.h.write('void %sobject_manager_client_new_for_bus (\n'
                         '    GBusType                bus_type,\n'
                         '    GDBusObjectManagerClientFlags  flags,\n'
                         '    const gchar            *name,\n'
                         '    const gchar            *object_path,\n'
                         '    GCancellable           *cancellable,\n'
                         '    GAsyncReadyCallback     callback,\n'
                         '    gpointer                user_data);\n'
                         %(self.ns_lower))
            self.h.write('GDBusObjectManager *%sobject_manager_client_new_for_bus_finish (\n'
                         '    GAsyncResult        *res,\n'
                         '    GError             **error);\n'
                         %(self.ns_lower))
            self.h.write('GDBusObjectManager *%sobject_manager_client_new_for_bus_sync (\n'
                         '    GBusType                bus_type,\n'
                         '    GDBusObjectManagerClientFlags  flags,\n'
                         '    const gchar            *name,\n'
                         '    const gchar            *object_path,\n'
                         '    GCancellable           *cancellable,\n'
                         '    GError                **error);\n'
                         %(self.ns_lower))
            self.h.write('\n')

    # ----------------------------------------------------------------------------------------------------

    def generate_outro(self):
        self.h.write('\n'
                     'G_END_DECLS\n'
                     '\n'
                     '#endif /* __%s__ */\n'%(self.header_guard))

    # ----------------------------------------------------------------------------------------------------

    def generate_annotations(self, prefix, annotations):
        if annotations == None:
            return

        n = 0
        for a in annotations:
            #self.generate_annotations('%s_%d'%(prefix, n), a.get_annotations())

            # skip internal annotations
            if a.key.startswith('org.gtk.GDBus'):
                continue

            self.c.write('static const GDBusAnnotationInfo %s_%d =\n'
                         '{\n'
                         '  -1,\n'
                         '  (gchar *) "%s",\n'
                         '  (gchar *) "%s",\n'%(prefix, n, a.key, a.value))
            if len(a.annotations) == 0:
                self.c.write('  NULL\n')
            else:
                self.c.write('  (GDBusAnnotationInfo **) &%s_%d_pointers\n'%(prefix, n))
            self.c.write('};\n'
                         '\n')
            n += 1

        if n > 0:
            self.c.write('static const GDBusAnnotationInfo * const %s_pointers[] =\n'
                             '{\n'%(prefix))
            m = 0;
            for a in annotations:
                if a.key.startswith('org.gtk.GDBus'):
                    continue
                self.c.write('  &%s_%d,\n'%(prefix, m))
                m += 1
            self.c.write('  NULL\n'
                         '};\n'
                         '\n')
        return n

    def generate_args(self, prefix, args):
        for a in args:
            num_anno = self.generate_annotations('%s_arg_%s_annotation_info'%(prefix, a.name), a.annotations)

            self.c.write('static const _ExtendedGDBusArgInfo %s_%s =\n'
                         '{\n'
                         '  {\n'
                         '    -1,\n'
                         '    (gchar *) "%s",\n'
                         '    (gchar *) "%s",\n'%(prefix, a.name, a.name, a.signature))
            if num_anno == 0:
                self.c.write('    NULL\n')
            else:
                self.c.write('    (GDBusAnnotationInfo **) &%s_arg_%s_annotation_info_pointers\n'%(prefix, a.name))
            self.c.write('  },\n')
            if not utils.lookup_annotation(a.annotations, 'org.gtk.GDBus.C.ForceGVariant'):
                self.c.write('  FALSE\n')
            else:
                self.c.write('  TRUE\n')
            self.c.write('};\n'
                         '\n')

        if len(args) > 0:
            self.c.write('static const _ExtendedGDBusArgInfo * const %s_pointers[] =\n'
                             '{\n'%(prefix))
            for a in args:
                self.c.write('  &%s_%s,\n'%(prefix, a.name))
            self.c.write('  NULL\n'
                         '};\n'
                         '\n')

    def generate_introspection_for_interface(self, i):
            self.c.write('/* ---- Introspection data for %s ---- */\n'
                         '\n'%(i.name))

            if len(i.methods) > 0:
                for m in i.methods:
                    unix_fd = False
                    if utils.lookup_annotation(m.annotations, 'org.gtk.GDBus.C.UnixFD'):
                        unix_fd = True
                    self.generate_args('_%s_method_info_%s_IN_ARG'%(i.name_lower, m.name_lower), m.in_args)
                    self.generate_args('_%s_method_info_%s_OUT_ARG'%(i.name_lower, m.name_lower), m.out_args)

                    num_anno = self.generate_annotations('_%s_method_%s_annotation_info'%(i.name_lower, m.name_lower), m.annotations)

                    self.c.write('static const _ExtendedGDBusMethodInfo _%s_method_info_%s =\n'
                                 '{\n'
                                 '  {\n'
                                 '    -1,\n'
                                 '    (gchar *) "%s",\n'%(i.name_lower, m.name_lower, m.name))
                    if len(m.in_args) == 0:
                        self.c.write('    NULL,\n')
                    else:
                        self.c.write('    (GDBusArgInfo **) &_%s_method_info_%s_IN_ARG_pointers,\n'%(i.name_lower, m.name_lower))
                    if len(m.out_args) == 0:
                        self.c.write('    NULL,\n')
                    else:
                        self.c.write('    (GDBusArgInfo **) &_%s_method_info_%s_OUT_ARG_pointers,\n'%(i.name_lower, m.name_lower))
                    if num_anno == 0:
                        self.c.write('    NULL\n')
                    else:
                        self.c.write('    (GDBusAnnotationInfo **) &_%s_method_%s_annotation_info_pointers\n'%(i.name_lower, m.name_lower))
                    self.c.write('  },\n'
                                 '  "handle-%s",\n'
                                 '  %s\n'
                                 %(m.name_hyphen, 'TRUE' if unix_fd else 'FALSE'))
                    self.c.write('};\n'
                                 '\n')

                self.c.write('static const _ExtendedGDBusMethodInfo * const _%s_method_info_pointers[] =\n'
                             '{\n'%(i.name_lower))
                for m in i.methods:
                    self.c.write('  &_%s_method_info_%s,\n'%(i.name_lower, m.name_lower))
                self.c.write('  NULL\n'
                             '};\n'
                             '\n')

            # ---

            if len(i.signals) > 0:
                for s in i.signals:
                    self.generate_args('_%s_signal_info_%s_ARG'%(i.name_lower, s.name_lower), s.args)

                    num_anno = self.generate_annotations('_%s_signal_%s_annotation_info'%(i.name_lower, s.name_lower), s.annotations)
                    self.c.write('static const _ExtendedGDBusSignalInfo _%s_signal_info_%s =\n'
                                 '{\n'
                                 '  {\n'
                                 '    -1,\n'
                                 '    (gchar *) "%s",\n'%(i.name_lower, s.name_lower, s.name))
                    if len(s.args) == 0:
                        self.c.write('    NULL,\n')
                    else:
                        self.c.write('    (GDBusArgInfo **) &_%s_signal_info_%s_ARG_pointers,\n'%(i.name_lower, s.name_lower))
                    if num_anno == 0:
                        self.c.write('    NULL\n')
                    else:
                        self.c.write('    (GDBusAnnotationInfo **) &_%s_signal_%s_annotation_info_pointers\n'%(i.name_lower, s.name_lower))
                    self.c.write('  },\n'
                                 '  "%s"\n'
                                 %(s.name_hyphen))
                    self.c.write('};\n'
                                 '\n')

                self.c.write('static const _ExtendedGDBusSignalInfo * const _%s_signal_info_pointers[] =\n'
                             '{\n'%(i.name_lower))
                for s in i.signals:
                    self.c.write('  &_%s_signal_info_%s,\n'%(i.name_lower, s.name_lower))
                self.c.write('  NULL\n'
                             '};\n'
                             '\n')

            # ---

            if len(i.properties) > 0:
                for p in i.properties:
                    if p.readable and p.writable:
                        access = 'G_DBUS_PROPERTY_INFO_FLAGS_READABLE | G_DBUS_PROPERTY_INFO_FLAGS_WRITABLE'
                    elif p.readable:
                        access = 'G_DBUS_PROPERTY_INFO_FLAGS_READABLE'
                    elif p.writable:
                        access = 'G_DBUS_PROPERTY_INFO_FLAGS_WRITABLE'
                    else:
                        access = 'G_DBUS_PROPERTY_INFO_FLAGS_NONE'
                    num_anno = self.generate_annotations('_%s_property_%s_annotation_info'%(i.name_lower, p.name_lower), p.annotations)
                    self.c.write('static const _ExtendedGDBusPropertyInfo _%s_property_info_%s =\n'
                                 '{\n'
                                 '  {\n'
                                 '    -1,\n'
                                 '    (gchar *) "%s",\n'
                                 '    (gchar *) "%s",\n'
                                 '    %s,\n'%(i.name_lower, p.name_lower, p.name, p.arg.signature, access))
                    if num_anno == 0:
                        self.c.write('    NULL\n')
                    else:
                        self.c.write('    (GDBusAnnotationInfo **) &_%s_property_%s_annotation_info_pointers\n'%(i.name_lower, p.name_lower))
                    self.c.write('  },\n'
                                 '  "%s",\n'
                                 %(p.name_hyphen))
                    if not utils.lookup_annotation(p.annotations, 'org.gtk.GDBus.C.ForceGVariant'):
                        self.c.write('  FALSE\n')
                    else:
                        self.c.write('  TRUE\n')
                    self.c.write('};\n'
                                 '\n')

                self.c.write('static const _ExtendedGDBusPropertyInfo * const _%s_property_info_pointers[] =\n'
                             '{\n'%(i.name_lower))
                for p in i.properties:
                    self.c.write('  &_%s_property_info_%s,\n'%(i.name_lower, p.name_lower))
                self.c.write('  NULL\n'
                             '};\n'
                             '\n')

            num_anno = self.generate_annotations('_%s_annotation_info'%(i.name_lower), i.annotations)
            self.c.write('static const _ExtendedGDBusInterfaceInfo _%s_interface_info =\n'
                         '{\n'
                         '  {\n'
                         '    -1,\n'
                         '    (gchar *) "%s",\n'%(i.name_lower, i.name))
            if len(i.methods) == 0:
                self.c.write('    NULL,\n')
            else:
                self.c.write('    (GDBusMethodInfo **) &_%s_method_info_pointers,\n'%(i.name_lower))
            if len(i.signals) == 0:
                self.c.write('    NULL,\n')
            else:
                self.c.write('    (GDBusSignalInfo **) &_%s_signal_info_pointers,\n'%(i.name_lower))
            if len(i.properties) == 0:
                self.c.write('    NULL,\n')
            else:
                self.c.write('    (GDBusPropertyInfo **) &_%s_property_info_pointers,\n'%(i.name_lower))
            if num_anno == 0:
                self.c.write('    NULL\n')
            else:
                self.c.write('    (GDBusAnnotationInfo **) &_%s_annotation_info_pointers\n'%(i.name_lower))
            self.c.write('  },\n'
                         '  "%s",\n'
                         '};\n'
                         '\n'
                         %(i.name_hyphen))
            self.c.write('\n')
            self.c.write(self.docbook_gen.expand(
                    '/**\n'
                    ' * %s_interface_info:\n'
                    ' *\n'
                    ' * Gets a machine-readable description of the #%s D-Bus interface.\n'
                    ' *\n'
                    ' * Returns: (transfer none): A #GDBusInterfaceInfo. Do not free.\n'
                    %(i.name_lower, i.name), False))
            self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
            self.c.write('GDBusInterfaceInfo *\n'
                         '%s_interface_info (void)\n'
                         '{\n'
                         '  return (GDBusInterfaceInfo *) &_%s_interface_info.parent_struct;\n'
                         '}\n'
                         '\n'%(i.name_lower, i.name_lower))

            self.c.write(self.docbook_gen.expand(
                    '/**\n'
                    ' * %s_override_properties:\n'
                    ' * @klass: The class structure for a #GObject<!-- -->-derived class.\n'
                    ' * @property_id_begin: The property id to assign to the first overridden property.\n'
                    ' *\n'
                    ' * Overrides all #GObject properties in the #%s interface for a concrete class.\n'
                    ' * The properties are overridden in the order they are defined.\n'
                    ' *\n'
                    ' * Returns: The last property id.\n'
                    %(i.name_lower, i.camel_name), False))
            self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
            self.c.write('guint\n'
                         '%s_override_properties (GObjectClass *klass, guint property_id_begin)\n'
                         '{\n'%(i.name_lower))
            for p in i.properties:
                self.c.write ('  g_object_class_override_property (klass, property_id_begin++, "%s");\n'%(p.name_hyphen))
            self.c.write('  return property_id_begin - 1;\n'
                         '}\n'
                         '\n')
            self.c.write('\n')

    # ----------------------------------------------------------------------------------------------------

    def generate_interface(self, i):
        self.c.write('\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %s:\n'
                ' *\n'
                ' * Abstract interface type for the D-Bus interface #%s.\n'
                %(i.camel_name, i.name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sIface:\n'
                ' * @parent_iface: The parent interface.\n'
                %(i.camel_name), False))

        doc_bits = {}
        if len(i.methods) > 0:
            for m in i.methods:
                key = (m.since, '_method_%s'%m.name_lower)
                value  = '@handle_%s: '%(m.name_lower)
                value += 'Handler for the #%s::handle-%s signal.'%(i.camel_name, m.name_hyphen)
                doc_bits[key] = value
        if len(i.signals) > 0:
            for s in i.signals:
                key = (s.since, '_signal_%s'%s.name_lower)
                value  = '@%s: '%(s.name_lower)
                value += 'Handler for the #%s::%s signal.'%(i.camel_name, s.name_hyphen)
                doc_bits[key] = value
        if len(i.properties) > 0:
            for p in i.properties:
                key = (p.since, '_prop_get_%s'%p.name_lower)
                value  = '@get_%s: '%(p.name_lower)
                value += 'Getter for the #%s:%s property.'%(i.camel_name, p.name_hyphen)
                doc_bits[key] = value
        for key in sorted(doc_bits.keys(), key=utils.version_cmp_key):
            self.c.write(' * %s\n'%doc_bits[key])

        self.c.write(self.docbook_gen.expand(
                ' *\n'
                ' * Virtual table for the D-Bus interface #%s.\n'
                %(i.name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('\n')

        self.c.write('typedef %sIface %sInterface;\n'%(i.camel_name, i.camel_name))
        self.c.write('G_DEFINE_INTERFACE (%s, %s, G_TYPE_OBJECT);\n'%(i.camel_name, i.name_lower))
        self.c.write('\n')

        self.c.write('static void\n'
                     '%s_default_init (%sIface *iface)\n'
                     '{\n'%(i.name_lower, i.camel_name));

        if len(i.methods) > 0:
            self.c.write('  /* GObject signals for incoming D-Bus method calls: */\n')
            for m in i.methods:
                unix_fd = False
                if utils.lookup_annotation(m.annotations, 'org.gtk.GDBus.C.UnixFD'):
                    unix_fd = True
                self.c.write(self.docbook_gen.expand(
                        '  /**\n'
                        '   * %s::handle-%s:\n'
                        '   * @object: A #%s.\n'
                        '   * @invocation: A #GDBusMethodInvocation.\n'
                        %(i.camel_name, m.name_hyphen, i.camel_name), False))
                if unix_fd:
                    self.c.write ('   * @fd_list: (allow-none): A #GUnixFDList or %NULL.\n')
                for a in m.in_args:
                    self.c.write ('   * @arg_%s: Argument passed by remote caller.\n'%(a.name))
                self.c.write(self.docbook_gen.expand(
                        '   *\n'
                        '   * Signal emitted when a remote caller is invoking the %s.%s() D-Bus method.\n'
                        '   *\n'
                        '   * If a signal handler returns %%TRUE, it means the signal handler will handle the invocation (e.g. take a reference to @invocation and eventually call %s_complete_%s() or e.g. g_dbus_method_invocation_return_error() on it) and no order signal handlers will run. If no signal handler handles the invocation, the %%G_DBUS_ERROR_UNKNOWN_METHOD error is returned.\n'
                        '   *\n'
                        '   * Returns: %%TRUE if the invocation was handled, %%FALSE to let other signal handlers run.\n'
                        %(i.name, m.name, i.name_lower, m.name_lower), False))
                self.write_gtkdoc_deprecated_and_since_and_close(m, self.c, 2)
                if unix_fd:
                    extra_args = 2
                else:
                    extra_args = 1
                self.c.write('  g_signal_new ("handle-%s",\n'
                             '    G_TYPE_FROM_INTERFACE (iface),\n'
                             '    G_SIGNAL_RUN_LAST,\n'
                             '    G_STRUCT_OFFSET (%sIface, handle_%s),\n'
                             '    g_signal_accumulator_true_handled,\n'
                             '    NULL,\n' # accu_data
                             '    g_cclosure_marshal_generic,\n'
                             '    G_TYPE_BOOLEAN,\n'
                             '    %d,\n'
                             '    G_TYPE_DBUS_METHOD_INVOCATION'
                             %(m.name_hyphen, i.camel_name, m.name_lower, len(m.in_args) + extra_args))
                if unix_fd:
                    self.c.write(', G_TYPE_UNIX_FD_LIST')
                for a in m.in_args:
                    self.c.write (', %s'%(a.gtype))
                self.c.write(');\n')
                self.c.write('\n')

        if len(i.signals) > 0:
            self.c.write('  /* GObject signals for received D-Bus signals: */\n')
            for s in i.signals:
                self.c.write(self.docbook_gen.expand(
                        '  /**\n'
                        '   * %s::%s:\n'
                        '   * @object: A #%s.\n'
                        %(i.camel_name, s.name_hyphen, i.camel_name), False))
                for a in s.args:
                    self.c.write ('   * @arg_%s: Argument.\n'%(a.name))
                self.c.write(self.docbook_gen.expand(
                        '   *\n'
                        '   * On the client-side, this signal is emitted whenever the D-Bus signal #%s::%s is received.\n'
                        '   *\n'
                        '   * On the service-side, this signal can be used with e.g. g_signal_emit_by_name() to make the object emit the D-Bus signal.\n'
                        %(i.name, s.name), False))
                self.write_gtkdoc_deprecated_and_since_and_close(s, self.c, 2)
                self.c.write('  g_signal_new ("%s",\n'
                             '    G_TYPE_FROM_INTERFACE (iface),\n'
                             '    G_SIGNAL_RUN_LAST,\n'
                             '    G_STRUCT_OFFSET (%sIface, %s),\n'
                             '    NULL,\n' # accumulator
                             '    NULL,\n' # accu_data
                             '    g_cclosure_marshal_generic,\n'
                             '    G_TYPE_NONE,\n'
                             '    %d'
                             %(s.name_hyphen, i.camel_name, s.name_lower, len(s.args)))
                for a in s.args:
                    self.c.write (', %s'%(a.gtype))
                self.c.write(');\n')
                self.c.write('\n')

        if len(i.properties) > 0:
            self.c.write('  /* GObject properties for D-Bus properties: */\n')
            for p in i.properties:
                if p.readable and p.writable:
                    hint = 'Since the D-Bus property for this #GObject property is both readable and writable, it is meaningful to both read from it and write to it on both the service- and client-side.'
                elif p.readable:
                    hint = 'Since the D-Bus property for this #GObject property is readable but not writable, it is meaningful to read from it on both the client- and service-side. It is only meaningful, however, to write to it on the service-side.'
                elif p.writable:
                    hint = 'Since the D-Bus property for this #GObject property is writable but not readable, it is meaningful to write to it on both the client- and service-side. It is only meaningful, however, to read from it on the service-side.'
                else:
                    raise RuntimeError('Cannot handle property %s that neither readable nor writable'%(p.name))
                self.c.write(self.docbook_gen.expand(
                        '  /**\n'
                        '   * %s:%s:\n'
                        '   *\n'
                        '   * Represents the D-Bus property #%s:%s.\n'
                        '   *\n'
                        '   * %s\n'
                        %(i.camel_name, p.name_hyphen, i.name, p.name, hint), False))
                self.write_gtkdoc_deprecated_and_since_and_close(p, self.c, 2)
                self.c.write('  g_object_interface_install_property (iface,\n')
                if p.arg.gtype == 'G_TYPE_VARIANT':
                    s = 'g_param_spec_variant ("%s", "%s", "%s", G_VARIANT_TYPE ("%s"), NULL'%(p.name_hyphen, p.name, p.name, p.arg.signature)
                elif p.arg.signature == 'b':
                    s = 'g_param_spec_boolean ("%s", "%s", "%s", FALSE'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'y':
                    s = 'g_param_spec_uchar ("%s", "%s", "%s", 0, 255, 0'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'n':
                    s = 'g_param_spec_int ("%s", "%s", "%s", G_MININT16, G_MAXINT16, 0'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'q':
                    s = 'g_param_spec_uint ("%s", "%s", "%s", 0, G_MAXUINT16, 0'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'i':
                    s = 'g_param_spec_int ("%s", "%s", "%s", G_MININT32, G_MAXINT32, 0'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'u':
                    s = 'g_param_spec_uint ("%s", "%s", "%s", 0, G_MAXUINT32, 0'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'x':
                    s = 'g_param_spec_int64 ("%s", "%s", "%s", G_MININT64, G_MAXINT64, 0'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 't':
                    s = 'g_param_spec_uint64 ("%s", "%s", "%s", 0, G_MAXUINT64, 0'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'd':
                    s = 'g_param_spec_double ("%s", "%s", "%s", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 's':
                    s = 'g_param_spec_string ("%s", "%s", "%s", NULL'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'o':
                    s = 'g_param_spec_string ("%s", "%s", "%s", NULL'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'g':
                    s = 'g_param_spec_string ("%s", "%s", "%s", NULL'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'ay':
                    s = 'g_param_spec_string ("%s", "%s", "%s", NULL'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'as':
                    s = 'g_param_spec_boxed ("%s", "%s", "%s", G_TYPE_STRV'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'ao':
                    s = 'g_param_spec_boxed ("%s", "%s", "%s", G_TYPE_STRV'%(p.name_hyphen, p.name, p.name)
                elif p.arg.signature == 'aay':
                    s = 'g_param_spec_boxed ("%s", "%s", "%s", G_TYPE_STRV'%(p.name_hyphen, p.name, p.name)
                else:
                    raise RuntimeError('Unsupported gtype %s for GParamSpec'%(p.arg.gtype))
                self.c.write('    %s, G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));'%s);
                self.c.write('\n')

        self.c.write('}\n'
                     '\n')

    # ----------------------------------------------------------------------------------------------------

    def generate_property_accessors(self, i):
        for p in i.properties:
            # getter
            if p.readable and p.writable:
                hint = 'Since this D-Bus property is both readable and writable, it is meaningful to use this function on both the client- and service-side.'
            elif p.readable:
                hint = 'Since this D-Bus property is readable, it is meaningful to use this function on both the client- and service-side.'
            elif p.writable:
                hint = 'Since this D-Bus property is not readable, it is only meaningful to use this function on the service-side.'
            else:
                raise RuntimeError('Cannot handle property %s that neither readable nor writable'%(p.name))
            self.c.write(self.docbook_gen.expand(
                    '/**\n'
                    ' * %s_get_%s: (skip)\n'
                    ' * @object: A #%s.\n'
                    ' *\n'
                    ' * Gets the value of the #%s:%s D-Bus property.\n'
                    ' *\n'
                    ' * %s\n'
                    ' *\n'
                    %(i.name_lower, p.name_lower, i.camel_name, i.name, p.name, hint), False))
            if p.arg.free_func != None:
                self.c.write(' * <warning>The returned value is only valid until the property changes so on the client-side it is only safe to use this function on the thread where @object was constructed. Use %s_dup_%s() if on another thread.</warning>\n'
                             ' *\n'
                             ' * Returns: (transfer none): The property value or %%NULL if the property is not set. Do not free the returned value, it belongs to @object.\n'
                             %(i.name_lower, p.name_lower))
            else:
                self.c.write(' * Returns: The property value.\n')
            self.write_gtkdoc_deprecated_and_since_and_close(p, self.c, 0)
            self.c.write('%s\n'
                         '%s_get_%s (%s *object)\n'
                         '{\n'%(p.arg.ctype_in, i.name_lower, p.name_lower, i.camel_name))
            self.c.write('  return %s%s_GET_IFACE (object)->get_%s (object);\n'%(i.ns_upper, i.name_upper, p.name_lower))
            self.c.write('}\n')
            self.c.write('\n')
            if p.arg.free_func != None:

                self.c.write(self.docbook_gen.expand(
                        '/**\n'
                        ' * %s_dup_%s: (skip)\n'
                        ' * @object: A #%s.\n'
                        ' *\n'
                        ' * Gets a copy of the #%s:%s D-Bus property.\n'
                        ' *\n'
                        ' * %s\n'
                        ' *\n'
                        ' * Returns: (transfer full): The property value or %%NULL if the property is not set. The returned value should be freed with %s().\n'
                        %(i.name_lower, p.name_lower, i.camel_name, i.name, p.name, hint, p.arg.free_func), False))
                self.write_gtkdoc_deprecated_and_since_and_close(p, self.c, 0)
                self.c.write('%s\n'
                             '%s_dup_%s (%s *object)\n'
                             '{\n'
                             '  %svalue;\n'%(p.arg.ctype_in_dup, i.name_lower, p.name_lower, i.camel_name, p.arg.ctype_in_dup))
                self.c.write('  g_object_get (G_OBJECT (object), "%s", &value, NULL);\n'%(p.name_hyphen))
                self.c.write('  return value;\n')
                self.c.write('}\n')
                self.c.write('\n')

            # setter
            if p.readable and p.writable:
                hint = 'Since this D-Bus property is both readable and writable, it is meaningful to use this function on both the client- and service-side.'
            elif p.readable:
                hint = 'Since this D-Bus property is not writable, it is only meaningful to use this function on the service-side.'
            elif p.writable:
                hint = 'Since this D-Bus property is writable, it is meaningful to use this function on both the client- and service-side.'
            else:
                raise RuntimeError('Cannot handle property %s that neither readable nor writable'%(p.name))
            self.c.write(self.docbook_gen.expand(
                    '/**\n'
                    ' * %s_set_%s: (skip)\n'
                    ' * @object: A #%s.\n'
                    ' * @value: The value to set.\n'
                    ' *\n'
                    ' * Sets the #%s:%s D-Bus property to @value.\n'
                    ' *\n'
                    ' * %s\n'
                    %(i.name_lower, p.name_lower, i.camel_name, i.name, p.name, hint), False))
            self.write_gtkdoc_deprecated_and_since_and_close(p, self.c, 0)
            self.c.write('void\n'
                         '%s_set_%s (%s *object, %svalue)\n'
                         '{\n'%(i.name_lower, p.name_lower, i.camel_name, p.arg.ctype_in, ))
            self.c.write('  g_object_set (G_OBJECT (object), "%s", value, NULL);\n'%(p.name_hyphen))
            self.c.write('}\n')
            self.c.write('\n')

    # ---------------------------------------------------------------------------------------------------

    def generate_signal_emitters(self, i):
        for s in i.signals:
            self.c.write(self.docbook_gen.expand(
                    '/**\n'
                    ' * %s_emit_%s:\n'
                    ' * @object: A #%s.\n'
                    %(i.name_lower, s.name_lower, i.camel_name), False))
            for a in s.args:
                self.c.write(' * @arg_%s: Argument to pass with the signal.\n'%(a.name))
            self.c.write(self.docbook_gen.expand(
                    ' *\n'
                    ' * Emits the #%s::%s D-Bus signal.\n'
                    %(i.name, s.name), False))
            self.write_gtkdoc_deprecated_and_since_and_close(s, self.c, 0)
            self.c.write('void\n'
                         '%s_emit_%s (\n'
                         '    %s *object'%(i.name_lower, s.name_lower, i.camel_name))
            for a in s.args:
                self.c.write(',\n    %sarg_%s'%(a.ctype_in, a.name))
            self.c.write(')\n'
                         '{\n'
                         '  g_signal_emit_by_name (object, "%s"'%(s.name_hyphen))
            for a in s.args:
                self.c.write(', arg_%s'%a.name)
            self.c.write(');\n')
            self.c.write('}\n'
                         '\n')

    # ---------------------------------------------------------------------------------------------------

    def generate_method_calls(self, i):
        for m in i.methods:
            unix_fd = False
            if utils.lookup_annotation(m.annotations, 'org.gtk.GDBus.C.UnixFD'):
                unix_fd = True
            # async begin
            self.c.write('/**\n'
                         ' * %s_call_%s:\n'
                         ' * @proxy: A #%sProxy.\n'
                         %(i.name_lower, m.name_lower, i.camel_name))
            for a in m.in_args:
                self.c.write(' * @arg_%s: Argument to pass with the method invocation.\n'%(a.name))
            if unix_fd:
                self.c.write(' * @fd_list: (allow-none): A #GUnixFDList or %NULL.\n')
            self.c.write(self.docbook_gen.expand(
                    ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                    ' * @callback: A #GAsyncReadyCallback to call when the request is satisfied or %%NULL.\n'
                    ' * @user_data: User data to pass to @callback.\n'
                    ' *\n'
                    ' * Asynchronously invokes the %s.%s() D-Bus method on @proxy.\n'
                    ' * When the operation is finished, @callback will be invoked in the <link linkend="g-main-context-push-thread-default">thread-default main loop</link> of the thread you are calling this method from.\n'
                    ' * You can then call %s_call_%s_finish() to get the result of the operation.\n'
                    ' *\n'
                    ' * See %s_call_%s_sync() for the synchronous, blocking version of this method.\n'
                    %(i.name, m.name, i.name_lower, m.name_lower, i.name_lower, m.name_lower), False))
            self.write_gtkdoc_deprecated_and_since_and_close(m, self.c, 0)
            self.c.write('void\n'
                         '%s_call_%s (\n'
                         '    %s *proxy'%(i.name_lower, m.name_lower, i.camel_name))
            for a in m.in_args:
                self.c.write(',\n    %sarg_%s'%(a.ctype_in, a.name))
            if unix_fd:
                self.c.write(',\n    GUnixFDList *fd_list')
            self.c.write(',\n'
                         '    GCancellable *cancellable,\n'
                         '    GAsyncReadyCallback callback,\n'
                         '    gpointer user_data)\n'
                         '{\n')
            if unix_fd:
                self.c.write('  g_dbus_proxy_call_with_unix_fd_list (G_DBUS_PROXY (proxy),\n')
            else:
                self.c.write('  g_dbus_proxy_call (G_DBUS_PROXY (proxy),\n')
            self.c.write('    "%s",\n'
                         '    g_variant_new ("('%(m.name))
            for a in m.in_args:
                self.c.write('%s'%(a.format_in))
            self.c.write(')"')
            for a in m.in_args:
                self.c.write(',\n                   arg_%s'%(a.name))
            self.c.write('),\n'
                         '    G_DBUS_CALL_FLAGS_NONE,\n'
                         '    -1,\n')
            if unix_fd:
                self.c.write('    fd_list,\n')
            self.c.write('    cancellable,\n'
                         '    callback,\n'
                         '    user_data);\n')
            self.c.write('}\n'
                         '\n')
            # async finish
            self.c.write('/**\n'
                         ' * %s_call_%s_finish:\n'
                         ' * @proxy: A #%sProxy.\n'
                         %(i.name_lower, m.name_lower, i.camel_name))
            for a in m.out_args:
                self.c.write(' * @out_%s: (out): Return location for return parameter or %%NULL to ignore.\n'%(a.name))
            if unix_fd:
                self.c.write(' * @out_fd_list: (out): Return location for a #GUnixFDList or %NULL.\n')
            self.c.write(self.docbook_gen.expand(
                    ' * @res: The #GAsyncResult obtained from the #GAsyncReadyCallback passed to %s_call_%s().\n'
                    ' * @error: Return location for error or %%NULL.\n'
                    ' *\n'
                    ' * Finishes an operation started with %s_call_%s().\n'
                    ' *\n'
                    ' * Returns: (skip): %%TRUE if the call succeded, %%FALSE if @error is set.\n'
                    %(i.name_lower, m.name_lower, i.name_lower, m.name_lower), False))
            self.write_gtkdoc_deprecated_and_since_and_close(m, self.c, 0)
            self.c.write('gboolean\n'
                         '%s_call_%s_finish (\n'
                         '    %s *proxy'%(i.name_lower, m.name_lower, i.camel_name))
            for a in m.out_args:
                self.c.write(',\n    %sout_%s'%(a.ctype_out, a.name))
            if unix_fd:
                self.c.write(',\n    GUnixFDList **out_fd_list')
            self.c.write(',\n'
                         '    GAsyncResult *res,\n'
                         '    GError **error)\n'
                         '{\n'
                         '  GVariant *_ret;\n')
            if unix_fd:
                self.c.write('  _ret = g_dbus_proxy_call_with_unix_fd_list_finish (G_DBUS_PROXY (proxy), out_fd_list, res, error);\n')
            else:
                self.c.write('  _ret = g_dbus_proxy_call_finish (G_DBUS_PROXY (proxy), res, error);\n')
            self.c.write('  if (_ret == NULL)\n'
                         '    goto _out;\n')
            self.c.write('  g_variant_get (_ret,\n'
                         '                 \"(')
            for a in m.out_args:
                self.c.write('%s'%(a.format_out))
            self.c.write(')"')
            for a in m.out_args:
                self.c.write(',\n                 out_%s'%(a.name))
            self.c.write(');\n'
                         '  g_variant_unref (_ret);\n')
            self.c.write('_out:\n'
                         '  return _ret != NULL;\n'
                         '}\n'
                         '\n')


            # sync
            self.c.write('/**\n'
                         ' * %s_call_%s_sync:\n'
                         ' * @proxy: A #%sProxy.\n'
                         %(i.name_lower, m.name_lower, i.camel_name))
            for a in m.in_args:
                self.c.write(' * @arg_%s: Argument to pass with the method invocation.\n'%(a.name))
            if unix_fd:
                self.c.write(' * @fd_list: (allow-none): A #GUnixFDList or %NULL.\n')
            for a in m.out_args:
                self.c.write(' * @out_%s: (out): Return location for return parameter or %%NULL to ignore.\n'%(a.name))
            if unix_fd:
                self.c.write(' * @out_fd_list: (out): Return location for a #GUnixFDList or %NULL.\n')
            self.c.write(self.docbook_gen.expand(
                    ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                    ' * @error: Return location for error or %%NULL.\n'
                    ' *\n'
                    ' * Synchronously invokes the %s.%s() D-Bus method on @proxy. The calling thread is blocked until a reply is received.\n'
                    ' *\n'
                    ' * See %s_call_%s() for the asynchronous version of this method.\n'
                    ' *\n'
                    ' * Returns: (skip): %%TRUE if the call succeded, %%FALSE if @error is set.\n'
                    %(i.name, m.name, i.name_lower, m.name_lower), False))
            self.write_gtkdoc_deprecated_and_since_and_close(m, self.c, 0)
            self.c.write('gboolean\n'
                         '%s_call_%s_sync (\n'
                         '    %s *proxy'%(i.name_lower, m.name_lower, i.camel_name))
            for a in m.in_args:
                self.c.write(',\n    %sarg_%s'%(a.ctype_in, a.name))
            if unix_fd:
                self.c.write(',\n    GUnixFDList  *fd_list')
            for a in m.out_args:
                self.c.write(',\n    %sout_%s'%(a.ctype_out, a.name))
            if unix_fd:
                self.c.write(',\n    GUnixFDList **out_fd_list')
            self.c.write(',\n'
                         '    GCancellable *cancellable,\n'
                         '    GError **error)\n'
                         '{\n'
                         '  GVariant *_ret;\n')
            if unix_fd:
                self.c.write('  _ret = g_dbus_proxy_call_with_unix_fd_list_sync (G_DBUS_PROXY (proxy),\n')
            else:
                self.c.write('  _ret = g_dbus_proxy_call_sync (G_DBUS_PROXY (proxy),\n')
            self.c.write('    "%s",\n'
                         '    g_variant_new ("('%(m.name))
            for a in m.in_args:
                self.c.write('%s'%(a.format_in))
            self.c.write(')"')
            for a in m.in_args:
                self.c.write(',\n                   arg_%s'%(a.name))
            self.c.write('),\n'
                         '    G_DBUS_CALL_FLAGS_NONE,\n'
                         '    -1,\n')
            if unix_fd:
                self.c.write('    fd_list,\n'
                             '    out_fd_list,\n')
            self.c.write('    cancellable,\n'
                         '    error);\n'
                         '  if (_ret == NULL)\n'
                         '    goto _out;\n')
            self.c.write('  g_variant_get (_ret,\n'
                         '                 \"(')
            for a in m.out_args:
                self.c.write('%s'%(a.format_out))
            self.c.write(')"')
            for a in m.out_args:
                self.c.write(',\n                 out_%s'%(a.name))
            self.c.write(');\n'
                         '  g_variant_unref (_ret);\n')
            self.c.write('_out:\n'
                         '  return _ret != NULL;\n'
                         '}\n'
                         '\n')

    # ---------------------------------------------------------------------------------------------------

    def generate_method_completers(self, i):
        for m in i.methods:
            unix_fd = False
            if utils.lookup_annotation(m.annotations, 'org.gtk.GDBus.C.UnixFD'):
                unix_fd = True
            self.c.write('/**\n'
                         ' * %s_complete_%s:\n'
                         ' * @object: A #%s.\n'
                         ' * @invocation: (transfer full): A #GDBusMethodInvocation.\n'
                         %(i.name_lower, m.name_lower, i.camel_name))
            if unix_fd:
                self.c.write (' * @fd_list: (allow-none): A #GUnixFDList or %NULL.\n')
            for a in m.out_args:
                self.c.write(' * @%s: Parameter to return.\n'%(a.name))
            self.c.write(self.docbook_gen.expand(
                    ' *\n'
                    ' * Helper function used in service implementations to finish handling invocations of the %s.%s() D-Bus method. If you instead want to finish handling an invocation by returning an error, use g_dbus_method_invocation_return_error() or similar.\n'
                    ' *\n'
                    ' * This method will free @invocation, you cannot use it afterwards.\n'
                    %(i.name, m.name), False))
            self.write_gtkdoc_deprecated_and_since_and_close(m, self.c, 0)
            self.c.write('void\n'
                         '%s_complete_%s (\n'
                         '    %s *object,\n'
                         '    GDBusMethodInvocation *invocation'%(i.name_lower, m.name_lower, i.camel_name))
            if unix_fd:
                self.c.write(',\n    GUnixFDList *fd_list')
            for a in m.out_args:
                self.c.write(',\n    %s%s'%(a.ctype_in, a.name))
            self.c.write(')\n'
                         '{\n')

            if unix_fd:
                self.c.write('  g_dbus_method_invocation_return_value_with_unix_fd_list (invocation,\n'
                             '    g_variant_new ("(')
            else:
                self.c.write('  g_dbus_method_invocation_return_value (invocation,\n'
                             '    g_variant_new ("(')
            for a in m.out_args:
                self.c.write('%s'%(a.format_in))
            self.c.write(')"')
            for a in m.out_args:
                self.c.write(',\n                   %s'%(a.name))
            if unix_fd:
                self.c.write('),\n    fd_list);\n')
            else:
                self.c.write('));\n')
            self.c.write('}\n'
                         '\n')

    # ---------------------------------------------------------------------------------------------------

    def generate_proxy(self, i):
        # class boilerplate
        self.c.write('/* ------------------------------------------------------------------------ */\n'
                     '\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sProxy:\n'
                ' *\n'
                ' * The #%sProxy structure contains only private data and should only be accessed using the provided API.\n'
                %(i.camel_name, i.camel_name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sProxyClass:\n'
                ' * @parent_class: The parent class.\n'
                ' *\n'
                ' * Class structure for #%sProxy.\n'
                %(i.camel_name, i.camel_name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('\n')

        self.c.write('struct _%sProxyPrivate\n'
                     '{\n'
                     '  GData *qdata;\n'
                     '};\n'
                     '\n'%i.camel_name)

        self.c.write('static void %s_proxy_iface_init (%sIface *iface);\n'
                     '\n'%(i.name_lower, i.camel_name))
        self.c.write('#if GLIB_VERSION_MAX_ALLOWED >= GLIB_VERSION_2_38\n')
        self.c.write('G_DEFINE_TYPE_WITH_CODE (%sProxy, %s_proxy, G_TYPE_DBUS_PROXY,\n'%(i.camel_name, i.name_lower))
        self.c.write('                         G_ADD_PRIVATE (%sProxy)\n'%(i.camel_name))
        self.c.write('                         G_IMPLEMENT_INTERFACE (%sTYPE_%s, %s_proxy_iface_init));\n\n'%(i.ns_upper, i.name_upper, i.name_lower))
        self.c.write('#else\n')
        self.c.write('G_DEFINE_TYPE_WITH_CODE (%sProxy, %s_proxy, G_TYPE_DBUS_PROXY,\n'%(i.camel_name, i.name_lower))
        self.c.write('                         G_IMPLEMENT_INTERFACE (%sTYPE_%s, %s_proxy_iface_init));\n\n'%(i.ns_upper, i.name_upper, i.name_lower))
        self.c.write('#endif\n')

        # finalize
        self.c.write('static void\n'
                     '%s_proxy_finalize (GObject *object)\n'
                     '{\n'%(i.name_lower))
        self.c.write('  %sProxy *proxy = %s%s_PROXY (object);\n'%(i.camel_name, i.ns_upper, i.name_upper))
        self.c.write('  g_datalist_clear (&proxy->priv->qdata);\n')
        self.c.write('  G_OBJECT_CLASS (%s_proxy_parent_class)->finalize (object);\n'
                     '}\n'
                     '\n'%(i.name_lower))

        # property accessors
        #
        # Note that we are guaranteed that prop_id starts at 1 and is
        # laid out in the same order as introspection data pointers
        #
        self.c.write('static void\n'
                     '%s_proxy_get_property (GObject      *object,\n'
                     '  guint         prop_id,\n'
                     '  GValue       *value,\n'
                     '  GParamSpec   *pspec G_GNUC_UNUSED)\n'
                     '{\n'%(i.name_lower))
        if len(i.properties) > 0:
            self.c.write('  const _ExtendedGDBusPropertyInfo *info;\n'
                         '  GVariant *variant;\n'
                         '  g_assert (prop_id != 0 && prop_id - 1 < %d);\n'
                         '  info = _%s_property_info_pointers[prop_id - 1];\n'
                         '  variant = g_dbus_proxy_get_cached_property (G_DBUS_PROXY (object), info->parent_struct.name);\n'
                         '  if (info->use_gvariant)\n'
                         '    {\n'
                         '      g_value_set_variant (value, variant);\n'
                         '    }\n'
                         '  else\n'
                         '    {\n'
                         # could be that we don't have the value in cache - in that case, we do
                         # nothing and the user gets the default value for the GType
                         '      if (variant != NULL)\n'
                         '        g_dbus_gvariant_to_gvalue (variant, value);\n'
                         '    }\n'
                         '  if (variant != NULL)\n'
                         '    g_variant_unref (variant);\n'
                         %(len(i.properties), i.name_lower))
        self.c.write('}\n'
                     '\n')
        if len(i.properties) > 0:
            self.c.write('static void\n'
                         '%s_proxy_set_property_cb (GDBusProxy *proxy,\n'
                         '  GAsyncResult *res,\n'
                         '  gpointer      user_data)\n'
                         '{\n'%(i.name_lower))
            self.c.write('  const _ExtendedGDBusPropertyInfo *info = user_data;\n'
                         '  GError *error;\n'
                         '  GVariant *_ret;\n'
                         '  error = NULL;\n'
                         '  _ret = g_dbus_proxy_call_finish (proxy, res, &error);\n'
                         '  if (!_ret)\n'
                         '    {\n'
                         '      g_warning ("Error setting property \'%%s\' on interface %s: %%s (%%s, %%d)",\n'
                         '                 info->parent_struct.name, \n'
                         '                 error->message, g_quark_to_string (error->domain), error->code);\n'
                         '      g_error_free (error);\n'
                         '    }\n'
                         '  else\n'
                         '    {\n'
                         '      g_variant_unref (_ret);\n'
                         '    }\n'
                         %(i.name))
            self.c.write('}\n'
                         '\n')
        self.c.write('static void\n'
                     '%s_proxy_set_property (GObject      *object,\n'
                     '  guint         prop_id,\n'
                     '  const GValue *value,\n'
                     '  GParamSpec   *pspec G_GNUC_UNUSED)\n'
                     '{\n'%(i.name_lower))
        if len(i.properties) > 0:
            self.c.write('  const _ExtendedGDBusPropertyInfo *info;\n'
                         '  GVariant *variant;\n'
                         '  g_assert (prop_id != 0 && prop_id - 1 < %d);\n'
                         '  info = _%s_property_info_pointers[prop_id - 1];\n'
                         '  variant = g_dbus_gvalue_to_gvariant (value, G_VARIANT_TYPE (info->parent_struct.signature));\n'
                         '  g_dbus_proxy_call (G_DBUS_PROXY (object),\n'
                         '    "org.freedesktop.DBus.Properties.Set",\n'
                         '    g_variant_new ("(ssv)", "%s", info->parent_struct.name, variant),\n'
                         '    G_DBUS_CALL_FLAGS_NONE,\n'
                         '    -1,\n'
                         '    NULL, (GAsyncReadyCallback) %s_proxy_set_property_cb, (GDBusPropertyInfo *) &info->parent_struct);\n'
                         '  g_variant_unref (variant);\n'
                         %(len(i.properties), i.name_lower, i.name, i.name_lower))
        self.c.write('}\n'
                     '\n')

        # signal received
        self.c.write('static void\n'
                     '%s_proxy_g_signal (GDBusProxy *proxy,\n'
                     '  const gchar *sender_name G_GNUC_UNUSED,\n'
                     '  const gchar *signal_name,\n'
                     '  GVariant *parameters)\n'
                     '{\n'%(i.name_lower))
        self.c.write('  _ExtendedGDBusSignalInfo *info;\n'
                     '  GVariantIter iter;\n'
                     '  GVariant *child;\n'
                     '  GValue *paramv;\n'
                     '  guint num_params;\n'
                     '  guint n;\n'
                     '  guint signal_id;\n');
        # Note: info could be NULL if we are talking to a newer version of the interface
        self.c.write('  info = (_ExtendedGDBusSignalInfo *) g_dbus_interface_info_lookup_signal ((GDBusInterfaceInfo *) &_%s_interface_info.parent_struct, signal_name);\n'
                     '  if (info == NULL)\n'
                     '    return;\n'
                     %(i.name_lower))
        self.c.write ('  num_params = g_variant_n_children (parameters);\n'
                      '  paramv = g_new0 (GValue, num_params + 1);\n'
                      '  g_value_init (&paramv[0], %sTYPE_%s);\n'
                      '  g_value_set_object (&paramv[0], proxy);\n'
                      %(i.ns_upper, i.name_upper))
        self.c.write('  g_variant_iter_init (&iter, parameters);\n'
                     '  n = 1;\n'
                     '  while ((child = g_variant_iter_next_value (&iter)) != NULL)\n'
                     '    {\n'
                     '      _ExtendedGDBusArgInfo *arg_info = (_ExtendedGDBusArgInfo *) info->parent_struct.args[n - 1];\n'
                     '      if (arg_info->use_gvariant)\n'
                     '        {\n'
                     '          g_value_init (&paramv[n], G_TYPE_VARIANT);\n'
                     '          g_value_set_variant (&paramv[n], child);\n'
                     '          n++;\n'
                     '        }\n'
                     '      else\n'
                     '        g_dbus_gvariant_to_gvalue (child, &paramv[n++]);\n'
                     '      g_variant_unref (child);\n'
                     '    }\n'
                     )
        self.c.write('  signal_id = g_signal_lookup (info->signal_name, %sTYPE_%s);\n'
                     %(i.ns_upper, i.name_upper))
        self.c.write('  g_signal_emitv (paramv, signal_id, 0, NULL);\n')
        self.c.write('  for (n = 0; n < num_params + 1; n++)\n'
                     '    g_value_unset (&paramv[n]);\n'
                     '  g_free (paramv);\n')
        self.c.write('}\n'
                     '\n')

        # property changed
        self.c.write('static void\n'
                     '%s_proxy_g_properties_changed (GDBusProxy *_proxy,\n'
                     '  GVariant *changed_properties,\n'
                     '  const gchar *const *invalidated_properties)\n'
                     '{\n'%(i.name_lower))
        # Note: info could be NULL if we are talking to a newer version of the interface
        self.c.write('  %sProxy *proxy = %s%s_PROXY (_proxy);\n'
                     '  guint n;\n'
                     '  const gchar *key;\n'
                     '  GVariantIter *iter;\n'
                     '  _ExtendedGDBusPropertyInfo *info;\n'
                     '  g_variant_get (changed_properties, "a{sv}", &iter);\n'
                     '  while (g_variant_iter_next (iter, "{&sv}", &key, NULL))\n'
                     '    {\n'
                     '      info = (_ExtendedGDBusPropertyInfo *) g_dbus_interface_info_lookup_property ((GDBusInterfaceInfo *) &_%s_interface_info.parent_struct, key);\n'
                     '      g_datalist_remove_data (&proxy->priv->qdata, key);\n'
                     '      if (info != NULL)\n'
                     '        g_object_notify (G_OBJECT (proxy), info->hyphen_name);\n'
                     '    }\n'
                     '  g_variant_iter_free (iter);\n'
                     '  for (n = 0; invalidated_properties[n] != NULL; n++)\n'
                     '    {\n'
                     '      info = (_ExtendedGDBusPropertyInfo *) g_dbus_interface_info_lookup_property ((GDBusInterfaceInfo *) &_%s_interface_info.parent_struct, invalidated_properties[n]);\n'
                     '      g_datalist_remove_data (&proxy->priv->qdata, invalidated_properties[n]);\n'
                     '      if (info != NULL)\n'
                     '        g_object_notify (G_OBJECT (proxy), info->hyphen_name);\n'
                     '    }\n'
                     '}\n'
                     '\n'
                     %(i.camel_name, i.ns_upper, i.name_upper,
                       i.name_lower, i.name_lower))

        # property vfuncs
        for p in i.properties:
            nul_value = '0'
            if p.arg.free_func != None:
                nul_value = 'NULL'
            self.c.write('static %s\n'
                         '%s_proxy_get_%s (%s *object)\n'
                         '{\n'
                         '  %sProxy *proxy = %s%s_PROXY (object);\n'
                         '  GVariant *variant;\n'
                         '  %svalue = %s;\n'%(p.arg.ctype_in, i.name_lower, p.name_lower, i.camel_name,
                                              i.camel_name, i.ns_upper, i.name_upper,
                                              p.arg.ctype_in, nul_value))
            # For some property types, we have to free the returned
            # value (or part of it, e.g. the container) because of how
            # GVariant works.. see https://bugzilla.gnome.org/show_bug.cgi?id=657100
            # for details
            #
            free_container = False;
            if p.arg.gvariant_get == 'g_variant_get_strv' or p.arg.gvariant_get == 'g_variant_get_objpathv' or p.arg.gvariant_get == 'g_variant_get_bytestring_array':
                free_container = True;
            # If already using an old value for strv, objpathv, bytestring_array (see below),
            # then just return that... that way the result from multiple consecutive calls
            # to the getter are valid as long as they're freed
            #
            if free_container:
                self.c.write('  value = g_datalist_get_data (&proxy->priv->qdata, \"%s\");\n'
                             '  if (value != NULL)\n'
                             '    return value;\n'
                             %(p.name))
            self.c.write('  variant = g_dbus_proxy_get_cached_property (G_DBUS_PROXY (proxy), \"%s\");\n'%(p.name))
            if p.arg.gtype == 'G_TYPE_VARIANT':
                self.c.write('  value = variant;\n')
                self.c.write('  if (variant != NULL)\n')
                self.c.write('    g_variant_unref (variant);\n')
            else:
                self.c.write('  if (variant != NULL)\n'
                             '    {\n')
                extra_len = ''
                if p.arg.gvariant_get == 'g_variant_get_string' or p.arg.gvariant_get == 'g_variant_get_strv' or p.arg.gvariant_get == 'g_variant_get_objv' or p.arg.gvariant_get == 'g_variant_get_bytestring_array':
                    extra_len = ', NULL'
                self.c.write('      value = %s (variant%s);\n'%(p.arg.gvariant_get, extra_len))
                if free_container:
                    self.c.write('      g_datalist_set_data_full (&proxy->priv->qdata, \"%s\", (gpointer) value, g_free);\n'
                                 %(p.name))
                self.c.write('      g_variant_unref (variant);\n')
                self.c.write('    }\n')
            self.c.write('  return value;\n')
            self.c.write('}\n')
            self.c.write('\n')

        # class boilerplate
        self.c.write('static void\n'
                     '%s_proxy_init (%sProxy *proxy)\n'
                     '{\n'
                     '#if GLIB_VERSION_MAX_ALLOWED >= GLIB_VERSION_2_38\n'
                     '  proxy->priv = %s_proxy_get_instance_private (proxy);\n'
                     '#else\n'
                     '  proxy->priv = G_TYPE_INSTANCE_GET_PRIVATE (proxy, %sTYPE_%s_PROXY, %sProxyPrivate);\n'
                     '#endif\n\n'
                     '  g_dbus_proxy_set_interface_info (G_DBUS_PROXY (proxy), %s_interface_info ());\n'
                     '}\n'
                     '\n'
                     %(i.name_lower, i.camel_name,
                       i.name_lower,
                       i.ns_upper, i.name_upper, i.camel_name,
                       i.name_lower))
        self.c.write('static void\n'
                     '%s_proxy_class_init (%sProxyClass *klass)\n'
                     '{\n'
                     '  GObjectClass *gobject_class;\n'
                     '  GDBusProxyClass *proxy_class;\n'
                     '\n'
                     '  gobject_class = G_OBJECT_CLASS (klass);\n'
                     '  gobject_class->finalize     = %s_proxy_finalize;\n'
                     '  gobject_class->get_property = %s_proxy_get_property;\n'
                     '  gobject_class->set_property = %s_proxy_set_property;\n'
                     '\n'
                     '  proxy_class = G_DBUS_PROXY_CLASS (klass);\n'
                     '  proxy_class->g_signal = %s_proxy_g_signal;\n'
                     '  proxy_class->g_properties_changed = %s_proxy_g_properties_changed;\n'
                     '\n'%(i.name_lower, i.camel_name,
                           i.name_lower, i.name_lower, i.name_lower, i.name_lower, i.name_lower))
        if len(i.properties) > 0:
            self.c.write('  %s_override_properties (gobject_class, 1);\n\n'%(i.name_lower))
        self.c.write('#if GLIB_VERSION_MAX_ALLOWED < GLIB_VERSION_2_38\n'
                     '  g_type_class_add_private (klass, sizeof (%sProxyPrivate));\n'
                     '#endif\n'%(i.camel_name))
        self.c.write('}\n'
                     '\n')

        self.c.write('static void\n'
                     '%s_proxy_iface_init (%sIface *iface)\n'
                     '{\n'%(i.name_lower, i.camel_name))
        for p in i.properties:
            self.c.write('  iface->get_%s = %s_proxy_get_%s;\n'%(p.name_lower, i.name_lower, p.name_lower))
        self.c.write('}\n'
                     '\n')

        # constructors
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %s_proxy_new:\n'
                ' * @connection: A #GDBusConnection.\n'
                ' * @flags: Flags from the #GDBusProxyFlags enumeration.\n'
                ' * @name: (allow-none): A bus name (well-known or unique) or %%NULL if @connection is not a message bus connection.\n'
                ' * @object_path: An object path.\n'
                ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                ' * @callback: A #GAsyncReadyCallback to call when the request is satisfied.\n'
                ' * @user_data: User data to pass to @callback.\n'
                ' *\n'
                ' * Asynchronously creates a proxy for the D-Bus interface #%s. See g_dbus_proxy_new() for more details.\n'
                ' *\n'
                ' * When the operation is finished, @callback will be invoked in the <link linkend="g-main-context-push-thread-default">thread-default main loop</link> of the thread you are calling this method from.\n'
                ' * You can then call %s_proxy_new_finish() to get the result of the operation.\n'
                ' *\n'
                ' * See %s_proxy_new_sync() for the synchronous, blocking version of this constructor.\n'
                %(i.name_lower, i.name, i.name_lower, i.name_lower), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('void\n'
                     '%s_proxy_new (\n'
                     '    GDBusConnection     *connection,\n'
                     '    GDBusProxyFlags      flags,\n'
                     '    const gchar         *name,\n'
                     '    const gchar         *object_path,\n'
                     '    GCancellable        *cancellable,\n'
                     '    GAsyncReadyCallback  callback,\n'
                     '    gpointer             user_data)\n'
                     '{\n'
                     '  g_async_initable_new_async (%sTYPE_%s_PROXY, G_PRIORITY_DEFAULT, cancellable, callback, user_data, "g-flags", flags, "g-name", name, "g-connection", connection, "g-object-path", object_path, "g-interface-name", "%s", NULL);\n'
                     '}\n'
                     '\n'
                     %(i.name_lower, i.ns_upper, i.name_upper, i.name))
        self.c.write('/**\n'
                     ' * %s_proxy_new_finish:\n'
                     ' * @res: The #GAsyncResult obtained from the #GAsyncReadyCallback passed to %s_proxy_new().\n'
                     ' * @error: Return location for error or %%NULL\n'
                     ' *\n'
                     ' * Finishes an operation started with %s_proxy_new().\n'
                     ' *\n'
                     ' * Returns: (transfer full) (type %sProxy): The constructed proxy object or %%NULL if @error is set.\n'
                     %(i.name_lower, i.name_lower, i.name_lower, i.camel_name))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('%s *\n'
                     '%s_proxy_new_finish (\n'
                     '    GAsyncResult        *res,\n'
                     '    GError             **error)\n'
                     '{\n'
                     '  GObject *ret;\n'
                     '  GObject *source_object;\n'
                     '  source_object = g_async_result_get_source_object (res);\n'
                     '  ret = g_async_initable_new_finish (G_ASYNC_INITABLE (source_object), res, error);\n'
                     '  g_object_unref (source_object);\n'
                     '  if (ret != NULL)\n'
                     '    return %s%s (ret);\n'
                     '  else\n'
                     '    return NULL;\n'
                     '}\n'
                     '\n'
                     %(i.camel_name, i.name_lower, i.ns_upper, i.name_upper))
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %s_proxy_new_sync:\n'
                ' * @connection: A #GDBusConnection.\n'
                ' * @flags: Flags from the #GDBusProxyFlags enumeration.\n'
                ' * @name: (allow-none): A bus name (well-known or unique) or %%NULL if @connection is not a message bus connection.\n'
                ' * @object_path: An object path.\n'
                ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                ' * @error: Return location for error or %%NULL\n'
                ' *\n'
                ' * Synchronously creates a proxy for the D-Bus interface #%s. See g_dbus_proxy_new_sync() for more details.\n'
                ' *\n'
                ' * The calling thread is blocked until a reply is received.\n'
                ' *\n'
                ' * See %s_proxy_new() for the asynchronous version of this constructor.\n'
                ' *\n'
                ' * Returns: (transfer full) (type %sProxy): The constructed proxy object or %%NULL if @error is set.\n'
                %(i.name_lower, i.name, i.name_lower, i.camel_name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('%s *\n'
                     '%s_proxy_new_sync (\n'
                     '    GDBusConnection     *connection,\n'
                     '    GDBusProxyFlags      flags,\n'
                     '    const gchar         *name,\n'
                     '    const gchar         *object_path,\n'
                     '    GCancellable        *cancellable,\n'
                     '    GError             **error)\n'
                     '{\n'
                     '  GInitable *ret;\n'
                     '  ret = g_initable_new (%sTYPE_%s_PROXY, cancellable, error, "g-flags", flags, "g-name", name, "g-connection", connection, "g-object-path", object_path, "g-interface-name", "%s", NULL);\n'
                     '  if (ret != NULL)\n'
                     '    return %s%s (ret);\n'
                     '  else\n'
                     '    return NULL;\n'
                     '}\n'
                     '\n'
                     %(i.camel_name, i.name_lower, i.ns_upper, i.name_upper, i.name, i.ns_upper, i.name_upper))
        self.c.write('\n')
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %s_proxy_new_for_bus:\n'
                ' * @bus_type: A #GBusType.\n'
                ' * @flags: Flags from the #GDBusProxyFlags enumeration.\n'
                ' * @name: A bus name (well-known or unique).\n'
                ' * @object_path: An object path.\n'
                ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                ' * @callback: A #GAsyncReadyCallback to call when the request is satisfied.\n'
                ' * @user_data: User data to pass to @callback.\n'
                ' *\n'
                ' * Like %s_proxy_new() but takes a #GBusType instead of a #GDBusConnection.\n'
                ' *\n'
                ' * When the operation is finished, @callback will be invoked in the <link linkend="g-main-context-push-thread-default">thread-default main loop</link> of the thread you are calling this method from.\n'
                ' * You can then call %s_proxy_new_for_bus_finish() to get the result of the operation.\n'
                ' *\n'
                ' * See %s_proxy_new_for_bus_sync() for the synchronous, blocking version of this constructor.\n'
                %(i.name_lower, i.name_lower, i.name_lower, i.name_lower), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('void\n'
                     '%s_proxy_new_for_bus (\n'
                     '    GBusType             bus_type,\n'
                     '    GDBusProxyFlags      flags,\n'
                     '    const gchar         *name,\n'
                     '    const gchar         *object_path,\n'
                     '    GCancellable        *cancellable,\n'
                     '    GAsyncReadyCallback  callback,\n'
                     '    gpointer             user_data)\n'
                     '{\n'
                     '  g_async_initable_new_async (%sTYPE_%s_PROXY, G_PRIORITY_DEFAULT, cancellable, callback, user_data, "g-flags", flags, "g-name", name, "g-bus-type", bus_type, "g-object-path", object_path, "g-interface-name", "%s", NULL);\n'
                     '}\n'
                     '\n'
                     %(i.name_lower, i.ns_upper, i.name_upper, i.name))
        self.c.write('/**\n'
                     ' * %s_proxy_new_for_bus_finish:\n'
                     ' * @res: The #GAsyncResult obtained from the #GAsyncReadyCallback passed to %s_proxy_new_for_bus().\n'
                     ' * @error: Return location for error or %%NULL\n'
                     ' *\n'
                     ' * Finishes an operation started with %s_proxy_new_for_bus().\n'
                     ' *\n'
                     ' * Returns: (transfer full) (type %sProxy): The constructed proxy object or %%NULL if @error is set.\n'
                     %(i.name_lower, i.name_lower, i.name_lower, i.camel_name))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('%s *\n'
                     '%s_proxy_new_for_bus_finish (\n'
                     '    GAsyncResult        *res,\n'
                     '    GError             **error)\n'
                     '{\n'
                     '  GObject *ret;\n'
                     '  GObject *source_object;\n'
                     '  source_object = g_async_result_get_source_object (res);\n'
                     '  ret = g_async_initable_new_finish (G_ASYNC_INITABLE (source_object), res, error);\n'
                     '  g_object_unref (source_object);\n'
                     '  if (ret != NULL)\n'
                     '    return %s%s (ret);\n'
                     '  else\n'
                     '    return NULL;\n'
                     '}\n'
                     '\n'
                     %(i.camel_name, i.name_lower, i.ns_upper, i.name_upper))
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %s_proxy_new_for_bus_sync:\n'
                ' * @bus_type: A #GBusType.\n'
                ' * @flags: Flags from the #GDBusProxyFlags enumeration.\n'
                ' * @name: A bus name (well-known or unique).\n'
                ' * @object_path: An object path.\n'
                ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                ' * @error: Return location for error or %%NULL\n'
                ' *\n'
                ' * Like %s_proxy_new_sync() but takes a #GBusType instead of a #GDBusConnection.\n'
                ' *\n'
                ' * The calling thread is blocked until a reply is received.\n'
                ' *\n'
                ' * See %s_proxy_new_for_bus() for the asynchronous version of this constructor.\n'
                ' *\n'
                ' * Returns: (transfer full) (type %sProxy): The constructed proxy object or %%NULL if @error is set.\n'
                %(i.name_lower, i.name_lower, i.name_lower, i.camel_name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('%s *\n'
                     '%s_proxy_new_for_bus_sync (\n'
                     '    GBusType             bus_type,\n'
                     '    GDBusProxyFlags      flags,\n'
                     '    const gchar         *name,\n'
                     '    const gchar         *object_path,\n'
                     '    GCancellable        *cancellable,\n'
                     '    GError             **error)\n'
                     '{\n'
                     '  GInitable *ret;\n'
                     '  ret = g_initable_new (%sTYPE_%s_PROXY, cancellable, error, "g-flags", flags, "g-name", name, "g-bus-type", bus_type, "g-object-path", object_path, "g-interface-name", "%s", NULL);\n'
                     '  if (ret != NULL)\n'
                     '    return %s%s (ret);\n'
                     '  else\n'
                     '    return NULL;\n'
                     '}\n'
                     '\n'
                     %(i.camel_name, i.name_lower, i.ns_upper, i.name_upper, i.name, i.ns_upper, i.name_upper))
        self.c.write('\n')

    # ---------------------------------------------------------------------------------------------------

    def generate_skeleton(self, i):
        # class boilerplate
        self.c.write('/* ------------------------------------------------------------------------ */\n'
                     '\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sSkeleton:\n'
                ' *\n'
                ' * The #%sSkeleton structure contains only private data and should only be accessed using the provided API.\n'
                %(i.camel_name, i.camel_name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sSkeletonClass:\n'
                ' * @parent_class: The parent class.\n'
                ' *\n'
                ' * Class structure for #%sSkeleton.\n'
                %(i.camel_name, i.camel_name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('\n')

        self.c.write('struct _%sSkeletonPrivate\n'
                     '{\n'
                     '  GValue *properties;\n'
                     '  GList *changed_properties;\n'
                     '  GSource *changed_properties_idle_source;\n'
                     '  GMainContext *context;\n'
                     '  GMutex lock;\n'
                     '};\n'
                     '\n'%i.camel_name)

        self.c.write('static void\n'
                     '_%s_skeleton_handle_method_call (\n'
                     '  GDBusConnection *connection G_GNUC_UNUSED,\n'
                     '  const gchar *sender G_GNUC_UNUSED,\n'
                     '  const gchar *object_path G_GNUC_UNUSED,\n'
                     '  const gchar *interface_name,\n'
                     '  const gchar *method_name,\n'
                     '  GVariant *parameters,\n'
                     '  GDBusMethodInvocation *invocation,\n'
                     '  gpointer user_data)\n'
                     '{\n'
                     '  %sSkeleton *skeleton = %s%s_SKELETON (user_data);\n'
                     '  _ExtendedGDBusMethodInfo *info;\n'
                     '  GVariantIter iter;\n'
                     '  GVariant *child;\n'
                     '  GValue *paramv;\n'
                     '  guint num_params;\n'
                     '  guint num_extra;\n'
                     '  guint n;\n'
                     '  guint signal_id;\n'
                     '  GValue return_value = G_VALUE_INIT;\n'
                     %(i.name_lower, i.camel_name, i.ns_upper, i.name_upper))
        self.c.write('  info = (_ExtendedGDBusMethodInfo *) g_dbus_method_invocation_get_method_info (invocation);\n'
                     '  g_assert (info != NULL);\n'
                     %())
        self.c.write ('  num_params = g_variant_n_children (parameters);\n'
                      '  num_extra = info->pass_fdlist ? 3 : 2;'
                      '  paramv = g_new0 (GValue, num_params + num_extra);\n'
                      '  n = 0;\n'
                      '  g_value_init (&paramv[n], %sTYPE_%s);\n'
                      '  g_value_set_object (&paramv[n++], skeleton);\n'
                      '  g_value_init (&paramv[n], G_TYPE_DBUS_METHOD_INVOCATION);\n'
                      '  g_value_set_object (&paramv[n++], invocation);\n'
                      '  if (info->pass_fdlist)\n'
                      '    {\n'
                      '#ifdef G_OS_UNIX\n'
                      '      g_value_init (&paramv[n], G_TYPE_UNIX_FD_LIST);\n'
                      '      g_value_set_object (&paramv[n++], g_dbus_message_get_unix_fd_list (g_dbus_method_invocation_get_message (invocation)));\n'
                      '#else\n'
                      '      g_assert_not_reached ();\n'
                      '#endif\n'
                      '    }\n'
                      %(i.ns_upper, i.name_upper))
        self.c.write('  g_variant_iter_init (&iter, parameters);\n'
                     '  while ((child = g_variant_iter_next_value (&iter)) != NULL)\n'
                     '    {\n'
                     '      _ExtendedGDBusArgInfo *arg_info = (_ExtendedGDBusArgInfo *) info->parent_struct.in_args[n - num_extra];\n'
                     '      if (arg_info->use_gvariant)\n'
                     '        {\n'
                     '          g_value_init (&paramv[n], G_TYPE_VARIANT);\n'
                     '          g_value_set_variant (&paramv[n], child);\n'
                     '          n++;\n'
                     '        }\n'
                     '      else\n'
                     '        g_dbus_gvariant_to_gvalue (child, &paramv[n++]);\n'
                     '      g_variant_unref (child);\n'
                     '    }\n'
                     )
        self.c.write('  signal_id = g_signal_lookup (info->signal_name, %sTYPE_%s);\n'
                     %(i.ns_upper, i.name_upper))
        self.c.write('  g_value_init (&return_value, G_TYPE_BOOLEAN);\n'
                     '  g_signal_emitv (paramv, signal_id, 0, &return_value);\n'
                     '  if (!g_value_get_boolean (&return_value))\n'
                     '    g_dbus_method_invocation_return_error (invocation, G_DBUS_ERROR, G_DBUS_ERROR_UNKNOWN_METHOD, "Method %s is not implemented on interface %s", method_name, interface_name);\n'
                     '  g_value_unset (&return_value);\n'
                     )
        self.c.write('  for (n = 0; n < num_params + num_extra; n++)\n'
                     '    g_value_unset (&paramv[n]);\n'
                     '  g_free (paramv);\n')
        self.c.write('}\n'
                     '\n')

        self.c.write('static GVariant *\n'
                     '_%s_skeleton_handle_get_property (\n'
                     '  GDBusConnection *connection G_GNUC_UNUSED,\n'
                     '  const gchar *sender G_GNUC_UNUSED,\n'
                     '  const gchar *object_path G_GNUC_UNUSED,\n'
                     '  const gchar *interface_name G_GNUC_UNUSED,\n'
                     '  const gchar *property_name,\n'
                     '  GError **error,\n'
                     '  gpointer user_data)\n'
                     '{\n'
                     '  %sSkeleton *skeleton = %s%s_SKELETON (user_data);\n'
                     '  GValue value = G_VALUE_INIT;\n'
                     '  GParamSpec *pspec;\n'
                     '  _ExtendedGDBusPropertyInfo *info;\n'
                     '  GVariant *ret;\n'
                     %(i.name_lower, i.camel_name, i.ns_upper, i.name_upper))
        self.c.write('  ret = NULL;\n'
                     '  info = (_ExtendedGDBusPropertyInfo *) g_dbus_interface_info_lookup_property ((GDBusInterfaceInfo *) &_%s_interface_info.parent_struct, property_name);\n'
                     '  g_assert (info != NULL);\n'
                     '  pspec = g_object_class_find_property (G_OBJECT_GET_CLASS (skeleton), info->hyphen_name);\n'
                     '  if (pspec == NULL)\n'
                     '    {\n'
                     '      g_set_error (error, G_DBUS_ERROR, G_DBUS_ERROR_INVALID_ARGS, "No property with name %%s", property_name);\n'
                     '    }\n'
                     '  else\n'
                     '    {\n'
                     '      g_value_init (&value, pspec->value_type);\n'
                     '      g_object_get_property (G_OBJECT (skeleton), info->hyphen_name, &value);\n'
                     '      ret = g_dbus_gvalue_to_gvariant (&value, G_VARIANT_TYPE (info->parent_struct.signature));\n'
                     '      g_value_unset (&value);\n'
                     '    }\n'
                     '  return ret;\n'
                     '}\n'
                     '\n'
                     %(i.name_lower))

        self.c.write('static gboolean\n'
                     '_%s_skeleton_handle_set_property (\n'
                     '  GDBusConnection *connection G_GNUC_UNUSED,\n'
                     '  const gchar *sender G_GNUC_UNUSED,\n'
                     '  const gchar *object_path G_GNUC_UNUSED,\n'
                     '  const gchar *interface_name G_GNUC_UNUSED,\n'
                     '  const gchar *property_name,\n'
                     '  GVariant *variant,\n'
                     '  GError **error,\n'
                     '  gpointer user_data)\n'
                     '{\n'
                     '  %sSkeleton *skeleton = %s%s_SKELETON (user_data);\n'
                     '  GValue value = G_VALUE_INIT;\n'
                     '  GParamSpec *pspec;\n'
                     '  _ExtendedGDBusPropertyInfo *info;\n'
                     '  gboolean ret;\n'
                     %(i.name_lower, i.camel_name, i.ns_upper, i.name_upper))
        self.c.write('  ret = FALSE;\n'
                     '  info = (_ExtendedGDBusPropertyInfo *) g_dbus_interface_info_lookup_property ((GDBusInterfaceInfo *) &_%s_interface_info.parent_struct, property_name);\n'
                     '  g_assert (info != NULL);\n'
                     '  pspec = g_object_class_find_property (G_OBJECT_GET_CLASS (skeleton), info->hyphen_name);\n'
                     '  if (pspec == NULL)\n'
                     '    {\n'
                     '      g_set_error (error, G_DBUS_ERROR, G_DBUS_ERROR_INVALID_ARGS, "No property with name %%s", property_name);\n'
                     '    }\n'
                     '  else\n'
                     '    {\n'
                     '      if (info->use_gvariant)\n'
                     '        g_value_set_variant (&value, variant);\n'
                     '      else\n'
                     '        g_dbus_gvariant_to_gvalue (variant, &value);\n'
                     '      g_object_set_property (G_OBJECT (skeleton), info->hyphen_name, &value);\n'
                     '      g_value_unset (&value);\n'
                     '      ret = TRUE;\n'
                     '    }\n'
                     '  return ret;\n'
                     '}\n'
                     '\n'
                     %(i.name_lower))


        self.c.write('static const GDBusInterfaceVTable _%s_skeleton_vtable =\n'
                     '{\n'
                     '  _%s_skeleton_handle_method_call,\n'
                     '  _%s_skeleton_handle_get_property,\n'
                     '  _%s_skeleton_handle_set_property,\n'
                     '  {NULL}\n'
                     '};\n'
                     '\n'%(i.name_lower, i.name_lower, i.name_lower, i.name_lower))

        self.c.write('static GDBusInterfaceInfo *\n'
                     '%s_skeleton_dbus_interface_get_info (GDBusInterfaceSkeleton *skeleton G_GNUC_UNUSED)\n'
                     '{\n'
                     '  return %s_interface_info ();\n'
                     %(i.name_lower, i.name_lower))
        self.c.write('}\n'
                     '\n')

        self.c.write('static GDBusInterfaceVTable *\n'
                     '%s_skeleton_dbus_interface_get_vtable (GDBusInterfaceSkeleton *skeleton G_GNUC_UNUSED)\n'
                     '{\n'
                     '  return (GDBusInterfaceVTable *) &_%s_skeleton_vtable;\n'
                     %(i.name_lower, i.name_lower))
        self.c.write('}\n'
                     '\n')

        self.c.write('static GVariant *\n'
                     '%s_skeleton_dbus_interface_get_properties (GDBusInterfaceSkeleton *_skeleton)\n'
                     '{\n'
                     '  %sSkeleton *skeleton = %s%s_SKELETON (_skeleton);\n'
                     %(i.name_lower, i.camel_name, i.ns_upper, i.name_upper))
        self.c.write('\n'
                     '  GVariantBuilder builder;\n'
                     '  guint n;\n'
                     '  g_variant_builder_init (&builder, G_VARIANT_TYPE ("a{sv}"));\n'
                     '  if (_%s_interface_info.parent_struct.properties == NULL)\n'
                     '    goto out;\n'
                     '  for (n = 0; _%s_interface_info.parent_struct.properties[n] != NULL; n++)\n'
                     '    {\n'
                     '      GDBusPropertyInfo *info = _%s_interface_info.parent_struct.properties[n];\n'
                     '      if (info->flags & G_DBUS_PROPERTY_INFO_FLAGS_READABLE)\n'
                     '        {\n'
                     '          GVariant *value;\n'
                     '          value = _%s_skeleton_handle_get_property (g_dbus_interface_skeleton_get_connection (G_DBUS_INTERFACE_SKELETON (skeleton)), NULL, g_dbus_interface_skeleton_get_object_path (G_DBUS_INTERFACE_SKELETON (skeleton)), "%s", info->name, NULL, skeleton);\n'
                     '          if (value != NULL)\n'
                     '            {\n'
                     '              g_variant_take_ref (value);\n'
                     '              g_variant_builder_add (&builder, "{sv}", info->name, value);\n'
                     '              g_variant_unref (value);\n'
                     '            }\n'
                     '        }\n'
                     '    }\n'
                     'out:\n'
                     '  return g_variant_builder_end (&builder);\n'
                     '}\n'
                     '\n'
                     %(i.name_lower, i.name_lower, i.name_lower, i.name_lower, i.name))

        if len(i.properties) > 0:
            self.c.write('static gboolean _%s_emit_changed (gpointer user_data);\n'
                         '\n'
                         %(i.name_lower))

        self.c.write('static void\n'
                     '%s_skeleton_dbus_interface_flush (GDBusInterfaceSkeleton *_skeleton)\n'
                     '{\n'
                     %(i.name_lower))
        if len(i.properties) > 0:
            self.c.write('  %sSkeleton *skeleton = %s%s_SKELETON (_skeleton);\n'
                         '  gboolean emit_changed = FALSE;\n'
                         '\n'
                         '  g_mutex_lock (&skeleton->priv->lock);\n'
                         '  if (skeleton->priv->changed_properties_idle_source != NULL)\n'
                         '    {\n'
                         '      g_source_destroy (skeleton->priv->changed_properties_idle_source);\n'
                         '      skeleton->priv->changed_properties_idle_source = NULL;\n'
                         '      emit_changed = TRUE;\n'
                         '    }\n'
                         '  g_mutex_unlock (&skeleton->priv->lock);\n'
                         '\n'
                         '  if (emit_changed)\n'
                         '    _%s_emit_changed (skeleton);\n'
                         %(i.camel_name, i.ns_upper, i.name_upper, i.name_lower))
        self.c.write('}\n'
                     '\n')

        for s in i.signals:
            self.c.write('static void\n'
                         '_%s_on_signal_%s (\n'
                         '    %s *object'%(i.name_lower, s.name_lower, i.camel_name))
            for a in s.args:
                self.c.write(',\n    %sarg_%s'%(a.ctype_in, a.name))
            self.c.write(')\n'
                         '{\n'
                         '  %sSkeleton *skeleton = %s%s_SKELETON (object);\n\n'
                         '  GList      *connections, *l;\n'
                         '  GVariant   *signal_variant;\n'
                         '  connections = g_dbus_interface_skeleton_get_connections (G_DBUS_INTERFACE_SKELETON (skeleton));\n'
                         %(i.camel_name, i.ns_upper, i.name_upper))
            self.c.write('\n'
                         '  signal_variant = g_variant_ref_sink (g_variant_new ("(')
            for a in s.args:
                self.c.write('%s'%(a.format_in))
            self.c.write(')"')
            for a in s.args:
                self.c.write(',\n                   arg_%s'%(a.name))
            self.c.write('));\n')

            self.c.write('  for (l = connections; l != NULL; l = l->next)\n'
                         '    {\n'
                         '      GDBusConnection *connection = l->data;\n'
                         '      g_dbus_connection_emit_signal (connection,\n'
                         '        NULL, g_dbus_interface_skeleton_get_object_path (G_DBUS_INTERFACE_SKELETON (skeleton)), "%s", "%s",\n'
                         '        signal_variant, NULL);\n'
                         '    }\n'
                         %(i.name, s.name))
            self.c.write('  g_variant_unref (signal_variant);\n')
            self.c.write('  g_list_free_full (connections, g_object_unref);\n')
            self.c.write('}\n'
                         '\n')

        self.c.write('static void %s_skeleton_iface_init (%sIface *iface);\n'
                     %(i.name_lower, i.camel_name))

        self.c.write('#if GLIB_VERSION_MAX_ALLOWED >= GLIB_VERSION_2_38\n')
        self.c.write('G_DEFINE_TYPE_WITH_CODE (%sSkeleton, %s_skeleton, G_TYPE_DBUS_INTERFACE_SKELETON,\n'%(i.camel_name, i.name_lower))
        self.c.write('                         G_ADD_PRIVATE (%sSkeleton)\n'%(i.camel_name))
        self.c.write('                         G_IMPLEMENT_INTERFACE (%sTYPE_%s, %s_skeleton_iface_init));\n\n'%(i.ns_upper, i.name_upper, i.name_lower))
        self.c.write('#else\n')
        self.c.write('G_DEFINE_TYPE_WITH_CODE (%sSkeleton, %s_skeleton, G_TYPE_DBUS_INTERFACE_SKELETON,\n'%(i.camel_name, i.name_lower))
        self.c.write('                         G_IMPLEMENT_INTERFACE (%sTYPE_%s, %s_skeleton_iface_init));\n\n'%(i.ns_upper, i.name_upper, i.name_lower))
        self.c.write('#endif\n')

        # finalize
        self.c.write('static void\n'
                     '%s_skeleton_finalize (GObject *object)\n'
                     '{\n'%(i.name_lower))
        self.c.write('  %sSkeleton *skeleton = %s%s_SKELETON (object);\n'%(i.camel_name, i.ns_upper, i.name_upper))
        if len(i.properties) > 0:
            self.c.write('  guint n;\n'
                         '  for (n = 0; n < %d; n++)\n'
                         '    g_value_unset (&skeleton->priv->properties[n]);\n'%(len(i.properties)))
            self.c.write('  g_free (skeleton->priv->properties);\n')
        self.c.write('  g_list_free_full (skeleton->priv->changed_properties, (GDestroyNotify) _changed_property_free);\n')
        self.c.write('  if (skeleton->priv->changed_properties_idle_source != NULL)\n')
        self.c.write('    g_source_destroy (skeleton->priv->changed_properties_idle_source);\n')
        self.c.write('  g_main_context_unref (skeleton->priv->context);\n')
        self.c.write('  g_mutex_clear (&skeleton->priv->lock);\n')
        self.c.write('  G_OBJECT_CLASS (%s_skeleton_parent_class)->finalize (object);\n'
                     '}\n'
                     '\n'%(i.name_lower))

        # property accessors (TODO: generate PropertiesChanged signals in setter)
        if len(i.properties) > 0:
            self.c.write('static void\n'
                         '%s_skeleton_get_property (GObject      *object,\n'
                         '  guint         prop_id,\n'
                         '  GValue       *value,\n'
                         '  GParamSpec   *pspec G_GNUC_UNUSED)\n'
                         '{\n'%(i.name_lower))
            self.c.write('  %sSkeleton *skeleton = %s%s_SKELETON (object);\n'
                         '  g_assert (prop_id != 0 && prop_id - 1 < %d);\n'
                         '  g_mutex_lock (&skeleton->priv->lock);\n'
                         '  g_value_copy (&skeleton->priv->properties[prop_id - 1], value);\n'
                         '  g_mutex_unlock (&skeleton->priv->lock);\n'
                         %(i.camel_name, i.ns_upper, i.name_upper, len(i.properties)))
            self.c.write('}\n'
                         '\n')

            # if property is already scheduled then re-use entry.. though it could be
            # that the user did
            #
            #  foo_set_prop_bar (object, "");
            #  foo_set_prop_bar (object, "blah");
            #
            # say, every update... In this case, where nothing happens, we obviously
            # don't want a PropertiesChanged() event. We can easily check for this
            # by comparing against the _original value_ recorded before the first
            # change event. If the latest value is not different from the original
            # one, we can simply ignore the ChangedProperty
            #
            self.c.write('static gboolean\n'
                         '_%s_emit_changed (gpointer user_data)\n'
                         '{\n'
                         '  %sSkeleton *skeleton = %s%s_SKELETON (user_data);\n'
                         %(i.name_lower, i.camel_name, i.ns_upper, i.name_upper))
            self.c.write('  GList *l;\n'
                         '  GVariantBuilder builder;\n'
                         '  GVariantBuilder invalidated_builder;\n'
                         '  guint num_changes;\n'
                         '\n'
                         '  g_mutex_lock (&skeleton->priv->lock);\n'
                         '  g_variant_builder_init (&builder, G_VARIANT_TYPE ("a{sv}"));\n'
                         '  g_variant_builder_init (&invalidated_builder, G_VARIANT_TYPE ("as"));\n'
                         '  for (l = skeleton->priv->changed_properties, num_changes = 0; l != NULL; l = l->next)\n'
                         '    {\n'
                         '      ChangedProperty *cp = l->data;\n'
                         '      GVariant *variant;\n'
                         '      const GValue *cur_value;\n'
                         '\n'
                         '      cur_value = &skeleton->priv->properties[cp->prop_id - 1];\n'
                         '      if (!_g_value_equal (cur_value, &cp->orig_value))\n'
                         '        {\n'
                         '          variant = g_dbus_gvalue_to_gvariant (cur_value, G_VARIANT_TYPE (cp->info->parent_struct.signature));\n'
                         '          g_variant_builder_add (&builder, "{sv}", cp->info->parent_struct.name, variant);\n'
                         '          g_variant_unref (variant);\n'
                         '          num_changes++;\n'
                         '        }\n'
                         '    }\n'
                         '  if (num_changes > 0)\n'
                         '    {\n'
                         '      GList *connections, *ll;\n'
                         '      GVariant *signal_variant;'
                         '\n'
                         '      signal_variant = g_variant_ref_sink (g_variant_new ("(sa{sv}as)", "%s",\n'
                         '                                           &builder, &invalidated_builder));\n'
                         '      connections = g_dbus_interface_skeleton_get_connections (G_DBUS_INTERFACE_SKELETON (skeleton));\n'
                         '      for (ll = connections; ll != NULL; ll = ll->next)\n'
                         '        {\n'
                         '          GDBusConnection *connection = ll->data;\n'
                         '\n'
                         '          g_dbus_connection_emit_signal (connection,\n'
                         '                                         NULL, g_dbus_interface_skeleton_get_object_path (G_DBUS_INTERFACE_SKELETON (skeleton)),\n'
                         '                                         "org.freedesktop.DBus.Properties",\n'
                         '                                         "PropertiesChanged",\n'
                         '                                         signal_variant,\n'
                         '                                         NULL);\n'
                         '        }\n'
                         '      g_variant_unref (signal_variant);\n'
                         '      g_list_free_full (connections, g_object_unref);\n'
                         '    }\n'
                         '  else\n'
                         '    {\n'
                         '      g_variant_builder_clear (&builder);\n'
                         '      g_variant_builder_clear (&invalidated_builder);\n'
                         '    }\n'
                         %(i.name))
            self.c.write('  g_list_free_full (skeleton->priv->changed_properties, (GDestroyNotify) _changed_property_free);\n')
            self.c.write('  skeleton->priv->changed_properties = NULL;\n')
            self.c.write('  skeleton->priv->changed_properties_idle_source = NULL;\n')
            self.c.write('  g_mutex_unlock (&skeleton->priv->lock);\n')
            self.c.write('  return FALSE;\n'
                         '}\n'
                         '\n')
            # holding lock while being called
            self.c.write('static void\n'
                         '_%s_schedule_emit_changed (%sSkeleton *skeleton, const _ExtendedGDBusPropertyInfo *info, guint prop_id, const GValue *orig_value)\n'
                         '{\n'
                         '  ChangedProperty *cp;\n'
                         '  GList *l;\n'
                         '  cp = NULL;\n'
                         '  for (l = skeleton->priv->changed_properties; l != NULL; l = l->next)\n'
                         '    {\n'
                         '      ChangedProperty *i_cp = l->data;\n'
                         '      if (i_cp->info == info)\n'
                         '        {\n'
                         '          cp = i_cp;\n'
                         '          break;\n'
                         '        }\n'
                         '    }\n'
                         %(i.name_lower, i.camel_name))
            self.c.write('  if (cp == NULL)\n'
                         '    {\n'
                         '      cp = g_new0 (ChangedProperty, 1);\n'
                         '      cp->prop_id = prop_id;\n'
                         '      cp->info = info;\n'
                         '      skeleton->priv->changed_properties = g_list_prepend (skeleton->priv->changed_properties, cp);\n'
                         '      g_value_init (&cp->orig_value, G_VALUE_TYPE (orig_value));\n'
                         '      g_value_copy (orig_value, &cp->orig_value);\n'
                         '    }\n'
                         '}\n'
                         '\n'
                         %())

            # Postpone setting up the refresh source until the ::notify signal is emitted as
            # this allows use of g_object_freeze_notify()/g_object_thaw_notify() ...
            # This is useful when updating several properties from another thread than
            # where the idle will be emitted from
            self.c.write('static void\n'
                         '%s_skeleton_notify (GObject      *object,\n'
                         '  GParamSpec *pspec G_GNUC_UNUSED)\n'
                         '{\n'
                         '  %sSkeleton *skeleton = %s%s_SKELETON (object);\n'
                         '  g_mutex_lock (&skeleton->priv->lock);\n'
                         '  if (skeleton->priv->changed_properties != NULL &&\n'
                         '      skeleton->priv->changed_properties_idle_source == NULL)\n'
                         '    {\n'
                         '      skeleton->priv->changed_properties_idle_source = g_idle_source_new ();\n'
                         '      g_source_set_priority (skeleton->priv->changed_properties_idle_source, G_PRIORITY_DEFAULT);\n'
                         '      g_source_set_callback (skeleton->priv->changed_properties_idle_source, _%s_emit_changed, g_object_ref (skeleton), (GDestroyNotify) g_object_unref);\n'
                         '      g_source_attach (skeleton->priv->changed_properties_idle_source, skeleton->priv->context);\n'
                         '      g_source_unref (skeleton->priv->changed_properties_idle_source);\n'
                         '    }\n'
                         '  g_mutex_unlock (&skeleton->priv->lock);\n'
                         '}\n'
                         '\n'
                         %(i.name_lower, i.camel_name, i.ns_upper, i.name_upper, i.name_lower))

            self.c.write('static void\n'
                         '%s_skeleton_set_property (GObject      *object,\n'
                         '  guint         prop_id,\n'
                         '  const GValue *value,\n'
                         '  GParamSpec   *pspec)\n'
                         '{\n'%(i.name_lower))
            self.c.write('  %sSkeleton *skeleton = %s%s_SKELETON (object);\n'
                         '  g_assert (prop_id != 0 && prop_id - 1 < %d);\n'
                         '  g_mutex_lock (&skeleton->priv->lock);\n'
                         '  g_object_freeze_notify (object);\n'
                         '  if (!_g_value_equal (value, &skeleton->priv->properties[prop_id - 1]))\n'
                         '    {\n'
                         '      if (g_dbus_interface_skeleton_get_connection (G_DBUS_INTERFACE_SKELETON (skeleton)) != NULL)\n'
                         '        _%s_schedule_emit_changed (skeleton, _%s_property_info_pointers[prop_id - 1], prop_id, &skeleton->priv->properties[prop_id - 1]);\n'
                         '      g_value_copy (value, &skeleton->priv->properties[prop_id - 1]);\n'
                         '      g_object_notify_by_pspec (object, pspec);\n'
                         '    }\n'
                         '  g_mutex_unlock (&skeleton->priv->lock);\n'
                         '  g_object_thaw_notify (object);\n'
                         %(i.camel_name, i.ns_upper, i.name_upper, len(i.properties), i.name_lower, i.name_lower))
            self.c.write('}\n'
                         '\n')

        self.c.write('static void\n'
                     '%s_skeleton_init (%sSkeleton *skeleton)\n'
                     '{\n'
                     '#if GLIB_VERSION_MAX_ALLOWED >= GLIB_VERSION_2_38\n'
                     '  skeleton->priv = %s_skeleton_get_instance_private (skeleton);\n'
                     '#else\n'
                     '  skeleton->priv = G_TYPE_INSTANCE_GET_PRIVATE (skeleton, %sTYPE_%s_SKELETON, %sSkeletonPrivate);\n'
                     '#endif\n\n'
                     %(i.name_lower, i.camel_name,
                       i.name_lower,
                       i.ns_upper, i.name_upper, i.camel_name))
        self.c.write('  g_mutex_init (&skeleton->priv->lock);\n')
        self.c.write('  skeleton->priv->context = g_main_context_ref_thread_default ();\n')
        if len(i.properties) > 0:
            self.c.write('  skeleton->priv->properties = g_new0 (GValue, %d);\n'%(len(i.properties)))
            n = 0
            for p in i.properties:
                self.c.write('  g_value_init (&skeleton->priv->properties[%d], %s);\n'%(n, p.arg.gtype))
                n += 1
        self.c.write('}\n'
                     '\n')

        # property vfuncs
        n = 0
        for p in i.properties:
            self.c.write('static %s\n'
                         '%s_skeleton_get_%s (%s *object)\n'
                         '{\n'
                         %(p.arg.ctype_in, i.name_lower, p.name_lower, i.camel_name))
            self.c.write('  %sSkeleton *skeleton = %s%s_SKELETON (object);\n'%(i.camel_name, i.ns_upper, i.name_upper))
            self.c.write('  %svalue;\n'
                         '  g_mutex_lock (&skeleton->priv->lock);\n'
                         '  value = %s (&(skeleton->priv->properties[%d]));\n'
                         '  g_mutex_unlock (&skeleton->priv->lock);\n'
                         %(p.arg.ctype_in_g, p.arg.gvalue_get, n))
            self.c.write('  return value;\n')
            self.c.write('}\n')
            self.c.write('\n')
            n += 1

        self.c.write('static void\n'
                     '%s_skeleton_class_init (%sSkeletonClass *klass)\n'
                     '{\n'
                     '  GObjectClass *gobject_class;\n'
                     '  GDBusInterfaceSkeletonClass *skeleton_class;\n'
                     '\n'
                     '  gobject_class = G_OBJECT_CLASS (klass);\n'
                     '  gobject_class->finalize = %s_skeleton_finalize;\n'
                     %(i.name_lower, i.camel_name, i.name_lower))
        if len(i.properties) > 0:
            self.c.write('  gobject_class->get_property = %s_skeleton_get_property;\n'
                         '  gobject_class->set_property = %s_skeleton_set_property;\n'
                         '  gobject_class->notify       = %s_skeleton_notify;\n'
                         '\n'%(i.name_lower, i.name_lower, i.name_lower))
            self.c.write('\n'
                         '  %s_override_properties (gobject_class, 1);\n'%(i.name_lower))
        self.c.write('\n'
                     '  skeleton_class = G_DBUS_INTERFACE_SKELETON_CLASS (klass);\n');
        self.c.write('  skeleton_class->get_info = %s_skeleton_dbus_interface_get_info;\n'%(i.name_lower))
        self.c.write('  skeleton_class->get_properties = %s_skeleton_dbus_interface_get_properties;\n'%(i.name_lower))
        self.c.write('  skeleton_class->flush = %s_skeleton_dbus_interface_flush;\n'%(i.name_lower))
        self.c.write('  skeleton_class->get_vtable = %s_skeleton_dbus_interface_get_vtable;\n'%(i.name_lower))

        self.c.write('\n'
                     '#if GLIB_VERSION_MAX_ALLOWED < GLIB_VERSION_2_38\n'
                     '  g_type_class_add_private (klass, sizeof (%sSkeletonPrivate));\n'
                     '#endif\n'%(i.camel_name))

        self.c.write('}\n'
                     '\n')

        self.c.write('static void\n'
                     '%s_skeleton_iface_init (%sIface *iface)\n'
                     '{\n'
                     %(i.name_lower, i.camel_name))
        for s in i.signals:
            self.c.write('  iface->%s = _%s_on_signal_%s;\n'
                         %(s.name_lower, i.name_lower, s.name_lower))
        for p in i.properties:
            self.c.write('  iface->get_%s = %s_skeleton_get_%s;\n'%(p.name_lower, i.name_lower, p.name_lower))
        self.c.write('}\n'
                     '\n')

        # constructors
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %s_skeleton_new:\n'
                ' *\n'
                ' * Creates a skeleton object for the D-Bus interface #%s.\n'
                ' *\n'
                ' * Returns: (transfer full) (type %sSkeleton): The skeleton object.\n'
                %(i.name_lower, i.name, i.camel_name), False))
        self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
        self.c.write('%s *\n'
                     '%s_skeleton_new (void)\n'
                     '{\n'
                     '  return %s%s (g_object_new (%sTYPE_%s_SKELETON, NULL));\n'
                     '}\n'
                     '\n'%(i.camel_name, i.name_lower, i.ns_upper, i.name_upper, i.ns_upper, i.name_upper))

    # ---------------------------------------------------------------------------------------------------

    def generate_object(self):
        self.c.write('/* ------------------------------------------------------------------------\n'
                     ' * Code for Object, ObjectProxy and ObjectSkeleton\n'
                     ' * ------------------------------------------------------------------------\n'
                     ' */\n'
                     '\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * SECTION:%sObject\n'
                ' * @title: %sObject\n'
                ' * @short_description: Specialized GDBusObject types\n'
                ' *\n'
                ' * This section contains the #%sObject, #%sObjectProxy, and #%sObjectSkeleton types which make it easier to work with objects implementing generated types for D-Bus interfaces.\n'
                ' */\n'
                %(self.namespace, self.namespace, self.namespace, self.namespace, self.namespace), False))
        self.c.write('\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sObject:\n'
                ' *\n'
                ' * The #%sObject type is a specialized container of interfaces.\n'
                ' */\n'
                %(self.namespace, self.namespace), False))
        self.c.write('\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sObjectIface:\n'
                ' * @parent_iface: The parent interface.\n'
                ' *\n'
                ' * Virtual table for the #%sObject interface.\n'
                ' */\n'
                %(self.namespace, self.namespace), False))
        self.c.write('\n')

        self.c.write('typedef %sObjectIface %sObjectInterface;\n'%(self.namespace, self.namespace))
        self.c.write('G_DEFINE_INTERFACE_WITH_CODE (%sObject, %sobject, G_TYPE_OBJECT, g_type_interface_add_prerequisite (g_define_type_id, G_TYPE_DBUS_OBJECT));\n'%(self.namespace, self.ns_lower))
        self.c.write('\n')
        self.c.write('static void\n'
                     '%sobject_default_init (%sObjectIface *iface)\n'
                     '{\n'
                     %(self.ns_lower, self.namespace));
        for i in self.ifaces:
            self.c.write(self.docbook_gen.expand(
                    '  /**\n'
                    '   * %sObject:%s:\n'
                    '   *\n'
                    '   * The #%s instance corresponding to the D-Bus interface #%s, if any.\n'
                    '   *\n'
                    '   * Connect to the #GObject::notify signal to get informed of property changes.\n'
                    %(self.namespace, i.name_hyphen, i.camel_name, i.name), False))
            self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 2)
            self.c.write('  g_object_interface_install_property (iface, g_param_spec_object ("%s", "%s", "%s", %sTYPE_%s, G_PARAM_READWRITE|G_PARAM_STATIC_STRINGS));\n'
                         '\n'
                         %(i.name_hyphen, i.name_hyphen, i.name_hyphen, self.ns_upper, i.name_upper))
        self.c.write('}\n'
                     '\n')

        for i in self.ifaces:
            self.c.write(self.docbook_gen.expand(
                    '/**\n'
                    ' * %sobject_get_%s:\n'
                    ' * @object: A #%sObject.\n'
                    ' *\n'
                    ' * Gets the #%s instance for the D-Bus interface #%s on @object, if any.\n'
                    ' *\n'
                    ' * Returns: (transfer full): A #%s that must be freed with g_object_unref() or %%NULL if @object does not implement the interface.\n'
                    %(self.ns_lower, i.name_upper.lower(), self.namespace, i.camel_name, i.name, i.camel_name), False))
            self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
            self.c.write ('%s *%sobject_get_%s (%sObject *object)\n'
                          %(i.camel_name, self.ns_lower, i.name_upper.lower(), self.namespace))
            self.c.write('{\n'
                         '  GDBusInterface *ret;\n'
                         '  ret = g_dbus_object_get_interface (G_DBUS_OBJECT (object), "%s");\n'
                         '  if (ret == NULL)\n'
                         '    return NULL;\n'
                         '  return %s%s (ret);\n'
                         '}\n'
                         '\n'
                         %(i.name, self.ns_upper, i.name_upper))
        self.c.write('\n')
        for i in self.ifaces:
            self.c.write(self.docbook_gen.expand(
                    '/**\n'
                    ' * %sobject_peek_%s: (skip)\n'
                    ' * @object: A #%sObject.\n'
                    ' *\n'
                    ' * Like %sobject_get_%s() but doesn\'t increase the reference count on the returned object.\n'
                    ' *\n'
                    ' * <warning>It is not safe to use the returned object if you are on another thread than the one where the #GDBusObjectManagerClient or #GDBusObjectManagerServer for @object is running.</warning>\n'
                    ' *\n'
                    ' * Returns: (transfer none): A #%s or %%NULL if @object does not implement the interface. Do not free the returned object, it is owned by @object.\n'
                    %(self.ns_lower, i.name_upper.lower(), self.namespace, self.ns_lower, i.name_upper.lower(), i.camel_name), False))
            self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
            self.c.write ('%s *%sobject_peek_%s (%sObject *object)\n'
                          %(i.camel_name, self.ns_lower, i.name_upper.lower(), self.namespace))
            self.c.write('{\n'
                         '  GDBusInterface *ret;\n'
                         '  ret = g_dbus_object_get_interface (G_DBUS_OBJECT (object), "%s");\n'
                         '  if (ret == NULL)\n'
                         '    return NULL;\n'
                         '  g_object_unref (ret);\n'
                         '  return %s%s (ret);\n'
                         '}\n'
                         '\n'
                         %(i.name, self.ns_upper, i.name_upper))
        self.c.write('\n')
        # shared by ObjectProxy and ObjectSkeleton classes
        self.c.write('static void\n'
                     '%sobject_notify (GDBusObject *object, GDBusInterface *interface)\n'
                     '{\n'
                     '  g_object_notify (G_OBJECT (object), ((_ExtendedGDBusInterfaceInfo *) g_dbus_interface_get_info (interface))->hyphen_name);\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower))

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sObjectProxy:\n'
                ' *\n'
                ' * The #%sObjectProxy structure contains only private data and should only be accessed using the provided API.\n'
                %(self.namespace, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('\n')
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sObjectProxyClass:\n'
                ' * @parent_class: The parent class.\n'
                ' *\n'
                ' * Class structure for #%sObjectProxy.\n'
                %(self.namespace, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('\n')
        # class boilerplate
        self.c.write('static void\n'
                     '%sobject_proxy__%sobject_iface_init (%sObjectIface *iface G_GNUC_UNUSED)\n'
                     '{\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower, self.ns_lower, self.namespace))
        self.c.write('static void\n'
                     '%sobject_proxy__g_dbus_object_iface_init (GDBusObjectIface *iface)\n'
                     '{\n'
                     '  iface->interface_added = %sobject_notify;\n'
                     '  iface->interface_removed = %sobject_notify;\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower, self.ns_lower, self.ns_lower))
        self.c.write('\n')
        self.c.write('G_DEFINE_TYPE_WITH_CODE (%sObjectProxy, %sobject_proxy, G_TYPE_DBUS_OBJECT_PROXY,\n'
                     '                         G_IMPLEMENT_INTERFACE (%sTYPE_OBJECT, %sobject_proxy__%sobject_iface_init)\n'
                     '                         G_IMPLEMENT_INTERFACE (G_TYPE_DBUS_OBJECT, %sobject_proxy__g_dbus_object_iface_init));\n'
                     '\n'
                     %(self.namespace, self.ns_lower, self.ns_upper, self.ns_lower, self.ns_lower, self.ns_lower))
        # class boilerplate
        self.c.write('static void\n'
                     '%sobject_proxy_init (%sObjectProxy *object G_GNUC_UNUSED)\n'
                     '{\n'
                     '}\n'
                     '\n'%(self.ns_lower, self.namespace))
        self.c.write('static void\n'
                     '%sobject_proxy_set_property (GObject      *gobject,\n'
                     '  guint         prop_id,\n'
                     '  const GValue *value G_GNUC_UNUSED,\n'
                     '  GParamSpec   *pspec)\n'
                     '{\n'
                     '  G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);\n'
                     %(self.ns_lower))
        self.c.write('}\n'
                     '\n'%())
        self.c.write('static void\n'
                     '%sobject_proxy_get_property (GObject      *gobject,\n'
                     '  guint         prop_id,\n'
                     '  GValue       *value,\n'
                     '  GParamSpec   *pspec)\n'
                     '{\n'
                     '  %sObjectProxy *object = %sOBJECT_PROXY (gobject);\n'
                     '  GDBusInterface *interface;\n'
                     '\n'
                     '  switch (prop_id)\n'
                     '    {\n'
                     %(self.ns_lower, self.namespace, self.ns_upper))
        n = 1
        for i in self.ifaces:
            self.c.write('    case %d:\n'
                         '      interface = g_dbus_object_get_interface (G_DBUS_OBJECT (object), "%s");\n'
                         '      g_value_take_object (value, interface);\n'
                         '      break;\n'
                         '\n'
                         %(n, i.name))
            n += 1
        self.c.write('    default:\n'
                     '      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);\n'
                     '      break;\n'
                     '  }\n'
                     '}\n'
                     '\n'%())
        self.c.write('static void\n'
                     '%sobject_proxy_class_init (%sObjectProxyClass *klass)\n'
                     '{\n'
                     '  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);\n'
                     '\n'
                     '  gobject_class->set_property = %sobject_proxy_set_property;\n'
                     '  gobject_class->get_property = %sobject_proxy_get_property;\n'
                     '\n'
                     %(self.ns_lower, self.namespace, self.ns_lower, self.ns_lower))
        n = 1
        for i in self.ifaces:
            self.c.write('  g_object_class_override_property (gobject_class, %d, "%s");'
                         '\n'
                         %(n, i.name_hyphen))
            n += 1
        self.c.write('}\n'
                     '\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sobject_proxy_new:\n'
                ' * @connection: A #GDBusConnection.\n'
                ' * @object_path: An object path.\n'
                ' *\n'
                ' * Creates a new proxy object.\n'
                ' *\n'
                ' * Returns: (transfer full): The proxy object.\n'
                ' */\n'
                %(self.ns_lower), False))
        self.c.write('%sObjectProxy *\n'
                     '%sobject_proxy_new (GDBusConnection *connection,\n'
                     '  const gchar *object_path)\n'
                     '{\n'
                     '  g_return_val_if_fail (G_IS_DBUS_CONNECTION (connection), NULL);\n'
                     '  g_return_val_if_fail (g_variant_is_object_path (object_path), NULL);\n'
                     '  return %sOBJECT_PROXY (g_object_new (%sTYPE_OBJECT_PROXY, "g-connection", connection, "g-object-path", object_path, NULL));\n'
                     '}\n'
                     '\n'%(self.namespace, self.ns_lower, self.ns_upper, self.ns_upper))

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sObjectSkeleton:\n'
                ' *\n'
                ' * The #%sObjectSkeleton structure contains only private data and should only be accessed using the provided API.\n'
                %(self.namespace, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('\n')
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sObjectSkeletonClass:\n'
                ' * @parent_class: The parent class.\n'
                ' *\n'
                ' * Class structure for #%sObjectSkeleton.\n'
                %(self.namespace, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('\n')
        # class boilerplate
        self.c.write('static void\n'
                     '%sobject_skeleton__%sobject_iface_init (%sObjectIface *iface G_GNUC_UNUSED)\n'
                     '{\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower, self.ns_lower, self.namespace))
        self.c.write('\n')
        self.c.write('static void\n'
                     '%sobject_skeleton__g_dbus_object_iface_init (GDBusObjectIface *iface)\n'
                     '{\n'
                     '  iface->interface_added = %sobject_notify;\n'
                     '  iface->interface_removed = %sobject_notify;\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower, self.ns_lower, self.ns_lower))
        self.c.write('G_DEFINE_TYPE_WITH_CODE (%sObjectSkeleton, %sobject_skeleton, G_TYPE_DBUS_OBJECT_SKELETON,\n'
                     '                         G_IMPLEMENT_INTERFACE (%sTYPE_OBJECT, %sobject_skeleton__%sobject_iface_init)\n'
                     '                         G_IMPLEMENT_INTERFACE (G_TYPE_DBUS_OBJECT, %sobject_skeleton__g_dbus_object_iface_init));\n'
                     '\n'
                     %(self.namespace, self.ns_lower, self.ns_upper, self.ns_lower, self.ns_lower, self.ns_lower))
        # class boilerplate
        self.c.write('static void\n'
                     '%sobject_skeleton_init (%sObjectSkeleton *object G_GNUC_UNUSED)\n'
                     '{\n'
                     '}\n'
                     '\n'%(self.ns_lower, self.namespace))
        self.c.write('static void\n'
                     '%sobject_skeleton_set_property (GObject      *gobject,\n'
                     '  guint         prop_id,\n'
                     '  const GValue *value,\n'
                     '  GParamSpec   *pspec)\n'
                     '{\n'
                     '  %sObjectSkeleton *object = %sOBJECT_SKELETON (gobject);\n'
                     '  GDBusInterfaceSkeleton *interface;\n'
                     '\n'
                     '  switch (prop_id)\n'
                     '    {\n'
                     %(self.ns_lower, self.namespace, self.ns_upper))
        n = 1
        for i in self.ifaces:
            self.c.write('    case %d:\n'
                         '      interface = g_value_get_object (value);\n'
                         '      if (interface != NULL)\n'
                         '        {\n'
                         '          g_warn_if_fail (%sIS_%s (interface));\n'
                         '          g_dbus_object_skeleton_add_interface (G_DBUS_OBJECT_SKELETON (object), interface);\n'
                         '        }\n'
                         '      else\n'
                         '        {\n'
                         '          g_dbus_object_skeleton_remove_interface_by_name (G_DBUS_OBJECT_SKELETON (object), "%s");\n'
                         '        }\n'
                         '      break;\n'
                         '\n'
                         %(n, self.ns_upper, i.name_upper, i.name))
            n += 1
        self.c.write('    default:\n'
                     '      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);\n'
                     '      break;\n'
                     '  }\n'
                     '}\n'
                     '\n'%())
        self.c.write('static void\n'
                     '%sobject_skeleton_get_property (GObject      *gobject,\n'
                     '  guint         prop_id,\n'
                     '  GValue       *value,\n'
                     '  GParamSpec   *pspec)\n'
                     '{\n'
                     '  %sObjectSkeleton *object = %sOBJECT_SKELETON (gobject);\n'
                     '  GDBusInterface *interface;\n'
                     '\n'
                     '  switch (prop_id)\n'
                     '    {\n'
                     %(self.ns_lower, self.namespace, self.ns_upper))
        n = 1
        for i in self.ifaces:
            self.c.write('    case %d:\n'
                         '      interface = g_dbus_object_get_interface (G_DBUS_OBJECT (object), "%s");\n'
                         '      g_value_take_object (value, interface);\n'
                         '      break;\n'
                         '\n'
                         %(n, i.name))
            n += 1
        self.c.write('    default:\n'
                     '      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);\n'
                     '      break;\n'
                     '  }\n'
                     '}\n'
                     '\n'%())
        self.c.write('static void\n'
                     '%sobject_skeleton_class_init (%sObjectSkeletonClass *klass)\n'
                     '{\n'
                     '  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);\n'
                     '\n'
                     '  gobject_class->set_property = %sobject_skeleton_set_property;\n'
                     '  gobject_class->get_property = %sobject_skeleton_get_property;\n'
                     '\n'
                     %(self.ns_lower, self.namespace, self.ns_lower, self.ns_lower))
        n = 1
        for i in self.ifaces:
            self.c.write('  g_object_class_override_property (gobject_class, %d, "%s");'
                         '\n'
                         %(n, i.name_hyphen))
            n += 1
        self.c.write('}\n'
                     '\n')
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sobject_skeleton_new:\n'
                ' * @object_path: An object path.\n'
                ' *\n'
                ' * Creates a new skeleton object.\n'
                ' *\n'
                ' * Returns: (transfer full): The skeleton object.\n'
                ' */\n'
                %(self.ns_lower), False))
        self.c.write('%sObjectSkeleton *\n'
                     '%sobject_skeleton_new (const gchar *object_path)\n'
                     '{\n'
                     '  g_return_val_if_fail (g_variant_is_object_path (object_path), NULL);\n'
                     '  return %sOBJECT_SKELETON (g_object_new (%sTYPE_OBJECT_SKELETON, "g-object-path", object_path, NULL));\n'
                     '}\n'
                     '\n'%(self.namespace, self.ns_lower, self.ns_upper, self.ns_upper))
        for i in self.ifaces:
            self.c.write(self.docbook_gen.expand(
                    '/**\n'
                    ' * %sobject_skeleton_set_%s:\n'
                    ' * @object: A #%sObjectSkeleton.\n'
                    ' * @interface_: (allow-none): A #%s or %%NULL to clear the interface.\n'
                    ' *\n'
                    ' * Sets the #%s instance for the D-Bus interface #%s on @object.\n'
                    %(self.ns_lower, i.name_upper.lower(), self.namespace, i.camel_name, i.camel_name, i.name), False))
            self.write_gtkdoc_deprecated_and_since_and_close(i, self.c, 0)
            self.c.write ('void %sobject_skeleton_set_%s (%sObjectSkeleton *object, %s *interface_)\n'
                          %(self.ns_lower, i.name_upper.lower(), self.namespace, i.camel_name))
            self.c.write('{\n'
                         '  g_object_set (G_OBJECT (object), "%s", interface_, NULL);\n'
                         '}\n'
                         '\n'
                         %(i.name_hyphen))
        self.c.write('\n')


    def generate_object_manager_client(self):
        self.c.write('/* ------------------------------------------------------------------------\n'
                     ' * Code for ObjectManager client\n'
                     ' * ------------------------------------------------------------------------\n'
                     ' */\n'
                     '\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * SECTION:%sObjectManagerClient\n'
                ' * @title: %sObjectManagerClient\n'
                ' * @short_description: Generated GDBusObjectManagerClient type\n'
                ' *\n'
                ' * This section contains a #GDBusObjectManagerClient that uses %sobject_manager_client_get_proxy_type() as the #GDBusProxyTypeFunc.\n'
                ' */\n'
                %(self.namespace, self.namespace, self.ns_lower), False))
        self.c.write('\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sObjectManagerClient:\n'
                ' *\n'
                ' * The #%sObjectManagerClient structure contains only private data and should only be accessed using the provided API.\n'
                %(self.namespace, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('\n')

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sObjectManagerClientClass:\n'
                ' * @parent_class: The parent class.\n'
                ' *\n'
                ' * Class structure for #%sObjectManagerClient.\n'
                %(self.namespace, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('\n')

        # class boilerplate
        self.c.write('G_DEFINE_TYPE (%sObjectManagerClient, %sobject_manager_client, G_TYPE_DBUS_OBJECT_MANAGER_CLIENT);\n'
                     '\n'
                     %(self.namespace, self.ns_lower))

        # class boilerplate
        self.c.write('static void\n'
                     '%sobject_manager_client_init (%sObjectManagerClient *manager G_GNUC_UNUSED)\n'
                     '{\n'
                     '}\n'
                     '\n'%(self.ns_lower, self.namespace))
        self.c.write('static void\n'
                     '%sobject_manager_client_class_init (%sObjectManagerClientClass *klass G_GNUC_UNUSED)\n'
                     '{\n'
                     '}\n'
                     '\n'%(self.ns_lower, self.namespace))

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sobject_manager_client_get_proxy_type:\n'
                ' * @manager: A #GDBusObjectManagerClient.\n'
                ' * @object_path: The object path of the remote object (unused).\n'
                ' * @interface_name: (allow-none): Interface name of the remote object or %%NULL to get the object proxy #GType.\n'
                ' * @user_data: User data (unused).\n'
                ' *\n'
                ' * A #GDBusProxyTypeFunc that maps @interface_name to the generated #GDBusObjectProxy<!-- -->- and #GDBusProxy<!-- -->-derived types.\n'
                ' *\n'
                ' * Returns: A #GDBusProxy<!-- -->-derived #GType if @interface_name is not %%NULL, otherwise the #GType for #%sObjectProxy.\n'
                %(self.ns_lower, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('GType\n'
                     '%sobject_manager_client_get_proxy_type (GDBusObjectManagerClient *manager G_GNUC_UNUSED, const gchar *object_path G_GNUC_UNUSED, const gchar *interface_name, gpointer user_data G_GNUC_UNUSED)\n'
                     '{\n'
                     %(self.ns_lower))
        self.c.write('  static gsize once_init_value = 0;\n'
                     '  static GHashTable *lookup_hash;\n'
                     '  GType ret;\n'
                     '\n'
                     '  if (interface_name == NULL)\n'
                     '    return %sTYPE_OBJECT_PROXY;\n'
                     '  if (g_once_init_enter (&once_init_value))\n'
                     '    {\n'
                     '      lookup_hash = g_hash_table_new (g_str_hash, g_str_equal);\n'
                     %(self.ns_upper))
        for i in self.ifaces:
            self.c.write('      g_hash_table_insert (lookup_hash, (gpointer) "%s", GSIZE_TO_POINTER (%sTYPE_%s_PROXY));\n'
                         %(i.name, i.ns_upper, i.name_upper))
        self.c.write('      g_once_init_leave (&once_init_value, 1);\n'
                     '    }\n')
        self.c.write('  ret = (GType) GPOINTER_TO_SIZE (g_hash_table_lookup (lookup_hash, interface_name));\n'
                     '  if (ret == (GType) 0)\n'
                     '    ret = G_TYPE_DBUS_PROXY;\n')
        self.c.write('  return ret;\n'
                     '}\n'
                     '\n')

        # constructors
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sobject_manager_client_new:\n'
                ' * @connection: A #GDBusConnection.\n'
                ' * @flags: Flags from the #GDBusObjectManagerClientFlags enumeration.\n'
                ' * @name: (allow-none): A bus name (well-known or unique) or %%NULL if @connection is not a message bus connection.\n'
                ' * @object_path: An object path.\n'
                ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                ' * @callback: A #GAsyncReadyCallback to call when the request is satisfied.\n'
                ' * @user_data: User data to pass to @callback.\n'
                ' *\n'
                ' * Asynchronously creates #GDBusObjectManagerClient using %sobject_manager_client_get_proxy_type() as the #GDBusProxyTypeFunc. See g_dbus_object_manager_client_new() for more details.\n'
                ' *\n'
                ' * When the operation is finished, @callback will be invoked in the <link linkend="g-main-context-push-thread-default">thread-default main loop</link> of the thread you are calling this method from.\n'
                ' * You can then call %sobject_manager_client_new_finish() to get the result of the operation.\n'
                ' *\n'
                ' * See %sobject_manager_client_new_sync() for the synchronous, blocking version of this constructor.\n'
                %(self.ns_lower, self.ns_lower, self.ns_lower, self.ns_lower), False))
        self.c.write(' */\n')
        self.c.write('void\n'
                     '%sobject_manager_client_new (\n'
                     '    GDBusConnection        *connection,\n'
                     '    GDBusObjectManagerClientFlags  flags,\n'
                     '    const gchar            *name,\n'
                     '    const gchar            *object_path,\n'
                     '    GCancellable           *cancellable,\n'
                     '    GAsyncReadyCallback     callback,\n'
                     '    gpointer                user_data)\n'
                     '{\n'
                     '  g_async_initable_new_async (%sTYPE_OBJECT_MANAGER_CLIENT, G_PRIORITY_DEFAULT, cancellable, callback, user_data, "flags", flags, "name", name, "connection", connection, "object-path", object_path, "get-proxy-type-func", %sobject_manager_client_get_proxy_type, NULL);\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower, self.ns_upper, self.ns_lower))
        self.c.write('/**\n'
                     ' * %sobject_manager_client_new_finish:\n'
                     ' * @res: The #GAsyncResult obtained from the #GAsyncReadyCallback passed to %sobject_manager_client_new().\n'
                     ' * @error: Return location for error or %%NULL\n'
                     ' *\n'
                     ' * Finishes an operation started with %sobject_manager_client_new().\n'
                     ' *\n'
                     ' * Returns: (transfer full) (type %sObjectManagerClient): The constructed object manager client or %%NULL if @error is set.\n'
                     %(self.ns_lower, self.ns_lower, self.ns_lower, self.namespace))
        self.c.write(' */\n')
        self.c.write('GDBusObjectManager *\n'
                     '%sobject_manager_client_new_finish (\n'
                     '    GAsyncResult        *res,\n'
                     '    GError             **error)\n'
                     '{\n'
                     '  GObject *ret;\n'
                     '  GObject *source_object;\n'
                     '  source_object = g_async_result_get_source_object (res);\n'
                     '  ret = g_async_initable_new_finish (G_ASYNC_INITABLE (source_object), res, error);\n'
                     '  g_object_unref (source_object);\n'
                     '  if (ret != NULL)\n'
                     '    return G_DBUS_OBJECT_MANAGER (ret);\n'
                     '  else\n'
                     '    return NULL;\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower))
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sobject_manager_client_new_sync:\n'
                ' * @connection: A #GDBusConnection.\n'
                ' * @flags: Flags from the #GDBusObjectManagerClientFlags enumeration.\n'
                ' * @name: (allow-none): A bus name (well-known or unique) or %%NULL if @connection is not a message bus connection.\n'
                ' * @object_path: An object path.\n'
                ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                ' * @error: Return location for error or %%NULL\n'
                ' *\n'
                ' * Synchronously creates #GDBusObjectManagerClient using %sobject_manager_client_get_proxy_type() as the #GDBusProxyTypeFunc. See g_dbus_object_manager_client_new_sync() for more details.\n'
                ' *\n'
                ' * The calling thread is blocked until a reply is received.\n'
                ' *\n'
                ' * See %sobject_manager_client_new() for the asynchronous version of this constructor.\n'
                ' *\n'
                ' * Returns: (transfer full) (type %sObjectManagerClient): The constructed object manager client or %%NULL if @error is set.\n'
                %(self.ns_lower, self.ns_lower, self.ns_lower, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('GDBusObjectManager *\n'
                     '%sobject_manager_client_new_sync (\n'
                     '    GDBusConnection        *connection,\n'
                     '    GDBusObjectManagerClientFlags  flags,\n'
                     '    const gchar            *name,\n'
                     '    const gchar            *object_path,\n'
                     '    GCancellable           *cancellable,\n'
                     '    GError                **error)\n'
                     '{\n'
                     '  GInitable *ret;\n'
                     '  ret = g_initable_new (%sTYPE_OBJECT_MANAGER_CLIENT, cancellable, error, "flags", flags, "name", name, "connection", connection, "object-path", object_path, "get-proxy-type-func", %sobject_manager_client_get_proxy_type, NULL);\n'
                     '  if (ret != NULL)\n'
                     '    return G_DBUS_OBJECT_MANAGER (ret);\n'
                     '  else\n'
                     '    return NULL;\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower, self.ns_upper, self.ns_lower))
        self.c.write('\n')
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sobject_manager_client_new_for_bus:\n'
                ' * @bus_type: A #GBusType.\n'
                ' * @flags: Flags from the #GDBusObjectManagerClientFlags enumeration.\n'
                ' * @name: A bus name (well-known or unique).\n'
                ' * @object_path: An object path.\n'
                ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                ' * @callback: A #GAsyncReadyCallback to call when the request is satisfied.\n'
                ' * @user_data: User data to pass to @callback.\n'
                ' *\n'
                ' * Like %sobject_manager_client_new() but takes a #GBusType instead of a #GDBusConnection.\n'
                ' *\n'
                ' * When the operation is finished, @callback will be invoked in the <link linkend="g-main-context-push-thread-default">thread-default main loop</link> of the thread you are calling this method from.\n'
                ' * You can then call %sobject_manager_client_new_for_bus_finish() to get the result of the operation.\n'
                ' *\n'
                ' * See %sobject_manager_client_new_for_bus_sync() for the synchronous, blocking version of this constructor.\n'
                %(self.ns_lower, self.ns_lower, self.ns_lower, self.ns_lower), False))
        self.c.write(' */\n')
        self.c.write('void\n'
                     '%sobject_manager_client_new_for_bus (\n'
                     '    GBusType                bus_type,\n'
                     '    GDBusObjectManagerClientFlags  flags,\n'
                     '    const gchar            *name,\n'
                     '    const gchar            *object_path,\n'
                     '    GCancellable           *cancellable,\n'
                     '    GAsyncReadyCallback     callback,\n'
                     '    gpointer                user_data)\n'
                     '{\n'
                     '  g_async_initable_new_async (%sTYPE_OBJECT_MANAGER_CLIENT, G_PRIORITY_DEFAULT, cancellable, callback, user_data, "flags", flags, "name", name, "bus-type", bus_type, "object-path", object_path, "get-proxy-type-func", %sobject_manager_client_get_proxy_type, NULL);\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower, self.ns_upper, self.ns_lower))
        self.c.write('/**\n'
                     ' * %sobject_manager_client_new_for_bus_finish:\n'
                     ' * @res: The #GAsyncResult obtained from the #GAsyncReadyCallback passed to %sobject_manager_client_new_for_bus().\n'
                     ' * @error: Return location for error or %%NULL\n'
                     ' *\n'
                     ' * Finishes an operation started with %sobject_manager_client_new_for_bus().\n'
                     ' *\n'
                     ' * Returns: (transfer full) (type %sObjectManagerClient): The constructed object manager client or %%NULL if @error is set.\n'
                     %(self.ns_lower, self.ns_lower, self.ns_lower, self.namespace))
        self.c.write(' */\n')
        self.c.write('GDBusObjectManager *\n'
                     '%sobject_manager_client_new_for_bus_finish (\n'
                     '    GAsyncResult        *res,\n'
                     '    GError             **error)\n'
                     '{\n'
                     '  GObject *ret;\n'
                     '  GObject *source_object;\n'
                     '  source_object = g_async_result_get_source_object (res);\n'
                     '  ret = g_async_initable_new_finish (G_ASYNC_INITABLE (source_object), res, error);\n'
                     '  g_object_unref (source_object);\n'
                     '  if (ret != NULL)\n'
                     '    return G_DBUS_OBJECT_MANAGER (ret);\n'
                     '  else\n'
                     '    return NULL;\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower))
        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * %sobject_manager_client_new_for_bus_sync:\n'
                ' * @bus_type: A #GBusType.\n'
                ' * @flags: Flags from the #GDBusObjectManagerClientFlags enumeration.\n'
                ' * @name: A bus name (well-known or unique).\n'
                ' * @object_path: An object path.\n'
                ' * @cancellable: (allow-none): A #GCancellable or %%NULL.\n'
                ' * @error: Return location for error or %%NULL\n'
                ' *\n'
                ' * Like %sobject_manager_client_new_sync() but takes a #GBusType instead of a #GDBusConnection.\n'
                ' *\n'
                ' * The calling thread is blocked until a reply is received.\n'
                ' *\n'
                ' * See %sobject_manager_client_new_for_bus() for the asynchronous version of this constructor.\n'
                ' *\n'
                ' * Returns: (transfer full) (type %sObjectManagerClient): The constructed object manager client or %%NULL if @error is set.\n'
                %(self.ns_lower, self.ns_lower, self.ns_lower, self.namespace), False))
        self.c.write(' */\n')
        self.c.write('GDBusObjectManager *\n'
                     '%sobject_manager_client_new_for_bus_sync (\n'
                     '    GBusType                bus_type,\n'
                     '    GDBusObjectManagerClientFlags  flags,\n'
                     '    const gchar            *name,\n'
                     '    const gchar            *object_path,\n'
                     '    GCancellable           *cancellable,\n'
                     '    GError                **error)\n'
                     '{\n'
                     '  GInitable *ret;\n'
                     '  ret = g_initable_new (%sTYPE_OBJECT_MANAGER_CLIENT, cancellable, error, "flags", flags, "name", name, "bus-type", bus_type, "object-path", object_path, "get-proxy-type-func", %sobject_manager_client_get_proxy_type, NULL);\n'
                     '  if (ret != NULL)\n'
                     '    return G_DBUS_OBJECT_MANAGER (ret);\n'
                     '  else\n'
                     '    return NULL;\n'
                     '}\n'
                     '\n'
                     %(self.ns_lower, self.ns_upper, self.ns_lower))
        self.c.write('\n')

    # ---------------------------------------------------------------------------------------------------

    def write_gtkdoc_deprecated_and_since_and_close(self, obj, f, indent):
        if len(obj.since) > 0:
            f.write('%*s *\n'
                    '%*s * Since: %s\n'
                    %(indent, '', indent, '', obj.since))
        if obj.deprecated:
            if isinstance(obj, dbustypes.Interface):
                thing = 'The D-Bus interface'
            elif isinstance(obj, dbustypes.Method):
                thing = 'The D-Bus method'
            elif isinstance(obj, dbustypes.Signal):
                thing = 'The D-Bus signal'
            elif isinstance(obj, dbustypes.Property):
                thing = 'The D-Bus property'
            else:
                raise RuntimeError('Cannot handle object ', obj)
            f.write(self.docbook_gen.expand(
                    '%*s *\n'
                    '%*s * Deprecated: %s has been deprecated.\n'
                    %(indent, '', indent, '', thing), False))
        f.write('%*s */\n'%(indent, ''))

    # ---------------------------------------------------------------------------------------------------

    def generate_interface_intro(self, i):
        self.c.write('/* ------------------------------------------------------------------------\n'
                     ' * Code for interface %s\n'
                     ' * ------------------------------------------------------------------------\n'
                     ' */\n'
                     '\n'%(i.name))

        self.c.write(self.docbook_gen.expand(
                '/**\n'
                ' * SECTION:%s\n'
                ' * @title: %s\n'
                ' * @short_description: Generated C code for the %s D-Bus interface\n'
                ' *\n'
                ' * This section contains code for working with the #%s D-Bus interface in C.\n'
                ' */\n'
                %(i.camel_name, i.camel_name, i.name, i.name), False))
        self.c.write('\n')

    def generate(self):
        self.generate_intro()
        self.declare_types()
        for i in self.ifaces:
            self.generate_interface_intro(i)
            self.generate_introspection_for_interface(i)
            self.generate_interface(i)
            self.generate_property_accessors(i)
            self.generate_signal_emitters(i)
            self.generate_method_calls(i)
            self.generate_method_completers(i)
            self.generate_proxy(i)
            self.generate_skeleton(i)
        if self.generate_objmanager:
            self.generate_object()
            self.generate_object_manager_client()

        self.generate_outro()
