#include "stk-synth.h"

/* STK stuff */
#include <RTWvOut.h>
#include <miditabl.h>
#include <Marimba.h>
#include <Vibraphn.h>
#include <AgogoBel.h>
#include <Plucked.h>
#include <Mandolin.h>
#include <Clarinet.h>
#include <Flute.h>
#include <Brass.h>
#include <Bowed.h>
#include <Rhodey.h>
#include <Wurley.h>
#include <TubeBell.h>
#include <HeavyMtl.h>
#include <PercFlut.h>
#include <BeeThree.h>
#include <FMVoices.h>
#include <Moog1.h>
#include <Simple.h>
#include <VoicForm.h>
#include <Shakers.h>
#include <DrumSynt.h>
#include <Instrmnt.h>

#define NCHANNELS 16
#define NPITCHES 128

class STKSynth {
protected:
  GSList *active_voices;
  Instrmnt *voices[NCHANNELS][NPITCHES];
  RTWvOut *output;

  Instrmnt *getInstrmnt(const unsigned char *ev, unsigned long len);
  Instrmnt *newInstrmnt(void);
  char *curinst;

public:
  STKSynth();
  ~STKSynth();
  void play(const unsigned char *ev, unsigned long len);
  void play_ev(const unsigned char *ev, unsigned long len);
  void changeInstrument(const char *name);
  void tick();
};

EXPORT gpointer
stk_synth_init(void)
{
  return (gpointer)(new STKSynth());
}

EXPORT void
stk_synth_play(gpointer mysynth, unsigned char *ev, unsigned long len)
{
  ((STKSynth *)mysynth)->play(ev, len);
}

EXPORT void
stk_synth_tick(gpointer mysynth)
{
  ((STKSynth *)mysynth)->tick();
}

EXPORT void
stk_synth_change_instrument(gpointer mysynth, const char *name)
{
  ((STKSynth *)mysynth)->changeInstrument(name);
}

EXPORT void
stk_synth_destroy(gpointer mysynth)
{
  delete ((STKSynth *)mysynth);
}

STKSynth::STKSynth()
{
  active_voices = NULL;
  output = new RTWvOut();
  memset(voices, '\0', sizeof(voices));
}


STKSynth::~STKSynth()
{
  delete output;
  changeInstrument("Clarinet");
  g_free(curinst);
}

Instrmnt *
STKSynth::newInstrmnt(void)
{
  Instrmnt *retval = NULL;

  if(!strcmp(curinst, "Clarinet")) {
    retval = new Clarinet(50.0);
  } else if(!strcmp(curinst, "Flute")) {
    retval = new Flute(50.0);
  } else if(!strcmp(curinst, "Brass")) {
    retval = new Brass(50.0);
  } else if(!strcmp(curinst, "Bowed")) {
    retval = new Bowed(50.0);
  } else if(!strcmp(curinst, "Plucked")) {
    retval = new Plucked(50.0);
  } else if(!strcmp(curinst, "Mandolin")) {
    retval = new Mandolin(50.0);
  } else if(!strcmp(curinst, "Marimba")) {
    retval = new Marimba;
  } else if(!strcmp(curinst, "Vibraphn")) {
    retval = new Vibraphn;
  } else if(!strcmp(curinst, "AgogoBel")) {
    retval = new AgogoBel;
  } else if(!strcmp(curinst, "Rhodey")) {
    retval = new Rhodey;
  } else if(!strcmp(curinst, "Wurley")) {
    retval = new Wurley;
  } else if(!strcmp(curinst, "TubeBell")) {
    retval = new TubeBell;
  } else if(!strcmp(curinst, "HeavyMtl")) {
    retval = new HeavyMtl;
  } else if(!strcmp(curinst, "PercFlut")) {
    retval = new PercFlut;
  } else if(!strcmp(curinst, "BeeThree")) {
    retval = new BeeThree;
  } else if(!strcmp(curinst, "Moog1")) {
    retval = new Moog1;
  } else if(!strcmp(curinst, "FMVoices")) {
    retval = new FMVoices;
  } else if(!strcmp(curinst, "VoicForm")) {
    retval = new VoicForm;
  } else if(!strcmp(curinst, "DrumSynt")) {
    retval = new DrumSynt;
  } else {
    g_error("Unknown instrument");
  }

  return retval;
}

void
STKSynth::changeInstrument(const char *name)
{
  int i, j;
  g_free(curinst);
  curinst = g_strdup(name);

  for(i = 0; i < NCHANNELS; i++) {
    for(j = 0; j < NPITCHES; j++) {
      if(voices[i][j]) {
	delete voices[i][j];
	voices[i][j] = NULL;
      }
    }
  }
  g_slist_free(active_voices);
  active_voices = NULL;
}

Instrmnt *
STKSynth::getInstrmnt(const unsigned char *ev, unsigned long len)
{
  Instrmnt *ins;
  unsigned int channel, step;

  channel = (ev[0] & 0x0F);
  step = ev[1];

  if(!voices[channel][step])
    voices[channel][step] = newInstrmnt();

  return voices[channel][step];
}

static gint
gmf_midi_event_len(const unsigned char *data, gulong data_len)
{
  guint retval = 0;
  guchar first_byte;

  g_return_val_if_fail(data_len > 0, -1);

  first_byte = data[0];

  switch(first_byte & 0xF0) {
  case 0xF0:
    switch(first_byte) {
    case 0xF7:
    case 0xF8:
    case 0xF9:
    case 0xFA:
    case 0xFB:
    case 0xFC:
    case 0xFE:
    case 0xFF:
      goto onebyteop;
    case 0xF1:
    case 0xF3:
      goto twobyteop;
    case 0xF0:
      { /* handle the dreadful sysex */
	int i;
	for(i = 1;
	    (i < data_len) && (data[i] & 0x80);
	    i++) /**/ ;
	
	if(data[i] & 0x80) {

	  if(data[i] == 0xF7)
	    i++; /* If they put an EOX byte in, then it
		    is part of the current message */

	  return i;
	} else
	  return -1;
      }
    break;
    default:
      break;
    }
  case 0x90:
  case 0x80:
  case 0xA0:
  case 0xB0:
  case 0xD0:
  case 0xE0:
    retval++;
  twobyteop:
  case 0xC0:
    retval++;
  onebyteop:
  default:
    retval++;
  }

  if(retval <= data_len)
    return retval;
  else
    return -1;
}

void STKSynth::play(const unsigned char *ev, unsigned long len)
{
  const unsigned char *evptr;
  unsigned long evlen;

  for(evptr = ev; evptr < (ev + len); evptr += evlen) {
    evlen = gmf_midi_event_len(evptr, len - (evptr - ev));
    play_ev(evptr, evlen);
  }
}

void
STKSynth::play_ev(const unsigned char *ev, unsigned long len)
{
  Instrmnt *infrom;
  MY_FLOAT pitch;
  MY_FLOAT amp;
  gboolean is_noteoff = FALSE;

  switch(ev[0] & 0xF0) {
  case 0x90:
    if(!ev[2]) {
      is_noteoff = TRUE;
      amp = 0.5;
    } else
      amp = ((MY_FLOAT)ev[2])/128.0;
    pitch = __MIDI_To_Pitch[(int)ev[1]];
    break;
  case 0x80:
    is_noteoff = TRUE;
    amp = ((MY_FLOAT)ev[2])/128.0;
    pitch = __MIDI_To_Pitch[(int)ev[1]];
    break;
  default:
    g_warning("Unsupported event type %x", ev[0]);
    return;
    break;
  }

  infrom = getInstrmnt(ev, len);

  if(is_noteoff) {
    infrom->noteOff(amp);
    active_voices = g_slist_remove(active_voices, infrom);
  } else {
    infrom->noteOn(pitch, amp);
    active_voices = g_slist_append(active_voices, infrom);
  }
}

void
STKSynth::tick()
{
  MY_FLOAT val;
  MY_FLOAT n;
  int nvoices;
  GSList *ltmp;

  nvoices = g_slist_length(active_voices);
  if(!nvoices) return;

  n = sqrt(sqrt(sqrt((MY_FLOAT)nvoices)));
  val = 0.0;
  for(ltmp = active_voices; ltmp; ltmp = g_slist_next(ltmp)) {
    val += ((Instrmnt *)ltmp->data)->tick()/n;
  }

  /*  val /= ((MY_FLOAT)n); */
  output->tick(val);
}
