/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include <gnome.h>
#include <gmf.h>

typedef struct {
  GtkWidget *mainwin;

  GtkWidget *slider;

  GtkWidget *rewind, *stop, *play, *ff;
  gboolean inside_toggle_handler;

  GtkAdjustment *position;

  enum { REWIND = -1, STOP, PLAY, FF } status;

  GMFFilterRegistry *freg;
} MPMainwin;

static MPMainwin *mp_mainwin_new(void);
static void mp_menu_file_open(GtkWidget *widget, MPMainwin *mw);
static void mp_menu_help_about(void);
static void mp_rewind_button_toggled(GtkToggleButton *btn, MPMainwin *mw);
static void mp_stop_button_toggled(GtkToggleButton *btn, MPMainwin *mw);
static void mp_play_button_toggled(GtkToggleButton *btn, MPMainwin *mw);
static void mp_ff_button_toggled(GtkToggleButton *btn, MPMainwin *mw);
static void mp_set_state(MPMainwin *mw, gint status);
static void mp_file_open(MPMainwin *mw, const char *fn);

int main(int argc, char *argv[])
{
  CORBA_ORB orb;
  CORBA_Environment ev;
  MPMainwin *mw;

  CORBA_exception_init(&ev);
  orb = gnome_CORBA_init("gmf-media-player", VERSION, &argc, argv,
			 GNORBA_INIT_SERVER_FUNC, &ev);

  mw = mp_mainwin_new();

  gtk_widget_show_all(mw->mainwin);

  gtk_main();

  return 0;
}

static GtkWidget *
make_stock_button(const char *name)
{
  GtkWidget *retval;

  retval = gtk_toggle_button_new();
  gtk_container_add(GTK_CONTAINER(retval),
		    gnome_stock_pixmap_widget(retval, name));

  return retval;
}

static MPMainwin *
mp_mainwin_new(void)
{
  static GnomeUIInfo filemenu[] = {
    GNOMEUIINFO_MENU_OPEN_ITEM(mp_menu_file_open, NULL),
    GNOMEUIINFO_MENU_EXIT_ITEM(gtk_main_quit, NULL),
    GNOMEUIINFO_END
  };
  static GnomeUIInfo helpmenu[] = {
    GNOMEUIINFO_HELP("gmf-media-player"),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_ABOUT_ITEM(mp_menu_help_about, NULL),
    GNOMEUIINFO_END
  };
  static GnomeUIInfo appmenus[] = {
    GNOMEUIINFO_SUBTREE("_File", filemenu),
    GNOMEUIINFO_SUBTREE("_Help", helpmenu),
    GNOMEUIINFO_END
  };
  GtkWidget *vbox, *hbox;
  MPMainwin *mw;

  mw = g_new0(MPMainwin, 1);
  mw->mainwin = gnome_app_new("gmf-media-player", "Media Player");

  gnome_app_create_menus_with_data(GNOME_APP(mw->mainwin), appmenus, mw);
  gtk_signal_connect(GTK_OBJECT(mw->mainwin), "delete_event",
		     GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  gnome_app_set_contents(GNOME_APP(mw->mainwin), vbox);

  mw->position = GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 100, 1, 10, 10));

  mw->slider = gtk_hscale_new(mw->position);
  gtk_container_add(GTK_CONTAINER(vbox), mw->slider);

  hbox = gtk_hbox_new(FALSE, GNOME_PAD_SMALL);
  gtk_container_add(GTK_CONTAINER(vbox), hbox);

  mw->rewind = make_stock_button(GNOME_STOCK_BUTTON_UP);
  gtk_signal_connect(GTK_OBJECT(mw->rewind), "toggled",
		     GTK_SIGNAL_FUNC(mp_rewind_button_toggled), mw);
  gtk_container_add(GTK_CONTAINER(hbox), mw->rewind);

  mw->stop = make_stock_button(GNOME_STOCK_PIXMAP_STOP);
  gtk_signal_connect(GTK_OBJECT(mw->stop), "toggled",
		     GTK_SIGNAL_FUNC(mp_stop_button_toggled), mw);
  gtk_container_add(GTK_CONTAINER(hbox), mw->stop);

  mw->play = make_stock_button(GNOME_STOCK_BUTTON_NEXT);
  gtk_signal_connect(GTK_OBJECT(mw->play), "toggled",
		     GTK_SIGNAL_FUNC(mp_play_button_toggled), mw);
  gtk_container_add(GTK_CONTAINER(hbox), mw->play);

  mw->ff = make_stock_button(GNOME_STOCK_BUTTON_DOWN);
  gtk_signal_connect(GTK_OBJECT(mw->ff), "toggled",
		     GTK_SIGNAL_FUNC(mp_ff_button_toggled), mw);
  gtk_container_add(GTK_CONTAINER(hbox), mw->ff);

  mw->freg = gmf_filter_registry_get();

  return mw;
}

static void
file_open_ok(GtkWidget *w, GtkWidget *filesel)
{
  MPMainwin *mw;
  char *fn;
  GtkWidget **dlgptr;

  mw = gtk_object_get_data(GTK_OBJECT(filesel), "MPMainwin");
  fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));

  mp_file_open(mw, fn);

  dlgptr = gtk_object_get_data(GTK_OBJECT(filesel), "dlgptr");
  gtk_widget_destroy(filesel);
  *dlgptr = NULL;
}

static void
file_open_cancel(GtkWidget *w, GtkWidget *filesel)
{
  GtkWidget **dlgptr;

  dlgptr = gtk_object_get_data(GTK_OBJECT(filesel), "dlgptr");
  gtk_widget_destroy(filesel);
  *dlgptr = NULL;
}

static void
mp_menu_file_open(GtkWidget *w, MPMainwin *mw)
{
  static GtkWidget *dlg = NULL;

  if(!dlg) {
    dlg = gtk_file_selection_new("Load file");
    
    gtk_object_set_data(GTK_OBJECT(dlg), "MPMainwin", mw);
    gtk_object_set_data(GTK_OBJECT(dlg), "dlgptr", &dlg);
    
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(dlg)->ok_button), "clicked",
		       GTK_SIGNAL_FUNC(file_open_ok), dlg);
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(dlg)->cancel_button), "clicked",
		       GTK_SIGNAL_FUNC(file_open_cancel), dlg);
    
    gtk_widget_show(dlg);
  } else
    gdk_window_raise(dlg->window);
}

static void
mp_menu_help_about(void)
{
  const char * const authors[] = {"Elliot Lee", NULL};
  GtkWidget *wtmp;

  wtmp = gnome_about_new("GMF Media Player", VERSION,
			 "Copyright (C) 1999 Elliot Lee",
			 (const gchar **)authors,
			 "A media player for the GNOME Media Framework.",
			 NULL);

  gtk_widget_show(wtmp);
}

static void
mp_rewind_button_toggled(GtkToggleButton *btn, MPMainwin *mw)
{
  if(mw->inside_toggle_handler) return;
  mw->inside_toggle_handler = TRUE;

  if(btn->active)
    mp_set_state(mw, REWIND);
  else
    mp_set_state(mw, STOP);

  mw->inside_toggle_handler = FALSE;
}

static void
mp_stop_button_toggled(GtkToggleButton *btn, MPMainwin *mw)
{
  if(mw->inside_toggle_handler) return;
  mw->inside_toggle_handler = TRUE;

  mp_set_state(mw, STOP);

  mw->inside_toggle_handler = FALSE;
}

static void
mp_play_button_toggled(GtkToggleButton *btn, MPMainwin *mw)
{
  if(mw->inside_toggle_handler) return;
  mw->inside_toggle_handler = TRUE;

  if(btn->active)
    mp_set_state(mw, PLAY);
  else
    mp_set_state(mw, STOP);

  mw->inside_toggle_handler = FALSE;
}

static void
mp_ff_button_toggled(GtkToggleButton *btn, MPMainwin *mw)
{
  if(mw->inside_toggle_handler) return;
  mw->inside_toggle_handler = TRUE;

  if(btn->active)
    mp_set_state(mw, FF);
  else
    mp_set_state(mw, STOP);

  mw->inside_toggle_handler = FALSE;
}

static void
mp_set_state(MPMainwin *mw, gint status)
{
  switch(status) {
  case STOP:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->rewind), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->ff), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->play), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->stop), TRUE);
    break;
  case REWIND:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->stop), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->play), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->ff), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->rewind), TRUE);
    break;
  case PLAY:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->rewind), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->stop), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->ff), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->play), TRUE);
    break;
  case FF:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->rewind), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->stop), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->play), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mw->ff), TRUE);
    break;
  default:
    g_error("Unknown state %d.", status);
  }

  if(mw->status == status) return;
}

static void
mp_file_open(MPMainwin *mw, const char *fn)
{
  /*
   * Steps:
   * 1. Figure out what MIME type the file is.
   * 2. Figure out what source filter to use to handle it.
   * 3. Figure out what rendering filter to use to handle it.
   */
}
