/* 
 * Copyright (C) 1999 JP Rosevear
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "gnome_chess.h"
#include <ctype.h>
#include <sys/time.h>
#include <stdarg.h>
#include <signal.h>

#include "cho1.h"
#include "pro.h"
#include "position.h"
#include "makros.h"
#include "movlist.h"

extern int yylex();
void yyerror(char *s);

static gint source;
static GdkInputCondition condition;

/* Menu callback functions */
static void engine_ics_menu_init(struct engine *e);
static void engine_ics_takeback_cb (GtkWidget *widget, void *data);
static void engine_ics_flag_cb (GtkWidget *widget, void *data);
static void engine_ics_draw_cb (GtkWidget *widget, void *data);
static void engine_ics_resign_cb (GtkWidget *widget, void *data);
static void engine_ics_start_cb (GtkWidget *widget, void *data);
static void engine_ics_previous_cb (GtkWidget *widget, void *data);
static void engine_ics_next_cb (GtkWidget *widget, void *data);
static void engine_ics_end_cb (GtkWidget *widget, void *data);

struct engine * engine_ics_new(char *host, char *user, char *pass, char *port, char *telnetprogram)
{
	struct engine *e;
	struct engine_ics *ei;
	char *cmd;
	char *arg[4];
	
	if (telnetprogram == NULL)
	  cmd = "telnet";
	else
	  cmd = telnetprogram;

	arg[0] = cmd;
	arg[1] = host;
	arg[2] = port;
	arg[3] = NULL;

	e = engine_new(cmd, arg);

	e->notation = SAN;

	e->data = malloc( sizeof(struct engine_ics));
	ei = (struct engine_ics *)e->data;

	if (user)  
		ei->user =  strdup(user);
	else
		ei->user = NULL;
	if (pass)
		ei->pass = strdup(pass);
	else
		ei->pass = NULL;
	ei->loggedin = FALSE;
	ei->game = 0;

	engine_ics_menu_init(e);

	gdk_input_add( fileno(e->read_from), GDK_INPUT_READ, engine_ics_cb, e);

	return e;
}

static void engine_ics_menu_init(struct engine *e) {

	e->menudata->menu_init = engine_ics_menu_init;
	e->menudata->menu_flag_cb = engine_ics_flag_cb;
	e->menudata->menu_draw_cb = engine_ics_draw_cb;
	e->menudata->menu_resign_cb = engine_ics_resign_cb;
	e->menudata->menu_start_cb = engine_ics_start_cb;
	e->menudata->menu_previous_cb = engine_ics_previous_cb;
	e->menudata->menu_next_cb = engine_ics_next_cb;
	e->menudata->menu_end_cb = engine_ics_end_cb;
	e->menudata->menu_takeback_cb = engine_ics_takeback_cb;

	enable_all_menus();
	uncheck_all_menus();
	disable_menu(MENU_NEW);
	disable_menu(MENU_NOW);
	disable_menu(MENU_GO);
	disable_menu(MENU_CWHITE);
	disable_menu(MENU_CBLACK);
	disable_menu(MENU_TWOC);
	disable_menu(MENU_ANALYZE);
	disable_menu(MENU_LEVEL);
	check_menu(MENU_TWOP);
}

char *engine_ics_get_user(struct engine *e) {
  struct engine_ics *ei;

  ei = (struct engine_ics *)e->data;
  return ei->user;
}

void engine_ics_set_user(struct engine *e, gchar *user) {
  struct engine_ics *ei;

  ei = (struct engine_ics *)e->data;
  if (ei->user)
	g_free(ei->user);
  ei->user = user;
}

char *engine_ics_get_pass(struct engine *e) {
  struct engine_ics *ei;

  ei = (struct engine_ics *)e->data;
  return ei->pass;
}

void engine_ics_set_pass(struct engine *e, gchar *pass) {
  struct engine_ics *ei;

  ei = (struct engine_ics *)e->data;
  if (ei->pass)
	g_free(ei->pass);
  ei->pass = pass;
}

gboolean engine_ics_is_loggedin(struct engine *e) {
  struct engine_ics *ei;

  ei = (struct engine_ics *)e->data;
  return ei->loggedin;
}

void engine_ics_set_loggedin(struct engine *e, gboolean li) {
  struct engine_ics *ei;

  ei = (struct engine_ics *)e->data;
  ei->loggedin = li;
}

int engine_ics_get_game(struct engine *e) {
  struct engine_ics *ei;

  ei = (struct engine_ics *)e->data;
  return ei->game;
}

void engine_ics_set_game(struct engine *e, int g) {
  struct engine_ics *ei;

  ei = (struct engine_ics *)e->data;
  ei->game = g;
}

static void engine_ics_takeback_cb (GtkWidget *widget, void *data) {
	engine_write(first_chess_programm,"takeback\n");
}

static void engine_ics_flag_cb (GtkWidget *widget, void *data) {
	engine_write(first_chess_programm,"flag\n");
}

static void engine_ics_draw_cb (GtkWidget *widget, void *data) {
	engine_write(first_chess_programm,"draw\n");
}

static void engine_ics_resign_cb (GtkWidget *widget, void *data) {
	engine_write(first_chess_programm,"resign\n");
}

static void engine_ics_start_cb (GtkWidget *widget, void *data) {
  gnomechess_movelist_move_start(board.movelist);
  gnomechess_board_update();
}

static void engine_ics_previous_cb (GtkWidget *widget, void *data) {
  gnomechess_movelist_move_back(board.movelist);
  gnomechess_board_update();
}

static void engine_ics_next_cb (GtkWidget *widget, void *data) {
  gnomechess_movelist_move_forward(board.movelist);
  gnomechess_board_update();
}

static void engine_ics_end_cb (GtkWidget *widget, void *data) {
  gnomechess_movelist_move_end(board.movelist);
  gnomechess_board_update();
}

int engine_ics_input (char *buf, int max) {
	char b[1024];
	ssize_t len;

	len = read(source, b, sizeof b);

	if (len > 0) {
		memcpy(buf, b, len);
	} else if (len == 0) {
		engine_destroy_all();
		gtk_main_quit();
	}
	return len;
}

static void engine_ics_set_board (char *b) {
  int i,j;

  for ( i=0;i<8;i++) {
	for ( j=0;j<8;j++) {
		int n= A8 - i * 10 + j;

		switch (b[(8*i)+j]) {
			case 'R':
				currPositionPtr->square[n] = WR;
				break;
			case 'N':
				currPositionPtr->square[n] = WN;
				break;
			case 'B':
				currPositionPtr->square[n] = WB;
				break;
			case 'Q':
				currPositionPtr->square[n] = WQ;
				break;
			case 'K':
				currPositionPtr->square[n] = WK;
				break;
			case 'P':
				currPositionPtr->square[n] = WP;
				break;
			case 'r':
				currPositionPtr->square[n] = BR;
				break;
			case 'n':
				currPositionPtr->square[n] = BN;
				break;
			case 'b':
				currPositionPtr->square[n] = BB;
				break;
			case 'q':
				currPositionPtr->square[n] = BQ;
				break;
			case 'k':
				currPositionPtr->square[n] = BK;
				break;
			case 'p':
				currPositionPtr->square[n] = BP;
				break;
			case '-':
				currPositionPtr->square[n] = EMPTY;
				break;
		}
    }
  }	
}

void engine_ics_update_board (char *boardstring) {
	char *temp;
	char b[65];
	char move[10];
	int state, rank=0, rel=0, game=0, setgame=0, flip=0, num=0, ply=0, maxply=0, colour=0;
	int from,to,piece;
	enum {START=0, BOARD=1, COLOUR=2, DOUBLE=3, WCS=4, WCL=5, BCS=6, BCL=7, 
			MOVE=8, GAME=9,	WNAME=10, BNAME=11, RELATION=12, TIME=13, INC=14,
			WSTR=15, BSTR=16, WTIME=17, BTIME=18, NUM=19, VERBOSE=20, LENGTH=21,
			NOTATION=22, FLIP=23};

	state = START;
	temp = boardstring;
	strcpy(b, "");
	debug_print(DEBUG_NORMAL,"%s\n",boardstring);
	temp = strtok(temp, " ");
	while (temp) {
		switch (state) {
			case START:
			case DOUBLE:
			case WCS:
			case WCL:
			case BCS:
			case BCL:
			case MOVE:
			case TIME:
			case INC:
			case WSTR:
			case BSTR:
			case LENGTH:
			case NOTATION:
				state++;
				break;
			case BOARD:
				rank++;
				strcat(b, temp);
				if (rank==8)
					state++;
				break;
			case COLOUR:
				if (!strcmp(temp, "W"))
					colour = WHITE;
				else
					colour  = BLACK;
				state++;
				break;
			case GAME:
				game = atoi(temp);
				state++;
				break;
			case RELATION:
				rel = atoi(temp);
				state++;
				break;
			case WNAME:
				gtk_label_set_text(GTK_LABEL(board.whitename), temp);
				state++;
				break;
			case BNAME:
				gtk_label_set_text(GTK_LABEL(board.blackname), temp);
				state++;
				break;
			case WTIME:
				gnomechess_timer_set(board.whitetime, atoi(temp));
				state++;
				break;
			case BTIME:
				gnomechess_timer_set(board.blacktime, atoi(temp));
				state++;
				break;
			case NUM:
				num = atoi(temp);
				state++;
				break;
			case VERBOSE:
				if (!strcmp(temp,"o-o") || !strcmp(temp, "o-o-o")) {
					strncpy(move, temp, strlen(temp)+1);
				} else if (strcmp(temp, "none")) {
					strncpy(move, temp+2, 2);
					move[0] = temp[2];
					move[1] = temp[3];
					move[2] = temp[5];
					move[3] = temp[6];
					move[4] = '\0';
				} else {
					strcpy(move, "none");
				}
				state++;
				break;
			case FLIP:
				flip = atoi(temp);
				state++;
		}
		temp = strtok(NULL, " ");
	}

	/* Flip the board if necessary */
	gnomechess_board_set_flip(flip);
	
	/* Start and stop the timer as required */
	if (rel == -3 || rel == -2 || rel == 2) {
		gnomechess_timer_stop(board.whitetime);
		gnomechess_timer_stop(board.blacktime);
	} else if (colour == WHITE) {
		gnomechess_timer_stop(board.blacktime);
		gnomechess_timer_start(board.whitetime);
	} else {
		gnomechess_timer_stop(board.whitetime);
		gnomechess_timer_start(board.blacktime);
	}

	/* Figure out what ply this is */
	if (!strcmp(move, "none")) {
		ply =  0;
	} else if (colour == WHITE) {
		ply = num*2-2;
	} else {
		ply = num*2-1;
	}

	/* Setup the board and figure out the move if there */
	/* was no dnd update or this is a different game */
	maxply = gnomechess_movelist_maxply(board.movelist);
	setgame = engine_ics_get_game(first_chess_programm);
	if (maxply < ply || game != setgame){
		if (strcmp(move, "none")) 
			ascii_to_move(move,&from,&to);

		if (from && ply-maxply == 1 && strcmp(move, "none") && game == setgame) {
			gnomechess_position_move(currPositionPtr, from, to);
		} else {
			gnomechess_position_set_initial(currPositionPtr);
			engine_ics_set_board (b);
		}
                gnomechess_sound_enginemove();

		if (strcmp(move, "none")) {
			piece = currPositionPtr->square[to];
			debug_print(DEBUG_NORMAL,"Move detected: %s %d %d %d\n",
						move,piece,from,to);
			gnomechess_movelist_add(board.movelist, piece,from,to);
		}
	}
	currPositionPtr->tomove = colour;

	/* Clear the movelist if this is a different game */
	if (game != setgame) {
		gnomechess_movelist_clear_initial(board.movelist, ply, currPositionPtr);
		debug_print(DEBUG_NORMAL,"Initializing at ply: %d\n", ply);
		gnomechess_movelist_set_result(board.movelist, "");
		engine_ics_set_game(first_chess_programm, game);
	}

	gnomechess_board_update();
}

void engine_ics_cb( gpointer d, gint s, GdkInputCondition c) {
	int token;
	source = s;
	condition = c;
	
	token = yylex();
}

