/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* game-view.c
 *
 * Copyright (C) 2001  JP Rosevear.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include "debug.h"
#include "makros.h"
#include "prefs.h"
#include "engine_null.h"
#include "movlist.h"
#include "board_info.h"
#include "game-view.h"

enum {
	INFO_SIGNAL,
	LAST_SIGNAL
};
static gint view_signals[LAST_SIGNAL] = { 0 };

static void class_init (GameViewClass *klass);
static void init (GameView *view);
static void destroy (GtkObject *object);

static void gv_engine_info_cb (GtkObject *obj, GameInfo *info, gpointer data);
static void gv_engine_game_cb (GtkObject *obj, gint ply, Position *pos, gpointer data);
static void gv_engine_move_cb (GtkObject *obj, gint from, gint to, gpointer data);
static void gv_engine_unmove_cb (GtkObject *obj, gint ply, gpointer data);
static void gv_board_move_cb (GtkWidget *widget, gint from, gint to, gpointer data);
static void gv_move_list_move_cb (GtkWidget *widget, gint plynum, gpointer data);

static GtkVBoxClass *parent_class = NULL;

struct _GameViewPrivate {
	guint id;
	guint mode;

	GtkWidget *board;
	GtkWidget *movelist;
	GtkWidget *board_info;
	GtkObject *engine;

	gchar *pgn_filename;
};


GtkType
game_view_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "GameView",
        sizeof (GameView),
        sizeof (GameViewClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gtk_vbox_get_type (), &info);
    }

  return type;
}

static void
class_init (GameViewClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (gtk_vbox_get_type ());

	view_signals[INFO_SIGNAL] =
		gtk_signal_new ("info",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GameViewClass, info),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	
	gtk_object_class_add_signals (object_class,
				      view_signals,
				      LAST_SIGNAL);

	object_class->destroy = destroy;
}


static void
init (GameView *view)
{
	GameViewPrivate *priv;

	priv = g_new0 (GameViewPrivate, 1);

	view->priv = priv;

	game_view_set_id (view, 0); 
	priv->mode = COMPUTER_BLACK;
}



GtkWidget *
game_view_new (void) 
{
	Engine *engine;

	engine = ENGINE (engine_null_new ());

	return game_view_new_with_engine (engine);
}

GtkWidget *
game_view_new_with_engine (Engine *engine)
{
	GameView *view;
	GameViewPrivate *priv;
	GtkWidget *hbox;
	guint8 r, g, b;
	
	view = gtk_type_new (game_view_get_type ());
	priv = view->priv;
	
	priv->engine = GTK_OBJECT (engine);

  	hbox = gtk_hbox_new (FALSE, 4);

	/* The Board */
	priv->board = board_new ();

	prefs_get_light_color (&r, &g, &b);
	board_set_light_color (BOARD (priv->board), 
			       GNOME_CANVAS_COLOR (r, g, b));

	prefs_get_dark_color (&r, &g, &b);
	board_set_dark_color (BOARD (priv->board), 
			      GNOME_CANVAS_COLOR (r, g, b));
	gtk_box_pack_start (GTK_BOX (hbox), priv->board, TRUE, TRUE, 2);
	gtk_widget_show (GTK_WIDGET (priv->board));  

	/* The Board Info */
	priv->board_info = board_info_new ();
	gtk_box_pack_start (GTK_BOX (hbox), priv->board_info, TRUE, TRUE, 0);
	gtk_widget_show (GTK_WIDGET (priv->board_info)); 

	/* Move List Area */
	priv->movelist = move_list_new ();
	gtk_box_pack_start (GTK_BOX (hbox), priv->movelist, FALSE, FALSE, 0);
	gtk_widget_show (priv->movelist);

	/* Engine  */
	engine_set_board (ENGINE (priv->engine), BOARD (priv->board));

	/* Connect to our callbacks */
	gtk_signal_connect (GTK_OBJECT (priv->engine), "info",
			    (GtkSignalFunc) gv_engine_info_cb,
			    view);
	gtk_signal_connect (GTK_OBJECT (priv->engine), "game",
			    (GtkSignalFunc) gv_engine_game_cb,
			    view);
	gtk_signal_connect (GTK_OBJECT (priv->engine), "move",
			    (GtkSignalFunc) gv_engine_move_cb,
			    view);
	gtk_signal_connect (GTK_OBJECT (priv->engine), "unmove",
			    (GtkSignalFunc) gv_engine_unmove_cb,
			    view);
	gtk_signal_connect (GTK_OBJECT (priv->board), "move",
			    (GtkSignalFunc) gv_board_move_cb,
			    view);
	gtk_signal_connect (GTK_OBJECT (priv->movelist), "move",
			    (GtkSignalFunc) gv_move_list_move_cb,
			    view);

	/* Show all the pieces */
	gtk_widget_show (hbox);

	/* Put the table into the window */
	gtk_container_add (GTK_CONTAINER (view), hbox);

	return GTK_WIDGET (view);
}

guint 
game_view_get_id (GameView *view) 
{
	GameViewPrivate *priv;
	
	priv = view->priv;
	
	return priv->id;
}

void 
game_view_set_id (GameView *view, guint id) 
{
	GameViewPrivate *priv;

	priv = view->priv;
	
	priv->id = id;
}

Board *
game_view_get_board (GameView *view)
{
	GameViewPrivate *priv;
	
	g_return_val_if_fail (view != NULL, NULL);
	g_return_val_if_fail (IS_GAME_VIEW (view), NULL);

	priv = view->priv;
	
	return BOARD (priv->board);
}

Engine *
game_view_get_engine (GameView *view)
{
	GameViewPrivate *priv;
	
	g_return_val_if_fail (view != NULL, NULL);
	g_return_val_if_fail (IS_GAME_VIEW (view), NULL);

	priv = view->priv;
	
	return ENGINE (priv->engine);
}

MoveList *
game_view_get_movelist (GameView *view)
{
	GameViewPrivate *priv;
	
	g_return_val_if_fail (view != NULL, NULL);
	g_return_val_if_fail (IS_GAME_VIEW (view), NULL);

	priv = view->priv;
	
	return MOVE_LIST (priv->movelist);
}

BoardInfo *
game_view_get_board_info (GameView *view)
{
	GameViewPrivate *priv;
	
	g_return_val_if_fail (view != NULL, NULL);
	g_return_val_if_fail (IS_GAME_VIEW (view), NULL);

	priv = view->priv;
	
	return BOARD_INFO (priv->board_info);
}

static void 
destroy (GtkObject *object)
{
	GameView *view;
	GameViewPrivate *priv;
	
	view = GAME_VIEW (object);
	priv = view->priv;

	gtk_object_destroy (priv->engine);
	g_free (priv);

	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
gv_engine_info_cb (GtkObject *obj, GameInfo *info, gpointer data)
{
	GameView *view;
	GameViewPrivate *priv;

	view = GAME_VIEW (data);
	priv = view->priv;

	board_info_set_game_info (BOARD_INFO (priv->board_info), info);
	
	gtk_signal_emit (GTK_OBJECT (view),
			 view_signals[INFO_SIGNAL],
			 info);
}

static void
gv_engine_game_cb (GtkObject *obj, gint ply, Position *pos, gpointer data)
{
	GameView *view;
	GameViewPrivate *priv;

	view = GAME_VIEW (data);
	priv = view->priv;

	board_set_position (BOARD (priv->board), pos);

	move_list_clear_initial (MOVE_LIST (priv->movelist), ply, pos);
}

static void
gv_engine_move_cb (GtkObject *obj, gint from, gint to, gpointer data)
{
	GameView *view;
	GameViewPrivate *priv;

	view = GAME_VIEW (data);
	priv = view->priv;

	board_move (BOARD (priv->board), from, to);

	move_list_add (MOVE_LIST (priv->movelist), from, to);
}

static void
gv_engine_unmove_cb (GtkObject *obj, gint ply, gpointer data)
{
	GameView *view;
	GameViewPrivate *priv;
	gint curr;	

	view = GAME_VIEW (data);
	priv = view->priv;

	curr = move_list_currply (MOVE_LIST (priv->movelist));
	if (curr < ply)
		return;
	
	move_list_clear_from (MOVE_LIST (priv->movelist), curr - ply);
}

static void 
gv_board_move_cb (GtkWidget *widget, gint from, gint to, gpointer data) 
{
	GameView *view;
	GameViewPrivate *priv;

	view = GAME_VIEW (data);
	priv = view->priv;

	engine_move (ENGINE (priv->engine), from, to);

	move_list_add (MOVE_LIST (priv->movelist), from, to);
}

static void 
gv_move_list_move_cb (GtkWidget *widget, gint plynum, gpointer data) 
{
	GameView *view;
	GameViewPrivate *priv;
	Position *pos;

	view = GAME_VIEW (data);
	priv = view->priv;

	debug_print (DEBUG_NORMAL,"MoveList Move To Ply: %d\n", plynum);

	pos = move_list_get_position (MOVE_LIST (priv->movelist), plynum);
	board_set_position (BOARD (priv->board), pos);
}
