/* 
 * Copyright (C) 1999 JP Rosevear
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdlib.h>
#include <string.h>
#include "makros.h"
#include "prefs.h"
#include "dialogs.h"
#include "position.h"


struct _PositionPrivate {
	gshort tomove;	/* Color to move */
	gshort wta;	/* Flag if the white 'a' rook has moved */
	gshort wth;	/* Flag if the white 'h' rook has moved */
	gshort wki;	/* The square the white king is on */
	gshort sta;	/* Flag if the black 'a' rook has moved */
	gshort sth;	/* Flag if the black 'h' rook has moved */
	gshort ski;	/* The square the black king is on */
	gshort ep;	/* En passent */
	gshort vf;	/* Last piece captured */
};

/* Move generation variables */
static const int jump [] = { 8, 12,19, 21,-8,-12,-19,-21,     
			     9, 11,-9,-11, 1, 10,-10, -1,     
			     9, 11, 1, 10,-1,  1, 10, -1,
			     -9,-11, 1,-10,-1     };
static gchar *nindex,*sindex;

/* Prototypes */
static void position_class_init (PositionClass *class);
static void position_init (Position *pos);
static void position_set_empty(Position *pos);

GtkType position_get_type ()
{
	static guint position_type = 0;

	if (!position_type) {
           GtkTypeInfo position_info = {
               "Position",
               sizeof (Position),
               sizeof (PositionClass),
               (GtkClassInitFunc) position_class_init,
               (GtkObjectInitFunc) position_init,
               (GtkArgSetFunc) NULL,
               (GtkArgGetFunc) NULL
             };
             position_type = gtk_type_unique (gtk_object_get_type (),
					      &position_info);
           }

         return position_type;
}

static void
position_finalize (GtkObject *object)
{
	Position *pos = (Position *) object;

	g_free (pos->priv);

	pos->priv = NULL;
}

static void
position_class_init (PositionClass *class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass*) class;

	object_class->finalize = position_finalize;
}

static void
position_init (Position *pos)
{
	pos->priv = g_new0 (PositionPrivate, 1);

	position_set_empty (pos);
}

GtkObject *
position_new ()
{
	return GTK_OBJECT (gtk_type_new (
		position_get_type ()));
}

GtkObject *
position_new_initial ()
{
	Position *pos;

	pos = POSITION (position_new ());
	position_set_initial (pos);

	return GTK_OBJECT (pos);
}

Position *
position_copy (Position *pos)
{
	Position *cpPos;

	cpPos = POSITION (position_new ());

	memcpy (cpPos->priv, pos->priv, sizeof (PositionPrivate));
	memcpy (cpPos->square, pos->square, sizeof (gchar) * 120);

	return cpPos;
}

static void
position_set_empty (Position *pos)
{
     unsigned int a;            

     for (a = 0 ; a < 120 ; a++)
          pos->square [a] = EMPTY;

     for (a = 0; a < 10; a++) {
          pos->square [a]          = BORDER;
          pos->square [a + 10]     = BORDER;
          pos->square [a + 100]    = BORDER;
          pos->square [a + 110]    = BORDER;
          pos->square [a * 10]     = BORDER;
          pos->square [a * 10 + 9] = BORDER;
     }

     pos->priv->wki = 0;
     pos->priv->wth = 0;
     pos->priv->wta = 0;
     pos->priv->ski = 0;
     pos->priv->sth = 0;
     pos->priv->sta = 0;
     pos->priv->vf = EMPTY;

     pos->priv->tomove = NONE;  

}

void
position_set_initial (Position *pos)
{
	unsigned int a;

	/* The white pieces */
    	pos->square [A1] = WR;
	pos->square [B1] = WN;
	pos->square [C1] = WB;
	pos->square [D1] = WQ;
    	pos->square [E1] = WK;
	pos->square [F1] = WB;
	pos->square [G1] = WN;
	pos->square [H1] = WR;

	/* The black pieces */
	pos->square [A8] = BR;
	pos->square [B8] = BN;
	pos->square [C8] = BB;
	pos->square [D8] = BQ;
	pos->square [E8] = BK;
	pos->square [F8] = BB;
	pos->square [G8] = BN;
	pos->square [H8] = BR;

	/* Pawns on the 2nd and 7th ranks */
	for (a = A2; a <= H2 ;a++) 
		pos->square [a] = WP;
	for (a = A7; a <= H7 ;a++)
		pos->square [a] = BP;

	/* The rest is blank */
	for (a = A3; a <= H3 ;a++)
		pos->square [a] = EMPTY;
	for (a = A4; a <= H4 ;a++)
		pos->square [a] = EMPTY;
	for (a = A5; a <= H5 ;a++)
		pos->square [a] = EMPTY;
	for (a = A6; a <= H6 ;a++)
		pos->square [a] = EMPTY;

	/* White to move, white king on E1 and black king on E8 */
	pos->priv->wta = 0;
	pos->priv->wth = 0;
	pos->priv->wki = E1;
	pos->priv->sta = 0;
	pos->priv->sth = 0;
	pos->priv->ski = E8;
	pos->priv->vf = EMPTY;
	pos->priv->tomove = WHITE;
}

/* 
 * Move Functions
 *
 * A set of functions to make a move in the context of the position 
 * passed in.
 */
static void
position_move_white_castle_short (Position *pos)
{
        pos->square [E1] = pos->square [H1] = EMPTY;
        pos->square [F1] = WR;
        pos->square [G1] = WK;
}

static void
position_move_white_castle_long (Position *pos)
{
        pos->square [E1] = pos->square [A1] = EMPTY;
        pos->square [C1] = WK;
        pos->square [D1] = WR;
}

static void
position_move_black_castle_short (Position *pos)
{
        pos->square [E8] = pos->square [H8] = EMPTY;
        pos->square [F8] = BR;
        pos->square [G8] = BK;
 }

static void
position_move_black_castle_long (Position *pos)
{
        pos->square [E8] = pos->square [A8] = EMPTY;
        pos->square [C8] = BK;
        pos->square [D8] = BR;
}

static void
position_move_white (Position *pos, int from, int to)
{
	gshort piece, zfeld;

	piece = pos->square [from];

	switch (piece) {
	case WP :
		/* If we are promoting a pawn */
		if (to & 128) {
			zfeld = (to & 7) + A8;            
			piece = ((to & 127) >> 3 ) + WP - 1;

			pos->priv->vf = pos->square [zfeld];
			pos->square [from]  = EMPTY;
			pos->square [zfeld] = piece;
			pos->priv->ep = EMPTY;
			return;
		}

		/* If we are capturing en passent */
		if ((to - from) != 10) {
			if((to - from) != 20) {
				if(pos->square [to] == EMPTY) {
					pos->square [to - 10] = EMPTY;
					pos->priv->ep = EMPTY;
					pos->square [to] = WP;
					pos->square [from] = EMPTY;
					pos->priv->vf = EMPTY;
					return;
				}
			}
		}

		pos->priv->vf = pos->square [to];
		pos->square [to]   = piece;
		pos->square [from] = EMPTY;

		if ((to - from) == 20)
			pos->priv->ep = to;
		else  
			pos->priv->ep = EMPTY;
		return;

	case WK :
		pos->priv->ep = EMPTY;
		pos->priv->wki = to;
		pos->priv->wta += 1;
		pos->priv->wth += 1;

		/*  If we are not castling */
		if (from != E1 || abs (to - from) != 2) {
			pos->priv->vf = pos->priv->vf = pos->square [to];
			pos->square [to]   = piece; 
			pos->square [from] = EMPTY;
			return;
		}

		/*  If we are castling */
		switch (to) {
		case G1 :
			position_move_white_castle_short (pos);
			break;  
		case C1 :
			position_move_white_castle_long (pos);
			break;  
		default :       
			abort ();
		}
		return;

	default :
		pos->priv->ep = EMPTY;
		pos->priv->vf = pos->square [to];
		pos->square [to] = piece;
		pos->square [from]  = EMPTY;

		/* Indicate if a rook has moved */
		if (piece == WR && from == A1)
			pos->priv->wta += 1;
		if (piece == WR && from == H1)
			pos->priv->wth += 1;
		return;
	}
}

static void
position_move_black (Position *pos, int from, int to)
{
	gshort piece, zfeld;

	piece = pos->square [from];

	switch (piece) {
	case BP :
		/* If we are promoting a pawn */
		if (to & 128) {
			zfeld = (to & 7) + A1;
			piece = ((to & 127) >> 3) + BP - 1;
			pos->priv->vf = pos->square [zfeld];
			pos->square [from]  = EMPTY;
			pos->square [zfeld] = piece;
			pos->priv->ep = EMPTY;
			return;
		}

		/* If we are capturing en passent */
		if ((from - to) != 10) {
			if ((from - to) != 20) {
				if (pos->square [to] == EMPTY) {
					pos->square [to + 10] = EMPTY;
					pos->priv->ep = EMPTY;
					pos->square [to] = BP;
					pos->square [from] = EMPTY;
					pos->priv->vf = EMPTY;
					return;
				}
			}
		}

		pos->priv->vf = pos->square [to];
		pos->square [to] = piece;
		pos->square [from] = EMPTY;

		if ((from - to) == 20)
			pos->priv->ep = to;
		else                   
			pos->priv->ep = EMPTY;	
		return;

	case BK :
		pos->priv->ep = EMPTY;
		pos->priv->ski = to;
		pos->priv->sta += 1;
		pos->priv->sth += 1;

		/*  If we are not castling */
		if (from != E8 || abs (to - from) != 2) {
			pos->priv->vf = pos->square [to];
			pos->square [to] = piece;
			pos->square [from]  = EMPTY;
			return;
		}

		/*  If we are castling */
		switch (to) {
		case G8 :       
			position_move_black_castle_short (pos);
			break; 
		case C8 :
			position_move_black_castle_long (pos);
			break; 
		default :
			abort();
		}
		return;

	default :
		pos->priv->ep = EMPTY;
		pos->priv->vf = pos->square [to];
		pos->square [to]   = piece;
		pos->square [from] = EMPTY;

		/* Indicate if a rook has moved */
		if (piece == BR && from == A8)
			pos->priv->sta += 1;
		if (piece == BR && from == H8)
			pos->priv->sth += 1;
		return;
	}
}

void position_move(Position *pos, int from, int to) {

	switch (pos->priv->tomove) {
		case WHITE:
			position_move_white (pos, from, to);
			pos->priv->tomove = BLACK;
			break;
		case BLACK:
			position_move_black (pos, from, to);
			pos->priv->tomove = WHITE;
			break;
        	default:
			abort();
	}
}

static gshort
position_move_normalize_promotion (Position *pos, gshort to, gshort n1,
				   gshort n2, gshort n3, gshort n4)
{
        gchar help;
        gshort fi=0, c;

        if  (pos->priv->tomove == WHITE)
		help = to - A8;
        else
		help = to - A1;

        if (help == (n1 & 7)) {
		if (prefs_get_promotetoqueen ())
			c = 0;
		else
			c = dialog_promotion ();

		if (pos->priv->tomove == WHITE) 
			switch (c) {
			case 0 : fi = WQ; break;
			case 1 : fi = WR; break;
			case 2 : fi = WB; break;
			case 3 : fi = WN; break;
			}
		else 
			switch (c) {
			case 0 : fi = BQ; break;
			case 1 : fi = BR; break;
			case 2 : fi = BB; break;
			case 3 : fi = BN; break;
			}

		if (pos->priv->tomove == WHITE)
			help = 128 + to - A8 + (fi - WP + 1) * 8;
		else
			help = 128 + to - A1 + (fi - BP + 1) * 8;

		if (help == n1)
			return n1;
		if (help == n2)
			return n2;
		if (help == n3)
			return n3;
		if (help == n4)
			return n4;
	  
        }
        return FALSE;
}

gshort
position_move_normalize (Position *pos, gshort from, gshort to)
{
	gshort anz, anz_n, anz_s, ret;
	gshort i;
	gchar *ap;
	gchar *aq;
	gchar movelist [AB_ZUGL];

	ap =  movelist + AB_ZUG_S;
	anz = position_legal_move (pos, &ap, &anz_s, &anz_n);

	for (aq = ap, i = 0; i < anz; i++, aq += 2 ) {
		if (from == *aq) {
			if (to == *(aq + 1)) 
				return to;
			else {
				if (*(aq + 1) < 0) { /* Promotion */
					ret = position_move_normalize_promotion (
					pos, to, *(aq + 1), *(aq + 3),
					*(aq + 5), *(aq + 7));
					if (ret)
						return ret;
					aq += 6; /* skip 3 moves */
				}
			}
		}
	}
	return FALSE;
}

/* 
 * Move Reverse Functions
 *
 * A set of functions to reverse a previous move in the context of the position 
 * passed in.
 *
 */
static void
position_move_reverse_castling_white_short (Position *pos)
{
        pos->square [E1] = WK;
        pos->square [F1] = pos->square [G1] = EMPTY;
        pos->square [H1] = WR;
}

static void
position_move_reverse_castling_white_long (Position *pos)
{
        pos->square [A1] = WR;
        pos->square [C1] = pos->square [D1] = EMPTY;
        pos->square [E1] = WK;
}

static void
position_move_reverse_castling_black_short (Position *pos)
{
        pos->square [E8] = BK;
        pos->square [F8] = pos->square [G8] = EMPTY;
        pos->square [H8] = BR;
}

static void
position_move_reverse_castling_black_long (Position *pos)
{
        pos->square [A8] = BR;
        pos->square [D8] = pos->square [C8] = EMPTY;
        pos->square [E8] = BK;
}

static void
position_move_reverse_promotion_white (Position *pos, int from, int to)
{
	pos->square [from] = WP;
	pos->square [(to & 7) + A8] = pos->priv->vf;
}

static void
position_move_reverse_promotion_black (Position *pos, int from, int to)
{
	pos->square [from] = BP;
	pos->square [(to & 7) + A1] = pos->priv->vf;
}

void
position_move_reverse_white (Position *pos, int from, int to) 
{
	int fi;

	pos->priv->tomove = WHITE; /* change tomove */

	if (to & 128) {        /*    Promotion     */
		position_move_reverse_promotion_white (pos,from,to);
		return;
	}

        fi = pos->square[to];

        /*    look for castling   */
        if (fi == WK) {
        	pos->priv->wki = from;
		pos->priv->wta -= 1;
		pos->priv->wth -= 1;
		
        	if (from != E1)  {  /*    no castling   */
               		pos->square[from]  = fi;
               		pos->square[to] = pos->priv->vf;
               		return;
        	}
        	if (abs(from-to) != 2)  {  /*  no castling  */
               		pos->square[from]  = fi;
               		pos->square[to] = pos->priv->vf;
               		return;
        	}

        	if (to == G1) {
			position_move_reverse_castling_white_short (pos);
			return;
		} else if (to == C1) {
			position_move_reverse_castling_white_long (pos);
			return;
		}
        	abort();
	}

	if (fi == WR && from == A1)
		pos->priv->wta -= 1;
	if (fi == WR && from == H1)
		pos->priv->wth -= 1;

        if (fi == WP) {
        	if ((to - from) != 10) {
          		if ((to - from) != 20) {
            			if (pos->priv->vf == EMPTY) {
                			pos->square [to - 10] = BP;
                			pos->square [to] = EMPTY;
                			pos->square [from] = WP;
          			        return;
        			}
			}
		}
        	pos->square [from] = fi;
        	pos->square [to] = pos->priv->vf;
		return;
	}
        pos->square [from]  = fi;
        pos->square [to] = pos->priv->vf;
}

void
position_move_reverse_black (Position *pos, int from, int to) 
{
	int fi;

	pos->priv->tomove = BLACK;      /* change tomove */

	if (to & 128) {        /*    Promotion  */
		position_move_reverse_promotion_black(pos,from,to);
		return;
	}

        fi = pos->square[to];

        /* look for castling */
        if (fi == BK) {
        	pos->priv->ski = from;
		pos->priv->sta -= 1;
		pos->priv->sth -= 1;
		
        	if (from != E8)  {  /*    no castling   */
               		pos->square[from]  = fi;
               		pos->square[to] = pos->priv->vf;
               		return;
        	}
        	if (abs(from-to) != 2)  {  /*    no castling  */
               		pos->square[from]  = fi;
               		pos->square[to] = pos->priv->vf;
               		return;
        	}

        	if (to == G8) {
			position_move_reverse_castling_black_short(pos);
			return;
		} else if (to == C8) {
			position_move_reverse_castling_black_long(pos);
			return;
		}
        	abort();
	}

	if (fi == BR && from == A8)
		pos->priv->sta -= 1;
	if (fi == BR && from == H8)
		pos->priv->sth -= 1;

        if (fi == BP) {
        	if ((from-to) !=10) {
          		if ((from-to) !=20) {
            			if (pos->priv->vf == EMPTY) {
                			pos->square[to+10] = WP;
                			pos->square[to] = EMPTY;
                			pos->square[from] = BP;
          			        return;
        			}
			}
		}
        	pos->square [from] = fi;
        	pos->square [to]= pos->priv->vf;
		return;
	}
	pos->square [from]  = fi;
        pos->square [to] = pos->priv->vf;
}

void
position_move_reverse (Position *pos, int from, int to)
{
	switch (pos->priv->tomove) {
		case WHITE:
			position_move_reverse_black (pos, from, to);
			break;
		case BLACK:
			position_move_reverse_white (pos, from, to);
			break;
        	default:
			abort();
	}
}

/* 
 * Move Generate Functions
 *
 * A set of functions to generate moves in the context of the position 
 * passed in.
 *
 */

static void inline new_move(gshort from,gshort to) {
	*nindex = (gchar) from;
	*( nindex + 1 ) = (gchar) to;
	nindex += 2;
}


static void new_smove(gshort from,gshort to) {
	sindex -= 2;
	*sindex = (gchar) from;
	*(sindex + 1 )  = (gchar) to;
}

static void wdouble(Position *pos, gshort af,gshort a,gshort b) {
	gshort jp, ef = af, piece;

	for ( ; a < b ; a ++ ) {
		ef = af;
		jp = jump[a];

		do {
			ef += jp;
			piece  = pos->square[ef];

			if ( piece == EMPTY )
				new_move(af,ef);
			else if ( piece == BORDER )
				break;
			else if ( BPIECE(piece) ) {
				new_smove(af,ef);
				break;
			} else
				break;
		} while ( TRUE );
        }
}

static void white_promotion(gshort af,gshort ef) {
	register gshort b,i;

	for (i=2;i<6;i++) {
		b = 128 + 8 * i + ef - A8;
		new_smove(af,b);
	}
}

static void wpawn2(Position *pos, gshort af) {
	register gshort ef;

	ef = af + 10;    
	if (pos->square[ef] == EMPTY ) {
		new_move(af,ef);
	        ef = af + 20;
		if (pos->square[ef] == EMPTY ) {
			new_move(af,ef);
		}
	}

	if (BPIECE(pos->square[af + 9 ])) new_smove(af,af+9);
	if (BPIECE(pos->square[af +11 ])) new_smove(af,af+11);
}

static void wpawn3(Position *pos, gshort af) {
	register gshort ef;

	ef = af + 10;
	if ( pos->square[ef] == EMPTY ) new_move(af,ef);

	if ( BPIECE( pos->square[af + 9 ] ) ) new_smove(af, af + 9 );
	if ( BPIECE( pos->square[af +11 ] ) ) new_smove(af, af + 11 );
}

static void wpawn5(Position *pos, gshort af) {

	wpawn3(pos,af);

	if ( ( af - 1 ) == pos->priv->ep )  
		new_smove(af,af+9);
	else if ( ( af + 1 ) == pos->priv->ep )  
		new_smove(af,af+11);
}

static void wpawn7(Position *pos, gshort af) {
	register gshort ef;

	ef = af + 10;
	if (pos->square[ef] == EMPTY ) white_promotion(af,ef);

	if (BPIECE( pos->square[af + 9] )) white_promotion( af , af + 9);
	if (BPIECE( pos->square[af +11] )) white_promotion(af , af +11);
}

static void wknight(Position *pos, gshort af) {
	register gshort a,piece,bb;

	for (a = 0 ; a < 8;a++) {
		bb=af+jump[a];
		piece=pos->square[bb];

		switch (piece) {
			case EMPTY:
				new_move(af,bb);
				break;
			case BORDER:  
				break;
			default:
				if (BPIECE(piece))
					new_smove(af,bb);
		}
	}
}

static void wbishop(Position *pos, gshort af) {
	gshort a = 8,
        b = 12;

	wdouble(pos, af,a,b);
}

static void wrook(Position *pos, gshort af) {
	gshort a = 12, b = 16;

	wdouble(pos, af,a,b);
}

static void wqueen(Position *pos, gshort af) {
	gshort a = 8, b = 16;

	wdouble(pos, af,a,b);
}

static void wking(Position *pos, gshort af) {
	register gshort a;
	register gshort piece,bb;

	for (a = 8; a < 16 ; a++) {
		bb=af+jump[a];
		piece = pos->square[bb];

		switch (piece) {
			case EMPTY:
				new_move(af,bb);
				break;
			case BORDER:
				break;
			default:
				if (BPIECE(piece))
					new_smove(af,bb);
		}
	}
}

static void w_ro_k(Position *pos) {
	if ( pos->square[F1] == EMPTY
	  && pos->square[G1] == EMPTY
	  && pos->square[H1] == WR)
		new_move(E1,G1);
}

static void w_ro_l(Position *pos){
	if (pos->square[D1] == EMPTY &&
	  pos->square[C1] == EMPTY &&
	  pos->square[B1] == EMPTY &&
	  pos->square[A1] == WR) 
		new_move(E1,C1);
}

static void
wkingro (Position *pos, gshort af)
{
	register gshort a;
	register gshort piece,bb;

	for (a = 8; a < 16 ; a++) {
		bb=af+jump[a];
		piece=pos->square[bb];

		switch (piece) {
		case EMPTY:
			new_move(af,bb);
			break;
		case BORDER:
			break;
		default:
			if (BPIECE(piece))
				new_smove(af,bb);
		}
	}

	if (pos->priv->wki != E1)
		return;
	if (!pos->priv->wth)
		w_ro_k(pos);
	if (!pos->priv->wta)
		w_ro_l(pos);
}

static void sdouble(Position *pos, gshort af,gshort a,gshort b) {
	gshort jp, ef = af, piece;

        for ( ; a < b ; a ++ ) {
		ef = af;
		jp = jump[a];

		do {
			ef += jp;
                        piece  = pos->square[ef];

			if ( piece == EMPTY )
				new_move(af,ef);
			else if ( piece == BORDER )
				break;
			else if ( WPIECE(piece) ) { 
				new_smove(af,ef);
				break;
			} else
				break;
                } while ( TRUE );
        }
}

static void black_promotion(gshort af,gshort ef) {
	register gshort b,i;

	for (i=2;i<6;i++) {
		b = 128 + 8 * i + ef - A1;
		new_smove(af,b);
	}
}

static void bpawn7(Position *pos, gshort af) {
	register gshort ef;

	ef = af - 10;                
	if (pos->square[ef] == EMPTY ) {
		new_move(af,ef);
		ef = af - 20;            
		if (pos->square[ef] == EMPTY ) {
			new_move(af,ef);
		}
	}

	if (WPIECE(pos->square[af - 9 ])) new_smove(af,af-9);
	if (WPIECE(pos->square[af -11 ])) new_smove(af,af-11);
}

static void bpawn6(Position *pos, gshort af) {
	register gshort ef;

	ef = af - 10;
	if ( pos->square[ef] == EMPTY ) new_move(af,ef);

	if ( WPIECE( pos->square[af - 9 ] ) ) new_smove(af, af - 9 );
	if ( WPIECE( pos->square[af -11 ] ) ) new_smove(af, af - 11 );
}


static void bpawn4(Position *pos, gshort af) {

	bpawn6(pos, af);

	if (( af - 1) == pos->priv->ep )
		new_smove( af, af-11);
	if (( af + 1) == pos->priv->ep ) 
		new_smove( af, af-9);
}

static void bpawn2(Position *pos, gshort af) {
	register gshort ef;

	ef = af - 10;
	if (pos->square[ef] == EMPTY ) black_promotion(af,ef);


	if (WPIECE( pos->square[af - 9] )) black_promotion( af , af - 9);
	if (WPIECE( pos->square[af -11] )) black_promotion (af , af -11);
}

static void bknight(Position *pos, gshort af) {
	register gshort a,piece,bb;

	for (a = 0 ; a < 8;a++) {
		bb=af+jump[a];
		piece=pos->square[bb];

		switch (piece) {
			case EMPTY:
				new_move(af,bb);
				break;
			case BORDER:
				break;
			default:
				if (WPIECE(piece))
					new_smove(af,bb);
		}
	}
}

static void bbishop(Position *pos, gshort af) {
	gshort a=8, b=12;

	sdouble(pos, af,a,b);
}

static void brook(Position *pos, gshort af) {
	gshort a = 12, b = 16;

	sdouble(pos, af,a,b);
}

static void bqueen(Position *pos, gshort af) {
	gshort a=8, b=16;

	sdouble(pos, af,a,b);
}

static void bking(Position *pos, gshort af) {
	register gshort a;
	register gshort piece,bb;

	for (a = 8; a < 16 ; a++) {
		bb=af+jump[a];
		piece=pos->square[bb];

		switch (piece) {
			case EMPTY:
				new_move(af,bb);
				break;
			case BORDER:
				break;
			default:
				if (WPIECE(piece))
					new_smove(af,bb);
		}
	}
}

static void b_ro_k(Position *pos) {

  if ( pos->square[F8] == EMPTY &&
       pos->square[G8] == EMPTY && 
       pos->square[H8] == BR)
    new_move(E8,G8);

}

static void b_ro_l(Position *pos) {
  if (pos->square[D8] == EMPTY &&
      pos->square[C8] == EMPTY &&
      pos->square[B8] == EMPTY &&
      pos->square[A8] == BR) 
    new_move(E8,C8);
}

static void
bkingro (Position *pos, gshort af) 
{
	register gshort a;
	register gshort piece,bb;

	for (a = 8; a < 16 ; a++) {
		bb=af+jump[a];
		piece=pos->square[bb];

		switch (piece) {
		case EMPTY:
			new_move(af,bb);break;
		case BORDER:
			break;
		default:
			if (WPIECE(piece))
				new_smove(af,bb);
		}
	}

	if ( pos->priv->ski != E8)
		return;
	if (!pos->priv->sth)
		b_ro_k(pos);  
	if (!pos->priv->sta)
		b_ro_l(pos);
}

static int position_move_generator_white(Position *pos, gchar **index0,gshort *anz_s,gshort *anz_n)
{
	register gshort rank,af,piece;

	nindex = sindex = *index0;

	for(rank=1;rank<=8;rank++) { 
		for( af = A1 + ((rank - 1) * 10); af <= H1 + ((rank - 1) * 10); af++ ) {
			piece = pos->square[af];
			if ( WPIECE(piece)) { 
				switch (piece)   {
					case WP: 
						switch (rank) {
							case 1:
							case 8:
								abort();break;
							case 2:
								wpawn2(pos, af);break;
							case 3:
							case 4:
							case 6:
								wpawn3(pos, af);break;
							case 5:
								wpawn5(pos, af);break;
							case 7:
								wpawn7(pos, af);break;
						}
						break;
					case WN: wknight(pos, af);break;
					case WB: wbishop(pos, af);break;
					case WR: wrook(pos, af);break;
					case WQ: wqueen(pos, af);break;
			                case WK:
						if (rank == 1) 
							wkingro(pos, af);
						else
							wking(pos, af);
						break;
				}
			}
		}
	}

	*anz_n = (gshort) (( nindex  - *index0) / 2) ;
	*anz_s = (gshort) (( *index0 - sindex) / 2 );

	af = *anz_n + *anz_s ;
	*index0 = sindex;

	return af;
}

static int position_move_generator_black(Position *pos, gchar **index0,gshort *anz_s,gshort *anz_n)
{
   register gshort af,piece,rank;

   nindex = sindex = *index0;

	for(rank=1;rank<=8;rank++) { 
		for( af = A1 + ((rank - 1) * 10); af <= H1 + ((rank - 1) * 10); af++ ) {
			piece = pos->square[af];
			if ( BPIECE(piece)) { 
				switch (piece)   {
					case BP: 
						switch (rank) {
							case 1:
							case 8:
								abort();break;
							case 2:
								bpawn2(pos, af);break;
							case 3:
							case 5:
							case 6:
								bpawn6(pos, af);break;
							case 4:
								bpawn4(pos, af);break;
							case 7:
								bpawn7(pos, af);break;
						}
						break;
					case BN: bknight(pos, af);break;
					case BB: bbishop(pos, af);break;
					case BR: brook(pos, af);break;
					case BQ: bqueen(pos, af);break;
			                case BK:
						if (rank == 8) 
							bkingro(pos, af);
						else
							bking(pos, af);
						break;
				}
			}
		}
	}

	*anz_n = (gshort) (( nindex  - *index0) / 2) ;
	*anz_s = (gshort) (( *index0 - sindex) / 2 );
     
	af = *anz_n + *anz_s ;
	*index0 = sindex;

	return af;
}

gint position_move_generator(Position *pos, gchar **index0,gshort *anz_s,gshort *anz_n) {

	if (pos->priv->tomove == WHITE ) 
		return position_move_generator_white(pos, index0,anz_s,anz_n);
	else if (pos->priv->tomove == BLACK ) 
		return position_move_generator_black(pos, index0,anz_s,anz_n);
	else
		abort();
}

/* 
 * Position Characteristic Functions
 *
 * A set of functions to give information about the position
 * passed in.
 *
 */

#define CHECK(king_place,direction,piece1,piece2)\
{  int 	i= king_place; \
   do { i += direction; } \
   while(!pos->square[i]); \
   if (pos->square[i] == piece1) return piece1; \
   if (pos->square[i] == piece2) return piece2; \
}

static int long4(Position *pos, int ort,int r1,int r2,int r3,int r4,int f1,int f2)
{

  CHECK(ort,r1,f1,f2);
  CHECK(ort,r2,f1,f2);
  CHECK(ort,r3,f1,f2);
  CHECK(ort,r4,f1,f2);
  return FALSE;
}   

#define KURZ_TEST(r)   if (pos->square[ort+r] == f1) return f1

static int short8(Position *pos, int ort,int r1,int r2,int r3,int r4,int r5,int r6,int r7,int r8,int f1)
{
  KURZ_TEST(r1);
  KURZ_TEST(r2);
  KURZ_TEST(r3);
  KURZ_TEST(r4);
  KURZ_TEST(r5);
  KURZ_TEST(r6);
  KURZ_TEST(r7);
  KURZ_TEST(r8);

  return FALSE;
}

gboolean position_white_king_attack(Position *pos) {
	int ret;
	int k= pos->priv->wki;

	ret = long4(pos,k,9,11,-9,-11,BQ,BB);
	if (ret) return ret;
   
	ret = long4(pos,k,1,10,-10,-1,BQ,BR);
	if (ret) return ret;
   
	if( short8(pos, k,8,12,19,21,-8,-12,-19,-21,BN)) return BN;
	if( short8(pos, k,9,11,-9,-11,1,10,-10,-1,BK)) return BK;
   
	if (pos->square[k+OL] == BP)
		return BP;
	if (pos->square[k+OR] == BP)
		return BP;

	return FALSE;
}

gboolean
position_black_king_attack (Position *pos)
{
	int ret;
	int k = pos->priv->ski;
	
	ret = long4(pos,k,9,11,-9,-11,WQ,WB);
	if (ret) return ret;
   
	ret = long4(pos,k,1,10,-10,-1,WQ,WR);
	if (ret) return ret;
   
	if( short8(pos, k,8,12,19,21,-8,-12,-19,-21,WN)) return WN;
	if( short8(pos, k,9,11,-9,-11,1,10,-10,-1,WK)) return WK;
   
	if (pos->square[k+UL] == WP)
	   	return WP;
	if (pos->square[k+UR] == WP)
		return WP;

	return FALSE;
}

gshort
position_legal_move (Position *pos, gchar **zl, gshort *as, gshort *an)
{
	Position temp;
	gshort a;
	gchar *ap,*ap2;
	gshort anz,anz_s,anz_n,anzahl,check=0;
	gshort tomove = pos->priv->tomove;
	gchar zugl[AB_ZUGL];

        ap =  zugl + AB_ZUG_S;
        anz = position_move_generator(pos,&ap,&anz_s,&anz_n);
        anzahl = 0;
        ap2 = *zl;

        for(a=0;a<anz;a++)
        {

                temp = *pos;
                position_move(pos, *ap,*(ap+1));

                switch (tomove)
                {
                        case WHITE: check = position_white_king_attack(pos);
                                  break;
                        case BLACK: check = position_black_king_attack(pos);
                                  break;
                        default : abort();
                }

                if (!check) {
                        *ap2++=*ap++;
                        *ap2++=*ap++;
                        anzahl++;
                } else {
		        ap+=2;
		}
                switch (tomove) {
                        case WHITE:
                                position_move_reverse_white(pos,*(ap-2),*(ap-1));
                                break;
                        case BLACK:
                                position_move_reverse_black(pos,*(ap-2),*(ap-1));
                                break;
                }
                *pos = temp;
        }
        *as = anzahl;
	*an = 0;
        return anzahl;
}

short
position_last_piece_captured (Position *pos)
{
	g_return_val_if_fail (pos != NULL, EMPTY);

	return pos->priv->vf;
}

short
position_get_color_to_move (Position *pos)
{
	g_return_val_if_fail (pos != NULL, WHITE);

	return pos->priv->tomove;
}

void
position_set_white_king (Position *pos, short wki)
{
	g_return_if_fail (pos != NULL);

	pos->priv->wki = wki;
}

void
position_set_black_king (Position *pos, short ski)
{
	g_return_if_fail (pos != NULL);

	pos->priv->ski = ski;
}

void
position_set_color_to_move (Position *pos, short tomove)
{
	g_return_if_fail (pos != NULL);

	pos->priv->tomove = tomove;
}
