/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* server-term.c
 *
 * Copyright (C) 2001  JP Rosevear.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include "zvt.h"
#include "server-term.h"

static void class_init (ServerTermClass *klass);
static void init (ServerTerm *term);

static void st_command_activate_cb (GtkWidget *widget, gpointer data);
static gboolean st_command_key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer data);

static GtkVBoxClass *parent_class = NULL;

struct _ServerTermPrivate {
	GtkWidget *term;

	GList *above;
	gchar *current;
	GList *below;
	gboolean on_stack;
	
	GtkWidget *prompt;
	GtkWidget *command;
};

/* Class signals */
enum {
	COMMAND_SIGNAL,
	LAST_SIGNAL
};
static gint term_signals[LAST_SIGNAL] = { 0 };

GtkType
server_term_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "ServerTerm",
        sizeof (ServerTerm),
        sizeof (ServerTermClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gtk_vbox_get_type (), &info);
    }

  return type;
}

static void
class_init (ServerTermClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (gtk_vbox_get_type ());

	term_signals[COMMAND_SIGNAL]
		= gtk_signal_new ("command",
				  GTK_RUN_FIRST,
				  object_class->type,
				  GTK_SIGNAL_OFFSET (ServerTermClass, command),
				  gtk_marshal_NONE__STRING,
				  GTK_TYPE_NONE, 1,
				  GTK_TYPE_STRING);

         gtk_object_class_add_signals (object_class,
				       term_signals,
				       LAST_SIGNAL);
}


static void
init (ServerTerm *term)
{
	ServerTermPrivate *priv;
	GtkWidget *sb, *hbox;
	gushort red[18] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0xaaaa,0x0000};
	gushort green[18] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0xaaaa,0x0000};
	gushort blue[18] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0xaaaa,0x0000};
	
	priv = g_new0 (ServerTermPrivate, 1);

	term->priv = priv;

	gtk_box_set_spacing (GTK_BOX (term), 4);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox);
	
	priv->term = zvt_term_new_with_size (80, 12);
	zvt_term_set_blink (ZVT_TERM (priv->term), FALSE);
	zvt_term_set_scrollback (ZVT_TERM (priv->term), 100);
	zvt_term_set_color_scheme (ZVT_TERM (priv->term), red, green, blue);
	gtk_widget_show (priv->term);

	sb = gtk_vscrollbar_new (GTK_ADJUSTMENT (ZVT_TERM (priv->term)->adjustment));
	GTK_WIDGET_UNSET_FLAGS (sb, GTK_CAN_FOCUS);
	gtk_widget_show (sb);

	gtk_box_pack_start (GTK_BOX (hbox), priv->term, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), sb, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (term), hbox, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 5);
	gtk_widget_show (hbox);

	priv->prompt = gtk_label_new ("Prompt");
	gtk_widget_show (priv->prompt);
	gtk_box_pack_start (GTK_BOX (hbox), priv->prompt, FALSE, FALSE, 0);
	
	priv->command = gtk_entry_new ();
	gtk_widget_show (priv->command);
	gtk_box_pack_start (GTK_BOX (hbox), priv->command, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (term), hbox, FALSE, FALSE, 0);

	/* Attach the command call backs */
	gtk_signal_connect (GTK_OBJECT (priv->command), "key-press-event",
			    (GtkSignalFunc) st_command_key_press_cb,
			    term);			    
	gtk_signal_connect (GTK_OBJECT (priv->command), "activate",
			    (GtkSignalFunc) st_command_activate_cb,
			    term);
}



GtkWidget *
server_term_new (void)
{
	GtkWidget *widget;
	
	widget = gtk_type_new (server_term_get_type ());

	return widget;
}

void
server_term_set_prompt (ServerTerm *term, gchar *prompt)
{
	ServerTermPrivate *priv;

	g_return_if_fail (term != NULL);
	g_return_if_fail (IS_SERVER_TERM (term));
	g_return_if_fail (prompt != NULL);

	priv = term->priv;

	gtk_label_set (GTK_LABEL (priv->prompt), prompt); 
}

void
server_term_print (ServerTerm *term, gchar *text)
{
	ServerTermPrivate *priv;
	
	g_return_if_fail (term != NULL);
	g_return_if_fail (IS_SERVER_TERM (term));
	g_return_if_fail (text != NULL);

	priv = term->priv;
	
	zvt_print_raw_text (priv->term, text);
}

static gboolean
st_command_key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	ServerTerm *term = SERVER_TERM (data);
	ServerTermPrivate *priv;
	
	priv = term->priv;

	switch (event->keyval) {
	case GDK_Up:
		if (priv->above != NULL) {
			g_free (priv->current);
			priv->current = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
			priv->below = g_list_prepend (priv->below, priv->current);
			priv->current = priv->above->data;
			priv->above = g_list_remove (priv->above, priv->current);
			priv->on_stack = TRUE;

			gtk_entry_set_text (GTK_ENTRY (widget), priv->current);
		}
		gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), 
					      "key-press-event");
		break;
	case GDK_Down:
		if (priv->below != NULL) {			
			priv->above = g_list_prepend (priv->above, priv->current);
			priv->current = priv->below->data;
			priv->below = g_list_remove (priv->below, priv->current);

			 gtk_entry_set_text (GTK_ENTRY (widget), priv->current);
		}		
		gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), 
					      "key-press-event");
		break;
	}

	return TRUE;
}

static void
st_command_activate_cb (GtkWidget *widget, gpointer data)
{
	ServerTerm *term = SERVER_TERM (data);
	ServerTermPrivate *priv;
	GList *l;
	gchar *command;
	
	priv = term->priv;

        command = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
	if (command == NULL || !strcmp (command, ""))
		return;

	/* Push the current on to the stack */
	if (priv->current != NULL)
		priv->above = g_list_prepend (priv->above, priv->current);
	priv->current = NULL;

	/* Push the remaining items on to the stack */	
	for (l = priv->below; l != NULL; l = l->next) {
		gchar *command = l->data;
		
		priv->above = g_list_prepend (priv->above, command);
	}
	g_list_free (priv->below);
	priv->below = NULL;	

	/* Push the current command on to the stack */
	if (priv->on_stack)
		priv->above = g_list_remove (priv->above, priv->above->data);
	priv->above = g_list_prepend (priv->above, command);
	priv->on_stack = FALSE;
	
	gtk_signal_emit (GTK_OBJECT (term), 
			 term_signals[COMMAND_SIGNAL], 
			 command);

	gtk_entry_set_text (GTK_ENTRY (widget), "");
}


