/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-device.h"
#include "imageloader.h"


struct _GnomeCmdDevicePrivate {	
	gchar *alias;
	gchar *device_fn;
	gchar *mountp;
	gchar *icon_path;

	GnomeCmdPixmap *pixmap;
	GnomeCmdPixmap *umnt_pixmap;
};


GnomeCmdDevice*
gnome_cmd_device_new ()
{
	GnomeCmdDevice *dev = g_malloc (sizeof (GnomeCmdDevice));
	dev->priv = g_malloc (sizeof (GnomeCmdDevicePrivate));
	dev->priv->alias = NULL;
	dev->priv->device_fn = NULL;
	dev->priv->mountp = NULL;
	dev->priv->icon_path = NULL;

	dev->priv->pixmap = NULL;
	dev->priv->umnt_pixmap = NULL;

	return dev;
}


GnomeCmdDevice*
gnome_cmd_device_new_with_values (const gchar *alias,
								  const gchar *device_fn, 
								  const gchar *mountp,
								  const gchar *icon_path)
{
	GnomeCmdDevice *dev = gnome_cmd_device_new ();

	gnome_cmd_device_set_alias (dev, alias);
	gnome_cmd_device_set_device_fn (dev, device_fn);
	gnome_cmd_device_set_mountp (dev, mountp);
	gnome_cmd_device_set_icon_path (dev, icon_path);

	return dev;
}


void
gnome_cmd_device_free (GnomeCmdDevice *dev)
{
	g_return_if_fail (dev != NULL);
	g_return_if_fail (dev->priv != NULL);
	
	if (dev->priv->alias)
		g_free (dev->priv->alias);
	if (dev->priv->device_fn)
		g_free (dev->priv->device_fn);
	if (dev->priv->mountp)
		g_free (dev->priv->mountp);
	if (dev->priv->icon_path)
		g_free (dev->priv->icon_path);

	g_free (dev->priv);
	g_free (dev);
}


void
gnome_cmd_device_set_alias (GnomeCmdDevice *dev, const gchar *alias)
{
	g_return_if_fail (dev != NULL);
	g_return_if_fail (dev->priv != NULL);
	g_return_if_fail (alias != NULL);
	
	if (dev->priv->alias)
		g_free (dev->priv->alias);
	
	dev->priv->alias = g_strdup (alias);
}


void
gnome_cmd_device_set_device_fn (GnomeCmdDevice *dev, const gchar *device_fn)
{
	g_return_if_fail (dev != NULL);
	g_return_if_fail (dev->priv != NULL);
	if (!device_fn) return;
	
	if (dev->priv->device_fn)
		g_free (dev->priv->device_fn);
	
	dev->priv->device_fn = g_strdup (device_fn);
}


void
gnome_cmd_device_set_mountp (GnomeCmdDevice *dev, const gchar *mountp)
{
	g_return_if_fail (dev != NULL);
	g_return_if_fail (dev->priv != NULL);
	if (!mountp) return;
	
	if (dev->priv->mountp)
		g_free (dev->priv->mountp);
	
	dev->priv->mountp = g_strdup (mountp);
}


void
gnome_cmd_device_set_icon_path (GnomeCmdDevice *dev, const gchar *icon_path)
{
	g_return_if_fail (dev != NULL);
	g_return_if_fail (dev->priv != NULL);
	if (!icon_path) return;
	
	if (dev->priv->icon_path)
		g_free (dev->priv->icon_path);
	
	dev->priv->icon_path = g_strdup (icon_path);

	if (dev->priv->pixmap)
		gnome_cmd_pixmap_free (dev->priv->pixmap);
	if (dev->priv->umnt_pixmap) {
		gnome_cmd_pixmap_free (dev->priv->umnt_pixmap);
		dev->priv->umnt_pixmap = NULL;
	}
	
	dev->priv->pixmap = gnome_cmd_pixmap_new_from_file (icon_path);
	if (dev->priv->pixmap) {
		GdkPixbuf *tmp = IMAGE_get_pixbuf(PIXMAP_OVERLAY_UMOUNT);
		if (tmp) {
			int w, h;
			GdkPixbuf *overlay = gdk_pixbuf_copy (dev->priv->pixmap->pixbuf);
			
			w = gdk_pixbuf_get_width (tmp);
			h = gdk_pixbuf_get_height (tmp);
			if (w > 14) w=14;
			if (h > 14) h=14;
			
			gdk_pixbuf_copy_area (tmp, 0, 0, w, h, overlay, 0, 0);
			dev->priv->umnt_pixmap = gnome_cmd_pixmap_new_from_pixbuf (overlay);
		}
	}
}


const gchar *
gnome_cmd_device_get_alias (GnomeCmdDevice *dev)
{
	g_return_val_if_fail (dev != NULL, NULL);
	g_return_val_if_fail (dev->priv != NULL, NULL);

	return dev->priv->alias;
}


const gchar *
gnome_cmd_device_get_device_fn (GnomeCmdDevice *dev)
{
	g_return_val_if_fail (dev != NULL, NULL);
	g_return_val_if_fail (dev->priv != NULL, NULL);

	return dev->priv->device_fn;
}


const gchar *
gnome_cmd_device_get_mountp (GnomeCmdDevice *dev)
{
	g_return_val_if_fail (dev != NULL, NULL);
	g_return_val_if_fail (dev->priv != NULL, NULL);

	return dev->priv->mountp;
}


const gchar *
gnome_cmd_device_get_icon_path (GnomeCmdDevice *dev)
{
	g_return_val_if_fail (dev != NULL, NULL);
	g_return_val_if_fail (dev->priv != NULL, NULL);

	return dev->priv->icon_path;
}


GnomeCmdPixmap *
gnome_cmd_device_get_pixmap (GnomeCmdDevice *dev)
{
	return dev->priv->pixmap;
}


GnomeCmdPixmap *
gnome_cmd_device_get_umount_pixmap (GnomeCmdDevice *dev)
{
	return dev->priv->umnt_pixmap;
}
