/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-file-funcs.h"
#include "gnome-cmd-main-win-types.h"
#include "dir_pool.h"
#include "gnome-cmd-fam.h"
#include "gnome-cmd-data.h"
#include "utils.h"

#define DIR_FILES_PER_NOTIFICATION 100

int dir_counter = 0;

enum {
	FILE_CREATED,
	FILE_DELETED,
	FILE_CHANGED,
	LIST_UPDATED,
	LAST_SIGNAL
};

struct _GnomeCmdDirPrivate
{
	GnomeVFSAsyncHandle *list_handle;
	GtkWidget *progwin;
	GtkWidget *progtext;
	GtkWidget *cancel_btn;
	GList *infolist;
	gboolean done;
	gboolean lock;
	gint count;	
};

static GtkObjectClass *parent_class = NULL;

static guint dir_signals[LAST_SIGNAL] = { 0 };

/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdDir *dir = GNOME_CMD_DIR (object);

	DEBUG ('d', "dir destroying 0x%x %s\n", (guint)dir, dir->path);
	
#ifdef HAVE_LIBFAM
	if (dir->request != NULL)
		fam_monitor_cancel (dir);
#endif

	dir_pool_remove (dir);
	g_list_foreach (dir->files, (GFunc)gnome_cmd_file_unref, NULL);
	gnome_vfs_uri_unref (dir->uri);

	if (dir->uri_str)
		g_free (dir->uri_str);
	if (dir->path)
		g_free (dir->path);
	if (dir->priv)
		g_free (dir->priv);
	dir_counter--;

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
class_init (GnomeCmdDirClass *class)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (class);
	parent_class = gtk_type_class (gtk_object_get_type ());

	dir_signals[FILE_CREATED] =
		gtk_signal_new ("file_created",
			GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GnomeCmdDirClass, file_created),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE,
			1, GTK_TYPE_POINTER);

	dir_signals[FILE_DELETED] =
		gtk_signal_new ("file_deleted",
			GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GnomeCmdDirClass, file_deleted),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE,
			1, GTK_TYPE_POINTER);
	
	dir_signals[FILE_CHANGED] =
		gtk_signal_new ("file_changed",
			GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GnomeCmdDirClass, file_changed),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE,
			1, GTK_TYPE_POINTER);
	
	dir_signals[LIST_UPDATED] =
		gtk_signal_new ("list_updated",
			GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GnomeCmdDirClass, list_updated),
		    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE,
			0);
	
	gtk_object_class_add_signals (object_class, dir_signals, LAST_SIGNAL);
	
	object_class->destroy = destroy;
	class->file_created = NULL;
	class->file_deleted = NULL;
	class->file_changed = NULL;
	class->list_updated = NULL;
}

static void
init (GnomeCmdDir *dir)
{
	dir->priv = g_new (GnomeCmdDirPrivate, 1);
	dir->priv->lock = FALSE;
	dir->ref_cnt = 0;
	dir->path = NULL;
	dir->uri = NULL;
	dir->uri_str = NULL;
	dir->files = NULL;
	dir->content_changed = FALSE;
	dir->prev_error = FALSE;
	dir->not_listed = TRUE;
	dir->voffset = 0;
	dir_counter++;
}



/***********************************
 * Public functions
 ***********************************/

GtkType
gnome_cmd_dir_get_type         (void)
{
	static GtkType type = 0;

	if (type == 0)
	{
		GtkTypeInfo info =
		{
			"GnomeCmdDir",
			sizeof (GnomeCmdDir),
			sizeof (GnomeCmdDirClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gtk_object_get_type (), &info);
	}
	return type;
}


GnomeCmdDir *gnome_cmd_dir_new_with_values (GnomeVFSURI *baseuri,
											const gchar *path,
											gboolean add_to_pool)
{
	GnomeCmdDir *dir;

	g_return_val_if_fail (baseuri != NULL, NULL);
	g_return_val_if_fail (path != NULL, NULL);
	
	dir = gtk_type_new (gnome_cmd_dir_get_type ());
	g_return_val_if_fail (dir != NULL, NULL);

	dir->uri = gnome_vfs_uri_append_path (baseuri, path);
	dir->uri_str = gnome_vfs_uri_to_string (dir->uri, 0);
	dir->path = g_strdup (path);

#ifdef HAVE_LIBFAM
	dir->request = NULL;
	if (gnome_cmd_data_get_use_fam ())
		fam_monitor_start (dir);
#endif

	if (add_to_pool)
		dir_pool_add (dir);
	
	return dir;
}


GnomeCmdDir *gnome_cmd_dir_new_from_uri_str (const gchar *dir_uri_str,
											 gboolean add_to_pool)
{
	GnomeCmdDir *dir;

	g_return_val_if_fail (dir_uri_str != NULL, NULL);
	
	dir = gtk_type_new (gnome_cmd_dir_get_type ());
	g_return_val_if_fail (dir != NULL, NULL);

	dir->uri = gnome_vfs_uri_new (dir_uri_str);
	dir->uri_str = g_strdup (dir_uri_str);
	dir->path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (dir->uri), 0);

#ifdef HAVE_LIBFAM
	dir->request = NULL;
	if (gnome_cmd_data_get_use_fam ())
		fam_monitor_start (dir);
#endif

	if (add_to_pool)
		dir_pool_add (dir);
	
	return dir;
}


GnomeCmdDir *gnome_cmd_dir_get_parent (GnomeCmdDir *dir)
{
	const gchar *cur_path;
	gchar *new_uri_str;
	GnomeVFSURI *new_uri;
	GnomeCmdDir *new_dir;

	cur_path = gnome_vfs_uri_get_path (gnome_cmd_dir_get_uri (dir));
	if (!cur_path || strcmp (cur_path, "/") == 0)
		return NULL;	
	
	new_uri = gnome_vfs_uri_get_parent (gnome_cmd_dir_get_uri (dir));
	new_uri_str = gnome_vfs_uri_to_string (new_uri, 0);
	gnome_vfs_uri_unref (new_uri);

	new_dir = dir_pool_get (new_uri_str);
	g_free (new_uri_str);
	
	return new_dir;
}


static void show_error_dialog (const gchar *msg)
{
	GtkWidget *dlg;
		
	dlg = gnome_error_dialog (msg);
	
	gtk_window_set_transient_for (GTK_WINDOW (dlg), GTK_WINDOW (main_win));
	gtk_window_set_position (GTK_WINDOW (dlg), GTK_WIN_POS_CENTER);
	gtk_window_set_modal (GTK_WINDOW (dlg), TRUE);
	gtk_widget_show (dlg);
}


void gnome_cmd_dir_ref (GnomeCmdDir *dir)
{
	g_return_if_fail (GNOME_CMD_IS_DIR (dir));

	dir->ref_cnt++;
	DEBUG ('d', "dir refing: 0x%x %s to %d\n", (guint)dir, dir->path, dir->ref_cnt);
}


void gnome_cmd_dir_unref (GnomeCmdDir *dir)
{
	g_return_if_fail (GNOME_CMD_IS_DIR (dir));
	
	dir->ref_cnt--;
	DEBUG ('d', "dir un-refing: 0x%x %s to %d\n", (guint)dir, dir->path, dir->ref_cnt);
	
	if (dir->ref_cnt < 1)
		gtk_object_unref (GTK_OBJECT (dir));
}


/******************************************************************************
*
*   Function: gnome_cmd_dir_list_files
*
*   Purpose: Return a list of all files in this directory. When calling this function
*            the first time the files will be fetched but repeated calls will just return
*            the same list. See the gnome_cmd_dir_relist function if you want to force an
*            update.
*   Params: 
*
*   Returns: A GList containing the files in this directory. The list is the same as the
*            insternal one in the directory so be sure to make a copy if you intend to save
*            it.
*
*   Statuses: 
*
******************************************************************************/

GList *gnome_cmd_dir_list_files (GnomeCmdDir *dir, gboolean visprog)
{
	g_return_val_if_fail (GNOME_CMD_IS_DIR (dir), NULL);

	if (!dir->files || dir->content_changed || dir->not_listed)
	{
		DEBUG ('l', "going to relist in list due to content change or no previous files where present\n");
		return gnome_cmd_dir_relist_files (dir, visprog);
	}
	else
	{
		DEBUG ('l', "list\n");
	}

	return dir->files;
}


static void
on_cancel_list (GtkWidget *btn, GnomeCmdDir *dir)
{
	g_return_if_fail (GNOME_CMD_IS_DIR (dir));

	DEBUG ('l', "Directory listing was cancelled\n");
	gnome_vfs_async_cancel (dir->priv->list_handle);
	dir->priv->done = TRUE;
}


static void
create_progwin (GnomeCmdDir *dir, gboolean visprog)
{
	GtkWidget *win = NULL;
	GtkWidget *frame = NULL;
	GtkWidget *lbl = NULL;
	GtkWidget *btn = NULL;
	GtkWidget *box = NULL;
	GtkWidget *bbox = NULL;

	if (visprog && main_win && GTK_WIDGET (main_win)->window) {
		gint x, y, w, h;
		
		win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
		gtk_widget_ref (win);
		gtk_widget_set_usize (win, 200, 60);
		gtk_window_set_title (GTK_WINDOW (win), _("Loading directory..."));
		gtk_object_set_data_full (GTK_OBJECT (win),
								  "win", win,
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_window_set_transient_for (GTK_WINDOW (win), GTK_WINDOW (main_win));
		gtk_window_set_modal (GTK_WINDOW (win), TRUE);

		frame = gtk_frame_new (NULL);
		gtk_widget_ref (win);
		gtk_container_add (GTK_CONTAINER (win), frame);
		gtk_object_set_data_full (GTK_OBJECT (win),
								  "frame", frame,
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
		gtk_widget_show (frame);
		

		box = gtk_vbox_new (FALSE, 2);
		gtk_widget_ref (box);
		gtk_container_add (GTK_CONTAINER (frame), box);
		gtk_object_set_data_full (GTK_OBJECT (win),
								  "box", box,
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_widget_show (box);

		lbl = gtk_label_new (_("Preparing..."));
		gtk_widget_ref (lbl);
		gtk_box_pack_start (GTK_BOX (box), lbl, TRUE, TRUE, 0);
		gtk_object_set_data_full (GTK_OBJECT (win),
								  "lbl", lbl,
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_widget_show (lbl);

		bbox = gtk_hbutton_box_new ();
		gtk_widget_ref (bbox);
		gtk_box_pack_start (GTK_BOX (box), bbox, FALSE, TRUE, 0);
		gtk_object_set_data_full (GTK_OBJECT (win),
								  "bbox", bbox,
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_widget_show (bbox);

		btn = gnome_stock_button (GNOME_STOCK_BUTTON_CANCEL);
		gtk_widget_ref (btn);
		gtk_box_pack_start (GTK_BOX (bbox), btn, FALSE, TRUE, 0);
		gtk_object_set_data_full (GTK_OBJECT (win),
								  "btn", btn,
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_widget_show (btn);

		gtk_signal_connect (GTK_OBJECT (btn), "clicked",
							GTK_SIGNAL_FUNC (on_cancel_list), dir);

		gdk_window_get_position (GTK_WIDGET (main_win)->window, &x, &y);
		gdk_window_get_size (GTK_WIDGET (main_win)->window, &w, &h);

		gtk_widget_set_uposition (win, x+w/2-100, y+h/2-30);
		gtk_window_set_transient_for (GTK_WINDOW (win), GTK_WINDOW (main_win));
		gtk_window_set_modal (GTK_WINDOW (win), TRUE);
	}
	
	dir->priv->progwin = win;
	dir->priv->progtext = lbl;
	dir->priv->cancel_btn = btn;
}


static void
on_files_listed (GnomeVFSAsyncHandle *handle,
				 GnomeVFSResult result,
				 GList *list,
				 guint entries_read,
				 GnomeCmdDir *dir)
{
	g_return_if_fail (GNOME_CMD_IS_DIR (dir));

	if (result != GNOME_VFS_OK && result != GNOME_VFS_ERROR_EOF) {
		warn_print ("Directory listing failed, %s\n", gnome_vfs_result_to_string (result));	
		dir->prev_error = TRUE;
		dir->priv->done = TRUE;
		show_error_dialog (gnome_vfs_result_to_string (result));
		return;
	}	
	
	if (entries_read > 0 && list != NULL) {

		g_list_foreach (list, (GFunc)gnome_vfs_file_info_ref, NULL);
		dir->priv->infolist = g_list_concat (dir->priv->infolist, g_list_copy (list));
		dir->priv->count += entries_read;

		if (dir->priv->progtext) {
			gchar *text = g_strdup_printf (ngettext("Loaded %d file...\n","Loaded %d files...\n",dir->priv->count), dir->priv->count);
			gtk_label_set_text (GTK_LABEL (dir->priv->progtext), text);
			g_free (text);
		}
	}
	
	if (result == GNOME_VFS_ERROR_EOF) {
		dir->priv->done = TRUE;
		return;
	}
}



static GList *
visprog_list (GnomeCmdDir *dir)
{
	GnomeVFSFileInfoOptions infoOpts;

	infoOpts = GNOME_VFS_FILE_INFO_FOLLOW_LINKS|GNOME_VFS_FILE_INFO_GET_MIME_TYPE;

	dir->priv->done = FALSE;
	dir->priv->count = 0;
	dir->priv->infolist = NULL;

	gnome_vfs_async_load_directory_uri (
		&dir->priv->list_handle,
		gnome_cmd_dir_get_uri (dir),
		infoOpts,
		GNOME_VFS_DIRECTORY_FILTER_NONE,
        GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR,
		NULL,
		DIR_FILES_PER_NOTIFICATION,
		(GnomeVFSAsyncDirectoryLoadCallback)on_files_listed,
		dir);

	while (!dir->priv->done) {
		if (gtk_events_pending ())
			gtk_main_iteration_do (FALSE);
	};

	return dir->priv->infolist;
}


static GList *
blocking_list (GnomeCmdDir *dir)
{
	GnomeVFSResult result;
	GnomeVFSFileInfoOptions infoOpts;
	GnomeVFSDirectoryFilter *filter;

	infoOpts = GNOME_VFS_FILE_INFO_FOLLOW_LINKS|GNOME_VFS_FILE_INFO_GET_MIME_TYPE;

	dir->priv->done = FALSE;
	dir->priv->count = 0;
	dir->priv->infolist = NULL;

	filter = gnome_vfs_directory_filter_new (
		GNOME_VFS_DIRECTORY_FILTER_NONE,
		GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR,
		NULL);
	
	result = gnome_vfs_directory_list_load (
		&dir->priv->infolist,
		gnome_cmd_dir_get_uri_str (dir),
		infoOpts,
		filter);

	gnome_vfs_directory_filter_destroy (filter);

	if (result == GNOME_VFS_OK)
		return dir->priv->infolist;

	create_error_dialog (gnome_vfs_result_to_string (result));
	
	return NULL;
}


/******************************************************************************
*
*   Function: gnome_cmd_dir_relist_files
*
*   Purpose: Returns a fresh list of all files in this directory every time its called
*
*   Params: 
*
*   Returns: A GList containing the files in this directory. The list is the same as the
*            insternal one in the directory so be sure to make a copy if you intend to save
*            it.
*
*   Statuses: 
*
******************************************************************************/

GList *gnome_cmd_dir_relist_files (GnomeCmdDir *dir, gboolean visprog)
{
	GList *list;
	GList *tmp;
	
	if (dir->priv->lock) return NULL;
	dir->priv->lock = TRUE;
	
	g_return_val_if_fail (GNOME_CMD_IS_DIR (dir), NULL);
	g_return_val_if_fail (dir->uri != NULL, NULL);

	if (dir->prev_error) {
		DEBUG ('l', "Skipping relist due to previous error\n");
		dir->priv->lock = FALSE;
		return NULL;
	}
	
	DEBUG ('l', "relist\n");
	if (dir->files != NULL) {
		gnome_cmd_file_list_free (dir->files);
		dir->files = NULL;
	}

	if (visprog) {
		create_progwin (dir, visprog);
		if (dir->priv->progwin)
			gtk_widget_show (dir->priv->progwin);
		
		list = visprog_list (dir);
	}
	else {
		list = blocking_list (dir);
	}

	if (list) {
		if (visprog && dir->priv->progtext)
			gtk_label_set_text (GTK_LABEL (dir->priv->progtext), _("Processing files..."));

		/* create a new list with GnomeCmdFile objects */
		tmp = list;
		while (tmp) {
			GnomeCmdFile *finfo;
			GnomeVFSFileInfo *info = (GnomeVFSFileInfo*)tmp->data;
			if (info && info->name) {
				finfo = gnome_cmd_file_new_with_values (info, dir);
				gnome_cmd_file_ref (finfo);
				dir->files = g_list_append (dir->files, finfo);
			}
			tmp = tmp->next;
		}

		gnome_vfs_file_info_list_free (list);
	}

	if (visprog && dir->priv->progwin)
		gtk_widget_destroy (dir->priv->progwin);
	
	dir->content_changed = FALSE;
	
	/* only send the list-updated signal if this is not the first relist */
	if (dir->not_listed)
		dir->not_listed = FALSE;
	else
		gtk_signal_emit (GTK_OBJECT (dir), dir_signals[LIST_UPDATED]);

	dir->priv->lock = FALSE;
	return dir->files;
}


const gchar *gnome_cmd_dir_get_path (GnomeCmdDir *dir)
{
	g_return_val_if_fail (GNOME_CMD_IS_DIR (dir), NULL);
	g_return_val_if_fail (dir->uri != NULL, NULL);
	
	return dir->path;
}


GnomeVFSURI *gnome_cmd_dir_get_uri (GnomeCmdDir *dir)
{
	g_return_val_if_fail (GNOME_CMD_IS_DIR (dir), NULL);
	g_return_val_if_fail (dir->uri != NULL, NULL);
	
	return dir->uri;
}


const gchar *gnome_cmd_dir_get_uri_str (GnomeCmdDir *dir)
{
	g_return_val_if_fail (GNOME_CMD_IS_DIR (dir), NULL);

	return dir->uri_str;
}


GnomeVFSURI *gnome_cmd_dir_get_file_uri (GnomeCmdDir *dir, const gchar *filename)
{
	g_return_val_if_fail (GNOME_CMD_IS_DIR (dir), NULL);
	g_return_val_if_fail (dir->uri != NULL, NULL);

	return gnome_vfs_uri_append_file_name (dir->uri, filename);
}


static gboolean
file_already_exists (GnomeCmdDir *dir, const gchar *filename)
{
	GList *tmp;

	g_return_val_if_fail (GNOME_CMD_IS_DIR (dir), TRUE);
	g_return_val_if_fail (filename != NULL, TRUE);

	tmp = dir->files;
	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		if (strcmp (gnome_cmd_file_get_name (finfo), filename) == 0)
			return TRUE;
		tmp = tmp->next;
	}

	return FALSE;
}


/* A file has been created. Create a new GnomeCmdFile object for that file
 */
void gnome_cmd_dir_file_created (GnomeCmdDir *dir, const gchar *filename)
{
	GnomeVFSURI *uri;
	GnomeVFSResult res;
	GnomeVFSFileInfo *info;
	GnomeCmdFile *finfo;
	GnomeVFSFileInfoOptions infoOpts =
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS|GNOME_VFS_FILE_INFO_GET_MIME_TYPE;

	g_return_if_fail (GNOME_CMD_IS_DIR (dir));
	g_return_if_fail (filename != NULL);

	if (file_already_exists (dir, filename))
		return;
	
	uri = gnome_cmd_dir_get_file_uri (dir, filename);	
	info = gnome_vfs_file_info_new ();	
	res = gnome_vfs_get_file_info_uri (
		uri, info, infoOpts);
	gnome_vfs_uri_unref (uri);
	
	finfo = gnome_cmd_file_new_with_values (info, dir);
	dir->files = g_list_append (dir->files, finfo);
	gnome_cmd_file_ref (finfo);

	gtk_signal_emit (GTK_OBJECT (dir), dir_signals[FILE_CREATED], finfo);
}


/* A file has been deleted. Remove the corresponding GnomeCmdFile
 */
void gnome_cmd_dir_file_deleted (GnomeCmdDir *dir, const gchar *filename)
{
	GList *tmp = dir->files;

	g_return_if_fail (GNOME_CMD_IS_DIR (dir));
	g_return_if_fail (filename != NULL);
	
	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		if (strcmp (finfo->info->name, filename) == 0) {
			dir->files = g_list_remove (dir->files, finfo);
			gtk_signal_emit (GTK_OBJECT (dir), dir_signals[FILE_DELETED], finfo);
			gnome_cmd_file_unref (finfo);
			return;
		}
		tmp = tmp->next;
	}
}


/* A file has been changed. Find the corresponding GnomeCmdFile, update its
 * GnomeVFSFileInfo
 */
void gnome_cmd_dir_file_changed (GnomeCmdDir *dir, const gchar *filename)
{
	GList *tmp;

	g_return_if_fail (GNOME_CMD_IS_DIR (dir));
	g_return_if_fail (filename != NULL);

	tmp = dir->files;
	
	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		if (strcmp (finfo->info->name, filename) == 0) {
			GnomeVFSResult res;
			GnomeVFSURI *uri;
			GnomeVFSFileInfo *info;
			GnomeVFSFileInfoOptions infoOpts = GNOME_VFS_FILE_INFO_GET_MIME_TYPE;
	
			uri = gnome_cmd_dir_get_file_uri (dir, filename);
			info = gnome_vfs_file_info_new ();
			res = gnome_vfs_get_file_info_uri (
				uri, info, infoOpts);
			gnome_vfs_uri_unref (uri);
			
			gnome_cmd_file_update_info (finfo, info);
			gtk_signal_emit (GTK_OBJECT (dir), dir_signals[FILE_CHANGED], finfo);
			return;
		}
		tmp = tmp->next;
	}
}


void gnome_cmd_dir_file_renamed (GnomeCmdDir *dir, GnomeCmdFile *finfo)
{
	g_return_if_fail (GNOME_CMD_IS_DIR (dir));
	g_return_if_fail (GNOME_CMD_IS_FILE (finfo));

	gtk_signal_emit (GTK_OBJECT (dir), dir_signals[FILE_CHANGED], finfo);
}


gboolean
gnome_cmd_dir_uses_fam (GnomeCmdDir *dir)
{
#ifdef HAVE_LIBFAM
	return (dir->request != NULL);
#endif
	return FALSE;
}
