/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
#include "config.h"

#include "snmpkit/snmpkit"

#include <stdio.h>

#include <string.h>
#include <iostream>
#include <string>

static const std::string HPSTR("JETDIRECT");
static const std::string LEXMARKSTR("Lexmark");
static const std::string TEKTRONIXSTR("Tektronix");
static const std::string XEROXDC230STR("131;C1H011131;");
static const std::string XEROXDC230STR2(";C1H017730;");
static const std::string XEROXDC265STR("3UP060485");
static const std::string XEROXSTR("Xerox");
static const std::string QMSSTR("QMS");
static const std::string IBMSTR("IBM");
static const std::string EFISTR("EFI Fiery Color Printer Server");
static const std::string EFISTR2("EFI Fiery Server ZX");
static const std::string FUJISTR("Able Model-PRII");

static const std::string SYSDESC("1.3.6.1.2.1.1.1.0");
static const std::string HPINFO("1.3.6.1.4.1.11.2.3.9.1.1.7.0"); // undocumented
static const std::string HRDEVICEDESC("1.3.6.1.2.1.25.3.2.1.3.1");
static const std::string TKMODEL("1.3.6.1.4.1.128.2.1.3.1.2.0");

class PrinterException { };

struct PrinterInfo {
	std::string vendor;
	std::string model;

	std::string sysDesc;
	char *info;
};

static char *printer_info = NULL;

static void
printer_info_insert_sysDesc (void *dest_ptr,const char *str)
{
	PrinterInfo *dest = reinterpret_cast<PrinterInfo*>(dest_ptr);
	dest->sysDesc=str;
}

static void
printer_info_insert_info (void *dest_ptr, const char *str)
{
	PrinterInfo *dest = reinterpret_cast<PrinterInfo*>(dest_ptr);
	dest->info = strdup(str);
}

static void *
do_req (SNMP_session *printer)
{
	PrinterInfo prinfo;
	int len;

	try{
		SNMP_structFiller table(*printer);
		table.append_string (SYSDESC, printer_info_insert_sysDesc);
		table.get(&prinfo);
		table.remove(SYSDESC);/* ITS4: ignore */

		/* ------- HP ------- */
		if (prinfo.sysDesc.find (HPSTR) != std::string::npos) {
			prinfo.vendor="HP";
			table.append_string (HPINFO,printer_info_insert_info);
			table.get (&prinfo);
			char *model_begin=strstr (prinfo.info,"MODEL:");
			if (model_begin!=NULL) {
				model_begin+=6;
				if (!strncmp (model_begin,"HP ",3))
					model_begin+=3;
			} else if ((model_begin=strstr (prinfo.info,"MDL:"))!=NULL)
				model_begin+=4;
			else
				throw PrinterException ();
			char *end=strchr (model_begin,';');
			if (end==NULL)
				end=model_begin+strlen (model_begin);
			prinfo.model=std::string (model_begin,end-model_begin);

		/* ------- Lexmark ------- */
		} else if (! prinfo.sysDesc.compare (LEXMARKSTR)) {
			prinfo.vendor="Lexmark";
			prinfo.model=prinfo.sysDesc.substr (8,prinfo.sysDesc.find ("  "));

		/* ------- Tektronix ------- */
		} else if (! prinfo.sysDesc.compare (TEKTRONIXSTR)) {
			prinfo.vendor="Tektronix";
			table.append_string (HRDEVICEDESC,printer_info_insert_info);
			try {
				table.get (&prinfo);
			} catch (SNMPBadOidException e) {
				table.remove (HRDEVICEDESC);/* ITS4: ignore */
				table.append_string (TKMODEL,printer_info_insert_info);
				table.get (&prinfo);
			}
			char *begin=strstr (prinfo.info,"Inc., ");
			if (begin) {
				begin+=6;
				prinfo.model=std::string (begin,strchr (begin,',')-begin);
			} else
				throw PrinterException ();
		/* ------- Xerox Desktop ------- */
		} else if (!prinfo.sysDesc.compare (XEROXSTR)) {
			prinfo.vendor="Xerox";
			if (prinfo.sysDesc.find ("???")!=std::string::npos)
				prinfo.model="???";
			else
				prinfo.model=prinfo.sysDesc.substr (prinfo.sysDesc.find (' ')+1);      
		} else if (!prinfo.sysDesc.compare (XEROXDC230STR) ||
			   !prinfo.sysDesc.compare (XEROXDC230STR2)) {
			prinfo.vendor="Xerox";
			prinfo.model="Document Centre 230ST";
		} else if (!prinfo.sysDesc.compare (XEROXDC265STR)) {
			prinfo.vendor="Xerox";
			prinfo.model="Document Centre 265";

		} else if (!prinfo.sysDesc.compare (EFISTR) || 
			   !prinfo.sysDesc.compare (EFISTR2)) {
			prinfo.vendor="EFI";
			prinfo.model="???";

		} else if (!prinfo.sysDesc.compare(QMSSTR)) {
			prinfo.vendor="QMS";
			prinfo.model=prinfo.sysDesc.substr (4);

		} else if (!prinfo.sysDesc.compare (IBMSTR)) {
			prinfo.vendor="IBM";
			prinfo.model=prinfo.sysDesc.substr (4);

		} else if (!prinfo.sysDesc.compare (FUJISTR)) {
			prinfo.vendor="Fuji";
			prinfo.model="Able PRII";

		} else
			return NULL;


		//=========================================================
		// Append new data to global std::string.
		//=========================================================
		if (printer_info == NULL)
			printer_info = strdup ("");

		std::string tempstrng ("printer=");
		tempstrng += printer->Hostname ();
		tempstrng += ";vendor=";
		tempstrng += prinfo.vendor;
		tempstrng += ";model=";
		tempstrng += prinfo.model;
		tempstrng += "\n";

		len = strlen (tempstrng.c_str());
		printer_info = (char*)realloc (printer_info,
					       strlen (printer_info) + len + 1);
		strcat (printer_info, tempstrng.c_str());

		//=========================================================

		return NULL;
	} catch (SNMPNoResponseException e) { // this is not a problem
		return NULL;
	} catch (ProgrammerException e) {
		std::cerr << "inside programmer error\n";
	} catch (DecodeException e) {
		std::cerr << "inside decode error\n";
	} catch (SNMPException e) {
		std::cerr << "inside SNMPException\n";
	} catch (...) {
		std::cerr << "inside other error\n";
	} 

	int *i=new int;
	*i=1;
	return i;
}

/*----------------------------------------------------------------
  This function lets our C library access the C++ snmpkit library.
Note: 'extern "C"' is required to get around C++ name mangling.
----------------------------------------------------------------*/
extern "C" char *
get_snmp_printers (char *specstr, int *retval)
{
	set_snmpsock_props(5,2);
	std::list<SNMP_session*> sessions;
	try {
		std::string hostspec(specstr);
		SNMP_sessions(sessions,hostspec,do_req);
		SNMP_sessions_done();
	} catch (ProgrammerException e) {
		*retval = 1;
		std::cerr << "programmer error\n";
	} catch (DecodeException e) {
		*retval = 1;
		std::cerr << "decode error\n";
	} catch (SessionHostNotFoundException e) {
		*retval = 1;
		std::cerr << "cannot resolve host " << specstr << std::endl;
		return NULL;
	} catch (...) {
		*retval = 1;
		std::cerr << "other error\n";
		return NULL;
	}

	return (char *) printer_info;
}  
