/*
 * isnmpsock.C: Implementation of the underlying socket
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later 
 * version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * See the AUTHORS file for a list of people who have hacked on 
 * this code. 
 * See the ChangeLog file for a list of changes.
 *
 */

#include "snmpsock.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <ctype.h>
#include <errno.h>

#include <map>
#include <string>

#define MAXPACKSIZE 10240

pthread_mutex_t pending_m=PTHREAD_MUTEX_INITIALIZER;

class request_t{
  pthread_cond_t req_cv;
  //filled in on the return
  unsigned char *retbuf;
  int retlen; /* This is an int vs. an unsigned int because send(2)
		 returns an int vs. an unsigned int. */
  
  int errnum;

public:
  inline request_t():retbuf(NULL),errnum(-1){
    pthread_cond_init(&req_cv,NULL);
  }
  inline ~request_t(){
    if(retbuf)
      delete retbuf;
  }

  inline void wakeup(int err){
#ifdef DEBUG
    cerr << "wakeup error\n" << flush;
#endif
    
    errnum=err;
    retlen=0;
    retbuf=NULL;
    pthread_cond_signal(&req_cv);
  }
  void wakeup(unsigned char *buf,unsigned int len){
    errnum=0;
    retbuf=buf;
    retlen=len;
    pthread_cond_signal(&req_cv);
  }
  // sleep assumes that the pending mutex is locked.
  inline int sleep(timespec &tv){ 
    return pthread_cond_timedwait(&req_cv,&pending_m,&tv);
  }
  inline int error(){return errnum;}
  inline unsigned char *claimbuf(int &buflen){ 
    unsigned char *retval=retbuf;
    retbuf=NULL;
    buflen=retlen;
    return retval;
  }
};
/* The string is a addr in disquise 
   This is to only be manipulated in conjunction with pending_m
*/
std::map<std::string,request_t*> pending;

inline unsigned int to_ui(char i){
  return static_cast<unsigned int>(static_cast<unsigned char>(i));
}

void *receiver(void *sockp)
  throw(std::bad_alloc,ReceiverSelectException,ReceiverReadException){
  int sock=*(int*)sockp;
  int readcnt;
  unsigned int fromlen;
  for(;;){
    sockaddr_in from;
    memset(&from,0,sizeof(from));
    fromlen=sizeof(sockaddr_in);
    unsigned char *buf=new unsigned char[MAXPACKSIZE];
    fd_set rfds;
    timeval tv;
    int retval;
    FD_ZERO(&rfds);
    FD_SET(sock,&rfds);
    tv.tv_sec=5;
    tv.tv_usec=0;
    if((retval=select(sock+1,&rfds,NULL,NULL,&tv))==-1)
      throw ReceiverSelectException(errno);
    pthread_testcancel();

    // check every 5 seconds to see if we were cancelled
    if(retval!=1 || !FD_ISSET(sock,&rfds))
      continue;

    // actually got some data
    readcnt=recvfrom(sock,buf,MAXPACKSIZE,0,(sockaddr*)&from,&fromlen);
    if(readcnt==-1)
      if(errno==ECONNREFUSED)
	continue; // just ignore those icmp unreachable errors
      else
	throw ReceiverReadException(errno);
    
    pthread_mutex_lock(&pending_m);

    std::string tmpstr(reinterpret_cast<char*>(&from.sin_addr),4);
    // begin DEBUG 
    /* char *addr2=reinterpret_cast<char*>(&from.sin_addr);
    fprintf(stderr,"packet recieved from %u.%u.%u.%u\n",
	    to_ui(addr2[0])&0xff,to_ui(addr2[1])&0xff,to_ui(addr2[2])&0xff,
	    to_ui(addr2[3])&0xff); */
    // end DEBUG
    std::map<std::string,request_t*>::iterator cur=pending.find(tmpstr);

    if(cur==pending.end()){
      char *addr=reinterpret_cast<char*>(&from.sin_addr);
      /* ITS4: ignore */
      fprintf(stderr,"Warning: stray packet recieved from %u.%u.%u.%u\n",
	      to_ui(addr[0])&0xff,to_ui(addr[1])&0xff,to_ui(addr[2])&0xff,
	      to_ui(addr[3])&0xff);
      pthread_mutex_unlock(&pending_m);
      continue;
    }

    if(readcnt==-1){
      cur->second->wakeup(errno);
    }else{
      cur->second->wakeup(buf,readcnt);
    }
    pending.erase(cur);
      
    pthread_mutex_unlock(&pending_m);
  }
}

SNMP_socket::SNMP_socket(int tmo, int rt, int pt)
  throw(SocketNoUDPException,SocketCreateFailException,
	ReceiverCreateException):
  timeout(tmo),retries(rt),port(pt){
  // create socket
  struct protoent *pe;
  struct servent *se;
  
  if((pe=getprotobyname("udp"))==NULL) throw SocketNoUDPException();
  if((sock=socket(AF_INET,SOCK_DGRAM,pe->p_proto))==-1) 
    throw SocketCreateFailException(errno);

  if(port==0){
    se = getservbyname("snmp","udp");
    if (NULL != se){
      port=se->s_port;
      endservent();
    } else
      port = 161; /* Fall back to hardcoded value */
  }  else 
    port=htons(port);

  // create listening thread
  int i=pthread_create(&listening_thr,NULL,receiver,&sock);
  if(i!=0)
    throw ReceiverCreateException(i);
}

SNMP_socket::~SNMP_socket(){
  // zap listening thread 
  pthread_cancel(listening_thr);
  pthread_join(listening_thr,NULL);
  // close socket
  close(sock);
}

unsigned char *SNMP_socket::call(char *addr,int addrlen,int addrtype,
				 const unsigned char *data,int &buflen)
  throw(SocketSendShortExecption){
  request_t curreq;
#ifdef DEBUG
  cerr << "call curreq-" << &curreq << endl << flush;
#endif
  std::string addrstr(addr,4);

  // construct address
  struct sockaddr_in sain;
  memset((caddr_t)&sain,0,sizeof(sain));
  sain.sin_family=addrtype,
  sain.sin_port=port;
  memcpy((caddr_t)&sain.sin_addr,addr,addrlen);
  // begin DEBUG 
  /*  char *addr2=reinterpret_cast<char*>(&sain.sin_addr);
  fprintf(stderr,"packet being sent to %u.%u.%u.%u\n",
	  to_ui(addr2[0])&0xff,to_ui(addr2[1])&0xff,to_ui(addr2[2])&0xff,
	  to_ui(addr2[3])&0xff); */
  // end DEBUG

  // insert it on the list
  pthread_mutex_lock(&pending_m);
  pending[addrstr]=&curreq;
  fflush(stderr);

  //while there are still retries
  int rt;
  for(rt=retries+1;rt!=0;rt--){
  retry:
    int sendcnt=sendto(sock,data,buflen,0,(sockaddr*)&sain,
		       sizeof(sockaddr_in));
    fflush(stderr);
    /* work around potential problem where ICMP port unreachable 
       propegates up IP stack and can appear on socket even 
       though problem is not with this particular packet. */
    if(sendcnt==-1){
      if(errno==ECONNREFUSED)
	goto retry;
      // remove from list and return error
      buflen=errno;
      pthread_mutex_lock(&pending_m);
      pending.erase(addr);
      pthread_mutex_unlock(&pending_m);
      return NULL;
    }
    if(sendcnt!=buflen) throw SocketSendShortExecption();
    // wait on condition variable
    timespec tv;
    tv.tv_sec=time(NULL)+timeout;
    tv.tv_nsec=0;
    
    if(curreq.sleep(tv)!=ETIMEDOUT)
      break;
  }
  if(rt==0)
    pending.erase(addr);
  pthread_mutex_unlock(&pending_m);

  // kick out bad data
  if(curreq.error()){
    buflen=curreq.error();
    return NULL;
  }
  
  return curreq.claimbuf(buflen);
}
