/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"

/* data types */
struct GSQLFE_DriverInfo
{
  gboolean is_ok;
  gchar label[32];
#ifndef HAVE_GDA
  GnomeSqlDriver driver;
#endif  
};

#ifndef HAVE_GDA
/* function prototypes */
static GnomeSqlDriver get_driver_from_string (gchar *);
/* global variables */
static struct GSQLFE_DriverInfo l_drivers[] =
  {
    { TRUE, "MySQL", GNOME_SQL_DRIVER_MYSQL },
    { TRUE, "ODBC", GNOME_SQL_DRIVER_ODBC },
    { TRUE, "PostgreSQL", GNOME_SQL_DRIVER_POSTGRES },
    { FALSE, "\0", GNOME_SQL_DRIVER_NONE }
  };
#else
static struct GSQLFE_DriverInfo l_drivers[] =
{
  { TRUE, "gda-mysql"},
  { TRUE, "gda-odbc"},
  { FALSE, ""},
#define GNOME_SQL_DRIVER_NONE 0
};
#endif

static gboolean l_connecting = FALSE;
#ifdef HAVE_GDA
static Gda_Connection* l_database = NULL;
#else
static GnomeDatabase *l_database = NULL;
#endif
static GtkWidget *w_dbname, *w_login, *w_pwd, *w_host,
  *w_port, *w_options, *w_combo;

/* static function */
static void 
dialog_clicked_cb(GnomeDialog *dialog, gint button_number, 
                  gpointer data)
{
  register gint cnt;
  extern CORBA_ORB orb;
  switch (button_number)
    {
    case 0 : /* OK button */
      {
        gint cur_tab = gtk_notebook_current_page(GTK_NOTEBOOK(data));
	switch (cur_tab)
	  {
	  case 0 : /* 'new' tab */
	    {
#ifndef HAVE_GDA
	      GnomeDatabase *db;
	      GnomeSqlDriver cur_driver;
#else
	      Gda_Connection* db;
	      gchar* cur_driver;
#endif	      
	      gchar *drv_str, *dbname, *login, *pwd, *host, *port, *options;
	      /* get fields' values */
	      dbname = gtk_entry_get_text(GTK_ENTRY(w_dbname));
	      login = gtk_entry_get_text(GTK_ENTRY(w_login));
	      pwd = gtk_entry_get_text(GTK_ENTRY(w_pwd));
	      host = gtk_entry_get_text(GTK_ENTRY(w_host));
	      port = gtk_entry_get_text(GTK_ENTRY(w_port));
	      options = gtk_entry_get_text(GTK_ENTRY(w_options));
	      drv_str = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(w_combo)->entry));
	      cur_driver = get_driver_from_string(drv_str);
	      if (cur_driver != GNOME_SQL_DRIVER_NONE)
		{
		  /* create driver connection */
#ifdef HAVE_GDA
		  gint rc;
		  db = gda_connection_new(orb);
		  db->provider = g_strdup(drv_str);
		  rc = gda_connection_open(db, dbname, login, pwd);
		  if (rc == 0)
		    {
		      l_connecting = FALSE;
		      db_new_connection(db);
		      gnome_dialog_close(dialog);
		    }
		  else
		    ui_show_error("Coule not open connection to database %s", dbname);
#else
		  db = gnome_sql_connect(cur_driver, dbname, login, pwd, host);
		  if (db != NULL)
		    {
		      /* close dialog */
		      l_connecting = FALSE;
		      db_new_connection(db);
		      gnome_dialog_close(dialog);
		    }
		  else ui_show_error("Could not open connection to database %s", dbname);
#endif
		  
		}
	    }
	  case 1 : /* recent connections tab */
	    break;
	  }
        break;
      }
    case 1 : /* Cancel button */
      /* close dialog */
      l_connecting = FALSE;
      gnome_dialog_close(dialog);
      break;
    }
}

static GtkWidget *
create_new_connection_tab (GtkWidget *notebook)
{
  GtkWidget *tab;
  register gint cnt;
#ifdef HAVE_GDA
  GList*   data_sources;
#endif  
  /* check parameters */
  g_return_val_if_fail(notebook != NULL, NULL);
  /* create the table */
  tab = gtk_table_new(5, 4, FALSE);
  gtk_widget_show(tab);
  /* create widgets in the following order: */
  /* driver's combo */
#ifdef HAVE_GDA
  ui_new_label_in_table(tab, _("Provider"), 0, 0, 1, 1);
#else
  ui_new_label_in_table(tab, "Driver:", 0, 0, 1, 1);
#endif
  w_combo = ui_new_combo_in_table(tab, 1, 0, 3, 1);
  for (cnt = 0; l_drivers[cnt].is_ok; cnt++)
    {
      GtkWidget *li = gtk_list_item_new_with_label(l_drivers[cnt].label);
      gtk_container_add(GTK_CONTAINER(GTK_COMBO(w_combo)->list), li);
      gtk_widget_show(li);
    }
  /* 1 - database name */
  ui_new_label_in_table(tab, "Database:", 0, 1, 1, 2);
  w_dbname = ui_new_entry_in_table(tab, "DBNAME", NULL, 1, 1, 2, 2);
  /* 2 - login */
  ui_new_label_in_table(tab, "User:", 0, 2, 1, 3);
  w_login = ui_new_entry_in_table(tab, "LOGIN", g_get_user_name(), 1, 2, 2, 3);
  /* 3 - password */
  ui_new_label_in_table(tab, "Password:", 2, 2, 3, 3);
  w_pwd = ui_new_entry_in_table(tab, "PASSWORD", NULL, 3, 2, 4, 3);
  gtk_entry_set_visibility(GTK_ENTRY(w_pwd), FALSE);
  /* 4 - host */
  ui_new_label_in_table(tab, "Host:", 0, 3, 1, 4);
  w_host = ui_new_entry_in_table(tab, "HOST", NULL, 1, 3, 2, 4);
  /* 5 - port */
  ui_new_label_in_table(tab, "Port:", 2, 3, 3, 4);
  w_port = ui_new_entry_in_table(tab, "PORT", NULL, 3, 3, 4, 4);
  /* 6 - options */
  ui_new_label_in_table(tab, "Options:", 0, 4, 1, 5);
  w_options = ui_new_entry_in_table(tab, "OPTIONS", NULL, 1, 4, 2, 5);
  return (tab);
}

static GtkWidget *
create_new_recent_tab (GtkWidget *notebook)
{
  GtkWidget *tab;
  /* check parameters */
  g_return_val_if_fail(notebook != NULL, NULL);
  /* create the table */
  tab = gtk_table_new(2, 4, FALSE);
  gtk_widget_show(tab);

  return (tab);
}

#ifndef HAVE_GDA
static GnomeSqlDriver 
get_driver_from_string (gchar *str)
{
  register gint cnt;
  for (cnt = 0; l_drivers[cnt].is_ok; cnt++)
    {
      if (!strcmp(l_drivers[cnt].label, str))
	return (l_drivers[cnt].driver);
    }
  return (GNOME_SQL_DRIVER_NONE);
}
#else
static gchar*
get_driver_from_string (gchar *str)
{
  register gint cnt;
  for (cnt = 0; l_drivers[cnt].is_ok; cnt++)
    {
      if (!strcmp(l_drivers[cnt].label, str))
	return str;
    }
  return (GNOME_SQL_DRIVER_NONE);
}
#endif
/* function definitions */
void 
db_open_connection (GtkWidget *w, gpointer data)
{
  register int cnt;
  GtkWidget *notebook, *driver_tab, *recent_tab, *label;
  static GtkWidget *dialog;
  /* if we are already connecting... */
  if (l_connecting)
    {
      gtk_widget_show(dialog);	/* FIXME: want to get focus!!! */
      gtk_widget_grab_focus(dialog);
      return;
    }
  l_connecting = TRUE;
  /* create dialog box */
  dialog = gnome_dialog_new("Open Connection", 
                            GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL,
                            NULL);
  gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
  /* create notebook */
  notebook = gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), notebook,
                     TRUE, TRUE, GNOME_PAD);
  gtk_signal_connect(GTK_OBJECT(dialog), "clicked",
                     GTK_SIGNAL_FUNC(dialog_clicked_cb), (gpointer) notebook);
  /* add database drivers connection screen */
  driver_tab = create_new_connection_tab(notebook);
  if (driver_tab != NULL)
    {
      /* create the tab's label */
      label = gtk_label_new("New Connection");
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), driver_tab,
			       label);
    }
  /* create recent connections tab */
  //recent_tab = create_new_recent_tab(notebook);
  //if (recent_tab != NULL)
  //  {
  //    /* create the tab's field */
  //    label = gtk_label_new("Recent");
  //    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), recent_tab, label);
  //  }
  gtk_widget_show(notebook);
  /* finally show the dialog */
  gtk_widget_show(dialog);
}
