/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"

/* global variables */
GList *l_last_commands = NULL;  /* not static, because settings.c needs it */

/* private functions */
static void
select_dialog_clicked_cb (GnomeDialog *dialog, gint button_number, 
                          gpointer data)
{
  GtkList *list = (GtkList *) data;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  switch (button_number)
    {
    case 0 :						/* 'Ok' button */
      if (info != NULL && list->selection != NULL)
        {
          GList *node = g_list_first(list->selection);
          if (node != NULL && node->data != NULL)
            {
              gchar *command;
              GtkObject *list_item;
              /* get command string */
              list_item = GTK_OBJECT(node->data);
              command = gtk_object_get_data(list_item, glb_list_item_data_key);
              if (command != NULL)
                {
                  /* set buffer text */
                  gtk_editable_delete_text(GTK_EDITABLE(info->cmd_win), 0,
                               gtk_text_get_length(GTK_TEXT(info->cmd_win)));
                  gtk_text_freeze(GTK_TEXT(info->cmd_win));
                  gtk_text_insert(GTK_TEXT(info->cmd_win), NULL, NULL, NULL,
                                  command, strlen(command));
                  gtk_text_thaw(GTK_TEXT(info->cmd_win));
                }
            }
        }
    case 1 :						/* 'Cancel' button */
      gnome_dialog_close(dialog);
      break;
    }
}

/* clipboard stuff */
void
sql_clear_window (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      /* empty SQL command window */
      gtk_editable_delete_text(GTK_EDITABLE(info->cmd_win), 0,
                               gtk_text_get_length(GTK_TEXT(info->cmd_win)));
      /* empty grid */
      info->grid = grid_new_empty(info->grid, GTK_CONTAINER(info->scrolled_window));
    }
}

void
sql_copy_text (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    gtk_editable_copy_clipboard(GTK_EDITABLE(info->cmd_win));
}

void
sql_cut_text (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    gtk_editable_cut_clipboard(GTK_EDITABLE(info->cmd_win));
}

void
sql_paste_text (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    gtk_editable_paste_clipboard(GTK_EDITABLE(info->cmd_win));
}

/* load a file into the SQL window */
void
sql_load_script (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  gchar *filename = ui_get_file_name("Select SQL Script");
  if (filename != NULL)
    {
      if (info != NULL)
	ui_load_file_in_text(GTK_TEXT(info->cmd_win), filename);
      else editor_open_window(NULL, filename);
      /* free memory */
      g_free((gpointer) filename);
    }
}

/* return last executed command */
gchar *
sql_get_last_command (void)
{
  GList *node = g_list_last(l_last_commands);
  return (node != NULL ? (gchar *) node->data : NULL);
}

/* requery grid */
void
sql_refresh_grid (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      gchar *command = sql_get_last_command();
      if (command != NULL)
        {
          Gda_Command *cmd = gda_command_new();
          Gda_Recordset *rowset;
          gulong reccount;
          gda_command_set_connection(cmd, info->db);
	  gda_command_set_text(cmd, command);
          rowset = gda_command_execute(command, &reccount, 0);
          if (rowset != NULL)
            {
              info->grid = grid_show_resultset(info->grid, rowset,
					       GTK_CONTAINER(info->scrolled_window));
              gda_recordset_close(rowset);
	      gda_recordset_free(rowset);
            }
          else ui_show_error("Error executing command");
	  gda_command_free(cmd);
        }
    }
}

/* run SQL buffer as a script */
void
sql_run_script (GtkWidget *w, gpointer data)
{
  gchar *script;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      /* get text from command window */
      script = gtk_editable_get_chars(GTK_EDITABLE(info->cmd_win), 0, 
                           gtk_text_get_length(GTK_TEXT(info->cmd_win)));
      if (script != NULL)
        {
          gchar *token = strtok(script, ";");
          while (token != NULL)
            {
              gulong reccount;
              Gda_Command *cmd = gda_command_new();
              gda_command_set_connection(cmd, info->db);
              gda_command_set_text(cmd, token);
              /* execute command */
              if (!gda_command_execute(info->db, &reccount, token))
                ui_show_error("Error executing command");
              else sql_store_command(token);
              /* get next token */
              token = strtok(NULL, ";");
            }
        }
    }
}

/* save SQL window to file */
void
sql_save_script (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      gchar *filename;
      /* prompt user for file name */
      filename = ui_get_file_name("Save SQL Buffer");
      if (filename != NULL)
        {
          ui_save_text_to_file(GTK_TEXT(info->cmd_win), filename);
          g_free((gpointer) filename);
        }
    }
}

/* let the user choose one of the last executed commands */
void
sql_select_command (GtkWidget *w, gpointer data)
{
  GtkWidget *dialog, *scrolled_window, *list;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      /* create the dialog box */
      dialog = gnome_dialog_new("Select SQL Command",
                                GNOME_STOCK_BUTTON_OK,
                                GNOME_STOCK_BUTTON_CANCEL,
                                NULL);
      gtk_widget_set_usize(dialog, 400, 400);
      gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);              
      /* create scrolled window to put list inside */
      scrolled_window = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrolled_window,
                         TRUE, TRUE, GNOME_PAD);
      gtk_widget_show(scrolled_window);
      /* create table list */
      list = gtk_list_new();
      gtk_signal_connect(GTK_OBJECT(dialog), "clicked",
                         GTK_SIGNAL_FUNC(select_dialog_clicked_cb), 
                         (gpointer) list); 
      if (l_last_commands != NULL)
        ui_fill_list(GTK_LIST(list), l_last_commands);
      gtk_list_set_selection_mode(GTK_LIST(list), GTK_SELECTION_SINGLE);
      gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(scrolled_window), list);
      gtk_widget_show(list);
      /* finally, show the dialog box */
      gtk_widget_show(dialog);
    }
}

/* store a command on linked list */
void
sql_store_command (const gchar *cmd)
{
  gchar *new_str;
  g_return_if_fail(cmd != NULL);
  if ((new_str = g_strdup(cmd)))
    {
      new_str = g_strstrip(new_str);
      l_last_commands = g_list_prepend(l_last_commands, (gpointer) new_str);
      if (g_list_length(l_last_commands) > MAX_COMMAND_LIST)
        {
          GList *node = g_list_first(l_last_commands);
          l_last_commands = g_list_remove(l_last_commands, node->data);
        }
    }
  else ui_show_error("Cannot allocate %ld bytes", strlen(cmd));
}
