/* GNOME DB library
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(HAVE_CONFIG_H)
#  include <config.h>
#endif
#include "gda-ui.h"

#define DESKTOP_SNAP_X         80
#define DESKTOP_SNAP_Y         80
#define ICON_LIST_SEPARATORS   " /-_."
#define ICON_LIST_ROW_SPACING  1
#define ICON_LIST_COL_SPACING  1
#define ICON_LIST_ICON_BORDER  2
#define ICON_LIST_TEXT_SPACING 2

static void gnome_db_icon_list_class_init (GnomeDbIconListClass *klass);
static void gnome_db_icon_list_init       (GnomeDbIconList *icon_list);
static void gnome_db_icon_list_destroy    (GnomeDbIconList *icon_list);

/*
 * GnomeDbIconList signals
 */
enum
{
  GNOME_DB_ICON_LIST_LOADED,
  GNOME_DB_ICON_LIST_ADD_ROW,
  GNOME_DB_ICON_LIST_SELECT_ROW,
  GNOME_DB_ICON_LIST_LAST_SIGNAL
};
static gint gnome_db_icon_list_signals[GNOME_DB_ICON_LIST_LAST_SIGNAL] = { 0 };

/*
 * Private functions
 */
static void
select_icon_cb (GnomeIconList *w, gint num, GdkEvent *event, GnomeDbIconList *icon_list)
{
  g_return_if_fail(GNOME_DB_IS_ICON_LIST(icon_list));
  gtk_signal_emit(GTK_OBJECT(icon_list),
                  gnome_db_icon_list_signals[GNOME_DB_ICON_LIST_SELECT_ROW]);
}

static gint
timeout_callback (GnomeDbIconList *icon_list)
{
  gint position;
  
  g_return_val_if_fail(GNOME_DB_IS_ICON_LIST(icon_list), 0);
  g_return_val_if_fail(GNOME_IS_ICON_LIST(icon_list->list), 0);
  g_return_val_if_fail(IS_GDA_RECORDSET(icon_list->recset), 0);
  
  position = gda_recordset_move(icon_list->recset, 1, 0);
  if (position != GDA_RECORDSET_INVALID_POSITION && !gda_recordset_eof(icon_list->recset))
    {
      Gda_Field* field;
      
      field = gda_recordset_field_idx(icon_list->recset, icon_list->pos);
      if (field)
	{
	  gchar *row = gda_stringify_value(0, 0, field);
	  gnome_icon_list_freeze(GNOME_ICON_LIST(icon_list->list));
	  gnome_icon_list_append(GNOME_ICON_LIST(icon_list->list),
	                         icon_list->pixmap_file,
	                         row);
	  gtk_signal_emit(GTK_OBJECT(icon_list),
	                  gnome_db_icon_list_signals[GNOME_DB_ICON_LIST_ADD_ROW]);
	  gnome_icon_list_thaw(GNOME_ICON_LIST(icon_list->list));
	  icon_list->total_rows++;
	  icon_list->values = g_list_append(icon_list->values, (gpointer) g_strdup(row));
	}
    }
  else
    {
      gtk_signal_emit(GTK_OBJECT(icon_list),
                      gnome_db_icon_list_signals[GNOME_DB_ICON_LIST_LOADED]);
      icon_list->timeout_handle = -1;
      return (0);
    }
  return (1);
}

/*
 * GnomeDbIconList widget interface
 */
static void
gnome_db_icon_list_class_init (GnomeDbIconListClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  gnome_db_icon_list_signals[GNOME_DB_ICON_LIST_LOADED] =
           gtk_signal_new("loaded", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbIconListClass, loaded),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);
  gnome_db_icon_list_signals[GNOME_DB_ICON_LIST_ADD_ROW] =
           gtk_signal_new("add_row", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbIconListClass, add_row),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);
  gnome_db_icon_list_signals[GNOME_DB_ICON_LIST_SELECT_ROW] =
           gtk_signal_new("select_row", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbIconListClass, select_row),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, gnome_db_icon_list_signals,
                               GNOME_DB_ICON_LIST_LAST_SIGNAL);
                               
  object_class->destroy = gnome_db_icon_list_destroy;
  klass->loaded = 0;
  klass->add_row = 0;
  klass->select_row = 0;
}

static void
gnome_db_icon_list_init (GnomeDbIconList *icon_list)
{
  GtkWidget* frame;
  
  frame = gda_ui_new_frame_widget(0);
  gtk_box_pack_start(GTK_BOX(icon_list), frame, 1, 1, 0);
  
  /* set scrolled window */
  icon_list->scrolled_window = gda_ui_new_scrolled_window_widget();
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(icon_list->scrolled_window),
                                 GTK_POLICY_AUTOMATIC,
                                 GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(frame), icon_list->scrolled_window);
  gtk_widget_show(icon_list->scrolled_window);

  /* create icon list */
  icon_list->list = gnome_icon_list_new(DESKTOP_SNAP_X, NULL, TRUE);
  gtk_signal_connect(GTK_OBJECT(icon_list->list),
                     "select_icon",
                     GTK_SIGNAL_FUNC(select_icon_cb),
                     (gpointer) icon_list);
  gtk_container_set_border_width(GTK_CONTAINER(icon_list->list), 3);
  gnome_icon_list_set_separators(GNOME_ICON_LIST(icon_list->list), ICON_LIST_SEPARATORS);
  gnome_icon_list_set_row_spacing(GNOME_ICON_LIST(icon_list->list), ICON_LIST_ROW_SPACING);
  gnome_icon_list_set_col_spacing(GNOME_ICON_LIST(icon_list->list), ICON_LIST_COL_SPACING);
  gnome_icon_list_set_icon_border(GNOME_ICON_LIST(icon_list->list), ICON_LIST_ICON_BORDER);
  gnome_icon_list_set_text_spacing(GNOME_ICON_LIST(icon_list->list), ICON_LIST_TEXT_SPACING);
  gnome_icon_list_set_selection_mode(GNOME_ICON_LIST(icon_list->list), GTK_SELECTION_MULTIPLE);
  GTK_WIDGET_SET_FLAGS(icon_list->list, GTK_CAN_FOCUS);
  
  gtk_container_add(GTK_CONTAINER(icon_list->scrolled_window), icon_list->list);
  gtk_widget_show(icon_list->list);
  
  /* init members */
  icon_list->recset = 0;
  icon_list->pos = -1;
  icon_list->total_rows = 0;
  icon_list->timeout_handle = -1;
  icon_list->pixmap_file = gnome_pixmap_file("gnome-folder.png");
  icon_list->values = 0;
}

guint
gnome_db_icon_list_get_type (void)
{
  static guint db_icon_list_type = 0;
  
  if (!db_icon_list_type)
    {
      GtkTypeInfo db_icon_list_info =
      {
        "GnomeDbIconList",
        sizeof (GnomeDbIconList),
        sizeof (GnomeDbIconListClass),
        (GtkClassInitFunc) gnome_db_icon_list_class_init,
        (GtkObjectInitFunc) gnome_db_icon_list_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_icon_list_type = gtk_type_unique(gtk_vbox_get_type(),
                                          &db_icon_list_info);
    }
  return (db_icon_list_type);
}

/**
 * gnome_db_icon_list_new
 * @recset: a @Gda_Recordset object
 * @pos: field position
 *
 * Creates a new GnomeDbIconList widget and associates the given recordset
 * object to it
 */
GtkWidget *
gnome_db_icon_list_new (Gda_Recordset *recset, gint pos)
{
  GtkWidget* icon_list = gtk_type_new(gnome_db_icon_list_get_type());
  
  if (recset != NULL)
    gnome_db_icon_list_set_recordset(GNOME_DB_ICON_LIST(icon_list), recset, pos);
  return (icon_list);
}

static void
gnome_db_icon_list_destroy (GnomeDbIconList *icon_list)
{
  g_return_if_fail(GNOME_DB_IS_ICON_LIST(icon_list));
  
  if (icon_list->timeout_handle != -1) gtk_timeout_remove(icon_list->timeout_handle);
  if (icon_list->recset) gda_recordset_free(icon_list->recset);
  if (icon_list->pixmap_file) g_free((gpointer) icon_list->pixmap_file);
  g_list_foreach(icon_list->values, (GFunc) g_free, NULL);
  g_list_free(icon_list->values);
}

/**
 * gnome_db_icon_list_clear
 * @icon_list: a @GnomeDbIconList widget
 *
 * Clears the @GnomeDbIconList widget
 */
void
gnome_db_icon_list_clear (GnomeDbIconList *icon_list)
{
  g_return_if_fail(GNOME_DB_IS_ICON_LIST(icon_list));
  
  if (icon_list->timeout_handle != -1) gtk_timeout_remove(icon_list->timeout_handle);
  if (icon_list->recset != 0) gda_recordset_free(icon_list->recset);
  
  icon_list->recset = 0;
  icon_list->pos = -1;  
  icon_list->total_rows = 0;
  icon_list->timeout_handle = -1;
  g_list_foreach(icon_list->values, (GFunc) g_free, NULL);
  g_list_free(icon_list->values);
  icon_list->values = 0;
  
  gnome_icon_list_clear(GNOME_ICON_LIST(icon_list->list));
}

/**
 * gnome_db_icon_list_get_string
 * @icon_list: a @GnomeDbIconList widget
 *
 * Returns currently selected string in the icon list widget
 */
const gchar *
gnome_db_icon_list_get_string (GnomeDbIconList *icon_list)
{
  GList* selected, *node;
  gint   row;
  gchar* str;
  
  g_return_val_if_fail(GNOME_DB_IS_ICON_LIST(icon_list), 0);
  g_return_val_if_fail(GNOME_IS_ICON_LIST(icon_list->list), 0);
  
  selected = GNOME_ICON_LIST(icon_list->list)->selection;
  if (selected)
    {
      row = GPOINTER_TO_INT(selected->data);
      node = g_list_nth(icon_list->values, row);
      if (node != 0) return ((const gchar *) node->data);
    }
  return (0);
}

/**
 * gnome_db_icon_list_get_recordset
 * @icon_list: a @GnomeDbIconList widget
 *
 * Returns the associated recordset for the given widget
 */
Gda_Recordset *
gnome_db_icon_list_get_recordset (GnomeDbIconList *icon_list)
{
  g_return_val_if_fail(GNOME_DB_IS_ICON_LIST(icon_list), 0);
  return (icon_list->recset);
}

/**
 * gnome_db_icon_list_set_recordset
 * @icon_list: a @GnomeDbIconList widget
 * @recset: recordset to be shown
 * @pos: field position
 *
 * Associates a @Gda_Recordset object with the given object, forcing an
 * immediate loading of the data from the underlying database. The data
 * is loaded asynchronously, so control returns to your program
 * immediately
 */
void
gnome_db_icon_list_set_recordset (GnomeDbIconList *icon_list, Gda_Recordset *recset, gint pos)
{
  g_return_if_fail(GNOME_DB_IS_ICON_LIST(icon_list));
  g_return_if_fail(IS_GDA_RECORDSET(recset));
  
  gnome_db_icon_list_clear(icon_list);
  
  icon_list->recset = recset;
  icon_list->pos = pos;
  icon_list->total_rows = 0;
  
  icon_list->timeout_handle = gtk_timeout_add(150,
                                              (GtkFunction) timeout_callback,
                                              (gpointer) icon_list);
}

/**
 * gnome_db_icon_list_get_pixmap
 * @icon_list: a @GnomeDbIconList widget
 *
 * Returns the pixmap file being used for the given icon list widget
 */
const gchar *
gnome_db_icon_list_get_pixmap (GnomeDbIconList *icon_list)
{
  g_return_val_if_fail(GNOME_DB_IS_ICON_LIST(icon_list), 0);
  return (icon_list->pixmap_file);
}

/**
 * gnome_db_icon_set_pixmap
 * @icon_list: a @GnomeDbIconList widget
 * @pixmap: pixmap file
 *
 * Sets the pixmap file to be used for the icons in the given
 * @GnomeDbIconList widget
 */
void
gnome_db_icon_list_set_pixmap (GnomeDbIconList *icon_list, gchar *pixmap)
{
  g_return_if_fail(GNOME_DB_IS_ICON_LIST(icon_list));
  
  if (icon_list->pixmap_file) g_free((gpointer) icon_list->pixmap_file);
  icon_list->pixmap_file = g_strdup(pixmap);
}