/* GNOME DB library
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(HAVE_CONFIG_H)
#  include <config.h>
#endif
#include "gda-ui.h"

static void     gnome_db_xml_viewer_class_init (GnomeDbXmlViewerClass *klass);
static void     gnome_db_xml_viewer_destroy    (GnomeDbXmlViewer *viewer);
static void     gnome_db_xml_viewer_init       (GnomeDbXmlViewer *viewer);

static void     add_property_cb                (GtkWidget *w, GnomeDbXmlViewer *viewer);
static gboolean enum_properties_cb             (Gda_XmlObject *obj,
                                                const gchar *prop,
                                                gpointer user_data);
static void     remove_property_cb             (GtkWidget *w, GnomeDbXmlViewer *viewer);
static void     select_child_cb                (GtkTree *tree, GtkWidget *child, gpointer data);
static void     unselect_child_cb              (GtkTree *tree, GtkWidget *child, gpointer data);

/*
 * GnomeDbXmlViewer signals
 */
enum
{
  GNOME_DB_XML_VIEWER_DATABASE_CHANGED,
  GNOME_DB_XML_VIEWER_LAST_SIGNAL
};
static gint gnome_db_xml_viewer_signals[GNOME_DB_XML_VIEWER_LAST_SIGNAL] = { 0 };

/*
 * Menus and toolbars
 */
GnomeUIInfo propertypopupmenu[] =
{
  { GNOME_APP_UI_ITEM, N_("Add property"), N_("Add a new property for this object"),
    add_property_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Remove property"), N_("Remove selected properties"),
    remove_property_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static GtkWidget *
find_node_by_type (GnomeDbXmlViewer *viewer, const gchar *type)
{
  GList* node;

  g_return_val_if_fail(GNOME_DB_IS_XML_VIEWER(viewer), 0);
  g_return_val_if_fail(type != 0, 0);

  node = g_list_first(viewer->nodes);
  while (node)
    {
      GtkWidget* item = (GtkWidget *) node->data;
      if (item)
        {
          gchar* tmp_type;

          tmp_type = (gchar *) gtk_object_get_data(GTK_OBJECT(item), "GDAUI_XmlViewer_NodeType");
          if (type && !g_strcasecmp(tmp_type, type)) return (item);
        }
      node = g_list_next(node);
    }
  return (0); /* not found */
}

/*
 * Callbacks
 */
static void
add_property_cb (GtkWidget *w, GnomeDbXmlViewer *viewer)
{
  GtkWidget *dialog, *table, *label, *name_entry, *value_entry;
  gint       btn;

  g_return_if_fail(GNOME_DB_IS_XML_VIEWER(viewer));
  g_return_if_fail(IS_GDA_XML_DATABASE(viewer->db));

  if (!viewer->selected_object) return;

  /* create the dialog */
  dialog = gnome_dialog_new(_("Add Property"),
                            GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL, NULL);
  table = gtk_table_new(2, 2, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, 0);
  
  label = gda_ui_new_label_widget(_("Property"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  name_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), name_entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

  label = gda_ui_new_label_widget(_("Value"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  value_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), value_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

  /* run the dialog */
  btn = gnome_dialog_run(GNOME_DIALOG(dialog));
  if (btn == 0)
    {
      gchar*         name;
      gchar*         value;
      Gda_XmlObject* obj = viewer->selected_object;

      name = gtk_entry_get_text(GTK_ENTRY(name_entry));
      value = gtk_entry_get_text(GTK_ENTRY(value_entry));
      if (name && value)
        {
          gda_xml_object_set_property(obj, name, value);
          gda_ui_clear_clist(GTK_CLIST(viewer->description));
          gda_xml_object_enum_properties(obj, (Gda_XmlEnumPropFunc) enum_properties_cb, viewer);
        }
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

static gboolean
enum_properties_cb (Gda_XmlObject *obj, const gchar *prop, gpointer user_data)
{
  gchar*            row[2];
  GnomeDbXmlViewer* viewer;

  g_return_val_if_fail(obj != 0, FALSE);
  g_return_val_if_fail(prop != 0, FALSE);
  g_return_val_if_fail(GNOME_DB_IS_XML_VIEWER(user_data), FALSE);

  /* add property to GtkCList */
  viewer = GNOME_DB_XML_VIEWER(user_data);
  row[0] = (gchar *) prop;
  row[1] = (gchar *) gda_xml_object_get_property(obj, prop);
  gtk_clist_freeze(GTK_CLIST(viewer->description));
  gtk_clist_append(GTK_CLIST(viewer->description), (gchar **) row);
  gtk_clist_thaw(GTK_CLIST(viewer->description));
  return (TRUE);
}

static void
remove_property_cb (GtkWidget *w, GnomeDbXmlViewer *viewer)
{
  g_return_if_fail(GNOME_DB_IS_XML_VIEWER(viewer));
  g_return_if_fail(IS_GDA_XML_DATABASE(viewer->db));

  if (!viewer->selected_object) return;
}

static void
select_child_cb (GtkTree *tree, GtkWidget *child, gpointer data)
{
  Gda_XmlObject* obj;

  g_return_if_fail(GTK_IS_WIDGET(child));
  g_return_if_fail(GNOME_DB_IS_XML_VIEWER(data));

  obj = (Gda_XmlObject *) gtk_object_get_data(GTK_OBJECT(child), "GDAUI_XmlViewer_TreeXmlObj");
  if (obj != 0)
    {
      GnomeDbXmlViewer* viewer = GNOME_DB_XML_VIEWER(data);
      viewer->selected_object = obj;
      gda_ui_clear_clist(GTK_CLIST(viewer->description));
      gda_xml_object_enum_properties(obj, (Gda_XmlEnumPropFunc) enum_properties_cb, data);
    }
  else GNOME_DB_XML_VIEWER(data)->selected_object = 0;
}

static void
unselect_child_cb (GtkTree *tree, GtkWidget *child, gpointer data)
{
  g_return_if_fail(GTK_IS_WIDGET(child));
  g_return_if_fail(GNOME_DB_IS_XML_VIEWER(data));

  GNOME_DB_XML_VIEWER(data)->selected_object = 0;
}

static gboolean
xml_enum_children_cb (Gda_XmlDatabase *db, Gda_XmlObject *obj, GtkWidget *tree_item)
{
  gchar*            type;
  gchar*            txt;
  xmlNodePtr        xmlnode;
  GtkWidget*        tree;
  GnomeDbXmlViewer* viewer;

  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), FALSE);
  g_return_val_if_fail(obj != 0, FALSE);
  g_return_val_if_fail(GTK_IS_TREE_ITEM(tree_item), FALSE);

  type = (gchar *) gda_xml_object_type(obj);
  tree = GTK_TREE(gtk_object_get_data(GTK_OBJECT(tree_item), "GDAUI_XmlViewer_NodeTree"));
  viewer = GNOME_DB_XML_VIEWER(gtk_object_get_data(GTK_OBJECT(tree_item),
                                                   "GDAUI_XmlViewer_ViewerWidget"));
  if (!tree)
    {
      tree = gda_ui_new_tree_widget();
      gtk_signal_connect(GTK_OBJECT(tree),
                         "select-child",
                         GTK_SIGNAL_FUNC(select_child_cb),
                         (gpointer) viewer);
      gtk_signal_connect(GTK_OBJECT(tree),
                         "unselect-child",
                         GTK_SIGNAL_FUNC(unselect_child_cb),
                         (gpointer) viewer);
      gtk_object_set_data(GTK_OBJECT(tree_item),
                          "GDAUI_XmlViewer_NodeTree",
                          (gpointer) tree);
      gtk_tree_set_view_mode(GTK_TREE(tree), GTK_TREE_VIEW_ITEM);
      gtk_tree_item_set_subtree(GTK_TREE_ITEM(tree_item), tree);
    }

  /* create the object */
  txt = g_strdup_printf("%s %s",
                        type ? type : "??",
                        (gchar *) gda_xml_object_get_property(obj,
                                                              GDA_XML_PROPERTY_NAME));
  tree_item = gtk_tree_item_new_with_label(txt);
  gtk_object_set_data(GTK_OBJECT(tree_item), "GDAUI_XmlViewer_ViewerWidget", (gpointer) viewer);
  gtk_object_set_data(GTK_OBJECT(tree_item), "GDAUI_XmlViewer_TreeXmlObj", (gpointer) obj);
  gtk_widget_show(tree_item);
  gtk_tree_append(GTK_TREE(tree), tree_item);
  g_free((gpointer) txt);

  /* add all children */
  return gda_xml_object_enum_children(obj,
                                      (Gda_XmlEnumFunc) xml_enum_children_cb,
                                      (const gchar *) 0,
                                      (gpointer) tree_item);
}

static gboolean
xml_enum_cb (Gda_XmlDatabase *db, Gda_XmlObject *obj, GnomeDbXmlViewer *viewer)
{
  GtkWidget*    tree;
  GtkWidget*    tree_item;

  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), FALSE);
  g_return_val_if_fail(obj != 0, FALSE);
  g_return_val_if_fail(GNOME_DB_IS_XML_VIEWER(viewer), FALSE);
  
  tree_item = find_node_by_type(viewer, gda_xml_object_type(obj));

  /* create tree node */
  if (!tree_item)
    {
      tree_item = gtk_tree_item_new_with_label((gchar *) gda_xml_object_type(obj));
      gtk_object_set_data(GTK_OBJECT(tree_item),
                          "GDAUI_XmlViewer_NodeType",
                          (gpointer) gda_xml_object_type(obj));
      gtk_widget_show(tree_item);
      gtk_tree_append(GTK_TREE(viewer->tree), (gpointer) tree_item);
      viewer->nodes = g_list_append(viewer->nodes, (gpointer) tree_item);

      tree = gda_ui_new_tree_widget();
      gtk_signal_connect(GTK_OBJECT(tree),
                         "select-child",
                         GTK_SIGNAL_FUNC(select_child_cb),
                         (gpointer) viewer);
      gtk_signal_connect(GTK_OBJECT(tree),
                         "unselect-child",
                         GTK_SIGNAL_FUNC(unselect_child_cb),
                         (gpointer) viewer);
      gtk_object_set_data(GTK_OBJECT(tree_item),
                          "GDAUI_XmlViewer_NodeTree",
                          (gpointer) tree);
      gtk_tree_set_view_mode(GTK_TREE(tree), GTK_TREE_VIEW_ITEM);
      gtk_tree_item_set_subtree(GTK_TREE_ITEM(tree_item), tree);
      gtk_tree_item_expand(GTK_TREE_ITEM(tree_item));
    }
  else
    {
      tree = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(tree_item),
                                               "GDAUI_XmlViewer_NodeTree");
      if (!tree)
        {
          gda_ui_show_error(_("Invalid behaviour in %s"), __FILE__);
        }
    }
    
  /* create the object */
  tree_item = gtk_tree_item_new_with_label((gchar *) gda_xml_object_get_property(obj,
                                                                    GDA_XML_PROPERTY_NAME));
  gtk_object_set_data(GTK_OBJECT(tree_item), "GDAUI_XmlViewer_ViewerWidget", (gpointer) viewer);
  gtk_object_set_data(GTK_OBJECT(tree_item), "GDAUI_XmlViewer_TreeXmlObj", (gpointer) obj);
  gtk_widget_show(tree_item);
  gtk_tree_append(GTK_TREE(tree), tree_item);

  /* add all children */
  gda_xml_object_enum_children(obj, (Gda_XmlEnumFunc) xml_enum_children_cb, 0, (gpointer) tree_item);
  return (TRUE);
}

/*
 * GnomeDbXmlViewer widget interface
 */
static void
gnome_db_xml_viewer_class_init (GnomeDbXmlViewerClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  gnome_db_xml_viewer_signals[GNOME_DB_XML_VIEWER_DATABASE_CHANGED] =
           gtk_signal_new("database_changed", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbXmlViewerClass, database_changed),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, gnome_db_xml_viewer_signals,
                               GNOME_DB_XML_VIEWER_LAST_SIGNAL);
                               
  object_class->destroy = gnome_db_xml_viewer_destroy;
  klass->database_changed = 0;
}

static void
gnome_db_xml_viewer_init (GnomeDbXmlViewer *viewer)
{
  GtkWidget* box, *box2, *table, *label, *scroll, *popup_menu;
  const gchar* titles[] = { N_("Property"), N_("Value") };
  
  box = gtk_hbox_new(0, 0);
  gtk_widget_show(box);
  gtk_box_pack_start(GTK_BOX(viewer), box, 1, 1, 0);
  
  /* create tree widget */
  scroll = gda_ui_new_scrolled_window_widget();
  gtk_box_pack_start(GTK_BOX(box), scroll, 1, 1, 0);
  viewer->tree = gda_ui_new_tree_widget();
  gtk_signal_connect(GTK_OBJECT(viewer->tree),
                     "select-child",
                     GTK_SIGNAL_FUNC(select_child_cb),
                     (gpointer) viewer);
  gtk_signal_connect(GTK_OBJECT(viewer->tree),
                     "unselect-child",
                     GTK_SIGNAL_FUNC(unselect_child_cb),
                     (gpointer) viewer);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), viewer->tree);
  
  box2 = gtk_vbox_new(0, 0);
  gtk_widget_show(box2);
  gtk_box_pack_start(GTK_BOX(box), box2, 1, 1, 0);

  table = gtk_table_new(4, 2, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(box2), table, 0, 0, 0);
  
  label = gda_ui_new_label_widget(_("Database name"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  viewer->dbname_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), viewer->dbname_entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  label = gda_ui_new_label_widget(_("File name"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  viewer->filename_entry = gda_ui_new_file_entry_widget("FE_Designer_history");
  gtk_table_attach(GTK_TABLE(table), viewer->filename_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  
  /* create object description */
  scroll = gda_ui_new_scrolled_window_widget();
  gtk_box_pack_start(GTK_BOX(box2), scroll, 1, 1, 0);
  viewer->description = gda_ui_new_clist_widget((gchar **) titles, 
                                                sizeof(titles) / sizeof(titles[0]));
  popup_menu = gda_ui_new_popup_menu(viewer->description, propertypopupmenu, (gpointer) viewer);
  gtk_container_add(GTK_CONTAINER(scroll), viewer->description);

  viewer->nodes = 0;
  viewer->db = gda_xml_database_new("XML Database");
}

guint
gnome_db_xml_viewer_get_type (void)
{
  static guint db_xml_viewer_type = 0;
  
  if (!db_xml_viewer_type)
    {
      GtkTypeInfo db_xml_viewer_info =
      {
        "GnomeDbXmlViewer",
        sizeof (GnomeDbXmlViewer),
        sizeof (GnomeDbXmlViewerClass),
        (GtkClassInitFunc) gnome_db_xml_viewer_class_init,
        (GtkObjectInitFunc) gnome_db_xml_viewer_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_xml_viewer_type = gtk_type_unique(gtk_vbox_get_type(), &db_xml_viewer_info);
    }
  return (db_xml_viewer_type);
}

/**
 * gnome_db_xml_viewer_new
 * @db: an existing Gda_XmlDatabase object
 *
 * Creates a new GnomeDbXmlViewer widget, which allows you to present visually
 * a file generated by the XML interface functions in the GDA client library.
 * It also lets you modify the file at run-time, displaying all the changes
 * you made
 */
GtkWidget *
gnome_db_xml_viewer_new (Gda_XmlDatabase *db)
{
  GtkWidget* viewer = gtk_type_new(gnome_db_xml_viewer_get_type());
  
  if (IS_GDA_XML_DATABASE(db))
    gnome_db_xml_viewer_set_database(GNOME_DB_XML_VIEWER(viewer), db);
  return (viewer);
}

/**
 * gnome_db_xml_viewer_new_from_file
 * @filename: XML file name
 *
 * Creates a new GnomeDbXmlViewer from an existing XML file
 */
GtkWidget *
gnome_db_xml_viewer_new_from_file (const gchar *filename)
{
  GtkWidget* viewer = gtk_type_new(gnome_db_xml_viewer_get_type());
  gnome_db_xml_viewer_load_file(GNOME_DB_XML_VIEWER(viewer), filename);
  return (viewer);
}

static void
gnome_db_xml_viewer_destroy (GnomeDbXmlViewer *viewer)
{
  g_return_if_fail(GNOME_DB_IS_XML_VIEWER(viewer));
  
  g_warning("gnome_db_xml_viewer_destroy called");
  if (IS_GDA_XML_DATABASE(viewer->db)) gda_xml_database_free(viewer->db);
}

/**
 * gnome_db_xml_viewer_clear
 * @db:a GnomeDbXmlViewer widget
 *
 * Clears the given widget. That is, all the child widgets are cleared, and all
 * the associated memory is freed.
 */
void
gnome_db_xml_viewer_clear (GnomeDbXmlViewer *viewer)
{
  const gchar* empty_string = "";
  GList*       children;

  g_return_if_fail(GNOME_DB_IS_XML_VIEWER(viewer));
  
  if (IS_GDA_XML_DATABASE(viewer->db)) gda_xml_database_free(viewer->db);
  
  gtk_entry_set_text(GTK_ENTRY(viewer->dbname_entry), empty_string);
  gtk_entry_set_text(gnome_entry_gtk_entry(GNOME_FILE_ENTRY(viewer->filename_entry)), empty_string);

  /* FIXME: how to clear a GtkTree? */
  children = gtk_container_children(GTK_CONTAINER(viewer->tree));
  gtk_widget_ref(viewer->tree);
  gtk_tree_remove_items(GTK_TREE(viewer->tree), children);
  g_list_free(children);

  gda_ui_clear_clist(GTK_CLIST(viewer->description));

  viewer->db = gda_xml_database_new("XML Database");
  viewer->selected_object = 0;
}

/**
 * gnome_db_xml_viewer_load_file
 * @db:a GnomeDbXmlViewer widget
 * @filename: XML file name
 *
 * Loads an existing XML file and displays its structure in the given
 * GnomeDbXmlViewer widget
 */
void
gnome_db_xml_viewer_load_file (GnomeDbXmlViewer *viewer, const gchar *filename)
{
  Gda_XmlDatabase *db;
  
  g_return_if_fail(GNOME_DB_IS_XML_VIEWER(viewer));
  g_return_if_fail(filename != 0);
  
  /* load file */
  db = gda_xml_database_new_from_file(filename);
  if (db != 0)
    {
      gnome_db_xml_viewer_set_database(viewer, db);
    }
}
/**
 * gnome_db_xml_viewer_get_database
 * @viewer: a GnomeDbXmlViewer widget
 *
 * Returns the Gda_XmlDatabase object being used for the given widget
 */
Gda_XmlDatabase *
gnome_db_xml_viewer_get_database (GnomeDbXmlViewer *viewer)
{
  g_return_val_if_fail(GNOME_DB_IS_XML_VIEWER(viewer), 0);
  return (viewer->db);
}

/**
 * gnome_db_xml_viewer_set_database
 * @viewer: a GnomeDbXmlViewer widget
 * @db: a Gda_XmlDatabase object
 */
void
gnome_db_xml_viewer_set_database (GnomeDbXmlViewer *viewer, Gda_XmlDatabase *db)
{
  gchar* str;

  g_return_if_fail(GNOME_DB_IS_XML_VIEWER(viewer));
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  
  gnome_db_xml_viewer_clear(viewer);
  viewer->db = db;
  
  /* show information */
  gda_xml_database_enum_objects(db, (Gda_XmlEnumFunc) xml_enum_cb, 0, (gpointer) viewer);
  
  if ((str = gda_xml_database_get_name(db))) 
    gtk_entry_set_text(GTK_ENTRY(viewer->dbname_entry), str);

  if ((str = gda_xml_database_get_filename(db)))
    gtk_entry_set_text(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(viewer->filename_entry)), str);
}

/**
 * gnome_db_xml_viewer_get_selected
 * @viewer: a GnomeDbXmlViewer widget
 *
 * Returns the currently selected object in the XML tree as a 
 * #Gda_XmlObjet structure. This allows you to directly call
 * the GDA XML interface functions on the returned object
 */
Gda_XmlObject *
gnome_db_xml_viewer_get_selected (GnomeDbXmlViewer *viewer)
{
  g_return_val_if_fail(GNOME_DB_IS_XML_VIEWER(viewer), 0);
  return (viewer->selected_object);
}