/* GNOME-DB
 * Copyright (c) 1998-2000 by Rodrigo Moya
 * Copyright (c) 2000 by Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <ctype.h>
#include "gda.h"
#include "gda-common.h"
#include "gda-postgres-types.h"
#include "gda-postgres-command.h"
#include "gda-postgres-connection.h"
#include "gda-postgres-error.h"
#include "gda-postgres-recset.h"
#include <stdlib.h>

typedef Gda_POSTGRES_Recordset* (*schema_ops_fn)(Gda_POSTGRES_Error *,
                                                 Gda_POSTGRES_Connection *,
                                                 GDA_Connection_Constraint *,
                                                 gint );

schema_ops_fn schema_ops[GDA_Connection_GDCN_SCHEMA_LAST] =
{
  0,
};


typedef struct _Gda_connection_data 
{
  gchar *dsn;
  gchar *user;
  guint nreg;
  Gda_POSTGRES_Types_Array *types_array;
} Gda_connection_data;


#define Gda_POSTGRES_Types_Array_Nb 23
Gda_POSTGRES_Types_Array types_array[Gda_POSTGRES_Types_Array_Nb] = 
{
  {"datetime", 0, GDA_TypeDbTimestamp, SQL_C_TIMESTAMP},
  {"timestamp", 0, GDA_TypeDbTimestamp, SQL_C_TIMESTAMP},
  {"abstime", 0, GDA_TypeDbTimestamp, SQL_C_TIMESTAMP}, 
  {"bool", 0, GDA_TypeBoolean, SQL_C_BIT},
  {"bpchar", 0, GDA_TypeChar, SQL_C_CHAR}, 
  {"char", 0, GDA_TypeChar, SQL_C_CHAR},
  {"date", 0, GDA_TypeDbDate, SQL_C_DATE},
  {"float4", 0, GDA_TypeSingle, SQL_C_FLOAT}, 
  {"float8", 0, GDA_TypeDouble, SQL_C_DOUBLE}, 
  {"int2", 0, GDA_TypeSmallint, SQL_C_SSHORT},
  {"int4", 0, GDA_TypeInteger, SQL_C_SLONG},
  {"text", 0, GDA_TypeLongvarchar, SQL_C_CHAR}, 
  {"varchar", 0 , GDA_TypeVarchar, SQL_C_CHAR},
  {"char2", 0, GDA_TypeChar, SQL_C_CHAR},
  {"char4", 0, GDA_TypeChar, SQL_C_CHAR},
  {"char8", 0, GDA_TypeChar, SQL_C_CHAR},
  {"char16", 0, GDA_TypeChar, SQL_C_CHAR},
  {"name", 0, GDA_TypeChar, SQL_C_CHAR},
  {"bytea", 0, GDA_TypeVarbin, SQL_C_BINARY},
  {"oid", 0, GDA_TypeInteger, SQL_C_SLONG},
  {"xid", 0, GDA_TypeInteger, SQL_C_SLONG},
  {"time", 0, GDA_TypeDbTime, SQL_C_TIME},
  {"money", 0, GDA_TypeSingle, SQL_C_FLOAT}
};
GSList *global_connection_data_list = NULL;

/* private functions */
static gboolean
gda_postgres_connection_is_type_known (Gda_POSTGRES_Connection *cnc, 
				       gulong sql_type);

static gint
execute_command (Gda_POSTGRES_Connection *cnc, gchar *cmd)
{
  g_return_val_if_fail(cnc != 0, -1);
  g_return_val_if_fail(cmd != 0, -1);
  if (PQstatus(cnc->pq_conn) == CONNECTION_OK)
    {
      PGresult *rc = PQexec(cnc->pq_conn, cmd);
      switch (PQresultStatus(rc))
        {
        case PGRES_COMMAND_OK :
        case PGRES_EMPTY_QUERY :
        case PGRES_TUPLES_OK :
        case PGRES_COPY_OUT :
        case PGRES_COPY_IN :
          PQclear(rc);
          return (0);
        case PGRES_BAD_RESPONSE :
        case PGRES_NONFATAL_ERROR :
        case PGRES_FATAL_ERROR :
          break;
        }
      gda_postgres_error_make(0, 0, cnc, "PQexec");
      if (rc != 0)
        PQclear(rc);
    }
  return (-1);
}

static Gda_POSTGRES_Recordset *
schema_columns (Gda_POSTGRES_Error *error,
                Gda_POSTGRES_Connection *cnc, GDA_Connection_Constraint *constraint,
                gint length)
{
  Gda_POSTGRES_Command *cmd;
  Gda_POSTGRES_Recordset *recset = 0;
  GDA_Connection_Constraint *ptr;
  gulong affected;
  gchar *table_qualifier = 0, *table_owner = 0, *table_name = 0, *column_name = 0;
  
  /* create the command object */
  cmd = gda_postgres_cmd_new();
  cmd->cnc = cnc;
  
  /* get table name */
  ptr = constraint;
  while (length)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_OBJECT_CATALOG : /* not used in postgres! */
          table_qualifier = ptr->value;
          fprintf(stderr, "schema_columns: table_qualifier = '%s' UNUSED!\n", 
		  table_qualifier);
          break;
        case GDA_Connection_OBJECT_SCHEMA :
          table_owner = ptr->value;
          fprintf(stderr, "schema_columns: table_owner = '%s'\n", table_owner);
          break;
        case GDA_Connection_OBJECT_NAME :
          table_name = ptr->value;
          fprintf(stderr, "schema_columns: table_name = '%s'\n", table_name);
          break;
        case GDA_Connection_COLUMN_NAME :
          column_name = ptr->value;
          fprintf(stderr, "schema_columns: column_name = '%s'\n", column_name);
          break;
        default:
          fprintf(stderr,"schema_columns: invalid constraint type %d\n", 
		  ptr->ctype);
          return (0);
        }
      length--;
      ptr++;
    }

  if (!table_name)
    {
      fprintf(stderr, "schema_columns: table name is null\n");
      return (0);
    }

  cmd->cmd = g_strdup_printf("SELECT b.attname AS \"Name\", "
                             "c.typname AS \"Type\", "
                             "textcat(b.attlen, textcat('/', b.atttypmod)) "
			     "AS \"Size\", "
                             "0 AS \"Precision\", "
                             "NOT(b.attnotnull) AS \"Nullable\", "
			     "textcat('%s ', b.attnum) AS \"Key\", "
			     "textcat('%s ', b.attname) AS \"Default Value\", "
                             "obj_description(b.oid) AS \"Comments\" "
                             "FROM pg_type c, "
                             "pg_attribute b, "
                             "pg_class a "
                             "WHERE a.relname = '%s' "
                             "AND b.attrelid = a.oid "
                             "AND c.oid = b.atttypid "
                             "AND b.attnum > 0 "
                             "ORDER BY b.attnum", table_name,
			     table_name, table_name);

  /* execute the command */
  if (cmd->cmd != 0)
    {
      recset = gda_postgres_cmd_execute(cmd, error, 0, &affected, 0);
      fprintf(stderr, "Nb of columns: %ld\n", affected);
      gda_postgres_cmd_free(cmd);
      if (recset)
	{ /* Setting replacements for column length */
	  Gda_POSTGRES_Recordset_Replacement *repl;
	  repl = (Gda_POSTGRES_Recordset_Replacement *) 
	    g_malloc(sizeof(Gda_POSTGRES_Recordset_Replacement));
	  repl->colnum = 2;
	  repl->newtype = GDA_TypeSmallint;
	  repl->trans_func = replace_TABLE_FIELD_with_length;
	  recset->replacements = g_slist_append(recset->replacements,
						repl);
	  /* Setting replacement for default value */
	  repl = (Gda_POSTGRES_Recordset_Replacement *) 
	    g_malloc(sizeof(Gda_POSTGRES_Recordset_Replacement));
	  repl->colnum = 6;
	  repl->newtype = GDA_TypeVarchar;
	  repl->trans_func = replace_TABLE_FIELD_with_defaultval;
	  recset->replacements = g_slist_append(recset->replacements,
						repl);
	  /* Setting replacements for key or not */
	  repl = (Gda_POSTGRES_Recordset_Replacement *) 
	    g_malloc(sizeof(Gda_POSTGRES_Recordset_Replacement));
	  repl->colnum = 5;
	  repl->newtype = GDA_TypeBoolean;
	  repl->trans_func = replace_TABLE_FIELD_with_iskey;
	  recset->replacements = g_slist_append(recset->replacements,
						repl);
	}
      return (recset);
    }
  gda_postgres_cmd_free(cmd);
  return (0);
}

static Gda_POSTGRES_Recordset *
schema_procedures (Gda_POSTGRES_Error *error,
                   Gda_POSTGRES_Connection *cnc,
                   GDA_Connection_Constraint *constraint,
                   gint length)
{
  gboolean                   extra_info = FALSE;
  gint                       cnt, i, cntnt, cntntsym;
  GDA_Connection_Constraint* ptr;
  Gda_POSTGRES_Command*      cmd;
  Gda_POSTGRES_Recordset*    recset;
  gchar*                     proc_name = 0;
  gchar*                     proc_owner = 0;
  GString*                   query=NULL;
  GString*                   and_value=NULL;
  gboolean                   where;
  gulong                     affected;
  Gda_Builtin_Result*        bres;
  PGresult                   *res, *notypes, *notypes_sym;
  gchar                      **row;
  gint                       nbcols; /* nb of cols in the bres */

  /* process constraints */
  ptr = constraint;
  for (cnt = 0; cnt < length && ptr != 0; cnt++)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_OBJECT_NAME :
          proc_name = ptr->value;
          fprintf(stderr, "schema_procedures: proc_name = '%s'\n", proc_name);
	  if (!and_value)
	    {
	      and_value = g_string_new("");
	      g_string_sprintfa(and_value, "p.proname = '%s' ", ptr->value);
	    }
	  else
	    g_string_sprintfa(and_value, "AND p.proname = '%s' ", ptr->value);
          break;
        case GDA_Connection_OBJECT_SCHEMA :
          proc_owner = ptr->value;
          fprintf(stderr, "schema_procedures: proc_owner = '%s'\n",proc_owner);
	  if (!and_value)
	    {
	      and_value = g_string_new("");
	      g_string_sprintfa(and_value, "u.usename = '%s' ", ptr->value);
	    }
	  else
	    g_string_sprintfa(and_value, "AND u.usename = '%s' ", ptr->value);
          break;
        case GDA_Connection_OBJECT_CATALOG :
          fprintf(stderr, "schema_procedures: proc_catalog = '%s' UNUSED!\n", 
		  ptr->value);
          break;
        case GDA_Connection_EXTRA_INFO :
          extra_info = TRUE;
          break;
        default:
          fprintf(stderr, "schema_procedures: invalid constraint type %d\n", ptr->ctype);
          return (0);
        }
      ptr++;
    }

  /* build command object */
  cmd = gda_postgres_cmd_new();
  gda_postgres_cmd_set_connection(cmd, cnc);
  cmd->cmd = 0;

  /* build the query */
#define IN_PG_PROCS_NON_ALLOWED "('abstime', 'array_assgn', 'array_clip', 'array_in', 'array_ref', 'array_set', 'bpcharin', 'btabstimecmp', 'btcharcmp', 'btfloat4cmp', 'btfloat8cmp', 'btint24cmp', 'btint2cmp', 'btint42cmp', 'btint4cmp', 'btnamecmp', 'bttextcmp', 'byteaGetBit', 'byteaGetByte', 'byteaGetSize', 'byteaSetBit', 'byteaSetByte', 'date', 'date_cmp', 'datetime', 'datetime_cmp', 'float8', 'hashbpchar', 'hashchar', 'hashfloat4', 'hashfloat8', 'hashint2', 'hashint4', 'hashname', 'hashtext', 'hashvarchar', 'int', 'int2', 'int4', 'lo_close', 'lo_lseek', 'lo_tell', 'loread', 'lowrite', 'pg_get_ruledef', 'pg_get_userbyid', 'pg_get_viewdef', 'pqtest', 'time_cmp', 'timestamp', 'userfntest', 'varcharcmp', 'varcharin', 'xideq', 'keyfirsteq')"
  where = FALSE;
  if (extra_info)
    {
      query = g_string_new("SELECT p.proname AS \"Name\", "
			   "p.oid AS \"Object Id\", "
			   "u.usename AS \"Owner\", "
			   "obj_description(p.oid) AS \"Comments\", "
			   "p.pronargs AS \"Number of args\", "
			   "p.oid AS \"SQL Def.\", "
			   "p.prorettype, p.proargtypes " /* for checking */
			   "FROM pg_proc p, pg_user u "
			   "WHERE u.usesysid=p.proowner AND "
			   "(pronargs = 0 or oid8types(p.proargtypes)!= '') ");
      where = TRUE;
      nbcols = 6;
    }
  else 
    {
      query = g_string_new("SELECT p.proname AS \"Name\", "
			   "p.oid AS \"Object Id\", "
			   "obj_description(p.oid) AS \"Comments\", "
			   "p.prorettype, p.proargtypes " /* for checking */
			   "FROM pg_proc p ");
      nbcols = 3;
    }
  if (proc_owner && !extra_info)
    {
      g_string_append(query, ", pg_user u WHERE u.usesysid=p.proowner ");
      where = TRUE;
    }

  if (and_value)
    {
      if (!where)
	g_string_append(query, "WHERE ");
      else
	g_string_append(query, "AND ");
      g_string_sprintfa(query, "%s", and_value->str);
      g_string_free(and_value, TRUE);
    }
  if (!where)
    g_string_append(query, "WHERE ");
  else
    g_string_append(query, "AND ");
  g_string_append(query, "p.proname NOT IN " IN_PG_PROCS_NON_ALLOWED " ");
  g_string_append(query, "ORDER BY proname, prorettype");

  /* build the bres */
  bres = Gda_Builtin_Result_new(nbcols, "result", 0, -1);
  Gda_Builtin_Result_set_att(bres, 0, "Name", 
	        gda_postgres_connection_get_sql_type(cnc, "varchar"), -1);
  Gda_Builtin_Result_set_att(bres, 1, "Object Id", 
		gda_postgres_connection_get_sql_type(cnc, "varchar"), -1);
  if (extra_info)
    {
      Gda_Builtin_Result_set_att(bres, 2, "Owner", 
		gda_postgres_connection_get_sql_type(cnc, "varchar"), -1);
      Gda_Builtin_Result_set_att(bres, 3, "Comments", 
		gda_postgres_connection_get_sql_type(cnc, "varchar"), -1);
      Gda_Builtin_Result_set_att(bres, 4, "Number of Args.", 
		gda_postgres_connection_get_sql_type(cnc, "int"), 4);
      Gda_Builtin_Result_set_att(bres, 5, "Sql Def.", 
		gda_postgres_connection_get_sql_type(cnc, "varchar"), -1);
    }
  else
    Gda_Builtin_Result_set_att(bres, 2, "Comments", 
		gda_postgres_connection_get_sql_type(cnc, "varchar"), -1);

  /* do the query, get a PGresult, and for each tuple:
     - see if the function is authorized (not in le result of notypes)
     - see if all the parameters and return type are in the known postgres 
       types. If that is the case, add the tuple to the builtin result. 
     Then clear the PGresult */
  res = PQexec(cnc->pq_conn, query->str);
  fprintf(stderr, "Query: %s\n", query->str);
  g_string_free(query, TRUE);

  /* REM: contents of notypes and of notype_sym have en empty intersection */
  /* notypes is the list of functions' oid which can be obtained 
     using operators */
  notypes =PQexec(cnc->pq_conn, 
		  "SELECT distinct text(oprcode) FROM "
		  "pg_operator ORDER BY oprcode");
  /* notypes_sym is the list of functions' oid which are synonyms with other 
     functions */
  notypes_sym = PQexec(cnc->pq_conn, 
		       "select p.oid from pg_proc p where text(p.proname)!=p.prosrc and p.prosrc in (select text(proname) from pg_proc where text(proname)= p.prosrc) order by p.oid");
  if ((!res || (PQresultStatus(res) != PGRES_TUPLES_OK) ||
       (PQntuples(res) < 1)) ||
      (!notypes || (PQresultStatus(notypes) != PGRES_TUPLES_OK) ||
       (PQntuples(notypes) < 1)) ||
      (!notypes_sym || (PQresultStatus(notypes_sym) != PGRES_TUPLES_OK) ||
       (PQntuples(notypes_sym) < 1)))
    {
      gda_postgres_cmd_free(cmd);
      fprintf(stderr, "Error: res=%p and notypes=%p and notypes_syn=%p\n", 
	      res, notypes, notypes_sym);
      if (res)
	PQclear(res);
      if (notypes)
	PQclear(notypes);
      if (notypes_sym)
	PQclear(notypes_sym);
      return NULL;
    }
 
  cnt = PQntuples(res);
  cntnt = PQntuples(notypes);
  cntntsym = PQntuples(notypes_sym);
  row = g_new(gchar*, nbcols);
  for (i=0; i<cnt; i++)
    {
      gboolean insert = TRUE, cont;
      GSList *list;
      int j;

      /* check if the proc is not in the list of notypes */
      cont = TRUE;
      j = 0;
      while ((j<cntnt) && insert && cont)
	{
	  gint cmpres;
	  cmpres = atoi(PQgetvalue(res, i, 1)) - /* object id is tested */
	    atoi(PQgetvalue(notypes, j, 0));
	  if (!cmpres)
	    insert = FALSE;
	  if (cmpres < 0)
	    cont = FALSE;
	  j++;
	}

      /* check if the proc is not in the list of notypes_sym */
      cont = TRUE;
      j = 0;
      while ((j<cntntsym) && insert && cont)
	{
	  gint cmpres;
	  cmpres = atoi(PQgetvalue(res, i, 1)) - /* object id is tested */
	    atoi(PQgetvalue(notypes_sym, j, 0));
	  if (!cmpres)
	    insert = FALSE;
	  if (cmpres < 0)
	    cont = FALSE;
	  j++;
	}

      /* check if PROC is ok:
       * the return type is in col nbcols 
       * and the arg types is in col nbcols+1 
       */
      if (insert && !gda_postgres_connection_is_type_known(cnc, 
				  atoi(PQgetvalue(res, i, nbcols))))
	insert = FALSE;
      if (insert) /* checking on the IN params */
	{
	  list = convert_tabular_to_list((PQgetvalue(res, i, nbcols+1)));
	  while (list)
	    {
	      if (!gda_postgres_connection_is_type_known(cnc,
				       atoi((gchar*)(list->data))))
		insert = FALSE;
	      g_free(list->data);
	      list = g_slist_remove_link(list, list);
	    }
	}
      
      /* if OK, we put it into bres */
      if (insert)
	{
	  for (j=0; j<nbcols; j++)
	    row[j] = PQgetvalue(res, i, j);
	  Gda_Builtin_Result_add_row(bres, row);      
	}
    }
  PQclear(res);
  PQclear(notypes);
  PQclear(notypes_sym);
  g_free(row);
  recset = gda_postgres_cmd_execute_from_builtin(cmd, bres, &affected);
  fprintf(stderr, "Nb of procs: %ld\n", affected);
  gda_postgres_cmd_free(cmd);
  if (recset && extra_info)
    { /* Setting replacements for nb of params */
      Gda_POSTGRES_Recordset_Replacement *repl; 

      /* Setting replacements for SQL definition */
      repl = (Gda_POSTGRES_Recordset_Replacement *) 
	g_malloc(sizeof(Gda_POSTGRES_Recordset_Replacement));
      repl->colnum = 5;
      repl->newtype = GDA_TypeVarchar;
      repl->trans_func = replace_FUNCTION_OID_with_SQL;
      recset->replacements = g_slist_append(recset->replacements,
					    repl);
    }
  return (recset);
}

static Gda_POSTGRES_Recordset *
schema_proc_params (Gda_POSTGRES_Error *error,
		    Gda_POSTGRES_Connection *cnc,
		    GDA_Connection_Constraint *constraint,
		    gint length)
{
  gint                       cnt;
  GDA_Connection_Constraint* ptr;
  Gda_POSTGRES_Command*      cmd;
  Gda_POSTGRES_Recordset*    recset;
  gchar*                     proc_name = 0;
  gchar*                     query;
  gulong                     affected;
  Gda_Builtin_Result*        bres;
  PGresult*                  res;
  gchar                      *row[2];
  GSList*                    list;

  /* process constraints */
  ptr = constraint;
  for (cnt = 0; cnt < length && ptr != 0; cnt++)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_OBJECT_NAME :
          proc_name = ptr->value;
          fprintf(stderr, "schema_proc_params: proc_name = '%s'\n", proc_name);
          break;
        default:
          fprintf(stderr, "schema_procedures: invalid constraint type %d\n", ptr->ctype);
          return (0);
        }
      ptr++;
    }

  if (!proc_name)
    {
      fprintf(stderr, "schema_proc_params: proc name is null\n");
      return (0);
    }  

  /* building initial builtin result */
  bres = Gda_Builtin_Result_new(2, "result", 0, -1);
  Gda_Builtin_Result_set_att(bres, 0, "InOut", 
			  gda_postgres_connection_get_sql_type(cnc, "varchar"),
			     -1);
  Gda_Builtin_Result_set_att(bres, 1, "Type", 
			  gda_postgres_connection_get_sql_type(cnc, "varchar"),
			     -1);

  cmd = gda_postgres_cmd_new();
  gda_postgres_cmd_set_connection(cmd, cnc);
  cmd->cmd = 0;

  /* fetch what we want, and put it into out builtin result */
  query = g_strdup_printf("SELECT prorettype, proargtypes FROM pg_proc "
			  "WHERE (pronargs = 0 or oid8types(proargtypes)!= '')"
			  " AND oid = %s", proc_name);
  res = PQexec(cnc->pq_conn, query);
  g_free(query);
  if (!res || (PQresultStatus(res) != PGRES_TUPLES_OK) ||
      (PQntuples(res) < 1))
    {
      gda_postgres_cmd_free(cmd);
      if (res)
	PQclear(res);
      return NULL;
    }
  row[0] = "out";
  row[1] = gda_postgres_connection_get_type_name(cnc, 
						 atoi(PQgetvalue(res, 0, 0)));
  if (row[1])
    Gda_Builtin_Result_add_row(bres, row);
  list = convert_tabular_to_list(PQgetvalue(res, 0, 1));
  PQclear(res);
  row[0] = "in";
  while (list)
    {
      row[1] = gda_postgres_connection_get_type_name(cnc, 
						 atoi((gchar*)(list->data)));
      if (row[1])
	Gda_Builtin_Result_add_row(bres, row);
      if (list->data)
	g_free(list->data);
      list = g_slist_remove_link(list, list);
    }

  /* final step */
  recset = gda_postgres_cmd_execute_from_builtin(cmd, bres, &affected);
  fprintf(stderr, "Nb of proc params: %ld\n", affected);
  gda_postgres_cmd_free(cmd);
  return (recset);

  /* COMPLETE QUERY:
     SELECT p.proname as function, t.typname, u.usename, p.proargtypes as arguments,  obj_description(p.oid) as description FROM pg_proc p, pg_type t, pg_user u WHERE u.usesysid=p.proowner AND p.prorettype = t.oid and (pronargs = 0 or oid8types(p.proargtypes)!= '') ORDER BY function, typname;
  */
}

static Gda_POSTGRES_Recordset *
schema_aggregates (Gda_POSTGRES_Error *error,
                   Gda_POSTGRES_Connection *cnc,
                   GDA_Connection_Constraint *constraint,
                   gint length)
{
  GString*                   query;
  Gda_POSTGRES_Command*      cmd;
  Gda_POSTGRES_Recordset*    recset = 0;
  GDA_Connection_Constraint* ptr;
  gint                       cnt;
  GString*                   and_condition = 0;
  gulong                     affected;
  gboolean                   extra_info=FALSE;

  /* process constraints */
  ptr = constraint;
  for (cnt = 0; cnt < length && ptr != 0; cnt++)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_EXTRA_INFO :
	  extra_info = TRUE;
	  break;
        case GDA_Connection_OBJECT_NAME :
          if (!and_condition) and_condition = g_string_new("");
          g_string_sprintfa(and_condition, "AND a.oid='%s' ", ptr->value);
          fprintf(stderr, "schema_tables: aggregate name = '%s'\n", 
		  ptr->value);
          break;
        case GDA_Connection_OBJECT_SCHEMA :
          if (!and_condition) and_condition = g_string_new("");
          g_string_sprintfa(and_condition, "AND b.usename='%s' ", ptr->value);
          fprintf(stderr, "schema_tables: table_schema = '%s'\n", ptr->value);
          break;
        case GDA_Connection_OBJECT_CATALOG :
          fprintf(stderr, "schema_procedures: proc_catalog = '%s' UNUSED!\n", 
		  ptr->value);
          break;
        default :
          fprintf(stderr, "schema_tables: invalid constraint type %d\n", 
		  ptr->ctype);
          g_string_free(and_condition, TRUE);
          return (0);
        }
      ptr++;
    }

  /* build the command object */
  cmd = gda_postgres_cmd_new();
  cmd->cnc = cnc;

  /* build the query */
  query = g_string_new("SELECT a.aggname AS \"Name\", "
		       "a.oid AS \"Object Id\", "
		       "t.typname as \"IN Type\", ");
  if (extra_info) 
    g_string_append(query, "b.usename AS \"Owner\", "
                    "obj_description(a.oid) AS \"Comments\", "
                    "a.oid AS \"SQL\" ");
  else 
    g_string_append(query, "obj_description(a.oid) AS \"Comments\" ");

  g_string_append(query, "FROM pg_aggregate a, pg_type t, pg_user b "
		  "WHERE a.aggbasetype = t.oid AND b.usesysid=a.aggowner ");
  if (and_condition)
    {
      g_string_append(query, and_condition->str);
      g_string_free(and_condition, TRUE);
    }

  g_string_append(query, "ORDER BY aggname, typname");		  

  cmd->cmd = query->str;

  /* execute the command */
  if (cmd->cmd != 0)
    {
      recset = gda_postgres_cmd_execute(cmd, error, 0, &affected, 0);
      fprintf(stderr, "Nb of tables: %ld\n", affected);
    }
  else recset = 0;
  g_string_free(query, FALSE);
  if (recset && extra_info)
    {
      Gda_POSTGRES_Recordset_Replacement *repl;

      repl = (Gda_POSTGRES_Recordset_Replacement *) 
	g_malloc(sizeof(Gda_POSTGRES_Recordset_Replacement));
      repl->colnum = 5;
      repl->newtype = GDA_TypeVarchar;
      repl->trans_func = replace_AGGREGATE_OID_with_SQL;
      recset->replacements = g_slist_append(recset->replacements,
					    repl);
    }
  
  gda_postgres_cmd_free(cmd);
  return (recset);
}


static Gda_POSTGRES_Recordset *
schema_tables (Gda_POSTGRES_Error *error,
               Gda_POSTGRES_Connection *cnc,
               GDA_Connection_Constraint *constraint,
               gint length)
{
  GString*                   query;
  Gda_POSTGRES_Command*      cmd;
  Gda_POSTGRES_Recordset*    recset = 0;
  GDA_Connection_Constraint* ptr;
  gboolean                   extra_info = FALSE;
  gint                       cnt;
  GString*                   and_condition = 0;
  gulong                     affected;

  /* process constraints */
  ptr = constraint;
  for (cnt = 0; cnt < length && ptr != 0; cnt++)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_EXTRA_INFO :
          extra_info = TRUE;
          break;
        case GDA_Connection_OBJECT_NAME :
          if (!and_condition) and_condition = g_string_new("");
          g_string_sprintfa(and_condition, "AND a.relname='%s' ", ptr->value);
          fprintf(stderr, "schema_tables: table name = '%s'\n", ptr->value);
          break;
        case GDA_Connection_OBJECT_SCHEMA :
          if (!and_condition) and_condition = g_string_new("");
          g_string_sprintfa(and_condition, "AND b.usename='%s' ", ptr->value);
          fprintf(stderr, "schema_tables: table_schema = '%s'\n", ptr->value);
          break;
        case GDA_Connection_OBJECT_CATALOG :
          fprintf(stderr, "schema_procedures: proc_catalog = '%s' UNUSED!\n", 
		  ptr->value);
          break;
        default :
          fprintf(stderr, "schema_tables: invalid constraint type %d\n", 
		  ptr->ctype);
          g_string_free(and_condition, TRUE);
          return (0);
        }
      ptr++;
    }

  /* build the command object */
  cmd = gda_postgres_cmd_new();
  cmd->cnc = cnc;

  /* build the query */
  query = g_string_new("SELECT a.relname AS \"Name\", ");
  if (extra_info) 
    g_string_append(query, "b.usename AS \"Owner\", "
                    "obj_description(a.oid) AS \"Comments\", "
                    "a.relname AS \"SQL\" ");
  else 
    g_string_append(query, "obj_description(a.oid) AS \"Comments\" ");
  g_string_append(query, " FROM pg_class a, pg_user b "
                  "WHERE ( relkind = 'r') and relname !~ '^pg_' "
                  "AND relname !~ '^xin[vx][0-9]+' AND "
                  "b.usesysid = a.relowner ");
  if (and_condition) 
    g_string_sprintfa(query, "%s", and_condition->str);
 
  g_string_append(query, " ORDER BY a.relname");
  if (and_condition) g_string_free(and_condition, TRUE);
  cmd->cmd = query->str;

  /* execute the command */
  if (cmd->cmd != 0)
    {
      recset = gda_postgres_cmd_execute(cmd, error, 0, &affected, 0);
      fprintf(stderr, "Nb of tables: %ld\n", affected);
    }
  else recset = 0;
  g_string_free(query, FALSE);
  if (recset && extra_info)
    {
      Gda_POSTGRES_Recordset_Replacement *repl;
      repl = (Gda_POSTGRES_Recordset_Replacement *) 
	g_malloc(sizeof(Gda_POSTGRES_Recordset_Replacement));
      repl->colnum = 3;
      repl->newtype = GDA_TypeVarchar;
      repl->trans_func = replace_TABLE_NAME_with_SQL;
      recset->replacements = g_slist_append(recset->replacements,
					    repl);
    }
  
  gda_postgres_cmd_free(cmd);
  return (recset);
}

static Gda_POSTGRES_Recordset *
schema_tab_parents (Gda_POSTGRES_Error *error,
		    Gda_POSTGRES_Connection *cnc,
		    GDA_Connection_Constraint *constraint,
		    gint length)
{
  GString*                   query;
  Gda_POSTGRES_Command*      cmd;
  Gda_POSTGRES_Recordset*    recset = 0;
  GDA_Connection_Constraint* ptr;
  gint                       cnt, oid;
  gchar*                     table_name=NULL;
  PGresult*                  res;
  gulong                     affected;

  /* process constraints */
  ptr = constraint;
  for (cnt = 0; cnt < length && ptr != 0; cnt++)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_OBJECT_CATALOG : /* not used in postgres */
	  break;
        case GDA_Connection_OBJECT_NAME :
          table_name = ptr->value;
          fprintf(stderr, "schema_tab_parents: table_name = '%s'\n", 
		  table_name);
          break;
        default:
          fprintf(stderr, "schema_tab_parents: invalid constraint type %d\n", 
		  ptr->ctype);
          return (0);
        }
      ptr++;
    }

  if (!table_name)
    {
      fprintf(stderr, "schema_tab_parents: table name is null\n");
      return (0);
    }

  /* build the command object */
  cmd = gda_postgres_cmd_new();
  cmd->cnc = cnc;

  /* find the oid of the table */
  query = g_string_new("SELECT oid from pg_class where ");
  g_string_sprintfa(query, "relname = '%s'", table_name);
  res = PQexec(cnc->pq_conn, query->str);
  g_string_free(query, TRUE);
  if (!res || (PQresultStatus(res) != PGRES_TUPLES_OK) ||
      (PQntuples(res) < 1))
    {
      if (res)
	PQclear(res);
      return NULL;
    }
  oid = atoi(PQgetvalue(res, 0, 0));
  PQclear(res);

  /* build the query */
  query = g_string_new("SELECT a.relname AS \"Table Name\", "
		       "b.inhseqno AS \"Sequence\" FROM "
		       "pg_inherits b, pg_class a WHERE a.oid=b.inhparent "
		       "AND b.inhrel = ");
  g_string_sprintfa(query, "%d order by b.inhseqno", oid);
  cmd->cmd = query->str;
  g_string_free(query, FALSE);

  /* execute the command */
  if (cmd->cmd != 0)
    {
      recset = gda_postgres_cmd_execute(cmd, error, 0, &affected, 0);
      fprintf(stderr, "Nb of tab parents: %ld\n", affected);
    }
  else recset = 0;
  
  gda_postgres_cmd_free(cmd);
  return (recset);
}


static Gda_POSTGRES_Recordset *
schema_sequences (Gda_POSTGRES_Error *error,
		  Gda_POSTGRES_Connection *cnc,
		  GDA_Connection_Constraint *constraint,
		  gint length)
{
  GString*                   query;
  Gda_POSTGRES_Command*      cmd;
  Gda_POSTGRES_Recordset*    recset = 0;
  GDA_Connection_Constraint* ptr;
  gboolean                   extra_info = FALSE;
  gint                       cnt;
  GString*                   and_condition = 0;
  gulong                     affected;

  /* process constraints */
  ptr = constraint;
  for (cnt = 0; cnt < length && ptr != 0; cnt++)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_EXTRA_INFO :
          extra_info = TRUE;
          break;
        case GDA_Connection_OBJECT_CATALOG : /* not used in postgres */
	  break;
        case GDA_Connection_OBJECT_NAME :
          if (!and_condition) and_condition = g_string_new("");
          g_string_sprintfa(and_condition, "AND a.relname='%s' ", ptr->value);
          fprintf(stderr, "schema_sequences: seq name = '%s'\n", ptr->value);
          break;
        case GDA_Connection_OBJECT_SCHEMA :
          if (!and_condition) and_condition = g_string_new("");
          g_string_sprintfa(and_condition, "AND b.usename='%s' ", ptr->value);
          fprintf(stderr, "schema_sequences: seq schema = '%s'\n", ptr->value);
          break;
        default :
          fprintf(stderr, "schema_sequences: invalid constraint type %d\n", 
		  ptr->ctype);
          g_string_free(and_condition, TRUE);
          return (0);
        }
      ptr++;
    }

  /* build the command object */
  cmd = gda_postgres_cmd_new();
  cmd->cnc = cnc;

  /* build the query */
  query = g_string_new("SELECT a.relname AS \"Name\", ");
  if (extra_info) 
    g_string_append(query, "b.usename AS \"Owner\", "
                    "obj_description(a.oid) AS \"Comments\", "
                    "a.relname AS \"SQL\" ");
  else 
    g_string_append(query, "obj_description(a.oid) AS \"Comments\" ");
  g_string_append(query, " FROM pg_class a, pg_user b "
                  "WHERE ( relkind = 'S') and relname !~ '^pg_' "
                  "AND relname !~ '^xin[vx][0-9]+' AND "
                  "b.usesysid = a.relowner ");
  if (and_condition) 
    g_string_sprintfa(query, "%s", and_condition->str);
 
  g_string_append(query, " ORDER BY a.relname");
  if (and_condition) g_string_free(and_condition, TRUE);
  cmd->cmd = query->str;

  /* execute the command */
  if (cmd->cmd != 0)
    {
      recset = gda_postgres_cmd_execute(cmd, error, 0, &affected, 0);
      fprintf(stderr, "Nb of sequences: %ld\n", affected);
    }
  else recset = 0;
  g_string_free(query, FALSE);
  if (recset && extra_info)
    {
      Gda_POSTGRES_Recordset_Replacement *repl;
      repl = (Gda_POSTGRES_Recordset_Replacement *) 
	g_malloc(sizeof(Gda_POSTGRES_Recordset_Replacement));
      repl->colnum = 3;
      repl->newtype = GDA_TypeVarchar;
      repl->trans_func = replace_SEQUENCE_NAME_with_SQL;
      recset->replacements = g_slist_append(recset->replacements,
					    repl);
    }
  
  gda_postgres_cmd_free(cmd);
  return (recset);
}


static Gda_POSTGRES_Recordset *
schema_types (Gda_POSTGRES_Error *error,
              Gda_POSTGRES_Connection *cnc,
              GDA_Connection_Constraint *constraint,
              gint length)
{
  gboolean                   extra_info = FALSE;
  gint                       cnt;
  GDA_Connection_Constraint* ptr;
  Gda_POSTGRES_Command*      cmd;
  Gda_POSTGRES_Recordset*    recset;
  GString *query=NULL, *and_condition=NULL;
  gulong                     affected;
#define IN_PG_TYPES_ALLOWED "('abstime', 'bool', 'box', 'bpchar', 'bytea', 'char', 'cidr', 'circle', 'date', 'datetime', 'filename', 'float4', 'float8', 'inet', 'int2', 'int4', 'int8', 'line', 'lseg', 'macaddr', 'money', 'name', 'numeric', 'path', 'point', 'polygon', 'reltime', 'text', 'time', 'timespan', 'timestamp', 'tinterval', 'varchar')"

  /* process constraints */
  ptr = constraint;
  for (cnt = 0; cnt < length && ptr != 0; cnt++)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_OBJECT_NAME :
          if (!and_condition) and_condition = g_string_new("");
          g_string_sprintfa(and_condition, "AND a.typname='%s' ", ptr->value);
          fprintf(stderr, "schema_tables: type name = '%s'\n", ptr->value);
          break;
        case GDA_Connection_OBJECT_SCHEMA :
          if (!and_condition) and_condition = g_string_new("");
          g_string_sprintfa(and_condition, "AND b.usename='%s' ", ptr->value);
          fprintf(stderr, "schema_tables: owner     = '%s'\n", ptr->value);
          break;
        case GDA_Connection_OBJECT_CATALOG : /* not used in postgres */
	  break;
        case GDA_Connection_EXTRA_INFO :
          extra_info = TRUE;
          break;
        default :
          fprintf(stderr, "schema_types: invalid constraint type %d\n", ptr->ctype);
          return (0);
        }
      ptr++;
    }

  /* build command object */
  cmd = gda_postgres_cmd_new();
  gda_postgres_cmd_set_connection(cmd, cnc);
  cmd->cmd = 0;

  /* build the query */
  if (extra_info)
    {
      query = g_string_new("SELECT a.typname AS \"Name\", b.usename "
			   "AS \"Owner\", "
			   "obj_description(a.oid) AS \"Comments\", "
			   "a.oid AS \"Gda Type\", "
			   "a.oid as \"Local Type\" "
			   "FROM pg_type a, pg_user b WHERE typrelid = 0 "
			   "AND a.typowner=b.usesysid ");
    }
  else
    {
      query = g_string_new("SELECT a.typname AS \"Name\", "
			   "obj_description(a.oid) AS \"Comments\" "
			   "FROM pg_type a, pg_user b "
			   "WHERE typrelid = 0 ");
    }
  if (and_condition)
    {
      g_string_sprintfa(query, "%s", and_condition->str);
      g_string_free(and_condition, TRUE);
    }
  g_string_append(query, "AND (typname in " IN_PG_TYPES_ALLOWED 
		  " OR (usename != 'postgres' AND b.usesysid=a.typowner)) ");
  g_string_append(query, /*"AND obj_description(a.oid) is not null "*/
		  "ORDER BY typname");
  cmd->cmd = query->str;
  g_string_free(query, FALSE);
  if (cmd->cmd != 0)
    {
      recset = gda_postgres_cmd_execute(cmd, error, 0, &affected, 0);
      fprintf(stderr, "Nb of types: %ld\n", affected);
      gda_postgres_cmd_free(cmd);
      /* completion with gda types if extra_info */
      if (extra_info)
	{
	  Gda_POSTGRES_Recordset_Replacement *repl;
	  repl = (Gda_POSTGRES_Recordset_Replacement *) 
	    g_malloc(sizeof(Gda_POSTGRES_Recordset_Replacement));
	  repl->colnum = 3;
	  repl->newtype = GDA_TypeSmallint;
	  repl->trans_func = replace_PROV_TYPES_with_gdatype;
	  recset->replacements = g_slist_append(recset->replacements,
						repl);
	}
      return (recset);
    }
  gda_postgres_cmd_free(cmd);
  return (0);
}

static Gda_POSTGRES_Recordset *
schema_views (Gda_POSTGRES_Error *error,
              Gda_POSTGRES_Connection *cnc,
              GDA_Connection_Constraint *constraint,
              gint length)
{
  gchar*                     query = 0;
  gchar*                     view_name = 0;
  gint cnt;
  Gda_POSTGRES_Command*      cmd;
  Gda_POSTGRES_Recordset*    recset = 0;
  GDA_Connection_Constraint* ptr;
  gboolean                   extra_info = FALSE;
  gulong                     affected;

  /* process constraints */
  ptr = constraint;
  for (cnt = 0; cnt < length && ptr != 0; cnt++)
    {
      switch (ptr->ctype)
        {
        case GDA_Connection_EXTRA_INFO :
          extra_info = TRUE;
          break;
        case GDA_Connection_OBJECT_NAME :
          view_name = ptr->value;
          fprintf(stderr, "schema_views: view name = '%s'\n", view_name);
          break;
        default :
          fprintf(stderr, "schema_views: invalid constraint type %d\n", ptr->ctype);
          return (0);
        }
      ptr++;
    }

  /* build the command object */
  cmd = gda_postgres_cmd_new();
  cmd->cnc = cnc;
  
  /* build the query */
  if (extra_info)
    {
      gchar *where_clause;

      if (view_name != 0)
        {
          where_clause = g_strdup_printf("WHERE viewname = '%s'", view_name);
        }
      else where_clause = g_strdup(" ");
      query = g_strdup_printf("SELECT viewname AS \"Name\", "
                              "       viewowner AS \"Owner\", "
                              "       obj_description(oid) AS \"Comments\", "
                              "       definition AS \"SQL\" "
                              " FROM pg_views "
                              " %s "
                              " ORDER BY viewname", where_clause);
      g_free((gpointer) where_clause);
    }
  else
    {
      query = g_strdup("SELECT viewname AS \"Name\", "
                       "       null AS \"Comments\" "
                       " FROM pg_views "
                       " ORDER BY viewname");
    }
  cmd->cmd = query;

  /* execute the command */
  if (cmd->cmd != 0)
    {
      recset = gda_postgres_cmd_execute(cmd, error, 0, &affected, 0);
      fprintf(stderr, "Nb of views: %ld\n", affected);
    }
  else recset = 0;

  gda_postgres_cmd_free(cmd);
  return (recset);
}

static void
initialize_schema_ops (void)
{
  schema_ops[GDA_Connection_GDCN_SCHEMA_TABLES] = schema_tables;
  schema_ops[GDA_Connection_GDCN_SCHEMA_COLS] = schema_columns;
  schema_ops[GDA_Connection_GDCN_SCHEMA_PROCS] = schema_procedures;
  schema_ops[GDA_Connection_GDCN_SCHEMA_PROC_PARAMS] = schema_proc_params;
  schema_ops[GDA_Connection_GDCN_SCHEMA_AGGREGATES] = schema_aggregates;
  schema_ops[GDA_Connection_GDCN_SCHEMA_PROV_TYPES] = schema_types;
  schema_ops[GDA_Connection_GDCN_SCHEMA_VIEWS] = schema_views;
  schema_ops[GDA_Connection_GDCN_SCHEMA_SEQUENCES] = schema_sequences;
  schema_ops[GDA_Connection_GDCN_SCHEMA_TAB_PARENTS] = schema_tab_parents;
}

/* create new connection object */
Gda_POSTGRES_Connection *
gda_postgres_connection_new (void)
{
  static gint initialized = 0;
  Gda_POSTGRES_Connection *c = g_new0(Gda_POSTGRES_Connection, 1);
  if (!initialized)
    {
      initialize_schema_ops();
      initialized = 1;
    }

  /* initialize everything to 0 to avoid core dumped */
  c->pq_host = 0;
  c->pq_port = 0;
  c->pq_options = 0;
  c->pq_tty = 0;
  c->pq_db = 0;
  c->pq_login = 0;
  c->pq_pwd = 0;
  c->pq_conn = 0;
  c->errors = 0;
  c->types_array = NULL;
  return (c);
}

/* free given connection object */
void
gda_postgres_connection_free (Gda_POSTGRES_Connection *c)
{
  g_return_if_fail(c != 0);
  gda_postgres_connection_close(c);
  if (c->pq_host != 0) g_free((gpointer) c->pq_host);
  if (c->pq_port != 0) g_free((gpointer) c->pq_port);
  if (c->pq_options != 0) g_free((gpointer) c->pq_options);
  if (c->pq_tty != 0) g_free((gpointer) c->pq_tty);
  if (c->pq_db != 0) g_free((gpointer) c->pq_db);
  if (c->pq_login != 0) g_free((gpointer) c->pq_login);
  if (c->pq_pwd != 0) g_free((gpointer) c->pq_pwd);
  g_free((gpointer) c);
}

/* open new connection to database server */
static gchar*
get_value(gchar* ptr)
{
  while (*ptr && *ptr != '=')
    ptr++;
  if (!*ptr)
    return 0;
  ptr++;
  if (!*ptr)
    return 0;
  while (*ptr && isspace(*ptr))
    ptr++;
  return (g_strdup(ptr));
}

gint
gda_postgres_connection_open (Gda_POSTGRES_Connection *cnc, const gchar *dsn,
			      const gchar *user, const gchar *password)
{
  gchar *ptr_s, *ptr_e;
  PGresult *res;
  gint i, j, cmp, length;
  gboolean found;
  GSList *list;
  Gda_connection_data *data=NULL;

  fprintf(stderr, "DSN: %s\n", dsn);
  /* get options from connection string */
  ptr_s = (gchar *) dsn;
  while (ptr_s && *ptr_s)
    {
      ptr_e = strchr(ptr_s, ';');
      if (ptr_e) *ptr_e = '\0';
      if (strcasecmp(ptr_s, "HOST") == 0)
        cnc->pq_host = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "DATABASE", strlen("DATABASE")) == 0)
        cnc->pq_db = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PORT", strlen("PORT")) == 0)
        cnc->pq_port = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "OPTIONS", strlen("OPTIONS")) == 0)
        cnc->pq_options = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "TTY", strlen("TTY")) == 0)
        cnc->pq_tty = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "LOGIN", strlen("LOGIN")) == 0)
        cnc->pq_login = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PASSWORD", strlen("PASSWORD")) == 0)
        cnc->pq_pwd = get_value(ptr_s);
      ptr_s = ptr_e;
      if (ptr_s)
        ptr_s++;
    }
  if (!cnc->pq_login) 
    cnc->pq_login = user != 0 ? g_strdup(user) : 0;
  if (!cnc->pq_pwd)
    cnc->pq_pwd = password != 0 ? g_strdup(password) : 0;
  /* actually establish the connection */
  cnc->pq_conn = PQsetdbLogin(cnc->pq_host, cnc->pq_port, cnc->pq_options,
                              cnc->pq_tty, cnc->pq_db, cnc->pq_login, cnc->pq_pwd);
  if (PQstatus(cnc->pq_conn) != CONNECTION_OK)
    {
      Gda_POSTGRES_Error* error;
      error = gda_postgres_error_new();
      gda_postgres_error_make(error, 0, cnc, "gda_postgres_connection_open");
      return (-1);
    }

  /*
   * is there already a types_array filled (same dsn, same user? 
   */
  found = FALSE;
  list = global_connection_data_list;
  while (list && !found)
    {
      data = (Gda_connection_data *)(list->data);
      if (!strcmp(data->dsn, dsn) && !strcmp(data->user, user))
	found = TRUE;
      else
	list = g_slist_next(list);
    }
  if (found)
    {
      /* use the types_array previously defined */
      cnc->types_array = data->types_array;
      data->nreg++;
    }
  else 
    {
      /* 
       * now loads the correspondance between postgres 
       * data types and GDA types 
       */
      cnc->types_array = g_malloc(sizeof(types_array));
      memcpy(cnc->types_array, types_array, sizeof(types_array));
      res = PQexec(cnc->pq_conn,
		   "SELECT typname, oid FROM pg_type "
		   "WHERE typrelid = 0 ORDER BY typname");
      if (!res || (PQresultStatus(res) != PGRES_TUPLES_OK))
	{
	  Gda_POSTGRES_Error* error;
	  error = gda_postgres_error_new();
	  gda_postgres_error_make(error, 0, cnc, 
				  "gda_postgres_connection_open");
	  if (res) 
	    PQclear(res);
	  return (-1);
	}
      length = PQntuples(res);
      for (i=0; i<Gda_POSTGRES_Types_Array_Nb; i++) {
	j = 0;
	found = FALSE;
	while ((j<length) && !found) 
	  {
	    cmp = strcmp(PQgetvalue(res, j, 0), 
			 cnc->types_array[i].postgres_type); 
	    if (!cmp) 
	      {
		cnc->types_array[i].oid = atoi(PQgetvalue(res, j, 1));
		fprintf(stderr, "Postgres type %s found, oid=%ld!\n",  
			cnc->types_array[i].postgres_type, 
			cnc->types_array[i].oid);
		found = TRUE;
	      }
	    else
	      if (cmp > 0) 
		j = length; /* so we leave the while */
	    j++;
	  }
	if (!found) 
	  fprintf(stderr, "Postgres type %s not found!\n",  
		  cnc->types_array[i].postgres_type);
      }
      PQclear(res);
      data = (Gda_connection_data *) g_malloc(sizeof(Gda_connection_data));
      data->dsn = g_strdup(dsn);
      data->user = g_strdup(user);
      data->nreg = 1;
      data->types_array = cnc->types_array;
      global_connection_data_list = 
	g_slist_append(global_connection_data_list, data);
    }

  /*
   * Sets the DATE format for all the current session to DD-MM-YYYY
   */
  res = PQexec(cnc->pq_conn, "SET DATESTYLE TO 'SQL, US'");
  PQclear(res);
  /*
   * the TIMEZONE is left to what is the default, without trying to impose
   * one. Otherwise the command would be:
   * "SET TIME ZONE '???'" or "SET TIME ZONE DEFAULT"
   */

  return (0);
}

static gboolean
gda_postgres_connection_is_type_known (Gda_POSTGRES_Connection *cnc, 
				       gulong sql_type)
{
  gboolean found;
  gint i;

  found = FALSE;
  i = 0;
  while ((i< Gda_POSTGRES_Types_Array_Nb) && !found) 
    {
      if (cnc->types_array[i].oid == sql_type) 
	found = TRUE;
      i++;
    }
  return found;
}

GDA_ValueType           
gda_postgres_connection_get_gda_type (Gda_POSTGRES_Connection *cnc, 
				      gulong sql_type)
{
  GDA_ValueType gda_type = GDA_TypeVarchar;
  gboolean found;
  gint i;

  found = FALSE;
  i = 0;
  while ((i< Gda_POSTGRES_Types_Array_Nb) && !found) 
    {
      if (cnc->types_array[i].oid == sql_type) 
	{
	  gda_type = cnc->types_array[i].gda_type;
	  found = TRUE;
	}
      i++;
    }
  return gda_type;
}

Gda_POSTGRES_CType     
gda_postgres_connection_get_c_type (Gda_POSTGRES_Connection *cnc, 
				    gulong sql_type)
{
  Gda_POSTGRES_CType c_type = SQL_C_CHAR;
  gboolean found;
  gint i;

  found = FALSE;
  i = 0;
  while ((i< Gda_POSTGRES_Types_Array_Nb) && !found) 
    {
      if (cnc->types_array[i].oid == sql_type) 
	{
	  c_type = cnc->types_array[i].c_type;
	  found = TRUE;
	}
      i++;
    }
  return c_type;
}

Gda_POSTGRES_CType     
gda_postgres_connection_get_c_type_from_gda (Gda_POSTGRES_Connection *cnc, 
					     GDA_ValueType gda_type)
{
  Gda_POSTGRES_CType c_type = SQL_C_CHAR;
  gboolean found;
  gint i;

  found = FALSE;
  i = 0;
  while ((i< Gda_POSTGRES_Types_Array_Nb) && !found) 
    {
      if (cnc->types_array[i].gda_type == gda_type) 
	{
	  c_type = cnc->types_array[i].c_type;
	  found = TRUE;
	}
      i++;
    }
  return c_type;
}

gulong
gda_postgres_connection_get_sql_type(Gda_POSTGRES_Connection *cnc, 
				     gchar *postgres_type)
{
  gulong oid = 0;
  gboolean found;
  gint i;

  found = FALSE;
  i = 0;
  while ((i< Gda_POSTGRES_Types_Array_Nb) && !found) 
    {
      if (!strcmp(cnc->types_array[i].postgres_type, postgres_type)) 
	{
	  oid = cnc->types_array[i].oid;
	  found = TRUE;
	}
      i++;
    }

  return oid;
}

gchar* gda_postgres_connection_get_type_name(Gda_POSTGRES_Connection *cnc, 
					     gulong oid)
{
  gchar* str=NULL;
  gboolean found;
  gint i;

  found = FALSE;
  i = 0;
  while ((i< Gda_POSTGRES_Types_Array_Nb) && !found) 
    {
      if (cnc->types_array[i].oid == oid) 
	{
	  str = cnc->types_array[i].postgres_type;
	  found = TRUE;
	}
      i++;
    }

  return str;
}

/* schemas */
Gda_POSTGRES_Recordset *
gda_postgres_connection_open_schema (Gda_POSTGRES_Connection *cnc,
                                     Gda_POSTGRES_Error *e,
                                     GDA_Connection_QType t,
                                     GDA_Connection_Constraint *constraints,
                                     gint length)
{
  schema_ops_fn fn = schema_ops[(gint) t];

  if (fn)
    return fn(e, cnc, constraints, length);
  else
    g_log("GDA POSTGRES", G_LOG_LEVEL_INFO,
	  "schema_open_schema: Unhandled SCHEMA_QTYPE %d\n", (gint) t);
  return 0;
}

gboolean
gda_postgres_connection_supports (Gda_POSTGRES_Connection* cnc,
				  GDA_Connection_Feature feature)
{
  gboolean retval;

  fprintf(stderr, "You are asking for feature %d\n", feature);
  switch (feature)
    {
    case GDA_Connection_FEATURE_TRANSACTIONS:
    case GDA_Connection_FEATURE_SEQUENCES:
    case GDA_Connection_FEATURE_PROCS:
    case GDA_Connection_FEATURE_INHERITANCE:
      retval = TRUE;
      break;

    default :
      retval = FALSE;
    }

  return retval;
}

/* close given connection to database */
gint
gda_postgres_connection_close (Gda_POSTGRES_Connection *c)
{
  gboolean found;
  GSList *list;
  Gda_connection_data *data=NULL;

  if (c->types_array)
    {
      list = global_connection_data_list;
      found = FALSE;
      while (list && !found)
	{
	  data = (Gda_connection_data*)(list->data);
	  if (c->types_array == data->types_array)
	      found = TRUE;
	  else
	    list = g_slist_next(list);
	}
      if (found)
	{ 
	  if (data->nreg == 1)
	    {
	      g_free(data->dsn);
	      g_free(data->user);
	      g_free(data->types_array);
	      global_connection_data_list = 
		g_slist_remove_link(global_connection_data_list, list);
	      g_free(data);
	    }
	  else /* some other connections are using it */
	    {
	      data->nreg--;
	    }
	}
      else /* types_array not in list if there was an error while loading */
	g_free(c->types_array);
      c->types_array = NULL;
    }
  /* check connection status */
  if (PQstatus(c->pq_conn) == CONNECTION_OK)
    {
      PQfinish(c->pq_conn);
      c->pq_conn = 0;
      return (0);
    }
  gda_postgres_error_make(0, 0, c, "gda_postgres_connection_close");
  return (-1);
}

/* BEGIN TRANS */
gint
gda_postgres_connection_begin_transaction (Gda_POSTGRES_Connection *cnc)
{
  return (execute_command(cnc, "BEGIN"));
}

/* COMMIT */
gint
gda_postgres_connection_commit_transaction (Gda_POSTGRES_Connection *cnc)
{
  return (execute_command(cnc, "COMMIT"));
}

/* ROLLBACK */
gint
gda_postgres_connection_rollback_transaction (Gda_POSTGRES_Connection *cnc)
{
  return (execute_command(cnc, "ROLLBACK"));
}

/* logging */
gint
gda_postgres_connection_start_logging (Gda_POSTGRES_Connection *cnc, gchar *filename)
{
  FILE *f;
  gchar *str;
  
  /* check parameters */
  g_return_val_if_fail(cnc != 0, -1);
  if (PQstatus(cnc->pq_conn) != CONNECTION_OK)
    return (-1);
    
  /* open given file or default one if 0 */
  str = filename ? filename : g_strdup_printf("%s/gda.log", g_get_home_dir());
  if (!(f = fopen(str, "w")))
    {
      if (str != filename)
        g_free(str);
      return (-1);
    }
  PQtrace(cnc->pq_conn, f);

  /* free memory */
  if (str != filename)
    g_free(str);
  return (0);
}

gint
gda_postgres_connection_stop_logging (Gda_POSTGRES_Connection *cnc)
{
  /* check parameters */
  g_return_val_if_fail(cnc != 0, -1);
  if (PQstatus(cnc->pq_conn) != CONNECTION_OK)
    return (-1);
  PQuntrace(cnc->pq_conn);
  return (0);
}

gchar *
gda_postgres_connection_create_recset (Gda_POSTGRES_Connection *cnc,
                                       GDA_RowAttributes *columns)
{
  return (0);
}
