/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"
#include <unistd.h>

static void add_environ_cb      (GtkButton *button, gpointer data);
static void clear_text_cb       (GtkWidget *w, gpointer data);
static void connect_on_entry_cb (GtkCheckButton *button, gpointer data);
static void current_environ_cb  (GtkButton *button, gpointer data);
static void dialog_clicked_cb   (GnomeDialog * dialog, gint button_number, gpointer data);
static void font_set_cb         (GnomeFontPicker *fontpicker, gchar *font_name, gpointer user_data);
static void remove_environ_cb   (GtkButton *button, gpointer data);
static void save_as_cb          (GtkWidget *w, gpointer data);
static void save_config_cb      (GtkWidget *w, gpointer data);
static void save_on_exit_cb     (GtkCheckButton *button, gpointer data);
static void set_language_cb     (GtkEntry *entry, gpointer data);
static void set_shell_cb        (GtkEntry *entry, gpointer data);
static void set_tabs_cb         (GtkToggleButton *button, gpointer data);

static GtkWidget* config_create_tab (void);
static GtkWidget* environ_create_tab (void);
static GtkWidget* general_create_tab (void);

char **environ = 0;
static GtkWidget* l_dialog = 0;
static GtkWidget* l_notebook = 0;
static struct
{
  GtkWidget* (*create_tab)(void);
  gchar*     title;
} l_tabs[] =
{
  { general_create_tab, N_("General") },
  { environ_create_tab, N_("Environment") },
  { config_create_tab, N_("Config File") }
};

/*
 * Private functions
 */
static GtkWidget *
config_create_tab (void)
{
  GtkWidget *table, *scroll, *text, *box, *button;
  gchar *str;
  
  table = gtk_table_new(3, 3, FALSE);
  
  /* add editor */
  scroll = gda_ui_new_scrolled_window_widget();
  text = fe_new_text_widget();
  str = g_strdup_printf("%s/.gnome/gdafe", g_get_home_dir());
  fe_load_file_in_text(GTK_TEXT(text), str);
  g_free((gpointer) str);
  gtk_container_add(GTK_CONTAINER(scroll), text);
  gtk_table_attach(GTK_TABLE(table), scroll, 0, 2, 0, 3,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
                   
  /* add buttons */
  box = gtk_vbox_new(TRUE, 0);
  button = gda_ui_new_button_widget_with_pixmap(_("Save"), GNOME_STOCK_MENU_SAVE);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(save_config_cb), (gpointer) text);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  button = gda_ui_new_button_widget_with_pixmap(_("Save As"), GNOME_STOCK_MENU_SAVE_AS);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(save_as_cb), (gpointer) text);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  button = gda_ui_new_button_widget_with_pixmap(_("Clear"), GNOME_STOCK_MENU_TRASH);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(clear_text_cb), (gpointer) text);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 2, 3, 0, 1,
                   GTK_SHRINK,
                   GTK_SHRINK,
                   3, 3);
                   
  gtk_widget_show(table);
  return (table);
}

static GtkWidget *
environ_create_tab (void)
{
  GtkWidget *table, *scroll, *list, *button, *box;
  gpointer iterator;
  gchar *name, *value;
  static const gchar *cols[] = { N_("Variable"), N_("Value") };
  
  table = gtk_table_new(4, 3, FALSE);
  
  /* add list of environment variables */
  scroll = gda_ui_new_scrolled_window_widget();
  list = gda_ui_new_clist_widget((gchar **) cols, sizeof(cols) / sizeof(cols[0]));
  iterator = gnome_config_init_iterator(FE_CONFIG_SECTION_ENVIRONMENT);
  while ((iterator = gnome_config_iterator_next(iterator, &name, &value)))
    {
      gchar *row[] = { name, value };
      gtk_clist_append(GTK_CLIST(list), row);
    }
  gtk_container_add(GTK_CONTAINER(scroll), list);
  gtk_table_attach(GTK_TABLE(table), scroll, 0, 2, 0, 4,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
                   
  /* add buttons */
  box = gtk_vbox_new(TRUE, 0);
  button = gda_ui_new_button_widget_with_pixmap(_("Add"), GNOME_STOCK_MENU_NEW);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(add_environ_cb), (gpointer) list);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  button = gda_ui_new_button_widget_with_pixmap(_("Remove"), GNOME_STOCK_MENU_TRASH);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(remove_environ_cb), (gpointer) list);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  button = gda_ui_new_button_widget_with_pixmap(_("Current"), GNOME_STOCK_MENU_JUMP_TO);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(current_environ_cb), 0);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 2, 3, 0, 1,
                   GTK_SHRINK,
                   GTK_SHRINK,
                   3, 3);

  gtk_widget_show(table);
  return (table);
}

static GtkWidget *
general_create_tab (void)
{
  GtkWidget *table, *label, *box, *entry, *button;
  int i;
  
  table = gtk_table_new(7, 2, FALSE); // era 5, 4
  
  /* on/off options */
  box = gtk_vbox_new(FALSE, 0);
  button = gda_ui_new_check_button_widget(_("Save options on exit"),
                                          gnome_config_get_int(FE_CONFIG_GENERAL_SAVE_ON_EXIT));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(save_on_exit_cb), 0);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  
  button = gda_ui_new_check_button_widget(_("Open connection dialog on entry"),
                                          gnome_config_get_int(FE_CONFIG_GENERAL_CONNECT_ON_ENTRY));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(connect_on_entry_cb), 0);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 0, 2, 0, 1,
                   GTK_FILL | GTK_EXPAND,
                   GTK_FILL | GTK_EXPAND,
                   3, 3);

  /* Fonts */
  label = gda_ui_new_label_widget(_("Editor Font"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
                   GTK_FILL,
                   GTK_FILL, 3, 3);
  box = gtk_hbox_new(FALSE, 0);
  entry = gda_ui_new_entry_widget(0, FALSE);
  gtk_entry_set_text(GTK_ENTRY(entry),
                     gnome_config_get_string(FE_CONFIG_GENERAL_EDITOR_FONT));
  gtk_object_set_data(GTK_OBJECT(entry), "FE_Config_Font", FE_CONFIG_GENERAL_EDITOR_FONT);
  gtk_box_pack_start(GTK_BOX(box), entry, 0, 0, 0);
  button = gnome_font_picker_new();
  gtk_signal_connect(GTK_OBJECT(button), "font-set",
                     GTK_SIGNAL_FUNC(font_set_cb), (gpointer) entry);
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 1, 2, 1, 2,
                   GTK_FILL,
                   GTK_FILL, 3, 3);
                   
  label = gda_ui_new_label_widget(_("Grid Font"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
                   GTK_FILL,
                   GTK_FILL, 3, 3);
  box = gtk_hbox_new(FALSE, 0);
  entry = gda_ui_new_entry_widget(0, FALSE);
  gtk_entry_set_text(GTK_ENTRY(entry),
                     gnome_config_get_string(FE_CONFIG_GENERAL_GRID_FONT));
  gtk_object_set_data(GTK_OBJECT(entry), "FE_Config_Font", FE_CONFIG_GENERAL_GRID_FONT);
  gtk_box_pack_start(GTK_BOX(box), entry, 0, 0, 0);
  button = gnome_font_picker_new();
  gtk_signal_connect(GTK_OBJECT(button), "font-set",
                     GTK_SIGNAL_FUNC(font_set_cb), (gpointer) entry);
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 1, 2, 2, 3,
                   GTK_FILL,
                   GTK_FILL, 3, 3);

  /* language */
  box = gtk_hbox_new(FALSE, 0);
  label = gda_ui_new_label_widget(_("Language"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4,
                   GTK_FILL,
                   GTK_FILL, 3, 3);
  button = gda_ui_new_combo_widget();
  gtk_combo_set_popdown_strings(GTK_COMBO(button), 
                                gnome_i18n_get_language_list("LC_COLLATE"));
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(button)->entry),
                     gnome_config_get_string(FE_CONFIG_GENERAL_LANGUAGE));
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(button)->entry), "changed",
                     GTK_SIGNAL_FUNC(set_language_cb), NULL);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 1, 2, 3, 4,
                   GTK_FILL,
                   GTK_FILL,
                   3, 3);
                   
  /* shell */
  label = gda_ui_new_label_widget(_("Shell"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5,
                   GTK_FILL,
                   GTK_FILL, 3, 3);
  entry = gda_ui_new_file_entry_widget("FE_Config_shell_history");
  gtk_entry_set_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry))),
                     gnome_config_get_string(FE_CONFIG_GENERAL_SHELL));
  gtk_signal_connect(GTK_OBJECT(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry))),
                     "changed",
                     GTK_SIGNAL_FUNC(set_shell_cb), 0);
  gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 4, 5,
                   GTK_FILL,
                   GTK_FILL, 3, 3);
             
  /* window tabs */
  label = gda_ui_new_label_widget(_("Tabs position"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6,
                   GTK_FILL,
                   GTK_FILL, 3, 3);

  i = gnome_config_get_int(FE_CONFIG_DOCUMENT_TABS);
  box = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(box);
  button = gda_ui_new_radio_button_widget(_("Top"), 0);
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
                     GTK_SIGNAL_FUNC(set_tabs_cb),
                     GINT_TO_POINTER(GTK_POS_TOP));
  if (i == GTK_POS_TOP)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    }
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  button = gda_ui_new_radio_button_widget(_("Bottom"), GTK_RADIO_BUTTON(button));
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
                     GTK_SIGNAL_FUNC(set_tabs_cb),
                     GINT_TO_POINTER(GTK_POS_BOTTOM));
  if (i == GTK_POS_BOTTOM)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    }
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  button = gda_ui_new_radio_button_widget(_("Left"), GTK_RADIO_BUTTON(button));
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
                     GTK_SIGNAL_FUNC(set_tabs_cb),
                     GINT_TO_POINTER(GTK_POS_LEFT));
  if (i == GTK_POS_LEFT)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    }
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  button = gda_ui_new_radio_button_widget(_("Right"), GTK_RADIO_BUTTON(button));
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
                     GTK_SIGNAL_FUNC(set_tabs_cb),
                     GINT_TO_POINTER(GTK_POS_RIGHT));
  if (i == GTK_POS_RIGHT)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    }
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);

  gtk_table_attach(GTK_TABLE(table), box, 1, 2, 5, 6,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   3, 3);
                   
  gtk_widget_show(table);
  return (table);
}

static void
real_close (void)
{
  if (l_dialog != 0)
    {
      gnome_dialog_close(GNOME_DIALOG(l_dialog));
      l_dialog = 0;
    }
}

/*
 * Callbacks
 */
static void
add_environ_cb (GtkButton *button, gpointer data)
{
  GtkWidget *dialog, *table, *label, *name_entry, *value_entry;
  GtkWidget *list = GTK_WIDGET(data);
  
  g_return_if_fail(GTK_IS_CLIST(list));
  
  /* create the dialog box */
  dialog = gnome_dialog_new(_("Add Environment Variable"),
                            GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL,
                            0);
                            
  table = gtk_table_new(2, 2, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, GNOME_PAD);
  
  /* create fields */
  label = gda_ui_new_label_widget(_("Name"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  name_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), name_entry, 1, 2, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  label = gda_ui_new_label_widget(_("Value"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  value_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), value_entry, 1, 2, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
                            
  /* run the dialog box */
  if (gnome_dialog_run(GNOME_DIALOG(dialog)) == 0)
    {
      gchar *str = g_strdup_printf("%s/%s", FE_CONFIG_SECTION_ENVIRONMENT,
                                   gtk_entry_get_text(GTK_ENTRY(name_entry)));
      gchar *row[2];
      gnome_config_set_string(str, gtk_entry_get_text(GTK_ENTRY(value_entry)));
      g_free((gpointer) str);
      
      /* add row to variable list */
      row[0] = gtk_entry_get_text(GTK_ENTRY(name_entry));
      row[1] = gtk_entry_get_text(GTK_ENTRY(value_entry));
      gtk_clist_append(GTK_CLIST(list), row);
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
clear_text_cb (GtkWidget *w, gpointer data)
{
  GtkWidget *text = (GtkWidget *) data;
  
  g_return_if_fail(GTK_IS_TEXT(text));
  gtk_editable_delete_text(GTK_EDITABLE(text), 0, -1);
}

static void
connect_on_entry_cb (GtkCheckButton *button, gpointer data)
{
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
    {
      gnome_config_set_int(FE_CONFIG_GENERAL_CONNECT_ON_ENTRY, 1);
    }
  else
    {
      gnome_config_set_int(FE_CONFIG_GENERAL_CONNECT_ON_ENTRY, 0);
    }
}

static void
current_environ_cb (GtkButton *button, gpointer data)
{
  GtkWidget *dialog, *list, *scroll;
  const gchar *cols[] = { N_("Variable"), N_("Value") };
  gchar *row[2];
  gchar** ptr;
  register gint cnt;
  
  /* create dialog */
  dialog = gnome_dialog_new(_("Current Environment"), GNOME_STOCK_BUTTON_CLOSE, 0);
  gtk_widget_set_usize(dialog, 450, 300);
  
  scroll = gda_ui_new_scrolled_window_widget();
  list = gda_ui_new_clist_widget((gchar **) cols, 2);
  gtk_container_add(GTK_CONTAINER(scroll), list);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scroll, 1, 1, GNOME_PAD);
  
  /* fill environment list */
  gtk_clist_freeze(GTK_CLIST(list));
  for (ptr = environ; ptr; ptr++)
    {
      gchar* empty_string = "";

      row[0] = ptr ? ptr : empty_string;
      row[1] = g_getenv(ptr);
      if (!row[1]) row[1] = empty_string;

      gtk_clist_append(GTK_CLIST(list), (const gchar **) row);
    }
  gtk_clist_thaw(GTK_CLIST(list));

  /* run dialog and finish */
  gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
}

static void
dialog_clicked_cb (GnomeDialog * dialog, gint button_number, gpointer data)
{
  switch (button_number)
    {
    case 0 : /* OK button */
      gnome_config_sync();
      fe_load_settings(0, 0);
      real_close();
      break;
    case 1 : /* Apply button */
      gnome_config_sync();
      fe_load_settings(0, 0);
      break;
    case 2 : /* Close */
      gnome_config_drop_all();
      real_close();
      break;
    }
}

static void
font_set_cb (GnomeFontPicker *fontpicker, gchar *font_name, gpointer user_data)
{
  if (GTK_IS_ENTRY(user_data))
    {
      gchar *font = gtk_object_get_data(GTK_OBJECT(user_data), "FE_Config_Font");
      gtk_entry_set_text(GTK_ENTRY(user_data), font_name);
      
      if (font)
        {
          gnome_config_set_string(font, font_name);
        }
    }
}

static void
remove_environ_cb (GtkButton *button, gpointer data)
{
  GList *sel;
  GtkWidget *list = GTK_WIDGET(data);
  
  g_return_if_fail(GTK_IS_CLIST(list));
  
  sel = GTK_CLIST(list)->selection;
  if (sel != 0)
    {
      gchar *txt = 0;
      gint row = GPOINTER_TO_INT(sel->data);
      gtk_clist_get_text(GTK_CLIST(list), row, 0, &txt);
      if (txt != 0)
        {
          gchar *str = g_strdup_printf("%s/%s", FE_CONFIG_SECTION_ENVIRONMENT,
                                       txt);
          gnome_config_clean_key(str);
          g_free((gpointer) str);
          gtk_clist_remove(GTK_CLIST(list), row);
        }
    }
}

static void
save_as_cb (GtkWidget *w, gpointer data)
{
  GtkWidget *text = (GtkWidget *) data;
  gchar *str;
  
  g_return_if_fail(GTK_IS_TEXT(text));

  str = fe_get_file_name(_("Save As..."));
  if (str)
    {
      fe_save_text_to_file(GTK_TEXT(text), str);
      g_free((gpointer) str);
    }
}

static void
save_config_cb (GtkWidget *w, gpointer data)
{
  GtkWidget *text = (GtkWidget *) data;
  gchar *str;
  
  g_return_if_fail(GTK_IS_TEXT(text));

  str = g_strdup_printf("%s/.gnome/gdafe", g_get_home_dir());
  fe_save_text_to_file(GTK_TEXT(text), str);
  g_free((gpointer) str);
}

static void
save_on_exit_cb (GtkCheckButton *button, gpointer data)
{
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
    {
      gnome_config_set_int(FE_CONFIG_GENERAL_SAVE_ON_EXIT, 1);
    }
  else
    {
      gnome_config_set_int(FE_CONFIG_GENERAL_SAVE_ON_EXIT, 0);
    }
}

static void
set_language_cb (GtkEntry *entry, gpointer data)
{
  gchar *empty = "";
  gchar *str = gtk_entry_get_text(entry);
  if (!str) str = empty;
  gnome_config_set_string(FE_CONFIG_GENERAL_LANGUAGE, str);
}

static void
set_shell_cb (GtkEntry *entry, gpointer data)
{
  gchar *empty = "";
  gchar *str = gtk_entry_get_text(entry);
  if (!str) str = empty;
  gnome_config_set_string(FE_CONFIG_GENERAL_SHELL, str);
}

static void
set_tabs_cb (GtkToggleButton *button, gpointer data)
{
  if (gtk_toggle_button_get_active(button))
    {
      gnome_config_set_int(FE_CONFIG_DOCUMENT_TABS, GPOINTER_TO_INT(data));
      gtk_notebook_set_tab_pos(GTK_NOTEBOOK(glb_WorkArea), GPOINTER_TO_INT(data));
    }
}

/*
 * Public functions
 */
void
fe_open_config (GtkWidget *w, gpointer data)
{
  GtkWidget *notebook;
  register gint cnt;
  if (l_dialog == 0)
    {
      l_dialog = gnome_dialog_new(_("Preferences"),
                                  GNOME_STOCK_BUTTON_OK,
                                  GNOME_STOCK_BUTTON_APPLY,
                                  GNOME_STOCK_BUTTON_CLOSE,
                                  NULL);
      gtk_signal_connect(GTK_OBJECT(l_dialog), "clicked",
                         GTK_SIGNAL_FUNC(dialog_clicked_cb), NULL);
      l_notebook = gda_ui_new_notebook_widget();
      gtk_widget_show(l_notebook);
      
      /* add tabs */
      for (cnt = 0; cnt < sizeof(l_tabs) / sizeof(l_tabs[0]); cnt++)
        {
          GtkWidget *tab = (*l_tabs[cnt].create_tab)();
          gchar *title = l_tabs[cnt].title;
          gtk_notebook_append_page(GTK_NOTEBOOK(l_notebook), tab,
                                gtk_label_new(title));
        }
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(l_dialog)->vbox), l_notebook, 1, 1, GNOME_PAD);
    }
  gtk_widget_show(l_dialog);
}
