/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

typedef struct
{
  FE_ConnectionInfo* cnc_info;
  GtkWidget*         tree_widget;
  GtkWidget*         tables_tree_item;
  GtkWidget*         tables_tree;
  GtkWidget*         views_tree_item;
  GtkWidget*         views_tree;
  GtkWidget*         file_entry;
  GtkWidget*         name_entry;
  GtkWidget*         export_data_button;
  GtkWidget*         show_file_button;
  GtkWidget*         extra_info_button;
  GtkWidget*         log_window;
} FE_ExportInfo;

static void clear_export_cb   (GtkWidget *w, gpointer data);
static void close_export_cb   (GtkWidget *w, gpointer data);
static void export_to_xml_cb  (GtkWidget *w, gpointer data);
static void refresh_export_cb (GtkWidget *w, gpointer data);

static GnomeUIInfo exporttoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("To XML"), N_("Export selected data to XML file"),
    export_to_xml_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
    NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("To SQL"), N_("Export selected data as a SQL script"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_NONE,
    GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh data"),
    refresh_export_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear all widgets"),
    clear_export_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close this window"),
    close_export_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
fill_subtree (GtkTree *subtree, Gda_Connection *cnc, GDA_Connection_QType t)
{
  Gda_Recordset* recset;
  GList*         children;

  g_return_if_fail(GTK_IS_TREE(subtree));
  g_return_if_fail(IS_GDA_CONNECTION(cnc));
  g_return_if_fail(gda_connection_is_open(cnc));

  /* first clear subtree */
  children = gtk_container_children(GTK_CONTAINER(subtree));
  gtk_tree_remove_items(subtree, children);
  g_list_free(children);
  
  /* FIXME: must show objects belonging only to the current user */
  recset = gda_connection_open_schema(cnc, t, GDA_Connection_no_CONSTRAINT, 0);
  if (recset)
    {
      gulong position = gda_recordset_move(recset, 1, 0);
      while (position != GDA_RECORDSET_INVALID_POSITION &&
             !gda_recordset_eof(recset))
        {
          GtkWidget* tree_item;
          Gda_Field* f = gda_recordset_field_idx(recset, 0);
          
          tree_item = gtk_tree_item_new_with_label(gda_stringify_value(0, 0, f));
          gtk_widget_show(tree_item);
          gtk_tree_append(subtree, tree_item);
          position = gda_recordset_move(recset, 1, 0);
        }

      /* free the recordset */
      gda_recordset_free(recset);
    }
}

static void
real_close (GtkWidget *w, gpointer data)
{
  FE_ExportInfo* exp_info;

  g_return_if_fail(GTK_IS_WIDGET(data));

  exp_info = (FE_ExportInfo *) fe_get_view_data(GTK_WIDGET(data), "FE_ExportInfo");
  if (exp_info)
    {
      g_free((gpointer) exp_info);
    }
}

/*
 * Callbacks
 */
static void
clear_export_cb (GtkWidget *w, gpointer data)
{
  FE_ExportInfo* exp_info;
  GtkWidget*     view = fe_get_current_view();

  if (view && (exp_info = fe_get_view_data(view, "FE_ExportInfo")))
    {
    }
}

static void
close_connection_cb (Gda_Connection *cnc, GtkWidget *view)
{
  g_return_if_fail(IS_GDA_CONNECTION(cnc));
  g_return_if_fail(GTK_IS_WIDGET(view));
  
  if (view && fe_get_view_data(view, "FE_ExportInfo"))
    fe_destroy_view(view);
}

static void
close_export_cb (GtkWidget *w, gpointer data)
{
  GtkWidget *view = fe_get_current_view();
  if (view && fe_get_view_data(view, "FE_ExportInfo"))
    {
      fe_destroy_view(view);
    }
}

static void
export_to_xml_cb (GtkWidget *w, gpointer data)
{
  FE_ExportInfo* exp_info;
  GtkWidget*     view = fe_get_current_view();

  if (view && (exp_info = fe_get_view_data(view, "FE_ExportInfo")))
    {
      GList*   selection;
      gchar*   filename;
      gchar*   dbname;
      gboolean export_data;

      /* check that all needed data is provided */
      filename = gtk_entry_get_text(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(exp_info->file_entry)));
      if (!filename || !strlen(filename))
        {
          gda_ui_show_error(_("You have to provide a file name"));
          return;
        }
      dbname = gtk_entry_get_text(GTK_ENTRY(exp_info->name_entry));
      if (!dbname || !strlen(dbname))
        {
          gda_ui_show_error(_("You have to specify a name for the database export"));
          return;
        }

      /* export selected objects */
      selection = GTK_TREE(exp_info->tree_widget)->selection;
      if (selection)
        {
          Gda_XmlDatabase* xmldb;

          export_data = gtk_toggle_button_get_active(
                            GTK_TOGGLE_BUTTON(exp_info->export_data_button));
          xmldb = gda_xml_database_new(dbname);
          do
            {
              if (GTK_IS_TREE_ITEM(selection->data))
                {
                  gchar *name = 0;

                  gtk_label_get(GTK_LABEL(GTK_BIN(selection->data)->child), &name);
                  fe_status_bar_message(_("Exporting object named '%s'"), name);
                  if (name)
                    {
                      g_warning("object name = %s", name);
                      if (GTK_TREE_ITEM_SUBTREE(selection->data)
                          == exp_info->tables_tree)
                        {
                          Gda_Recordset* recset;
                          gulong         reccount;
                          gchar*         sql = g_strdup_printf("select * from %s", name);
                          
                          recset = gda_connection_execute(exp_info->cnc_info->cnc,
                                                          sql,
                                                          &reccount,
                                                          0);
                          gda_xml_table_new_from_gda_recordset(xmldb, name, recset, export_data);
                          g_free((gpointer) sql);
                        }
                      else if (GTK_TREE_ITEM_SUBTREE(selection->data)
                               == exp_info->views_tree)
                        {
                          /* it's a view */
                          g_warning("%s is a view", name);
                        }
                    }
                }
              selection = g_list_next(selection);
            } while (selection);

          /* save XML file */
          fe_status_bar_message(_("Saving to '%s' ..."), filename);
          if (!gda_xml_database_save(xmldb, filename))
            gda_ui_show_error(_("Error saving XML file\n'%s'"), filename);
          gda_xml_database_free(xmldb);

          /* show the generated file if the user wants to */
          if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(exp_info->show_file_button)))
            {
            }
        }
      else gda_ui_show_error(_("No objects are selected"));
    }
}

static void
refresh_export_cb (GtkWidget *w, gpointer data)
{
  FE_ExportInfo* exp_info;
  GtkWidget*     view = fe_get_current_view();

  if (view && (exp_info = fe_get_view_data(view, "FE_ExportInfo")))
    {
      /* fill in each subtree */
      fill_subtree(GTK_TREE(exp_info->tables_tree),
                   exp_info->cnc_info->cnc,
                   GDA_Connection_GDCN_SCHEMA_TABLES);
      fill_subtree(GTK_TREE(exp_info->views_tree),
                   exp_info->cnc_info->cnc,
                   GDA_Connection_GDCN_SCHEMA_VIEWS);
    }
}

/*
 * Public functions
 */
void
fe_open_export (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo* cnc_info = fe_get_current_connection();
  
  if (cnc_info != 0)
    {
      GtkWidget*     view;
      GtkWidget*     scroll;
      GtkWidget*     label;
      GtkWidget*     table;
      GtkWidget*     frame;
      FE_ExportInfo* exp_info;
      gchar*         str;

      /* create the internal strcture */
      exp_info = g_new0(FE_ExportInfo, 1);
      exp_info->cnc_info = cnc_info;

      /* create the view window */
      str = g_strdup_printf(_("Export: %s@%s"), gda_connection_get_user(cnc_info->cnc),
                            cnc_info->dsn);
      view = fe_new_view(str, 4, 6, exporttoolbar);
      fe_set_view_data(view, "FE_ConnectionInfo", (gpointer) cnc_info);
      g_free((gpointer) str);

      /* create widgets */
      scroll = gda_ui_new_scrolled_window_widget();
      fe_add_widget_to_view(view, 0, 1, 2, 6, scroll);
      exp_info->tree_widget = gda_ui_new_tree_widget();
      gtk_tree_set_selection_mode(GTK_TREE(exp_info->tree_widget),
                                  GTK_SELECTION_MULTIPLE);
      gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll),
                                            exp_info->tree_widget);

      exp_info->tables_tree_item = gtk_tree_item_new_with_label(_("TABLES"));
      gtk_widget_show(exp_info->tables_tree_item);
      gtk_tree_append(GTK_TREE(exp_info->tree_widget), exp_info->tables_tree_item);
      exp_info->tables_tree = gda_ui_new_tree_widget();
      gtk_tree_set_selection_mode(GTK_TREE(exp_info->tables_tree), GTK_SELECTION_MULTIPLE);
      gtk_tree_set_view_mode(GTK_TREE(exp_info->tables_tree), GTK_TREE_VIEW_ITEM);
      gtk_tree_item_set_subtree(GTK_TREE_ITEM(exp_info->tables_tree_item),
                                exp_info->tables_tree);
      gtk_tree_item_expand(GTK_TREE_ITEM(exp_info->tables_tree_item));

      exp_info->views_tree_item = gtk_tree_item_new_with_label(_("VIEWS"));
      gtk_widget_show(exp_info->views_tree_item);
      gtk_tree_append(GTK_TREE(exp_info->tree_widget), exp_info->views_tree_item);
      exp_info->views_tree = gda_ui_new_tree_widget();
      gtk_tree_set_selection_mode(GTK_TREE(exp_info->views_tree), GTK_SELECTION_MULTIPLE);
      gtk_tree_set_view_mode(GTK_TREE(exp_info->views_tree), GTK_TREE_VIEW_ITEM);
      gtk_tree_item_set_subtree(GTK_TREE_ITEM(exp_info->views_tree_item),
                                exp_info->views_tree);
      gtk_tree_item_expand(GTK_TREE_ITEM(exp_info->views_tree_item));

      frame = gda_ui_new_frame_widget(_("Properties"));
      table = gtk_table_new(8, 2, FALSE);
      gtk_container_set_border_width(GTK_CONTAINER(table), 3);
      gtk_widget_show(table);
      gtk_container_add(GTK_CONTAINER(frame), table);
      fe_add_widget_to_view(view, 2, 1, 3, 2, frame);

      label = gda_ui_new_label_widget(_("File name"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
      exp_info->file_entry = gda_ui_new_file_entry_widget("FE_Export_history");
      gtk_table_attach(GTK_TABLE(table), exp_info->file_entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);

      label = gda_ui_new_label_widget(_("Name"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
      exp_info->name_entry = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), exp_info->name_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);

      exp_info->export_data_button = gda_ui_new_check_button_widget(_("Export data"), TRUE);
      gtk_table_attach(GTK_TABLE(table), exp_info->export_data_button, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 0, 0);

      exp_info->show_file_button = gda_ui_new_check_button_widget(_("Show file when done"), FALSE);
      gtk_table_attach(GTK_TABLE(table), exp_info->show_file_button, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 0, 0);

      exp_info->extra_info_button = gda_ui_new_check_button_widget(_("Add object extra info"), TRUE);
      gtk_table_attach(GTK_TABLE(table), exp_info->extra_info_button, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 0, 0);

      scroll = gda_ui_new_scrolled_window_widget();
      fe_add_widget_to_view(view, 2, 2, 4, 6, scroll);
      exp_info->log_window = fe_new_text_widget();
      gtk_text_set_editable(GTK_TEXT(exp_info->log_window), FALSE);
      gtk_container_add(GTK_CONTAINER(scroll), exp_info->log_window);

      /* connect to Gda_Connection's close signal */
      gtk_signal_connect(GTK_OBJECT(cnc_info->cnc), "close",
                         GTK_SIGNAL_FUNC(close_connection_cb), (gpointer) view);
      fe_set_view_destroy_func(view, (FE_DestroyViewFunc) real_close, (gpointer) view);
      fe_set_view_data(view, "FE_ExportInfo", (gpointer) exp_info);
      fe_display_view(view);

      refresh_export_cb(NULL, NULL);
    }
  else fe_status_bar_message(_("No connection open"));
}