/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

static void dialog_clicked_cb (GnomeDialog *dialog, gint button_number, gpointer data);
static void new_source_cb (GtkWidget *w, gpointer data);
static void remove_source_cb (GtkWidget *w, gpointer data);
static void select_list_cb (GtkWidget *w, gpointer data);

static GtkWidget* l_dialog = NULL;
static GtkWidget* l_list = NULL;
static GtkWidget* l_dsnconf = NULL;
static GList*     l_dsns = NULL;

/*
 * Private functions
 */
static Gda_Dsn *
find_source_by_name (gchar *name)
{
  GList *node = g_list_first(l_dsns);

  g_return_val_if_fail(name != NULL, NULL);
  g_return_val_if_fail(l_dialog != NULL, NULL);
  
  while (node != NULL)
    {
      Gda_Dsn *data = (Gda_Dsn *) node->data;
      if (data != NULL && !strcmp(name, GDA_DSN_GDA_NAME(data)))
        {
          return (data);
        }
      node = g_list_next(node);
    }
  return (NULL);
}

static gchar*
trim_string(gchar* s)
{
  gint length = strlen(s);

  while (length && isspace(s[--length]))
    ;
  s[length+1] = '\0';
  return s;
}

static void
get_data_sources (void)
{
  GList *node;

  /* free old DSN list */
  gda_dsn_free_list(l_dsns);
  gtk_list_clear_items(GTK_LIST(l_list), 0, -1);

  l_dsns = gda_dsn_list();
  node = g_list_first(l_dsns);
  while (node)
    {
      Gda_Dsn *src = (Gda_Dsn *) node->data;

      if (src) fe_add_list_item(GTK_LIST(l_list), GDA_DSN_GDA_NAME(src));
      node = g_list_next(node);
    }
}

static void
real_close (void)
{
  if (l_dialog != NULL)
    {
      if (l_dsns != NULL)
        {
          gda_dsn_free_list(l_dsns);
        }
      l_dsns = NULL;
      gnome_dialog_close(GNOME_DIALOG(l_dialog));
      l_dialog = NULL;
    }
}

static void
show_source_data (Gda_Dsn *data)
{
  const gchar *empty_string = "";
  
  g_return_if_fail(data != NULL);
  g_return_if_fail(l_dialog != NULL);
  
  gnome_db_dsn_config_set_name(GNOME_DB_DSN_CONFIG(l_dsnconf), data->gda_name);
}

/*
 * Callbacks
 */
static void
dialog_clicked_cb (GnomeDialog * dialog, gint button_number, gpointer data)
{
  switch (button_number) 
    {
    case 0: /* OK button */
      //gnome_config_sync();
      gnome_db_dsn_config_save(GNOME_DB_DSN_CONFIG(l_dsnconf));
      real_close();
      break;
    case 1: /* Apply button */
      gnome_db_dsn_config_save(GNOME_DB_DSN_CONFIG(l_dsnconf));
      get_data_sources();
      break;
    case 2: /* Cancel button */
      real_close();
      break;
    default:
      g_assert_not_reached();
    };
}
             
static void
new_source_cb (GtkWidget *w, gpointer data)
{
  const gchar *empty_string = "";
  
  g_return_if_fail(l_dialog != 0);
  
  gnome_db_dsn_config_clear(GNOME_DB_DSN_CONFIG(l_dsnconf));
}

static void
remove_source_cb (GtkWidget *w, gpointer data)
{
  gchar*   src = gnome_db_dsn_config_get_name(GNOME_DB_DSN_CONFIG(l_dsnconf));
  Gda_Dsn* dsn = find_source_by_name(src);
  if (dsn != NULL)
    {
      gnome_db_dsn_config_set_name(GNOME_DB_DSN_CONFIG(l_dsnconf), src);
      gnome_db_dsn_config_remove(GNOME_DB_DSN_CONFIG(l_dsnconf));
      get_data_sources();
      gnome_db_dsn_config_clear(l_dsnconf);
    }
}

static void
select_list_cb (GtkWidget *w, gpointer data)
{
  if (l_dialog != NULL)
    {
      gchar *name = fe_get_list_current_string(GTK_LIST(l_list));
      if (name != NULL)
        {
          Gda_Dsn *data = find_source_by_name(name);
          if (data != NULL)
            show_source_data(data);
        }
    }
}

/*
 * Public functions
 */
void
fe_open_gda_config (GtkWidget *w, gpointer data)
{
  GtkWidget *box, *scrolled_window, *table, *label, *button, *frame;
  GList *node;
  if (l_dialog == NULL)
    {
      l_dialog = gnome_dialog_new(_("GDA Config"),
                                  GNOME_STOCK_BUTTON_OK,
                                  GNOME_STOCK_BUTTON_APPLY,
                                  GNOME_STOCK_BUTTON_CLOSE,
                                  NULL);
      gtk_signal_connect(GTK_OBJECT(l_dialog), "clicked",
                         GTK_SIGNAL_FUNC(dialog_clicked_cb),
                         NULL);

      frame = gda_ui_new_frame_widget(_("Data Sources"));
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(l_dialog)->vbox), frame, 1, 1, GNOME_PAD);
      
      box = gtk_hbox_new(FALSE, 0);
      gtk_container_add(GTK_CONTAINER(frame), box);
      scrolled_window = gda_ui_new_scrolled_window_widget();
      l_list = fe_new_list_widget();
      gtk_signal_connect(GTK_OBJECT(l_list), "selection_changed",
                         GTK_SIGNAL_FUNC(select_list_cb), NULL);
      get_data_sources();
      gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window),
                                            l_list);
      gtk_box_pack_start(GTK_BOX(box), scrolled_window, 1, 1, GNOME_PAD);

      table = gtk_table_new(7, 2, FALSE);
      gtk_widget_show(table);
   
      gtk_box_pack_start(GTK_BOX(box), table, 1, 1, GNOME_PAD);
      gtk_widget_show(box);

      l_dsnconf = gnome_db_dsn_config_new(NULL);
      gtk_widget_show(l_dsnconf);
      gtk_table_attach(GTK_TABLE(table), l_dsnconf, 0, 2, 0, 6,
      		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
      		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
      		       3, 3);

      box = gtk_hbox_new(TRUE, 0);
      gtk_table_attach(GTK_TABLE(table), box, 0, 2, 6, 7,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
      button = gda_ui_new_button_widget_with_pixmap(_("New"), GNOME_STOCK_MENU_NEW);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(new_source_cb), 0);
      gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
      gtk_widget_show(button);
      button = gda_ui_new_button_widget_with_pixmap(_("Delete"), GNOME_STOCK_MENU_TRASH);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(remove_source_cb), 0);
      gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
      gtk_widget_show(button);
      gtk_widget_show(box);
    }
  gtk_widget_set_usize(l_dialog, 450, 350);
  gtk_widget_show(l_dialog);
}





