/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

/*
 * Common widget creation/management functions
 * This file contains lots of utility functions that allow widget creation and
 * management using a common place for it, thus allowing customization of the
 * entire application look and feel
 */

GdkFont *glb_GridFont = 0;
GdkFont *glb_EditorFont = 0;

/*
 * Private functions
 */
static void
browse_button_click_cb (GtkWidget *w, gpointer data)
{
  gchar *filename = fe_get_file_name(_("Select File"));
  if (filename != NULL)
    {
      gtk_entry_set_text(GTK_ENTRY(data), (const gchar *) filename);
    }
}

/*
 * Public functions
 */
GtkWidget *
fe_new_browse_button_widget (GtkEntry *entry)
{
  GtkWidget *button;
  button = gda_ui_new_button_widget("...");
  if (entry != NULL)
    {
      gtk_signal_connect(GTK_OBJECT(button), "clicked", 
                         GTK_SIGNAL_FUNC(browse_button_click_cb), 
                         (gpointer) entry);
    }
  gtk_widget_show(button);
  return (button);
}

GtkWidget *
fe_new_grid_widget (Gda_Recordset *recset)
{
  GtkWidget *grid;

  /* create grid */  
  grid = gnome_db_grid_new(recset);
  
  /* set font and other properties */
  if (glb_GridFont != 0)
    {
      GtkStyle *style = gtk_widget_get_style(grid);
      style->font = glb_GridFont;
      gtk_widget_set_style(grid, style);
    }
    
  gtk_widget_show(grid);
  return (grid);
}

GtkWidget *
fe_new_list_widget (void)
{
  GtkWidget *list;
  list = gtk_list_new();
  gtk_list_set_selection_mode(GTK_LIST(list), GTK_SELECTION_SINGLE);
  gtk_widget_show(list);
  return (list);
}

/*
 * Text editor widget
 */
static void
copy_text_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GTK_IS_TEXT(data));

  gtk_editable_copy_clipboard(GTK_EDITABLE(data));
}

static void
cut_text_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GTK_IS_TEXT(data));

  gtk_editable_cut_clipboard(GTK_EDITABLE(data));
  gtk_editable_changed(GTK_EDITABLE(data));
}

static void
insert_file_cb (GtkWidget *w, gpointer data)
{
  gchar *filename;

  g_return_if_fail(GTK_IS_TEXT(data));

  filename = fe_get_file_name(_("Select File"));
  if (filename != NULL)
    {
      fe_insert_file_into_text(GTK_TEXT(data), filename);
      gtk_editable_changed(GTK_EDITABLE(data));
      g_free(filename);
    }
}

static void
paste_text_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GTK_IS_TEXT(data));

  gtk_editable_paste_clipboard(GTK_EDITABLE(data));
  gtk_editable_changed(GTK_EDITABLE(data));
}

static void
save_to_file_cb (GtkWidget *w, gpointer data)
{
  gchar *filename;

  g_return_if_fail(GTK_IS_TEXT(data));

  filename = fe_get_file_name(_("Select File"));
  if (filename != NULL)
    {
      fe_save_text_to_file(GTK_TEXT(data), filename);
      g_free(filename);
    }
}

static void
select_all_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GTK_IS_TEXT(data));

  gtk_editable_select_region(GTK_EDITABLE(data), 0, -1);
}

GnomeUIInfo textpopupmenu[] =
{
  { GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text to clipboard"),
    cut_text_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text to clipboard"),
    copy_text_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste text from clipboard"),
    paste_text_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PASTE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Select all"), N_("Select all text in buffer"),
    select_all_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
    NULL, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Insert file..."), N_("Insert file into current position"),
    insert_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save to file..."), N_("Save widget contents to file"),
    save_to_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_SAVE_AS, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Print..."), N_("Print buffer"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PRINT, 0, 0, NULL },
  GNOMEUIINFO_END
};

GtkWidget *
fe_new_text_widget (void)
{
  GtkWidget *text, *scroll_bar, *box, *popup_menu;

  /* create text widget */
#ifdef HAVE_GTKEDITOR
  text = gtk_editor_new(NULL, NULL);
#else
  text = gtk_text_new(NULL, NULL);
#endif
  
  /* set font and other properties */
  if (glb_EditorFont != 0)
    {
      GtkStyle *style = gtk_widget_get_style(text);
      style->font = glb_EditorFont;
      gtk_widget_set_style(text, style);
    }
  gtk_text_set_editable(GTK_TEXT(text), TRUE);
  gtk_text_set_word_wrap(GTK_TEXT(text), TRUE);
  gtk_widget_set_sensitive(text, TRUE);
  gtk_widget_show(text);
  /* create popup-menu */
  popup_menu = gda_ui_new_popup_menu(text, textpopupmenu, (gpointer) text);
  return (text);
}

GtkWidget *
fe_new_toggle_button_widget (gchar *label, gint state)
{
  GtkWidget *toggle = label ? gtk_toggle_button_new_with_label(label) : gtk_toggle_button_new();
  gtk_widget_show(toggle);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(toggle), state);
  return (toggle);
}

GtkWidget *
fe_new_toolbar_widget (GtkOrientation orient, GtkToolbarStyle style)
{
  GtkWidget *toolbar = gtk_toolbar_new(orient, style);
  gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar), GTK_RELIEF_NONE);
  gtk_widget_show(toolbar);
  return (toolbar);
}
