/* GNOME DB library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gda-ui.h>

static void gnome_db_log_viewer_class_init (GnomeDbLogViewerClass *klass);
static void gnome_db_log_viewer_init       (GnomeDbLogViewer *viewer);
static void gnome_db_log_viewer_destroy    (GnomeDbLogViewer *viewer);

/*
 * GnomeDbLogViewer signals
 */
enum
{
  GNOME_DB_LOG_VIEWER_LAST_SIGNAL
};

static gint gnome_db_log_viewer_signals[GNOME_DB_LOG_VIEWER_LAST_SIGNAL] = { 0 };

/*
 * GnomeDbLogViewer widget interface
 */
static void
gnome_db_log_viewer_class_init (GnomeDbLogViewerClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);

  object_class->destroy = gnome_db_log_viewer_destroy;
}

static void
gnome_db_log_viewer_init (GnomeDbLogViewer *viewer)
{
  GtkWidget* frame;
  GtkWidget* scroll;
  gchar*     col_titles[] = { N_("Date"), N_("Time"), N_("Message") };

  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));

  frame = gda_ui_new_frame_widget(NULL);
  gtk_box_pack_start(GTK_BOX(viewer), frame, 1, 1, 0);

  scroll = gda_ui_new_scrolled_window_widget();
  gtk_container_add(GTK_CONTAINER(frame), scroll);
  viewer->msg_list = gda_ui_new_clist_widget(col_titles,
					     sizeof(col_titles) / sizeof(col_titles[0]));
  gtk_container_add(GTK_CONTAINER(scroll), viewer->msg_list);
}

GtkType
gnome_db_log_viewer_get_type (void)
{
  static GtkType db_log_viewer_type = 0;
  
  if (!db_log_viewer_type)
    {
      GtkTypeInfo db_log_viewer_info =
      {
        "GnomeDbLogViewer",
        sizeof (GnomeDbLogViewer),
        sizeof (GnomeDbLogViewerClass),
        (GtkClassInitFunc) gnome_db_log_viewer_class_init,
        (GtkObjectInitFunc) gnome_db_log_viewer_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_log_viewer_type = gtk_type_unique(gtk_vbox_get_type(), &db_log_viewer_info);
    }
  return (db_log_viewer_type);
}

/**
 * gnome_db_log_viewer_new
 *
 * Create a new #GnomeDbLogViewer widget, which lets you navigate through
 * the different log files generated by the GNOME-DB processes.
 *
 * Returns: a pointer to the newly created widget, or NULL on error
 */
GtkWidget *
gnome_db_log_viewer_new (void)
{
  GtkWidget* viewer = GTK_WIDGET(gtk_type_new(gnome_db_log_viewer_get_type()));
  return viewer;
}

static void
gnome_db_log_viewer_destroy (GnomeDbLogViewer *viewer)
{
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));
}

/**
 * gnome_db_log_viewer_load_file
 * @viewer: a log viewer widget
 * @filename: file to load
 */
gboolean
gnome_db_log_viewer_load_file (GnomeDbLogViewer *viewer, const gchar *filename)
{
  gpointer section_iterator, item_iterator;
  gchar*   key = NULL;
  gchar*   name = NULL;
  gchar*   value = NULL;
  gchar*   tmp;

  g_return_val_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer), FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);

  gda_ui_clear_clist(GTK_CLIST(viewer->msg_list));
  gtk_clist_freeze(GTK_CLIST(viewer->msg_list));

  tmp = g_strdup_printf("=%s=", filename);
  section_iterator = gnome_config_init_iterator_sections(tmp);
  g_free((gpointer) tmp);
  while ((section_iterator = gnome_config_iterator_next(section_iterator, &key, &name)))
    {
      tmp = g_strdup_printf("=%s=/%s", filename, key);
      item_iterator = gnome_config_init_iterator(tmp);
      //g_free((gpointer) name);
      while ((item_iterator = gnome_config_iterator_next(item_iterator, &name, &value)))
	{
	  gchar* row[3];
	  row[0] = key;
	  row[1] = name;
	  row[2] = value;
	  gtk_clist_append(GTK_CLIST(viewer->msg_list), row);
	  //g_free((gpointer) name);
	  //g_free((gpointer) value);
	}
      g_free((gpointer) tmp);
      //g_free((gpointer) key);
    }
  gtk_clist_thaw(GTK_CLIST(viewer->msg_list));

  g_free((gpointer) viewer->filename);
  viewer->filename = g_strdup(filename);
  return TRUE;
}

/**
 * gnome_db_log_viewer_refresh
 */
void
gnome_db_log_viewer_refresh (GnomeDbLogViewer *viewer)
{
  gchar* file;

  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));

  file = g_strdup(viewer->filename);
  gnome_db_log_viewer_load_file(viewer, file);
  g_free((gpointer) file);
}

/**
 * gnome_db_log_viewer_clear
 */
void
gnome_db_log_viewer_clear (GnomeDbLogViewer *viewer)
{
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));

  gda_ui_clear_clist(GTK_CLIST(viewer->msg_list));
  g_free((gpointer) viewer->filename);
}
