/* GNOME DB Common Library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-common.h"

static gboolean log_enabled = TRUE;

/*
 * Private functions
 */
static void
write_to_log (const gchar *str, gboolean error)
{
  gchar* msg;

  g_return_if_fail(str != NULL);

  msg = g_strdup_printf("%s%s", error ? _("ERROR: ") : _("MESSAGE: "), str);
  if (log_enabled)
    {
      time_t     t;
      struct tm* now;
      gchar*     config_entry;

      /* retrieve current date */
      t = time(NULL);
      now = localtime(&t);
      if (now)
	{
	  config_entry = g_strdup_printf("=/tmp/%s.log=/%04d-%02d-%02d/%02d:%02d:%02d.%ld",
					 g_get_prgname(),
					 now->tm_year + 1900,
					 now->tm_mon + 1,
					 now->tm_mday,
					 now->tm_hour,
					 now->tm_min,
					 now->tm_sec,
					 clock());
	  gnome_config_set_string(config_entry, msg);
	  gnome_config_sync();
	  g_free((gpointer) config_entry);
	}
    }
  g_warning(msg);
  g_free((gpointer) msg);
}

/**
 * gda_log_enable
 *
 * Enables GDA logs
 */
void
gda_log_enable (void)
{
  log_enabled = TRUE;
}

/**
 * gda_log_disable
 */
void
gda_log_disable (void)
{
  log_enabled = FALSE;
}

/**
 * gda_log_is_enabled
 */
gboolean
gda_log_is_enabled (void)
{
  return log_enabled;
}

/**
 * gda_log_message
 * @format: message string
 *
 * Logs the given message in the GDA log file
 */
void
gda_log_message (const gchar *format, ...)
{
  va_list args;
  gchar   buffer[512];

  g_return_if_fail(format != NULL);

  va_start(args, format);
  vsprintf(buffer, format, args);
  va_end(args);

  write_to_log(buffer, FALSE);
}

/**
 * gda_log_error
 */
void
gda_log_error (const gchar *format, ...)
{
  va_list args;
  gchar   buffer[512];

  g_return_if_fail(format != NULL);

  va_start(args, format);
  vsprintf(buffer, format, args);
  va_end(args);

  write_to_log(buffer, TRUE);
}
