/* GNOME DB Front End
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-fe.h"

static GList* last_commands = NULL;
static gint   max_last_command = 100;

/*
 * Public functions
 */
void
fe_config_add_command (const gchar *cmd)
{
  g_return_if_fail(cmd != NULL);

  last_commands = g_list_prepend(last_commands, (gpointer) g_strdup(cmd));
  if (g_list_length(last_commands) > max_last_command)
    {
      GList* node = g_list_last(last_commands);
      gchar* str = (gchar *) node->data;
      last_commands = g_list_remove(last_commands, (gpointer) str);
      g_free((gpointer) str);
    }
}

GList *
fe_config_get_last_commands (void)
{
  return last_commands;
}

void
fe_config_load (void)
{
  gchar*        name;
  gchar*        value;
  gpointer      iterator;
  register gint cnt;

  /* load list of last commands */
  max_last_command = gnome_config_get_int("/gda-fe/SQLCommands/MaxCommands");
  if (max_last_command <= 0) max_last_command = 100;
  for (cnt = 0; cnt < max_last_command; cnt++)
    {
      gchar* conf;
      gchar* str = g_strdup_printf("/gda-fe/SQLCommands/Command%d", cnt + 1);

      conf = gnome_config_get_string(str);
      if (conf)
	last_commands = g_list_append(last_commands, conf);
      g_free((gpointer) str);
    }

  /* load environment variables */
  iterator = gnome_config_init_iterator("/gda-fe/Environment");
  while ((iterator = gnome_config_iterator_next(iterator, &name, &value)))
    {
      setenv(name, value, 1);
    }
}

void
fe_config_save (void)
{
  GList* node;
  gint   cnt = 1;

  /* save SQL stuff */
  gnome_config_set_int("/gda-fe/SQLCommands/MaxCommands", max_last_command);
  node = g_list_first(last_commands);
  while (node)
    {
      gchar* str;

      if (cnt > max_last_command) break;

      str = g_strdup_printf("/gda-fe/SQLCommands/Command%d", cnt);
      gnome_config_set_string(str, (gchar *) node->data);
      g_free((gpointer) str);

      node = g_list_next(node);
      cnt++;
    }

  /* save UI related configuration */
  gda_ui_save_window_config("MainWindow", gnome_db_container_get_widget(glb_container));

  gnome_config_sync();
}

/*
 * Configuration dialog
 */
static void
add_environ_cb (GtkButton *button, gpointer data)
{
  GtkWidget* dialog;
  GtkWidget* frame;
  GtkWidget* table;
  GtkWidget* label;
  GtkWidget* name_entry;
  GtkWidget* value_entry;
  GtkWidget* clist = GTK_WIDGET(data);

  g_return_if_fail(GTK_IS_CLIST(clist));

  /* create the dialog box */
  dialog = gnome_dialog_new(_("Add Environment Variable"),
			    GNOME_STOCK_BUTTON_OK,
			    GNOME_STOCK_BUTTON_CANCEL,
			    NULL);

  frame = gda_ui_new_frame_widget(NULL);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
  table = gda_ui_new_table_widget(2, 2, FALSE);
  gtk_container_add(GTK_CONTAINER(frame), table);

  /* create fields */
  label = gda_ui_new_label_widget(_("Name"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  name_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), name_entry, 1, 2, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  label = gda_ui_new_label_widget(_("Value"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  value_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), value_entry, 1, 2, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);

  /* run the dialog box */
  if (!gnome_dialog_run(GNOME_DIALOG(dialog)))
    {
      gchar* str = g_strdup_printf("/gda-fe/Environment/%s",
				   gtk_entry_get_text(GTK_ENTRY(name_entry)));
      gchar* row[2];

      gnome_config_set_string(str, gtk_entry_get_text(GTK_ENTRY(value_entry)));
      g_free((gpointer) str);

      /* add row to variable list */
      row[0] = gtk_entry_get_text(GTK_ENTRY(name_entry));
      row[1] = gtk_entry_get_text(GTK_ENTRY(value_entry));
      gtk_clist_append(GTK_CLIST(clist), row);

      setenv(row[0], row[1], 1);
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

static GtkWidget *
environ_create_tab (GnomeDialog *dialog)
{
  GtkWidget*   tab;
  GtkWidget*   scroll;
  GtkWidget*   clist;
  GtkWidget*   button;
  gpointer     iterator;
  gchar*       name;
  gchar*       value;
  const gchar* titles[2] = { N_("Name"), N_("Value") };

  tab = gda_ui_new_table_widget(4, 3, FALSE);

  scroll = gda_ui_new_scrolled_window_widget();
  gtk_table_attach(GTK_TABLE(tab), scroll, 0, 3, 0, 3,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		   0, 0);
  clist = gda_ui_new_clist_widget(titles, 2);
  iterator = gnome_config_init_iterator("/gda-fe/Environment");
  while ((iterator = gnome_config_iterator_next(iterator, &name, &value)))
    {
      gchar *row[] = { name, value };
      gtk_clist_append(GTK_CLIST(clist), row);
    }
  gtk_container_add(GTK_CONTAINER(scroll), clist);

  button = gda_ui_new_button_widget_with_pixmap(_("Add"), GNOME_STOCK_MENU_NEW);
  gtk_signal_connect(GTK_OBJECT(button),
		     "clicked",
		     GTK_SIGNAL_FUNC(add_environ_cb),
		     (gpointer) clist);
  gtk_table_attach(GTK_TABLE(tab), button, 3, 4, 0, 1, GTK_FILL, GTK_SHRINK, 0, 0);

  button = gda_ui_new_button_widget_with_pixmap(_("Remove"), GNOME_STOCK_MENU_TRASH);
  gtk_table_attach(GTK_TABLE(tab), button, 3, 4, 1, 2, GTK_FILL, GTK_SHRINK, 0, 0);

  return tab;
}

static GtkWidget *
gda_create_tab (GnomeDialog *dialog)
{
  GtkWidget* tab;

  tab = gda_ui_new_table_widget(1, 1, FALSE);
  return tab;
}

static GtkWidget *
ui_create_tab (GnomeDialog *dialog)
{
  GtkWidget* tab;
  GtkWidget* label;
  GtkWidget* button;
  GtkWidget* frame;
  GtkWidget* table;
  GtkWidget* menu;
  GtkWidget* menu_item;
  GtkWidget* box;

  tab = gda_ui_new_table_widget(4, 10, FALSE);
  
  label = gda_ui_new_label_widget(_("Editor Font"));
  gtk_table_attach(GTK_TABLE(tab), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  button = gda_ui_new_font_picker_widget();
  gtk_table_attach(GTK_TABLE(tab), button, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

  label = gda_ui_new_label_widget(_("Tab Position"));
  gtk_table_attach(GTK_TABLE(tab), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  button = gda_ui_new_option_menu_widget();
  menu = gtk_menu_new();
  menu_item = gda_ui_new_menu_item_widget(GTK_MENU(menu), _("Top"));
  menu_item = gda_ui_new_menu_item_widget(GTK_MENU(menu), _("Bottom"));
  menu_item = gda_ui_new_menu_item_widget(GTK_MENU(menu), _("Left"));
  menu_item = gda_ui_new_menu_item_widget(GTK_MENU(menu), _("Right"));
  gtk_widget_show(menu);
  gtk_option_menu_set_menu(GTK_OPTION_MENU(button), menu);
  gtk_table_attach(GTK_TABLE(tab), button, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

  frame = gda_ui_new_frame_widget(_("Grid"));
  gtk_table_attach(GTK_TABLE(tab), frame, 0, 4, 2, 3,
		   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		   1, 1);
  table = gda_ui_new_table_widget(4, 4, FALSE);
  gtk_container_add(GTK_CONTAINER(frame), table);
  box = gtk_hbox_new(0, FALSE);
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 0, 4, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  label = gda_ui_new_label_widget(_("Font"));
  gtk_box_pack_start(GTK_BOX(box), label, 0, 0, 0);
  button = gda_ui_new_font_picker_widget();
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  label = gda_ui_new_label_widget(_("Foreground Color"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  button = gda_ui_new_color_picker_widget();
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  label = gda_ui_new_label_widget(_("Background Color"));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  button = gda_ui_new_color_picker_widget();
  gtk_table_attach(GTK_TABLE(table), button, 3, 4, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  
  button = gda_ui_new_check_button_widget(_("Show row numbers"), TRUE);
  gtk_table_attach(GTK_TABLE(table), button, 0, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
  button = gda_ui_new_check_button_widget(_("Show columns"), TRUE);
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

  return tab;
}

void
fe_config_open_dialog (void)
{
  GtkWidget*        dialog;
  GtkWidget*        notebook;
  gint              btn;
  gint              cnt;
  struct
  {
    gchar*     title;
    GtkWidget* (*create_tab_func)(GnomeDialog *);
  } tabs_info[] =
    {
      { N_("User Interface"), ui_create_tab },
      { N_("Environment"), environ_create_tab },
      { N_("GDA"), gda_create_tab },
    };

  /* create dialog */
  dialog = gnome_dialog_new(_("Preferences"),
			    GNOME_STOCK_BUTTON_OK,
			    GNOME_STOCK_BUTTON_CLOSE,
			    NULL);
  gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);

  notebook = gda_ui_new_notebook_widget();
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), notebook, 1, 1, 0);

  for (cnt = 0; cnt < sizeof(tabs_info) / sizeof(tabs_info[0]); cnt++)
    {
      GtkWidget* tab = tabs_info[cnt].create_tab_func(GNOME_DIALOG(dialog));
      if (GTK_IS_WIDGET(tab))
	{
	  GtkWidget* frame;

	  gtk_widget_show(tab);
	  frame = gda_ui_new_frame_widget(NULL);
	  gtk_container_add(GTK_CONTAINER(frame), tab);
	  gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
				   frame,
				   gtk_label_new(tabs_info[cnt].title));
	}
    }

  /* and run the dialog */
  btn = gnome_dialog_run(GNOME_DIALOG(dialog));
  if (btn == 0)
    {
      /* save all changes */
      gnome_config_sync();
    }
  else
    {
      /* discard all changes */
      gnome_config_drop_all();
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}





