/* GDA Manager
 * Copyright (C) 1998,1999 Michael Lausch
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-mgr.h"

static void add_dsn_cb        (GtkWidget *widget, gpointer data);
static void browse_dsn_cb     (GtkWidget *widget, gpointer data);
static void config_dsn_cb     (GtkWidget *widget, gpointer data);
static void edit_dsn_cb       (GtkWidget *widget, gpointer data);
static void remove_dsn_cb     (GtkWidget *widget, gpointer data);
static void view_providers_cb (GtkWidget *widget, gpointer data);

static void view_provider_log_cb (GtkWidget *w, gpointer data);

static GtkWidget* l_ProviderList = 0;
static GtkWidget* l_DsnList = 0;

static GnomeUIInfo gdatoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Add"), N_("Add new data source"),
    add_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Edit"), N_("Edit selected data source"),
    edit_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PREF, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Remove"), N_("Remove selected data source"),
    remove_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Config"), N_("Configure underlying database"),
    config_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_BOOK_RED, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Browse"), N_("Browse selected data source"),
    browse_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PROP, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Providers"), N_("View GDA providers"),
    view_providers_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_HOME, 0, 0, NULL },
  GNOMEUIINFO_END
};
static GnomeUIInfo providerstoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Logs"), N_("View provider log"),
    view_provider_log_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_MAIL, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("New"), N_("Install a new GDA provider"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Edit"), N_("Edit selected provider"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PREF, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Remove"), N_("Remove selected provider"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
fill_dsn_list (GtkCList *clist)
{
  GList* list, *node;
  gint   cnt = 0;
  static GtkWidget* on_pixmap = 0;
  static GtkWidget* off_pixmap = 0;

  g_return_if_fail(GTK_IS_CLIST(clist));

  /* create pixmaps */
  if (!on_pixmap)
    {
      on_pixmap = gnome_stock_pixmap_widget(GTK_WIDGET(clist),
                                            GNOME_STOCK_MENU_BOOK_OPEN);
      if (!on_pixmap) return;
      gtk_widget_show(on_pixmap);
    }
  if (!off_pixmap)
    {
      off_pixmap = gnome_stock_pixmap_widget(GTK_WIDGET(clist),
                                             GNOME_STOCK_MENU_BOOK_GREEN);
      if (!off_pixmap) return;
      gtk_widget_show(off_pixmap);
    }

  /* fill list */
  gda_ui_clear_clist(clist);
  gtk_clist_freeze(clist);
  list = gda_dsn_list();
  node = g_list_first(list);
  while (node)
    {
      Gda_Dsn* dsn = (Gda_Dsn *) node->data;
      if (dsn)
        {
          gchar* row[3];
          gchar* empty_string = "";

          row[0] = GDA_DSN_GDA_NAME(dsn);
          row[1] = GDA_DSN_DESCRIPTION(dsn);
          row[2] = empty_string;

          gtk_clist_append(clist, (const gchar **) row);
          if (GDA_DSN_IS_GLOBAL(dsn))
            {
              gtk_clist_set_pixmap(clist, cnt, 2,
                                   GNOME_PIXMAP(on_pixmap)->pixmap,
                                   GNOME_PIXMAP(on_pixmap)->mask);
            }
          else
            {
              gtk_clist_set_pixmap(clist, cnt, 2,
                                   GNOME_PIXMAP(off_pixmap)->pixmap,
                                   GNOME_PIXMAP(off_pixmap)->mask);
            }
          cnt++;
        }
      node = g_list_next(node);
    }
  gda_dsn_free_list(list);
  gtk_clist_thaw(clist);
}

static void
fill_provider_list (GtkCList *clist)
{
  gchar* empty_string = "";
  gchar* row[4];
  GList* server_list;
  GList* node;

  g_return_if_fail(GTK_IS_CLIST(clist));

  gtk_clist_freeze(clist);
  server_list = gda_server_list();
  node = g_list_first(server_list);
  while (node)
    {
      gchar* type_exe = "exe";
      gchar* type_lib = "shlib";
      gchar* type_relay = "relay";

      Gda_Server* server = (Gda_Server *) node->data;
      if (server)
        {
          row[0] = GDA_SERVER_NAME(server);
          if (GDA_SERVER_GOAD_TYPE(server) == GOAD_SERVER_EXE) row[1] = type_exe;
          else if (GDA_SERVER_GOAD_TYPE(server) == GOAD_SERVER_SHLIB) row[1] = type_lib;
	  else if (GDA_SERVER_GOAD_TYPE(server) == GOAD_SERVER_RELAY) row[1] = type_relay;
          else row[1] = empty_string;
          row[2] = GDA_SERVER_COMMENT(server);
          row[3] = GDA_SERVER_LOCATION(server);

          gtk_clist_append(clist, (const gchar **) row);
        }
      node = g_list_next(node);
    }
  gda_server_free_list(server_list);
  gtk_clist_thaw(clist);
}

static const gchar *
get_current_dsn (void)
{
  GList* selected;

  g_return_val_if_fail(GTK_IS_CLIST(l_DsnList), NULL);

  selected = GTK_CLIST(l_DsnList)->selection;
  if (selected)
    {
      gchar* dsn = 0;

      guint row = GPOINTER_TO_UINT(selected->data);
      gtk_clist_get_text(GTK_CLIST(l_DsnList), row, 0, &dsn);
      return dsn;
    }
  return NULL;
}

static void
manage_dsn (gchar *name)
{
  GtkWidget* dialog;
  GtkWidget* dsnconf;
  gint       btn;

  /* create the dialog */
  dialog = gnome_dialog_new(_("Data Source"),
                            GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL,
                            0);
  gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
  gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(glb_App));

  dsnconf = gnome_db_dsn_config_new(name);
  gtk_widget_show(dsnconf);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), dsnconf, 1, 1, 0);

  btn = gnome_dialog_run(GNOME_DIALOG(dialog));
  if (btn == 0) /* 'OK' button */
    {
      gnome_db_dsn_config_save(dsnconf);
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

/*
 * Callbacks
 */
static void
add_dsn_cb (GtkWidget *widget, gpointer data)
{
  manage_dsn(NULL);
}

static void
browse_dsn_cb (GtkWidget *widget, gpointer data)
{
  GtkWidget* dialog;
  GtkWidget* frame;
  GtkWidget* table;
  GtkWidget* label;
  GtkWidget* user_entry;
  GtkWidget* pwd_entry;
  gint       btn;
  gchar*     dsn_name;

  dsn_name = get_current_dsn();
  if (dsn_name)
    {
      /* create dialog for asking the user name and password */
      dialog = gnome_dialog_new(_("Browse Data Source"),
				GNOME_STOCK_BUTTON_OK,
				GNOME_STOCK_BUTTON_CANCEL,
				NULL);
      gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
      frame = gda_ui_new_frame_widget(NULL);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
      table = gda_ui_new_table_widget(2, 2, FALSE);
      gtk_container_add(GTK_CONTAINER(frame), table);
      
      label = gda_ui_new_label_widget(_("Username"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      user_entry = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), user_entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      label = gda_ui_new_label_widget(_("Password"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
      pwd_entry = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), pwd_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

      btn = gnome_dialog_run(GNOME_DIALOG(dialog));
      if (!btn)
	{
	  gchar*     user;
	  gchar*     pwd;
	  Gda_Dsn*   dsn;
	  GtkWidget* browser_control;

	  user = gtk_entry_get_text(GTK_ENTRY(user_entry));
	  pwd = gtk_entry_get_text(GTK_ENTRY(pwd_entry));
	  g_warning("user %s password = %s", user, pwd);
	  gnome_dialog_close(GNOME_DIALOG(dialog));

	  dsn = gda_dsn_find_by_name(dsn_name);
	  if (dsn)
	    {
	      browser_control = gnome_db_control_widget_new("control:gnome-db-browser");
	      if (GNOME_DB_IS_CONTROL_WIDGET(browser_control))
              //  && gnome_db_control_widget_isok(browser_control))
		{
		  gboolean true = TRUE;

		  gtk_widget_show(browser_control);
		  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(browser_control),
						       GNOME_DB_CONTROL_PROP_DSN,
						       (gconstpointer) GDA_DSN_GDA_NAME(dsn));
		  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(browser_control),
						       GNOME_DB_CONTROL_PROP_IDENT_NAME,
						       (gconstpointer) user);
		  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(browser_control),
						       GNOME_DB_CONTROL_PROP_IDENT_PASS,
						       (gconstpointer) pwd);
		  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(browser_control),
						       GNOME_DB_CONTROL_PROP_START,
						       (gconstpointer) &true);

		  dialog = gnome_dialog_new(dsn_name, GNOME_STOCK_BUTTON_CLOSE, NULL);
		  gtk_widget_set_usize(dialog, 500, 450);
		  frame = gda_ui_new_frame_widget(NULL);
		  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
		  gtk_container_add(GTK_CONTAINER(frame), browser_control);

		  gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
		}
	      else
                {
                  gda_ui_show_error(_("could not load browser control"));
                  gtk_object_destroy(GTK_OBJECT(browser_control));
                }
	      gda_dsn_free(dsn); /* free memory */
	    }
	}
      else gnome_dialog_close(GNOME_DIALOG(dialog));
    }
}

static void
config_dsn_cb (GtkWidget *widget, gpointer data)
{
  gchar* dsn_name = get_current_dsn();
  if (dsn_name)
    {
      GtkWidget* dialog;
      GtkWidget* frame;
      GtkWidget* table;
      GtkWidget* label;
      GtkWidget* entry;
      GtkWidget* user_entry;
      GtkWidget* pwd_entry;
      GtkWidget* users_button;
      GtkWidget* access_button;
      GtkWidget* all_button;
      GtkWidget* sep;

      /* create the dialog */
      dialog = gnome_dialog_new(_("Load config component"),
				GNOME_STOCK_BUTTON_OK,
				GNOME_STOCK_BUTTON_CANCEL,
				NULL);
      frame = gda_ui_new_frame_widget(NULL);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
      table = gda_ui_new_table_widget(2, 4, FALSE);
      gtk_container_add(GTK_CONTAINER(frame), table);

      label = gda_ui_new_label_widget(_("GDA Name"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      gtk_entry_set_text(GTK_ENTRY(entry), dsn_name);

      label = gda_ui_new_label_widget(_("Username"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
      user_entry = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), user_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Password"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
      pwd_entry = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), pwd_entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

      frame = gda_ui_new_frame_widget(_("Module"));
      gtk_table_attach(GTK_TABLE(table), frame, 0, 2, 3, 4,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       3, 3);
      table = gda_ui_new_table_widget(1, 4, FALSE);
      gtk_container_add(GTK_CONTAINER(frame), table);

      users_button = gda_ui_new_radio_button_widget(_("Users"), NULL);
      gtk_table_attach(GTK_TABLE(table), users_button, 0, 1, 0, 1,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND, GTK_FILL, 1, 1);
      access_button = gda_ui_new_radio_button_widget(_("User access rights"),
						     GTK_RADIO_BUTTON(users_button));
      gtk_table_attach(GTK_TABLE(table), access_button, 0, 1, 1, 2,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND, GTK_FILL, 1, 1);
      sep = gtk_hseparator_new();
      gtk_widget_show(sep);
      gtk_table_attach(GTK_TABLE(table), sep, 0, 1, 2, 3,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND, GTK_FILL, 1, 1);
      all_button = gda_ui_new_radio_button_widget(_("All"),
						  GTK_RADIO_BUTTON(users_button));
      gtk_table_attach(GTK_TABLE(table), all_button, 0, 1, 3, 4,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND, GTK_FILL, 1, 1);

      if (!gnome_dialog_run(GNOME_DIALOG(dialog)))
	{
	  gchar*   username;
	  gchar*   password;
	  gchar*   module;
	  Gda_Dsn* dsn;

	  dsn = gda_dsn_find_by_name(dsn_name);
	  if (dsn)
	    {
	      username = gtk_entry_get_text(GTK_ENTRY(user_entry));
	      password = gtk_entry_get_text(GTK_ENTRY(pwd_entry));

	      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(users_button)))
		module = g_strdup_printf("control:%s-users-list", GDA_DSN_PROVIDER(dsn));
	      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(access_button)))
		module = g_strdup_printf("control:%s-users-ac", GDA_DSN_PROVIDER(dsn));
	      else module = g_strdup_printf("control:%s-config", GDA_DSN_PROVIDER(dsn));

	      /* load the component */
	      manager_container_new(module, dsn_name, username, password);

	      g_free((gpointer) module);
	      gda_dsn_free(dsn);
	    }
	}
      gnome_dialog_close(GNOME_DIALOG(dialog));
    }
}

static void
edit_dsn_cb (GtkWidget *widget, gpointer data)
{
  gchar* dsn_name = get_current_dsn();
  if (dsn_name)
    {
      manage_dsn(dsn_name);
    }
}

static void
remove_dsn_cb (GtkWidget *widget, gpointer data)
{
  gchar* dsn_name = get_current_dsn();
  if (dsn_name)
    {
      Gda_Dsn* dsn = gda_dsn_find_by_name(dsn_name);
      if (dsn)
	{
	  gda_dsn_remove(dsn);
	  gda_dsn_free(dsn);
	  fill_dsn_list(GTK_CLIST(l_DsnList));
	}
      else gda_ui_show_error(_("Data source not found in GDA configuration"));
    }
}

static void
view_provider_log_cb (GtkWidget *widget, gpointer data)
{
  GList* selected;

  g_return_if_fail(GTK_IS_CLIST(data));

  selected = GTK_CLIST(data)->selection;
  if (selected)
    {
      gchar* provider = 0;

      guint row = GPOINTER_TO_UINT(selected->data);
      gtk_clist_get_text(GTK_CLIST(data), row, 0, &provider);
      if (provider)
        {
          gchar* filename = g_strdup_printf("/tmp/%s-srv.log", provider);
          gda_ui_show_file(filename);
          g_free((gpointer) filename);
        }
    }
}

static void
view_providers_cb (GtkWidget *widget, gpointer data)
{
  GtkWidget* dialog;
  GtkWidget* table;
  GtkWidget* scroll;
  GtkWidget* clist;
  GtkWidget* frame;
  GtkWidget* toolbar;
  gchar*     col_titles[4] = { N_("Name"), N_("Type"), N_("Comments"), N_("Location") };

  /* create dialog */
  dialog = gnome_dialog_new(_("GDA Providers"),
			    GNOME_STOCK_BUTTON_CLOSE,
			    NULL);
  gtk_widget_set_usize(dialog, 500, 350);
  table = gda_ui_new_table_widget(4, 3, FALSE);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, 0);
  scroll = gda_ui_new_scrolled_window_widget();
  gtk_table_attach(GTK_TABLE(table), scroll, 0, 3, 0, 3,
		   GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		   GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		   3, 3);
  clist = gda_ui_new_clist_widget(col_titles, 4);
  fill_provider_list(GTK_CLIST(clist));
  gtk_container_add(GTK_CONTAINER(scroll), clist);

  toolbar = gda_ui_new_toolbar_widget(GTK_ORIENTATION_VERTICAL,
				      GTK_TOOLBAR_BOTH,
				      providerstoolbar,
				      (gpointer) clist);
  gtk_table_attach(GTK_TABLE(table), toolbar, 3, 4, 0, 3, GTK_FILL, GTK_FILL, 3, 3);
      
  gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
}

/*
 * Public functions
 */
void
manager_gda_init (void)
{
  const gchar* dsn_titles[] = { N_("GDA Name"), N_("Description"), N_("Global") };
  GtkWidget*   frame;
  GtkWidget*   table;
  GtkWidget*   table2;
  GtkWidget*   scroll;
  GtkWidget*   toolbar;

  g_return_if_fail(GTK_IS_NOTEBOOK(glb_Notebook));

  table = gtk_table_new(2, 2, FALSE);
  gtk_widget_show(table);
  gtk_notebook_append_page(GTK_NOTEBOOK(glb_Notebook), table, gtk_label_new(_("GDA Configuration")));

  /* data sources frame */
  frame = gda_ui_new_frame_widget(_("Data Sources"));
  gtk_table_attach(GTK_TABLE(table), frame, 0, 1, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  table2 = gtk_table_new(1, 2, FALSE);
  gtk_widget_show(table2);
  gtk_container_add(GTK_CONTAINER(frame), table2);
  scroll = gda_ui_new_scrolled_window_widget();
  gtk_table_attach(GTK_TABLE(table2), scroll, 0, 1, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  l_DsnList = gda_ui_new_clist_widget(dsn_titles,
                                      sizeof(dsn_titles) / sizeof(dsn_titles[0]));
  gtk_container_add(GTK_CONTAINER(scroll), l_DsnList);
  fill_dsn_list(GTK_CLIST(l_DsnList));

  toolbar = gda_ui_new_toolbar_widget(GTK_ORIENTATION_VERTICAL,
				      GTK_TOOLBAR_BOTH,
				      gdatoolbar,
				      NULL);
  gtk_table_attach(GTK_TABLE(table2), toolbar, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
}
