/* GNOME DB Postgres Config Component
 * Copyright (C) 2000 Rodrigo Moya
 * Copyright (C) 2000 Vivien Malerba
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _MAIN_CONFIG_
#include "gda-postgres-config.h"
#undef _MAIN_CONFIG_
#include "gda-postgres-config-users.h"
#include "gda-postgres-config-ac.h"
#include <gnome-db-control-widget.h>

/*
 * functions called when a component is actually created,
 * called by the gnome-db-control framework
 */
static GtkWidget* create_config_widget_cb (void);
void config_widget_new_instance_cb(GnomeDbControl *control,
				   GnomeDbControlInstance *control_instance,
				   gpointer data);
void config_widget_get_property_cb(GnomeDbControl *control,
				   GnomeDbControlInstance *control_instance,
				   const gchar *name,
				   gpointer data);
void config_widget_set_property_cb(GnomeDbControl *control,
				   GnomeDbControlInstance *control_instance,
				   const gchar *name,
				   gconstpointer value,
				   gpointer data);

/* To destroy the associated data with a control */
void component_widget_destroy_cb(GtkObject *obj, gpointer data)
{
  PostgresPropsInfo *info;
  GSList *list;
  info = gtk_object_get_data(GTK_OBJECT(obj), "propinfo");
  if (info)
    {
      if (info->dsn)
	g_free(info->dsn);
      g_free(info);
    }
  list = g_slist_find(controls_list, info);
  if (list)
    controls_list = g_slist_remove_link(controls_list, list);
  g_print("Object %p is being destroyed now (prop info freed)!!!\n", obj);
}

/*
 * A component should invoke this function wherever something has been 
 * modified which makes that other components should update their display 
 */
static gchar *get_host(gchar *dsn)
{
  gchar *ptr_s, *hold, *ptr_e, *host=NULL;

  ptr_s = hold = g_strdup(dsn);
  while (ptr_s && *ptr_s)
    {
      ptr_e = strchr(ptr_s, ';');
      if (ptr_e) *ptr_e = '\0';
      if (strncasecmp(ptr_s, "HOST", strlen("HOST")) == 0)
        host = get_value(ptr_s);
      ptr_s = ptr_e;
      if (ptr_s)
        ptr_s++;
    }
  g_free(hold);
  return host;
}

void require_components_refresh(PostgresPropsInfo *info)
{
  GSList *list;
  PostgresPropsInfo *dinfo;
  gchar *host1, *host2;

  /* compare only the HOST part and not the entire DSN */
  host1 = get_host(info->dsn);
  if (host1)
    g_print("Components on host %s to be updated\n", host1);
  list = controls_list;
  while (host1 && list)
    {
      dinfo = (PostgresPropsInfo *)(list->data);
      if (dinfo != info)
	{
	  host2 = get_host(dinfo->dsn);
	  if (host2 && !strcmp(host2, host1))
	    dinfo->refresh_component(dinfo);
	  if (host2) 
	    g_free(host2);
	}

      list = g_slist_next(list);
    }
  if (host1)
    g_free(host1);
}


/*
 *
 * Implementation of framework 'callback' functions 
 *
 */

static GtkWidget *create_config_widget_cb (void)
{
  GtkWidget *nb, *wid, *label;
  PostgresPropsInfo *info;


  nb = gtk_notebook_new();

  info = g_new0(PostgresPropsInfo, 1);
  info->ident = TRUE; /* default */
  gtk_object_set_data(GTK_OBJECT(nb), "propinfo", info);

  /* the pages are bonobo components themselves */

  /* users list component */
  wid = gnome_db_control_widget_new("control:gda-postgres-users-list");
  info->u.global.users_list = wid;
  label = gtk_label_new("Users management");
  gtk_notebook_append_page(GTK_NOTEBOOK(nb), wid, label);

  /* users access config component */
  wid = gnome_db_control_widget_new("control:gda-postgres-users-ac");
  info->u.global.users_ac = wid;
  label = gtk_label_new("Users access control");
  gtk_notebook_append_page(GTK_NOTEBOOK(nb), wid, label);


  return nb;
}

void config_widget_new_instance_cb(GnomeDbControl *control,
				   GnomeDbControlInstance *control_instance,
				   gpointer data)
{
  PostgresPropsInfo *info;

  /* fetching the info */
  info = gtk_object_get_data(GTK_OBJECT(control_instance->widget), "propinfo");
  /* info is the control instance's data */
  gnome_db_control_set_instance_data(control_instance, info);
  gnome_db_control_add_property(control_instance,
				"dsn", _("DSN for the connection"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				"ident_name", _("User identity"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				"ident_password", _("User passord"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				"ident_show", _("Show/hide the user id"),
				GNOME_DB_CONTROL_ARG_BOOLEAN, NULL);
  gnome_db_control_add_property(control_instance,"reset", 
				_("Give it TRUE to reset/start the component"),
				GNOME_DB_CONTROL_ARG_BOOLEAN, NULL);
}

void config_widget_get_property_cb(GnomeDbControl *control,
				   GnomeDbControlInstance *control_instance,
				   const gchar *name,
				   gpointer data)
{
  PostgresPropsInfo *info;
  gboolean done=FALSE;

  info = gnome_db_control_get_instance_data(control_instance);
  if (!strcmp(name, "dsn"))
    {

      if (info->dsn)
	gnome_db_control_set_property(control_instance, "dsn", info->dsn);
      else
	gnome_db_control_set_property(control_instance, "dsn", "");
      done = TRUE;
    }

  /* returns something silly because we can't return multiple values */
  if (!strcmp(name, "ident_name") || !strcmp(name, "ident_password"))
    {
      gnome_db_control_set_property(control_instance, name, "");
      done = TRUE;
    }

  if (!done && !strcmp(name, "ident_show"))
    {
      gnome_db_control_set_property(control_instance, "ident_show", 
				    &(info->ident));
      done = TRUE;
    }

  if (!done && !strcmp(name, "reset"))
    {
      gnome_db_control_set_property(control_instance, "reset", 
				    &(info->running));
      done = TRUE;
    }

  if (!done)
    g_warning("Unknown property '%s'\n", name);
}


void config_widget_set_property_cb(GnomeDbControl *control,
				   GnomeDbControlInstance *control_instance,
				   const gchar *name,
				   gconstpointer value,
				   gpointer data)
{
  PostgresPropsInfo *info;
  gboolean done=FALSE;

  info = gnome_db_control_get_instance_data(control_instance);
  if (!strcmp(name, "dsn") || !strcmp(name, "ident_name") || 
      !strcmp(name, "ident_password") || !strcmp(name, "ident_show") ||
      !strcmp(name, "reset"))
    {
      gnome_db_control_widget_set_prop_ptr
	(GNOME_DB_CONTROL_WIDGET(info->u.global.users_list),
	 name, value);
      gnome_db_control_widget_set_prop_ptr
	(GNOME_DB_CONTROL_WIDGET(info->u.global.users_ac),
	 name, value);
      done = TRUE;
    }

  if (!strcmp(name, "dsn"))
    {
      if (info->dsn)
	g_free(info->dsn);
      info->dsn = g_strdup((gchar *) value);
    }

  if (!done && !strcmp(name, "ident_show"))
    {
      info->ident = *((gboolean *) value);
    }

  if (!done && !strcmp(name, "reset"))
    {
      if (*((gboolean *) value))
	{
	  info->running = TRUE;
	}
    }

  if (!done)
    g_warning("Unknown property '%s'\n", name);
}




/* creates a simple name/password entries group */
GtkWidget* create_user_id_config(PostgresPropsInfo *info)
{
  GtkWidget *frame, *hb, *wid;

  frame = gtk_frame_new(_("User identity"));
  
  hb = gtk_hbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(hb), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), hb);

  wid = gtk_label_new(_("Name:"));
  gtk_box_pack_start(GTK_BOX(hb), wid, FALSE, FALSE, 0);
  wid = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(hb), wid, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(frame), "name", wid);
  info->user_name = wid;

  wid = gtk_label_new(_("Password:"));
  gtk_box_pack_start(GTK_BOX(hb), wid, FALSE, FALSE, 0);
  wid = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(wid), FALSE);
  gtk_box_pack_start(GTK_BOX(hb), wid, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(frame), "pass", wid);
  info->user_passwd = wid;

  gtk_widget_show_all(hb);
  return frame;
}



/*
 *
 * MAIN
 *
 */
int
main (int argc, char *argv[])
{
  CORBA_Environment ev;
  GnomeDbControl*   control;

  /* initialize CORBA/bonobo */
  CORBA_exception_init(&ev);
  gnome_CORBA_init("gda-postgres-config", VERSION,
		   &argc, argv,
		   GNORBA_INIT_SERVER_FUNC,
		   &ev);

  if (ev._major != CORBA_NO_EXCEPTION)
    g_error(_("Could not initialize CORBA"));

  if (!bonobo_init(gnome_CORBA_ORB(), NULL, NULL))
    g_error(_("Could not initialize Bonobo"));

  CORBA_exception_free(&ev);
  controls_list = NULL;

  /* register the users list control */
  control = 
    GNOME_DB_CONTROL(gnome_db_control_new("gda-postgres-users-list", 
					  create_users_list_widget_cb));

  if (control)
    {
      gtk_signal_connect(GTK_OBJECT(control), "new_instance",
			 GTK_SIGNAL_FUNC(users_list_new_instance_cb), NULL);
      gtk_signal_connect(GTK_OBJECT(control), "get_property",
			 GTK_SIGNAL_FUNC(users_list_get_property_cb), NULL);
      gtk_signal_connect(GTK_OBJECT(control), "set_property",
			 GTK_SIGNAL_FUNC(users_list_set_property_cb), NULL);
    }
  else
    g_error(_("Could not instantiate the gnome-db-control"));

  g_print("Control gda-postgres-user-list %p created\n", control);

  /* register the users access control */
  control = 
    GNOME_DB_CONTROL(gnome_db_control_new("gda-postgres-users-ac", 
					  create_users_ac_widget_cb));

  if (control)
    {
      gtk_signal_connect(GTK_OBJECT(control), "new_instance",
			 GTK_SIGNAL_FUNC(users_ac_new_instance_cb), NULL);
      gtk_signal_connect(GTK_OBJECT(control), "get_property",
			 GTK_SIGNAL_FUNC(users_ac_get_property_cb), NULL);
      gtk_signal_connect(GTK_OBJECT(control), "set_property",
			 GTK_SIGNAL_FUNC(users_ac_set_property_cb), NULL);
    }
  else
    g_error(_("Could not instantiate the gnome-db-control"));

  g_print("Control gda-postgres-user-ac %p created\n", control);

  /* register the global config control */
  control = 
    GNOME_DB_CONTROL(gnome_db_control_new("gda-postgres-config", 
					  create_config_widget_cb));
  if (control)
    {
      /* FIXME */
      gtk_signal_connect(GTK_OBJECT(control), "new_instance",
			 GTK_SIGNAL_FUNC(config_widget_new_instance_cb), NULL);
      gtk_signal_connect(GTK_OBJECT(control), "get_property",
			 GTK_SIGNAL_FUNC(config_widget_get_property_cb), NULL);
      gtk_signal_connect(GTK_OBJECT(control), "set_property",
			 GTK_SIGNAL_FUNC(config_widget_set_property_cb), NULL);
    }
  else
    g_error(_("Could not instantiate the gnome-db-control"));

  g_print("Control gda-postgres-config %p created\n", control);

  bonobo_main();
  return 0;
}

gchar* get_value(gchar* ptr)
{
  while (*ptr && *ptr != '=')
    ptr++;
  if (!*ptr)
    return 0;
  ptr++;
  if (!*ptr)
    return 0;
  while (*ptr && isspace(*ptr))
    ptr++;
  return (g_strdup(ptr));
}

/* WARN: str will be freed and new one allocated */
gchar *escape_chars(gchar *str)
{
  gchar *ptr=str, *ret, *retptr;
  gint size; 

  /* determination of the new string size */
  size = 1;
  while (*ptr != '\0') {
    if (*ptr=='\'') {
      if (ptr == str)
        size +=2;
      else {
        if (*(ptr-1) == '\\')
          size +=1;
        else
          size +=2;
      }
    }
    else 
      size +=1;
    ptr++;
  }
  
  ptr=str;
  ret = (gchar *) malloc(sizeof(gchar)*size);
  retptr = ret;
  while(*ptr!='\0') {
    if (*ptr == '\'') {
      if (ptr == str) {
        *retptr = '\\';
        retptr++;
      }
      else 
        if (*(ptr-1) != '\\') {
          *retptr = '\\';
          retptr++;
        }
    }
    *retptr = *ptr;
    retptr++;
    ptr++;
  }
  *retptr = '\0';
  g_free(str);
  return ret;
}


/*
 * Return a PGconn which is either NULL or a valid one
 */
PGconn *get_setup_conn(PostgresPropsInfo *info)
{
  PGconn *conn;
  gchar *host=NULL, *port=NULL, *options=NULL, *tty=NULL, 
    *db=NULL, *login=NULL, *passwd=NULL, *str;
  gchar *ptr_s, *ptr_e, *hold;

  /* parse the DSN to get the parameters */
  ptr_s = hold = g_strdup(info->dsn);
  while (ptr_s && *ptr_s)
    {
      ptr_e = strchr(ptr_s, ';');
      if (ptr_e) *ptr_e = '\0';
      if (strncasecmp(ptr_s, "HOST", strlen("HOST")) == 0)
        host = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "DATABASE", strlen("DATABASE")) == 0)
        db = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PORT", strlen("PORT")) == 0)
        port = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "OPTIONS", strlen("OPTIONS")) == 0)
        options = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "TTY", strlen("TTY")) == 0)
        tty = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "LOGIN", strlen("LOGIN")) == 0)
        login = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PASSWORD", strlen("PASSWORD")) == 0)
        passwd = get_value(ptr_s);
      ptr_s = ptr_e;
      if (ptr_s)
        ptr_s++;
    }
  g_free(hold);

  /* if login/passwd specified, set it to them */
  str = gtk_entry_get_text(GTK_ENTRY(info->user_name));
  if (str && (*str!=0))
    {
      if (login)
	g_free(login);
      login = g_strdup(str);
      str = gtk_entry_get_text(GTK_ENTRY(info->user_passwd));
      if (passwd)
	g_free(passwd);
      passwd = g_strdup(str);
    }
  
  /* if no database name, use 'template1' */
  if (!db)
    db = g_strdup("template1");

  /* opens connection */
  conn = PQsetdbLogin(host, port, options, tty, db, login, passwd);
  if (host) g_free(host);
  if (port) g_free(port);
  if (options) g_free(options);
  if (tty) g_free(tty);
  if (db) g_free(db);
  if (login) g_free(login);
  if (passwd) g_free(passwd);
  
  return conn;
}

/* converts a abstime like '2000-05-12 16:43:29+02' to a struct tm */
struct tm *str_to_tmstruct_timestamp(gchar *time) 
{
  int hh, mm, ss;
  int day, month, year;
  char *ptr1, *ptr2;
  char mtime[23];
  struct tm *stm;
  
  if ((time==NULL) || (*time=='\0')) 
      return NULL;

  strncpy(mtime, time, 22);
  mtime[22] = '\0';
  /* warning: do all the fractionning here because str_to_tmstruct_*
     functions use strtok as well! */
  ptr1=(char *) strtok(mtime, " "); /* the complete date */
  if (!(ptr2=(char *) strtok(NULL, ".+-")))  /* the complete time */
      return NULL; 

  stm = str_to_tmstruct_date2(ptr1);
  if (!stm)
      return NULL;
  year = stm->tm_year;
  month = stm->tm_mon;
  day = stm->tm_mday;
  g_free(stm);

  stm = str_to_tmstruct_time(ptr2);
  if (!stm)
      return NULL;
  hh = stm->tm_hour;
  mm = stm->tm_min;
  ss = stm->tm_sec;
  g_free(stm);

  stm = g_new0(struct tm, 1);/*(struct tm *) g_malloc(sizeof(struct tm));*/
  stm->tm_year = year;
  stm->tm_mon = month;
  stm->tm_mday = day;
  stm->tm_hour = hh;
  stm->tm_min = mm;
  stm->tm_sec = ss;

  return stm;
}

/* Converting YYYY-MM-DD to struct tm */
struct tm *str_to_tmstruct_date2(gchar *date)
{
  int day, month, year;
  char *ptr;
  char mdate[11];
  struct tm *stm;
  
  stm = g_new0(struct tm, 1);/*(struct tm *) g_malloc(sizeof(struct tm));*/
  if ((date==NULL) || (*date=='\0')) 
    {
      g_free(stm);
      return NULL;
    }

  strncpy(mdate, date, 10);
  mdate[10] = '\0';
  ptr=(char *) strtok(mdate, "-/.");
  year = atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/."))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  month=atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/."))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  day=atoi(ptr);
  
  stm->tm_mday = day;
  stm->tm_mon = month -1;
  stm->tm_year = year - 1900;
  
  return stm;
}

struct tm *str_to_tmstruct_time(gchar *time)
{
  int hh, mm, ss;
  char *ptr;
  char mtime[9];
  struct tm *stm;
  
  stm = g_new0(struct tm, 1);/*(struct tm *) g_malloc(sizeof(struct tm));*/
  if ((time==NULL) || (*time=='\0')) 
    {
      g_free(stm);
      return NULL;
    }

  strncpy(mtime, time, 8);
  mtime[8] = '\0';
  ptr=(char *) strtok(mtime, "-/.:");
  hh = atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/.:"))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  mm=atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/.:"))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  ss=atoi(ptr);

  if ((hh>23) || (mm>60) || (ss>60))
    {
      g_free(stm);
      return NULL; /* Error */
    }

  stm->tm_hour = hh;
  stm->tm_min = mm;
  stm->tm_sec = ss;
  
  return stm;
}

gboolean is_pgresult_valid(PGresult *res)
{
  gboolean retval=FALSE;
  if ((PQresultStatus(res) == PGRES_COMMAND_OK) ||
      (PQresultStatus(res) == PGRES_TUPLES_OK))
    retval = TRUE;

  return retval;
}
