/* GNOME DB Server Library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-server-impl.h"
#include <gda-common.h>

static void gda_server_impl_init       (Gda_ServerImpl *server_impl);
static void gda_server_impl_class_init (Gda_ServerImplClass *klass);

static void gda_server_impl_destroy    (Gda_ServerImpl *server_impl);

static GList* server_list = NULL;

/*
 * Private functions
 */
static void
gda_server_impl_init (Gda_ServerImpl *server_impl)
{
  g_return_if_fail(IS_GDA_SERVER_IMPL(server_impl));

  server_impl->name = NULL;
  //  memset((void *) &server_impl, 0, sizeof(Gda_ServerImplFunctions));
}

static void
gda_server_impl_class_init (Gda_ServerImplClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *) klass;

  object_class->destroy = gda_server_impl_destroy;
}

static void
gda_server_impl_destroy (Gda_ServerImpl *server_impl)
{
  g_return_if_fail(IS_GDA_SERVER_IMPL(server_impl));

  if (server_impl->name) g_free((gpointer) server_impl->name);
}

GtkType
gda_server_impl_get_type (void)
{
  static guint type = 0;

  if (!type)
    {
      GtkTypeInfo info =
      {
	"Gda_ServerImpl",
	sizeof (Gda_ServerImpl),
	sizeof (Gda_ServerImplClass),
	(GtkClassInitFunc) gda_server_impl_class_init,
	(GtkObjectInitFunc) gda_server_impl_init,
	(GtkArgSetFunc)NULL,
        (GtkArgSetFunc)NULL,
      };
      type = gtk_type_unique(gtk_object_get_type(), &info);
    }
  return type;
}

/**
 * gda_server_impl_new
 * @name: name of the server
 * @functions: callback functions
 *
 * Create a new GDA provider implementation object. This function initializes
 * all the needed CORBA stuff, registers the server in the system's object
 * directory, and initializes all internal data. After successful return,
 * you've got a ready-to-go GDA provider. To start it, use
 * #gda_server_impl_start
 */
Gda_ServerImpl *
gda_server_impl_new (const gchar *name, Gda_ServerImplFunctions *functions)
{
  Gda_ServerImpl*    server_impl;
  PortableServer_POA root_poa;
  CORBA_char*        objref;
  CORBA_Object       name_service;

  g_return_val_if_fail(name != NULL, NULL);

  /* look for an already running instance */
  server_impl = gda_server_impl_find(name);
  if (server_impl) return server_impl;

  /* start logging stuff */

  /* create provider instance */
  server_impl = GDA_SERVER_IMPL(gtk_type_new(gda_server_impl_get_type()));
  server_impl->name = g_strdup(name);
  if (functions)
    {
      memcpy((void *) &server_impl->functions,
	     (const void *) functions,
	     sizeof(Gda_ServerImplFunctions));
    }
  else g_warning(_("Starting provider %s with no implementation functions"), name);

  server_impl->connections = NULL;
  server_impl->is_running = FALSE;

  /* create CORBA connection factory */
  server_impl->ev = g_new0(CORBA_Environment, 1);
  server_impl->root_poa = (PortableServer_POA)
    CORBA_ORB_resolve_initial_references(gnome_CORBA_ORB(), "RootPOA", server_impl->ev);
  gda_server_impl_exception(server_impl->ev);

  server_impl->connection_factory_obj = impl_GDA_ConnectionFactory__create(server_impl->root_poa,
									   server_impl->ev);
  gda_server_impl_exception(server_impl->ev);
  objref = CORBA_ORB_object_to_string(gnome_CORBA_ORB(),
				      server_impl->connection_factory_obj,
				      server_impl->ev);
  gda_server_impl_exception(server_impl->ev);

  name_service = gnome_name_service_get();
  if (CORBA_Object_is_nil(name_service, server_impl->ev))
    gda_log_error(_("Could not get CORBA name service object"));
  else
    {
      goad_server_register(name_service, server_impl->connection_factory_obj,
			   server_impl->name, "object", server_impl->ev);
      gda_log_message(_("Registered with ID = %s"), objref);
    }
  
  server_list = g_list_append(server_list, (gpointer) server_impl);
  return server_impl;
}

/**
 * gda_server_impl_find
 * @id: object id
 *
 * Searches the list of loaded server implementations by object activation
 * identification
 */
Gda_ServerImpl *
gda_server_impl_find (const gchar *id)
{
  GList* node;

  g_return_val_if_fail(id != NULL, NULL);

  node = g_list_first(server_list);
  while (node)
    {
      Gda_ServerImpl* server_impl = GDA_SERVER_IMPL(node->data);
      if (server_impl && !strcmp(server_impl->name, id))
	return server_impl;
      node = g_list_next(node);
    }
  return NULL;
}

/**
 * gda_server_impl_start
 * @server_impl: server implementation
 *
 * Starts the given GDA provider
 */
void
gda_server_impl_start (Gda_ServerImpl *server_impl)
{
  PortableServer_POAManager pm;

  g_return_if_fail(server_impl != NULL);
  g_return_if_fail(server_impl->is_running == FALSE);

  pm = PortableServer_POA__get_the_POAManager(server_impl->root_poa, server_impl->ev);
  gda_server_impl_exception(server_impl->ev);
  PortableServer_POAManager_activate(pm, server_impl->ev);
  gda_server_impl_exception(server_impl->ev);

  server_impl->is_running = TRUE;
  CORBA_ORB_run(gnome_CORBA_ORB(), server_impl->ev);
}

/**
 * gda_server_impl_stop
 * @server_impl: server implementation
 *
 * Stops the given server implementation
 */
void
gda_server_impl_stop (Gda_ServerImpl *server_impl)
{
  g_return_if_fail(IS_GDA_SERVER_IMPL(server_impl));
  g_return_if_fail(server_impl->is_running);

  CORBA_ORB_shutdown(gnome_CORBA_ORB(), TRUE, server_impl->ev);
  server_impl->is_running = FALSE;
}

/**
 * gda_server_impl_connection_new
 */
Gda_ServerImplConnection *
gda_server_impl_connection_new (Gda_ServerImpl *server_impl)
{
  Gda_ServerImplConnection* cnc;

  g_return_val_if_fail(server_impl != NULL, NULL);

  /* FIXME: could be possible to share connections */
  cnc = g_new0(Gda_ServerImplConnection, 1);
  cnc->server_impl = server_impl;
  cnc->users = 1;

  /* notify 'real' provider */
  gda_server_connection_new(cnc);
  cnc->server_impl->connections = g_list_append(cnc->server_impl->connections, (gpointer) cnc);

  return cnc;
}

/**
 * gda_server_impl_connection_get_dsn
 */
gchar *
gda_server_impl_connection_get_dsn (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, NULL);
  return cnc->dsn;
}

/**
 * gda_server_impl_connection_set_dsn
 */
void
gda_server_impl_connection_set_dsn (Gda_ServerImplConnection *cnc, const gchar *dsn)
{
  g_return_if_fail(cnc != NULL);

  if (cnc->dsn) g_free((gpointer) cnc->dsn);
  if (dsn) cnc->dsn = g_strdup(dsn);
  else cnc->dsn = NULL;
}

/**
 * gda_server_impl_connection_get_username
 */
gchar *
gda_server_impl_connection_get_username (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, NULL);
  return cnc->username;
}

/**
 * gda_server_impl_connection_set_username
 */
void
gda_server_impl_connection_set_username (Gda_ServerImplConnection *cnc, const gchar *username)
{
  g_return_if_fail(cnc != NULL);

  if (cnc->username) g_free((gpointer) cnc->username);
  if (username) cnc->username = g_strdup(username);
  else cnc->username = NULL;
}

/**
 * gda_server_impl_connection_get_password
 */
gchar *
gda_server_impl_connection_get_password (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, NULL);
  return cnc->password;
}

/**
 * gda_server_impl_connection_set_password
 */
void
gda_server_impl_connection_set_password (Gda_ServerImplConnection *cnc, const gchar *password)
{
  g_return_if_fail(cnc != NULL);

  if (cnc->password) g_free((gpointer) cnc->password);
  if (password) cnc->password = g_strdup(password);
  else cnc->password = NULL;
}

/**
 * gda_server_impl_connection_add_error
 */
void
gda_server_impl_connection_add_error (Gda_ServerImplConnection *cnc, Gda_ServerImplError *error)
{
  g_return_if_fail(cnc != NULL);
  g_return_if_fail(error != NULL);

  cnc->errors = g_list_append(cnc->errors, (gpointer) error);
}

/**
 * gda_server_impl_connection_get_user_data
 */
gpointer
gda_server_impl_connection_get_user_data (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, NULL);
  return cnc->user_data;
}

/**
 * gda_server_impl_connection_set_user_data
 */
void
gda_server_impl_connection_set_user_data (Gda_ServerImplConnection *cnc, gpointer user_data)
{
  g_return_if_fail(cnc != NULL);
  cnc->user_data = user_data;
}

/**
 * gda_server_impl_connection_free
 */
void
gda_server_impl_connection_free (Gda_ServerImplConnection *cnc)
{
  g_return_if_fail(cnc != NULL);

  cnc->users--;
  if (!cnc->users)
    {
      if (cnc->dsn) g_free((gpointer) cnc->dsn);
      if (cnc->username) g_free((gpointer) cnc->username);
      if (cnc->password) g_free((gpointer) cnc->password);
      g_list_foreach(cnc->commands, (GFunc) gda_server_impl_command_free, NULL);
      g_list_foreach(cnc->errors, (GFunc) gda_server_impl_error_free, NULL);

      if (cnc->server_impl)
	{
	  cnc->server_impl->connections = g_list_remove(cnc->server_impl->connections, (gpointer) cnc);
	}
      gda_server_connection_free(cnc);
      g_free((gpointer) cnc);
    }
}

/**
 * gda_server_impl_command_new
 */
Gda_ServerImplCommand *
gda_server_impl_command_new (Gda_ServerImplConnection *cnc)
{
  Gda_ServerImplCommand* cmd;

  g_return_val_if_fail(cnc != NULL, NULL);

  cmd = g_new0(Gda_ServerImplCommand, 1);
  cmd->cnc = cnc;
  cmd->users = 1;

  gda_server_command_new(cmd);
  cmd->cnc->commands = g_list_append(cmd->cnc->commands, (gpointer) cmd);

  return cmd;
}

/**
 * gda_server_impl_command_get_connection
 */
Gda_ServerImplConnection *
gda_server_impl_command_get_connection (Gda_ServerImplCommand *cmd)
{
  g_return_val_if_fail(cmd != NULL, NULL);
  return cmd->cnc;
}

/**
 * gda_server_impl_command_get_text
 */
gchar *
gda_server_impl_command_get_text (Gda_ServerImplCommand *cmd)
{
  g_return_val_if_fail(cmd != NULL, NULL);
  return cmd->text;
}

/**
 * gda_server_impl_command_set_text
 */
void
gda_server_impl_command_set_text (Gda_ServerImplCommand *cmd, const gchar *text)
{
  g_return_if_fail(cmd != NULL);

  if (cmd->text) g_free((gpointer) cmd->text);
  if (text) cmd->text = g_strdup(text);
  else cmd->text = NULL;
}

/**
 * gda_server_impl_command_get_type
 */
gulong
gda_server_impl_command_get_type (Gda_ServerImplCommand *cmd)
{
  g_return_val_if_fail(cmd != NULL, 0);
  return cmd->type;
}

/**
 * gda_server_impl_command_set_type
 */
void
gda_server_impl_command_set_type (Gda_ServerImplCommand *cmd, gulong type)
{
  g_return_if_fail(cmd != NULL);
  cmd->type = type;
}

/**
 * gda_server_impl_command_get_user_data
 */
gpointer
gda_server_impl_command_get_user_data (Gda_ServerImplCommand *cmd)
{
  g_return_val_if_fail(cmd != NULL, NULL);
  return cmd->user_data;
}

/**
 * gda_server_impl_command_set_user_data
 */
void
gda_server_impl_command_set_user_data (Gda_ServerImplCommand *cmd, gpointer user_data)
{
  g_return_if_fail(cmd != NULL);
  cmd->user_data = user_data;
}

/**
 * gda_server_impl_command_free
 */
void
gda_server_impl_command_free (Gda_ServerImplCommand *cmd)
{
  g_return_if_fail(cmd != NULL);

  cmd->users--;
  if (!cmd->users)
    {
      gda_server_command_free(cmd);
      cmd->cnc->commands = g_list_remove(cmd->cnc->commands, (gpointer) cmd);
      if (cmd->text) g_free((gpointer) cmd->text);
      g_free((gpointer) cmd);
    }
}

/**
 * gda_server_impl_recordset_new
 */
Gda_ServerImplRecordset *
gda_server_impl_recordset_new (Gda_ServerImplConnection *cnc)
{
  Gda_ServerImplRecordset* recset;

  g_return_val_if_fail(cnc != NULL, NULL);

  recset = g_new0(Gda_ServerImplRecordset, 1);
  recset->cnc = cnc;
  recset->fields = NULL;
  recset->position = -1;
  recset->at_begin = FALSE;
  recset->at_end = FALSE;
  recset->users = 1;
  
  gda_server_recordset_new(recset);
  return recset;
}

/**
 * gda_server_impl_recordset_get_connection
 */
Gda_ServerImplConnection *
gda_server_impl_recordset_get_connection (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, NULL);
  return recset->cnc;
}

/**
 * gda_server_impl_recordset_add_field
 */
void
gda_server_impl_recordset_add_field (Gda_ServerImplRecordset *recset, Gda_ServerImplField *field)
{
  g_return_if_fail(recset != NULL);
  g_return_if_fail(field != NULL);

  recset->fields = g_list_append(recset->fields, (gpointer) field);
}

/**
 * gda_server_impl_recordset_get_fields
 */
GList *
gda_server_impl_recordset_get_fields (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, NULL);
  return recset->fields;
}

/**
 * gda_server_impl_recordset_is_at_begin
 */
gboolean
gda_server_impl_recordset_is_at_begin (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, FALSE);
  return recset->at_begin;
}

/**
 * gda_server_impl_recordset_set_at_begin
 */
void
gda_server_impl_recordset_set_at_begin (Gda_ServerImplRecordset *recset, gboolean at_begin)
{
  g_return_if_fail(recset != NULL);
  recset->at_begin = at_begin;
}

/**
 * gda_server_impl_recordset_is_at_end
 */
gboolean
gda_server_impl_recordset_is_at_end (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, FALSE);
  return recset->at_end;
}

/**
 * gda_server_impl_recordset_set_at_end
 */
void
gda_server_impl_recordset_set_at_end (Gda_ServerImplRecordset *recset, gboolean at_end)
{
  g_return_if_fail(recset != NULL);
  recset->at_end = at_end;
}

/**
 * gda_server_impl_recordset_get_user_data
 */
gpointer
gda_server_impl_recordset_get_user_data (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, NULL);
  return recset->user_data;
}

/**
 * gda_server_impl_recordset_set_user_data
 */
void
gda_server_impl_recordset_set_user_data (Gda_ServerImplRecordset *recset, gpointer user_data)
{
  g_return_if_fail(recset != NULL);
  recset->user_data = user_data;
}

/**
 * gda_server_impl_recordset_free
 */
void
gda_server_impl_recordset_free (Gda_ServerImplRecordset *recset)
{
  g_return_if_fail(recset != NULL);

  recset->users--;
  if (!recset->users)
    {
      gda_server_recordset_free(recset);
      g_list_foreach(recset->fields, (GFunc) gda_server_impl_field_free, NULL);
      g_free((gpointer) recset);
    }
}

/**
 * gda_server_impl_field_new
 */
Gda_ServerImplField *
gda_server_impl_field_new (void)
{
  return g_new0(Gda_ServerImplField, 1);
}

/**
 * gda_server_impl_field_set_name
 */
void
gda_server_impl_field_set_name (Gda_ServerImplField *field, const gchar *name)
{
  g_return_if_fail(field != NULL);

  if (field->name) g_free((gpointer) field->name);
  field->name = g_strdup(name);
}

/**
 * gda_server_impl_field_set_sql_type
 */
void
gda_server_impl_field_set_sql_type (Gda_ServerImplField *field, gulong sql_type)
{
  g_return_if_fail(field != NULL);
  field->sql_type = sql_type;
}

/**
 * gda_server_impl_field_set_defined_length
 */
void
gda_server_impl_field_set_defined_length (Gda_ServerImplField *field, glong length)
{
  g_return_if_fail(field != NULL);
  field->defined_length = length;
}

/**
 * gda_server_impl_field_set_actual_length
 */
void
gda_server_impl_field_set_actual_length (Gda_ServerImplField *field, glong length)
{
  g_return_if_fail(field != NULL);
  field->actual_length = length;
}

/**
 * gda_server_impl_field_set_scale
 */
void
gda_server_impl_field_set_scale (Gda_ServerImplField *field, gshort scale)
{
  g_return_if_fail(field != NULL);
  field->num_scale = scale;
}

/**
 * gda_server_impl_field_free
 */
void
gda_server_impl_field_free (Gda_ServerImplField *field)
{
  g_return_if_fail(field != NULL);

  if (field->name) g_free((gpointer) field->name);

  g_free((gpointer) field);
}

/**
 * gda_server_impl_error_new
 * @cnc: connection object
 */
Gda_ServerImplError *
gda_server_impl_error_new (void)
{
  return g_new0(Gda_ServerImplError, 1);
}

/**
 * gda_server_impl_error_get_description
 */
gchar *
gda_server_impl_error_get_description (Gda_ServerImplError *error)
{
  g_return_val_if_fail(error != NULL, NULL);
  return error->description;
}

/**
 * gda_server_impl_error_set_description
 */
void
gda_server_impl_error_set_description (Gda_ServerImplError *error, const gchar *description)
{
  g_return_if_fail(error != NULL);

  if (error->description) g_free((gpointer) error->description);
  error->description = g_strdup(description);
}

/**
 * gda_server_impl_error_get_number
 */
glong
gda_server_impl_error_get_number (Gda_ServerImplError *error)
{
  g_return_val_if_fail(error != NULL, -1);
  return error->number;
}

/**
 * gda_server_impl_error_set_number
 */
void
gda_server_impl_error_set_number (Gda_ServerImplError *error, glong number)
{
  g_return_if_fail(error != NULL);
  error->number = number;
}

/**
 * gda_server_impl_error_set_source
 */
void
gda_server_impl_error_set_source (Gda_ServerImplError *error, const gchar *source)
{
  g_return_if_fail(error != NULL);

  if (error->source) g_free((gpointer) error->source);
  error->source = g_strdup(source);
}

/**
 * gda_server_impl_error_set_help_file
 */
void
gda_server_impl_error_set_help_file (Gda_ServerImplError *error, const gchar *helpfile)
{
  g_return_if_fail(error != NULL);

  if (error->helpfile) g_free((gpointer) error->helpfile);
  error->helpfile = g_strdup(helpfile);
}

/**
 * gda_server_impl_error_set_help_context
 */
void
gda_server_impl_error_set_help_context (Gda_ServerImplError *error, const gchar *helpctxt)
{
  g_return_if_fail(error != NULL);

  if (error->helpctxt) g_free((gpointer) error->helpctxt);
  error->helpctxt = g_strdup(helpctxt);
}

/**
 * gda_server_impl_error_set_sqlstate
 */
void
gda_server_impl_error_set_sqlstate (Gda_ServerImplError *error, const gchar *sqlstate)
{
  g_return_if_fail(error != NULL);

  if (error->sqlstate) g_free((gpointer) error->sqlstate);
  error->sqlstate = g_strdup(sqlstate);
}

/**
 * gda_server_impl_error_set_native
 */
void
gda_server_impl_error_set_native (Gda_ServerImplError *error, const gchar *native)
{
  g_return_if_fail(error != NULL);

  if (error->native) g_free((gpointer) error->native);
  error->native = g_strdup(native);
}

/**
 * gda_server_impl_error_free
 * @error: error object
 */
void
gda_server_impl_error_free (Gda_ServerImplError *error)
{
  g_return_if_fail(error != NULL);

  if (error->description) g_free((gpointer) error->description);
  if (error->source) g_free((gpointer) error->source);
  if (error->helpfile) g_free((gpointer) error->helpfile);
  if (error->helpctxt) g_free((gpointer) error->helpctxt);
  if (error->sqlstate) g_free((gpointer) error->sqlstate);
  if (error->native) g_free((gpointer) error->native);
  g_free((gpointer) error);
}

/**
 * gda_server_connection_new
 */
gboolean
gda_server_connection_new (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, FALSE);
  g_return_val_if_fail(cnc->server_impl != NULL, FALSE);
  g_return_val_if_fail(cnc->server_impl->functions.connection_new != NULL, FALSE);

  return cnc->server_impl->functions.connection_new(cnc);
}

/**
 * gda_server_connection_open
 */
gint
gda_server_connection_open (Gda_ServerImplConnection *cnc,
			    const gchar *dsn,
			    const gchar *user,
			    const gchar *password)
{
  gint rc;

  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(dsn != NULL, -1);
  g_return_val_if_fail(cnc->server_impl != NULL, -1);
  g_return_val_if_fail(cnc->server_impl->functions.connection_open != NULL, -1);

  rc = cnc->server_impl->functions.connection_open(cnc, dsn, user, password);
  if (rc != -1)
    {
      gda_server_impl_connection_set_dsn(cnc, dsn);
      gda_server_impl_connection_set_username(cnc, user);
      gda_server_impl_connection_set_password(cnc, password);
      rc = 0;
    }
  return rc;
}

/**
 * gda_server_connection_close
 */
void
gda_server_connection_close (Gda_ServerImplConnection *cnc)
{
  g_return_if_fail(cnc != NULL);
  g_return_if_fail(cnc->server_impl != NULL);
  g_return_if_fail(cnc->server_impl->functions.connection_close != NULL);

  cnc->server_impl->functions.connection_close(cnc);
  gda_server_connection_free(cnc);
}

/**
 * gda_server_connection_begin_transaction
 */
gint
gda_server_connection_begin_transaction (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->server_impl != NULL, -1);
  g_return_val_if_fail(cnc->server_impl->functions.connection_begin_transaction != NULL, -1);

  return cnc->server_impl->functions.connection_begin_transaction(cnc);
}

/**
 * gda_server_connection_commit_transaction
 */
gint
gda_server_connection_commit_transaction (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->server_impl != NULL, -1);
  g_return_val_if_fail(cnc->server_impl->functions.connection_commit_transaction != NULL, -1);

  return cnc->server_impl->functions.connection_commit_transaction(cnc);
}

/**
 * gda_server_connection_rollback_transaction
 */
gint
gda_server_connection_rollback_transaction (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->server_impl != NULL, -1);
  g_return_val_if_fail(cnc->server_impl->functions.connection_rollback_transaction != NULL, -1);

  return cnc->server_impl->functions.connection_rollback_transaction(cnc);
}

/**
 * gda_server_connection_open_schema
 */
Gda_ServerImplRecordset *
gda_server_connection_open_schema (Gda_ServerImplConnection *cnc,
				   Gda_ServerImplError *error,
				   GDA_Connection_QType t,
				   GDA_Connection_Constraint *constraints,
				   gint length)
{
  g_return_val_if_fail(cnc != NULL, NULL);
  g_return_val_if_fail(cnc->server_impl != NULL, NULL);
  g_return_val_if_fail(cnc->server_impl->functions.connection_open_schema != NULL, NULL);

  return cnc->server_impl->functions.connection_open_schema(cnc, error, t, constraints, length);
}

/**
 * gda_server_connection_start_logging
 */
gint
gda_server_connection_start_logging (Gda_ServerImplConnection *cnc, const gchar *filename)
{
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->server_impl != NULL, -1);
  g_return_val_if_fail(cnc->server_impl->functions.connection_start_logging != NULL, -1);

  return cnc->server_impl->functions.connection_start_logging(cnc, filename);
}

/**
 * gda_server_connection_stop_logging
 */
gint
gda_server_connection_stop_logging (Gda_ServerImplConnection *cnc)
{
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->server_impl != NULL, -1);
  g_return_val_if_fail(cnc->server_impl->functions.connection_stop_logging != NULL, -1);

  return cnc->server_impl->functions.connection_stop_logging(cnc);
}

/**
 * gda_server_connection_create_table
 */
gchar *
gda_server_connection_create_table (Gda_ServerImplConnection *cnc, GDA_RowAttributes *columns)
{
  g_return_val_if_fail(cnc != NULL, NULL);
  g_return_val_if_fail(cnc->server_impl != NULL, NULL);
  g_return_val_if_fail(cnc->server_impl->functions.connection_create_table != NULL, NULL);
  g_return_val_if_fail(columns != NULL, NULL);

  return cnc->server_impl->functions.connection_create_table(cnc, columns);
}

/**
 * gda_server_connection_supports
 */
gboolean
gda_server_connection_supports (Gda_ServerImplConnection *cnc, GDA_Connection_Feature feature)
{
  g_return_val_if_fail(cnc != NULL, FALSE);
  g_return_val_if_fail(cnc->server_impl != NULL, FALSE);
  g_return_val_if_fail(cnc->server_impl->functions.connection_supports != NULL, FALSE);

  return cnc->server_impl->functions.connection_supports(cnc, feature);
}

/**
 * gda_server_connection_get_gda_type
 */
GDA_ValueType
gda_server_connection_get_gda_type (Gda_ServerImplConnection *cnc, gulong sql_type)
{
  g_return_val_if_fail(cnc != NULL, GDA_TypeNull);
  g_return_val_if_fail(cnc->server_impl != NULL, GDA_TypeNull);
  g_return_val_if_fail(cnc->server_impl->functions.connection_get_gda_type != NULL, GDA_TypeNull);

  return cnc->server_impl->functions.connection_get_gda_type(cnc, sql_type);
}

/**
 * gda_server_connection_get_c_type
 */
gshort
gda_server_connection_get_c_type (Gda_ServerImplConnection *cnc, GDA_ValueType type)
{
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->server_impl != NULL, -1);
  g_return_val_if_fail(cnc->server_impl->functions.connection_get_c_type != NULL, -1);

  return cnc->server_impl->functions.connection_get_c_type(cnc, type);
}

/**
 * gda_server_connection_free
 */
void
gda_server_connection_free (Gda_ServerImplConnection *cnc)
{
  g_return_if_fail(cnc != NULL);
  g_return_if_fail(cnc->server_impl != NULL);
  g_return_if_fail(cnc->server_impl->functions.connection_free != NULL);

  cnc->server_impl->functions.connection_free(cnc);
}

/**
 * gda_server_command_new
 */
gboolean
gda_server_command_new (Gda_ServerImplCommand *cmd)
{
  g_return_val_if_fail(cmd != NULL, FALSE);
  g_return_val_if_fail(cmd->cnc != NULL, FALSE);
  g_return_val_if_fail(cmd->cnc->server_impl != NULL, FALSE);
  g_return_val_if_fail(cmd->cnc->server_impl->functions.command_new != NULL, FALSE);

  return cmd->cnc->server_impl->functions.command_new(cmd);
}

/**
 * gda_server_command_execute
 */
Gda_ServerImplRecordset *
gda_server_command_execute (Gda_ServerImplCommand *cmd,
			    Gda_ServerImplError *error,
			    const GDA_CmdParameterSeq *params,
			    gulong *affected,
			    gulong options)
{
  g_return_val_if_fail(cmd != NULL, NULL);
  g_return_val_if_fail(cmd->cnc != NULL, NULL);
  g_return_val_if_fail(cmd->cnc->server_impl != NULL, NULL);
  g_return_val_if_fail(cmd->cnc->server_impl->functions.command_execute != NULL, NULL);

  return cmd->cnc->server_impl->functions.command_execute(cmd, error, params, affected, options);
}

/**
 * gda_server_command_free
 */
void
gda_server_command_free (Gda_ServerImplCommand *cmd)
{
  g_return_if_fail(cmd != NULL);
  g_return_if_fail(cmd->cnc != NULL);
  g_return_if_fail(cmd->cnc->server_impl != NULL);
  g_return_if_fail(cmd->cnc->server_impl->functions.command_free != NULL);

  cmd->cnc->server_impl->functions.command_free(cmd);
}

/**
 * gda_server_recordset_new
 */
gboolean
gda_server_recordset_new (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, FALSE);
  g_return_val_if_fail(recset->cnc != NULL, FALSE);
  g_return_val_if_fail(recset->cnc->server_impl != NULL, FALSE);
  g_return_val_if_fail(recset->cnc->server_impl->functions.recordset_new != NULL, FALSE);

  return recset->cnc->server_impl->functions.recordset_new(recset);
}

/**
 * gda_server_recordset_move_next
 */
gint
gda_server_recordset_move_next (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, -1);
  g_return_val_if_fail(recset->cnc != NULL, -1);
  g_return_val_if_fail(recset->cnc->server_impl != NULL, -1);
  g_return_val_if_fail(recset->cnc->server_impl->functions.recordset_move_next != NULL, -1);

  return recset->cnc->server_impl->functions.recordset_move_next(recset);
}

/**
 * gda_server_recordset_move_prev
 */
gint
gda_server_recordset_move_prev (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, -1);
  g_return_val_if_fail(recset->cnc != NULL, -1);
  g_return_val_if_fail(recset->cnc->server_impl != NULL, -1);
  g_return_val_if_fail(recset->cnc->server_impl->functions.recordset_move_prev != NULL, -1);

  return recset->cnc->server_impl->functions.recordset_move_prev(recset);
}

/**
 * gda_server_recordset_close
 */
gint
gda_server_recordset_close (Gda_ServerImplRecordset *recset)
{
  g_return_val_if_fail(recset != NULL, -1);
  g_return_val_if_fail(recset->cnc != NULL, -1);
  g_return_val_if_fail(recset->cnc->server_impl != NULL, -1);
  g_return_val_if_fail(recset->cnc->server_impl->functions.recordset_close != NULL, -1);

  return recset->cnc->server_impl->functions.recordset_close(recset);
}

/**
 * gda_server_recordset_free
 */
void
gda_server_recordset_free (Gda_ServerImplRecordset *recset)
{
  g_return_if_fail(recset != NULL);
  g_return_if_fail(recset->cnc != NULL);
  g_return_if_fail(recset->cnc->server_impl != NULL);
  g_return_if_fail(recset->cnc->server_impl->functions.recordset_free != NULL);

  recset->cnc->server_impl->functions.recordset_free(recset);
}

/**
 * gda_server_error_make
 */
void
gda_server_error_make (Gda_ServerImplError *error,
		       Gda_ServerImplRecordset *recset,
		       Gda_ServerImplConnection *cnc,
		       gchar *where)
{
  Gda_ServerImplConnection* cnc_to_use = NULL;

  g_return_if_fail(error != NULL);

  if (cnc) cnc_to_use = cnc;
  else if (recset) cnc_to_use = recset->cnc;

  if (!cnc_to_use)
    {
      gda_log_message(_("Could not get pointer to server implementation"));
      return;
    }

  g_return_if_fail(cnc_to_use->server_impl != NULL);
  g_return_if_fail(cnc_to_use->server_impl->functions.error_make != NULL);

  cnc_to_use->server_impl->functions.error_make(error, recset, cnc, where);

  gda_server_impl_connection_add_error(cnc_to_use, error);
}
		       
/*
 * Convenience functions
 */
gint
gda_server_impl_exception (CORBA_Environment *ev)
{
  g_return_val_if_fail(ev != NULL, -1);

  switch (ev->_major)
    {
    case CORBA_SYSTEM_EXCEPTION :
      gda_log_error(_("CORBA system exception %s"), CORBA_exception_id(ev));
      return -1;
    case CORBA_USER_EXCEPTION :
      gda_log_error(_("CORBA user exception: %s"), CORBA_exception_id( ev ));
      return -1;
    default:
      break;
    }
  return 0;
}

GDA_Error *
gda_server_impl_make_error_buffer (Gda_ServerImplConnection *cnc)
{
  gint       idx;
  GList*     ptr;
  GDA_Error* rc;

  g_return_val_if_fail(cnc != NULL, CORBA_OBJECT_NIL);

  rc  = CORBA_sequence_GDA_Error_allocbuf(g_list_length(cnc->errors));
  idx = 0;
  ptr = cnc->errors;

  while (ptr)
    {
      Gda_ServerImplError* e = ptr->data;
      rc[idx].description = CORBA_string_dup(e->description);
      rc[idx].number = e->number;
      rc[idx].source = CORBA_string_dup(e->source);
#if 0
      rc->_buffer[idx].helpfile = CORBA_string_dup(e->helpfile);
      rc->_buffer[idx].helpctxt = CORBA_string_dup(e->helpctxt);
#endif
      rc[idx].sqlstate = CORBA_string_dup(e->sqlstate);
      rc[idx].nativeMsg   = CORBA_string_dup(e->native);
      gda_server_impl_error_free(e);
      ptr = g_list_next(ptr);
      idx++;
    }
  g_list_free(cnc->errors);
  cnc->errors = NULL;
  return rc;
}


