/* GNOME DB Front End
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gda-fe.h>

static void clear_viewer_cb    (GtkWidget *w, gpointer data);
static void close_log_cb       (GtkWidget *w, gpointer data);
static void refresh_viewer_cb  (GtkWidget *w, gpointer data);
static void select_provider_cb (GtkMenuItem *item, gpointer data);
static void view_entry_cb      (GtkWidget *w, gpointer data);

static GtkWidget* log_tab = NULL;
static GtkWidget* option_menu = NULL;
static GtkWidget* log_viewer = NULL;
static GList*     mem_to_free = NULL;
static GnomeUIInfo log_toolbox[] =
{
  { GNOME_APP_UI_ITEM, N_("View"), N_("View selected entry"),
    view_entry_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PROP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear log file"),
    clear_viewer_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh log viewer"),
    refresh_viewer_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_REFRESH, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close log window"),
    close_log_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
fill_provider_list (GtkOptionMenu *optmenu)
{
  GList*     servlist;
  GList*     node;
  GtkWidget* menu;

  g_return_if_fail(GTK_IS_OPTION_MENU(optmenu));

  menu = gtk_menu_new();
  servlist = node = gda_server_list();
  while (node)
    {
      gchar*     str = g_strdup((GDA_SERVER_NAME((Gda_Server *) node->data)));
      GtkWidget* item = gtk_menu_item_new_with_label(str);
      gtk_signal_connect(GTK_OBJECT(item),
			 "activate",
			 GTK_SIGNAL_FUNC(select_provider_cb),
			 str);
      mem_to_free = g_list_append(mem_to_free, (gpointer) str);
      gtk_widget_show(item);
      gtk_menu_append(GTK_MENU(menu), item);
      node = g_list_next(node);
    }
  gda_server_free_list(servlist);
  gtk_widget_show(menu);
  gtk_option_menu_set_menu(optmenu, menu);
}

/*
 * Callbacks
 */
static void
clear_viewer_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(log_viewer));
  gnome_db_log_viewer_clear(GNOME_DB_LOG_VIEWER(log_viewer));
}

static void
close_log_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GTK_IS_WIDGET(log_tab));

  fe_workarea_remove_tab(log_tab);
  log_tab = NULL;

  /* free all memory */
  g_list_foreach(mem_to_free, (GFunc) g_free, NULL);
  g_list_free(mem_to_free);
  mem_to_free = NULL;
}

static void
refresh_viewer_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(log_viewer));
  gnome_db_log_viewer_refresh(GNOME_DB_LOG_VIEWER(log_viewer));
}

static void
select_provider_cb (GtkMenuItem *item, gpointer data)
{
  gchar* str = (gchar *) data;

  g_return_if_fail(GTK_IS_MENU_ITEM(item));
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(log_viewer));

  if (str)
    {
      gchar* filename = g_strdup_printf("/tmp/%s-srv.log", str);
      if (!gnome_db_log_viewer_load_file(GNOME_DB_LOG_VIEWER(log_viewer), filename))
	gda_ui_show_error(_("Error loading file '%s'"), filename);
      g_free((gpointer) filename);
    }
}

static void
view_entry_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(log_viewer));
  gnome_db_log_viewer_show_current(GNOME_DB_LOG_VIEWER(log_viewer));
}

/*
 * Public functions
 */
void
fe_log_open_window (void)
{
  if (!log_tab)
    {
      GtkWidget* table;
      GtkWidget* label;

      log_tab = gda_ui_new_frame_widget(NULL);

      table = gda_ui_new_table_widget(8, 4, FALSE);

      label = gda_ui_new_label_widget(_("Provider"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 3, 3);
      option_menu = gda_ui_new_option_menu_widget();
      fill_provider_list(GTK_OPTION_MENU(option_menu));
      gtk_table_attach(GTK_TABLE(table), option_menu, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

      log_viewer = gnome_db_log_viewer_new();
      gtk_widget_show(log_viewer);
      gtk_table_attach(GTK_TABLE(table), log_viewer, 0, 8, 3, 4,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       3, 3);
      gtk_container_add(GTK_CONTAINER(log_tab), table);

      fe_workarea_add_tab(_("Logs"), log_tab, log_toolbox);
    }

  fe_workarea_activate_tab(log_tab);
}
