/* GNOME DB Front End
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-fe.h"

static void activate_service_cb (GtkWidget *w, gpointer data);
static void close_window_cb     (GtkWidget *w, gpointer data);
static void refresh_list_cb     (GtkWidget *w, gpointer data);

static GtkWidget* service_tab = NULL;
static GtkWidget* service_list = NULL;
static GtkWidget* service_name_entry = NULL;
static GtkWidget* service_desc_entry = NULL;
static GtkWidget* service_type_entry = NULL;
static GtkWidget* service_loc_entry = NULL;
static GtkWidget* service_repo_list = NULL;
static GdkPixmap* pm_active;
static GdkPixmap* pm_inactive;
static GdkBitmap* pm_active_mask;
static GdkBitmap* pm_inactive_mask;
static GnomeUIInfo service_toolbox[] =
{
  { GNOME_APP_UI_ITEM, N_("Activate"), N_("Activate selected service"),
    activate_service_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Stop"), N_("Stop selected service"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_STOP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh service list"),
    refresh_list_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_REFRESH, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close log window"),
    close_window_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static gboolean
service_is_active (CORBA_Object ns, GoadServer *s, CORBA_Environment *ev)
{
  CORBA_Object                   otmp;
  static CosNaming_NameComponent nc[3] = { { "GNOME", "subcontext"},
					   { "Servers", "subcontext"}
                                         };
  static CosNaming_Name          nom = {0, 3, nc, CORBA_FALSE};
  gboolean                       retval;

  nc[2].id = s->server_id;
  nc[2].kind = "server";

  otmp = CosNaming_NamingContext_resolve(ns, &nom, ev);

  if (ev->_major != CORBA_NO_EXCEPTION)
    otmp = CORBA_OBJECT_NIL;

  CORBA_exception_free(ev);

  if (CORBA_Object_is_nil(otmp, ev))
    retval = FALSE;
  else
    {
      retval = TRUE;
      CORBA_Object_release(otmp, ev);
    }
  return retval;
}

static void
load_service_list (GtkCList *clist)
{
  GoadServerList*   servlist;
  GoadServer*       serv;
  CORBA_Environment ev;
  CORBA_Object      ns;
  gchar* err_msg = N_("Could not find any installed services. Either you don't have\n"
		      "any services installed or I don't know the path to find them.\n"
		      "In either case, please check your GNOME-DB installation.");

  g_return_if_fail(GTK_IS_CLIST(clist));

  gda_ui_clear_clist(clist);
  servlist = goad_server_list_get();
  serv = servlist->list;
  if (serv)
    {
      gint i, j;

      CORBA_exception_init(&ev);
      ns = gnome_name_service_get();

      gtk_clist_freeze(clist);
      for (i = 0; serv[i].repo_id; i++)
	{
	  gchar* cols[3];
	  gint   currow;

	  for (j = 0; serv[i].repo_id[j]; j++)
	    {
	      if (!strcmp(serv[i].repo_id[j], "IDL:GDA/ConnectionFactory:1.0") ||
		  !strcmp(serv[i].repo_id[j], "IDL:BonoboControl/gnome-db-config:1.0") ||
		  !strcmp(serv[i].repo_id[j], "IDL:BonoboControl/gnome-db-control:1.0"))
		{
		  cols[0] = "";
		  cols[1] = serv[i].server_id;
		  cols[2] = serv[i].description;
		  currow = gtk_clist_append(clist, cols);

		  if (service_is_active(ns, &serv[i], &ev))
		    gtk_clist_set_pixmap(clist, currow, 0, pm_active, pm_active_mask);
		  else
		    gtk_clist_set_pixmap(clist, currow, 0, pm_inactive, pm_inactive_mask);
		}
	    }
	}

      gtk_clist_thaw(clist);
      CORBA_Object_release(ns, &ev);
      CORBA_exception_free(&ev);
    }
  else gda_ui_show_error(err_msg);

  goad_server_list_free(servlist);
}

/*
 * Callbacks
 */
static void
activate_service_cb (GtkWidget *w, gpointer data)
{
  GList* selection;

  g_return_if_fail(GTK_IS_CLIST(service_list));

  selection = GTK_CLIST(service_list)->selection;
  if (selection)
    {
      gchar* server_id = NULL;

      if (gtk_clist_get_text(GTK_CLIST(service_list),
			     GPOINTER_TO_INT(selection->data),
			     1,
			     &server_id))
	{
	  CORBA_Environment ev;
	  CORBA_Object      obj;
	  GoadServerList*   servlist = goad_server_list_get();

	  CORBA_exception_init(&ev);
	  obj = goad_server_activate_with_id(servlist, server_id, 0, NULL);
	  if (!CORBA_Object_is_nil(obj, &ev))
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					    _("Service '%s' activated"), server_id);
	      load_service_list(GTK_CLIST(service_list));
	    }
	  else gda_ui_show_error(_("Could not activate service '%s'"), server_id);

	  goad_server_list_free(servlist);
	  CORBA_exception_free(&ev);
	}
    }
}

static void
close_window_cb (GtkWidget *w, gpointer data)
{
  if (GTK_IS_WIDGET(service_tab))
    {
      fe_workarea_remove_tab(service_tab);
    }
  service_tab = NULL;
}

static void
row_selected_cb (GtkCList *clist,
		 gint row,
		 gint column,
		 GdkEventButton *event,
		 gpointer user_data)
{
  gchar* serv_id = NULL;

  if (gtk_clist_get_text(clist, row, 1, &serv_id))
    {
      GoadServer*     serv;
      GoadServerList* servlist = goad_server_list_get();
      serv = servlist->list;
      if (serv)
	{
	  gint i, j;

	  for (i = 0; serv[i].repo_id; i++)
	    {
	      if (!strcasecmp(serv[i].server_id, serv_id))
		{
		  gtk_entry_set_text(GTK_ENTRY(service_name_entry), serv[i].server_id);
		  gtk_entry_set_text(GTK_ENTRY(service_desc_entry), serv[i].description);
		  gtk_entry_set_text(GTK_ENTRY(service_loc_entry), serv[i].location_info);

		  if (serv[i].type == GOAD_SERVER_SHLIB)
		    gtk_entry_set_text(GTK_ENTRY(service_type_entry), "shlib");
		  else if (serv[i].type == GOAD_SERVER_EXE)
		    gtk_entry_set_text(GTK_ENTRY(service_type_entry), "exe");
		  else if (serv[i].type == GOAD_SERVER_RELAY)
		    gtk_entry_set_text(GTK_ENTRY(service_type_entry), "relay");
		  else if (serv[i].type == GOAD_SERVER_FACTORY)
		    gtk_entry_set_text(GTK_ENTRY(service_type_entry), "factory");
		  else
		    gtk_entry_set_text(GTK_ENTRY(service_type_entry), "");

		  /* fill repo id list */
		  gda_ui_clear_clist(GTK_CLIST(service_repo_list));
		  gtk_clist_freeze(GTK_CLIST(service_repo_list));
		  for (j = 0; serv[i].repo_id[j]; j++)
		    {
		      gchar* col[1];
		      col[0] = serv[i].repo_id[j];
		      gtk_clist_append(GTK_CLIST(service_repo_list), col);
		    }
		  gtk_clist_thaw(GTK_CLIST(service_repo_list));
		  break;
		}
	    }
	}
      goad_server_list_free(servlist);
    }
  gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container), _("Ready"));
}

static void
row_unselected_cb (GtkCList *clist,
		   gint row,
		   gint column,
		   GdkEventButton *event,
		   gpointer user_data)
{
  gtk_entry_set_text(GTK_ENTRY(service_name_entry), "");
  gtk_entry_set_text(GTK_ENTRY(service_desc_entry), "");
  gtk_entry_set_text(GTK_ENTRY(service_type_entry), "");
  gtk_entry_set_text(GTK_ENTRY(service_loc_entry), "");
  gda_ui_clear_clist(GTK_CLIST(service_repo_list));
}

static void
refresh_list_cb (GtkWidget *w, gpointer data)
{
  load_service_list(GTK_CLIST(service_list));
  gtk_entry_set_text(GTK_ENTRY(service_name_entry), "");
  gtk_entry_set_text(GTK_ENTRY(service_desc_entry), "");
  gtk_entry_set_text(GTK_ENTRY(service_type_entry), "");
  gtk_entry_set_text(GTK_ENTRY(service_loc_entry), "");
  gda_ui_clear_clist(GTK_CLIST(service_repo_list));
}

/*
 * Public functions
 */
void
fe_service_open_window (void)
{
  static first_time = TRUE;

  if (!GTK_IS_WIDGET(service_tab))
    {
      GtkWidget* table;
      GtkWidget* scroll;
      GtkWidget* frame;
      GtkWidget* table2;
      GtkWidget* label;
      gchar*     serv_cols[3] = { "", N_("Name"), N_("Description") };
      gchar*     repo_cols[1] = { N_("Repo ID") };

      /* create pixmaps */
      if (first_time)
	{
	  gdk_imlib_load_file_to_pixmap(gnome_pixmap_file("yes.xpm"),
					&pm_active, &pm_active_mask);
	  gdk_imlib_load_file_to_pixmap(gnome_pixmap_file("no.xpm"),
					&pm_inactive, &pm_inactive_mask);
	  first_time = FALSE;
	}

      service_tab = gda_ui_new_frame_widget(NULL);
      table = gda_ui_new_table_widget(6, 2, FALSE);
      gtk_container_add(GTK_CONTAINER(service_tab), table);

      /* create service list */
      scroll = gda_ui_new_scrolled_window_widget();
      gtk_table_attach(GTK_TABLE(table), scroll, 0, 6, 0, 1,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       3, 3);
      service_list = gda_ui_new_clist_widget(serv_cols, 3);
      gtk_signal_connect(GTK_OBJECT(service_list),
			 "select-row",
			 GTK_SIGNAL_FUNC(row_selected_cb),
			 NULL);
      gtk_signal_connect(GTK_OBJECT(service_list),
			 "unselect-row",
			 GTK_SIGNAL_FUNC(row_unselected_cb),
			 NULL);
      load_service_list(GTK_CLIST(service_list));
      gtk_container_add(GTK_CONTAINER(scroll), service_list);

      /* add detail frame */
      frame = gda_ui_new_frame_widget(_("Properties"));
      gtk_table_attach(GTK_TABLE(table), frame, 0, 6, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
      table2 = gda_ui_new_table_widget(3, 4, FALSE);
      gtk_container_add(GTK_CONTAINER(frame), table2);

      label = gda_ui_new_label_widget(_("Name"));
      gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      service_name_entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_table_attach(GTK_TABLE(table2), service_name_entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Description"));
      gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
      service_desc_entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_table_attach(GTK_TABLE(table2), service_desc_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Type"));
      gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
      service_type_entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_table_attach(GTK_TABLE(table2), service_type_entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Location"));
      gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
      service_loc_entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_table_attach(GTK_TABLE(table2), service_loc_entry, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);

      scroll = gda_ui_new_scrolled_window_widget();
      gtk_table_attach(GTK_TABLE(table2), scroll, 2, 3, 0, 4,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       3, 3);
      service_repo_list = gda_ui_new_clist_widget(repo_cols, 1);
      gtk_container_add(GTK_CONTAINER(scroll), service_repo_list);

      fe_workarea_add_tab(_("Service Control"), service_tab, service_toolbox);
    }

  fe_workarea_activate_tab(service_tab);
}







