/* GDA Manager
 * Copyright (C) 1998,1999 Michael Lausch
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-mgr.h"
#include <gnome-db-control-widget.h>

#if defined(HAVE_BONOBO)
BonoboUIHandler* glb_UIHandler = 0;
#endif
static GList* l_containers = 0;

/*
 * Private functions
 */
static void
get_control_properties (MANAGER_Container *container)
{
#if defined(HAVE_BONOBO)
  g_return_if_fail(MANAGER_IS_CONTAINER(container));
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(container->widget));
#endif
}

/*
 * Callbacks
 */
static void
activate_container_cb (GtkWidget *shortcut, gchar *label, MANAGER_Container *container)
{
  g_return_if_fail(MANAGER_IS_CONTAINER(container));

  gtk_notebook_set_page(GTK_NOTEBOOK(glb_Notebook),
			gtk_notebook_page_num(GTK_NOTEBOOK(glb_Notebook),
					      container->widget));
}

static void
manager_container_destroy (MANAGER_Container *container)
{
  if (container->module) g_free((gpointer) container->module);
  //GTK_OBJECT_CLASS(container_parent_class)->destroy(container);
}

/*
 * MANAGER_Container widget implementation
 */
static void
manager_container_class_init (MANAGER_ContainerClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);

  object_class->destroy = manager_container_destroy;
}

static void
manager_container_init (MANAGER_Container *container)
{
  container->module = NULL;
  container->widget = NULL;
  container->pbc = NULL;
}

GtkType
manager_container_get_type (void)
{
  static GtkType type = 0;

  if (!type)
    {
      GtkTypeInfo info =
      {
	"MANAGER_Container",
	sizeof(MANAGER_Container),
	sizeof(MANAGER_ContainerClass),
	(GtkClassInitFunc) manager_container_class_init,
	(GtkObjectInitFunc) manager_container_init,
	NULL,
	NULL,
	(GtkClassInitFunc) NULL
      };
#if defined(HAVE_BONOBO)
      type = gtk_type_unique(bonobo_object_get_type(), &info);
#else
      type = gtk_type_unique(gtk_object_get_type(), &info);
#endif
    }
  return type;
}

MANAGER_Container *
manager_container_new (const gchar *module,
		       const gchar *dsn,
		       const gchar *username,
		       const gchar *password)
{
  MANAGER_Container*  container;

  g_return_val_if_fail(module != 0, NULL);

  /* check if container is already loaded */
  container = manager_container_find(module);
  if (container)
    {
      gtk_notebook_set_page(GTK_NOTEBOOK(glb_Notebook),
			    gtk_notebook_page_num(GTK_NOTEBOOK(glb_Notebook),
						  container->widget));
      return container;
    }

  /* load configuration module */
  container = gtk_type_new(manager_container_get_type());

#if defined(HAVE_BONOBO)
  container->widget = gnome_db_control_widget_new(module);
#else
  container->widget = gda_ui_new_label_widget(_("You need Bonobo to load components!"));
#endif
  if (GTK_IS_WIDGET(container->widget))
    {
      gboolean true = TRUE;

      container->module = g_strdup(module);
      get_control_properties(container);
      gtk_widget_show(container->widget);
      gtk_object_set_data(GTK_OBJECT(container->widget),
			  MANAGER_WIDGET_DATA_CONTAINER,
			  (gpointer) container);
      gtk_notebook_append_page(GTK_NOTEBOOK(glb_Notebook),
			       container->widget,
			       gtk_label_new(container->module));
      gtk_notebook_set_page(GTK_NOTEBOOK(glb_Notebook),
			    gtk_notebook_page_num(GTK_NOTEBOOK(glb_Notebook),
						  container->widget));
      l_containers = g_list_append(l_containers, (gpointer) container);

      /* set control properties */
      if (dsn)
	gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(container->widget),
					     GNOME_DB_CONTROL_PROP_DSN,
					     (gconstpointer) dsn);
      if (username)
	{
	  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(container->widget),
					       GNOME_DB_CONTROL_PROP_IDENT_NAME,
					       (gconstpointer) username);
	  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(container->widget),
					       GNOME_DB_CONTROL_PROP_USER_NAME,
					       (gconstpointer) username);
	}
      if (password)
	gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(container->widget),
					     GNOME_DB_CONTROL_PROP_IDENT_PASS,
					     (gconstpointer) password);
      gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(container->widget),
					   GNOME_DB_CONTROL_PROP_START,
					   (gconstpointer) &true);
      
      /* add icon to shortcut bar */
      gda_ui_shortcut_append(glb_Shortcut,
			     container->module,
			     gnome_pixmap_file("gnome-calendar.png"),
			     activate_container_cb,
			     (gpointer) container);
      return container;
    }
  else gda_ui_show_error(_("Could not activate module %s"), module);

  bonobo_object_destroy(container);
  return NULL;
}

void
manager_container_close (MANAGER_Container *container)
{
  g_return_if_fail(MANAGER_IS_CONTAINER(container));

  gda_ui_shortcut_remove(glb_Shortcut, container->module);
  g_free((gpointer) container->module);
  gtk_notebook_remove_page(GTK_NOTEBOOK(glb_Notebook),
			   gtk_notebook_page_num(GTK_NOTEBOOK(glb_Notebook),
						 container->widget));

  l_containers = g_list_remove(l_containers, (gpointer) container);
#if defined(HAVE_BONOBO)
  bonobo_object_unref(BONOBO_OBJECT(container));
#else
  gtk_object_unref(GTK_OBJECT(container));
#endif
}

void
manager_container_close_all (void)
{
  GList* node = g_list_first(l_containers);
  while (node)
    {
      MANAGER_Container* container = (MANAGER_Container *) node->data;
      if (MANAGER_IS_CONTAINER(container))
	{
	  manager_container_close(container);
	}
      node = l_containers = g_list_remove(l_containers, (gpointer) container);
    }
}

MANAGER_Container *
manager_container_find (const gchar* module)
{
  GList* node;

  g_return_val_if_fail(module != NULL, NULL);

  node = g_list_first(l_containers);
  while (node)
    {
      MANAGER_Container* container = (MANAGER_Container *) node->data;
      if (container && !g_strcasecmp(container->module, module))
	return container;
      node = g_list_next(node);
    }
  return NULL;
}



