/* GNOME DB library
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(HAVE_CONFIG_H)
#  include <config.h>
#endif
#include "gnome-db.h"

static void gnome_db_dsn_config_class_init (GnomeDbDsnConfigClass *klass);
static void gnome_db_dsn_config_init       (GnomeDbDsnConfig *dsnconf);

/*
 * GnomeDbDsnConfig widget interface
 */
static void
gnome_db_dsn_config_class_init (GnomeDbDsnConfigClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *) klass;
}

static void
gnome_db_dsn_config_init (GnomeDbDsnConfig *dsnconf)
{
  GtkWidget* frame;
  GtkWidget* table;
  GtkWidget* label;
  GList*     provider_list;
  GList*     node;

  frame = gnome_db_new_frame_widget(NULL);
  gtk_box_pack_start(GTK_BOX(dsnconf), frame, 1, 1, 0);

  table = gtk_table_new(7, 2, FALSE);
  gtk_widget_show(table);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gnome_db_new_label_widget(_("GDA Name"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 3, 3);
  dsnconf->gda_name_entry = gnome_db_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), dsnconf->gda_name_entry, 1, 2, 0, 1,
		   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		   GTK_SHRINK,
		   3, 3);

  label = gnome_db_new_label_widget(_("Provider"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_SHRINK, GTK_SHRINK, 3, 3);
  dsnconf->provider_entry = gnome_db_new_combo_widget();
  gtk_table_attach(GTK_TABLE(table), dsnconf->provider_entry, 1, 2, 1, 2,
		   GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		   GTK_SHRINK, 3, 3);
  provider_list = node = gda_provider_list();
  while (node)
    {
      Gda_Provider* provider = (Gda_Provider *) node->data;
      if (provider)
	{
	  GtkWidget* li = gtk_list_item_new_with_label(GDA_PROVIDER_NAME(provider));
	  gtk_widget_show(li);
	  gtk_container_add(GTK_CONTAINER(GTK_COMBO(dsnconf->provider_entry)->list), li);
	}
      node = g_list_next(node);
    }
  gda_provider_free_list(provider_list);

  label = gnome_db_new_label_widget(_("DSN"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_SHRINK, GTK_SHRINK, 3, 3);
  dsnconf->dsn_entry = gnome_db_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), dsnconf->dsn_entry, 1, 2, 2, 3,
		   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		   GTK_SHRINK,
		   3, 3);

  label = gnome_db_new_label_widget(_("Description"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, GTK_SHRINK, GTK_SHRINK, 3, 3);
  dsnconf->description_entry = gnome_db_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), dsnconf->description_entry, 1, 2, 3, 4,
		   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		   GTK_SHRINK,
		   3, 3);

  label = gnome_db_new_label_widget(_("Username"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, GTK_SHRINK, GTK_SHRINK, 3, 3);
  dsnconf->user_entry = gnome_db_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), dsnconf->user_entry, 1, 2, 4, 5,
		   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		   GTK_SHRINK,
		   3, 3);

  label = gnome_db_new_label_widget(_("Config"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, GTK_SHRINK, GTK_SHRINK, 3, 3);
  dsnconf->config_entry = gnome_db_new_entry_widget(0, FALSE);
  gtk_table_attach(GTK_TABLE(table), dsnconf->config_entry, 1, 2, 5, 6,
		   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		   GTK_SHRINK,
		   3, 3);

  dsnconf->global_button = gnome_db_new_check_button_widget(_("Global"), FALSE);
  gtk_table_attach(GTK_TABLE(table), dsnconf->global_button, 1, 2, 6, 7,
		   GTK_SHRINK, GTK_SHRINK,
		   3, 3);
}

guint
gnome_db_dsn_config_get_type (void)
{
  static guint db_dsn_config_type = 0;
  
  if (!db_dsn_config_type)
    {
      GtkTypeInfo db_dsn_config_info =
      {
        "GnomeDbDsnConfig",
        sizeof (GnomeDbDsnConfig),
        sizeof (GnomeDbDsnConfigClass),
        (GtkClassInitFunc) gnome_db_dsn_config_class_init,
        (GtkObjectInitFunc) gnome_db_dsn_config_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_dsn_config_type = gtk_type_unique(gtk_vbox_get_type(), &db_dsn_config_info);
    }
  return (db_dsn_config_type);
}

/**
 * gnome_db_dsn_config_new
 * @name: data source name
 *
 * Create a new GnomeDbDsnConfig, which is used to manage existing
 * DSN entries in your system GDA configuration. The widget provides
 * functions to let you view, modify and remove entries.
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_dsn_config_new (const gchar *name)
{
  GtkWidget *dsnconf = gtk_type_new(gnome_db_dsn_config_get_type());

  if (name)
    {
      gnome_db_dsn_config_set_name(GNOME_DB_DSN_CONFIG(dsnconf), name);
    }
  return dsnconf;
}

/**
 * gnome_db_dsn_config_clear
 * @dsnconf: GnomeDbDsnConfig widget
 *
 * Clears all widgets in the DSN config
 */
void
gnome_db_dsn_config_clear (GnomeDbDsnConfig *dsnconf)
{
  gchar* empty_string = "";

  g_return_if_fail(GNOME_DB_IS_DSN_CONFIG(dsnconf));

  gtk_entry_set_text(GTK_ENTRY(dsnconf->gda_name_entry), empty_string);
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dsnconf->provider_entry)->entry), empty_string);
  gtk_entry_set_text(GTK_ENTRY(dsnconf->dsn_entry), empty_string);
  gtk_entry_set_text(GTK_ENTRY(dsnconf->description_entry), empty_string);
  gtk_entry_set_text(GTK_ENTRY(dsnconf->user_entry), empty_string);
  gtk_entry_set_text(GTK_ENTRY(dsnconf->config_entry), empty_string);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dsnconf->global_button), FALSE);
}

/**
 * gnome_db_dsn_config_save
 * @dsnconf: GnomeDbDsnConfig widget
 *
 * Saves the given DSN being configured in the GDA configuration files
 */
gboolean
gnome_db_dsn_config_save (GnomeDbDsnConfig *dsnconf)
{
  gchar*   gda_name;

  g_return_val_if_fail(GNOME_DB_IS_DSN_CONFIG(dsnconf), FALSE);

  gda_name = gtk_entry_get_text(GTK_ENTRY(dsnconf->gda_name_entry));
  if (gda_name)
    {
      Gda_Dsn* dsn = gda_dsn_new();
      gda_dsn_set_name(dsn, gda_name);
      gda_dsn_set_provider(dsn,
			   gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dsnconf->provider_entry)->entry)));
      gda_dsn_set_dsn(dsn,
		      gtk_entry_get_text(GTK_ENTRY(dsnconf->dsn_entry)));
      gda_dsn_set_description(dsn,
			      gtk_entry_get_text(GTK_ENTRY(dsnconf->description_entry)));
      gda_dsn_set_username(dsn,
			   gtk_entry_get_text(GTK_ENTRY(dsnconf->user_entry)));
      gda_dsn_set_config(dsn,
			 gtk_entry_get_text(GTK_ENTRY(dsnconf->config_entry)));
      gda_dsn_set_global(dsn,
			 gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dsnconf->global_button)));
      if (gda_dsn_save(dsn))
	{
	  gda_dsn_free(dsn);
	  return TRUE;
	}
      else gnome_db_show_error(_("Error while saving configuration!"));
      gda_dsn_free(dsn);
    }
  else gnome_db_show_error(_("Data source has no name!"));
  return FALSE;
}

/**
 * gnome_db_dsn_config_remove
 * @dsnconf: GnomeDbDsnConfig widget
 */
gboolean
gnome_db_dsn_config_remove (GnomeDbDsnConfig *dsnconf)
{
  g_return_val_if_fail(GNOME_DB_IS_DSN_CONFIG(dsnconf), FALSE);
}

/**
 * gnome_db_dsn_config_get_name
 * @dsnconf: GnomeDbDsnConfig widget
 */
const gchar *
gnome_db_dsn_config_get_name (GnomeDbDsnConfig *dsnconf)
{
  g_return_val_if_fail(GNOME_DB_IS_DSN_CONFIG(dsnconf), NULL);
  return gtk_entry_get_text(GTK_ENTRY(dsnconf->gda_name_entry));
}

/**
 * gnome_db_dsn_config_set_name
 * @dsnconf: GnomeDbDsnConfig widget
 * @name: data source name
 */
void
gnome_db_dsn_config_set_name (GnomeDbDsnConfig *dsnconf, const gchar *name)
{
  GList* dsns;
  GList* node;

  g_return_if_fail(GNOME_DB_IS_DSN_CONFIG(dsnconf));
  g_return_if_fail(name != 0);

  dsns = node = gda_dsn_list();
  while (node)
    {
      Gda_Dsn* data = (Gda_Dsn *) node->data;
      if (data)
	{
	  if (!g_strcasecmp(GDA_DSN_GDA_NAME(data), name))
	    {
	      gtk_entry_set_text(GTK_ENTRY(dsnconf->gda_name_entry), GDA_DSN_GDA_NAME(data));
	      gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dsnconf->provider_entry)->entry),
				 GDA_DSN_PROVIDER(data));
	      gtk_entry_set_text(GTK_ENTRY(dsnconf->dsn_entry), GDA_DSN_DSN(data));
	      gtk_entry_set_text(GTK_ENTRY(dsnconf->description_entry),
				 GDA_DSN_DESCRIPTION(data));
	      gtk_entry_set_text(GTK_ENTRY(dsnconf->user_entry), GDA_DSN_USERNAME(data));
	      gtk_entry_set_text(GTK_ENTRY(dsnconf->config_entry), GDA_DSN_CONFIG(data));
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dsnconf->global_button),
					   GDA_DSN_IS_GLOBAL(data));
	      break;
	    }
	}
      node = g_list_next(node);
    }
  gda_dsn_free_list(dsns);
}




