/*  GNOME-DB Components
 *  Copyright (C) 2000 Rodrigo Moya
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gnomedb-com.h"
#include "e-vpaned.h"

#define CONNECTION_INSTANCE_DATA "ConnectionInstanceData"

typedef struct {
	GnomeDbControl* control;
	GtkWidget*      tab;
	GtkWidget*      notebook;
	GList*          open_cncs;
} connection_control_data_t;

typedef struct {
	connection_control_data_t* control_data;
	GdaConnection*             cnc;
	GtkWidget*                 tab;
	gchar*                     gda_dsn;
	
	/* SQL tab */
	GtkWidget*                 sql_tab;
	GtkWidget*                 sql_cmd;
	GtkWidget*                 sql_grid;
	
	/* browser tab */
	GtkWidget*                 browser_tab;

	/* batch tab */
	GtkWidget*                 batch_tab;
	GtkWidget*                 batch_file_list;
	GtkWidget*                 batch_transaction;
	GtkWidget*                 batch_log;
} connection_instance_data_t;

static void activate_batch_cb (GtkWidget *w, GnomeDbControl *control);
static void activate_browser_cb (GtkWidget *w, GnomeDbControl *control);
static void activate_sql_cb (GtkWidget *w, GnomeDbControl *control);
static void begin_transaction_cb (GtkWidget *w, GnomeDbControl *control);
static void commit_transaction_cb (GtkWidget *w, GnomeDbControl *control);
static void disconnect_cb (GtkWidget *w, GnomeDbControl *control);
static void export_database_cb (GtkWidget *w, GnomeDbControl *control);
static void open_connection_cb (GtkWidget *w, GnomeDbControl *control);
static void rollback_transaction_cb (GtkWidget *w, GnomeDbControl *control);
static void show_connection_properties_cb (GtkWidget *w, GnomeDbControl *control);

static void do_copy_cb  (GnomeDbControl *control, connection_control_data_t *control_data);
static void do_cut_cb  (GnomeDbControl *control, connection_control_data_t *control_data);
static void do_paste_cb (GnomeDbControl *control, connection_control_data_t *control_data);
static void do_print_cb (GnomeDbControl *control, connection_control_data_t *control_data);

static void sql_clear_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_copy_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_cut_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_open_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_paste_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_run_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_save_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_select_cb (GtkWidget *w, connection_instance_data_t *instance_data);

static GnomeUIInfo databasemenu[] = {
	{ GNOME_APP_UI_ITEM, N_("Connect"), N_("Open new database connection"),
	  open_connection_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_CONVERT, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Disconnect"), N_("Close current connection"),
	  disconnect_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_STOP, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("Browser"), N_("Activate database browser window"),
	  activate_browser_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_PROP, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("SQL"), N_("Activate SQL window"),
	  activate_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Batch"), N_("Activate dbatch job window"),
	  activate_batch_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_TIMER, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("Begin Transaction"), N_("Start a new database transaction"),
	  begin_transaction_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
	  NULL, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Commit"), N_("Commit current transaction"),
	  commit_transaction_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
	  NULL, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Rollback"), N_("Rollback current transaction"),
	  rollback_transaction_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
	  NULL, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("Properties"), N_("View current connection properties"),
	  show_connection_properties_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_PREF, 0, 0, NULL },
	GNOMEUIINFO_END
};
static GnomeUIInfo toolsmenu[] = {
	{ GNOME_APP_UI_ITEM, N_("Export database"), N_("Make a copy of a database"),
	  export_database_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_FORWARD, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Import database"), N_("Import an export file into a database"),
	  NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_BACK, 0, 0, NULL },
	GNOMEUIINFO_END
};
static GnomeUIInfo helpmenu[] = {
	GNOMEUIINFO_END
};
static GnomeUIInfo connection_control_menu[] = {
	GNOMEUIINFO_SUBTREE(N_("_Database"), databasemenu),
	GNOMEUIINFO_SUBTREE(N_("_Tools"), toolsmenu),
	GNOMEUIINFO_SUBTREE(N_("_Help"), helpmenu),
	GNOMEUIINFO_END
};
static GnomeUIInfo connection_control_toolbar[] = {
	{ GNOME_APP_UI_ITEM, N_("Connect"), N_("Open new database connection"),
	  open_connection_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_PIXMAP_CONVERT, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Disconnect"), N_("Close current connection"),
	  disconnect_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_PIXMAP_STOP, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("Browser"), N_("Activate database browser window"),
	  activate_browser_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_PIXMAP_PROPERTIES, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("SQL"), N_("Activate SQL window"),
	  activate_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Batch"), N_("Activate batch job window"),
	  activate_batch_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_PIXMAP_TIMER, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("Properties"), N_("View current connection properties"),
	  show_connection_properties_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_PIXMAP_PREFERENCES, 0, 0, NULL },
	GNOMEUIINFO_END
};

/*
 * Private functions
 */

static void
fill_saved_query_list (GtkCList *clist)
{
	GList* commands;

	g_return_if_fail(GTK_IS_CLIST(clist));

	commands = gnome_db_config_get_saved_sql_queries();
	if (commands) {
		GList* node;

		gtk_clist_freeze(clist);
		for (node = g_list_first(commands); node != NULL; node = g_list_next(node)) {
			gchar* row[2];
			gchar* tmp;

			row[0] = (gchar *) node->data;
			tmp = g_strdup_printf("%s/%s",
					      GNOME_DB_CONFIG_SECTION_SAVED_SQL_QUERIES,
					      (gchar *) node->data);
			row[1] = gda_config_get_string(tmp);
			g_free((gpointer) tmp);
			gtk_clist_append(clist, row);
		}
		gtk_clist_thaw(clist);
		gda_config_free_list(commands);
	}
}

static connection_instance_data_t *
get_current_connection_from_control (GnomeDbControl *control)
{
	connection_control_data_t* control_data;
	
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	
	control_data = gnome_db_control_get_user_data(control);
	if (control_data) {
		gint       page_num;
		GtkWidget* w;
		
		if (GTK_IS_NOTEBOOK(control_data->notebook)) {
			page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(control_data->notebook));
			w = gtk_notebook_get_nth_page(GTK_NOTEBOOK(control_data->notebook), page_num);
			if (GTK_IS_WIDGET(w))
				return gtk_object_get_data(GTK_OBJECT(w), CONNECTION_INSTANCE_DATA);
		}
	}
	return NULL;
}

static void
init_batch_window (connection_instance_data_t *instance_data)
{
	GtkWidget*   frame;
	GtkWidget*   table;
	GtkWidget*   scroll;
	const gchar* file_titles[] = { N_("Select"), N_("State"), N_("File name") };
	gboolean     has_transactions;

	g_return_if_fail(instance_data != NULL);

	has_transactions = gda_connection_supports(instance_data->cnc,
	                                           GDA_Connection_FEATURE_TRANSACTIONS);
	instance_data->batch_tab = gnome_db_new_table_widget(4, 4, FALSE);

	/* create tollbar */

	/* create file options area */
	frame = gnome_db_new_frame_widget(_("Options"));
	gtk_table_attach(GTK_TABLE(instance_data->batch_tab), frame, 0, 4, 1, 2,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);
	table = gnome_db_new_table_widget(4, 4, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 3, 0, 2,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);
	instance_data->batch_file_list = gnome_db_new_clist_widget(file_titles, 3);
	gtk_container_add(GTK_CONTAINER(scroll), instance_data->batch_file_list);

	instance_data->batch_transaction = gnome_db_new_check_button_widget(_("Transaction mode"), has_transactions);
	gtk_table_attach(GTK_TABLE(table), instance_data->batch_transaction, 3, 4, 0, 1,
	                 GTK_FILL, GTK_FILL, 3, 3);
	if (!has_transactions)
		gtk_widget_set_sensitive(instance_data->batch_transaction, FALSE);

	/* create log window */
	frame = gnome_db_new_frame_widget(_("Log"));
	gtk_table_attach(GTK_TABLE(instance_data->batch_tab), frame, 0, 4, 2, 3,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_container_add(GTK_CONTAINER(frame), scroll);
	instance_data->batch_log = gnome_db_new_text_widget();
	gtk_text_set_editable(GTK_TEXT(instance_data->batch_log), FALSE);
	gtk_container_add(GTK_CONTAINER(scroll), instance_data->batch_log);

	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->tab),
	                         instance_data->batch_tab,
	                         gtk_label_new(_("Batch")));
}

static void
init_browser_window (connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	
	instance_data->browser_tab = gnome_db_browser_new(instance_data->cnc);
	gnome_db_set_widget_tooltip(instance_data->browser_tab,
				    _("This screen lets you navigate through the contents "
				      "of your database, looking at the tables, views, "
				      "procedures, etc that may have created. Clicking "
				      "on an icon in one of the lists on the left, will "
				      "display extensive information about that selected "
				      "object in the right-hand pane"));
	gtk_widget_show(instance_data->browser_tab);
	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->tab),
	                         instance_data->browser_tab,
	                         gtk_label_new(_("Browser")));
}

static GnomeUIInfo sql_toolbar[] = {
	{ GNOME_APP_UI_ITEM, N_("Run"), N_("Execute current command"),
	  sql_run_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Select"), N_("Select command from previous"),
	  sql_select_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_UP, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Open"), N_("Open SQL command"),
	  sql_open_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Save"), N_("Save command for later retrieval"),
	  sql_save_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text"),
	  sql_cut_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_CUT, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text"),
	  sql_copy_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_COPY, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste text from clipboard"),
	  sql_paste_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_PASTE, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear widget contents"),
	  sql_clear_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
	GNOMEUIINFO_END
};

static void
init_sql_window (connection_instance_data_t *instance_data)
{
	GtkWidget* toolbar;
	GtkWidget* scroll;
	GtkWidget* paned;
	
	g_return_if_fail(instance_data != NULL);
	
	instance_data->sql_tab = gnome_db_new_table_widget(1, 2, FALSE);
	toolbar = gnome_db_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
	                                      GTK_TOOLBAR_ICONS,
	                                      sql_toolbar,
	                                      (gpointer) instance_data);
	gtk_table_attach(GTK_TABLE(instance_data->sql_tab), toolbar, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	paned = e_vpaned_new();
	gtk_widget_show(paned);
	gtk_table_attach(GTK_TABLE(instance_data->sql_tab), paned, 0, 1, 1, 2,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 3, 3);

	scroll = gnome_db_new_scrolled_window_widget();
	e_paned_add1(E_PANED(paned), scroll);
	instance_data->sql_cmd = gnome_db_new_text_widget();
	gnome_db_set_widget_tooltip(instance_data->sql_cmd,
								_("This widget allows you to send SQL commands "
								  "to the database you're connected to. Just type "
								  "in the command you want to run, and press the "
								  "'Run' button on the small toolbar"));
	gtk_container_add(GTK_CONTAINER(scroll), instance_data->sql_cmd);
  
	instance_data->sql_grid = GTK_WIDGET(gnome_db_new_grid_widget(NULL));
	gnome_db_set_widget_tooltip(instance_data->sql_grid,
								_("Here will be shown the results returned by the "
								  "database server in response to the command you "
								  "typed in"));
	e_paned_add2(E_PANED(paned), instance_data->sql_grid);
	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->tab),
	                         instance_data->sql_tab,
	                         gtk_label_new(_("SQL")));
}

/*
 * Callbacks
 */
static void
activate_batch_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->tab),
		                      gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->tab),
		                                            instance_data->batch_tab));
	}
}

static void
activate_browser_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->tab),
		                      gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->tab),
		                                            instance_data->browser_tab));
	}
}

static void
activate_sql_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->tab),
		                      gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->tab),
		                                            instance_data->sql_tab));
	}
}

static void
begin_transaction_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		if (gda_connection_supports(instance_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS)) {
			gda_connection_begin_transaction(instance_data->cnc);
		}
		else gnome_db_show_error(_("Transactions not supported"));
	}
}

static void
commit_transaction_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		if (gda_connection_supports(instance_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS)) {
			gda_connection_commit_transaction(instance_data->cnc);
		}
		else gnome_db_show_error(_("Transactions not supported"));
	}
}

static void
disconnect_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;

	g_return_if_fail(GNOME_DB_IS_CONTROL(control));

	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		/* remove this connection from our list */
		if (instance_data->control_data) {
			instance_data->control_data->open_cncs =
				g_list_remove(instance_data->control_data->open_cncs,
							  (gpointer) instance_data);
			gtk_notebook_remove_page(GTK_NOTEBOOK(instance_data->control_data->notebook),
									 gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->control_data->notebook),
														   instance_data->tab));
		}

		components_connection_close(instance_data->cnc);
		g_free((gpointer) instance_data->gda_dsn);
		g_free((gpointer) instance_data);
	}
}

static void
do_copy_cb (GnomeDbControl *control, connection_control_data_t *control_data)
{
	gint                        current_page;
	GtkWidget*                  tab;
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(instance_data->tab));
		tab = gtk_notebook_get_nth_page(GTK_NOTEBOOK(instance_data->tab), current_page);
		if (tab == instance_data->sql_tab)
			sql_copy_cb(NULL, instance_data);
		else if (tab == instance_data->browser_tab) ;
	}
}

static void
do_cut_cb (GnomeDbControl *control, connection_control_data_t *control_data)
{
	gint                        current_page;
	GtkWidget*                  tab;
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(instance_data->tab));
		tab = gtk_notebook_get_nth_page(GTK_NOTEBOOK(instance_data->tab), current_page);
		if (tab == instance_data->sql_tab)
			sql_cut_cb(NULL, instance_data);
		else if (tab == instance_data->browser_tab) ;
	}
}

static void
do_paste_cb (GnomeDbControl *control, connection_control_data_t *control_data)
{
	gint                        current_page;
	GtkWidget*                  tab;
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(instance_data->tab));
		tab = gtk_notebook_get_nth_page(GTK_NOTEBOOK(instance_data->tab), current_page);
		if (tab == instance_data->sql_tab)
			sql_paste_cb(NULL, instance_data);
		else if (tab == instance_data->browser_tab) ;
	}
}

static void
do_print_cb (GnomeDbControl *control, connection_control_data_t *control_data)
{
	gint                        current_page;
	GtkWidget*                  tab;
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(instance_data->tab));
		tab = gtk_notebook_get_nth_page(GTK_NOTEBOOK(instance_data->tab), current_page);
		if (tab == instance_data->sql_tab) ; // gnome_db_grid_print
		else if (tab == instance_data->browser_tab) ;
	}
}

static void
export_database_cb (GtkWidget *w, GnomeDbControl *control)
{
	GtkWidget *dialog;
	GtkWidget *export;

	/* create the dialog box */
	dialog = gnome_dialog_new (_("Export database"),
				   GNOME_STOCK_BUTTON_CLOSE,
				   NULL);

	export = gnome_db_export_new ();
	gtk_widget_show (export);
	gnome_db_export_set_pool (export, components_connection_get_pool ());
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), export, TRUE, TRUE, 0);

	gtk_widget_show (dialog);
}

static void
open_connection_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;
	GtkWidget*                  dialog;
	GtkWidget*                  login;

	g_return_if_fail(GNOME_DB_IS_CONTROL(control));

	instance_data = g_new0(connection_instance_data_t, 1);
	instance_data->control_data = gnome_db_control_get_user_data(control);

	/* create login dialog box */
	dialog = gnome_dialog_new(_("Open connection"),
							  GNOME_STOCK_BUTTON_OK,
							  GNOME_STOCK_BUTTON_CANCEL,
							  NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	login = gnome_db_login_new(NULL);
	gtk_widget_show(login);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), login, 1, 1, 0);

	/* run dialog */
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		instance_data->cnc =
			components_connection_open(gnome_db_login_get_gda_name(GNOME_DB_LOGIN(login)),
									   gnome_db_login_get_username(GNOME_DB_LOGIN(login)),
									   gnome_db_login_get_password(GNOME_DB_LOGIN(login)));
		if (instance_data->cnc) {
			gchar* label;

			instance_data->gda_dsn = g_strdup(gnome_db_login_get_gda_name(GNOME_DB_LOGIN(login)));

			/* create tab widget (a GtkNotebook) */
			instance_data->tab = gnome_db_new_notebook_widget();
			gtk_notebook_set_show_tabs(GTK_NOTEBOOK(instance_data->tab), FALSE);
			gtk_notebook_set_show_border(GTK_NOTEBOOK(instance_data->tab), FALSE);
			gtk_notebook_popup_disable(GTK_NOTEBOOK(instance_data->tab));

			init_browser_window(instance_data);
			init_sql_window(instance_data);
			init_batch_window(instance_data);
	
			/* add this instance to list of open connections */
			if (instance_data->control_data) {
				gint page_num;

				instance_data->control_data->open_cncs =
					g_list_append(instance_data->control_data->open_cncs,
								  (gpointer) instance_data);
				label = g_strdup_printf("%s@%s", gda_connection_get_user(instance_data->cnc),
										instance_data->gda_dsn);
				gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->control_data->notebook),
										 instance_data->tab,
										 gtk_label_new(label));
				g_free((gpointer) label);

				gtk_object_set_data(GTK_OBJECT(instance_data->tab), CONNECTION_INSTANCE_DATA, instance_data);
				page_num = gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->control_data->notebook),
												 instance_data->tab);
				gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->control_data->notebook), page_num);
			}
		}
		else
			g_free((gpointer) instance_data);
	}
	else
		g_free((gpointer) instance_data);
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
rollback_transaction_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		if (gda_connection_supports(instance_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS)) {
			gda_connection_rollback_transaction(instance_data->cnc);
		}
		else gnome_db_show_error(_("Transactions not supported"));
	}
}

static void
show_connection_properties_cb (GtkWidget *w, GnomeDbControl *control)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		GtkWidget* dialog;
		GtkWidget* frame;
		GtkWidget* table;
		GtkWidget* label;
		GtkWidget* entry;
		GdaDsn*   dsn_info;
		
		dialog = gnome_dialog_new(_("Connection Properties"),
		                          GNOME_STOCK_BUTTON_CLOSE,
		                          NULL);
		frame = gnome_db_new_frame_widget(NULL);
		gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
		table = gnome_db_new_table_widget(3, 6, FALSE);
		gtk_container_add(GTK_CONTAINER(frame), table);
		
		label = gnome_db_new_label_widget(_("GDA Name"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), instance_data->gda_dsn);
		gnome_db_set_widget_tooltip(entry, _("This is the name you use to refer to this connection"));
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
		
		dsn_info = gda_dsn_find_by_name(instance_data->gda_dsn);
		label = gnome_db_new_label_widget(_("Provider"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), GDA_DSN_PROVIDER(dsn_info));
		gnome_db_set_widget_tooltip(entry, _("This is the provider being used to access the database"));
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
		
		label = gnome_db_new_label_widget(_("User name"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_user(instance_data->cnc));
		gnome_db_set_widget_tooltip(entry, _("This is the user name you use to log into the database"));
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
		
		label = gnome_db_new_label_widget(_("Version"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_version(instance_data->cnc));
		gnome_db_set_widget_tooltip(entry, _("This is the version supported by the GDA provider"));
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
		
		gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
	}
}

static void
sql_clear_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
	
	gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
	gnome_db_grid_set_recordset(GNOME_DB_GRID(instance_data->sql_grid), NULL);
}

static void
sql_copy_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
	
	gtk_editable_copy_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
}

static void
sql_cut_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
	
	gtk_editable_cut_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
}

static void
sql_open_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	GtkWidget*   dialog;
	GtkWidget*   frame;
	GtkWidget*   table;
	GtkWidget*   scroll;
	GtkWidget*   label;
	GtkWidget*   list;
	GtkWidget*   from_conf;
	GtkWidget*   from_file;
	GtkWidget*   from_file_name;
	const gchar* saved_titles[] = { N_("Name"), N_("Command") };

	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));

	/* create dialog */
	dialog = gnome_dialog_new(_("Open SQL command"),
	                          GNOME_STOCK_BUTTON_OK,
	                          GNOME_STOCK_BUTTON_CANCEL,
	                          NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	gtk_widget_set_usize(dialog, 400, 250);
	frame = gnome_db_new_frame_widget(NULL);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
	table = gnome_db_new_table_widget(3, 3, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	/* from config stuff */
	from_conf = gnome_db_new_radio_button_widget(_("From saved query"), NULL);
	gtk_table_attach(GTK_TABLE(table), from_conf, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 3, 1, 2,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);

	list = gnome_db_new_clist_widget(saved_titles, 2);
	gnome_db_set_widget_tooltip(list, _("Select the named query to be loaded"));
	fill_saved_query_list(GTK_CLIST(list));
	gtk_container_add(GTK_CONTAINER(scroll), list);

	/* from file stuff */
	from_file = gnome_db_new_radio_button_widget(_("From file"), GTK_RADIO_BUTTON(from_conf));
	gtk_table_attach(GTK_TABLE(table), from_file, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
	from_file_name = gnome_db_new_file_entry_widget("GNOME_DB_Database_FileName");
	gnome_db_set_widget_tooltip(from_file_name, _("Select the file containing the command to be loaded"));
	gtk_table_attach(GTK_TABLE(table), from_file_name, 1, 3, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

	/* run the dialog */
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(from_file))) {
			gchar* filename = gtk_entry_get_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(from_file_name))));
			if (filename) {
				gchar* text;

				/* load command(s) from file */
				text = gda_util_load_file(filename);
				if (text) {
					gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
					gtk_text_insert(GTK_TEXT(instance_data->sql_cmd), NULL, NULL, NULL, text, strlen(text));
					g_free((gpointer) text);
				}
				else gnome_db_show_error(_("Error loading file %s"), filename);
			}
		}
		else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(from_conf))) {
			GList* selection;
			
			/* get CList's selection */
			selection = GTK_CLIST(list)->selection;
			if (selection) {
				gchar* text = NULL;
				if (gtk_clist_get_text(GTK_CLIST(list),
				                       GPOINTER_TO_INT(selection->data), 1, &text)) {
					gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
					gtk_text_insert(GTK_TEXT(instance_data->sql_cmd), NULL, NULL, NULL, text, strlen(text));
				}
			}
		}
	}
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
sql_paste_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
	
	gtk_editable_paste_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
}

static void
sql_run_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	gchar* sql;
	
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_TEXT(instance_data->sql_cmd));
	
	sql = gtk_editable_get_chars(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
	if (sql) {
		gulong         reccount;
		GdaRecordset* recset;
		
		/* execute command */
		recset = gda_connection_execute(instance_data->cnc, sql, &reccount, 0);
		if (recset) {
			/* FIXME: notify user of success in container's status bar */
		}
		gnome_db_grid_set_recordset(GNOME_DB_GRID(instance_data->sql_grid), recset);
		components_config_add_last_command((const gchar *) sql);
	}
}

static void
sql_save_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	GtkWidget* dialog;
	GtkWidget* frame;
	GtkWidget* table;
	GtkWidget* to_query;
	GtkWidget* to_query_name;
	GtkWidget* to_file;
	GtkWidget* to_file_name;

	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));

	/* create dialog */
	dialog = gnome_dialog_new(_("Save SQL command"),
	                          GNOME_STOCK_BUTTON_OK,
	                          GNOME_STOCK_BUTTON_CANCEL,
	                          NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	frame = gnome_db_new_frame_widget(NULL);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
	table = gnome_db_new_table_widget(3, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	/* to saved query stuff */
	to_query = gnome_db_new_radio_button_widget(_("To query"), NULL);
	gtk_table_attach(GTK_TABLE(table), to_query, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	to_query_name = gnome_db_new_entry_widget(0, TRUE);
	gnome_db_set_widget_tooltip(to_query_name, _("Save command to a named query"));
	gtk_table_attach(GTK_TABLE(table), to_query_name, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	/* to file stuff */
	to_file = gnome_db_new_radio_button_widget(_("To file"), GTK_RADIO_BUTTON(to_query));
	gtk_table_attach(GTK_TABLE(table), to_file, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
	to_file_name = gnome_db_new_file_entry_widget("GNOME_DB_Database_FileName");
	gnome_db_set_widget_tooltip(to_file_name, _("Select the file to save the command to"));
	gtk_table_attach(GTK_TABLE(table), to_file_name, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	/* run the dialog */
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		gchar* sql_txt = gtk_editable_get_chars(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);

		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(to_query))) {
			gchar* query_name = gtk_entry_get_text(GTK_ENTRY(to_query_name));

			/* save to "SQL Queries" configuration area */
			if (query_name) {
				gchar* key;

				key = g_strdup_printf("%s/%s", GNOME_DB_CONFIG_SECTION_SAVED_SQL_QUERIES, query_name);
				gda_config_set_string(key, sql_txt);
				g_free((gpointer) key);
			}
			else gnome_db_show_error(_("Query must have a name"));
		}
		else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(to_file))) {
			gchar* filename = gtk_entry_get_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(to_file_name))));
			
			/* save to file */
			if (filename) {
				if (!gda_util_save_file(filename, sql_txt))
					gnome_db_show_error(_("Could not save file %s"), filename);
			}
		}
	}
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
command_selected_cb (GtkCList *clist, gint row, gint col, GdkEvent *event, gpointer user_data)
{
	GtkText* sql = (GtkText *) user_data;
	gchar*   txt = NULL;

	g_return_if_fail(GTK_IS_CLIST(clist));
	g_return_if_fail(GTK_IS_TEXT(sql));

	if (gtk_clist_get_text(clist, row, 1, &txt)) {
		gtk_editable_delete_text(GTK_EDITABLE(sql), 0, -1);
		gtk_text_insert(GTK_TEXT(sql), NULL, NULL, NULL, txt, strlen(txt));
	}
}

static void
sql_select_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	GtkWidget*   dialog;
	GtkWidget*   frame;
	GtkWidget*   scroll;
	GtkWidget*   table;
	GtkWidget*   list;
	GtkWidget*   label;
	GtkWidget*   sql;
	GList*       commands;
	const gchar* cmd_titles[] = { "#", N_("Command") };
	const gchar* saved_titles[] = { N_("Name"), N_("Command") };

	g_return_if_fail(instance_data != NULL);

	/* create the dialog */
	dialog = gnome_dialog_new(_("Select SQL Command"),
	                          GNOME_STOCK_BUTTON_OK,
	                          GNOME_STOCK_BUTTON_CANCEL,
	                          NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	gtk_widget_set_usize(dialog, 500, 350);
	frame = gnome_db_new_frame_widget(NULL);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
	table = gnome_db_new_table_widget(2, 4, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	/* create GtkText to show the current selection */
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 2, 3, 4,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);
	sql = gnome_db_new_text_widget();
	gnome_db_set_widget_tooltip(sql, _("Here is shown the currently selected command's text"));
	gtk_container_add(GTK_CONTAINER(scroll), sql);

	/* create list of last commands */
	label = gnome_db_new_label_widget(_("Last commands"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 1, 1, 3,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);

	list = gnome_db_new_clist_widget(cmd_titles, 2);
	gnome_db_set_widget_tooltip(list, _("This is a list of the last commands you've executed"));
	gtk_signal_connect(GTK_OBJECT(list),
	                   "select_row",
	                   GTK_SIGNAL_FUNC(command_selected_cb),
	                   (gpointer) sql);
	commands = components_config_get_last_commands();
	if (commands) {
		GList* node;
		gint   cnt;
		
		gtk_clist_freeze(GTK_CLIST(list));
		for (cnt = 1, node = g_list_first(commands); node; node = g_list_next(node), cnt++) {
			gchar* row[2];
			
			row[0] = g_strdup_printf("%d", cnt);
			row[1] = (gchar *) node->data;
			gtk_clist_append(GTK_CLIST(list), row);
		}
		gtk_clist_thaw(GTK_CLIST(list));
	}
	gtk_container_add(GTK_CONTAINER(scroll), list);
	
	/* create list of saved queries */
	label = gnome_db_new_label_widget(_("Saved queries"));
	gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 1, 2, 1, 3,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);

	list = gnome_db_new_clist_widget(saved_titles, 2);
	gnome_db_set_widget_tooltip(list, _("This is a list of all the named queries you've stored"));
	gtk_signal_connect(GTK_OBJECT(list),
	                   "select_row",
	                   GTK_SIGNAL_FUNC(command_selected_cb),
	                   (gpointer) sql);
	fill_saved_query_list(GTK_CLIST(list));
	gtk_container_add(GTK_CONTAINER(scroll), list);

	/* run the dialog */
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		gchar* txt;
		
		txt = gtk_editable_get_chars(GTK_EDITABLE(sql), 0, -1);
		if (txt) {
			gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
			gtk_text_insert(GTK_TEXT(instance_data->sql_cmd), NULL, NULL, NULL, txt, strlen(txt));
		}
	}
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

/*
 * Public functions
 */
BonoboObject *
components_connection_new (void)
{
	GtkWidget*      tab;
	GnomeDbControl* control = NULL;
	
	tab = gnome_db_new_table_widget(4, 2, FALSE);
	control = gnome_db_control_new(tab);
	if (GNOME_DB_IS_CONTROL(control)) {
		connection_control_data_t* control_data;

		control_data = g_new0(connection_control_data_t, 1);
		control_data->control = control;
		control_data->tab = tab;
		                 
		/* create notebook */
		control_data->notebook = gnome_db_new_notebook_widget();
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(control_data->notebook), GTK_POS_BOTTOM);
		gtk_table_attach(GTK_TABLE(control_data->tab), control_data->notebook, 0, 4, 0, 2,
		                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		                 3, 3);

		gnome_db_control_set_menubar(control_data->control, connection_control_menu);
		gnome_db_control_set_toolbar(control_data->control, connection_control_toolbar);
		gnome_db_control_set_user_data(control_data->control, control_data);
		
		gtk_signal_connect(GTK_OBJECT(control_data->control),
		                   "do_print",
		                   GTK_SIGNAL_FUNC(do_print_cb),
		                   (gpointer) control_data);
		gtk_signal_connect(GTK_OBJECT(control_data->control),
		                   "do_cut_clipboard",
		                   GTK_SIGNAL_FUNC(do_cut_cb),
		                   (gpointer) control_data);
		gtk_signal_connect(GTK_OBJECT(control_data->control),
		                   "do_copy_clipboard",
		                   GTK_SIGNAL_FUNC(do_copy_cb),
		                   (gpointer) control_data);
		gtk_signal_connect(GTK_OBJECT(control_data->control),
		                   "do_paste_clipboard",
		                   GTK_SIGNAL_FUNC(do_paste_cb),
		                   (gpointer) control_data);
	}
	else gtk_widget_destroy(GTK_OBJECT(tab));
	
	return BONOBO_OBJECT(control);
}
