/* GNOME DB library
 * Copyright (C) 1999-2001 The Free Software Foundation
 *
 * Authors:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-export.h"
#include "gnome-db-util.h"
#include <gtk/gtkclist.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtknotebook.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>

struct _GnomeDbExportPrivate {
	GtkWidget *cnc_list;
	GtkWidget *notebook;
	GdaConnectionPool *pool;
	GdaExport *gda_export;

	GtkWidget *options_tab;
	GtkWidget *filename_entry;
	GtkWidget *export_table_data;

	GtkWidget *objects_tab;
	GtkWidget *object_notebook;
	GtkWidget *tables_list;
	GtkWidget *selected_objects;

	GtkWidget *log_tab;
	GtkWidget *log_output;
};

static void gnome_db_export_class_init (GnomeDbExportClass *klass);
static void gnome_db_export_init       (GnomeDbExport *export);
static void gnome_db_export_destroy    (GtkObject *object);

static void connect_button_clicked_cb  (GtkButton *button, gpointer user_data);
static void connection_selected_cb     (GtkMenuItem *menu_item, gpointer user_data);
static void object_tab_changed_cb      (GtkNotebook *notebook,
					GtkNotebookPage *page,
					guint page_num,
					gpointer user_data);

enum {
	LAST_SIGNAL
};

static gint db_export_signals[LAST_SIGNAL] = { 0, };

/*
 * Private functions
 */
static void
add_pool_connection (GdaConnectionPool *pool,
		     GdaConnection *cnc,
		     const gchar *gda_name,
		     const gchar *username,
		     const gchar *password,
		     gpointer user_data)
{
	GtkWidget *menu_item;
	GtkWidget *menu = (GtkWidget *) user_data;
	GnomeDbExport *export;

	g_return_if_fail (IS_GDA_CONNECTION (cnc));
	g_return_if_fail (GTK_IS_MENU (menu));

	export = gtk_object_get_data (GTK_OBJECT (menu), "GNOME_DB_Export_ExportWidget");
	if (gda_name) {
		gchar *tmp_str = g_strdup (gda_name);

		menu_item = gtk_menu_item_new_with_label (tmp_str);
		gtk_object_set_data_full (GTK_OBJECT (menu_item),
					  "GNOME_DB_MenuItemLabel",
					  (gpointer) tmp_str,
					  (GtkDestroyNotify) g_free);
		gtk_object_set_data (GTK_OBJECT (menu_item), "GNOME_DB_Export_GdaConnection", cnc);
		gtk_signal_connect (GTK_OBJECT (menu_item),
				    "activate",
				    GTK_SIGNAL_FUNC (connection_selected_cb),
				    (gpointer) export);

		gtk_widget_show (menu_item);
		gtk_menu_append (GTK_MENU (menu), menu_item);
	}
}

static void
fill_connection_list (GnomeDbExport *export)
{
	GtkWidget *menu;

	g_return_if_fail (GNOME_DB_IS_EXPORT (export));

	if (GTK_IS_OPTION_MENU (export->priv->cnc_list)) {
		gtk_option_menu_remove_menu (GTK_OPTION_MENU (export->priv->cnc_list));
		menu = gtk_menu_new ();
		gtk_object_set_data (GTK_OBJECT (menu), "GNOME_DB_Export_ExportWidget", export);

		/* if we don't have a GdaConnectionPool, create one */
		if (!IS_GDA_CONNECTION_POOL (export->priv->pool))
			export->priv->pool = gda_connection_pool_new ();
		gda_connection_pool_foreach (export->priv->pool,
					     (GdaConnectionPoolForeachFunc) add_pool_connection,
					     (gpointer) menu);

		gtk_option_menu_set_menu (GTK_OPTION_MENU (export->priv->cnc_list), menu);
	}
}

static void
show_list_in_clist (GtkCList *clist, GList *list)
{
	GList *l;

	g_return_if_fail (GTK_IS_CLIST (clist));

	gnome_db_clear_clist (clist);
	gtk_clist_freeze (clist);
	for (l = g_list_first (list); l != NULL; l = g_list_next (l)) {
		gchar *row[1] = { (gchar *) l->data };
		gtk_clist_append (clist, row);
	}
	gtk_clist_thaw (clist);
}

/*
 * Callbacks
 */
static void
connect_button_clicked_cb (GtkButton *button, gpointer user_data)
{
	GnomeDbExport *export = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (export));
}

static void
connection_selected_cb (GtkMenuItem *menu_item, gpointer user_data)
{
	GnomeDbExport *export = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (export));

	/*
	 * ugly yes, but it does what we need, which is to refresh the
	 * object lists
	 */
	object_tab_changed_cb (GTK_NOTEBOOK (export->priv->object_notebook),
			       NULL,
			       gtk_notebook_get_current_page (
				       GTK_NOTEBOOK (export->priv->object_notebook)),
			       export);
}

static void
object_tab_changed_cb (GtkNotebook *notebook,
		       GtkNotebookPage *page,
		       guint page_num,
		       gpointer user_data)
{
	GtkWidget *tab;
	GtkWidget *object_list;
	GList *list;
	GList *selection;
	GnomeDbExport *export = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (export));

	tab = gtk_notebook_get_nth_page (notebook, page_num);
	object_list = gtk_object_get_data (GTK_OBJECT (tab), "GNOME_DB_Export_ObjectList");
	if (GTK_IS_WIDGET (object_list)) {
		if (object_list == export->priv->tables_list) {
			list = gda_export_get_tables (export->priv->gda_export);
			selection = gda_export_get_selected_tables (export->priv->gda_export);
		}

		show_list_in_clist (GTK_CLIST (object_list), list);
		show_list_in_clist (GTK_CLIST (export->priv->selected_objects), selection);

		g_list_free (list);
		g_list_free (selection);
	}
}

/*
 * GnomeDbExport implementation
 */

static void
gnome_db_export_class_init (GnomeDbExportClass *klass)
{
	GtkObjectClass *object_class = GTK_OBJECT_CLASS (klass);

	object_class->destroy = gnome_db_export_destroy;
}

static void
gnome_db_export_init (GnomeDbExport *export)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *notebook;
	GtkWidget *scroll;

	export->priv = g_new0 (GnomeDbExportPrivate, 1);
	export->priv->gda_export = gda_export_new (NULL);

	/* create the main container */
	table = gnome_db_new_table_widget (4, 4, FALSE);
	gtk_box_pack_start (GTK_BOX (export), table, TRUE, TRUE, 0);

	/* create the information widgets */
	label = gnome_db_new_label_widget (_("Database"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	export->priv->cnc_list = gnome_db_new_option_menu_widget ();
	gnome_db_set_widget_tooltip (export->priv->cnc_list,
				     _("Select the database you want to export. If you want"
				       ", you can also open a new connection by pressing"
				       " the 'Connect' button"));
	gtk_table_attach (GTK_TABLE (table), export->priv->cnc_list, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget_with_pixmap ("", GNOME_STOCK_MENU_CONVERT);
	gnome_db_set_widget_tooltip (button, _("Press this button if you want to open a"
					       " new connection to the database to be exported"));
	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    GTK_SIGNAL_FUNC (connect_button_clicked_cb),
			    (gpointer) export);
	gtk_table_attach (GTK_TABLE (table), button, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	export->priv->notebook = gnome_db_new_notebook_widget ();
	gtk_table_attach (GTK_TABLE (table), export->priv->notebook, 0, 4, 2, 4,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  3, 3);

	/* create the 'Options' tab */
	export->priv->options_tab = gnome_db_new_table_widget (6, 3, FALSE);
	gtk_notebook_append_page (GTK_NOTEBOOK (export->priv->notebook),
				  export->priv->options_tab,
				  gtk_label_new (_("Options")));

	label = gnome_db_new_label_widget (_("Output file"));
	gtk_table_attach (GTK_TABLE (export->priv->options_tab), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 3, 3);
	export->priv->filename_entry = gnome_db_new_file_entry_widget ("GNOME_DB_Export_OutputFile");
	gnome_db_set_widget_tooltip (export->priv->filename_entry,
				     _("Select the file to which you want to save the"
				       " database contents"));
	gtk_table_attach (GTK_TABLE (export->priv->options_tab), export->priv->filename_entry,
			  1, 3, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	export->priv->export_table_data =
		gnome_db_new_check_button_widget (_("Export table data"), TRUE);
	gnome_db_set_widget_tooltip (export->priv->export_table_data,
				     _("When selected, the export process also saves"
				      " the exported tables' data. If not selected,"
				      " only the table structure is exported"));
	gtk_table_attach (GTK_TABLE (export->priv->options_tab), export->priv->export_table_data,
			  0, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	/* create the 'Objects' tab */
	export->priv->objects_tab = gnome_db_new_table_widget (3, 6, FALSE);
	gtk_notebook_append_page (GTK_NOTEBOOK (export->priv->notebook),
				  export->priv->objects_tab,
				  gtk_label_new (_("Objects")));
	export->priv->object_notebook = gnome_db_new_notebook_widget ();
	gtk_signal_connect (GTK_OBJECT (export->priv->object_notebook),
			    "switch_page",
			    GTK_SIGNAL_FUNC (object_tab_changed_cb),
			    (gpointer) export);
	gtk_table_attach (GTK_TABLE (export->priv->objects_tab), export->priv->object_notebook,
			  0, 1, 0, 6,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  3, 3);

	scroll = gnome_db_new_scrolled_window_widget ();
	export->priv->tables_list = gnome_db_new_clist_widget (NULL, 1);
	gtk_container_add (GTK_CONTAINER (scroll), export->priv->tables_list);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
				  scroll,
				  gtk_label_new (_("Tables")));
	gtk_object_set_data (GTK_OBJECT (scroll),
			     "GNOME_DB_Export_ObjectList",
			     export->priv->tables_list);

	scroll = gnome_db_new_scrolled_window_widget ();
	export->priv->selected_objects = gnome_db_new_clist_widget (NULL, 1);

	/* create the 'Log' tab */
	export->priv->log_tab = gnome_db_new_scrolled_window_widget ();
	gtk_notebook_append_page (GTK_NOTEBOOK (export->priv->notebook),
				  export->priv->log_tab,
				  gtk_label_new (_("Log")));
	export->priv->log_output = gnome_db_new_text_widget ();
	gtk_text_set_editable (GTK_TEXT (export->priv->log_output), FALSE);
	gtk_container_add (GTK_CONTAINER (export->priv->log_tab), export->priv->log_output);
}

static void
gnome_db_export_destroy (GtkObject *object)
{
	GtkObjectClass *parent_class;
	GnomeDbExport *export = (GnomeDbExport *) object;

	g_return_if_fail (GNOME_DB_IS_EXPORT (export));

	if (IS_GDA_EXPORT (export->priv->gda_export)) {
		gda_export_free (export->priv->gda_export);
		export->priv->gda_export = NULL;
	}
	if (IS_GDA_CONNECTION (export->priv->pool)) {
		gda_connection_pool_free (export->priv->pool);
		export->priv->pool = NULL;
	}

	g_free ((gpointer) export->priv);

	parent_class = gtk_type_class (gtk_vbox_get_type ());
	if (parent_class && parent_class->destroy)
		parent_class->destroy (object);
}

/**
 * gnome_db_export_get_type
 */
GtkType
gnome_db_export_get_type (void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"GnomeDbExport",
			sizeof (GnomeDbExport),
			sizeof (GnomeDbExportClass),
			(GtkClassInitFunc) gnome_db_export_class_init,
			(GtkObjectInitFunc) gnome_db_export_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		type = gtk_type_unique (gtk_vbox_get_type (), &info);
	}

	return type;
}

/**
 * gnome_db_export_new
 *
 * Create a new #GnomeDbExport widget. This widget is just a frontend
 * to the #GdaExport object provided in libgda, which lets you easily
 * add database exporting to your GNOME applications.
 *
 * Returns: the newly created widget
 */
GtkWidget *
gnome_db_export_new (void)
{
	GnomeDbExport *export;

	export = GNOME_DB_EXPORT (gtk_type_new (gnome_db_export_get_type ()));
	return export;
}

/**
 * gnome_db_export_get_pool
 * @export: a #GnomeDbExport widget
 *
 * Return the #GdaConnectionPool object being used by the given
 * #GnomeDbExport widget
 *
 * Returns: a #GdaConnectionPool object
 */
GdaConnectionPool *
gnome_db_export_get_pool (GnomeDbExport *export)
{
	g_return_val_if_fail (GNOME_DB_IS_EXPORT (export), NULL);
	return export->priv->pool;
}

/**
 * gnome_db_export_set_pool
 * @export: a #GnomeDbExport widget
 * @pool: a #GdaConnectionPool object
 *
 * Associate a #GdaConnectionPool object with the given #GnomeDbExport
 * widget. By doing this, you force the #GnomeDbExport widget to
 * use that connection pool to get the connections it needs, and
 * also reusing the ones that are already open on the connection
 * pool
 */
void
gnome_db_export_set_pool (GnomeDbExport *export, GdaConnectionPool *pool)
{
	g_return_if_fail (GNOME_DB_IS_EXPORT (export));

	if (IS_GDA_CONNECTION_POOL (export->priv->pool))
		gda_connection_pool_free (export->priv->pool);

	export->priv->pool = pool;
	if (IS_GDA_CONNECTION_POOL (export->priv->pool))
		gtk_object_ref (GTK_OBJECT (export->priv->pool));

	fill_connection_list (export);
}
