/* GNOME-DB Front End
 * Copyrigth (C) 2000 Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gnomedb-fe.h>

void
fe_config_load (void)
{
	/* generate default configuration if it's the first time */
	if (!gda_config_has_section(GNOMEDB_FE_CONFIG_SECTION_MAIN)) {
		gda_config_set_boolean(GNOMEDB_FE_CONFIG_KEY_SAVE_WINDOW_POS, TRUE);
		gda_config_set_boolean(GNOME_DB_CONFIG_KEY_SHOW_TOOLTIPS, TRUE);
	}

	/* now, default components to be loaded */
	if (!gda_config_has_section(GNOMEDB_FE_CONFIG_SECTION_LOAD)) {
		gda_config_set_string(GNOMEDB_FE_CONFIG_SECTION_LOAD "/" GNOME_DB_OAFIID_LOG_VIEWER, GNOME_DB_OAFIID_LOG_VIEWER);
		gda_config_set_string(GNOMEDB_FE_CONFIG_SECTION_LOAD "/" GNOME_DB_OAFIID_MANAGER, GNOME_DB_OAFIID_MANAGER);
		gda_config_set_string(GNOMEDB_FE_CONFIG_SECTION_LOAD "/" GNOME_DB_OAFIID_CONNECTION, GNOME_DB_OAFIID_CONNECTION);
		gda_config_set_string(GNOMEDB_FE_CONFIG_SECTION_LOAD "/" GNOME_DB_OAFIID_DESIGNER, GNOME_DB_OAFIID_DESIGNER);
	}
}

void
fe_config_save (void)
{
	/* save UI settings */
	if (gda_config_get_boolean(GNOMEDB_FE_CONFIG_KEY_SAVE_WINDOW_POS))
		gnome_db_save_window_config("MainWindow", glb_mainwin);
}

/*
 * Configuration dialog
 */
static void
fill_shell_components_list (GtkCList *clist)
{
	GList* shell_com_list;
	
	g_return_if_fail(GTK_IS_CLIST(clist));
	
	gnome_db_clear_clist(clist);
	shell_com_list = gnome_db_config_get_shell_components();
	if (shell_com_list) {
		GList* node;
		
		for (node = g_list_first(shell_com_list); node; node = g_list_next(node)) {
			GnomeDbConfigComponent* com = (GnomeDbConfigComponent *) node->data;
			if (com) {
				gchar* row[3] = { "", "", "" };
				gchar* str;
				gint   row_number;
				
				row[1] = gnome_db_config_component_get_description(com);
				row[2] = gnome_db_config_component_get_name(com);
				row_number = gtk_clist_append(GTK_CLIST(clist), row);
				
				str = g_strdup_printf("%s/%s",
				                      GNOMEDB_FE_CONFIG_SECTION_LOAD,
				                      gnome_db_config_component_get_name(com));
				if (gda_config_has_key(str)) {
					GnomePixmap* pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_EXEC);
					gtk_clist_set_pixmap(GTK_CLIST(clist), row_number, 0,
					                     pixmap->pixmap, pixmap->mask);
				}
				else {
					GnomePixmap* pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_STOP);
					gtk_clist_set_pixmap(GTK_CLIST(clist), row_number, 0,
					                     pixmap->pixmap, pixmap->mask);
				}
				g_free((gpointer) str);
			}
		}
	}
}

static void
set_load_component_cb (GtkWidget *w, GtkCList *clist)
{
	GnomePixmap* pixmap;
	GList*       selected;
	
	g_return_if_fail(GTK_IS_CLIST(clist));
	
	selected = clist->selection;
	if (selected) {
		pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_EXEC);
		if (pixmap) {
			gtk_clist_set_pixmap(GTK_CLIST(clist), GPOINTER_TO_INT(selected->data),
			                     0, pixmap->pixmap, pixmap->mask);
		}
	}
}

static void
set_unload_component_cb (GtkWidget *w, GtkCList *clist)
{
	GnomePixmap* pixmap;
	GList*       selected;
	
	g_return_if_fail(GTK_IS_CLIST(clist));
	
	selected = clist->selection;
	if (selected) {
		pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_STOP);
		if (pixmap) {
			gtk_clist_set_pixmap(GTK_CLIST(clist), GPOINTER_TO_INT(selected->data),
			                     0, pixmap->pixmap, pixmap->mask);
		}
	}
}

static GtkWidget *
components_init (GtkNotebook *notebook)
{
	GtkWidget* table;
	GtkWidget* scroll;
	GtkWidget* clist;
	GtkWidget* button;
	gchar*     clist_titles[] = { N_("Load"), N_("Component"), N_("OAFIID") };
	
	g_return_val_if_fail(GTK_IS_NOTEBOOK(notebook), NULL);
	
	table = gnome_db_new_table_widget(4, 6, FALSE);
	
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 3, 0, 6,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
	clist = gnome_db_new_clist_widget(clist_titles, 3);
	gtk_object_set_data(GTK_OBJECT(table), "FE_Config_ShellComponentList", (gpointer) clist);
	fill_shell_components_list(GTK_CLIST(clist));
	gtk_container_add(GTK_CONTAINER(scroll), clist);
	
	button = gnome_db_new_button_widget_with_pixmap(_("Load"), GNOME_STOCK_MENU_OPEN);
	gtk_signal_connect(GTK_OBJECT(button),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(set_load_component_cb),
	                   (gpointer) clist);
	gtk_table_attach(GTK_TABLE(table), button, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	
	button = gnome_db_new_button_widget_with_pixmap(_("Unload"), GNOME_STOCK_MENU_CLOSE);
	gtk_signal_connect(GTK_OBJECT(button),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(set_unload_component_cb),
	                   (gpointer) clist);
	gtk_table_attach(GTK_TABLE(table), button, 3, 4, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
	
	gtk_notebook_append_page(notebook, table, gtk_label_new(_("Components")));
	return table;
}

static void
components_save (GtkWidget *tab)
{
	GtkWidget*   clist;
	GnomePixmap* active_pixmap;
	
	active_pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_EXEC);
	if (!active_pixmap) return;
	
	clist = GTK_CLIST(gtk_object_get_data(GTK_OBJECT(tab), "FE_Config_ShellComponentList"));
	if (GTK_IS_CLIST(clist)) {
		GdkPixmap* pixmap = NULL;
		GdkBitmap* mask = NULL;
		gint       cnt;
		
		for (cnt = 0; cnt < GTK_CLIST(clist)->rows; cnt++) {
			gtk_clist_get_pixmap(GTK_CLIST(clist), cnt, 0, &pixmap, &mask);
			if (pixmap && mask) {
				gchar* name = NULL;
				gchar* str;
				
				/* traverse all rows in clist to save configuration */
				gtk_clist_get_text(GTK_CLIST(clist), cnt, 2, &name);
				if (!name) continue;
				
				str = g_strdup_printf("%s/%s", GNOMEDB_FE_CONFIG_SECTION_LOAD, name);
				if (GNOME_PIXMAP(active_pixmap)->pixmap == pixmap &&
				    GNOME_PIXMAP(active_pixmap)->mask == mask) {
					gda_config_set_string(str, str);
				}
				else gda_config_remove_key(str);
				g_free((gpointer) str);
			}
		}
	}
}

static GtkWidget *
general_init (GtkNotebook *notebook)
{
	GtkWidget* table;
	GtkWidget* save_window_pos;
	GtkWidget* show_tooltips;
	
	g_return_val_if_fail(GTK_IS_NOTEBOOK(notebook), NULL);
	
	table = gnome_db_new_table_widget(4, 10, FALSE);
	gtk_notebook_append_page(notebook, table, gtk_label_new(_("General")));

	/* 'save window position' option */
	save_window_pos = gnome_db_new_check_button_widget(_("Save window position"),
	                                                   gda_config_get_boolean(GNOMEDB_FE_CONFIG_KEY_SAVE_WINDOW_POS));
	gnome_db_set_widget_tooltip(save_window_pos, _("If enabled, save the window position on exit"));
	gtk_object_set_data(GTK_OBJECT(table), "FE_Config_SaveWindowPos", (gpointer) save_window_pos);
	gtk_table_attach(GTK_TABLE(table), save_window_pos, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	/* 'show tooltips' option */
	show_tooltips = gnome_db_new_check_button_widget(_("Show tooltips"),
													 gda_config_get_boolean(GNOME_DB_CONFIG_KEY_SHOW_TOOLTIPS));
	gnome_db_set_widget_tooltip(show_tooltips, _("Show tooltips for UI widgets"));
	gtk_object_set_data(GTK_OBJECT(table), "FE_Config_ShowTooltips", (gpointer) show_tooltips);
	gtk_table_attach(GTK_TABLE(table), show_tooltips, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
 
	return table;
}

static void
general_save (GtkWidget *tab)
{
	GtkWidget* w;
	
	g_return_if_fail(GTK_IS_WIDGET(tab));
	
	/* SaveWindowPos -> save main window position for future sessions */
	w = gtk_object_get_data(GTK_OBJECT(tab), "FE_Config_SaveWindowPos");
	if (GTK_IS_CHECK_BUTTON(w)) {
		gda_config_set_boolean(GNOMEDB_FE_CONFIG_KEY_SAVE_WINDOW_POS,
		                       gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)));
	}

	/* ShowTooltips -> show tooltips on UI widgets */
	w = gtk_object_get_data(GTK_OBJECT(tab), "FE_Config_ShowTooltips");
	if (GTK_IS_CHECK_BUTTON(w)) {
		gda_config_set_boolean(GNOME_DB_CONFIG_KEY_SHOW_TOOLTIPS,
							   gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)));
	}
}

void
fe_config_open_dialog (void)
{
	GtkWidget* dialog;
	GtkWidget* notebook;
	GtkWidget* com_tab;
	GtkWidget* general_tab;
	
	/* create dialog */
	dialog = gnome_dialog_new(_("Preferences"), GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL, NULL);
	gtk_widget_set_usize(dialog, 400, 300);
	notebook = gnome_db_new_notebook_widget();
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), notebook, 1, 1, 0);
	
	general_tab = general_init(GTK_NOTEBOOK(notebook));
	com_tab = components_init(GTK_NOTEBOOK(notebook));
	
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		general_save(general_tab);
		components_save(com_tab);
	}
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

