/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome-db-label.h>

struct _GnomeDbLabelPrivate {
	GdaRecordset* recset;
	gint           col;
};

/*
 * Callbacks
 */
static void
row_changed_cb (GdaRecordset *rs, GnomeDbLabel *label) {
	GdaField* field;
	
	g_return_if_fail(IS_GDA_RECORDSET(rs));
	g_return_if_fail(GNOME_DB_IS_LABEL(label));
	g_return_if_fail(label->priv->recset == rs);
	
	field = gda_recordset_field_idx(label->priv->recset, label->priv->col);
	if (field) {
		gnome_db_label_set_text(label, gda_stringify_value(0, 0, field));
	}
}

/*
 * Private functions
 */
static void
_gnome_db_label_destroy (GnomeDbLabel *label, gpointer user_data) {
	g_return_if_fail(GNOME_DB_IS_LABEL(label));
	
	if (label->priv) {
		if (IS_GDA_RECORDSET(label->priv->recset))
			gtk_object_unref(GTK_OBJECT(label->priv->recset));
		g_free((gpointer) label->priv);
	}
}

static void
_gnome_db_label_class_init (GnomeDbLabelClass *klass) {
}

static void
_gnome_db_label_init (GnomeDbLabel *label) {
	g_return_if_fail(GNOME_DB_IS_LABEL(label));
	
	label->priv = g_new0(GnomeDbLabelPrivate, 1);
	gtk_signal_connect(GTK_OBJECT(label),
	                   "destroy",
	                   GTK_SIGNAL_FUNC(_gnome_db_label_destroy),
	                   NULL);
}

/*
 * Public functions
 */
GtkType
gnome_db_label_get_type (void) {
	static GtkType db_label_type = 0;
	
	if (!db_label_type) {
		GtkTypeInfo db_label_info = {
			"GnomeDbLabel",
			sizeof (GnomeDbLabel),
			sizeof (GnomeDbLabelClass),
			(GtkClassInitFunc) _gnome_db_label_class_init,
			(GtkObjectInitFunc) _gnome_db_label_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		db_label_type = gtk_type_unique(gtk_label_get_type(), &db_label_info);
	}
	return (db_label_type);
}

/**
 * gnome_db_label_new
 * @recset: recordset to bind to label
 * @col: column index in recordset
 */
GtkWidget *
gnome_db_label_new (GdaRecordset *recset, gint col) {
	GnomeDbLabel* label;
	
	label = GNOME_DB_LABEL(gtk_type_new(gnome_db_label_get_type()));
	if (IS_GDA_RECORDSET(recset))
		gnome_db_label_set_recordset(label, recset, col);
	return GTK_WIDGET(label);
}

/**
 * gnome_db_label_get_recordset
 */
GdaRecordset *
gnome_db_label_get_recordset (GnomeDbLabel *label) {
	g_return_val_if_fail(GNOME_DB_IS_LABEL(label), NULL);
	g_return_val_if_fail(label->priv != NULL, NULL);
	
	return label->priv->recset;
}

/**
 * gnome_db_label_set_recordset
 */
void
gnome_db_label_set_recordset (GnomeDbLabel *label, GdaRecordset *recset, gint col) {
	g_return_if_fail(GNOME_DB_IS_LABEL(label));
	g_return_if_fail(label->priv != NULL);
	
	/* remove the currently active recordset */
	if (IS_GDA_RECORDSET(label->priv->recset)) {
		gtk_signal_disconnect(GTK_OBJECT(label->priv->recset),
		                      gtk_signal_lookup("row_changed", gda_recordset_get_type()));
		gtk_object_unref(GTK_OBJECT(label->priv->recset));
	}
	
	/* set the new recordset */
	label->priv->recset = recset;
	label->priv->col = col;
	gtk_signal_connect(GTK_OBJECT(label->priv->recset),
	                   "row_changed",
	                   GTK_SIGNAL_FUNC(row_changed_cb),
	                   (gpointer) label);
	gtk_object_ref(GTK_OBJECT(label->priv->recset));
}

/**
 * gnome_db_label_get_text
 */
const gchar *
gnome_db_label_get_text (GnomeDbLabel *label) {
	gchar* str = NULL;
	
	g_return_val_if_fail(GNOME_DB_IS_LABEL(label), NULL);
	gtk_label_get(GTK_LABEL(label), &str);
	return (const gchar *) str;
}

/**
 * gnome_db_label_set_text
 */
void
gnome_db_label_set_text (GnomeDbLabel *label, const gchar *txt) {
	g_return_if_fail(GNOME_DB_IS_LABEL(label));
	gtk_label_set_text(GTK_LABEL(label), txt);
}

