/*  GNOME-DB Components
 *  Copyright (C) 2000 Rodrigo Moya
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gnome-db-control.h"
#include "gnomedb-com.h"
#include "e-vpaned.h"

#define CONNECTION_INSTANCE_DATA "ConnectionInstanceData"

typedef struct {
	GnomeDbControl* control;
	GtkWidget*      tab;
	GtkWidget*      notebook;
	GList*          open_cncs;
} connection_control_data_t;

typedef struct {
	connection_control_data_t* control_data;
	GdaConnection*             cnc;
	GtkWidget*                 tab;
	gchar*                     gda_dsn;
	
	/* SQL tab */
	GtkWidget*                 sql_tab;
	GtkWidget*                 sql_cmd;
	GtkWidget*                 sql_grid;
	
	/* browser tab */
	GtkWidget*                 browser_tab;

	/* batch tab */
	GtkWidget*                 batch_tab;
	GtkWidget*                 batch_file_list;
	GtkWidget*                 batch_transaction;
	GtkWidget*                 batch_log;
} connection_instance_data_t;

static void activate_batch_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void activate_browser_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void activate_sql_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void begin_transaction_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void commit_transaction_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void disconnect_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void export_database_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void open_command_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void open_connection_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void rollback_transaction_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void save_command_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);
static void show_connection_properties_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path);

static void do_copy_cb  (GnomeDbControl *control, connection_control_data_t *control_data);
static void do_cut_cb  (GnomeDbControl *control, connection_control_data_t *control_data);
static void do_paste_cb (GnomeDbControl *control, connection_control_data_t *control_data);
static void do_print_cb (GnomeDbControl *control, connection_control_data_t *control_data);

static void grid_loaded_cb (GnomeDbGrid *grid, gpointer user_data);

static void sql_clear_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_copy_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_cut_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_open_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_paste_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_run_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_save_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_select_cb (GtkWidget *w, connection_instance_data_t *instance_data);

/* menu verbs */
BonoboUIVerb connection_verbs [] = {
	BONOBO_UI_VERB ("DatabaseOpenCommand", open_command_cb),
	BONOBO_UI_VERB ("DatabaseSaveCommand", save_command_cb),
	BONOBO_UI_VERB ("DatabaseConnect", open_connection_cb),
	BONOBO_UI_VERB ("DatabaseDisconnect", disconnect_cb),
	BONOBO_UI_VERB ("DatabaseBrowser", activate_browser_cb),
	BONOBO_UI_VERB ("DatabaseSql", activate_sql_cb),
	BONOBO_UI_VERB ("DatabaseBeginTrans", begin_transaction_cb),
	BONOBO_UI_VERB ("DatabaseCommit", commit_transaction_cb),
	BONOBO_UI_VERB ("DatabaseRollback", rollback_transaction_cb),
	BONOBO_UI_VERB ("DatabaseProperties", show_connection_properties_cb),
	BONOBO_UI_VERB ("DatabaseToolsExport", export_database_cb),
	BONOBO_UI_VERB ("DatabaseToolsImport", NULL),
	BONOBO_UI_VERB_END
};
/*
 * Private functions
 */

static void
fill_saved_query_list (GtkCList *clist)
{
	GList* commands;

	g_return_if_fail(GTK_IS_CLIST(clist));

	commands = gnome_db_config_get_saved_sql_queries();
	if (commands) {
		GList* node;

		gtk_clist_freeze(clist);
		for (node = g_list_first(commands); node != NULL; node = g_list_next(node)) {
			gchar* row[2];
			gchar* tmp;

			row[0] = (gchar *) node->data;
			tmp = g_strdup_printf("%s/%s",
					      GNOME_DB_CONFIG_SECTION_SAVED_SQL_QUERIES,
					      (gchar *) node->data);
			row[1] = gda_config_get_string(tmp);
			g_free((gpointer) tmp);
			gtk_clist_append(clist, row);
		}
		gtk_clist_thaw(clist);
		gda_config_free_list(commands);
	}
}

static connection_instance_data_t *
get_current_connection_from_control (GnomeDbControl *control)
{
	connection_control_data_t* control_data;
	
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	
	control_data = gnome_db_control_get_user_data(control);
	if (control_data) {
		gint       page_num;
		GtkWidget* w;
		
		if (GTK_IS_NOTEBOOK(control_data->notebook)) {
			page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(control_data->notebook));
			w = gtk_notebook_get_nth_page(GTK_NOTEBOOK(control_data->notebook), page_num);
			if (GTK_IS_WIDGET(w))
				return gtk_object_get_data(GTK_OBJECT(w), CONNECTION_INSTANCE_DATA);
		}
	}
	return NULL;
}

static void
init_batch_window (connection_instance_data_t *instance_data)
{
	GtkWidget*   frame;
	GtkWidget*   table;
	GtkWidget*   scroll;
	const gchar* file_titles[] = { N_("Select"), N_("State"), N_("File name") };
	gboolean     has_transactions;

	g_return_if_fail(instance_data != NULL);

	has_transactions = gda_connection_supports(instance_data->cnc,
	                                           GDA_Connection_FEATURE_TRANSACTIONS);
	instance_data->batch_tab = gnome_db_new_table_widget(4, 4, FALSE);

	/* create tollbar */

	/* create file options area */
	frame = gnome_db_new_frame_widget(_("Options"));
	gtk_table_attach(GTK_TABLE(instance_data->batch_tab), frame, 0, 4, 1, 2,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);
	table = gnome_db_new_table_widget(4, 4, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 3, 0, 2,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);
	instance_data->batch_file_list = gnome_db_new_clist_widget(file_titles, 3);
	gtk_container_add(GTK_CONTAINER(scroll), instance_data->batch_file_list);

	instance_data->batch_transaction = gnome_db_new_check_button_widget(_("Transaction mode"), has_transactions);
	gtk_table_attach(GTK_TABLE(table), instance_data->batch_transaction, 3, 4, 0, 1,
	                 GTK_FILL, GTK_FILL, 3, 3);
	if (!has_transactions)
		gtk_widget_set_sensitive(instance_data->batch_transaction, FALSE);

	/* create log window */
	frame = gnome_db_new_frame_widget(_("Log"));
	gtk_table_attach(GTK_TABLE(instance_data->batch_tab), frame, 0, 4, 2, 3,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_container_add(GTK_CONTAINER(frame), scroll);
	instance_data->batch_log = gnome_db_new_text_widget();
	gtk_text_set_editable(GTK_TEXT(instance_data->batch_log), FALSE);
	gtk_container_add(GTK_CONTAINER(scroll), instance_data->batch_log);

	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->tab),
	                         instance_data->batch_tab,
	                         gtk_label_new(_("Batch")));
}

static void
init_browser_window (connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	
	instance_data->browser_tab = gnome_db_browser_new(instance_data->cnc);
	gtk_widget_show(instance_data->browser_tab);
	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->tab),
	                         instance_data->browser_tab,
	                         gtk_label_new(_("Browser")));
}

static GnomeUIInfo sql_toolbar[] = {
	{ GNOME_APP_UI_ITEM, N_("Run"), N_("Execute current command"),
	  sql_run_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Select"), N_("Select command from previous"),
	  sql_select_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_UP, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Open"), N_("Open SQL command"),
	  sql_open_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Save"), N_("Save command for later retrieval"),
	  sql_save_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text"),
	  sql_cut_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_CUT, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text"),
	  sql_copy_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_COPY, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste text from clipboard"),
	  sql_paste_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_PASTE, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear widget contents"),
	  sql_clear_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
	GNOMEUIINFO_END
};

static void
init_sql_window (connection_instance_data_t *instance_data)
{
	GtkWidget* toolbar;
	GtkWidget* scroll;
	GtkWidget* paned;
	
	g_return_if_fail(instance_data != NULL);
	
	instance_data->sql_tab = gnome_db_new_table_widget(1, 2, FALSE);
	toolbar = gnome_db_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
	                                      GTK_TOOLBAR_ICONS,
	                                      sql_toolbar,
	                                      (gpointer) instance_data);
	gtk_table_attach(GTK_TABLE(instance_data->sql_tab), toolbar, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	paned = e_vpaned_new();
	gtk_widget_show(paned);
	gtk_table_attach(GTK_TABLE(instance_data->sql_tab), paned, 0, 1, 1, 2,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 3, 3);

	scroll = gnome_db_new_scrolled_window_widget();
	e_paned_add1(E_PANED(paned), scroll);
	instance_data->sql_cmd = gnome_db_new_text_widget();
	gtk_container_add(GTK_CONTAINER(scroll), instance_data->sql_cmd);
  
	instance_data->sql_grid = GTK_WIDGET(gnome_db_new_grid_widget(NULL));
	gtk_signal_connect (GTK_OBJECT (instance_data->sql_grid),
			    "loaded",
			    GTK_SIGNAL_FUNC (grid_loaded_cb),
			    instance_data->control_data->control);
	e_paned_add2(E_PANED(paned), instance_data->sql_grid);
	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->tab),
	                         instance_data->sql_tab,
	                         gtk_label_new(_("SQL")));
}

/*
 * Callbacks
 */
static void
activate_batch_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->tab),
		                      gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->tab),
		                                            instance_data->batch_tab));
	}
}

static void
activate_browser_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->tab),
		                      gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->tab),
		                                            instance_data->browser_tab));
	}
}

static void
activate_sql_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->tab),
		                      gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->tab),
		                                            instance_data->sql_tab));
	}
}

static void
begin_transaction_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		if (gda_connection_supports(instance_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS)) {
			if (gda_connection_begin_transaction(instance_data->cnc) == 0) {
				gnome_db_control_set_status (control,
							     _("Transaction started"));
			}
		}
		else
			gnome_db_show_error (_("Transactions not supported"));
	}
}

static void
commit_transaction_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		if (gda_connection_supports(instance_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS)) {
			if (gda_connection_commit_transaction(instance_data->cnc) == 0) {
				gnome_db_control_set_status (control,
							     _("Transaction finished successfully"));
			}
		}
		else gnome_db_show_error(_("Transactions not supported"));
	}
}

static void
disconnect_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;

	g_return_if_fail(GNOME_DB_IS_CONTROL(control));

	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		/* remove this connection from our list */
		if (instance_data->control_data) {
			instance_data->control_data->open_cncs =
				g_list_remove(instance_data->control_data->open_cncs,
							  (gpointer) instance_data);
			gtk_notebook_remove_page (
				GTK_NOTEBOOK(instance_data->control_data->notebook),
				gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->control_data->notebook),
						      instance_data->tab));
		}

		components_connection_close(instance_data->cnc);
		g_free((gpointer) instance_data->gda_dsn);
		g_free((gpointer) instance_data);

		gnome_db_control_set_status (control, _("Connection closed"));
	}
}

static void
do_copy_cb (GnomeDbControl *control, connection_control_data_t *control_data)
{
	gint                        current_page;
	GtkWidget*                  tab;
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(instance_data->tab));
		tab = gtk_notebook_get_nth_page(GTK_NOTEBOOK(instance_data->tab), current_page);
		if (tab == instance_data->sql_tab)
			sql_copy_cb(NULL, instance_data);
		else if (tab == instance_data->browser_tab) ;
	}
}

static void
do_cut_cb (GnomeDbControl *control, connection_control_data_t *control_data)
{
	gint                        current_page;
	GtkWidget*                  tab;
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(instance_data->tab));
		tab = gtk_notebook_get_nth_page(GTK_NOTEBOOK(instance_data->tab), current_page);
		if (tab == instance_data->sql_tab)
			sql_cut_cb(NULL, instance_data);
		else if (tab == instance_data->browser_tab) ;
	}
}

static void
do_paste_cb (GnomeDbControl *control, connection_control_data_t *control_data)
{
	gint                        current_page;
	GtkWidget*                  tab;
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(instance_data->tab));
		tab = gtk_notebook_get_nth_page(GTK_NOTEBOOK(instance_data->tab), current_page);
		if (tab == instance_data->sql_tab)
			sql_paste_cb(NULL, instance_data);
		else if (tab == instance_data->browser_tab) ;
	}
}

static void
do_print_cb (GnomeDbControl *control, connection_control_data_t *control_data)
{
	gint                        current_page;
	GtkWidget*                  tab;
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(instance_data->tab));
		tab = gtk_notebook_get_nth_page(GTK_NOTEBOOK(instance_data->tab), current_page);
		if (tab == instance_data->sql_tab) ; // gnome_db_grid_print
		else if (tab == instance_data->browser_tab) ;
	}
}

static void
export_database_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	GtkWidget *dialog;
	GtkWidget *export;

	/* create the dialog box */
	dialog = gnome_dialog_new (_("Export database"),
				   GNOME_STOCK_BUTTON_CLOSE,
				   NULL);
	gtk_signal_connect (GTK_OBJECT (dialog),
			    "clicked",
			    GTK_SIGNAL_FUNC (gnome_dialog_close),
			    NULL);

	export = gnome_db_export_new ();
	gtk_widget_show (export);
	gnome_db_export_set_pool (export, components_connection_get_pool ());
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), export, TRUE, TRUE, 0);

	gtk_widget_show (dialog);
}

static void
grid_loaded_cb (GnomeDbGrid *grid, gpointer user_data)
{
	gchar *str;
	gint row_count;
	GnomeDbControl *control = (GnomeDbControl *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (GNOME_DB_IS_CONTROL (control));

	row_count = gnome_db_grid_get_row_count (grid);
	if (row_count == 1)
		str = g_strdup (_("1 row returned"));
	else
		str = g_strdup_printf (_("%d rows returned"), row_count);

	gnome_db_control_set_status (control, str);
	g_free (str);
}

static void
open_command_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	g_return_if_fail (GNOME_DB_IS_CONTROL (control));
	sql_open_cb (NULL, get_current_connection_from_control (control));
}

static void
open_connection_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;
	GtkWidget*                  dialog;
	GtkWidget*                  login;

	g_return_if_fail(GNOME_DB_IS_CONTROL(control));

	instance_data = g_new0(connection_instance_data_t, 1);
	instance_data->control_data = gnome_db_control_get_user_data(control);

	/* create login dialog box */
	dialog = gnome_dialog_new(_("Open connection"),
				  GNOME_STOCK_BUTTON_OK,
				  GNOME_STOCK_BUTTON_CANCEL,
				  NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	login = gnome_db_login_new(NULL);
	gtk_widget_show(login);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), login, 1, 1, 0);

	/* run dialog */
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		instance_data->cnc =
			components_connection_open(gnome_db_login_get_gda_name(GNOME_DB_LOGIN(login)),
						   gnome_db_login_get_username(GNOME_DB_LOGIN(login)),
						   gnome_db_login_get_password(GNOME_DB_LOGIN(login)));
		if (instance_data->cnc) {
			gchar* label;

			instance_data->gda_dsn = g_strdup(gnome_db_login_get_gda_name(GNOME_DB_LOGIN(login)));

			/* create tab widget (a GtkNotebook) */
			instance_data->tab = gnome_db_new_notebook_widget();
			gtk_notebook_set_show_tabs(GTK_NOTEBOOK(instance_data->tab), FALSE);
			gtk_notebook_set_show_border(GTK_NOTEBOOK(instance_data->tab), FALSE);
			gtk_notebook_popup_disable(GTK_NOTEBOOK(instance_data->tab));

			init_browser_window(instance_data);
			init_sql_window(instance_data);
			init_batch_window(instance_data);
	
			/* add this instance to list of open connections */
			if (instance_data->control_data) {
				gint page_num;

				instance_data->control_data->open_cncs =
					g_list_append(instance_data->control_data->open_cncs,
						      (gpointer) instance_data);
				label = g_strdup_printf("%s@%s", gda_connection_get_user(instance_data->cnc),
							instance_data->gda_dsn);
				gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->control_data->notebook),
							 instance_data->tab,
							 gtk_label_new(label));
				g_free((gpointer) label);

				gtk_object_set_data(GTK_OBJECT(instance_data->tab), CONNECTION_INSTANCE_DATA, instance_data);
				page_num = gtk_notebook_page_num (
					GTK_NOTEBOOK(instance_data->control_data->notebook),
					instance_data->tab);
				gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->control_data->notebook), page_num);
			}

			gnome_db_control_set_status (control, _("Connection opened"));
		}
		else
			g_free((gpointer) instance_data);
	}
	else
		g_free((gpointer) instance_data);
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
rollback_transaction_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		if (gda_connection_supports(instance_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS)) {
			if (gda_connection_rollback_transaction(instance_data->cnc) == 0) {
				gnome_db_control_set_status (control,
							     _("Transaction aborted"));
			}
		}
		else gnome_db_show_error(_("Transactions not supported"));
	}
}

static void
show_connection_properties_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	connection_instance_data_t* instance_data;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	instance_data = get_current_connection_from_control(control);
	if (instance_data) {
		GtkWidget* dialog;
		GtkWidget* table;
		GtkWidget* label;
		GtkWidget* entry;
		GdaDsn*   dsn_info;
		
		dialog = gnome_dialog_new(_("Connection Properties"),
		                          GNOME_STOCK_BUTTON_CLOSE,
		                          NULL);
		table = gnome_db_new_table_widget(3, 6, FALSE);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table, 1, 1, 0);
		
		label = gnome_db_new_label_widget(_("GDA Name"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), instance_data->gda_dsn);
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
		
		dsn_info = gda_dsn_find_by_name(instance_data->gda_dsn);
		label = gnome_db_new_label_widget(_("Provider"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), GDA_DSN_PROVIDER(dsn_info));
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
		
		label = gnome_db_new_label_widget(_("User name"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_user(instance_data->cnc));
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
		
		label = gnome_db_new_label_widget(_("Version"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_version(instance_data->cnc));
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
		
		gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
	}
}

static void
save_command_cb (BonoboUIComponent *uic, GnomeDbControl *control, const gchar *path)
{
	g_return_if_fail (GNOME_DB_IS_CONTROL (control));
	sql_save_cb (NULL, get_current_connection_from_control (control));
}

static void
sql_clear_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
	
	gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
	gnome_db_grid_set_recordset(GNOME_DB_GRID(instance_data->sql_grid), NULL);
}

static void
sql_copy_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
	
	gtk_editable_copy_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
	gnome_db_control_set_status (instance_data->control_data->control,
				     _("Text copied to clipboard"));
}

static void
sql_cut_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
	
	gtk_editable_cut_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
	gnome_db_control_set_status (instance_data->control_data->control,
				     _("Text cut into the clipboard"));
}

static void
sql_open_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	GtkWidget*   dialog;
	GtkWidget*   table;
	GtkWidget*   scroll;
	GtkWidget*   label;
	GtkWidget*   list;
	GtkWidget*   from_conf;
	GtkWidget*   from_file;
	GtkWidget*   from_file_name;
	const gchar* saved_titles[] = { N_("Name"), N_("Command") };

	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));

	/* create dialog */
	dialog = gnome_dialog_new(_("Open SQL command"),
	                          GNOME_STOCK_BUTTON_OK,
	                          GNOME_STOCK_BUTTON_CANCEL,
	                          NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	gtk_widget_set_usize(dialog, 400, 250);
	table = gnome_db_new_table_widget(3, 3, FALSE);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table, 1, 1, 0);

	/* from config stuff */
	from_conf = gnome_db_new_radio_button_widget(_("From saved query"), NULL);
	gtk_table_attach(GTK_TABLE(table), from_conf, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 3, 1, 2,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);

	list = gnome_db_new_clist_widget(saved_titles, 2);
	fill_saved_query_list(GTK_CLIST(list));
	gtk_container_add(GTK_CONTAINER(scroll), list);

	/* from file stuff */
	from_file = gnome_db_new_radio_button_widget(_("From file"), GTK_RADIO_BUTTON(from_conf));
	gtk_table_attach(GTK_TABLE(table), from_file, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
	from_file_name = gnome_db_new_file_entry_widget("GNOME_DB_Database_FileName");
	gtk_table_attach(GTK_TABLE(table), from_file_name, 1, 3, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

	/* run the dialog */
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(from_file))) {
			gchar* filename = gtk_entry_get_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(from_file_name))));
			if (filename) {
				gchar* text;

				/* load command(s) from file */
				text = gda_util_load_file(filename);
				if (text) {
					gtk_editable_delete_text (
						GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
					gtk_text_insert(GTK_TEXT(instance_data->sql_cmd),
							NULL, NULL, NULL, text, strlen(text));
					g_free((gpointer) text);
				}
				else
					gnome_db_show_error(_("Error loading file %s"), filename);
			}
		}
		else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(from_conf))) {
			GList* selection;
			
			/* get CList's selection */
			selection = GTK_CLIST(list)->selection;
			if (selection) {
				gchar* text = NULL;
				if (gtk_clist_get_text(GTK_CLIST(list),
				                       GPOINTER_TO_INT(selection->data), 1, &text)) {
					gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
					gtk_text_insert(GTK_TEXT(instance_data->sql_cmd), NULL, NULL, NULL, text, strlen(text));
				}
			}
		}
	}
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
sql_paste_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
	
	gtk_editable_paste_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
	gnome_db_control_set_status (instance_data->control_data->control,
				     _("Text pasted from the clipboard"));
}

static void
sql_run_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	gchar* sql;
	
	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_TEXT(instance_data->sql_cmd));
	
	sql = gtk_editable_get_chars(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
	if (sql) {
		gulong         reccount;
		GdaRecordset* recset;
		
		/* execute command */
		recset = gda_connection_execute(instance_data->cnc, sql, &reccount, 0);
		if (recset) {
			gnome_db_control_set_status (instance_data->control_data->control,
						     _("Command executed successfully"));
		}
		gnome_db_grid_set_recordset(GNOME_DB_GRID(instance_data->sql_grid), recset);
		gda_recordset_free (recset);
		components_config_add_last_command((const gchar *) sql);
	}
}

static void
sql_save_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	GtkWidget* dialog;
	GtkWidget* table;
	GtkWidget* to_query;
	GtkWidget* to_query_name;
	GtkWidget* to_file;
	GtkWidget* to_file_name;

	g_return_if_fail(instance_data != NULL);
	g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));

	/* create dialog */
	dialog = gnome_dialog_new(_("Save SQL command"),
	                          GNOME_STOCK_BUTTON_OK,
	                          GNOME_STOCK_BUTTON_CANCEL,
	                          NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	table = gnome_db_new_table_widget(3, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table, 1, 1, 0);

	/* to saved query stuff */
	to_query = gnome_db_new_radio_button_widget(_("To query"), NULL);
	gtk_table_attach(GTK_TABLE(table), to_query, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	to_query_name = gnome_db_new_entry_widget(0, TRUE);
	gnome_db_set_widget_tooltip(to_query_name, _("Save command to a named query"));
	gtk_table_attach(GTK_TABLE(table), to_query_name, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	/* to file stuff */
	to_file = gnome_db_new_radio_button_widget(_("To file"), GTK_RADIO_BUTTON(to_query));
	gtk_table_attach(GTK_TABLE(table), to_file, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
	to_file_name = gnome_db_new_file_entry_widget("GNOME_DB_Database_FileName");
	gnome_db_set_widget_tooltip(to_file_name, _("Select the file to save the command to"));
	gtk_table_attach(GTK_TABLE(table), to_file_name, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	/* run the dialog */
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		gchar* sql_txt = gtk_editable_get_chars(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);

		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(to_query))) {
			gchar* query_name = gtk_entry_get_text(GTK_ENTRY(to_query_name));

			/* save to "SQL Queries" configuration area */
			if (query_name) {
				gchar* key;

				key = g_strdup_printf("%s/%s", GNOME_DB_CONFIG_SECTION_SAVED_SQL_QUERIES, query_name);
				gda_config_set_string(key, sql_txt);
				g_free((gpointer) key);
			}
			else gnome_db_show_error(_("Query must have a name"));
		}
		else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(to_file))) {
			gchar* filename = gtk_entry_get_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(to_file_name))));
			
			/* save to file */
			if (filename) {
				if (!gda_util_save_file(filename, sql_txt))
					gnome_db_show_error(_("Could not save file %s"), filename);
			}
		}
	}
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
command_selected_cb (GtkCList *clist, gint row, gint col, GdkEvent *event, gpointer user_data)
{
	GtkText* sql = (GtkText *) user_data;
	gchar*   txt = NULL;

	g_return_if_fail(GTK_IS_CLIST(clist));
	g_return_if_fail(GTK_IS_TEXT(sql));

	if (gtk_clist_get_text(clist, row, 1, &txt)) {
		gtk_editable_delete_text(GTK_EDITABLE(sql), 0, -1);
		gtk_text_insert(GTK_TEXT(sql), NULL, NULL, NULL, txt, strlen(txt));
	}
}

static void
sql_select_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
	GtkWidget*   dialog;
	GtkWidget*   scroll;
	GtkWidget*   table;
	GtkWidget*   list;
	GtkWidget*   label;
	GtkWidget*   sql;
	GList*       commands;
	const gchar* cmd_titles[] = { "#", N_("Command") };
	const gchar* saved_titles[] = { N_("Name"), N_("Command") };

	g_return_if_fail(instance_data != NULL);

	/* create the dialog */
	dialog = gnome_dialog_new(_("Select SQL Command"),
	                          GNOME_STOCK_BUTTON_OK,
	                          GNOME_STOCK_BUTTON_CANCEL,
	                          NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	gtk_widget_set_usize(dialog, 500, 350);
	table = gnome_db_new_table_widget(2, 4, FALSE);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table, 1, 1, 0);

	/* create GtkText to show the current selection */
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 2, 3, 4,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);
	sql = gnome_db_new_text_widget();
	gnome_db_set_widget_tooltip(sql, _("Here is shown the currently selected command's text"));
	gtk_container_add(GTK_CONTAINER(scroll), sql);

	/* create list of last commands */
	label = gnome_db_new_label_widget(_("Last commands"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 1, 1, 3,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);

	list = gnome_db_new_clist_widget(cmd_titles, 2);
	gnome_db_set_widget_tooltip(list, _("This is a list of the last commands you've executed"));
	gtk_signal_connect(GTK_OBJECT(list),
	                   "select_row",
	                   GTK_SIGNAL_FUNC(command_selected_cb),
	                   (gpointer) sql);
	commands = components_config_get_last_commands();
	if (commands) {
		GList* node;
		gint   cnt;
		
		gtk_clist_freeze(GTK_CLIST(list));
		for (cnt = 1, node = g_list_first(commands); node; node = g_list_next(node), cnt++) {
			gchar* row[2];
			
			row[0] = g_strdup_printf("%d", cnt);
			row[1] = (gchar *) node->data;
			gtk_clist_append(GTK_CLIST(list), row);
		}
		gtk_clist_thaw(GTK_CLIST(list));
	}
	gtk_container_add(GTK_CONTAINER(scroll), list);
	
	/* create list of saved queries */
	label = gnome_db_new_label_widget(_("Saved queries"));
	gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 1, 2, 1, 3,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	                 3, 3);

	list = gnome_db_new_clist_widget(saved_titles, 2);
	gnome_db_set_widget_tooltip(list, _("This is a list of all the named queries you've stored"));
	gtk_signal_connect(GTK_OBJECT(list),
	                   "select_row",
	                   GTK_SIGNAL_FUNC(command_selected_cb),
	                   (gpointer) sql);
	fill_saved_query_list(GTK_CLIST(list));
	gtk_container_add(GTK_CONTAINER(scroll), list);

	/* run the dialog */
	if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
		gchar* txt;
		
		txt = gtk_editable_get_chars(GTK_EDITABLE(sql), 0, -1);
		if (txt) {
			gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
			gtk_text_insert(GTK_TEXT(instance_data->sql_cmd), NULL, NULL, NULL, txt, strlen(txt));
		}
	}
	gnome_dialog_close(GNOME_DIALOG(dialog));
}

/*
 * Public functions
 */
BonoboObject *
components_connection_new (void)
{
	GtkWidget*      tab;
	GnomeDbControl* control = NULL;
	
	tab = gnome_db_new_table_widget(4, 2, FALSE);
	control = gnome_db_control_new(tab);
	if (GNOME_DB_IS_CONTROL(control)) {
		connection_control_data_t* control_data;

		control_data = g_new0(connection_control_data_t, 1);
		control_data->control = control;
		control_data->tab = tab;

		gnome_db_control_set_ui (control_data->control,
					 GNOMEDB_DATADIR,
					 "gnomedb-com-connection.xml",
					 connection_verbs,
					 control);
		gnome_db_control_set_user_data(control_data->control, control_data);
		
		gtk_signal_connect(GTK_OBJECT(control_data->control),
		                   "do_print",
		                   GTK_SIGNAL_FUNC(do_print_cb),
		                   (gpointer) control_data);
		gtk_signal_connect(GTK_OBJECT(control_data->control),
		                   "do_cut_clipboard",
		                   GTK_SIGNAL_FUNC(do_cut_cb),
		                   (gpointer) control_data);
		gtk_signal_connect(GTK_OBJECT(control_data->control),
		                   "do_copy_clipboard",
		                   GTK_SIGNAL_FUNC(do_copy_cb),
		                   (gpointer) control_data);
		gtk_signal_connect(GTK_OBJECT(control_data->control),
		                   "do_paste_clipboard",
		                   GTK_SIGNAL_FUNC(do_paste_cb),
		                   (gpointer) control_data);

		/* create notebook */
		control_data->notebook = gnome_db_new_notebook_widget();
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(control_data->notebook), GTK_POS_BOTTOM);
		gtk_table_attach(GTK_TABLE(control_data->tab), control_data->notebook, 0, 4, 0, 2,
		                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		                 3, 3);
	}
	else gtk_widget_destroy (tab);
	
	return BONOBO_OBJECT (control);
}
