/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */
using GLib;
using Gee;
using Gst;

namespace DVB {

    public static delegate void ForcedStopNotify (Channel channel);

    /**
     * This class handles watching channels one physical device.
     * 
     * It's possible to watch multiple channels at the same time
     * if they are all on the same transport stream.
     *
     * The class is able to reuse channels that are already watched
     * and forward EPG data to #EPGScanner.
     */
    public class PlayerThread : GLib.Object {
        
        private class ChannelElements {
            public uint sid;
            public ArrayList<Gst.Element> sinks;
            public Gst.Element tee;
            public bool forced;
            public ForcedStopNotify notify_func;
        }

        /** 
         * Emitted when we came across EIT table
         */
        public signal void eit_structure (Gst.Structure structure);
        
        // List of channels that are currently in use
        public HashSet<Channel> active_channels {get; construct;}
        // The device in use
        public Device device {get; construct;}
        // Whether watching one of the channels was forced
        public bool forced {
            get {
                bool val = false;
                lock (this.elements_map) {
                    foreach (ChannelElements celem in this.elements_map.get_values ()) {
                        if (celem.forced) {
                            val = true;
                            break;
                        }
                    }
                }
                return val;
            }
        }
        
        private Element? pipeline;
        private HashMap<uint, ChannelElements> elements_map;
        private EPGScanner? epgscanner;
        private Element? dvbbasebin;
        private bool destroyed;
        
        construct {
            this.elements_map = new HashMap<uint, ChannelElements> ();
            this.active_channels = new HashSet<Channel> ();
            this.destroyed = false;
        }
        
        /**
         * @device: The device to use
         * @epgscanner: #EPGScanner to forward EIT to
         */
        public PlayerThread (Device device, EPGScanner? epgscanner) {
            this.device = device;
            this.epgscanner = epgscanner;
        }
        
        public Gst.Element? get_pipeline () {
            return this.pipeline;
        }
        
        public Gst.Element? get_dvbbasebin () {
            return this.dvbbasebin;
        }
        
        /**
         * @returns: GstBin containing queue and @sink_element 
         *
         * Start watching @channel and link it with @sink_element
         */
        public Gst.Element? get_element (Channel channel, owned Gst.Element sink_element,
                bool forced, ForcedStopNotify? notify_func) {
            uint channel_sid = channel.Sid;
            string channel_sid_str = channel_sid.to_string ();
            bool create_channel;
            
            Element? bin, tee = null;
            if (this.pipeline == null) {
                // New channel and new pipeline
                debug ("Creating new pipeline");
                lock (this.pipeline) {
                    // Setup new pipeline
                    this.pipeline = new Pipeline ("recording");
                
                    Gst.Bus bus = this.pipeline.get_bus();
                    bus.add_signal_watch();
                    bus.message += this.bus_watch_func;
                    
                    this.dvbbasebin = ElementFactory.make ("dvbbasebin", null);
                    if (this.dvbbasebin == null) {
                        critical ("Could not create dvbbasebin element");
                        return null;
                    }
                    this.dvbbasebin.pad_added += this.on_dvbbasebin_pad_added;
                    
                    channel.setup_dvb_source (this.dvbbasebin);
                    
                    this.dvbbasebin.set ("program-numbers", channel_sid_str);
                    this.dvbbasebin.set ("adapter", this.device.Adapter);
                    this.dvbbasebin.set ("frontend", this.device.Frontend);
                    
                    // don't use add_many because of problems with ownership transfer    
                    ((Bin) this.pipeline).add (this.dvbbasebin);
                    
                    tee = ElementFactory.make ("tee", null);
                    this.add_element (tee);
                    
                    bin = this.add_sink_bin (sink_element);
                    if (!tee.link (bin)) {
                        critical ("Could not link tee and bin");
                        return null;
                    }
                    
                    create_channel = true;
                }
            } else {
                // Use current pipeline and add new sink
                debug ("Reusing existing pipeline");
                if (this.dvbbasebin == null) {
                    critical ("No dvbbasebin element");
                    return null;
                }

                if (!this.active_channels.contains (channel)) {
                    // existing pipeline and new channel
                    lock (this.pipeline) {
                        tee = ElementFactory.make ("tee", null);
                        this.add_element (tee);
                        
                        bin = this.add_sink_bin (sink_element);
                        if (!tee.link (bin)) {
                            critical ("Could not link tee and bin");
                            return null;
                        }
                    
                        this.pipeline.set_state (State.PAUSED);
                        
                        string programs;
                        dvbbasebin.get ("program-numbers", out programs);
                        
                        string new_programs = "%s:%s".printf (programs,
                            channel_sid_str);
                        debug ("Changing program-numbers from %s to %s", programs,
                            new_programs);
                        this.dvbbasebin.set ("program-numbers", new_programs);
                        
                        create_channel = true;
                    }
                } else { // existing pipeline and existing channel
                    ChannelElements c_element;
                    lock (this.elements_map) {
                        c_element = this.elements_map.get (channel_sid);
                    }
                    tee = c_element.tee;

                    lock (this.pipeline) {
                        this.pipeline.set_state (State.PAUSED);
                        bin = this.add_sink_bin (sink_element);
                    }

                    debug ("Linking %s with %s", tee.get_name (), bin.get_name ());
                    if (!tee.link (bin)) {
                        critical ("Could not link tee and bin");
                        return null;
                    }

                    c_element.sinks.add (bin);
                    create_channel = false;
                }
            }
            
            if (create_channel) {
                ChannelElements celems = new ChannelElements ();
                celems.sid = channel_sid;
                celems.sinks = new ArrayList<Gst.Element> ();
                celems.sinks.add (bin);
                celems.tee = tee;
                celems.forced = forced;
                celems.notify_func = notify_func;

                lock (this.elements_map) {
                    this.elements_map.set (channel_sid, celems);
                }
                this.active_channels.add (channel);
            }
            
            return bin;
        }
        
        private Gst.Element add_sink_bin (Gst.Element sink_element) {
            Element queue = ElementFactory.make ("queue", null);
            queue.set ("max-size-buffers", 0);
            
            Gst.Element bin = new Gst.Bin (null);
            ((Gst.Bin)bin).add_many (queue, sink_element);
            if (!queue.link (sink_element)) {
                critical ("Could not link elements %s and %s", queue.get_name (),
                    sink_element.get_name ());
            }
            
            var pad = queue.get_static_pad ("sink");
            bin.add_pad (new Gst.GhostPad ("sink", pad));
            
            this.add_element (bin);

            return bin;
        }
            
        private static int find_element (void* av, void *bv) {
            Gst.Element a = (Gst.Element)av;
            Gst.Element b = (Gst.Element)bv;
            if (a == b) return 0;
            else return 1;
        }

        /**
         * @sid: Channel SID
         * @sink: The sink element that the bin should contain
         * @returns: GstBin containing queue and sink for the specified channel
         */
        public Gst.Element? get_sink_bin (uint sid, Gst.Element sink) {
            Gst.Element? result = null;
            debug ("Searching for sink %s (%p) of channel %u", sink.get_name (), sink, sid);
            lock (this.elements_map) {
                ChannelElements? celems = this.elements_map.get (sid);
                if (celems != null) {
                    foreach (Gst.Element sink_bin in celems.sinks) {
                        Gst.Iterator it = ((Gst.Bin)sink_bin).iterate_elements ();
                        Gst.Element element = (Gst.Element) it.find_custom (find_element, sink);
                        if (element != null) {
                            result = sink_bin;
                            break;
                        }
                    }
                }
            }
            
            return result;
        }
        
        /**
         * Stop watching @channel
         */
        public bool remove_channel (Channel channel, Gst.Element sink) {
            uint channel_sid = channel.Sid;
        
            if (!this.active_channels.contains (channel)) {
                critical ("Could not find channel with SID %u", channel_sid);
                return false;
            }
            
            // Check if that's the only channel in use
            if (this.active_channels.size > 1) {
                string channel_sid_string = channel_sid.to_string ();
                
                lock (this.pipeline) {
                    bool stop_channel;
                    lock (this.elements_map) {
                        ChannelElements celements = this.elements_map.get (channel_sid);
                        if (celements.sinks.size == 1) {
                            // this is the last sink
                            // (no one watches this channel anymore)
                            this.elements_map.remove (channel_sid);
                            stop_channel = true;
                        } else {
                            // we still have sinks left
                            // (others are still watching this channel)
                            celements.sinks.remove (sink);
                            stop_channel = false;
                        }
                    }
                   
                    if (stop_channel) { 
                        string programs;
                        dvbbasebin.get ("program-numbers", out programs);
                        string[] programs_arr = programs.split (":");
                        
                        // Remove SID of channel from program-numbers
                        SList<string> new_programs_list = new SList<string> ();
                        for (int i=0; i<programs_arr.length; i++) {
                            string val = programs_arr[i];
                            if (val != channel_sid_string)
                                new_programs_list.prepend (val);
                        }
                        
                        StringBuilder new_programs = new StringBuilder (new_programs_list.nth_data (0));
                        for (int i=1; i<new_programs_list.length (); i++) {
                            new_programs.append (":" + new_programs_list.nth_data (i));
                        }
                    
                        debug ("Changing program-numbers from %s to %s", programs,
                                new_programs.str);
                        this.pipeline.set_state (State.PAUSED);
                        
                        dvbbasebin.set ("program-numbers", new_programs.str);
                        
                        this.pipeline.set_state (State.PLAYING);
                        
                        this.active_channels.remove (channel);
                    }

                    this.remove_sink_bin (channel_sid, sink);
                }
            } else {
                lock (this.elements_map) {
                    ChannelElements celements = this.elements_map.get (channel_sid);
                    if (celements.sinks.size == 1) {
                        // this is the last sink
                        // (no one watches this channel anymore)
                        this.destroy ();
                    } else {
                        // we still have sinks left
                        // (others are still watching this channel)
                        this.remove_sink_bin (channel_sid, sink);
                        celements.sinks.remove (sink);
                    }
                }
            }
        
            return true;
        }
        
        private void remove_sink_bin (uint channel_sid, Gst.Element sink) {
            Gst.Element? sink_bin = this.get_sink_bin (channel_sid, sink);

            if (sink_bin == null) {
                warning ("Could not find sink bin for channel %u and sink %p",
                    channel_sid, sink);
                return;
            }
            debug ("Setting state of queue and sink to NULL");
            sink_bin.set_state (State.NULL);
            
            debug ("Removing queue and sink from pipeline");
            ((Bin)this.pipeline).remove (sink_bin);
        }
        
        /**
         * Stop pipeline and clean up everything else
         */
        public virtual void destroy (bool forced=false) {
            if (this.destroyed) return;
            lock (this.destroyed) {
                if (forced) {
                    lock (this.elements_map) {
                        foreach (ChannelElements celems in this.elements_map.get_values ()) {
                            if (celems.notify_func != null) {
                                Channel channel = this.device.Channels.get_channel (
                                    celems.sid);
                                celems.notify_func (channel);
                            }
                        }
                    }
                }

                lock (this.pipeline) {
                    if (this.pipeline != null) {
                        debug ("Stopping pipeline");
                        Gst.Bus bus = this.pipeline.get_bus ();
                        bus.remove_signal_watch ();
                        this.pipeline.set_state (State.NULL);
                    }
                    this.pipeline = null;
                }
                lock (this.elements_map) {
                    this.elements_map.clear ();
                }
                this.active_channels.clear ();
                this.destroyed = true;
            }
        }

        private bool add_element (owned Gst.Element elem) {
            string elem_name = elem.get_name ();
            if (!((Bin) this.pipeline).add (elem)) {
                critical ("Could not add element %s", elem_name);
                return false;
            }
            debug ("Element %s (%p) added to pipeline", elem_name, elem);
            return true;
        }
        
        /**
         * Link program_%d pad with tee
         */
        private void on_dvbbasebin_pad_added (Gst.Element elem, Gst.Pad pad) {
            string pad_name = pad.get_name();
            debug ("Pad %s added", pad_name);
            
            if (!pad_name.has_prefix ("program_"))
                return;
            
            uint sid;
            pad_name.scanf("program_%u", out sid);
            
            debug ("SID is '%u'", sid);
            // Check if we're interested in the pad
            lock (this.elements_map) {
                if (this.elements_map.contains (sid)) {
                    Element? sink = this.elements_map.get (sid).tee;
                    if (sink == null) {
                        critical ("Could not find sink for SID %u", sid);
                        return;
                    }
                    
                    debug ("Linking elements %s and %s", elem.get_name(), sink.get_name ());
                    Pad sinkpad = sink.get_static_pad ("sink");
                    
                    PadLinkReturn rc = pad.link (sinkpad);
                    if (rc != PadLinkReturn.OK) {
                        critical ("Could not link pads %s and %s", pad.get_name (),
                            sinkpad.get_name ());
                    } else {
                        debug ("Src pad %s linked with sink pad %s",
                            pad.get_name (), sinkpad.get_name ());
                    }
                }
            }
        }
        
        /**
         * Forward EIT structure
         */
        private void bus_watch_func (Gst.Bus bus, Gst.Message message) {
            switch (message.type) {
                case Gst.MessageType.ELEMENT:
                    string structure_name = message.structure.get_name();
                    if (structure_name == "eit") {
                        if (this.epgscanner != null)
                            this.epgscanner.on_eit_structure (message.structure);
                        this.eit_structure (message.structure);
                    }
                break;
                
                default:
                break;
            }
        }
    }

    /**
     * This class handles watching channels for a single #DeviceGroup
     */
    public class ChannelFactory : GLib.Object {
    
        // The device group the factory belongs to
        public unowned DeviceGroup device_group {get; construct;}
        
        // List of players that are currently in use
        private HashSet<PlayerThread> active_players;
        
        public ChannelFactory (DeviceGroup devgroup) {
            this.device_group = devgroup;
            this.active_players = new HashSet<PlayerThread> ();
        }
        
        /**
         * Stop all currently active players
         */
        public void destroy () {
            lock (this.active_players) {
                foreach (PlayerThread active_player in this.active_players) {
                    active_player.destroy ();
                }
                this.active_players.clear ();
            }
        }

        /**
         * @channel: channel to watch
         * @sink_element: The element the src pad should be linked with
         * @force: Whether to stop a player when there's currently no free device
         * @notify_func: The given function is called when watching the channel
         *   is aborted because a recording on a different transport streams is
         *   about to start
         * @returns: The #PlayerThread used to watch @channel
         *
         * Watch @channel and use @sink_element as sink element
         */
        public PlayerThread? watch_channel (Channel channel, owned Gst.Element sink_element,
                bool force=false, ForcedStopNotify? notify_func = null) {
            debug ("Watching channel %s (%u)", channel.Name, channel.Sid);
        
            bool create_new = true;
            PlayerThread? player = null;
            DVB.Device? free_device = null;
            lock (this.active_players) {
                foreach (PlayerThread active_player in this.active_players) {
                    foreach (Channel other_channel in active_player.active_channels) {
                        if (channel.on_same_transport_stream (other_channel)) {
                            create_new = false;
                            player = active_player;
                            break;
                        }
                    }
                }
            }
            
            debug ("Creating new PlayerThread: %s", create_new.to_string ());
            if (create_new) {
                // Stop epgscanner before starting recording
                EPGScanner? epgscanner = this.device_group.epgscanner;
                if (epgscanner != null) epgscanner.stop ();

                free_device = this.device_group.get_next_free_device ();
                if (free_device == null && force) {
                    // Stop first player
                    lock (this.active_players) {
                        foreach (PlayerThread active_player in this.active_players) {
                            if (!active_player.forced) {
                                active_player.destroy (true);
                                break;
                            } else {
                                critical ("No active players that are not forced");
                            }
                        }
                    }
                    free_device = this.device_group.get_next_free_device ();
                }
                if (free_device == null) {
                    critical ("All devices are busy");
                    return null;
                }

                player = this.create_player (free_device);
            }

            player.get_element (channel, sink_element, force, notify_func);
            lock (this.active_players) {
                this.active_players.add (player);
            }
            
            return player;
        }
        
        /**
         * @returns: TRUE on success
         *
         * Stop watching @channel
         */
        public bool stop_channel (Channel channel, Gst.Element sink) {
            debug ("Stopping channel %s (%u)", channel.Name, channel.Sid);
        
            bool success = false;
            PlayerThread? player = null;
            lock (this.active_players) {
                foreach (PlayerThread active_player in this.active_players) {
                    if (active_player.active_channels.contains (channel)) {
                        success = active_player.remove_channel (channel, sink);
                        player = active_player;
                        break;
                    }
                }
            
                if (success && player.active_channels.size == 0)
                    this.active_players.remove (player);
                
                if (this.active_players.size == 0) {
                    // Start EPG scanner again
                    EPGScanner? epgscanner = this.device_group.epgscanner;
                    if (epgscanner != null) epgscanner.start ();
                }
            }
            return success;
        }
        
        /**
         * @returns: a new #PlayerThread instance for @device
         */
        public virtual PlayerThread create_player (Device device) {
            return new PlayerThread (device, this.device_group.epgscanner);
        }
    
    }

}
