/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gst/gst.h>
#include <gee.h>
#include <gio/gio.h>
#include <stdlib.h>
#include <string.h>


#define DVB_TYPE_SCANNER (dvb_scanner_get_type ())
#define DVB_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SCANNER, DVBScanner))
#define DVB_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SCANNER, DVBScannerClass))
#define DVB_IS_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SCANNER))
#define DVB_IS_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SCANNER))
#define DVB_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SCANNER, DVBScannerClass))

typedef struct _DVBScanner DVBScanner;
typedef struct _DVBScannerClass DVBScannerClass;
typedef struct _DVBScannerPrivate DVBScannerPrivate;

#define DVB_TYPE_SCANNED_ITEM (dvb_scanned_item_get_type ())
#define DVB_SCANNED_ITEM(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SCANNED_ITEM, DVBScannedItem))
#define DVB_SCANNED_ITEM_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SCANNED_ITEM, DVBScannedItemClass))
#define DVB_IS_SCANNED_ITEM(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SCANNED_ITEM))
#define DVB_IS_SCANNED_ITEM_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SCANNED_ITEM))
#define DVB_SCANNED_ITEM_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SCANNED_ITEM, DVBScannedItemClass))

typedef struct _DVBScannedItem DVBScannedItem;
typedef struct _DVBScannedItemClass DVBScannedItemClass;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;

#define DVB_TYPE_DEVICE (dvb_device_get_type ())
#define DVB_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE, DVBDevice))
#define DVB_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE, DVBDeviceClass))
#define DVB_IS_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE))
#define DVB_IS_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE))
#define DVB_DEVICE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE, DVBDeviceClass))

typedef struct _DVBDevice DVBDevice;
typedef struct _DVBDeviceClass DVBDeviceClass;

#define DVB_TYPE_CHANNEL_LIST_WRITER (dvb_channel_list_writer_get_type ())
#define DVB_CHANNEL_LIST_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST_WRITER, DVBChannelListWriter))
#define DVB_CHANNEL_LIST_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST_WRITER, DVBChannelListWriterClass))
#define DVB_IS_CHANNEL_LIST_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST_WRITER))
#define DVB_IS_CHANNEL_LIST_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST_WRITER))
#define DVB_CHANNEL_LIST_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST_WRITER, DVBChannelListWriterClass))

typedef struct _DVBChannelListWriter DVBChannelListWriter;
typedef struct _DVBChannelListWriterClass DVBChannelListWriterClass;

/**
     * An abstract class responsible for scanning for new channels
     */
struct _DVBScanner {
	GObject parent_instance;
	DVBScannerPrivate * priv;
	DVBChannelList* channels;
	GstElement* pipeline;
	GQueue* frequencies;
	GstStructure* current_tuning_params;
	GeeHashSet* scanned_frequencies;
	GeeHashMap* transport_streams;
};

struct _DVBScannerClass {
	GObjectClass parent_class;
	void (*prepare) (DVBScanner* self);
	DVBScannedItem* (*get_scanned_item) (DVBScanner* self, const GstStructure* structure);
	DVBChannel* (*get_new_channel) (DVBScanner* self);
	void (*add_values_from_structure_to_channel) (DVBScanner* self, const GstStructure* delivery, DVBChannel* channel);
};

struct _DVBScannerPrivate {
	DVBDevice* _Device;
	GeeArrayList* new_channels;
	guint check_for_lock_event_id;
	guint wait_for_tables_event_id;
	gboolean nit_arrived;
	gboolean sdt_arrived;
	gboolean pat_arrived;
	gboolean pmt_arrived;
	gboolean locked;
};


static gpointer dvb_scanner_parent_class = NULL;

GType dvb_scanner_get_type (void);
GType dvb_scanned_item_get_type (void);
GType dvb_channel_get_type (void);
GType dvb_channel_list_get_type (void);
GType dvb_device_get_type (void);
#define DVB_SCANNER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_SCANNER, DVBScannerPrivate))
enum  {
	DVB_SCANNER_DUMMY_PROPERTY,
	DVB_SCANNER_DEVICE,
	DVB_SCANNER_CHANNELS
};
#define DVB_SCANNER_BASE_PIDS "16:17:18"
#define DVB_SCANNER_PIPELINE_TEMPLATE "dvbsrc name=dvbsrc adapter=%u frontend=%u pids=%s stats-reporting-interval=0 ! mpegtsparse ! fakesink silent=true"
void dvb_scanner_prepare (DVBScanner* self);
static void dvb_scanner_real_prepare (DVBScanner* self);
DVBScannedItem* dvb_scanner_get_scanned_item (DVBScanner* self, const GstStructure* structure);
static DVBScannedItem* dvb_scanner_real_get_scanned_item (DVBScanner* self, const GstStructure* structure);
DVBChannel* dvb_scanner_get_new_channel (DVBScanner* self);
static DVBChannel* dvb_scanner_real_get_new_channel (DVBScanner* self);
void dvb_scanner_add_values_from_structure_to_channel (DVBScanner* self, const GstStructure* delivery, DVBChannel* channel);
static void dvb_scanner_real_add_values_from_structure_to_channel (DVBScanner* self, const GstStructure* delivery, DVBChannel* channel);
DVBChannelList* dvb_channel_list_new (GFile* channels);
DVBChannelList* dvb_channel_list_construct (GType object_type, GFile* channels);
DVBDevice* dvb_scanner_get_Device (DVBScanner* self);
guint dvb_device_get_Adapter (DVBDevice* self);
guint dvb_device_get_Frontend (DVBDevice* self);
void dvb_scanner_bus_watch_func (DVBScanner* self, GstBus* bus, GstMessage* message);
static void _dvb_scanner_bus_watch_func_gst_bus_message (GstBus* _sender, GstMessage* message, gpointer self);
void dvb_scanner_start_scan (DVBScanner* self);
void dvb_scanner_Run (DVBScanner* self);
void dvb_scanner_remove_check_for_lock_timeout (DVBScanner* self);
void dvb_scanner_remove_wait_for_tables_timeout (DVBScanner* self);
void dvb_scanner_clear_and_reset_all (DVBScanner* self);
void dvb_channel_list_clear (DVBChannelList* self);
void dvb_scanner_Destroy (DVBScanner* self);
DVBChannelListWriter* dvb_channel_list_writer_new (GFile* file);
DVBChannelListWriter* dvb_channel_list_writer_construct (GType object_type, GFile* file);
GType dvb_channel_list_writer_get_type (void);
void dvb_channel_list_writer_write (DVBChannelListWriter* self, DVBChannel* channel, GError** error);
gboolean dvb_channel_list_writer_close (DVBChannelListWriter* self, GError** error);
gboolean dvb_scanner_WriteAllChannelsToFile (DVBScanner* self, const char* path);
DVBChannel* dvb_channel_list_get_channel (DVBChannelList* self, guint sid);
gboolean dvb_scanner_WriteChannelsToFile (DVBScanner* self, guint* channel_sids, int channel_sids_length1, const char* path);
void dvb_scanner_clear_frequencies (DVBScanner* self);
guint dvb_scanned_item_get_Frequency (DVBScannedItem* self);
void dvb_scanner_add_structure_to_scan (DVBScanner* self, GstStructure* structure);
void dvb_channel_list_remove (DVBChannelList* self, guint sid);
gboolean dvb_scanner_check_for_lock (DVBScanner* self);
static gboolean _dvb_scanner_check_for_lock_gsource_func (gpointer self);
gboolean dvb_scanner_wait_for_tables (DVBScanner* self);
void dvb_scanner_set_uint_property (GstElement* src, const GstStructure* params, const char* key);
static gboolean _dvb_scanner_wait_for_tables_gsource_func (gpointer self);
void dvb_scanner_on_dvb_frontend_stats_structure (DVBScanner* self, const GstStructure* structure);
void dvb_scanner_on_dvb_read_failure_structure (DVBScanner* self);
void dvb_scanner_on_pat_structure (DVBScanner* self, const GstStructure* structure);
gboolean dvb_channel_list_contains (DVBChannelList* self, guint sid);
void dvb_scanner_add_new_channel (DVBScanner* self, guint sid);
void dvb_channel_set_Scrambled (DVBChannel* self, gboolean value);
void dvb_channel_set_Name (DVBChannel* self, const char* value);
void dvb_channel_set_TransportStreamId (DVBChannel* self, guint value);
void dvb_channel_set_Network (DVBChannel* self, const char* value);
void dvb_scanner_on_sdt_structure (DVBScanner* self, const GstStructure* structure);
void dvb_channel_set_LogicalChannelNumber (DVBChannel* self, guint* value);
void dvb_scanner_on_nit_structure (DVBScanner* self, const GstStructure* structure);
void dvb_channel_set_VideoPID (DVBChannel* self, guint value);
GeeList* dvb_channel_get_AudioPIDs (DVBChannel* self);
void dvb_scanner_on_pmt_structure (DVBScanner* self, const GstStructure* structure);
guint dvb_channel_get_TransportStreamId (DVBChannel* self);
gboolean dvb_channel_is_valid (DVBChannel* self);
guint dvb_channel_get_VideoPID (DVBChannel* self);
char* dvb_channel_to_string (DVBChannel* self);
guint dvb_channel_get_Frequency (DVBChannel* self);
const char* dvb_channel_get_Name (DVBChannel* self);
const char* dvb_channel_get_Network (DVBChannel* self);
gboolean dvb_channel_get_Scrambled (DVBChannel* self);
void dvb_channel_set_Sid (DVBChannel* self, guint value);
void dvb_channel_list_add (DVBChannelList* self, DVBChannel* channel);
static void dvb_scanner_set_Device (DVBScanner* self, DVBDevice* value);
DVBChannelList* dvb_scanner_get_Channels (DVBScanner* self);
guint dvb_scanned_item_hash (DVBScannedItem* o);
static guint _dvb_scanned_item_hash_ghash_func (void* key);
gboolean dvb_scanned_item_equal (DVBScannedItem* o1, DVBScannedItem* o2);
static gboolean _dvb_scanned_item_equal_gequal_func (void* a, void* b);
static GObject * dvb_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static void dvb_scanner_finalize (GObject* obj);
static void dvb_scanner_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_scanner_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static int _vala_strcmp0 (const char * str1, const char * str2);


static void g_cclosure_user_marshal_VOID__UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);

static void dvb_scanner_real_prepare (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	g_critical ("Type `%s' does not implement abstract method `dvb_scanner_prepare'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return;
}


/**
         * Setup the pipeline correctly
         */
void dvb_scanner_prepare (DVBScanner* self) {
	DVB_SCANNER_GET_CLASS (self)->prepare (self);
}


static DVBScannedItem* dvb_scanner_real_get_scanned_item (DVBScanner* self, const GstStructure* structure) {
	g_return_val_if_fail (self != NULL, NULL);
	g_critical ("Type `%s' does not implement abstract method `dvb_scanner_get_scanned_item'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return NULL;
}


/**
         * Use the frequency and possibly other data to
         * mark the tuning paramters as already used
         */
DVBScannedItem* dvb_scanner_get_scanned_item (DVBScanner* self, const GstStructure* structure) {
	return DVB_SCANNER_GET_CLASS (self)->get_scanned_item (self, structure);
}


static DVBChannel* dvb_scanner_real_get_new_channel (DVBScanner* self) {
	g_return_val_if_fail (self != NULL, NULL);
	g_critical ("Type `%s' does not implement abstract method `dvb_scanner_get_new_channel'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return NULL;
}


/**
         * Return a new empty channel
         */
DVBChannel* dvb_scanner_get_new_channel (DVBScanner* self) {
	return DVB_SCANNER_GET_CLASS (self)->get_new_channel (self);
}


static void dvb_scanner_real_add_values_from_structure_to_channel (DVBScanner* self, const GstStructure* delivery, DVBChannel* channel) {
	g_return_if_fail (self != NULL);
	g_critical ("Type `%s' does not implement abstract method `dvb_scanner_add_values_from_structure_to_channel'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return;
}


/**
         * Retrieve the data from structure and add it to the Channel
         */
void dvb_scanner_add_values_from_structure_to_channel (DVBScanner* self, const GstStructure* delivery, DVBChannel* channel) {
	DVB_SCANNER_GET_CLASS (self)->add_values_from_structure_to_channel (self, delivery, channel);
}


static void _dvb_scanner_bus_watch_func_gst_bus_message (GstBus* _sender, GstMessage* message, gpointer self) {
	dvb_scanner_bus_watch_func (self, _sender, message);
}


/**
         * Start the scanner
         */
void dvb_scanner_Run (DVBScanner* self) {
	GError * _inner_error_;
	DVBChannelList* _tmp0_;
	GstBus* bus;
	g_return_if_fail (self != NULL);
	_inner_error_ = NULL;
	_tmp0_ = NULL;
	self->channels = (_tmp0_ = dvb_channel_list_new (NULL), (self->channels == NULL) ? NULL : (self->channels = (g_object_unref (self->channels), NULL)), _tmp0_);
	{
		char* _tmp1_;
		GstElement* _tmp2_;
		GstElement* _tmp3_;
		GstElement* _tmp4_;
		_tmp1_ = NULL;
		_tmp2_ = NULL;
		_tmp3_ = (_tmp2_ = gst_parse_launch (_tmp1_ = g_strdup_printf (DVB_SCANNER_PIPELINE_TEMPLATE, dvb_device_get_Adapter (self->priv->_Device), dvb_device_get_Frontend (self->priv->_Device), DVB_SCANNER_BASE_PIDS), &_inner_error_), _tmp1_ = (g_free (_tmp1_), NULL), _tmp2_);
		if (_inner_error_ != NULL) {
			goto __catch20_g_error;
			goto __finally20;
		}
		_tmp4_ = NULL;
		self->pipeline = (_tmp4_ = _tmp3_, (self->pipeline == NULL) ? NULL : (self->pipeline = (gst_object_unref (self->pipeline), NULL)), _tmp4_);
	}
	goto __finally20;
	__catch20_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_error ("Scanner.vala:147: Could not create pipeline: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			return;
		}
	}
	__finally20:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	bus = gst_element_get_bus (self->pipeline);
	gst_bus_add_signal_watch (bus);
	g_signal_connect_object (bus, "message", (GCallback) _dvb_scanner_bus_watch_func_gst_bus_message, self, 0);
	gst_element_set_state (self->pipeline, GST_STATE_READY);
	dvb_scanner_start_scan (self);
	(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
}


/**
         * Abort scanning and cleanup
         */
void dvb_scanner_Destroy (DVBScanner* self) {
	DVBChannelList* _tmp0_;
	g_return_if_fail (self != NULL);
	dvb_scanner_remove_check_for_lock_timeout (self);
	dvb_scanner_remove_wait_for_tables_timeout (self);
	dvb_scanner_clear_and_reset_all (self);
	dvb_channel_list_clear (self->channels);
	_tmp0_ = NULL;
	self->channels = (_tmp0_ = NULL, (self->channels == NULL) ? NULL : (self->channels = (g_object_unref (self->channels), NULL)), _tmp0_);
	g_signal_emit_by_name (self, "destroyed");
}


/** 
         * @path: Location where the file will be stored
         *
         * Write all the channels stored in this.Channels to file
         */
gboolean dvb_scanner_WriteAllChannelsToFile (DVBScanner* self, const char* path) {
	gboolean result;
	GError * _inner_error_;
	gboolean success;
	GFile* _tmp0_;
	DVBChannelListWriter* _tmp1_;
	DVBChannelListWriter* writer;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (path != NULL, FALSE);
	_inner_error_ = NULL;
	success = TRUE;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	writer = (_tmp1_ = dvb_channel_list_writer_new (_tmp0_ = g_file_new_for_path (path)), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
	{
		GeeIterator* _c_it;
		_c_it = gee_iterable_iterator ((GeeIterable*) self->channels);
		while (TRUE) {
			DVBChannel* c;
			if (!gee_iterator_next (_c_it)) {
				break;
			}
			c = (DVBChannel*) gee_iterator_get (_c_it);
			{
				dvb_channel_list_writer_write (writer, c, &_inner_error_);
				if (_inner_error_ != NULL) {
					goto __catch21_g_error;
					goto __finally21;
				}
			}
			goto __finally21;
			__catch21_g_error:
			{
				GError * e;
				e = _inner_error_;
				_inner_error_ = NULL;
				{
					g_critical ("Scanner.vala:184: Could not write to file: %s", e->message);
					success = FALSE;
					(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				}
			}
			__finally21:
			if (_inner_error_ != NULL) {
				(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
				(_c_it == NULL) ? NULL : (_c_it = (g_object_unref (_c_it), NULL));
				(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
				g_clear_error (&_inner_error_);
				return FALSE;
			}
			(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
		}
		(_c_it == NULL) ? NULL : (_c_it = (g_object_unref (_c_it), NULL));
	}
	{
		dvb_channel_list_writer_close (writer, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch22_g_error;
			goto __finally22;
		}
	}
	goto __finally22;
	__catch22_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("Scanner.vala:192: Could not close file handle: %s", e->message);
			success = FALSE;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally22:
	if (_inner_error_ != NULL) {
		(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return FALSE;
	}
	result = success;
	(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
	return result;
}


/**
         * @channel_sids: A list of channels' SIDs
         * @path: Location where the file will be stored
         *
         * Write the channels with the given SIDs to file @path
         */
gboolean dvb_scanner_WriteChannelsToFile (DVBScanner* self, guint* channel_sids, int channel_sids_length1, const char* path) {
	gboolean result;
	GError * _inner_error_;
	gboolean success;
	GFile* _tmp0_;
	DVBChannelListWriter* _tmp1_;
	DVBChannelListWriter* writer;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (path != NULL, FALSE);
	_inner_error_ = NULL;
	success = TRUE;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	writer = (_tmp1_ = dvb_channel_list_writer_new (_tmp0_ = g_file_new_for_path (path)), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
	{
		guint* sid_collection;
		int sid_collection_length1;
		int sid_it;
		sid_collection = channel_sids;
		sid_collection_length1 = channel_sids_length1;
		for (sid_it = 0; sid_it < channel_sids_length1; sid_it = sid_it + 1) {
			guint sid;
			sid = sid_collection[sid_it];
			{
				DVBChannel* c;
				c = dvb_channel_list_get_channel (self->channels, sid);
				if (c == NULL) {
					g_warning ("Scanner.vala:211: Channel with SID %u does not exist", sid);
					(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
					continue;
				}
				{
					dvb_channel_list_writer_write (writer, c, &_inner_error_);
					if (_inner_error_ != NULL) {
						goto __catch23_g_error;
						goto __finally23;
					}
				}
				goto __finally23;
				__catch23_g_error:
				{
					GError * e;
					e = _inner_error_;
					_inner_error_ = NULL;
					{
						g_critical ("Scanner.vala:217: Could not write to file: %s", e->message);
						success = FALSE;
						(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
					}
				}
				__finally23:
				if (_inner_error_ != NULL) {
					(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
					(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
					g_clear_error (&_inner_error_);
					return FALSE;
				}
				(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
			}
		}
	}
	{
		dvb_channel_list_writer_close (writer, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch24_g_error;
			goto __finally24;
		}
	}
	goto __finally24;
	__catch24_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("Scanner.vala:225: Could not close file handle: %s", e->message);
			success = FALSE;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally24:
	if (_inner_error_ != NULL) {
		(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return FALSE;
	}
	result = success;
	(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
	return result;
}


void dvb_scanner_clear_and_reset_all (DVBScanner* self) {
	GstStructure* _tmp1_;
	g_return_if_fail (self != NULL);
	if (self->pipeline != NULL) {
		GstBus* bus;
		GstElement* _tmp0_;
		bus = gst_element_get_bus (self->pipeline);
		gst_bus_remove_signal_watch (bus);
		gst_element_set_state (self->pipeline, GST_STATE_NULL);
		/* Free pipeline*/
		_tmp0_ = NULL;
		self->pipeline = (_tmp0_ = NULL, (self->pipeline == NULL) ? NULL : (self->pipeline = (gst_object_unref (self->pipeline), NULL)), _tmp0_);
		(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
	}
	gee_map_clear ((GeeMap*) self->transport_streams);
	gee_collection_clear ((GeeCollection*) self->scanned_frequencies);
	dvb_scanner_clear_frequencies (self);
	_tmp1_ = NULL;
	self->current_tuning_params = (_tmp1_ = NULL, (self->current_tuning_params == NULL) ? NULL : (self->current_tuning_params = (gst_structure_free (self->current_tuning_params), NULL)), _tmp1_);
	gee_collection_clear ((GeeCollection*) self->priv->new_channels);
}


void dvb_scanner_clear_frequencies (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	while (TRUE) {
		GstStructure* s;
		GstStructure* _tmp0_;
		if (!(!g_queue_is_empty (self->frequencies))) {
			break;
		}
		s = (GstStructure*) g_queue_pop_head (self->frequencies);
		/* Force that gst_structure_free is called*/
		_tmp0_ = NULL;
		s = (_tmp0_ = NULL, (s == NULL) ? NULL : (s = (gst_structure_free (s), NULL)), _tmp0_);
		(s == NULL) ? NULL : (s = (gst_structure_free (s), NULL));
	}
	g_queue_clear (self->frequencies);
}


void dvb_scanner_add_structure_to_scan (DVBScanner* self, GstStructure* structure) {
	DVBScannedItem* item;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	if (structure == NULL) {
		(structure == NULL) ? NULL : (structure = (gst_structure_free (structure), NULL));
		return;
	}
	item = dvb_scanner_get_scanned_item (self, structure);
	if (!gee_collection_contains ((GeeCollection*) self->scanned_frequencies, item)) {
		const GstStructure* _tmp0_;
		g_debug ("Scanner.vala:263: Queueing new frequency %u", dvb_scanned_item_get_Frequency (item));
		_tmp0_ = NULL;
		g_queue_push_tail (self->frequencies, (_tmp0_ = structure, (_tmp0_ == NULL) ? NULL : gst_structure_copy (_tmp0_)));
		gee_collection_add ((GeeCollection*) self->scanned_frequencies, item);
	}
	(item == NULL) ? NULL : (item = (g_object_unref (item), NULL));
	(structure == NULL) ? NULL : (structure = (gst_structure_free (structure), NULL));
}


static char* bool_to_string (gboolean self) {
	char* result;
	if (self) {
		result = g_strdup ("true");
		return result;
	} else {
		result = g_strdup ("false");
		return result;
	}
}


static gboolean _dvb_scanner_check_for_lock_gsource_func (gpointer self) {
	return dvb_scanner_check_for_lock (self);
}


/**
         * Pick up the next tuning paramters from the queue
         * and start scanning with them
         */
void dvb_scanner_start_scan (DVBScanner* self) {
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	gboolean all_tables;
	char* _tmp7_;
	char* _tmp6_;
	char* _tmp5_;
	char* _tmp4_;
	char* _tmp3_;
	GstStructure* _tmp8_;
	guint freq;
	GstElement* dvbsrc;
	g_return_if_fail (self != NULL);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	if (self->priv->sdt_arrived) {
		_tmp2_ = self->priv->nit_arrived;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		_tmp1_ = self->priv->pat_arrived;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = self->priv->pmt_arrived;
	} else {
		_tmp0_ = FALSE;
	}
	all_tables = _tmp0_;
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	g_debug ("Scanner.vala:276: Received all tables: %s (pat: %s, sdt: %s, nit: %s, pmt: %s)", _tmp3_ = bool_to_string (all_tables), _tmp4_ = bool_to_string (self->priv->pat_arrived), _tmp5_ = bool_to_string (self->priv->sdt_arrived), _tmp6_ = bool_to_string (self->priv->nit_arrived), _tmp7_ = bool_to_string (self->priv->pmt_arrived));
	_tmp7_ = (g_free (_tmp7_), NULL);
	_tmp6_ = (g_free (_tmp6_), NULL);
	_tmp5_ = (g_free (_tmp5_), NULL);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp3_ = (g_free (_tmp3_), NULL);
	self->priv->nit_arrived = FALSE;
	self->priv->sdt_arrived = FALSE;
	self->priv->pat_arrived = FALSE;
	self->priv->pmt_arrived = FALSE;
	self->priv->locked = FALSE;
	if (self->current_tuning_params != NULL) {
		guint old_freq;
		old_freq = 0U;
		gst_structure_get_uint (self->current_tuning_params, "frequency", &old_freq);
		g_signal_emit_by_name (self, "frequency-scanned", old_freq, self->frequencies->length);
	}
	if (g_queue_is_empty (self->frequencies)) {
		g_message ("Scanner.vala:294: Finished scanning");
		/* We don't have all the information for those channels
		 remove them*/
		g_debug ("Scanner.vala:297: %u channels still have missing or invalid information", (guint) gee_collection_get_size ((GeeCollection*) self->priv->new_channels));
		{
			GeeIterator* _sid_it;
			_sid_it = gee_iterable_iterator ((GeeIterable*) self->priv->new_channels);
			while (TRUE) {
				guint sid;
				if (!gee_iterator_next (_sid_it)) {
					break;
				}
				sid = GPOINTER_TO_UINT (gee_iterator_get (_sid_it));
				dvb_channel_list_remove (self->channels, sid);
			}
			(_sid_it == NULL) ? NULL : (_sid_it = (g_object_unref (_sid_it), NULL));
		}
		dvb_scanner_clear_and_reset_all (self);
		g_signal_emit_by_name (self, "finished");
		return;
	}
	_tmp8_ = NULL;
	self->current_tuning_params = (_tmp8_ = (GstStructure*) g_queue_pop_head (self->frequencies), (self->current_tuning_params == NULL) ? NULL : (self->current_tuning_params = (gst_structure_free (self->current_tuning_params), NULL)), _tmp8_);
	/* Remember that we already scanned this frequency*/
	freq = 0U;
	gst_structure_get_uint (self->current_tuning_params, "frequency", &freq);
	g_debug ("Scanner.vala:313: Starting scanning frequency %u (%u left)", freq, g_queue_get_length (self->frequencies));
	gst_element_set_state (self->pipeline, GST_STATE_READY);
	dvb_scanner_prepare (self);
	/* Reset PIDs*/
	dvbsrc = gst_bin_get_by_name (GST_BIN (self->pipeline), "dvbsrc");
	g_object_set ((GObject*) dvbsrc, "pids", DVB_SCANNER_BASE_PIDS, NULL);
	gst_element_set_state (self->pipeline, GST_STATE_PLAYING);
	self->priv->check_for_lock_event_id = g_timeout_add_seconds ((guint) 5, _dvb_scanner_check_for_lock_gsource_func, self);
	(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
}


/**
         * Check if we received a lock with the currently
         * used tuning parameters
         */
gboolean dvb_scanner_check_for_lock (DVBScanner* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	self->priv->check_for_lock_event_id = (guint) 0;
	if (!self->priv->locked) {
		gst_element_set_state (self->pipeline, GST_STATE_READY);
		dvb_scanner_start_scan (self);
	}
	result = FALSE;
	return result;
}


gboolean dvb_scanner_wait_for_tables (DVBScanner* self) {
	gboolean result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	g_return_val_if_fail (self != NULL, FALSE);
	self->priv->wait_for_tables_event_id = (guint) 0;
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	if (self->priv->sdt_arrived) {
		_tmp2_ = self->priv->nit_arrived;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		_tmp1_ = self->priv->pat_arrived;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = self->priv->pmt_arrived;
	} else {
		_tmp0_ = FALSE;
	}
	if (!_tmp0_) {
		gst_element_set_state (self->pipeline, GST_STATE_READY);
		dvb_scanner_start_scan (self);
	}
	result = FALSE;
	return result;
}


void dvb_scanner_remove_check_for_lock_timeout (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->check_for_lock_event_id != 0) {
		g_source_remove (self->priv->check_for_lock_event_id);
		self->priv->check_for_lock_event_id = (guint) 0;
	}
}


void dvb_scanner_remove_wait_for_tables_timeout (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->wait_for_tables_event_id != 0) {
		g_source_remove (self->priv->wait_for_tables_event_id);
		self->priv->wait_for_tables_event_id = (guint) 0;
	}
}


void dvb_scanner_set_uint_property (GstElement* src, const GstStructure* params, const char* key) {
	guint val;
	g_return_if_fail (src != NULL);
	g_return_if_fail (params != NULL);
	g_return_if_fail (key != NULL);
	val = 0U;
	gst_structure_get_uint (params, key, &val);
	g_object_set ((GObject*) src, key, val, NULL);
}


static gboolean _dvb_scanner_wait_for_tables_gsource_func (gpointer self) {
	return dvb_scanner_wait_for_tables (self);
}


void dvb_scanner_on_dvb_frontend_stats_structure (DVBScanner* self, const GstStructure* structure) {
	gboolean has_lock;
	gboolean _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	has_lock = FALSE;
	gst_structure_get_boolean (structure, "lock", &has_lock);
	_tmp0_ = FALSE;
	if (has_lock) {
		_tmp0_ = !self->priv->locked;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		g_debug ("Scanner.vala:378: Got lock");
		dvb_scanner_remove_check_for_lock_timeout (self);
		self->priv->wait_for_tables_event_id = g_timeout_add_seconds ((guint) 10, _dvb_scanner_wait_for_tables_gsource_func, self);
	}
}


void dvb_scanner_on_dvb_read_failure_structure (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	g_warning ("Scanner.vala:386: Read failure");
}


static char* string_substring (const char* self, glong offset, glong len) {
	char* result;
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	result = g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
	return result;
}


/*
            this.Destroy ();
            */
void dvb_scanner_on_pat_structure (DVBScanner* self, const GstStructure* structure) {
	GeeSet* pid_set;
	GValue programs;
	guint size;
	GValue val = {0};
	const GstStructure* program;
	GString* new_pids;
	char* pids;
	GstElement* dvbsrc;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	g_debug ("Scanner.vala:393: Received PAT");
	pid_set = (GeeSet*) gee_hash_set_new (G_TYPE_UINT, NULL, NULL, g_direct_hash, g_direct_equal);
	/* add BASE_PIDS*/
	gee_collection_add ((GeeCollection*) pid_set, GUINT_TO_POINTER ((guint) 16));
	gee_collection_add ((GeeCollection*) pid_set, GUINT_TO_POINTER ((guint) 17));
	gee_collection_add ((GeeCollection*) pid_set, GUINT_TO_POINTER ((guint) 18));
	programs = *gst_structure_get_value (structure, "programs");
	size = gst_value_list_get_size (&programs);
	program = NULL;
	{
		guint i;
		/* Iterate over programs*/
		i = (guint) 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				guint sid;
				guint pmt;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < size)) {
					break;
				}
				val = *gst_value_list_get_value (&programs, i);
				program = gst_value_get_structure (&val);
				sid = 0U;
				gst_structure_get_uint (program, "program-number", &sid);
				pmt = 0U;
				gst_structure_get_uint (program, "pid", &pmt);
				gee_collection_add ((GeeCollection*) pid_set, GUINT_TO_POINTER (pmt));
			}
		}
	}
	new_pids = g_string_new ("");
	{
		GeeIterator* _pid_it;
		_pid_it = gee_iterable_iterator ((GeeIterable*) pid_set);
		while (TRUE) {
			guint pid;
			char* _tmp1_;
			if (!gee_iterator_next (_pid_it)) {
				break;
			}
			pid = GPOINTER_TO_UINT (gee_iterator_get (_pid_it));
			_tmp1_ = NULL;
			g_string_append (new_pids, _tmp1_ = g_strdup_printf ("%u:", pid));
			_tmp1_ = (g_free (_tmp1_), NULL);
		}
		(_pid_it == NULL) ? NULL : (_pid_it = (g_object_unref (_pid_it), NULL));
	}
	pids = string_substring (new_pids->str, (glong) 0, new_pids->len - 1);
	g_debug ("Scanner.vala:425: Setting %d pids: %s", gee_collection_get_size ((GeeCollection*) pid_set), pids);
	/* We want to parse the pmt as well*/
	dvbsrc = gst_bin_get_by_name (GST_BIN (self->pipeline), "dvbsrc");
	g_object_set ((GObject*) dvbsrc, "pids", pids, NULL);
	self->priv->pat_arrived = TRUE;
	(pid_set == NULL) ? NULL : (pid_set = (g_object_unref (pid_set), NULL));
	(new_pids == NULL) ? NULL : (new_pids = (g_string_free (new_pids, TRUE), NULL));
	pids = (g_free (pids), NULL);
	(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
}


static char* string_replace (const char* self, const char* old, const char* replacement) {
	char* result;
	GError * _inner_error_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (old != NULL, NULL);
	g_return_val_if_fail (replacement != NULL, NULL);
	_inner_error_ = NULL;
	{
		char* _tmp0_;
		GRegex* _tmp1_;
		GRegex* regex;
		char* _tmp2_;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		regex = (_tmp1_ = g_regex_new (_tmp0_ = g_regex_escape_string (old, -1), 0, 0, &_inner_error_), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_REGEX_ERROR) {
				goto __catch25_g_regex_error;
			}
			goto __finally25;
		}
		_tmp2_ = g_regex_replace_literal (regex, self, (glong) (-1), 0, replacement, 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
			if (_inner_error_->domain == G_REGEX_ERROR) {
				goto __catch25_g_regex_error;
			}
			goto __finally25;
		}
		result = _tmp2_;
		(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
		return result;
	}
	goto __finally25;
	__catch25_g_regex_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_assert_not_reached ();
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally25:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
}


void dvb_scanner_on_sdt_structure (DVBScanner* self, const GstStructure* structure) {
	guint tsid;
	GValue services;
	guint size;
	GValue val = {0};
	const GstStructure* service;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	g_debug ("Scanner.vala:434: Received SDT");
	tsid = 0U;
	gst_structure_get_uint (structure, "transport-stream-id", &tsid);
	services = *gst_structure_get_value (structure, "services");
	size = gst_value_list_get_size (&services);
	service = NULL;
	{
		guint i;
		/* Iterate over services*/
		i = (guint) 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				const char* _tmp1_;
				char* name;
				char* _tmp2_;
				gint _tmp3_;
				gint sid;
				DVBChannel* channel;
				const char* _tmp7_;
				char* provider;
				gboolean _tmp8_;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < size)) {
					break;
				}
				val = *gst_value_list_get_value (&services, i);
				service = gst_value_get_structure (&val);
				/* Returns "service-%d"*/
				_tmp1_ = NULL;
				name = (_tmp1_ = gst_structure_get_name (service), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
				/* Get the number at the end*/
				_tmp2_ = NULL;
				sid = (_tmp3_ = atoi (_tmp2_ = string_substring (name, (glong) 8, g_utf8_strlen (name, -1) - 8)), _tmp2_ = (g_free (_tmp2_), NULL), _tmp3_);
				if (gst_structure_has_field (service, "name")) {
					char* _tmp5_;
					const char* _tmp4_;
					_tmp5_ = NULL;
					_tmp4_ = NULL;
					name = (_tmp5_ = (_tmp4_ = gst_structure_get_string (service, "name"), (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)), name = (g_free (name), NULL), _tmp5_);
				}
				if (!dvb_channel_list_contains (self->channels, (guint) sid)) {
					dvb_scanner_add_new_channel (self, (guint) sid);
				}
				channel = dvb_channel_list_get_channel (self->channels, (guint) sid);
				if (gst_structure_has_field (service, "scrambled")) {
					gboolean scrambled;
					scrambled = FALSE;
					gst_structure_get_boolean (service, "scrambled", &scrambled);
					dvb_channel_set_Scrambled (channel, scrambled);
				} else {
					dvb_channel_set_Scrambled (channel, FALSE);
				}
				if (g_utf8_validate (name, -1, NULL)) {
					char* _tmp6_;
					_tmp6_ = NULL;
					dvb_channel_set_Name (channel, _tmp6_ = string_replace (name, "\\s", " "));
					_tmp6_ = (g_free (_tmp6_), NULL);
				}
				dvb_channel_set_TransportStreamId (channel, tsid);
				_tmp7_ = NULL;
				provider = (_tmp7_ = gst_structure_get_string (service, "provider-name"), (_tmp7_ == NULL) ? NULL : g_strdup (_tmp7_));
				_tmp8_ = FALSE;
				if (provider != NULL) {
					_tmp8_ = g_utf8_validate (provider, -1, NULL);
				} else {
					_tmp8_ = FALSE;
				}
				if (_tmp8_) {
					dvb_channel_set_Network (channel, provider);
				} else {
					dvb_channel_set_Network (channel, "");
				}
				name = (g_free (name), NULL);
				(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
				provider = (g_free (provider), NULL);
			}
		}
	}
	self->priv->sdt_arrived = TRUE;
}


void dvb_scanner_on_nit_structure (DVBScanner* self, const GstStructure* structure) {
	char* name;
	GValue transports;
	guint size;
	GValue val = {0};
	const GstStructure* transport;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	g_debug ("Scanner.vala:488: Received NIT");
	name = NULL;
	if (gst_structure_has_field (structure, "network-name")) {
		char* _tmp1_;
		const char* _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		name = (_tmp1_ = (_tmp0_ = gst_structure_get_string (structure, "network-name"), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), name = (g_free (name), NULL), _tmp1_);
	} else {
		guint nid;
		char* _tmp2_;
		nid = 0U;
		gst_structure_get_uint (structure, "network-id", &nid);
		_tmp2_ = NULL;
		name = (_tmp2_ = g_strdup_printf ("%u", nid), name = (g_free (name), NULL), _tmp2_);
	}
	g_debug ("Scanner.vala:498: Network name '%s'", name);
	transports = *gst_structure_get_value (structure, "transports");
	size = gst_value_list_get_size (&transports);
	transport = NULL;
	{
		guint i;
		/* Iterate over transports*/
		i = (guint) 0;
		{
			gboolean _tmp3_;
			_tmp3_ = TRUE;
			while (TRUE) {
				guint tsid;
				if (!_tmp3_) {
					i++;
				}
				_tmp3_ = FALSE;
				if (!(i < size)) {
					break;
				}
				val = *gst_value_list_get_value (&transports, i);
				transport = gst_value_get_structure (&val);
				tsid = 0U;
				gst_structure_get_uint (transport, "transport-stream-id", &tsid);
				if (gst_structure_has_field (transport, "delivery")) {
					GValue delivery_val;
					const GstStructure* delivery;
					guint freq;
					const GstStructure* _tmp4_;
					delivery_val = *gst_structure_get_value (transport, "delivery");
					delivery = gst_value_get_structure (&delivery_val);
					g_debug ("Scanner.vala:517: Received TS %u", tsid);
					gee_map_set ((GeeMap*) self->transport_streams, GUINT_TO_POINTER (tsid), delivery);
					freq = 0U;
					gst_structure_get_uint (delivery, "frequency", &freq);
					/* Takes care of duplicates*/
					_tmp4_ = NULL;
					dvb_scanner_add_structure_to_scan (self, (_tmp4_ = delivery, (_tmp4_ == NULL) ? NULL : gst_structure_copy (_tmp4_)));
				}
				if (gst_structure_has_field (transport, "channels")) {
					GValue channels;
					guint channels_size;
					GValue channel_val = {0};
					const GstStructure* channel_struct;
					channels = *gst_structure_get_value (transport, "channels");
					channels_size = gst_value_list_get_size (&channels);
					channel_struct = NULL;
					{
						gint j;
						/* Iterate over channels*/
						j = 0;
						{
							gboolean _tmp5_;
							_tmp5_ = TRUE;
							while (TRUE) {
								guint sid;
								DVBChannel* dvb_channel;
								guint lcnumber;
								if (!_tmp5_) {
									j++;
								}
								_tmp5_ = FALSE;
								if (!(j < channels_size)) {
									break;
								}
								channel_val = *gst_value_list_get_value (&channels, (guint) j);
								channel_struct = gst_value_get_structure (&channel_val);
								sid = 0U;
								gst_structure_get_uint (channel_struct, "service-id", &sid);
								if (!dvb_channel_list_contains (self->channels, sid)) {
									dvb_scanner_add_new_channel (self, sid);
								}
								dvb_channel = dvb_channel_list_get_channel (self->channels, sid);
								if (g_utf8_validate (name, -1, NULL)) {
									dvb_channel_set_Network (dvb_channel, name);
								} else {
									dvb_channel_set_Network (dvb_channel, "");
								}
								lcnumber = 0U;
								gst_structure_get_uint (channel_struct, "logical-channel-number", &lcnumber);
								dvb_channel_set_LogicalChannelNumber (dvb_channel, &lcnumber);
								(dvb_channel == NULL) ? NULL : (dvb_channel = (g_object_unref (dvb_channel), NULL));
							}
						}
					}
				}
			}
		}
	}
	self->priv->nit_arrived = TRUE;
	name = (g_free (name), NULL);
}


void dvb_scanner_on_pmt_structure (DVBScanner* self, const GstStructure* structure) {
	guint program_number;
	DVBChannel* dvb_channel;
	GValue streams;
	guint size;
	GValue stream_val = {0};
	const GstStructure* stream;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	g_debug ("Scanner.vala:563: Received PMT");
	program_number = 0U;
	gst_structure_get_uint (structure, "program-number", &program_number);
	if (!dvb_channel_list_contains (self->channels, program_number)) {
		dvb_scanner_add_new_channel (self, program_number);
	}
	dvb_channel = dvb_channel_list_get_channel (self->channels, program_number);
	streams = *gst_structure_get_value (structure, "streams");
	size = gst_value_list_get_size (&streams);
	stream = NULL;
	{
		gint i;
		/* Iterate over streams*/
		i = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				guint pid;
				guint stream_type;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < size)) {
					break;
				}
				stream_val = *gst_value_list_get_value (&streams, (guint) i);
				stream = gst_value_get_structure (&stream_val);
				pid = 0U;
				gst_structure_get_uint (stream, "pid", &pid);
				/* See ISO/IEC 13818-1 Table 2-29*/
				stream_type = 0U;
				gst_structure_get_uint (stream, "stream-type", &stream_type);
				switch (stream_type) {
					case 0x01:
					case 0x02:
					case 0x1b:
					{
						g_debug ("Scanner.vala:595: Found video PID %u for channel %u", pid, program_number);
						dvb_channel_set_VideoPID (dvb_channel, pid);
						break;
					}
					case 0x03:
					case 0x04:
					case 0x0f:
					case 0x11:
					{
						g_debug ("Scanner.vala:603: Found audio PID %u for channel %u", pid, program_number);
						gee_collection_add ((GeeCollection*) dvb_channel_get_AudioPIDs (dvb_channel), GUINT_TO_POINTER (pid));
						break;
					}
					default:
					{
						g_debug ("Scanner.vala:608: Other stream type: 0x%02x", stream_type);
						break;
					}
				}
			}
		}
	}
	self->priv->pmt_arrived = TRUE;
	(dvb_channel == NULL) ? NULL : (dvb_channel = (g_object_unref (dvb_channel), NULL));
}


void dvb_scanner_bus_watch_func (DVBScanner* self, GstBus* bus, GstMessage* message) {
	gboolean _tmp4_;
	gboolean _tmp10_;
	gboolean _tmp11_;
	gboolean _tmp12_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (bus != NULL);
	g_return_if_fail (message != NULL);
	switch (message->type) {
		case GST_MESSAGE_ELEMENT:
		{
			{
				if (_vala_strcmp0 (gst_structure_get_name (message->structure), "dvb-frontend-stats") == 0) {
					dvb_scanner_on_dvb_frontend_stats_structure (self, message->structure);
				} else {
					if (_vala_strcmp0 (gst_structure_get_name (message->structure), "dvb-read-failure") == 0) {
						dvb_scanner_on_dvb_read_failure_structure (self);
					} else {
						if (_vala_strcmp0 (gst_structure_get_name (message->structure), "sdt") == 0) {
							dvb_scanner_on_sdt_structure (self, message->structure);
						} else {
							if (_vala_strcmp0 (gst_structure_get_name (message->structure), "nit") == 0) {
								dvb_scanner_on_nit_structure (self, message->structure);
							} else {
								if (_vala_strcmp0 (gst_structure_get_name (message->structure), "pat") == 0) {
									dvb_scanner_on_pat_structure (self, message->structure);
								} else {
									if (_vala_strcmp0 (gst_structure_get_name (message->structure), "pmt") == 0) {
										dvb_scanner_on_pmt_structure (self, message->structure);
									}
								}
							}
						}
					}
				}
				break;
			}
		}
		case GST_MESSAGE_ERROR:
		{
			{
				GError* gerror;
				char* debug;
				char* _tmp3_;
				char* _tmp2_;
				GError* _tmp1_;
				GError* _tmp0_;
				gerror = NULL;
				debug = NULL;
				_tmp3_ = NULL;
				_tmp2_ = NULL;
				_tmp1_ = NULL;
				_tmp0_ = NULL;
				(gst_message_parse_error (message, &_tmp0_, &_tmp2_), gerror = (_tmp1_ = _tmp0_, (gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL)), _tmp1_));
				debug = (_tmp3_ = _tmp2_, debug = (g_free (debug), NULL), _tmp3_);
				g_critical ("Scanner.vala:637: %s %s", gerror->message, debug);
				dvb_scanner_Destroy (self);
				(gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL));
				debug = (g_free (debug), NULL);
				break;
			}
		}
	}
	_tmp4_ = FALSE;
	if (self->priv->nit_arrived) {
		_tmp4_ = self->priv->sdt_arrived;
	} else {
		_tmp4_ = FALSE;
	}
	/* NIT gives us the transport stream, SDT links SID and TS ID */
	if (_tmp4_) {
		{
			GeeIterator* _sid_it;
			_sid_it = gee_iterable_iterator ((GeeIterable*) self->priv->new_channels);
			while (TRUE) {
				guint sid;
				DVBChannel* channel;
				guint tsid;
				if (!gee_iterator_next (_sid_it)) {
					break;
				}
				sid = GPOINTER_TO_UINT (gee_iterator_get (_sid_it));
				channel = dvb_channel_list_get_channel (self->channels, sid);
				tsid = dvb_channel_get_TransportStreamId (channel);
				/* Check if already came across the transport stream*/
				if (gee_map_contains ((GeeMap*) self->transport_streams, GUINT_TO_POINTER (tsid))) {
					GstStructure* _tmp5_;
					/* add values from Gst.Structure to Channel*/
					_tmp5_ = NULL;
					dvb_scanner_add_values_from_structure_to_channel (self, _tmp5_ = (GstStructure*) gee_map_get ((GeeMap*) self->transport_streams, GUINT_TO_POINTER (tsid)), channel);
					(_tmp5_ == NULL) ? NULL : (_tmp5_ = (gst_structure_free (_tmp5_), NULL));
				} else {
					g_debug ("Scanner.vala:656: TS %u for channel %u does not exist", tsid, sid);
					/* We haven't seen the transport stream, yet
					 Maybe it comes with a later bus message*/
					self->priv->nit_arrived = FALSE;
				}
				(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
			}
			(_sid_it == NULL) ? NULL : (_sid_it = (g_object_unref (_sid_it), NULL));
		}
		/* We received all tables at least once. Add valid channels.*/
		if (self->priv->pat_arrived) {
			GeeArrayList* del_channels;
			del_channels = gee_array_list_new (G_TYPE_UINT, NULL, NULL, g_direct_equal);
			{
				GeeIterator* _sid_it;
				_sid_it = gee_iterable_iterator ((GeeIterable*) self->priv->new_channels);
				while (TRUE) {
					guint sid;
					DVBChannel* channel;
					if (!gee_iterator_next (_sid_it)) {
						break;
					}
					sid = GPOINTER_TO_UINT (gee_iterator_get (_sid_it));
					channel = dvb_channel_list_get_channel (self->channels, sid);
					/* If this fails we may miss video or audio pid,
					 because we didn't came across the sdt or pmt, yet   */
					if (dvb_channel_is_valid (channel)) {
						const char* _tmp6_;
						const char* _tmp7_;
						char* type;
						char* _tmp8_;
						_tmp6_ = NULL;
						if (dvb_channel_get_VideoPID (channel) == 0) {
							_tmp6_ = "Radio";
						} else {
							_tmp6_ = "TV";
						}
						_tmp7_ = NULL;
						type = (_tmp7_ = _tmp6_, (_tmp7_ == NULL) ? NULL : g_strdup (_tmp7_));
						_tmp8_ = NULL;
						g_debug ("Scanner.vala:674: Channel added: %s", _tmp8_ = dvb_channel_to_string (channel));
						_tmp8_ = (g_free (_tmp8_), NULL);
						g_signal_emit_by_name (self, "channel-added", dvb_channel_get_Frequency (channel), sid, dvb_channel_get_Name (channel), dvb_channel_get_Network (channel), type, dvb_channel_get_Scrambled (channel));
						/* Mark channel for deletion of this.new_channels*/
						gee_collection_add ((GeeCollection*) del_channels, GUINT_TO_POINTER (sid));
						type = (g_free (type), NULL);
					} else {
						char* _tmp9_;
						_tmp9_ = NULL;
						g_debug ("Scanner.vala:681: Channel %u is not valid: %s", sid, _tmp9_ = dvb_channel_to_string (channel));
						_tmp9_ = (g_free (_tmp9_), NULL);
						self->priv->pmt_arrived = FALSE;
					}
					(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
				}
				(_sid_it == NULL) ? NULL : (_sid_it = (g_object_unref (_sid_it), NULL));
			}
			/* Only remove those channels we have all the information for*/
			{
				GeeIterator* _sid_it;
				/* Only remove those channels we have all the information for*/
				_sid_it = gee_iterable_iterator ((GeeIterable*) del_channels);
				/* Only remove those channels we have all the information for*/
				while (TRUE) {
					guint sid;
					/* Only remove those channels we have all the information for*/
					if (!gee_iterator_next (_sid_it)) {
						/* Only remove those channels we have all the information for*/
						break;
					}
					/* Only remove those channels we have all the information for*/
					sid = GPOINTER_TO_UINT (gee_iterator_get (_sid_it));
					gee_collection_remove ((GeeCollection*) self->priv->new_channels, GUINT_TO_POINTER (sid));
				}
				(_sid_it == NULL) ? NULL : (_sid_it = (g_object_unref (_sid_it), NULL));
			}
			(del_channels == NULL) ? NULL : (del_channels = (g_object_unref (del_channels), NULL));
		}
	}
	_tmp10_ = FALSE;
	_tmp11_ = FALSE;
	_tmp12_ = FALSE;
	if (self->priv->sdt_arrived) {
		_tmp12_ = self->priv->nit_arrived;
	} else {
		_tmp12_ = FALSE;
	}
	if (_tmp12_) {
		_tmp11_ = self->priv->pat_arrived;
	} else {
		_tmp11_ = FALSE;
	}
	if (_tmp11_) {
		_tmp10_ = self->priv->pmt_arrived;
	} else {
		_tmp10_ = FALSE;
	}
	/* If we collect all information we can continue scanning
	 the next frequency*/
	if (_tmp10_) {
		dvb_scanner_remove_wait_for_tables_timeout (self);
		dvb_scanner_start_scan (self);
	}
}


void dvb_scanner_add_new_channel (DVBScanner* self, guint sid) {
	DVBChannel* new_channel;
	g_return_if_fail (self != NULL);
	g_debug ("Scanner.vala:705: Adding new channel with SID %u", sid);
	new_channel = dvb_scanner_get_new_channel (self);
	dvb_channel_set_Sid (new_channel, sid);
	dvb_channel_list_add (self->channels, new_channel);
	gee_collection_add ((GeeCollection*) self->priv->new_channels, GUINT_TO_POINTER (sid));
	(new_channel == NULL) ? NULL : (new_channel = (g_object_unref (new_channel), NULL));
}


DVBDevice* dvb_scanner_get_Device (DVBScanner* self) {
	DVBDevice* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_Device;
	return result;
}


static void dvb_scanner_set_Device (DVBScanner* self, DVBDevice* value) {
	DVBDevice* _tmp1_;
	DVBDevice* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_Device = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_Device == NULL) ? NULL : (self->priv->_Device = (g_object_unref (self->priv->_Device), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "Device");
}


DVBChannelList* dvb_scanner_get_Channels (DVBScanner* self) {
	DVBChannelList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->channels;
	return result;
}


static guint _dvb_scanned_item_hash_ghash_func (void* key) {
	return dvb_scanned_item_hash (key);
}


static gboolean _dvb_scanned_item_equal_gequal_func (void* a, void* b) {
	return dvb_scanned_item_equal (a, b);
}


static GObject * dvb_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBScannerClass * klass;
	GObjectClass * parent_class;
	DVBScanner * self;
	klass = DVB_SCANNER_CLASS (g_type_class_peek (DVB_TYPE_SCANNER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_SCANNER (obj);
	{
		GeeHashSet* _tmp0_;
		GeeArrayList* _tmp1_;
		GQueue* _tmp2_;
		GeeHashMap* _tmp3_;
		_tmp0_ = NULL;
		self->scanned_frequencies = (_tmp0_ = gee_hash_set_new (DVB_TYPE_SCANNED_ITEM, (GBoxedCopyFunc) g_object_ref, g_object_unref, _dvb_scanned_item_hash_ghash_func, _dvb_scanned_item_equal_gequal_func), (self->scanned_frequencies == NULL) ? NULL : (self->scanned_frequencies = (g_object_unref (self->scanned_frequencies), NULL)), _tmp0_);
		_tmp1_ = NULL;
		self->priv->new_channels = (_tmp1_ = gee_array_list_new (G_TYPE_UINT, NULL, NULL, g_direct_equal), (self->priv->new_channels == NULL) ? NULL : (self->priv->new_channels = (g_object_unref (self->priv->new_channels), NULL)), _tmp1_);
		_tmp2_ = NULL;
		self->frequencies = (_tmp2_ = g_queue_new (), (self->frequencies == NULL) ? NULL : (self->frequencies = (g_queue_free (self->frequencies), NULL)), _tmp2_);
		_tmp3_ = NULL;
		self->transport_streams = (_tmp3_ = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, G_TYPE_POINTER, (GBoxedCopyFunc) gst_structure_copy, gst_structure_free, g_direct_hash, g_direct_equal, g_direct_equal), (self->transport_streams == NULL) ? NULL : (self->transport_streams = (g_object_unref (self->transport_streams), NULL)), _tmp3_);
	}
	return obj;
}


static void dvb_scanner_class_init (DVBScannerClass * klass) {
	dvb_scanner_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBScannerPrivate));
	DVB_SCANNER_CLASS (klass)->prepare = dvb_scanner_real_prepare;
	DVB_SCANNER_CLASS (klass)->get_scanned_item = dvb_scanner_real_get_scanned_item;
	DVB_SCANNER_CLASS (klass)->get_new_channel = dvb_scanner_real_get_new_channel;
	DVB_SCANNER_CLASS (klass)->add_values_from_structure_to_channel = dvb_scanner_real_add_values_from_structure_to_channel;
	G_OBJECT_CLASS (klass)->get_property = dvb_scanner_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_scanner_set_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_scanner_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_scanner_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_SCANNER_DEVICE, g_param_spec_object ("Device", "Device", "Device", DVB_TYPE_DEVICE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_SCANNER_CHANNELS, g_param_spec_object ("Channels", "Channels", "Channels", DVB_TYPE_CHANNEL_LIST, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_signal_new ("destroyed", DVB_TYPE_SCANNER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
	g_signal_new ("frequency_scanned", DVB_TYPE_SCANNER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT, G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
	g_signal_new ("channel_added", DVB_TYPE_SCANNER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN, G_TYPE_NONE, 6, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_BOOLEAN);
	g_signal_new ("finished", DVB_TYPE_SCANNER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}


static void dvb_scanner_instance_init (DVBScanner * self) {
	self->priv = DVB_SCANNER_GET_PRIVATE (self);
}


static void dvb_scanner_finalize (GObject* obj) {
	DVBScanner * self;
	self = DVB_SCANNER (obj);
	(self->priv->_Device == NULL) ? NULL : (self->priv->_Device = (g_object_unref (self->priv->_Device), NULL));
	(self->channels == NULL) ? NULL : (self->channels = (g_object_unref (self->channels), NULL));
	(self->pipeline == NULL) ? NULL : (self->pipeline = (gst_object_unref (self->pipeline), NULL));
	(self->frequencies == NULL) ? NULL : (self->frequencies = (g_queue_free (self->frequencies), NULL));
	(self->current_tuning_params == NULL) ? NULL : (self->current_tuning_params = (gst_structure_free (self->current_tuning_params), NULL));
	(self->scanned_frequencies == NULL) ? NULL : (self->scanned_frequencies = (g_object_unref (self->scanned_frequencies), NULL));
	(self->transport_streams == NULL) ? NULL : (self->transport_streams = (g_object_unref (self->transport_streams), NULL));
	(self->priv->new_channels == NULL) ? NULL : (self->priv->new_channels = (g_object_unref (self->priv->new_channels), NULL));
	G_OBJECT_CLASS (dvb_scanner_parent_class)->finalize (obj);
}


GType dvb_scanner_get_type (void) {
	static GType dvb_scanner_type_id = 0;
	if (dvb_scanner_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBScannerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_scanner_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBScanner), 0, (GInstanceInitFunc) dvb_scanner_instance_init, NULL };
		dvb_scanner_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBScanner", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return dvb_scanner_type_id;
}


static void dvb_scanner_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBScanner * self;
	gpointer boxed;
	self = DVB_SCANNER (object);
	switch (property_id) {
		case DVB_SCANNER_DEVICE:
		g_value_set_object (value, dvb_scanner_get_Device (self));
		break;
		case DVB_SCANNER_CHANNELS:
		g_value_set_object (value, dvb_scanner_get_Channels (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_scanner_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBScanner * self;
	self = DVB_SCANNER (object);
	switch (property_id) {
		case DVB_SCANNER_DEVICE:
		dvb_scanner_set_Device (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}



static void g_cclosure_user_marshal_VOID__UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__UINT_UINT) (gpointer data1, guint arg_1, guint arg_2, gpointer data2);
	register GMarshalFunc_VOID__UINT_UINT callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = (GCClosure *) closure;
	g_return_if_fail (n_param_values == 3);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = (GMarshalFunc_VOID__UINT_UINT) (marshal_data ? marshal_data : cc->callback);
	callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), data2);
}


static void g_cclosure_user_marshal_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN) (gpointer data1, guint arg_1, guint arg_2, const char* arg_3, const char* arg_4, const char* arg_5, gboolean arg_6, gpointer data2);
	register GMarshalFunc_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = (GCClosure *) closure;
	g_return_if_fail (n_param_values == 7);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = (GMarshalFunc_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN) (marshal_data ? marshal_data : cc->callback);
	callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), g_value_get_string (param_values + 3), g_value_get_string (param_values + 4), g_value_get_string (param_values + 5), g_value_get_boolean (param_values + 6), data2);
}



