/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <gst/gst.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <gobject/gvaluecollector.h>


#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_PLAYER_THREAD (dvb_player_thread_get_type ())
#define DVB_PLAYER_THREAD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_PLAYER_THREAD, DVBPlayerThread))
#define DVB_PLAYER_THREAD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_PLAYER_THREAD, DVBPlayerThreadClass))
#define DVB_IS_PLAYER_THREAD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_PLAYER_THREAD))
#define DVB_IS_PLAYER_THREAD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_PLAYER_THREAD))
#define DVB_PLAYER_THREAD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_PLAYER_THREAD, DVBPlayerThreadClass))

typedef struct _DVBPlayerThread DVBPlayerThread;
typedef struct _DVBPlayerThreadClass DVBPlayerThreadClass;
typedef struct _DVBPlayerThreadPrivate DVBPlayerThreadPrivate;

#define DVB_TYPE_DEVICE (dvb_device_get_type ())
#define DVB_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE, DVBDevice))
#define DVB_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE, DVBDeviceClass))
#define DVB_IS_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE))
#define DVB_IS_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE))
#define DVB_DEVICE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE, DVBDeviceClass))

typedef struct _DVBDevice DVBDevice;
typedef struct _DVBDeviceClass DVBDeviceClass;

#define DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS (dvb_player_thread_channel_elements_get_type ())
#define DVB_PLAYER_THREAD_CHANNEL_ELEMENTS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS, DVBPlayerThreadChannelElements))
#define DVB_PLAYER_THREAD_CHANNEL_ELEMENTS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS, DVBPlayerThreadChannelElementsClass))
#define DVB_PLAYER_THREAD_IS_CHANNEL_ELEMENTS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS))
#define DVB_PLAYER_THREAD_IS_CHANNEL_ELEMENTS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS))
#define DVB_PLAYER_THREAD_CHANNEL_ELEMENTS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS, DVBPlayerThreadChannelElementsClass))

typedef struct _DVBPlayerThreadChannelElements DVBPlayerThreadChannelElements;
typedef struct _DVBPlayerThreadChannelElementsClass DVBPlayerThreadChannelElementsClass;

#define DVB_TYPE_EPG_SCANNER (dvb_epg_scanner_get_type ())
#define DVB_EPG_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EPG_SCANNER, DVBEPGScanner))
#define DVB_EPG_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EPG_SCANNER, DVBEPGScannerClass))
#define DVB_IS_EPG_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EPG_SCANNER))
#define DVB_IS_EPG_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EPG_SCANNER))
#define DVB_EPG_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EPG_SCANNER, DVBEPGScannerClass))

typedef struct _DVBEPGScanner DVBEPGScanner;
typedef struct _DVBEPGScannerClass DVBEPGScannerClass;
typedef struct _DVBPlayerThreadChannelElementsPrivate DVBPlayerThreadChannelElementsPrivate;

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;
typedef struct _DVBPlayerThreadParamSpecChannelElements DVBPlayerThreadParamSpecChannelElements;

#define DVB_TYPE_CHANNEL_FACTORY (dvb_channel_factory_get_type ())
#define DVB_CHANNEL_FACTORY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_FACTORY, DVBChannelFactory))
#define DVB_CHANNEL_FACTORY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_FACTORY, DVBChannelFactoryClass))
#define DVB_IS_CHANNEL_FACTORY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_FACTORY))
#define DVB_IS_CHANNEL_FACTORY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_FACTORY))
#define DVB_CHANNEL_FACTORY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_FACTORY, DVBChannelFactoryClass))

typedef struct _DVBChannelFactory DVBChannelFactory;
typedef struct _DVBChannelFactoryClass DVBChannelFactoryClass;
typedef struct _DVBChannelFactoryPrivate DVBChannelFactoryPrivate;

#define DVB_TYPE_DEVICE_GROUP (dvb_device_group_get_type ())
#define DVB_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroup))
#define DVB_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))
#define DVB_IS_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE_GROUP))
#define DVB_IS_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE_GROUP))
#define DVB_DEVICE_GROUP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))

typedef struct _DVBDeviceGroup DVBDeviceGroup;
typedef struct _DVBDeviceGroupClass DVBDeviceGroupClass;

typedef void (*DVBForcedStopNotify) (DVBChannel* channel);
/**
     * This class handles watching channels one physical device.
     * 
     * It's possible to watch multiple channels at the same time
     * if they are all on the same transport stream.
     *
     * The class is able to reuse channels that are already watched
     * and forward EPG data to #EPGScanner.
     */
struct _DVBPlayerThread {
	GObject parent_instance;
	DVBPlayerThreadPrivate * priv;
};

struct _DVBPlayerThreadClass {
	GObjectClass parent_class;
	void (*destroy) (DVBPlayerThread* self, gboolean forced);
};

struct _DVBPlayerThreadPrivate {
	GeeHashSet* _active_channels;
	DVBDevice* _device;
	GstElement* pipeline;
	GStaticRecMutex __lock_pipeline;
	GeeHashMap* elements_map;
	GStaticRecMutex __lock_elements_map;
	DVBEPGScanner* epgscanner;
	GstElement* dvbbasebin;
	gboolean destroyed;
	GStaticRecMutex __lock_destroyed;
};

struct _DVBPlayerThreadChannelElements {
	GTypeInstance parent_instance;
	volatile int ref_count;
	DVBPlayerThreadChannelElementsPrivate * priv;
	guint sid;
	GeeArrayList* sinks;
	GstElement* tee;
	gboolean forced;
	DVBForcedStopNotify notify_func;
};

struct _DVBPlayerThreadChannelElementsClass {
	GTypeClass parent_class;
	void (*finalize) (DVBPlayerThreadChannelElements *self);
};

struct _DVBPlayerThreadParamSpecChannelElements {
	GParamSpec parent_instance;
};

/**
     * This class handles watching channels for a single #DeviceGroup
     */
struct _DVBChannelFactory {
	GObject parent_instance;
	DVBChannelFactoryPrivate * priv;
};

struct _DVBChannelFactoryClass {
	GObjectClass parent_class;
	DVBPlayerThread* (*create_player) (DVBChannelFactory* self, DVBDevice* device);
};

struct _DVBChannelFactoryPrivate {
	DVBDeviceGroup* _device_group;
	GeeHashSet* active_players;
	GStaticRecMutex __lock_active_players;
};


static gpointer dvb_player_thread_channel_elements_parent_class = NULL;
static gpointer dvb_player_thread_parent_class = NULL;
static gpointer dvb_channel_factory_parent_class = NULL;

GType dvb_channel_get_type (void);
GType dvb_player_thread_get_type (void);
GType dvb_device_get_type (void);
static gpointer dvb_player_thread_channel_elements_ref (gpointer instance);
static void dvb_player_thread_channel_elements_unref (gpointer instance);
static GParamSpec* dvb_player_thread_param_spec_channel_elements (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
static void dvb_player_thread_value_set_channel_elements (GValue* value, gpointer v_object);
static gpointer dvb_player_thread_value_get_channel_elements (const GValue* value);
static GType dvb_player_thread_channel_elements_get_type (void);
GType dvb_epg_scanner_get_type (void);
#define DVB_PLAYER_THREAD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_PLAYER_THREAD, DVBPlayerThreadPrivate))
enum  {
	DVB_PLAYER_THREAD_DUMMY_PROPERTY,
	DVB_PLAYER_THREAD_ACTIVE_CHANNELS,
	DVB_PLAYER_THREAD_DEVICE,
	DVB_PLAYER_THREAD_FORCED
};
static void dvb_player_thread_set_device (DVBPlayerThread* self, DVBDevice* value);
DVBPlayerThread* dvb_player_thread_new (DVBDevice* device, DVBEPGScanner* epgscanner);
DVBPlayerThread* dvb_player_thread_construct (GType object_type, DVBDevice* device, DVBEPGScanner* epgscanner);
GstElement* dvb_player_thread_get_pipeline (DVBPlayerThread* self);
GstElement* dvb_player_thread_get_dvbbasebin (DVBPlayerThread* self);
guint dvb_channel_get_Sid (DVBChannel* self);
static void dvb_player_thread_bus_watch_func (DVBPlayerThread* self, GstBus* bus, GstMessage* message);
static void _dvb_player_thread_bus_watch_func_gst_bus_message (GstBus* _sender, GstMessage* message, gpointer self);
static void dvb_player_thread_on_dvbbasebin_pad_added (DVBPlayerThread* self, GstElement* elem, GstPad* pad);
static void _dvb_player_thread_on_dvbbasebin_pad_added_gst_element_pad_added (GstElement* _sender, GstPad* pad, gpointer self);
void dvb_channel_setup_dvb_source (DVBChannel* self, GstElement* source);
DVBDevice* dvb_player_thread_get_device (DVBPlayerThread* self);
guint dvb_device_get_Adapter (DVBDevice* self);
guint dvb_device_get_Frontend (DVBDevice* self);
static gboolean dvb_player_thread_add_element (DVBPlayerThread* self, GstElement* elem);
static GstElement* dvb_player_thread_add_sink_bin (DVBPlayerThread* self, GstElement* sink_element);
GeeHashSet* dvb_player_thread_get_active_channels (DVBPlayerThread* self);
static DVBPlayerThreadChannelElements* dvb_player_thread_channel_elements_new (void);
static DVBPlayerThreadChannelElements* dvb_player_thread_channel_elements_construct (GType object_type);
GstElement* dvb_player_thread_get_element (DVBPlayerThread* self, DVBChannel* channel, GstElement* sink_element, gboolean forced, DVBForcedStopNotify notify_func);
static gint dvb_player_thread_find_element (void* av, void* bv);
static gint _dvb_player_thread_find_element_gcompare_func (void* a, void* b);
GstElement* dvb_player_thread_get_sink_bin (DVBPlayerThread* self, guint sid, GstElement* sink);
static void _g_slist_free_g_free (GSList* self);
static void dvb_player_thread_remove_sink_bin (DVBPlayerThread* self, guint channel_sid, GstElement* sink);
void dvb_player_thread_destroy (DVBPlayerThread* self, gboolean forced);
gboolean dvb_player_thread_remove_channel (DVBPlayerThread* self, DVBChannel* channel, GstElement* sink);
GType dvb_channel_list_get_type (void);
DVBChannelList* dvb_device_get_Channels (DVBDevice* self);
DVBChannel* dvb_channel_list_get_channel (DVBChannelList* self, guint sid);
static void dvb_player_thread_real_destroy (DVBPlayerThread* self, gboolean forced);
void dvb_epg_scanner_on_eit_structure (DVBEPGScanner* self, const GstStructure* structure);
static void dvb_player_thread_set_active_channels (DVBPlayerThread* self, GeeHashSet* value);
gboolean dvb_player_thread_get_forced (DVBPlayerThread* self);
static GObject * dvb_player_thread_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
enum  {
	DVB_PLAYER_THREAD_CHANNEL_ELEMENTS_DUMMY_PROPERTY
};
static void dvb_player_thread_channel_elements_finalize (DVBPlayerThreadChannelElements* obj);
static void dvb_player_thread_finalize (GObject* obj);
static void dvb_player_thread_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_player_thread_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
GType dvb_channel_factory_get_type (void);
GType dvb_device_group_get_type (void);
#define DVB_CHANNEL_FACTORY_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_CHANNEL_FACTORY, DVBChannelFactoryPrivate))
enum  {
	DVB_CHANNEL_FACTORY_DUMMY_PROPERTY,
	DVB_CHANNEL_FACTORY_DEVICE_GROUP
};
static void dvb_channel_factory_set_device_group (DVBChannelFactory* self, DVBDeviceGroup* value);
DVBChannelFactory* dvb_channel_factory_new (DVBDeviceGroup* devgroup);
DVBChannelFactory* dvb_channel_factory_construct (GType object_type, DVBDeviceGroup* devgroup);
void dvb_channel_factory_destroy (DVBChannelFactory* self);
const char* dvb_channel_get_Name (DVBChannel* self);
gboolean dvb_channel_on_same_transport_stream (DVBChannel* self, DVBChannel* channel);
DVBDeviceGroup* dvb_channel_factory_get_device_group (DVBChannelFactory* self);
DVBEPGScanner* dvb_device_group_get_epgscanner (DVBDeviceGroup* self);
void dvb_epg_scanner_stop (DVBEPGScanner* self);
DVBDevice* dvb_device_group_get_next_free_device (DVBDeviceGroup* self);
DVBPlayerThread* dvb_channel_factory_create_player (DVBChannelFactory* self, DVBDevice* device);
DVBPlayerThread* dvb_channel_factory_watch_channel (DVBChannelFactory* self, DVBChannel* channel, GstElement* sink_element, gboolean force, DVBForcedStopNotify notify_func);
gboolean dvb_epg_scanner_start (DVBEPGScanner* self);
gboolean dvb_channel_factory_stop_channel (DVBChannelFactory* self, DVBChannel* channel, GstElement* sink);
static DVBPlayerThread* dvb_channel_factory_real_create_player (DVBChannelFactory* self, DVBDevice* device);
static void dvb_channel_factory_finalize (GObject* obj);
static void dvb_channel_factory_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_channel_factory_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
         * @device: The device to use
         * @epgscanner: #EPGScanner to forward EIT to
         */
DVBPlayerThread* dvb_player_thread_construct (GType object_type, DVBDevice* device, DVBEPGScanner* epgscanner) {
	GParameter * __params;
	GParameter * __params_it;
	DVBPlayerThread * self;
	DVBEPGScanner* _tmp1_;
	DVBEPGScanner* _tmp0_;
	g_return_val_if_fail (device != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "device";
	g_value_init (&__params_it->value, DVB_TYPE_DEVICE);
	g_value_set_object (&__params_it->value, device);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->epgscanner = (_tmp1_ = (_tmp0_ = epgscanner, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->epgscanner == NULL) ? NULL : (self->priv->epgscanner = (g_object_unref (self->priv->epgscanner), NULL)), _tmp1_);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBPlayerThread* dvb_player_thread_new (DVBDevice* device, DVBEPGScanner* epgscanner) {
	return dvb_player_thread_construct (DVB_TYPE_PLAYER_THREAD, device, epgscanner);
}


GstElement* dvb_player_thread_get_pipeline (DVBPlayerThread* self) {
	GstElement* result;
	GstElement* _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0_ = NULL;
	result = (_tmp0_ = self->priv->pipeline, (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_));
	return result;
}


GstElement* dvb_player_thread_get_dvbbasebin (DVBPlayerThread* self) {
	GstElement* result;
	GstElement* _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0_ = NULL;
	result = (_tmp0_ = self->priv->dvbbasebin, (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_));
	return result;
}


static void _dvb_player_thread_bus_watch_func_gst_bus_message (GstBus* _sender, GstMessage* message, gpointer self) {
	dvb_player_thread_bus_watch_func (self, _sender, message);
}


static void _dvb_player_thread_on_dvbbasebin_pad_added_gst_element_pad_added (GstElement* _sender, GstPad* pad, gpointer self) {
	dvb_player_thread_on_dvbbasebin_pad_added (self, _sender, pad);
}


/**
         * @returns: GstBin containing queue and @sink_element 
         *
         * Start watching @channel and link it with @sink_element
         */
GstElement* dvb_player_thread_get_element (DVBPlayerThread* self, DVBChannel* channel, GstElement* sink_element, gboolean forced, DVBForcedStopNotify notify_func) {
	GstElement* result;
	guint channel_sid;
	char* channel_sid_str;
	gboolean create_channel;
	GstElement* bin;
	GstElement* tee;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (channel != NULL, NULL);
	g_return_val_if_fail (sink_element != NULL, NULL);
	channel_sid = dvb_channel_get_Sid (channel);
	channel_sid_str = g_strdup_printf ("%u", channel_sid);
	create_channel = FALSE;
	bin = NULL;
	tee = NULL;
	if (self->priv->pipeline == NULL) {
		/* New channel and new pipeline*/
		g_debug ("ChannelFactory.vala:114: Creating new pipeline");
		g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
		{
			GstElement* _tmp0_;
			GstBus* bus;
			GstElement* _tmp1_;
			GstElement* _tmp2_;
			GstElement* _tmp3_;
			GstElement* _tmp4_;
			GstElement* _tmp5_;
			/* Setup new pipeline*/
			_tmp0_ = NULL;
			self->priv->pipeline = (_tmp0_ = (GstElement*) ((GstPipeline*) gst_pipeline_new ("recording")), (self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL)), _tmp0_);
			bus = gst_element_get_bus (self->priv->pipeline);
			gst_bus_add_signal_watch (bus);
			g_signal_connect_object (bus, "message", (GCallback) _dvb_player_thread_bus_watch_func_gst_bus_message, self, 0);
			_tmp1_ = NULL;
			self->priv->dvbbasebin = (_tmp1_ = gst_element_factory_make ("dvbbasebin", NULL), (self->priv->dvbbasebin == NULL) ? NULL : (self->priv->dvbbasebin = (gst_object_unref (self->priv->dvbbasebin), NULL)), _tmp1_);
			if (self->priv->dvbbasebin == NULL) {
				g_critical ("ChannelFactory.vala:125: Could not create dvbbasebin element");
				result = NULL;
				(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
				channel_sid_str = (g_free (channel_sid_str), NULL);
				(bin == NULL) ? NULL : (bin = (gst_object_unref (bin), NULL));
				(tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL));
				(sink_element == NULL) ? NULL : (sink_element = (gst_object_unref (sink_element), NULL));
				return result;
			}
			g_signal_connect_object (self->priv->dvbbasebin, "pad-added", (GCallback) _dvb_player_thread_on_dvbbasebin_pad_added_gst_element_pad_added, self, 0);
			dvb_channel_setup_dvb_source (channel, self->priv->dvbbasebin);
			g_object_set ((GObject*) self->priv->dvbbasebin, "program-numbers", channel_sid_str, NULL);
			g_object_set ((GObject*) self->priv->dvbbasebin, "adapter", dvb_device_get_Adapter (self->priv->_device), NULL);
			g_object_set ((GObject*) self->priv->dvbbasebin, "frontend", dvb_device_get_Frontend (self->priv->_device), NULL);
			/* don't use add_many because of problems with ownership transfer    */
			_tmp2_ = NULL;
			gst_bin_add (GST_BIN (self->priv->pipeline), (_tmp2_ = self->priv->dvbbasebin, (_tmp2_ == NULL) ? NULL : gst_object_ref (_tmp2_)));
			_tmp3_ = NULL;
			tee = (_tmp3_ = gst_element_factory_make ("tee", NULL), (tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL)), _tmp3_);
			_tmp4_ = NULL;
			dvb_player_thread_add_element (self, (_tmp4_ = tee, (_tmp4_ == NULL) ? NULL : gst_object_ref (_tmp4_)));
			_tmp5_ = NULL;
			bin = (_tmp5_ = dvb_player_thread_add_sink_bin (self, sink_element), (bin == NULL) ? NULL : (bin = (gst_object_unref (bin), NULL)), _tmp5_);
			if (!gst_element_link (tee, bin)) {
				g_critical ("ChannelFactory.vala:144: Could not link tee and bin");
				result = NULL;
				(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
				channel_sid_str = (g_free (channel_sid_str), NULL);
				(bin == NULL) ? NULL : (bin = (gst_object_unref (bin), NULL));
				(tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL));
				(sink_element == NULL) ? NULL : (sink_element = (gst_object_unref (sink_element), NULL));
				return result;
			}
			create_channel = TRUE;
			(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
		}
		g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
	} else {
		/* Use current pipeline and add new sink*/
		g_debug ("ChannelFactory.vala:152: Reusing existing pipeline");
		if (self->priv->dvbbasebin == NULL) {
			g_critical ("ChannelFactory.vala:154: No dvbbasebin element");
			result = NULL;
			channel_sid_str = (g_free (channel_sid_str), NULL);
			(bin == NULL) ? NULL : (bin = (gst_object_unref (bin), NULL));
			(tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL));
			(sink_element == NULL) ? NULL : (sink_element = (gst_object_unref (sink_element), NULL));
			return result;
		}
		if (!gee_abstract_collection_contains ((GeeAbstractCollection*) self->priv->_active_channels, channel)) {
			g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
			{
				GstElement* _tmp6_;
				GstElement* _tmp7_;
				GstElement* _tmp8_;
				char* programs;
				char* new_programs;
				_tmp6_ = NULL;
				tee = (_tmp6_ = gst_element_factory_make ("tee", NULL), (tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL)), _tmp6_);
				_tmp7_ = NULL;
				dvb_player_thread_add_element (self, (_tmp7_ = tee, (_tmp7_ == NULL) ? NULL : gst_object_ref (_tmp7_)));
				_tmp8_ = NULL;
				bin = (_tmp8_ = dvb_player_thread_add_sink_bin (self, sink_element), (bin == NULL) ? NULL : (bin = (gst_object_unref (bin), NULL)), _tmp8_);
				if (!gst_element_link (tee, bin)) {
					g_critical ("ChannelFactory.vala:166: Could not link tee and bin");
					result = NULL;
					channel_sid_str = (g_free (channel_sid_str), NULL);
					(bin == NULL) ? NULL : (bin = (gst_object_unref (bin), NULL));
					(tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL));
					(sink_element == NULL) ? NULL : (sink_element = (gst_object_unref (sink_element), NULL));
					return result;
				}
				gst_element_set_state (self->priv->pipeline, GST_STATE_PAUSED);
				programs = NULL;
				g_object_get ((GObject*) self->priv->dvbbasebin, "program-numbers", &programs, NULL);
				new_programs = g_strdup_printf ("%s:%s", programs, channel_sid_str);
				g_debug ("ChannelFactory.vala:177: Changing program-numbers from %s to %s", programs, new_programs);
				g_object_set ((GObject*) self->priv->dvbbasebin, "program-numbers", new_programs, NULL);
				create_channel = TRUE;
				programs = (g_free (programs), NULL);
				new_programs = (g_free (new_programs), NULL);
			}
			g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
		} else {
			DVBPlayerThreadChannelElements* c_element;
			GstElement* _tmp11_;
			GstElement* _tmp10_;
			char* _tmp14_;
			char* _tmp13_;
			/* existing pipeline and existing channel*/
			c_element = NULL;
			g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
			{
				DVBPlayerThreadChannelElements* _tmp9_;
				_tmp9_ = NULL;
				c_element = (_tmp9_ = (DVBPlayerThreadChannelElements*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->elements_map, GUINT_TO_POINTER (channel_sid)), (c_element == NULL) ? NULL : (c_element = (dvb_player_thread_channel_elements_unref (c_element), NULL)), _tmp9_);
			}
			g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
			_tmp11_ = NULL;
			_tmp10_ = NULL;
			tee = (_tmp11_ = (_tmp10_ = c_element->tee, (_tmp10_ == NULL) ? NULL : gst_object_ref (_tmp10_)), (tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL)), _tmp11_);
			g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
			{
				GstElement* _tmp12_;
				gst_element_set_state (self->priv->pipeline, GST_STATE_PAUSED);
				_tmp12_ = NULL;
				bin = (_tmp12_ = dvb_player_thread_add_sink_bin (self, sink_element), (bin == NULL) ? NULL : (bin = (gst_object_unref (bin), NULL)), _tmp12_);
			}
			g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
			_tmp14_ = NULL;
			_tmp13_ = NULL;
			g_debug ("ChannelFactory.vala:195: Linking %s with %s", _tmp13_ = gst_object_get_name ((GstObject*) tee), _tmp14_ = gst_object_get_name ((GstObject*) bin));
			_tmp14_ = (g_free (_tmp14_), NULL);
			_tmp13_ = (g_free (_tmp13_), NULL);
			if (!gst_element_link (tee, bin)) {
				g_critical ("ChannelFactory.vala:197: Could not link tee and bin");
				result = NULL;
				(c_element == NULL) ? NULL : (c_element = (dvb_player_thread_channel_elements_unref (c_element), NULL));
				channel_sid_str = (g_free (channel_sid_str), NULL);
				(bin == NULL) ? NULL : (bin = (gst_object_unref (bin), NULL));
				(tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL));
				(sink_element == NULL) ? NULL : (sink_element = (gst_object_unref (sink_element), NULL));
				return result;
			}
			gee_abstract_collection_add ((GeeAbstractCollection*) c_element->sinks, bin);
			create_channel = FALSE;
			(c_element == NULL) ? NULL : (c_element = (dvb_player_thread_channel_elements_unref (c_element), NULL));
		}
	}
	if (create_channel) {
		DVBPlayerThreadChannelElements* celems;
		GeeArrayList* _tmp15_;
		GstElement* _tmp17_;
		GstElement* _tmp16_;
		celems = dvb_player_thread_channel_elements_new ();
		celems->sid = channel_sid;
		_tmp15_ = NULL;
		celems->sinks = (_tmp15_ = gee_array_list_new (GST_TYPE_ELEMENT, (GBoxedCopyFunc) gst_object_ref, gst_object_unref, g_direct_equal), (celems->sinks == NULL) ? NULL : (celems->sinks = (g_object_unref (celems->sinks), NULL)), _tmp15_);
		gee_abstract_collection_add ((GeeAbstractCollection*) celems->sinks, bin);
		_tmp17_ = NULL;
		_tmp16_ = NULL;
		celems->tee = (_tmp17_ = (_tmp16_ = tee, (_tmp16_ == NULL) ? NULL : gst_object_ref (_tmp16_)), (celems->tee == NULL) ? NULL : (celems->tee = (gst_object_unref (celems->tee), NULL)), _tmp17_);
		celems->forced = forced;
		celems->notify_func = notify_func;
		g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
		{
			gee_abstract_map_set ((GeeAbstractMap*) self->priv->elements_map, GUINT_TO_POINTER (channel_sid), celems);
		}
		g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
		gee_abstract_collection_add ((GeeAbstractCollection*) self->priv->_active_channels, channel);
		(celems == NULL) ? NULL : (celems = (dvb_player_thread_channel_elements_unref (celems), NULL));
	}
	result = bin;
	channel_sid_str = (g_free (channel_sid_str), NULL);
	(tee == NULL) ? NULL : (tee = (gst_object_unref (tee), NULL));
	(sink_element == NULL) ? NULL : (sink_element = (gst_object_unref (sink_element), NULL));
	return result;
}


static GstElement* dvb_player_thread_add_sink_bin (DVBPlayerThread* self, GstElement* sink_element) {
	GstElement* result;
	GstElement* queue;
	GstElement* bin;
	GstElement* _tmp1_;
	GstElement* _tmp0_;
	GstPad* pad;
	GstElement* _tmp4_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sink_element != NULL, NULL);
	queue = gst_element_factory_make ("queue", NULL);
	g_object_set ((GObject*) queue, "max-size-buffers", 0, NULL);
	bin = (GstElement*) ((GstBin*) gst_bin_new (NULL));
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	gst_bin_add_many (GST_BIN (bin), (_tmp0_ = queue, (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_)), (_tmp1_ = sink_element, (_tmp1_ == NULL) ? NULL : gst_object_ref (_tmp1_)), NULL);
	if (!gst_element_link (queue, sink_element)) {
		char* _tmp3_;
		char* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		g_critical ("ChannelFactory.vala:231: Could not link elements %s and %s", _tmp2_ = gst_object_get_name ((GstObject*) queue), _tmp3_ = gst_object_get_name ((GstObject*) sink_element));
		_tmp3_ = (g_free (_tmp3_), NULL);
		_tmp2_ = (g_free (_tmp2_), NULL);
	}
	pad = gst_element_get_static_pad (queue, "sink");
	gst_element_add_pad (bin, (GstPad*) ((GstGhostPad*) gst_ghost_pad_new ("sink", pad)));
	_tmp4_ = NULL;
	dvb_player_thread_add_element (self, (_tmp4_ = bin, (_tmp4_ == NULL) ? NULL : gst_object_ref (_tmp4_)));
	result = bin;
	(queue == NULL) ? NULL : (queue = (gst_object_unref (queue), NULL));
	(pad == NULL) ? NULL : (pad = (gst_object_unref (pad), NULL));
	return result;
}


static gint dvb_player_thread_find_element (void* av, void* bv) {
	gint result;
	GstElement* _tmp0_;
	GstElement* a;
	GstElement* _tmp1_;
	GstElement* b;
	_tmp0_ = NULL;
	a = (_tmp0_ = GST_ELEMENT (av), (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_));
	_tmp1_ = NULL;
	b = (_tmp1_ = GST_ELEMENT (bv), (_tmp1_ == NULL) ? NULL : gst_object_ref (_tmp1_));
	if (a == b) {
		result = 0;
		(a == NULL) ? NULL : (a = (gst_object_unref (a), NULL));
		(b == NULL) ? NULL : (b = (gst_object_unref (b), NULL));
		return result;
	} else {
		result = 1;
		(a == NULL) ? NULL : (a = (gst_object_unref (a), NULL));
		(b == NULL) ? NULL : (b = (gst_object_unref (b), NULL));
		return result;
	}
	(a == NULL) ? NULL : (a = (gst_object_unref (a), NULL));
	(b == NULL) ? NULL : (b = (gst_object_unref (b), NULL));
}


static gint _dvb_player_thread_find_element_gcompare_func (void* a, void* b) {
	return dvb_player_thread_find_element (a, b);
}


/**
         * @sid: Channel SID
         * @sink: The sink element that the bin should contain
         * @returns: GstBin containing queue and sink for the specified channel
         */
GstElement* dvb_player_thread_get_sink_bin (DVBPlayerThread* self, guint sid, GstElement* sink) {
	GstElement* result;
	GstElement* _result_;
	char* _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sink != NULL, NULL);
	_result_ = NULL;
	_tmp0_ = NULL;
	g_debug ("ChannelFactory.vala:257: Searching for sink %s (%p) of channel %u", _tmp0_ = gst_object_get_name ((GstObject*) sink), sink, sid);
	_tmp0_ = (g_free (_tmp0_), NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
	{
		DVBPlayerThreadChannelElements* celems;
		celems = (DVBPlayerThreadChannelElements*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->elements_map, GUINT_TO_POINTER (sid));
		if (celems != NULL) {
			{
				GeeIterator* _sink_bin_it;
				_sink_bin_it = gee_abstract_collection_iterator ((GeeAbstractCollection*) celems->sinks);
				while (TRUE) {
					GstElement* sink_bin;
					GstIterator* it;
					GstElement* _tmp1_;
					GstElement* element;
					if (!gee_iterator_next (_sink_bin_it)) {
						break;
					}
					sink_bin = (GstElement*) gee_iterator_get (_sink_bin_it);
					it = gst_bin_iterate_elements (GST_BIN (sink_bin));
					_tmp1_ = NULL;
					element = (_tmp1_ = GST_ELEMENT (gst_iterator_find_custom (it, _dvb_player_thread_find_element_gcompare_func, sink)), (_tmp1_ == NULL) ? NULL : gst_object_ref (_tmp1_));
					if (element != NULL) {
						GstElement* _tmp3_;
						GstElement* _tmp2_;
						_tmp3_ = NULL;
						_tmp2_ = NULL;
						_result_ = (_tmp3_ = (_tmp2_ = sink_bin, (_tmp2_ == NULL) ? NULL : gst_object_ref (_tmp2_)), (_result_ == NULL) ? NULL : (_result_ = (gst_object_unref (_result_), NULL)), _tmp3_);
						(sink_bin == NULL) ? NULL : (sink_bin = (gst_object_unref (sink_bin), NULL));
						(it == NULL) ? NULL : (it = (gst_iterator_free (it), NULL));
						(element == NULL) ? NULL : (element = (gst_object_unref (element), NULL));
						break;
					}
					(sink_bin == NULL) ? NULL : (sink_bin = (gst_object_unref (sink_bin), NULL));
					(it == NULL) ? NULL : (it = (gst_iterator_free (it), NULL));
					(element == NULL) ? NULL : (element = (gst_object_unref (element), NULL));
				}
				(_sink_bin_it == NULL) ? NULL : (_sink_bin_it = (g_object_unref (_sink_bin_it), NULL));
			}
		}
		(celems == NULL) ? NULL : (celems = (dvb_player_thread_channel_elements_unref (celems), NULL));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
	result = _result_;
	return result;
}


static void _g_slist_free_g_free (GSList* self) {
	g_slist_foreach (self, (GFunc) g_free, NULL);
	g_slist_free (self);
}


/**
         * Stop watching @channel
         */
gboolean dvb_player_thread_remove_channel (DVBPlayerThread* self, DVBChannel* channel, GstElement* sink) {
	gboolean result;
	guint channel_sid;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (channel != NULL, FALSE);
	g_return_val_if_fail (sink != NULL, FALSE);
	channel_sid = dvb_channel_get_Sid (channel);
	if (!gee_abstract_collection_contains ((GeeAbstractCollection*) self->priv->_active_channels, channel)) {
		g_critical ("ChannelFactory.vala:282: Could not find channel with SID %u", channel_sid);
		result = FALSE;
		return result;
	}
	/* Check if that's the only channel in use*/
	if (gee_collection_get_size ((GeeCollection*) self->priv->_active_channels) > 1) {
		char* channel_sid_string;
		channel_sid_string = g_strdup_printf ("%u", channel_sid);
		g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
		{
			gboolean stop_channel;
			stop_channel = FALSE;
			g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
			{
				DVBPlayerThreadChannelElements* celements;
				celements = (DVBPlayerThreadChannelElements*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->elements_map, GUINT_TO_POINTER (channel_sid));
				if (gee_collection_get_size ((GeeCollection*) celements->sinks) == 1) {
					/* this is the last sink
					 (no one watches this channel anymore)*/
					gee_abstract_map_remove ((GeeAbstractMap*) self->priv->elements_map, GUINT_TO_POINTER (channel_sid), NULL);
					stop_channel = TRUE;
				} else {
					/* we still have sinks left
					 (others are still watching this channel)*/
					gee_abstract_collection_remove ((GeeAbstractCollection*) celements->sinks, sink);
					stop_channel = FALSE;
				}
				(celements == NULL) ? NULL : (celements = (dvb_player_thread_channel_elements_unref (celements), NULL));
			}
			g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
			if (stop_channel) {
				char* programs;
				char** _tmp1_;
				gint programs_arr_size;
				gint programs_arr_length1;
				char** _tmp0_;
				char** programs_arr;
				GSList* new_programs_list;
				GString* new_programs;
				programs = NULL;
				g_object_get ((GObject*) self->priv->dvbbasebin, "program-numbers", &programs, NULL);
				_tmp1_ = NULL;
				_tmp0_ = NULL;
				programs_arr = (_tmp1_ = _tmp0_ = g_strsplit (programs, ":", 0), programs_arr_length1 = _vala_array_length (_tmp0_), programs_arr_size = programs_arr_length1, _tmp1_);
				/* Remove SID of channel from program-numbers*/
				new_programs_list = NULL;
				{
					gint i;
					i = 0;
					{
						gboolean _tmp2_;
						_tmp2_ = TRUE;
						while (TRUE) {
							const char* _tmp3_;
							char* val;
							if (!_tmp2_) {
								i++;
							}
							_tmp2_ = FALSE;
							if (!(i < programs_arr_length1)) {
								break;
							}
							_tmp3_ = NULL;
							val = (_tmp3_ = programs_arr[i], (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_));
							if (_vala_strcmp0 (val, channel_sid_string) != 0) {
								const char* _tmp4_;
								_tmp4_ = NULL;
								new_programs_list = g_slist_prepend (new_programs_list, (_tmp4_ = val, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)));
							}
							val = (g_free (val), NULL);
						}
					}
				}
				new_programs = g_string_new ((const char*) g_slist_nth_data (new_programs_list, (guint) 0));
				{
					gint i;
					i = 1;
					{
						gboolean _tmp5_;
						_tmp5_ = TRUE;
						while (TRUE) {
							char* _tmp6_;
							if (!_tmp5_) {
								i++;
							}
							_tmp5_ = FALSE;
							if (!(i < g_slist_length (new_programs_list))) {
								break;
							}
							_tmp6_ = NULL;
							g_string_append (new_programs, _tmp6_ = g_strconcat (":", (const char*) g_slist_nth_data (new_programs_list, (guint) i), NULL));
							_tmp6_ = (g_free (_tmp6_), NULL);
						}
					}
				}
				g_debug ("ChannelFactory.vala:325: Changing program-numbers from %s to %s", programs, new_programs->str);
				gst_element_set_state (self->priv->pipeline, GST_STATE_PAUSED);
				g_object_set ((GObject*) self->priv->dvbbasebin, "program-numbers", new_programs->str, NULL);
				gst_element_set_state (self->priv->pipeline, GST_STATE_PLAYING);
				gee_abstract_collection_remove ((GeeAbstractCollection*) self->priv->_active_channels, channel);
				programs = (g_free (programs), NULL);
				programs_arr = (_vala_array_free (programs_arr, programs_arr_length1, (GDestroyNotify) g_free), NULL);
				(new_programs_list == NULL) ? NULL : (new_programs_list = (_g_slist_free_g_free (new_programs_list), NULL));
				(new_programs == NULL) ? NULL : (new_programs = (g_string_free (new_programs, TRUE), NULL));
			}
			dvb_player_thread_remove_sink_bin (self, channel_sid, sink);
		}
		g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
		channel_sid_string = (g_free (channel_sid_string), NULL);
	} else {
		g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
		{
			DVBPlayerThreadChannelElements* celements;
			celements = (DVBPlayerThreadChannelElements*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->elements_map, GUINT_TO_POINTER (channel_sid));
			if (gee_collection_get_size ((GeeCollection*) celements->sinks) == 1) {
				/* this is the last sink
				 (no one watches this channel anymore)*/
				dvb_player_thread_destroy (self, FALSE);
			} else {
				/* we still have sinks left
				 (others are still watching this channel)*/
				dvb_player_thread_remove_sink_bin (self, channel_sid, sink);
				gee_abstract_collection_remove ((GeeAbstractCollection*) celements->sinks, sink);
			}
			(celements == NULL) ? NULL : (celements = (dvb_player_thread_channel_elements_unref (celements), NULL));
		}
		g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
	}
	result = TRUE;
	return result;
}


static void dvb_player_thread_remove_sink_bin (DVBPlayerThread* self, guint channel_sid, GstElement* sink) {
	GstElement* sink_bin;
	g_return_if_fail (self != NULL);
	g_return_if_fail (sink != NULL);
	sink_bin = dvb_player_thread_get_sink_bin (self, channel_sid, sink);
	if (sink_bin == NULL) {
		g_warning ("ChannelFactory.vala:361: Could not find sink bin for channel %u and sink %p", channel_sid, sink);
		(sink_bin == NULL) ? NULL : (sink_bin = (gst_object_unref (sink_bin), NULL));
		return;
	}
	g_debug ("ChannelFactory.vala:365: Setting state of queue and sink to NULL");
	gst_element_set_state (sink_bin, GST_STATE_NULL);
	g_debug ("ChannelFactory.vala:368: Removing queue and sink from pipeline");
	gst_bin_remove (GST_BIN (self->priv->pipeline), sink_bin);
	(sink_bin == NULL) ? NULL : (sink_bin = (gst_object_unref (sink_bin), NULL));
}


/**
         * Stop pipeline and clean up everything else
         */
static void dvb_player_thread_real_destroy (DVBPlayerThread* self, gboolean forced) {
	g_return_if_fail (self != NULL);
	if (self->priv->destroyed) {
		return;
	}
	g_static_rec_mutex_lock (&self->priv->__lock_destroyed);
	{
		if (forced) {
			g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
			{
				{
					GeeCollection* _tmp0_;
					GeeIterator* _tmp1_;
					GeeIterator* _celems_it;
					_tmp0_ = NULL;
					_tmp1_ = NULL;
					_celems_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_abstract_map_get_values ((GeeAbstractMap*) self->priv->elements_map))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
					while (TRUE) {
						DVBPlayerThreadChannelElements* celems;
						if (!gee_iterator_next (_celems_it)) {
							break;
						}
						celems = (DVBPlayerThreadChannelElements*) gee_iterator_get (_celems_it);
						if (celems->notify_func != NULL) {
							DVBChannel* channel;
							channel = dvb_channel_list_get_channel (dvb_device_get_Channels (self->priv->_device), celems->sid);
							celems->notify_func (channel);
							(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
						}
						(celems == NULL) ? NULL : (celems = (dvb_player_thread_channel_elements_unref (celems), NULL));
					}
					(_celems_it == NULL) ? NULL : (_celems_it = (g_object_unref (_celems_it), NULL));
				}
			}
			g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
		}
		g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
		{
			GstElement* _tmp2_;
			if (self->priv->pipeline != NULL) {
				GstBus* bus;
				g_debug ("ChannelFactory.vala:392: Stopping pipeline");
				bus = gst_element_get_bus (self->priv->pipeline);
				gst_bus_remove_signal_watch (bus);
				gst_element_set_state (self->priv->pipeline, GST_STATE_NULL);
				(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
			}
			_tmp2_ = NULL;
			self->priv->pipeline = (_tmp2_ = NULL, (self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL)), _tmp2_);
		}
		g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
		g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
		{
			gee_abstract_map_clear ((GeeAbstractMap*) self->priv->elements_map);
		}
		g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
		gee_abstract_collection_clear ((GeeAbstractCollection*) self->priv->_active_channels);
		self->priv->destroyed = TRUE;
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_destroyed);
}


void dvb_player_thread_destroy (DVBPlayerThread* self, gboolean forced) {
	DVB_PLAYER_THREAD_GET_CLASS (self)->destroy (self, forced);
}


static gboolean dvb_player_thread_add_element (DVBPlayerThread* self, GstElement* elem) {
	gboolean result;
	char* elem_name;
	GstElement* _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (elem != NULL, FALSE);
	elem_name = gst_object_get_name ((GstObject*) elem);
	_tmp0_ = NULL;
	if (!gst_bin_add (GST_BIN (self->priv->pipeline), (_tmp0_ = elem, (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_)))) {
		g_critical ("ChannelFactory.vala:410: Could not add element %s", elem_name);
		result = FALSE;
		elem_name = (g_free (elem_name), NULL);
		(elem == NULL) ? NULL : (elem = (gst_object_unref (elem), NULL));
		return result;
	}
	g_debug ("ChannelFactory.vala:413: Element %s (%p) added to pipeline", elem_name, elem);
	result = TRUE;
	elem_name = (g_free (elem_name), NULL);
	(elem == NULL) ? NULL : (elem = (gst_object_unref (elem), NULL));
	return result;
}


/**
         * Link program_%d pad with tee
         */
static void dvb_player_thread_on_dvbbasebin_pad_added (DVBPlayerThread* self, GstElement* elem, GstPad* pad) {
	char* pad_name;
	guint sid;
	g_return_if_fail (self != NULL);
	g_return_if_fail (elem != NULL);
	g_return_if_fail (pad != NULL);
	pad_name = gst_object_get_name ((GstObject*) pad);
	g_debug ("ChannelFactory.vala:422: Pad %s added", pad_name);
	if (!g_str_has_prefix (pad_name, "program_")) {
		pad_name = (g_free (pad_name), NULL);
		return;
	}
	sid = 0U;
	sscanf (pad_name, "program_%u", &sid);
	g_debug ("ChannelFactory.vala:430: SID is '%u'", sid);
	g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->elements_map, GUINT_TO_POINTER (sid))) {
			GstElement* _tmp1_;
			DVBPlayerThreadChannelElements* _tmp0_;
			GstElement* _tmp2_;
			GstElement* sink;
			char* _tmp4_;
			char* _tmp3_;
			GstPad* sinkpad;
			GstPadLinkReturn rc;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			_tmp2_ = NULL;
			sink = (_tmp2_ = (_tmp1_ = (_tmp0_ = (DVBPlayerThreadChannelElements*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->elements_map, GUINT_TO_POINTER (sid)))->tee, (_tmp1_ == NULL) ? NULL : gst_object_ref (_tmp1_)), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (dvb_player_thread_channel_elements_unref (_tmp0_), NULL)), _tmp2_);
			if (sink == NULL) {
				g_critical ("ChannelFactory.vala:436: Could not find sink for SID %u", sid);
				(sink == NULL) ? NULL : (sink = (gst_object_unref (sink), NULL));
				pad_name = (g_free (pad_name), NULL);
				return;
			}
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			g_debug ("ChannelFactory.vala:440: Linking elements %s and %s", _tmp3_ = gst_object_get_name ((GstObject*) elem), _tmp4_ = gst_object_get_name ((GstObject*) sink));
			_tmp4_ = (g_free (_tmp4_), NULL);
			_tmp3_ = (g_free (_tmp3_), NULL);
			sinkpad = gst_element_get_static_pad (sink, "sink");
			rc = gst_pad_link (pad, sinkpad);
			if (rc != GST_PAD_LINK_OK) {
				char* _tmp6_;
				char* _tmp5_;
				_tmp6_ = NULL;
				_tmp5_ = NULL;
				g_critical ("ChannelFactory.vala:445: Could not link pads %s and %s", _tmp5_ = gst_object_get_name ((GstObject*) pad), _tmp6_ = gst_object_get_name ((GstObject*) sinkpad));
				_tmp6_ = (g_free (_tmp6_), NULL);
				_tmp5_ = (g_free (_tmp5_), NULL);
			} else {
				char* _tmp8_;
				char* _tmp7_;
				_tmp8_ = NULL;
				_tmp7_ = NULL;
				g_debug ("ChannelFactory.vala:448: Src pad %s linked with sink pad %s", _tmp7_ = gst_object_get_name ((GstObject*) pad), _tmp8_ = gst_object_get_name ((GstObject*) sinkpad));
				_tmp8_ = (g_free (_tmp8_), NULL);
				_tmp7_ = (g_free (_tmp7_), NULL);
			}
			(sink == NULL) ? NULL : (sink = (gst_object_unref (sink), NULL));
			(sinkpad == NULL) ? NULL : (sinkpad = (gst_object_unref (sinkpad), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
	pad_name = (g_free (pad_name), NULL);
}


/**
         * Forward EIT structure
         */
static void dvb_player_thread_bus_watch_func (DVBPlayerThread* self, GstBus* bus, GstMessage* message) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (bus != NULL);
	g_return_if_fail (message != NULL);
	switch (message->type) {
		case GST_MESSAGE_ELEMENT:
		{
			const char* _tmp0_;
			char* structure_name;
			_tmp0_ = NULL;
			structure_name = (_tmp0_ = gst_structure_get_name (message->structure), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
			if (_vala_strcmp0 (structure_name, "eit") == 0) {
				if (self->priv->epgscanner != NULL) {
					dvb_epg_scanner_on_eit_structure (self->priv->epgscanner, message->structure);
				}
				g_signal_emit_by_name (self, "eit-structure", message->structure);
			}
			structure_name = (g_free (structure_name), NULL);
			break;
		}
		default:
		{
			break;
		}
	}
}


GeeHashSet* dvb_player_thread_get_active_channels (DVBPlayerThread* self) {
	GeeHashSet* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_active_channels;
	return result;
}


static void dvb_player_thread_set_active_channels (DVBPlayerThread* self, GeeHashSet* value) {
	GeeHashSet* _tmp1_;
	GeeHashSet* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_active_channels = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_active_channels == NULL) ? NULL : (self->priv->_active_channels = (g_object_unref (self->priv->_active_channels), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "active-channels");
}


DVBDevice* dvb_player_thread_get_device (DVBPlayerThread* self) {
	DVBDevice* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_device;
	return result;
}


static void dvb_player_thread_set_device (DVBPlayerThread* self, DVBDevice* value) {
	DVBDevice* _tmp1_;
	DVBDevice* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_device = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_device == NULL) ? NULL : (self->priv->_device = (g_object_unref (self->priv->_device), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "device");
}


gboolean dvb_player_thread_get_forced (DVBPlayerThread* self) {
	gboolean result;
	gboolean val;
	g_return_val_if_fail (self != NULL, FALSE);
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_elements_map);
	{
		{
			GeeCollection* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _celem_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_celem_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_abstract_map_get_values ((GeeAbstractMap*) self->priv->elements_map))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				DVBPlayerThreadChannelElements* celem;
				if (!gee_iterator_next (_celem_it)) {
					break;
				}
				celem = (DVBPlayerThreadChannelElements*) gee_iterator_get (_celem_it);
				if (celem->forced) {
					val = TRUE;
					(celem == NULL) ? NULL : (celem = (dvb_player_thread_channel_elements_unref (celem), NULL));
					break;
				}
				(celem == NULL) ? NULL : (celem = (dvb_player_thread_channel_elements_unref (celem), NULL));
			}
			(_celem_it == NULL) ? NULL : (_celem_it = (g_object_unref (_celem_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_elements_map);
	result = val;
	return result;
}


static GObject * dvb_player_thread_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBPlayerThreadClass * klass;
	GObjectClass * parent_class;
	DVBPlayerThread * self;
	klass = DVB_PLAYER_THREAD_CLASS (g_type_class_peek (DVB_TYPE_PLAYER_THREAD));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_PLAYER_THREAD (obj);
	{
		GeeHashMap* _tmp0_;
		GeeHashSet* _tmp1_;
		_tmp0_ = NULL;
		self->priv->elements_map = (_tmp0_ = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS, (GBoxedCopyFunc) dvb_player_thread_channel_elements_ref, dvb_player_thread_channel_elements_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->elements_map == NULL) ? NULL : (self->priv->elements_map = (g_object_unref (self->priv->elements_map), NULL)), _tmp0_);
		_tmp1_ = NULL;
		dvb_player_thread_set_active_channels (self, _tmp1_ = gee_hash_set_new (DVB_TYPE_CHANNEL, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (g_object_unref (_tmp1_), NULL));
		self->priv->destroyed = FALSE;
	}
	return obj;
}


static DVBPlayerThreadChannelElements* dvb_player_thread_channel_elements_construct (GType object_type) {
	DVBPlayerThreadChannelElements* self;
	self = (DVBPlayerThreadChannelElements*) g_type_create_instance (object_type);
	return self;
}


static DVBPlayerThreadChannelElements* dvb_player_thread_channel_elements_new (void) {
	return dvb_player_thread_channel_elements_construct (DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS);
}


static void dvb_player_thread_value_channel_elements_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void dvb_player_thread_value_channel_elements_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		dvb_player_thread_channel_elements_unref (value->data[0].v_pointer);
	}
}


static void dvb_player_thread_value_channel_elements_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = dvb_player_thread_channel_elements_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer dvb_player_thread_value_channel_elements_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* dvb_player_thread_value_channel_elements_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		DVBPlayerThreadChannelElements* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = dvb_player_thread_channel_elements_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* dvb_player_thread_value_channel_elements_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	DVBPlayerThreadChannelElements** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = dvb_player_thread_channel_elements_ref (value->data[0].v_pointer);
	}
	return NULL;
}


static GParamSpec* dvb_player_thread_param_spec_channel_elements (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	DVBPlayerThreadParamSpecChannelElements* spec;
	g_return_val_if_fail (g_type_is_a (object_type, DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


static gpointer dvb_player_thread_value_get_channel_elements (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS), NULL);
	return value->data[0].v_pointer;
}


static void dvb_player_thread_value_set_channel_elements (GValue* value, gpointer v_object) {
	DVBPlayerThreadChannelElements* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, DVB_PLAYER_THREAD_TYPE_CHANNEL_ELEMENTS));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		dvb_player_thread_channel_elements_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		dvb_player_thread_channel_elements_unref (old);
	}
}


static void dvb_player_thread_channel_elements_class_init (DVBPlayerThreadChannelElementsClass * klass) {
	dvb_player_thread_channel_elements_parent_class = g_type_class_peek_parent (klass);
	DVB_PLAYER_THREAD_CHANNEL_ELEMENTS_CLASS (klass)->finalize = dvb_player_thread_channel_elements_finalize;
}


static void dvb_player_thread_channel_elements_instance_init (DVBPlayerThreadChannelElements * self) {
	self->ref_count = 1;
}


static void dvb_player_thread_channel_elements_finalize (DVBPlayerThreadChannelElements* obj) {
	DVBPlayerThreadChannelElements * self;
	self = DVB_PLAYER_THREAD_CHANNEL_ELEMENTS (obj);
	(self->sinks == NULL) ? NULL : (self->sinks = (g_object_unref (self->sinks), NULL));
	(self->tee == NULL) ? NULL : (self->tee = (gst_object_unref (self->tee), NULL));
}


static GType dvb_player_thread_channel_elements_get_type (void) {
	static GType dvb_player_thread_channel_elements_type_id = 0;
	if (dvb_player_thread_channel_elements_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { dvb_player_thread_value_channel_elements_init, dvb_player_thread_value_channel_elements_free_value, dvb_player_thread_value_channel_elements_copy_value, dvb_player_thread_value_channel_elements_peek_pointer, "p", dvb_player_thread_value_channel_elements_collect_value, "p", dvb_player_thread_value_channel_elements_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (DVBPlayerThreadChannelElementsClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_player_thread_channel_elements_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBPlayerThreadChannelElements), 0, (GInstanceInitFunc) dvb_player_thread_channel_elements_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		dvb_player_thread_channel_elements_type_id = g_type_register_fundamental (g_type_fundamental_next (), "DVBPlayerThreadChannelElements", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return dvb_player_thread_channel_elements_type_id;
}


static gpointer dvb_player_thread_channel_elements_ref (gpointer instance) {
	DVBPlayerThreadChannelElements* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


static void dvb_player_thread_channel_elements_unref (gpointer instance) {
	DVBPlayerThreadChannelElements* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		DVB_PLAYER_THREAD_CHANNEL_ELEMENTS_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static void dvb_player_thread_class_init (DVBPlayerThreadClass * klass) {
	dvb_player_thread_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBPlayerThreadPrivate));
	DVB_PLAYER_THREAD_CLASS (klass)->destroy = dvb_player_thread_real_destroy;
	G_OBJECT_CLASS (klass)->get_property = dvb_player_thread_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_player_thread_set_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_player_thread_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_player_thread_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_PLAYER_THREAD_ACTIVE_CHANNELS, g_param_spec_object ("active-channels", "active-channels", "active-channels", GEE_TYPE_HASH_SET, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_PLAYER_THREAD_DEVICE, g_param_spec_object ("device", "device", "device", DVB_TYPE_DEVICE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_PLAYER_THREAD_FORCED, g_param_spec_boolean ("forced", "forced", "forced", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_signal_new ("eit_structure", DVB_TYPE_PLAYER_THREAD, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_marshal_VOID__POINTER, G_TYPE_NONE, 1, G_TYPE_POINTER);
}


static void dvb_player_thread_instance_init (DVBPlayerThread * self) {
	self->priv = DVB_PLAYER_THREAD_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_pipeline);
	g_static_rec_mutex_init (&self->priv->__lock_elements_map);
	g_static_rec_mutex_init (&self->priv->__lock_destroyed);
}


static void dvb_player_thread_finalize (GObject* obj) {
	DVBPlayerThread * self;
	self = DVB_PLAYER_THREAD (obj);
	(self->priv->_active_channels == NULL) ? NULL : (self->priv->_active_channels = (g_object_unref (self->priv->_active_channels), NULL));
	(self->priv->_device == NULL) ? NULL : (self->priv->_device = (g_object_unref (self->priv->_device), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_pipeline);
	(self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_elements_map);
	(self->priv->elements_map == NULL) ? NULL : (self->priv->elements_map = (g_object_unref (self->priv->elements_map), NULL));
	(self->priv->epgscanner == NULL) ? NULL : (self->priv->epgscanner = (g_object_unref (self->priv->epgscanner), NULL));
	(self->priv->dvbbasebin == NULL) ? NULL : (self->priv->dvbbasebin = (gst_object_unref (self->priv->dvbbasebin), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_destroyed);
	G_OBJECT_CLASS (dvb_player_thread_parent_class)->finalize (obj);
}


GType dvb_player_thread_get_type (void) {
	static GType dvb_player_thread_type_id = 0;
	if (dvb_player_thread_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBPlayerThreadClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_player_thread_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBPlayerThread), 0, (GInstanceInitFunc) dvb_player_thread_instance_init, NULL };
		dvb_player_thread_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBPlayerThread", &g_define_type_info, 0);
	}
	return dvb_player_thread_type_id;
}


static void dvb_player_thread_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBPlayerThread * self;
	gpointer boxed;
	self = DVB_PLAYER_THREAD (object);
	switch (property_id) {
		case DVB_PLAYER_THREAD_ACTIVE_CHANNELS:
		g_value_set_object (value, dvb_player_thread_get_active_channels (self));
		break;
		case DVB_PLAYER_THREAD_DEVICE:
		g_value_set_object (value, dvb_player_thread_get_device (self));
		break;
		case DVB_PLAYER_THREAD_FORCED:
		g_value_set_boolean (value, dvb_player_thread_get_forced (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_player_thread_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBPlayerThread * self;
	self = DVB_PLAYER_THREAD (object);
	switch (property_id) {
		case DVB_PLAYER_THREAD_ACTIVE_CHANNELS:
		dvb_player_thread_set_active_channels (self, g_value_get_object (value));
		break;
		case DVB_PLAYER_THREAD_DEVICE:
		dvb_player_thread_set_device (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


DVBChannelFactory* dvb_channel_factory_construct (GType object_type, DVBDeviceGroup* devgroup) {
	GParameter * __params;
	GParameter * __params_it;
	DVBChannelFactory * self;
	GeeHashSet* _tmp0_;
	g_return_val_if_fail (devgroup != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "device-group";
	g_value_init (&__params_it->value, DVB_TYPE_DEVICE_GROUP);
	g_value_set_object (&__params_it->value, devgroup);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	_tmp0_ = NULL;
	self->priv->active_players = (_tmp0_ = gee_hash_set_new (DVB_TYPE_PLAYER_THREAD, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal), (self->priv->active_players == NULL) ? NULL : (self->priv->active_players = (g_object_unref (self->priv->active_players), NULL)), _tmp0_);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBChannelFactory* dvb_channel_factory_new (DVBDeviceGroup* devgroup) {
	return dvb_channel_factory_construct (DVB_TYPE_CHANNEL_FACTORY, devgroup);
}


/**
         * Stop all currently active players
         */
void dvb_channel_factory_destroy (DVBChannelFactory* self) {
	g_return_if_fail (self != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_active_players);
	{
		{
			GeeIterator* _active_player_it;
			_active_player_it = gee_abstract_collection_iterator ((GeeAbstractCollection*) self->priv->active_players);
			while (TRUE) {
				DVBPlayerThread* active_player;
				if (!gee_iterator_next (_active_player_it)) {
					break;
				}
				active_player = (DVBPlayerThread*) gee_iterator_get (_active_player_it);
				dvb_player_thread_destroy (active_player, FALSE);
				(active_player == NULL) ? NULL : (active_player = (g_object_unref (active_player), NULL));
			}
			(_active_player_it == NULL) ? NULL : (_active_player_it = (g_object_unref (_active_player_it), NULL));
		}
		gee_abstract_collection_clear ((GeeAbstractCollection*) self->priv->active_players);
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_active_players);
}


static char* bool_to_string (gboolean self) {
	char* result;
	if (self) {
		result = g_strdup ("true");
		return result;
	} else {
		result = g_strdup ("false");
		return result;
	}
}


/**
         * @channel: channel to watch
         * @sink_element: The element the src pad should be linked with
         * @force: Whether to stop a player when there's currently no free device
         * @notify_func: The given function is called when watching the channel
         *   is aborted because a recording on a different transport streams is
         *   about to start
         * @returns: The #PlayerThread used to watch @channel
         *
         * Watch @channel and use @sink_element as sink element
         */
DVBPlayerThread* dvb_channel_factory_watch_channel (DVBChannelFactory* self, DVBChannel* channel, GstElement* sink_element, gboolean force, DVBForcedStopNotify notify_func) {
	DVBPlayerThread* result;
	gboolean create_new;
	DVBPlayerThread* player;
	DVBDevice* free_device;
	char* _tmp2_;
	GstElement* _tmp9_;
	GstElement* _tmp8_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (channel != NULL, NULL);
	g_return_val_if_fail (sink_element != NULL, NULL);
	g_debug ("ChannelFactory.vala:516: Watching channel %s (%u)", dvb_channel_get_Name (channel), dvb_channel_get_Sid (channel));
	create_new = TRUE;
	player = NULL;
	free_device = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_active_players);
	{
		{
			GeeIterator* _active_player_it;
			_active_player_it = gee_abstract_collection_iterator ((GeeAbstractCollection*) self->priv->active_players);
			while (TRUE) {
				DVBPlayerThread* active_player;
				if (!gee_iterator_next (_active_player_it)) {
					break;
				}
				active_player = (DVBPlayerThread*) gee_iterator_get (_active_player_it);
				{
					GeeIterator* _other_channel_it;
					_other_channel_it = gee_abstract_collection_iterator ((GeeAbstractCollection*) dvb_player_thread_get_active_channels (active_player));
					while (TRUE) {
						DVBChannel* other_channel;
						if (!gee_iterator_next (_other_channel_it)) {
							break;
						}
						other_channel = (DVBChannel*) gee_iterator_get (_other_channel_it);
						if (dvb_channel_on_same_transport_stream (channel, other_channel)) {
							DVBPlayerThread* _tmp1_;
							DVBPlayerThread* _tmp0_;
							create_new = FALSE;
							_tmp1_ = NULL;
							_tmp0_ = NULL;
							player = (_tmp1_ = (_tmp0_ = active_player, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (player == NULL) ? NULL : (player = (g_object_unref (player), NULL)), _tmp1_);
							(other_channel == NULL) ? NULL : (other_channel = (g_object_unref (other_channel), NULL));
							break;
						}
						(other_channel == NULL) ? NULL : (other_channel = (g_object_unref (other_channel), NULL));
					}
					(_other_channel_it == NULL) ? NULL : (_other_channel_it = (g_object_unref (_other_channel_it), NULL));
				}
				(active_player == NULL) ? NULL : (active_player = (g_object_unref (active_player), NULL));
			}
			(_active_player_it == NULL) ? NULL : (_active_player_it = (g_object_unref (_active_player_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_active_players);
	_tmp2_ = NULL;
	g_debug ("ChannelFactory.vala:533: Creating new PlayerThread: %s", _tmp2_ = bool_to_string (create_new));
	_tmp2_ = (g_free (_tmp2_), NULL);
	if (create_new) {
		DVBEPGScanner* _tmp3_;
		DVBEPGScanner* epgscanner;
		DVBDevice* _tmp4_;
		gboolean _tmp5_;
		DVBPlayerThread* _tmp7_;
		/* Stop epgscanner before starting recording*/
		_tmp3_ = NULL;
		epgscanner = (_tmp3_ = dvb_device_group_get_epgscanner (self->priv->_device_group), (_tmp3_ == NULL) ? NULL : g_object_ref (_tmp3_));
		if (epgscanner != NULL) {
			dvb_epg_scanner_stop (epgscanner);
		}
		_tmp4_ = NULL;
		free_device = (_tmp4_ = dvb_device_group_get_next_free_device (self->priv->_device_group), (free_device == NULL) ? NULL : (free_device = (g_object_unref (free_device), NULL)), _tmp4_);
		_tmp5_ = FALSE;
		if (free_device == NULL) {
			_tmp5_ = force;
		} else {
			_tmp5_ = FALSE;
		}
		if (_tmp5_) {
			DVBDevice* _tmp6_;
			g_static_rec_mutex_lock (&self->priv->__lock_active_players);
			{
				{
					GeeIterator* _active_player_it;
					_active_player_it = gee_abstract_collection_iterator ((GeeAbstractCollection*) self->priv->active_players);
					while (TRUE) {
						DVBPlayerThread* active_player;
						if (!gee_iterator_next (_active_player_it)) {
							break;
						}
						active_player = (DVBPlayerThread*) gee_iterator_get (_active_player_it);
						if (!dvb_player_thread_get_forced (active_player)) {
							dvb_player_thread_destroy (active_player, TRUE);
							(active_player == NULL) ? NULL : (active_player = (g_object_unref (active_player), NULL));
							break;
						} else {
							g_critical ("ChannelFactory.vala:548: No active players that are not forced");
						}
						(active_player == NULL) ? NULL : (active_player = (g_object_unref (active_player), NULL));
					}
					(_active_player_it == NULL) ? NULL : (_active_player_it = (g_object_unref (_active_player_it), NULL));
				}
			}
			g_static_rec_mutex_unlock (&self->priv->__lock_active_players);
			_tmp6_ = NULL;
			free_device = (_tmp6_ = dvb_device_group_get_next_free_device (self->priv->_device_group), (free_device == NULL) ? NULL : (free_device = (g_object_unref (free_device), NULL)), _tmp6_);
		}
		if (free_device == NULL) {
			g_critical ("ChannelFactory.vala:555: All devices are busy");
			result = NULL;
			(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
			(player == NULL) ? NULL : (player = (g_object_unref (player), NULL));
			(free_device == NULL) ? NULL : (free_device = (g_object_unref (free_device), NULL));
			(sink_element == NULL) ? NULL : (sink_element = (gst_object_unref (sink_element), NULL));
			return result;
		}
		_tmp7_ = NULL;
		player = (_tmp7_ = dvb_channel_factory_create_player (self, free_device), (player == NULL) ? NULL : (player = (g_object_unref (player), NULL)), _tmp7_);
		(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
	}
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	_tmp9_ = dvb_player_thread_get_element (player, channel, (_tmp8_ = sink_element, (_tmp8_ == NULL) ? NULL : gst_object_ref (_tmp8_)), force, notify_func);
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (gst_object_unref (_tmp9_), NULL));
	g_static_rec_mutex_lock (&self->priv->__lock_active_players);
	{
		gee_abstract_collection_add ((GeeAbstractCollection*) self->priv->active_players, player);
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_active_players);
	result = player;
	(free_device == NULL) ? NULL : (free_device = (g_object_unref (free_device), NULL));
	(sink_element == NULL) ? NULL : (sink_element = (gst_object_unref (sink_element), NULL));
	return result;
}


/**
         * @returns: TRUE on success
         *
         * Stop watching @channel
         */
gboolean dvb_channel_factory_stop_channel (DVBChannelFactory* self, DVBChannel* channel, GstElement* sink) {
	gboolean result;
	gboolean success;
	DVBPlayerThread* player;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (channel != NULL, FALSE);
	g_return_val_if_fail (sink != NULL, FALSE);
	g_debug ("ChannelFactory.vala:576: Stopping channel %s (%u)", dvb_channel_get_Name (channel), dvb_channel_get_Sid (channel));
	success = FALSE;
	player = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_active_players);
	{
		gboolean _tmp2_;
		{
			GeeIterator* _active_player_it;
			_active_player_it = gee_abstract_collection_iterator ((GeeAbstractCollection*) self->priv->active_players);
			while (TRUE) {
				DVBPlayerThread* active_player;
				if (!gee_iterator_next (_active_player_it)) {
					break;
				}
				active_player = (DVBPlayerThread*) gee_iterator_get (_active_player_it);
				if (gee_abstract_collection_contains ((GeeAbstractCollection*) dvb_player_thread_get_active_channels (active_player), channel)) {
					DVBPlayerThread* _tmp1_;
					DVBPlayerThread* _tmp0_;
					success = dvb_player_thread_remove_channel (active_player, channel, sink);
					_tmp1_ = NULL;
					_tmp0_ = NULL;
					player = (_tmp1_ = (_tmp0_ = active_player, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (player == NULL) ? NULL : (player = (g_object_unref (player), NULL)), _tmp1_);
					(active_player == NULL) ? NULL : (active_player = (g_object_unref (active_player), NULL));
					break;
				}
				(active_player == NULL) ? NULL : (active_player = (g_object_unref (active_player), NULL));
			}
			(_active_player_it == NULL) ? NULL : (_active_player_it = (g_object_unref (_active_player_it), NULL));
		}
		_tmp2_ = FALSE;
		if (success) {
			_tmp2_ = gee_collection_get_size ((GeeCollection*) dvb_player_thread_get_active_channels (player)) == 0;
		} else {
			_tmp2_ = FALSE;
		}
		if (_tmp2_) {
			gee_abstract_collection_remove ((GeeAbstractCollection*) self->priv->active_players, player);
		}
		if (gee_collection_get_size ((GeeCollection*) self->priv->active_players) == 0) {
			DVBEPGScanner* _tmp3_;
			DVBEPGScanner* epgscanner;
			/* Start EPG scanner again*/
			_tmp3_ = NULL;
			epgscanner = (_tmp3_ = dvb_device_group_get_epgscanner (self->priv->_device_group), (_tmp3_ == NULL) ? NULL : g_object_ref (_tmp3_));
			if (epgscanner != NULL) {
				dvb_epg_scanner_start (epgscanner);
			}
			(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_active_players);
	result = success;
	(player == NULL) ? NULL : (player = (g_object_unref (player), NULL));
	return result;
}


/**
         * @returns: a new #PlayerThread instance for @device
         */
static DVBPlayerThread* dvb_channel_factory_real_create_player (DVBChannelFactory* self, DVBDevice* device) {
	DVBPlayerThread* result;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (device != NULL, NULL);
	result = dvb_player_thread_new (device, dvb_device_group_get_epgscanner (self->priv->_device_group));
	return result;
}


DVBPlayerThread* dvb_channel_factory_create_player (DVBChannelFactory* self, DVBDevice* device) {
	return DVB_CHANNEL_FACTORY_GET_CLASS (self)->create_player (self, device);
}


DVBDeviceGroup* dvb_channel_factory_get_device_group (DVBChannelFactory* self) {
	DVBDeviceGroup* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_device_group;
	return result;
}


static void dvb_channel_factory_set_device_group (DVBChannelFactory* self, DVBDeviceGroup* value) {
	g_return_if_fail (self != NULL);
	self->priv->_device_group = value;
	g_object_notify ((GObject *) self, "device-group");
}


static void dvb_channel_factory_class_init (DVBChannelFactoryClass * klass) {
	dvb_channel_factory_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBChannelFactoryPrivate));
	DVB_CHANNEL_FACTORY_CLASS (klass)->create_player = dvb_channel_factory_real_create_player;
	G_OBJECT_CLASS (klass)->get_property = dvb_channel_factory_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_channel_factory_set_property;
	G_OBJECT_CLASS (klass)->finalize = dvb_channel_factory_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_CHANNEL_FACTORY_DEVICE_GROUP, g_param_spec_object ("device-group", "device-group", "device-group", DVB_TYPE_DEVICE_GROUP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void dvb_channel_factory_instance_init (DVBChannelFactory * self) {
	self->priv = DVB_CHANNEL_FACTORY_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_active_players);
}


static void dvb_channel_factory_finalize (GObject* obj) {
	DVBChannelFactory * self;
	self = DVB_CHANNEL_FACTORY (obj);
	g_static_rec_mutex_free (&self->priv->__lock_active_players);
	(self->priv->active_players == NULL) ? NULL : (self->priv->active_players = (g_object_unref (self->priv->active_players), NULL));
	G_OBJECT_CLASS (dvb_channel_factory_parent_class)->finalize (obj);
}


GType dvb_channel_factory_get_type (void) {
	static GType dvb_channel_factory_type_id = 0;
	if (dvb_channel_factory_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBChannelFactoryClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_channel_factory_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBChannelFactory), 0, (GInstanceInitFunc) dvb_channel_factory_instance_init, NULL };
		dvb_channel_factory_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBChannelFactory", &g_define_type_info, 0);
	}
	return dvb_channel_factory_type_id;
}


static void dvb_channel_factory_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBChannelFactory * self;
	gpointer boxed;
	self = DVB_CHANNEL_FACTORY (object);
	switch (property_id) {
		case DVB_CHANNEL_FACTORY_DEVICE_GROUP:
		g_value_set_object (value, dvb_channel_factory_get_device_group (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_channel_factory_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBChannelFactory * self;
	self = DVB_CHANNEL_FACTORY (object);
	switch (property_id) {
		case DVB_CHANNEL_FACTORY_DEVICE_GROUP:
		dvb_channel_factory_set_device_group (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




