/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gst/gst.h>
#include <time.h>


#define DVB_TYPE_TIMER (dvb_timer_get_type ())
#define DVB_TIMER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TIMER, DVBTimer))
#define DVB_TIMER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_TIMER, DVBTimerClass))
#define DVB_IS_TIMER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TIMER))
#define DVB_IS_TIMER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_TIMER))
#define DVB_TIMER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_TIMER, DVBTimerClass))

typedef struct _DVBTimer DVBTimer;
typedef struct _DVBTimerClass DVBTimerClass;
typedef struct _DVBTimerPrivate DVBTimerPrivate;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_OVERLAP_TYPE (dvb_overlap_type_get_type ())

/**
     * This class represents an event that should be recorded
     */
struct _DVBTimer {
	GObject parent_instance;
	DVBTimerPrivate * priv;
};

struct _DVBTimerClass {
	GObjectClass parent_class;
};

struct _DVBTimerPrivate {
	guint32 _Id;
	DVBChannel* _Channel;
	char* _Name;
	guint _Year;
	guint _Month;
	guint _Day;
	guint _Hour;
	guint _Minute;
	guint _Duration;
	GstElement* _sink;
	guint _EventID;
	struct tm starttime;
};

typedef enum  {
	DVB_OVERLAP_TYPE_UNKNOWN,
	DVB_OVERLAP_TYPE_NONE,
	DVB_OVERLAP_TYPE_PARTIAL,
	DVB_OVERLAP_TYPE_COMPLETE
} DVBOverlapType;


static gpointer dvb_timer_parent_class = NULL;

GType dvb_timer_get_type (void);
GType dvb_channel_get_type (void);
#define DVB_TIMER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_TIMER, DVBTimerPrivate))
enum  {
	DVB_TIMER_DUMMY_PROPERTY,
	DVB_TIMER_ID,
	DVB_TIMER_CHANNEL,
	DVB_TIMER_NAME,
	DVB_TIMER_YEAR,
	DVB_TIMER_MONTH,
	DVB_TIMER_DAY,
	DVB_TIMER_HOUR,
	DVB_TIMER_MINUTE,
	DVB_TIMER_DURATION,
	DVB_TIMER_SINK,
	DVB_TIMER_EVENT_ID
};
static void dvb_timer_set_Id (DVBTimer* self, guint32 value);
static void dvb_timer_set_Channel (DVBTimer* self, DVBChannel* value);
void dvb_timer_set_Name (DVBTimer* self, const char* value);
void dvb_timer_set_Year (DVBTimer* self, guint value);
void dvb_timer_set_Month (DVBTimer* self, guint value);
void dvb_timer_set_Day (DVBTimer* self, guint value);
void dvb_timer_set_Hour (DVBTimer* self, guint value);
void dvb_timer_set_Minute (DVBTimer* self, guint value);
void dvb_timer_set_EventID (DVBTimer* self, guint value);
void dvb_timer_set_Duration (DVBTimer* self, guint value);
struct tm dvb_utils_create_time (gint year, gint month, gint day, gint hour, gint minute, gint second);
DVBTimer* dvb_timer_new (guint32 id, DVBChannel* channel, gint year, gint month, gint day, gint hour, gint minute, guint duration, const char* name);
DVBTimer* dvb_timer_construct (GType object_type, guint32 id, DVBChannel* channel, gint year, gint month, gint day, gint hour, gint minute, guint duration, const char* name);
static time_t dvb_timer_get_start_time_timestamp (DVBTimer* self);
DVBChannel* dvb_timer_get_Channel (DVBTimer* self);
gboolean dvb_channel_on_same_transport_stream (DVBChannel* self, DVBChannel* channel);
gboolean dvb_channel_equals (DVBChannel* self, DVBChannel* channel);
static time_t dvb_timer_get_end_time_timestamp (DVBTimer* self);
gboolean dvb_timer_conflicts_with (DVBTimer* self, DVBTimer* t2);
GType dvb_overlap_type_get_type (void);
static DVBOverlapType dvb_timer_get_overlap (const time_t* this_start, const time_t* this_end, const time_t* other_start, const time_t* other_end);
DVBOverlapType dvb_timer_get_overlap_local (DVBTimer* self, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration);
DVBOverlapType dvb_timer_get_overlap_utc (DVBTimer* self, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration);
void dvb_timer_add_to_start_time (DVBTimer* self, gint minutes);
guint dvb_timer_get_Year (DVBTimer* self);
guint dvb_timer_get_Month (DVBTimer* self);
guint dvb_timer_get_Day (DVBTimer* self);
guint dvb_timer_get_Hour (DVBTimer* self);
guint dvb_timer_get_Minute (DVBTimer* self);
guint* dvb_timer_get_start_time (DVBTimer* self, int* result_length1);
struct tm dvb_timer_get_start_time_time (DVBTimer* self);
guint* dvb_timer_get_end_time (DVBTimer* self, int* result_length1);
gboolean dvb_timer_has_expired (DVBTimer* self);
gboolean dvb_timer_is_start_due (DVBTimer* self);
gboolean dvb_timer_is_end_due (DVBTimer* self);
guint dvb_channel_get_Sid (DVBChannel* self);
guint dvb_timer_get_Duration (DVBTimer* self);
char* dvb_timer_to_string (DVBTimer* self);
guint32 dvb_timer_get_Id (DVBTimer* self);
const char* dvb_timer_get_Name (DVBTimer* self);
GstElement* dvb_timer_get_sink (DVBTimer* self);
void dvb_timer_set_sink (DVBTimer* self, GstElement* value);
guint dvb_timer_get_EventID (DVBTimer* self);
static void dvb_timer_finalize (GObject* obj);
static void dvb_timer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_timer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);



DVBTimer* dvb_timer_construct (GType object_type, guint32 id, DVBChannel* channel, gint year, gint month, gint day, gint hour, gint minute, guint duration, const char* name) {
	GParameter * __params;
	GParameter * __params_it;
	DVBTimer * self;
	g_return_val_if_fail (channel != NULL, NULL);
	__params = g_new0 (GParameter, 2);
	__params_it = __params;
	__params_it->name = "Id";
	g_value_init (&__params_it->value, G_TYPE_UINT);
	g_value_set_uint (&__params_it->value, id);
	__params_it++;
	__params_it->name = "Channel";
	g_value_init (&__params_it->value, DVB_TYPE_CHANNEL);
	g_value_set_object (&__params_it->value, channel);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	dvb_timer_set_Name (self, name);
	dvb_timer_set_Year (self, (guint) year);
	dvb_timer_set_Month (self, (guint) month);
	dvb_timer_set_Day (self, (guint) day);
	dvb_timer_set_Hour (self, (guint) hour);
	dvb_timer_set_Minute (self, (guint) minute);
	dvb_timer_set_EventID (self, (guint) 0);
	dvb_timer_set_Duration (self, duration);
	self->priv->starttime = dvb_utils_create_time (year, month, day, hour, minute, 0);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBTimer* dvb_timer_new (guint32 id, DVBChannel* channel, gint year, gint month, gint day, gint hour, gint minute, guint duration, const char* name) {
	return dvb_timer_construct (DVB_TYPE_TIMER, id, channel, year, month, day, hour, minute, duration, name);
}


/**
         * Whether the timer conflicts with the other one
         */
gboolean dvb_timer_conflicts_with (DVBTimer* self, DVBTimer* t2) {
	gboolean result;
	time_t this_start;
	time_t other_start;
	DVBChannel* _tmp0_;
	DVBChannel* t1_c;
	DVBChannel* _tmp1_;
	DVBChannel* t2_c;
	gboolean _tmp2_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (t2 != NULL, FALSE);
	this_start = dvb_timer_get_start_time_timestamp (self);
	other_start = dvb_timer_get_start_time_timestamp (t2);
	_tmp0_ = NULL;
	t1_c = (_tmp0_ = self->priv->_Channel, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
	_tmp1_ = NULL;
	t2_c = (_tmp1_ = t2->priv->_Channel, (_tmp1_ == NULL) ? NULL : g_object_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (dvb_channel_on_same_transport_stream (t1_c, t2_c)) {
		_tmp2_ = !dvb_channel_equals (t1_c, t2_c);
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		/* Recordings on the same TS never conflict as long
		 as it isn't the same channel*/
		result = FALSE;
		(t1_c == NULL) ? NULL : (t1_c = (g_object_unref (t1_c), NULL));
		(t2_c == NULL) ? NULL : (t2_c = (g_object_unref (t2_c), NULL));
		return result;
	}
	if (this_start <= other_start) {
		time_t this_end;
		/* No conflict when this timer ends before other starts*/
		this_end = dvb_timer_get_end_time_timestamp (self);
		result = this_end > other_start;
		(t1_c == NULL) ? NULL : (t1_c = (g_object_unref (t1_c), NULL));
		(t2_c == NULL) ? NULL : (t2_c = (g_object_unref (t2_c), NULL));
		return result;
	} else {
		time_t other_end;
		/* No conflict when other timer ends before this starts*/
		other_end = dvb_timer_get_end_time_timestamp (t2);
		result = other_end > this_start;
		(t1_c == NULL) ? NULL : (t1_c = (g_object_unref (t1_c), NULL));
		(t2_c == NULL) ? NULL : (t2_c = (g_object_unref (t2_c), NULL));
		return result;
	}
	(t1_c == NULL) ? NULL : (t1_c = (g_object_unref (t1_c), NULL));
	(t2_c == NULL) ? NULL : (t2_c = (g_object_unref (t2_c), NULL));
}


/**
         * @duration: in minutes
         * @returns: The overlap between the timer and the given time range.
         * The timer is the reference, i.e. if the time range is completely
         * contained in the timer OverlapType.COMPLETE is returned.
         *
         * The given time range must be in local time.
         */
DVBOverlapType dvb_timer_get_overlap_local (DVBTimer* self, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration) {
	DVBOverlapType result;
	time_t this_start;
	time_t this_end;
	struct tm other_time;
	time_t other_start;
	time_t other_end;
	g_return_val_if_fail (self != NULL, 0);
	this_start = dvb_timer_get_start_time_timestamp (self);
	this_end = dvb_timer_get_end_time_timestamp (self);
	other_time = dvb_utils_create_time ((gint) start_year, (gint) start_month, (gint) start_day, (gint) start_hour, (gint) start_minute, 0);
	other_start = mktime (&other_time);
	other_time.tm_min = other_time.tm_min + ((gint) duration);
	other_end = mktime (&other_time);
	result = dvb_timer_get_overlap (&this_start, &this_end, &other_start, &other_end);
	return result;
}


/**
         * Same as get_overlap_local but the given time range is UTC time.
         */
DVBOverlapType dvb_timer_get_overlap_utc (DVBTimer* self, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration) {
	DVBOverlapType result;
	time_t this_start;
	time_t this_end;
	struct tm other_time;
	time_t other_start;
	time_t other_end;
	g_return_val_if_fail (self != NULL, 0);
	this_start = dvb_timer_get_start_time_timestamp (self);
	this_end = dvb_timer_get_end_time_timestamp (self);
	other_time = dvb_utils_create_time ((gint) start_year, (gint) start_month, (gint) start_day, (gint) start_hour, (gint) start_minute, 0);
	other_start = timegm (&other_time);
	other_time.tm_min = other_time.tm_min + ((gint) duration);
	other_end = timegm (&other_time);
	result = dvb_timer_get_overlap (&this_start, &this_end, &other_start, &other_end);
	return result;
}


static DVBOverlapType dvb_timer_get_overlap (const time_t* this_start, const time_t* this_end, const time_t* other_start, const time_t* other_end) {
	DVBOverlapType result;
	if ((*this_start) <= (*other_start)) {
		/* No conflict when this timer ends before other starts*/
		if ((*this_end) <= (*other_start)) {
			/* this starts before other and ends before other starts*/
			result = DVB_OVERLAP_TYPE_NONE;
			return result;
		} else {
			if ((*this_end) >= (*other_end)) {
				result = DVB_OVERLAP_TYPE_COMPLETE;
				return result;
			} else {
				result = DVB_OVERLAP_TYPE_PARTIAL;
				return result;
			}
		}
	} else {
		/* No conflict when other timer ends before this starts*/
		if ((*this_end) <= (*other_end)) {
			/* this starts after other and ends before other*/
			result = DVB_OVERLAP_TYPE_PARTIAL;
			return result;
		} else {
			if ((*this_start) < (*other_end)) {
				result = DVB_OVERLAP_TYPE_PARTIAL;
				return result;
			} else {
				result = DVB_OVERLAP_TYPE_NONE;
				return result;
			}
		}
	}
}


/**
         * Add the specified amount of minutes to the starting time
         */
void dvb_timer_add_to_start_time (DVBTimer* self, gint minutes) {
	g_return_if_fail (self != NULL);
	self->priv->starttime.tm_min = self->priv->starttime.tm_min + minutes;
	mktime (&self->priv->starttime);
	dvb_timer_set_Year (self, (guint) (self->priv->starttime.tm_year + 1900));
	dvb_timer_set_Month (self, (guint) (self->priv->starttime.tm_mon + 1));
	dvb_timer_set_Day (self, (guint) self->priv->starttime.tm_mday);
	dvb_timer_set_Hour (self, (guint) self->priv->starttime.tm_hour);
	dvb_timer_set_Minute (self, (guint) self->priv->starttime.tm_min);
}


guint* dvb_timer_get_start_time (DVBTimer* self, int* result_length1) {
	guint* result;
	guint* _tmp1_;
	gint start_size;
	gint start_length1;
	guint* _tmp0_;
	guint* start;
	guint* _tmp2_;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	start = (_tmp1_ = (_tmp0_ = g_new0 (guint, 5), _tmp0_[0] = self->priv->_Year, _tmp0_[1] = self->priv->_Month, _tmp0_[2] = self->priv->_Day, _tmp0_[3] = self->priv->_Hour, _tmp0_[4] = self->priv->_Minute, _tmp0_), start_length1 = 5, start_size = start_length1, _tmp1_);
	_tmp2_ = NULL;
	result = (_tmp2_ = start, *result_length1 = start_length1, _tmp2_);
	return result;
	start = (g_free (start), NULL);
}


struct tm dvb_timer_get_start_time_time (DVBTimer* self) {
	struct tm result;
	result = self->priv->starttime;
	return result;
}


guint* dvb_timer_get_end_time (DVBTimer* self, int* result_length1) {
	guint* result;
	time_t _tmp1_ = {0};
	struct tm _tmp0_ = {0};
	struct tm l;
	guint* _tmp2_;
	guint* _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	l = (localtime_r ((_tmp1_ = dvb_timer_get_end_time_timestamp (self), &_tmp1_), &_tmp0_), _tmp0_);
	_tmp2_ = NULL;
	_tmp3_ = NULL;
	result = (_tmp3_ = (_tmp2_ = g_new0 (guint, 5), _tmp2_[0] = (guint) (l.tm_year + 1900), _tmp2_[1] = (guint) (l.tm_mon + 1), _tmp2_[2] = (guint) l.tm_mday, _tmp2_[3] = (guint) l.tm_hour, _tmp2_[4] = (guint) l.tm_min, _tmp2_), *result_length1 = 5, _tmp3_);
	return result;
}


/**
         * Whether the start time of the timer is after the current local time
         * and the timer hasn't expired, yet.
         */
gboolean dvb_timer_is_start_due (DVBTimer* self) {
	gboolean result;
	time_t _tmp0_ = {0};
	time_t localtime;
	gboolean _tmp1_;
	g_return_val_if_fail (self != NULL, FALSE);
	localtime = (time (&_tmp0_), _tmp0_);
	_tmp1_ = FALSE;
	if ((localtime - mktime (&self->priv->starttime)) >= 0) {
		_tmp1_ = !dvb_timer_has_expired (self);
	} else {
		_tmp1_ = FALSE;
	}
	result = _tmp1_;
	return result;
}


/**
         * Whether the end time of the timer equals the current local time
         */
gboolean dvb_timer_is_end_due (DVBTimer* self) {
	gboolean result;
	time_t _tmp2_ = {0};
	time_t _tmp0_ = {0};
	struct tm _tmp1_ = {0};
	struct tm localtime;
	time_t _tmp4_ = {0};
	struct tm _tmp3_ = {0};
	struct tm endtime;
	gboolean _tmp5_;
	gboolean _tmp6_;
	gboolean _tmp7_;
	gboolean _tmp8_;
	g_return_val_if_fail (self != NULL, FALSE);
	localtime = (localtime_r ((_tmp2_ = (time (&_tmp0_), _tmp0_), &_tmp2_), &_tmp1_), _tmp1_);
	endtime = (localtime_r ((_tmp4_ = dvb_timer_get_end_time_timestamp (self), &_tmp4_), &_tmp3_), _tmp3_);
	_tmp5_ = FALSE;
	_tmp6_ = FALSE;
	_tmp7_ = FALSE;
	_tmp8_ = FALSE;
	if (endtime.tm_year == localtime.tm_year) {
		_tmp8_ = endtime.tm_mon == localtime.tm_mon;
	} else {
		_tmp8_ = FALSE;
	}
	if (_tmp8_) {
		_tmp7_ = endtime.tm_mday == localtime.tm_mday;
	} else {
		_tmp7_ = FALSE;
	}
	if (_tmp7_) {
		_tmp6_ = endtime.tm_hour == localtime.tm_hour;
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		_tmp5_ = endtime.tm_min == localtime.tm_min;
	} else {
		_tmp5_ = FALSE;
	}
	result = _tmp5_;
	return result;
}


/**
         * Whether the timer ends in the past
         */
gboolean dvb_timer_has_expired (DVBTimer* self) {
	gboolean result;
	time_t _tmp0_ = {0};
	time_t current_time;
	time_t end_time;
	g_return_val_if_fail (self != NULL, FALSE);
	current_time = (time (&_tmp0_), _tmp0_);
	end_time = dvb_timer_get_end_time_timestamp (self);
	result = end_time < current_time;
	return result;
}


char* dvb_timer_to_string (DVBTimer* self) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = g_strdup_printf ("channel: %u, start: %u-%u-%u %u:%u, duration: %u", dvb_channel_get_Sid (self->priv->_Channel), self->priv->_Year, self->priv->_Month, self->priv->_Day, self->priv->_Hour, self->priv->_Minute, self->priv->_Duration);
	return result;
}


static time_t dvb_timer_get_end_time_timestamp (DVBTimer* self) {
	time_t result;
	struct tm t;
	t = dvb_utils_create_time (self->priv->starttime.tm_year + 1900, self->priv->starttime.tm_mon + 1, self->priv->starttime.tm_mday, self->priv->starttime.tm_hour, self->priv->starttime.tm_min, 0);
	/* TODO Do we change the value of this.starttime each time?*/
	t.tm_min = t.tm_min + ((gint) self->priv->_Duration);
	result = mktime (&t);
	return result;
}


static time_t dvb_timer_get_start_time_timestamp (DVBTimer* self) {
	time_t result;
	struct tm t;
	t = dvb_timer_get_start_time_time (self);
	result = mktime (&t);
	return result;
}


guint32 dvb_timer_get_Id (DVBTimer* self) {
	guint32 result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Id;
	return result;
}


static void dvb_timer_set_Id (DVBTimer* self, guint32 value) {
	g_return_if_fail (self != NULL);
	self->priv->_Id = value;
	g_object_notify ((GObject *) self, "Id");
}


DVBChannel* dvb_timer_get_Channel (DVBTimer* self) {
	DVBChannel* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_Channel;
	return result;
}


static void dvb_timer_set_Channel (DVBTimer* self, DVBChannel* value) {
	DVBChannel* _tmp1_;
	DVBChannel* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_Channel = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_Channel == NULL) ? NULL : (self->priv->_Channel = (g_object_unref (self->priv->_Channel), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "Channel");
}


const char* dvb_timer_get_Name (DVBTimer* self) {
	const char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_Name;
	return result;
}


void dvb_timer_set_Name (DVBTimer* self, const char* value) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_Name = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->_Name = (g_free (self->priv->_Name), NULL), _tmp1_);
	g_object_notify ((GObject *) self, "Name");
}


guint dvb_timer_get_Year (DVBTimer* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Year;
	return result;
}


void dvb_timer_set_Year (DVBTimer* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_Year = value;
	g_object_notify ((GObject *) self, "Year");
}


guint dvb_timer_get_Month (DVBTimer* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Month;
	return result;
}


void dvb_timer_set_Month (DVBTimer* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_Month = value;
	g_object_notify ((GObject *) self, "Month");
}


guint dvb_timer_get_Day (DVBTimer* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Day;
	return result;
}


void dvb_timer_set_Day (DVBTimer* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_Day = value;
	g_object_notify ((GObject *) self, "Day");
}


guint dvb_timer_get_Hour (DVBTimer* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Hour;
	return result;
}


void dvb_timer_set_Hour (DVBTimer* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_Hour = value;
	g_object_notify ((GObject *) self, "Hour");
}


guint dvb_timer_get_Minute (DVBTimer* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Minute;
	return result;
}


void dvb_timer_set_Minute (DVBTimer* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_Minute = value;
	g_object_notify ((GObject *) self, "Minute");
}


guint dvb_timer_get_Duration (DVBTimer* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Duration;
	return result;
}


void dvb_timer_set_Duration (DVBTimer* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_Duration = value;
	g_object_notify ((GObject *) self, "Duration");
}


GstElement* dvb_timer_get_sink (DVBTimer* self) {
	GstElement* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_sink;
	return result;
}


void dvb_timer_set_sink (DVBTimer* self, GstElement* value) {
	GstElement* _tmp1_;
	GstElement* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_sink = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_)), (self->priv->_sink == NULL) ? NULL : (self->priv->_sink = (gst_object_unref (self->priv->_sink), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "sink");
}


guint dvb_timer_get_EventID (DVBTimer* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_EventID;
	return result;
}


void dvb_timer_set_EventID (DVBTimer* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_EventID = value;
	g_object_notify ((GObject *) self, "EventID");
}


static void dvb_timer_class_init (DVBTimerClass * klass) {
	dvb_timer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBTimerPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_timer_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_timer_set_property;
	G_OBJECT_CLASS (klass)->finalize = dvb_timer_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_ID, g_param_spec_uint ("Id", "Id", "Id", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_CHANNEL, g_param_spec_object ("Channel", "Channel", "Channel", DVB_TYPE_CHANNEL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_NAME, g_param_spec_string ("Name", "Name", "Name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_YEAR, g_param_spec_uint ("Year", "Year", "Year", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_MONTH, g_param_spec_uint ("Month", "Month", "Month", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_DAY, g_param_spec_uint ("Day", "Day", "Day", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_HOUR, g_param_spec_uint ("Hour", "Hour", "Hour", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_MINUTE, g_param_spec_uint ("Minute", "Minute", "Minute", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_DURATION, g_param_spec_uint ("Duration", "Duration", "Duration", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_SINK, g_param_spec_object ("sink", "sink", "sink", GST_TYPE_ELEMENT, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_TIMER_EVENT_ID, g_param_spec_uint ("EventID", "EventID", "EventID", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void dvb_timer_instance_init (DVBTimer * self) {
	self->priv = DVB_TIMER_GET_PRIVATE (self);
}


static void dvb_timer_finalize (GObject* obj) {
	DVBTimer * self;
	self = DVB_TIMER (obj);
	(self->priv->_Channel == NULL) ? NULL : (self->priv->_Channel = (g_object_unref (self->priv->_Channel), NULL));
	self->priv->_Name = (g_free (self->priv->_Name), NULL);
	(self->priv->_sink == NULL) ? NULL : (self->priv->_sink = (gst_object_unref (self->priv->_sink), NULL));
	G_OBJECT_CLASS (dvb_timer_parent_class)->finalize (obj);
}


GType dvb_timer_get_type (void) {
	static GType dvb_timer_type_id = 0;
	if (dvb_timer_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBTimerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_timer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBTimer), 0, (GInstanceInitFunc) dvb_timer_instance_init, NULL };
		dvb_timer_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBTimer", &g_define_type_info, 0);
	}
	return dvb_timer_type_id;
}


static void dvb_timer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBTimer * self;
	gpointer boxed;
	self = DVB_TIMER (object);
	switch (property_id) {
		case DVB_TIMER_ID:
		g_value_set_uint (value, dvb_timer_get_Id (self));
		break;
		case DVB_TIMER_CHANNEL:
		g_value_set_object (value, dvb_timer_get_Channel (self));
		break;
		case DVB_TIMER_NAME:
		g_value_set_string (value, dvb_timer_get_Name (self));
		break;
		case DVB_TIMER_YEAR:
		g_value_set_uint (value, dvb_timer_get_Year (self));
		break;
		case DVB_TIMER_MONTH:
		g_value_set_uint (value, dvb_timer_get_Month (self));
		break;
		case DVB_TIMER_DAY:
		g_value_set_uint (value, dvb_timer_get_Day (self));
		break;
		case DVB_TIMER_HOUR:
		g_value_set_uint (value, dvb_timer_get_Hour (self));
		break;
		case DVB_TIMER_MINUTE:
		g_value_set_uint (value, dvb_timer_get_Minute (self));
		break;
		case DVB_TIMER_DURATION:
		g_value_set_uint (value, dvb_timer_get_Duration (self));
		break;
		case DVB_TIMER_SINK:
		g_value_set_object (value, dvb_timer_get_sink (self));
		break;
		case DVB_TIMER_EVENT_ID:
		g_value_set_uint (value, dvb_timer_get_EventID (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_timer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBTimer * self;
	self = DVB_TIMER (object);
	switch (property_id) {
		case DVB_TIMER_ID:
		dvb_timer_set_Id (self, g_value_get_uint (value));
		break;
		case DVB_TIMER_CHANNEL:
		dvb_timer_set_Channel (self, g_value_get_object (value));
		break;
		case DVB_TIMER_NAME:
		dvb_timer_set_Name (self, g_value_get_string (value));
		break;
		case DVB_TIMER_YEAR:
		dvb_timer_set_Year (self, g_value_get_uint (value));
		break;
		case DVB_TIMER_MONTH:
		dvb_timer_set_Month (self, g_value_get_uint (value));
		break;
		case DVB_TIMER_DAY:
		dvb_timer_set_Day (self, g_value_get_uint (value));
		break;
		case DVB_TIMER_HOUR:
		dvb_timer_set_Hour (self, g_value_get_uint (value));
		break;
		case DVB_TIMER_MINUTE:
		dvb_timer_set_Minute (self, g_value_get_uint (value));
		break;
		case DVB_TIMER_DURATION:
		dvb_timer_set_Duration (self, g_value_get_uint (value));
		break;
		case DVB_TIMER_SINK:
		dvb_timer_set_sink (self, g_value_get_object (value));
		break;
		case DVB_TIMER_EVENT_ID:
		dvb_timer_set_EventID (self, g_value_get_uint (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}




