/* -*- Mode: vala; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-

   This file is part of GNOME Nibbles.

   Copyright (C) 2020 – Arnaud Bonatti <arnaud.bonatti@gmail.com>

   GNOME Nibbles is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   GNOME Nibbles is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GNOME Nibbles.  If not, see <https://www.gnu.org/licenses/>.
*/

namespace NibblesTest
{
    private static int main (string [] args)
    {
        Test.init (ref args);
        Test.add_func ("/Nibbles/test tests",
                                 test_tests);
        Test.add_func ("/Nibbles/test games",
                                 test_games);
        Test.add_func ("/Nibbles/test heads",
                                 test_heads);
        Test.add_func ("/Nibbles/test warps",
                                 test_warps);
        return Test.run ();
    }

    private static void test_tests ()
    {
        assert_true (1 + 1 == 2);
    }

    /*\
    * * utilities
    \*/

    private struct WormTest
    {
        int start_x;
        int start_y;
        int final_lives;
        int final_score;
    }

    private static void test_board (string [] board,
                                    uint8 regular_bonuses,
                                    uint8 final_bonuses,
                                    WormTest [] worms)
    {
        NibblesGame game = new NibblesGame (/* start level */ 1, /* speed */ 0, /* fakes */ false, board [0].char_count (), board.length, /* no random */ true);

        game.numhumans = 0;
        game.numai = worms.length;
        game.create_worms ();

        game.load_board (board, regular_bonuses);

        ulong [] worms_handlers = new ulong [game.worms.size];
        foreach (Worm worm in game.worms)
            // FIXME we should not have to connect to anything                  // TODO what’s this 3 in dematerialize?
            worms_handlers [worm.id] = worm.finish_added.connect (() => { worm.dematerialize (game.board, 3); worm.is_stopped = false; });

        assert_true (game.numworms   == worms.length);
        assert_true (game.worms.size == worms.length);

        uint8 applied_bonus = 0;
        ulong game_handler_1 = game.bonus_applied.connect ((bonus, worm) => { applied_bonus++; Test.message (@"worm $(worm.id) took bonus at ($(bonus.x), $(bonus.y))"); });

        game.add_worms ();
        game.start (/* add initial bonus */ true);

        for (uint8 i = 0; i < worms.length; i++)
        {
            Worm worm = game.worms.@get (i);
            WormTest worm_test = worms [i];

            assert_true (worm.lives == 6);
            assert_true (worm.score == 0);
//            Test.message (@"worm $(worm.id) start: ($(worm.head.x), $(worm.head.y))");
            assert_true (worm.head.x == worm_test.start_x && worm.head.y == worm_test.start_y);
        }

        // run until game is finished
        bool completed = false;
        ulong game_handler_2 = game.level_completed.connect (() => { completed = true; });
        MainContext context = MainContext.@default ();
        do context.iteration (/* may block */ false);
        while (!completed && (game.get_game_status () != GameStatus.GAMEOVER));

        for (uint8 i = 0; i < worms.length; i++)
        {
            Worm worm = game.worms.@get (i);
            WormTest worm_test = worms [i];

//            Test.message (@"worm $(worm.id) final lives: $(worm.lives)");
//            Test.message (@"worm $(worm.id) final score: $(worm.score)");
            assert_true (worm.lives == worm_test.final_lives);
            assert_true (worm.score == worm_test.final_score);
        }
//        Test.message (@"applied bonus: $applied_bonus");
        assert_true (applied_bonus == final_bonuses);

        foreach (Worm worm in game.worms)
            worm.disconnect (worms_handlers [worm.id]);
        game.disconnect (game_handler_1);
        game.disconnect (game_handler_2);

        Test.message ("");
    }

    /*\
    * * test games
    \*/

    private static void test_games ()
    {
        Test.message ("test board 008");
        test_board (level_008, /* regular bonus = 8 + numworms */ 12, /* final bonuses */ 15,
                    { WormTest () { start_x =  4, start_y = 14, final_lives = 6, final_score =  11 },
                      WormTest () { start_x = 18, start_y = 31, final_lives = 5, final_score =  14 },
                      WormTest () { start_x =  9, start_y = 39, final_lives = 6, final_score = 119 },
                      WormTest () { start_x = 51, start_y = 45, final_lives = 6, final_score =  19 }});

        Test.message ("test board 011");
        test_board (level_011, /* regular bonus = 8 + numworms */ 12, /* final bonuses */ 18,
                    { WormTest () { start_x = 15, start_y =  9, final_lives = 5, final_score =  12 },
                      WormTest () { start_x = 44, start_y =  9, final_lives = 6, final_score =   0 },
                      WormTest () { start_x = 76, start_y =  9, final_lives = 6, final_score =   0 },
                      WormTest () { start_x = 15, start_y = 56, final_lives = 6, final_score = 115 }});

        Test.message ("test board 025");
        test_board (level_025, /* regular bonus = 8 + numworms */ 12, /* final bonuses */ 14,
                    { WormTest () { start_x = 11, start_y =  6, final_lives = 5, final_score =   5 },
                      WormTest () { start_x = 80, start_y =  6, final_lives = 6, final_score =  26 },
                      WormTest () { start_x = 14, start_y = 37, final_lives = 6, final_score =  22 },
                      WormTest () { start_x = 79, start_y = 27, final_lives = 6, final_score =  22 }});
    }

    private const string [] level_008 = { // test because it has warps, left-to-right and top-to-bottom; not the exact code of the level (old style warps)
            "┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛........┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..++..................................................................................++..┃",
            "┃..+R...............................┏┓...........................┏┓....................+S..┃",
            "┃...u...............................┗╋┓..........................┃┃....................t...┃",
            "┃....................................┗╋┓.........................┃┃........................┃",
            "┃...........┏┓........................┗╋┓........................┃┃........................┃",
            "┃...........┃┃.........................┗╋┓.......................┃┃........................┃",
            "┃...▼.......┃┃..........................┗╋┓......................┃┃........................┃",
            "┃...........┃┃...........................┗╋┓.....................┃┃...┏━━━━━━━━┓...........┃",
            "┃...........┃┃............................┗╋┓....................┃┃...┗━━━━━━━━╋┓..........┃",
            "┃...........┃┃.............................┗╋┓...................┃┃............┗╋┓.........┃",
            "┃...........┃┃..............................┗╋┓..................┃┃.............┗╋┓........┃",
            "┃...........┃┃............┏┓.................┗╋┓.................┃┃..............┗╋━━┓.....┃",
            "┃...........┃┃...........┏╋┛..................┗╋┓................┃┃...............┗━━┛.....┃",
            "┃...........┃┃..........┏╋┛....................┗╋┓...............┃┃........................┃",
            "┛...........┗┛.........┏╋┛......................┗╋┓..............┃┃........................┗",
            "......................┏╋┛........................┗╋┓.............┃┃.........................",
            ".....................┏╋┛.........................┏╋┛.............┃┃.........................",
            "....................┏╋┛.........................┏╋┛..............┃┃.........................",
            "...................┏╋┛.........................┏╋┛...............┃┃.........................",
            "..................┏╋┛.........................┏╋┛................┃┃......┏┓.................",
            "..................┗┛.........................┏╋┛.................┃┃......┗╋┓................",
            "............................................┏╋┛..................┃┃.......┗╋┓...............",
            "............................................┗╋┓..................┃┃........┗╋┓..............",
            "┓.................▼..........................┗╋┓.................┃┃.........┗╋┓............┏",
            "┃.............................................┗╋┓................┃┃..........┗╋┓...........┃",
            "┃..............................................┗╋┓...............┃┃...........┗╋┓..........┃",
            "┃........┏┓.....................┏┓..............┗╋┓..............┃┃............┗┛..........┃",
            "┃........┃┃.....................┗╋┓..............┗╋┓.............┃┃........................┃",
            "┃........┃┃......................┗╋┓..............┗╋┓............┃┃........................┃",
            "┃........┃┃.......................┗╋┓..............┗╋┓...........┃┃........................┃",
            "┃........┃┃........................┗╋┓..............┗┛...........┃┃........................┃",
            "┃........┃┃.........................┗╋┓..........................┃┃........................┃",
            "┃........┃┗━━━━━━━┓..................┗╋┓.........................┃┃........................┃",
            "┃........┗━━━━━━━━┛...................┗╋┓........................┃┃........................┃",
            "┃......................................┗╋┓.......................┃┃........................┃",
            "┃.......................................┗╋┓......................┃┃....┏━━━━━━━━━━━━━┓.....┃",
            "┃...▶....................................┗╋┓.....................┃┃....┗━━━━━━━━━━━━━┛.....┃",
            "┃........................................┏╋┛.....................┃┃........................┃",
            "┃.......................................┏╋┛......................┃┃........................┃",
            "┃.................┏┓...................┏╋┛.......................┃┃........................┃",
            "┃.................┃┃..................┏╋┛........................┃┃........................┃",
            "┃...........┏┓....┃┃.................┏╋┛.........................┃┃........................┃",
            "┃..........┏╋┛....┃┃................┏╋┛..........................┃┃........................┃",
            "┃.........┏╋┛.....┃┃...............┏╋┛...........................┃┃........................┃",
            "┃........┏╋┛......┃┃...............┗┛............................┃┃........................┃",
            "┃.......┏╋┛.......┃┃.............................................┃┃........................┃",
            "┃......┏╋┛........┃┃.............................................┃┃........................┃",
            "┃......┗┛.........┃┃...............................▲.............┃┃........................┃",
            "┃.................┃┃.............................................┃┃........................┃",
            "┃.................┃┃.............................................┃┃........................┃",
            "┃.................┃┃.............................................┃┗━━━━━━━━━━━━━┓..........┃",
            "┃.................┃┃............┏━━━━━━━━━━━━━━━━━━━━━━┓.........┗━━━━━━━━━━━━━━┛..........┃",
            "┃.................┃┃............┗━━━━━━━━━━━━━━━━━━━━━━┛...................................┃",
            "┃.................┃┃.......................................................................┃",
            "┃.................┃┃...................................................................◀...┃",
            "┃.................┃┃.......................................................................┃",
            "┃.................┃┃......................◀................................................┃",
            "┃...s.............┃┃...................................................................r...┃",
            "┃..++.............┗┛...................................................................++..┃",
            "┃..+T..................................................................................+U..┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓........┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛"
        };

    private const string [] level_011 = { // test because it uses the two kind of warps
            "┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┳┳━━━━━━━━━━━━━━━━━━━━━━━━━━━━┳┳━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃..............▼..............┃┃............▼...............┃┃..............▼..............┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃......++............++......┃┃.............................┃",
            "┃.............................┃┃......+R............+S......┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃..............u..............┃┃............................┃┃..............t..............┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┗━┓........................┏━┛┃.............................┃",
            "┃.............................┗━━┛........................┗━━┛.............................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┣━━━━━━━━━━━━━━━━━━━━━┓.........┏┳┳┳┳┓....................┏┓.........┏━━━━━━━━━━━━━━━━━━━━━┫",
            "┣━━━━━━━━━━━━━━━━━━━━┓┃.........┗╋╋╋╋╋┓..................┏╋┛.........┃┏━━━━━━━━━━━━━━━━━━━━┫",
            "┃....................┃┃..........┗╋╋╋╋╋┓................┏╋┛..........┃┃....................┃",
            "┃....................┗┛...........┗╋╋╋╋╋┓..............┏╋┛...........┗┛....................┃",
            "┃..................................┗╋╋╋╋╋┓............┏╋┛..................................┃",
            "┃...................................┗╋╋╋╋╋┓..........┏╋┛...................................┃",
            "┃....................................┗╋╋╋╋╋┓........┏╋┛....................................┃",
            "┃.....................................┗╋╋╋╋╋┓......┏╋┛.....................................┃",
            "┃......................................┗╋╋╋╋╋┓....┏╋┛......................................┃",
            "┃.......................................┗╋╋╋╋╋┓..┏╋┛.......................................┃",
            "┃........................................┗╋╋╋╋┛.┏╋┛........................................┃",
            "┃.........++..............................┗╋╋┛.┏╋┛..............................++.........┃",
            "┃.........+V..............................┏╋┛.┏╋╋┓..............................+V.........┃",
            "┃........................................┏╋┛.┏╋╋╋╋┓........................................┃",
            "┃.......................................┏╋┛..┗╋╋╋╋╋┓.......................................┃",
            "┃......................................┏╋┛....┗╋╋╋╋╋┓......................................┃",
            "┃.....................................┏╋┛......┗╋╋╋╋╋┓.....................................┃",
            "┃....................................┏╋┛........┗╋╋╋╋╋┓....................................┃",
            "┃...................................┏╋┛..........┗╋╋╋╋╋┓...................................┃",
            "┃..................................┏╋┛............┗╋╋╋╋╋┓..................................┃",
            "┃....................┏┓...........┏╋┛..............┗╋╋╋╋╋┓...........┏┓....................┃",
            "┃....................┃┃..........┏╋┛................┗╋╋╋╋╋┓..........┃┃....................┃",
            "┣━━━━━━━━━━━━━━━━━━━━┛┃.........┏╋┛..................┗╋╋╋╋╋┓.........┃┗━━━━━━━━━━━━━━━━━━━━┫",
            "┣━━━━━━━━━━━━━━━━━━━━━┛.........┗┛....................┗┻┻┻┻┛.........┗━━━━━━━━━━━━━━━━━━━━━┫",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃.............................┏━━┓........................┏━━┓.............................┃",
            "┃.............................┃┏━┛........................┗━┓┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃..............s..............┃┃............................┃┃..............r..............┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃......++............++......┃┃.............................┃",
            "┃.............................┃┃......+T............+U......┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃..............▲..............┃┃................▲...........┃┃..............▲..............┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┃.............................┃┃............................┃┃.............................┃",
            "┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┻┻━━━━━━━━━━━━━━━━━━━━━━━━━━━━┻┻━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛"
        };

    private const string [] level_025 = { // test for ensuring the two first worms do not kill themselves
            "┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................┏┓..┏┓..........................................┃",
            "┃....┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┻┛..┗┻━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓....┃",
            "┃....┃▶..............................................................................◀┃....┃",
            "┃....┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┳┓..┏┳━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛....┃",
            "┃..........................................┗┛..┗┛..........................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃...............................┏┳━━┳┓.┏┳━━┳┓..┏┳━━┳┓.┏┳━━┳┓...............................┃",
            "┃...............................┣┛..┗┛.┗┛..┗┫..┣┛..┗┛.┗┛..┗┫...............................┃",
            "┃...............................┃...........┃..┃...........┃...............................┃",
            "┃...............................┃...........┃..┃...........┃...............................┃",
            "┃...............................┣┓.........┏┫..┣┓.........┏┫...............................┃",
            "┃...............................┗┛.........┗┛..┗┛.........┗┛...............................┃",
            "┃..........................................................................................┃",
            "┃...............................┏┓.........┏┓..┏┓.........┏┓...............................┃",
            "┃...............................┣┛.........┗┫..┣┛.........┗┫...............................┃",
            "┃...............................┃...........┃..┃...........┃...............................┃",
            "┃...............................┃...........┃..┃...........┃...............................┃",
            "┃...............................┣┓..┏┓.┏┓..┏┫..┣┓..┏┓.┏┓..┏┫...............................┃",
            "┃...............................┗┻━━┻┛.┗┻━━┻┛..┗┻━━┻┛.┗┻━━┻┛...............................┃",
            "┃.............▼................................................................▲...........┃",
            "┃..........................................................................................┃",
            "┃...............................┏┳━━┳┓.┏┳━━┳┓..┏┳━━┳┓.┏┳━━┳┓...............................┃",
            "┃...............................┣┛..┗┛.┗┛..┗┫..┣┛..┗┛.┗┛..┗┫...............................┃",
            "┃...............................┃...........┃..┃...........┃...............................┃",
            "┃...............................┃...........┃..┃...........┃...............................┃",
            "┃...............................┣┓.........┏┫..┣┓.........┏┫...............................┃",
            "┃...............................┗┛.........┗┛..┗┛.........┗┛...............................┃",
            "┃..........................................................................................┃",
            "┃...............................┏┓.........┏┓..┏┓.........┏┓...............................┃",
            "┃...............................┣┛.........┗┫..┣┛.........┗┫...............................┃",
            "┃...............................┃...........┃..┃...........┃...............................┃",
            "┃...............................┃...........┃..┃...........┃...............................┃",
            "┃...............................┣┓..┏┓.┏┓..┏┫..┣┓..┏┓.┏┓..┏┫...............................┃",
            "┃...............................┗┻━━┻┛.┗┻━━┻┛..┗┻━━┻┛.┗┻━━┻┛...............................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................┏┓..┏┓..........................................┃",
            "┃....┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┻┛..┗┻━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓....┃",
            "┃....┃▶..............................................................................◀┃....┃",
            "┃....┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┳┓..┏┳━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛....┃",
            "┃..........................................┗┛..┗┛..........................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┃..........................................................................................┃",
            "┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛"
        };

    /*\
    * * test heads
    \*/

    private static void test_heads ()
    {
        Test.message ("test heads 1");
        test_board (test_heads_1, 1, 1, { WormTest () { start_x =  6, start_y = 4, final_lives = 6, final_score = 1 },
                                          WormTest () { start_x = 11, start_y = 4, final_lives = 6, final_score = 0 }});

        Test.message ("test heads 2");
        test_board (test_heads_2, 1, 1, { WormTest () { start_x =  6, start_y = 4, final_lives = 6, final_score = 1 },
                                          WormTest () { start_x = 11, start_y = 4, final_lives = 4, final_score = 0 }});

        Test.message ("test heads 3");
        test_board (test_heads_3, 1, 1, { WormTest () { start_x =  6, start_y = 4, final_lives = 6, final_score = 1 },
                                          WormTest () { start_x = 10, start_y = 4, final_lives = 6, final_score = 0 }});

        Test.message ("test heads 4");
        test_board (test_heads_4, 1, 0, { WormTest () { start_x =  6, start_y = 4, final_lives = 0, final_score = 0 },
                                          WormTest () { start_x = 10, start_y = 4, final_lives = 0, final_score = 0 }});

        Test.message ("test heads 5");
        test_board (test_heads_5, 1, 1, { WormTest () { start_x =  6, start_y = 1, final_lives = 6, final_score = 1 },
                                          WormTest () { start_x =  6, start_y = 4, final_lives = 6, final_score = 0 }});

        Test.message ("test heads 6");
        test_board (test_heads_6, 1, 1, { WormTest () { start_x =  6, start_y = 1, final_lives = 4, final_score = 0 },
                                          WormTest () { start_x =  6, start_y = 4, final_lives = 6, final_score = 1 }});

        Test.message ("test heads 9");
        test_board (test_heads_9, 1, 1, { WormTest () { start_x =  6, start_y = 1, final_lives = 6, final_score = 1 },
                                          WormTest () { start_x =  6, start_y = 4, final_lives = 4, final_score = 0 }});

        Test.message ("test heads 7");
        test_board (test_heads_7, 1, 1, { WormTest () { start_x =  6, start_y = 2, final_lives = 6, final_score = 1 },
                                          WormTest () { start_x =  6, start_y = 4, final_lives = 6, final_score = 0 }});

        Test.message ("test heads 8");
        test_board (test_heads_8, 1, 0, { WormTest () { start_x =  6, start_y = 2, final_lives = 0, final_score = 0 },
                                          WormTest () { start_x =  6, start_y = 4, final_lives = 0, final_score = 0 }});

        Test.message ("test heads 0");
        test_board (test_heads_0, 1, 1, { WormTest () { start_x =  6, start_y = 2, final_lives = 6, final_score = 1 },
                                          WormTest () { start_x =  6, start_y = 4, final_lives = 6, final_score = 0 }});
    }

    private const int test_heads_width = 18;
    private const int test_heads_height = 6;
    private const string [] test_heads_1 = {
            "┏━━━━━━━━━━━━━━━━┓",
            "┃................┃",
            "┃................┃",
            "┣━━━━━━━..━━━━━━━┫",
            "┃▶..............◀┃",
            "┗━━━━━━━━━━━━━━━━┛"
        };  /* expected: 6, 6 */
    private const string [] test_heads_2 = {
            "┏━━━━━━━━━━━━━━━━┓",
            "┃................┃",
            "┃................┃",
            "┣━━━━━━━.━━━━━━━━┫",
            "┃▶..............◀┃",
            "┗━━━━━━━━━━━━━━━━┛"
        };  /* expected: 6, 5 */
    private const string [] test_heads_3 = {
            "┏━━━━━━━━━━━━━━━━┓",
            "┃................┃",
            "┃................┃",
            "┣━━━━━━━..━━━━━━┳┫",
            "┃▶.............◀┣┫",
            "┗━━━━━━━━━━━━━━━┻┛"
        };  /* expected: 6, 6 */
    private const string [] test_heads_4 = {
            "┏━━━━━━━━━━━━━━━━┓",
            "┃................┃",
            "┃................┃",
            "┣━━━━━━━.━━━━━━━┳┫",
            "┃▶.............◀┣┫",
            "┗━━━━━━━━━━━━━━━┻┛"
        };  /* expected: 0, 0 */
    private const string [] test_heads_5 = {
            "┏━━━━━━━┳━━━━━━━━┓",
            "┃▶......┃........┃",
            "┗━━━━━┓..........┃",
            "┏━━━━━┛..........┃",
            "┃▶......┃........┃",
            "┗━━━━━━━┻━━━━━━━━┛"
        };  /* expected: 6, 6 */
    private const string [] test_heads_6 = {
            "┏━━━━━━━┳━━━━━━━━┓",
            "┃▶......┃........┃",
            "┗━━━━━┓.┃........┃",
            "┏━━━━━┛..........┃",
            "┃▶......┃........┃",
            "┗━━━━━━━┻━━━━━━━━┛"
        };  /* expected: 4, 6 */
    private const string [] test_heads_9 = {
            "┏━━━━━━━┳━━━━━━━━┓",
            "┃▶......┃........┃",
            "┗━━━━━┓..........┃",
            "┏━━━━━┛.┃........┃",
            "┃▶......┃........┃",
            "┗━━━━━━━┻━━━━━━━━┛"
        };  /* expected: 6, 5 */
    private const string [] test_heads_7 = {
            "........┏━━━━━━━━┓",
            "┏━━━━━━━┛........┃",
            "┃▶...............┃",
            "┣━━━━━━..........┃",
            "┃▶......┃........┃",
            "┗━━━━━━━┻━━━━━━━━┛"
        };  /* expected: 6, 6 */
    private const string [] test_heads_8 = {
            "........┏━━━━━━━━┓",
            "┏━━━━━━━┫........┃",
            "┃▶......┃........┃",
            "┣━━━━━━..........┃",
            "┃▶......┃........┃",
            "┗━━━━━━━┻━━━━━━━━┛"
        };  /* expected: 0, 0 */
    private const string [] test_heads_0 = {
            "........┏━━━━━━━━┓",
            "┏━━━━━━━┫........┃",
            "┃▶......┃........┃",
            "┣━━━━━━..........┃",
            "┃▶...............┃",
            "┗━━━━━━━━━━━━━━━━┛"
        };  /* expected: 6, 6 */

    /*\
    * * test warps
    \*/

    private static void test_warps ()
    {
        Test.message ("test warps 1");
        test_board (test_warps_1, 1, 1, { WormTest () { start_x =  6, start_y = 4, final_lives = 6, final_score = 1 }});

        Test.message ("test warps 2");
        test_board (test_warps_2, 1, 0, { WormTest () { start_x =  6, start_y = 4, final_lives = 0, final_score = 0 }});

        Test.message ("test warps 3");
        test_board (test_warps_3, 1, 0, { WormTest () { start_x =  9, start_y = 4, final_lives = 0, final_score = 0 }});

        Test.message ("test warps 4");
        test_board (test_warps_4, 1, 1, { WormTest () { start_x =  9, start_y = 4, final_lives = 6, final_score = 1 }});
    }

    private const int test_warps_width = 11;
    private const int test_warps_height = 6;
    private const string [] test_warps_1 = {
            "┏━━┓┏━━┓┏━┓",
            "┃++┗┛..┗┛.┃",
            "┃+S....┏━━┫",
            "┣━━━━━━┛++┃",
            "┃▶......+S┃",
            "┗━━━━━━━━━┛"
        }; /* expected: 6 */
    private const string [] test_warps_2 = {
            "┏━━━━━━┓┏━┓",
            "┃++....┗┛.┃",
            "┃+S┏┓..┏━━┫",
            "┣━━┛┗━━┛++┃",
            "┃▶......+S┃",
            "┗━━━━━━━━━┛"
        }; /* expected: 0 */
    private const string [] test_warps_3 = {
            "┏━━━━━━┳━━┓",
            "┃......┃++┃",
            "┃.━┓...┃+S┃",
            "┃++┣━━━┻━.┃",
            "┃+S┃▶.....┃",
            "┗━━┻━━━━━━┛"
        }; /* expected: 0 */
    private const string [] test_warps_4 = {
            "┏━━━━━━┳━━┓",
            "┃......┃++┃",
            "┣━.┃...┃+S┃",
            "┃++┣━━━┻━.┃",
            "┃+S┃▶.....┃",
            "┗━━┻━━━━━━┛"
        }; /* expected: 6 */
}
