/* gnome-pilot-conduit-standard-abs.c
 * Copyright (C) 1999  Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome.h>
#include <pi-source.h>
#include <pi-socket.h>
#include <pi-dlp.h>
#include "gnome-pilot-conduit-standard-abs.h"
#include "manager.h"

enum {
	MATCH_RECORD,
	FREE_MATCH,
	ARCHIVE_LOCAL,
	ARCHIVE_REMOTE,
	STORE_REMOTE,
	CLEAR_STATUS_ARCHIVE_LOCAL,
	ITERATE,
	ITERATE_SPECIFIC,
	PURGE,
	SET_STATUS,
	SET_ARCHIVED,
	SET_PILOT_ID,
	COMPARE,
	COMPARE_BACKUP,
	FREE_TRANSMIT,
	DELETE_ALL,
	TRANSMIT,
	PRE_SYNC,
	LAST_SIGNAL
};

enum StandardAbsSyncDirection {
	SyncToRemote = 0x01,
	SyncToLocal  = 0x02
};

typedef enum StandardAbsSyncDirection StandardAbsSyncDirection;
#define SyncBothWays SyncToRemote|SyncToLocal


#define LOG_CASE6 _("local record modified on pilot, not deleting\n")
#define LOG_CASE10 _("merge conflict, pilots and local record swapped\n")
#define LOG_CASE5 _("pilot record modified locally, not deleting\n")

static void gnome_pilot_conduit_standard_abs_init		(GnomePilotConduitStandardAbs		 *pilot_conduit_standard_abs);
static void gnome_pilot_conduit_standard_abs_class_init		(GnomePilotConduitStandardAbsClass	 *klass);
static gint gnome_pilot_conduit_standard_real_copy_to_pilot     (GnomePilotConduitStandard *conduit,
								 GnomePilotDBInfo  *dbinfo);
static gint gnome_pilot_conduit_standard_real_copy_from_pilot   (GnomePilotConduitStandard *conduit,
								 GnomePilotDBInfo  *dbinfo);
static gint gnome_pilot_conduit_standard_real_merge_to_pilot    (GnomePilotConduitStandard *conduit,
								 GnomePilotDBInfo  *dbinfo);
static gint gnome_pilot_conduit_standard_real_merge_from_pilot  (GnomePilotConduitStandard *conduit,
								 GnomePilotDBInfo  *dbinfo);
static gint gnome_pilot_conduit_standard_real_synchronize       (GnomePilotConduitStandard *conduit,
								 GnomePilotDBInfo  *dbinfo);

int standard_abs_sync_record (GnomePilotConduitStandardAbs *, int, int, LocalRecord *, PilotRecord *, StandardAbsSyncDirection);
gint standard_abs_check_locally_deleted_records(GnomePilotConduitStandardAbs *, int, int, StandardAbsSyncDirection);
gint standard_abs_merge_to_remote (GnomePilotConduitStandardAbs *, int, int, StandardAbsSyncDirection);
gint standard_abs_merge_to_local (GnomePilotConduitStandardAbs *, int, int, StandardAbsSyncDirection);
int SlowSync (int, int, GnomePilotConduitStandardAbs *);
gint FastSync (int, int, GnomePilotConduitStandardAbs *);
gboolean gpilot_sync_pc_match(GnomePilotDBInfo*);
void standard_abs_open_db(GnomePilotConduitStandardAbs*,GnomePilotDBInfo*);
void standard_abs_close_db_and_purge_local(GnomePilotConduitStandardAbs*,GnomePilotDBInfo*);

static GnomePilotConduitStandardClass *parent_class = NULL;
static guint pilot_conduit_standard_abs_signals[LAST_SIGNAL] = { 0 };

GtkType
gnome_pilot_conduit_standard_abs_get_type (void)
{
	static GtkType pilot_conduit_standard_abs_type = 0;

	if (!pilot_conduit_standard_abs_type)
	{
		static const GtkTypeInfo pilot_conduit_standard_abs_info =
		{
			"GnomePilotConduitStandardAbs",
			sizeof (GnomePilotConduitStandardAbs),
			sizeof (GnomePilotConduitStandardAbsClass),
			(GtkClassInitFunc) gnome_pilot_conduit_standard_abs_class_init,
			(GtkObjectInitFunc) gnome_pilot_conduit_standard_abs_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		pilot_conduit_standard_abs_type = gtk_type_unique (gnome_pilot_conduit_standard_get_type (), &pilot_conduit_standard_abs_info);
	}

	return pilot_conduit_standard_abs_type;
}

GtkObject *
gnome_pilot_conduit_standard_abs_new (const char *db_name,
				      guint32 creator_id)
{
	GtkObject *retval;
	retval =  GTK_OBJECT (gtk_type_new (gnome_pilot_conduit_standard_abs_get_type ()));
	gnome_pilot_conduit_standard_construct (GNOME_PILOT_CONDUIT_STANDARD (retval),
						db_name,
						creator_id);

	return retval;
}

/* We have a bunch of marshalling functions that we need to declare.
 * We do so here */
typedef gint (*GtkSignal_INT__POINTER_POINTER) (GtkObject * object,
						gpointer arg1,
						gpointer arg2,
						gpointer user_data);
static void
gtk_marshal_INT__POINTER_POINTER (GtkObject * object,
				  GtkSignalFunc func,
				  gpointer func_data,
				  GtkArg * args)
{
	GtkSignal_INT__POINTER_POINTER rfunc;
	gint *return_val;
	return_val = GTK_RETLOC_INT (args[2]);
	rfunc = (GtkSignal_INT__POINTER_POINTER) func;
	*return_val = (*rfunc) (object,
				GTK_VALUE_POINTER (args[0]),
				GTK_VALUE_POINTER (args[1]),
				func_data);
}

typedef gint (*GtkSignal_INT__POINTER_INT) (GtkObject * object,
					    gpointer arg1,
					    gint arg2,
					    gpointer user_data);
static void
gtk_marshal_INT__POINTER_INT (GtkObject * object,
			      GtkSignalFunc func,
			      gpointer func_data,
			      GtkArg * args)
{
	GtkSignal_INT__POINTER_INT rfunc;
	gint *return_val;
	return_val = GTK_RETLOC_INT (args[2]);
	rfunc = (GtkSignal_INT__POINTER_INT) func;
	*return_val = (*rfunc) (object,
				GTK_VALUE_POINTER (args[0]),
				GTK_VALUE_INT (args[1]),
				func_data);
}

typedef gint (*GtkSignal_INT__POINTER_INT_INT) (GtkObject * object,
						gpointer arg1,
						gint arg2,
						gint arg3,
						gpointer user_data);
static void
gtk_marshal_INT__POINTER_INT_INT (GtkObject * object,
				  GtkSignalFunc func,
				  gpointer func_data,
				  GtkArg * args)
{
	GtkSignal_INT__POINTER_INT_INT rfunc;
	gint *return_val;
	return_val = GTK_RETLOC_INT (args[3]);
	rfunc = (GtkSignal_INT__POINTER_INT_INT) func;
	*return_val = (*rfunc) (object,
				GTK_VALUE_POINTER (args[0]),
				GTK_VALUE_INT (args[1]),
				GTK_VALUE_INT (args[2]),
				func_data);
}


typedef gint (*GtkSignal_INT__NONE) (GtkObject * object,
				     gpointer user_data);
static void
gtk_marshal_INT__NONE (GtkObject * object,
		       GtkSignalFunc func,
		       gpointer func_data,
		       GtkArg * args)
{
	GtkSignal_INT__NONE rfunc;
	gint *return_val;
	return_val = GTK_RETLOC_INT (args[0]);
	rfunc = (GtkSignal_INT__NONE) func;
	*return_val = (*rfunc) (object,
				func_data);
}

static void
gnome_pilot_conduit_standard_abs_class_init (GnomePilotConduitStandardAbsClass *klass)
{
	GtkObjectClass *object_class;
	GnomePilotConduitStandardClass *conduit_standard_class;

	object_class = (GtkObjectClass*) klass;
	conduit_standard_class = (GnomePilotConduitStandardClass *) klass;

	parent_class = gtk_type_class (gnome_pilot_conduit_standard_get_type ());

	pilot_conduit_standard_abs_signals[MATCH_RECORD] =
		gtk_signal_new ("match_record",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, match_record),
				gtk_marshal_INT__POINTER_POINTER,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[FREE_MATCH] =
		gtk_signal_new ("free_match",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, free_match),
				gtk_marshal_INT__POINTER,
				GTK_TYPE_INT, 1, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[ARCHIVE_LOCAL] =
		gtk_signal_new ("archive_local",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, archive_local),
				gtk_marshal_INT__POINTER,
				GTK_TYPE_INT, 1, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[ARCHIVE_REMOTE] =
		gtk_signal_new ("archive_remote",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, archive_remote),
				gtk_marshal_INT__POINTER_POINTER,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[STORE_REMOTE] =
		gtk_signal_new ("store_remote",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, store_remote),
				gtk_marshal_INT__POINTER,
				GTK_TYPE_INT, 1, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[CLEAR_STATUS_ARCHIVE_LOCAL] =
		gtk_signal_new ("clear_status_archive_local",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, clear_status_archive_local),
				gtk_marshal_INT__POINTER,
				GTK_TYPE_INT, 1, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[ITERATE] =
		gtk_signal_new ("iterate",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, iterate),
				gtk_marshal_INT__POINTER,
				GTK_TYPE_INT, 1, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[ITERATE_SPECIFIC] =
		gtk_signal_new ("iterate_specific",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, iterate_specific),
				gtk_marshal_INT__POINTER_INT_INT,
				GTK_TYPE_INT, 3, GTK_TYPE_POINTER, GTK_TYPE_INT, GTK_TYPE_INT);

	pilot_conduit_standard_abs_signals[PURGE] =
		gtk_signal_new ("purge",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, purge),
				gtk_marshal_INT__NONE,
				GTK_TYPE_INT, 0);

	pilot_conduit_standard_abs_signals[SET_STATUS] =
		gtk_signal_new ("set_status",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, set_status),
				gtk_marshal_INT__POINTER_INT,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_INT);

	pilot_conduit_standard_abs_signals[SET_ARCHIVED] =
		gtk_signal_new ("set_archived",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, set_archived),
				gtk_marshal_INT__POINTER_INT,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_INT);

	pilot_conduit_standard_abs_signals[SET_PILOT_ID] =
		gtk_signal_new ("set_pilot_id",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, set_pilot_id),
				gtk_marshal_INT__POINTER_INT,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_INT);

	pilot_conduit_standard_abs_signals[COMPARE] =
		gtk_signal_new ("compare",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, compare),
				gtk_marshal_INT__POINTER_POINTER,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[COMPARE_BACKUP] =
		gtk_signal_new ("compare_backup",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, compare_backup),
				gtk_marshal_INT__POINTER_POINTER,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[FREE_TRANSMIT] =
		gtk_signal_new ("free_transmit",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, free_transmit),
				gtk_marshal_INT__POINTER_POINTER,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[DELETE_ALL] =
		gtk_signal_new ("delete_all",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, delete_all),
				gtk_marshal_INT__NONE,
				GTK_TYPE_INT, 0);

	pilot_conduit_standard_abs_signals[TRANSMIT] =
		gtk_signal_new ("transmit",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, transmit),
				gtk_marshal_INT__POINTER_POINTER,
				GTK_TYPE_INT, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	pilot_conduit_standard_abs_signals[PRE_SYNC] =
		gtk_signal_new ("pre_sync",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomePilotConduitStandardAbsClass, pre_sync),
				gtk_marshal_INT__POINTER,
				GTK_TYPE_INT, 1, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, pilot_conduit_standard_abs_signals, LAST_SIGNAL);

	conduit_standard_class->copy_to_pilot = gnome_pilot_conduit_standard_real_copy_to_pilot;
	conduit_standard_class->copy_from_pilot = gnome_pilot_conduit_standard_real_copy_from_pilot;
	conduit_standard_class->merge_to_pilot = gnome_pilot_conduit_standard_real_merge_to_pilot;
	conduit_standard_class->merge_from_pilot = gnome_pilot_conduit_standard_real_merge_from_pilot;
	conduit_standard_class->synchronize = gnome_pilot_conduit_standard_real_synchronize;
}


/*
  Given PilotRecord with a "fresh" attr field, set by a dlp call,
  sets secret, archived and attr to GnomePilot values
 */
static void
standard_abs_compute_attr_field(PilotRecord *remote)
{
	remote->secret = remote->attr & dlpRecAttrSecret;
	remote->archived = remote->attr & dlpRecAttrArchived;
	if (remote->attr & dlpRecAttrDeleted)
		remote->attr = GnomePilotRecordDeleted;
	else if (remote->attr & dlpRecAttrDirty)
		remote->attr = GnomePilotRecordModified;
	else
		remote->attr = GnomePilotRecordNothing;
}

static void
gnome_pilot_conduit_standard_abs_init (GnomePilotConduitStandardAbs *pilot_conduit_standard_abs)
{
	pilot_conduit_standard_abs->recordIdsToIgnore = NULL;
	pilot_conduit_standard_abs->totalRecords = 0;
}

static gint
gnome_pilot_conduit_standard_real_copy_to_pilot     (GnomePilotConduitStandard *conduit_standard,
						     GnomePilotDBInfo  *dbinfo)
{
	GnomePilotConduitStandardAbs *conduit = NULL;
	LocalRecord *local = NULL;
	int retval = 0;
	recordid_t assigned_id;

	g_return_val_if_fail (conduit_standard != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit_standard), -1);

	conduit = GNOME_PILOT_CONDUIT_STANDARD_ABS (conduit_standard);

	standard_abs_open_db(conduit, dbinfo);

	if( gnome_pilot_conduit_standard_abs_pre_sync(conduit, dbinfo) != 0) {
		g_warning(_("Conduits initialization failed, aborting operation"));
		return -1;
	}
	if (dlp_DeleteRecord (dbinfo->pilot_socket, dbinfo->db_handle, 1, 0) < 0) {
		g_warning(_("Unable to delete all records in pilot database, aborting operation."));
		return -1;
	}
	while (gnome_pilot_conduit_standard_abs_iterate (conduit, &local) && local) {
		if (local->archived) {
			retval = gnome_pilot_conduit_standard_abs_clear_status_archive_local (conduit, local);
			gnome_pilot_conduit_standard_abs_set_status (conduit, local, GnomePilotRecordDeleted);
		} else if (local->attr != GnomePilotRecordDeleted) {
			PilotRecord *remote;
			gnome_pilot_conduit_standard_abs_transmit (conduit, local, &remote);
			if(remote==NULL) {
			  g_warning(_("Conduit did not return a record"));
			  break;
			}
			gnome_pilot_conduit_standard_abs_set_status (conduit, local, GnomePilotRecordNothing);
			retval = dlp_WriteRecord(dbinfo->pilot_socket,
						 dbinfo->db_handle,
						 remote->secret?dlpRecAttrSecret:0,
						 remote->ID,
						 remote->category,
						 remote->record,
						 remote->length,
						 &assigned_id);
			if(retval>0) gnome_pilot_conduit_standard_abs_set_pilot_id(conduit,local,assigned_id);
		        gnome_pilot_conduit_standard_abs_free_transmit (conduit, local, &remote);
		}
	}

	standard_abs_close_db_and_purge_local(conduit,dbinfo);

	return 0;
}

static gint
gnome_pilot_conduit_standard_real_copy_from_pilot   (GnomePilotConduitStandard *conduit_standard,
						     GnomePilotDBInfo  *dbinfo)
{
	GnomePilotConduitStandardAbs *conduit = NULL;
	unsigned char buffer[0xffff];
	int index = 0;
	PilotRecord remote;

	g_return_val_if_fail (conduit_standard != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit_standard), -1);

	conduit = GNOME_PILOT_CONDUIT_STANDARD_ABS (conduit_standard);

	remote.record = buffer;
	standard_abs_open_db(conduit,dbinfo);

	if(gnome_pilot_conduit_standard_abs_pre_sync(conduit, dbinfo)!=0) {
		g_warning(_("Conduits initialization failed, aborting operation"));
		return -1;
	}
	if (gnome_pilot_conduit_standard_abs_delete_all (conduit) < 0) {
		g_warning(_("Unable to delete all records in local database, aborting operation."));
		return -1;
	}

	while (dlp_ReadRecordByIndex (dbinfo->pilot_socket,
				      dbinfo->db_handle,
				      index,
				      remote.record,
				      &remote.ID,
				      &remote.length,
				      &remote.attr,
				      &remote.category) >= 0) {
		standard_abs_compute_attr_field(&remote);
		if (remote.archived) {
			remote.attr = GnomePilotRecordNothing;
			remote.archived = 0;
			gnome_pilot_conduit_standard_abs_archive_remote (conduit, 
									 NULL, 
									 &remote);
		} else if (remote.attr != GnomePilotRecordDeleted) {
			remote.attr = GnomePilotRecordNothing;
			remote.archived = 0;
			gnome_pilot_conduit_standard_abs_store_remote (conduit, &remote);
		}
		index++;
	}

	standard_abs_close_db_and_purge_local(conduit,dbinfo);

	return 0;
}

static gint
gnome_pilot_conduit_standard_real_merge_to_pilot    (GnomePilotConduitStandard *conduit_standard,
						     GnomePilotDBInfo  *dbinfo)
{
	GnomePilotConduitStandardAbs *conduit = NULL;

	g_return_val_if_fail (conduit_standard != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit_standard), -1);	

	conduit = GNOME_PILOT_CONDUIT_STANDARD_ABS (conduit_standard);

	standard_abs_open_db(conduit, dbinfo);
	if(gnome_pilot_conduit_standard_abs_pre_sync(conduit, dbinfo)!=0) {
		g_warning(_("Conduits initialization failed, aborting operation"));
		return -1;
	}
	standard_abs_merge_to_remote(conduit, dbinfo->pilot_socket, dbinfo->db_handle, SyncToRemote );

	standard_abs_close_db_and_purge_local(conduit,dbinfo);
	
	return 0;
}

static gint
gnome_pilot_conduit_standard_real_merge_from_pilot  (GnomePilotConduitStandard *conduit_standard,
						     GnomePilotDBInfo  *dbinfo)
{
	GnomePilotConduitStandardAbs *conduit = NULL;

	g_return_val_if_fail (conduit_standard != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit_standard), -1);

	conduit = GNOME_PILOT_CONDUIT_STANDARD_ABS (conduit_standard);

	standard_abs_open_db(conduit, dbinfo);
	if(gnome_pilot_conduit_standard_abs_pre_sync(conduit, dbinfo)!=0) {
		g_warning(_("Conduits initialization failed, aborting operation"));
		return -1;
	}
	standard_abs_merge_to_local(conduit, dbinfo->pilot_socket, dbinfo->db_handle, SyncToLocal );

	standard_abs_close_db_and_purge_local(conduit,dbinfo);

	return 0;
}

static gint
gnome_pilot_conduit_standard_real_synchronize       (GnomePilotConduitStandard *conduit_standard,
						     GnomePilotDBInfo  *dbinfo)
{
	int retval = 0;
	GnomePilotConduitStandardAbs *conduit = NULL;

	g_return_val_if_fail (conduit_standard != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit_standard), -1);

	conduit = GNOME_PILOT_CONDUIT_STANDARD_ABS(conduit_standard);

	standard_abs_open_db(conduit, dbinfo);
	if(gnome_pilot_conduit_standard_abs_pre_sync(conduit, dbinfo)!=0) {
		g_warning(_("Conduits initialization failed, aborting operation"));
		return -1;
	}

      	if (conduit_standard->slow==FALSE && gpilot_sync_pc_match (dbinfo)==TRUE) {
		retval = FastSync(dbinfo->pilot_socket, dbinfo->db_handle, conduit);
	} else {
		retval = SlowSync(dbinfo->pilot_socket, dbinfo->db_handle, conduit);
	}
	/* now disable the slow=true setting, so it doens't pass on to next time */
	if(conduit_standard->slow==TRUE)
		conduit_standard->slow = FALSE;

	standard_abs_merge_to_remote(conduit, dbinfo->pilot_socket, dbinfo->db_handle, SyncBothWays );
	standard_abs_check_locally_deleted_records(conduit, dbinfo->pilot_socket, 
						   dbinfo->db_handle, SyncBothWays);

	standard_abs_close_db_and_purge_local(conduit,dbinfo);

	return retval;
}

gint
gnome_pilot_conduit_standard_abs_match_record (GnomePilotConduitStandardAbs *conduit,
					       LocalRecord **local,
					       PilotRecord *remote)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [MATCH_RECORD],
			 local,
			 remote,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_free_match (GnomePilotConduitStandardAbs *conduit,
					     LocalRecord **local)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);


	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [FREE_MATCH],
			 local,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_archive_local (GnomePilotConduitStandardAbs *conduit,
						LocalRecord *local)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [ARCHIVE_LOCAL],
			 local,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_archive_remote (GnomePilotConduitStandardAbs *conduit,
						 LocalRecord *local,
						 PilotRecord *remote)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [ARCHIVE_REMOTE],
			 local,
			 remote,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_store_remote (GnomePilotConduitStandardAbs *conduit,
					       PilotRecord *remote)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [STORE_REMOTE],
			 remote,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_clear_status_archive_local (GnomePilotConduitStandardAbs *conduit,
							     LocalRecord *local)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [CLEAR_STATUS_ARCHIVE_LOCAL],
			 local,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_iterate (GnomePilotConduitStandardAbs *conduit,
					  LocalRecord **local)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [ITERATE],
			 local,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_iterate_specific (GnomePilotConduitStandardAbs *conduit,
						   LocalRecord **local,
						   gint flag,
						   gint archived)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [ITERATE_SPECIFIC],
			 local,
			 flag,
			 archived,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_purge (GnomePilotConduitStandardAbs *conduit)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [PURGE],
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_set_status (GnomePilotConduitStandardAbs *conduit,
					     LocalRecord *local,
					     gint status)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [SET_STATUS],
			 local,
			 status,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_set_archived (GnomePilotConduitStandardAbs *conduit,
					       LocalRecord *local,
					       gint archived)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [SET_ARCHIVED],
			 local,
			 archived
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_set_pilot_id (GnomePilotConduitStandardAbs *conduit,
					       LocalRecord *local,
					       guint32 id)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [SET_PILOT_ID],
			 local,
			 id,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_compare (GnomePilotConduitStandardAbs *conduit,
					  LocalRecord *local,
					  PilotRecord *remote)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [COMPARE],
			 local,
			 remote,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_compare_backup (GnomePilotConduitStandardAbs *conduit,
						 LocalRecord *local,
						 PilotRecord *remote)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [COMPARE_BACKUP],
			 local,
			 remote,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_free_transmit (GnomePilotConduitStandardAbs *conduit,
						LocalRecord *local,
						PilotRecord **remote)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [FREE_TRANSMIT],
			 local,
			 remote,
			 &retval);
	return retval;

}

gint
gnome_pilot_conduit_standard_abs_delete_all (GnomePilotConduitStandardAbs *conduit)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [DELETE_ALL],
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_transmit (GnomePilotConduitStandardAbs *conduit,
					   LocalRecord *local,
					   PilotRecord **remote)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [TRANSMIT],
			 local,
			 remote,
			 &retval);
	return retval;
}

gint
gnome_pilot_conduit_standard_abs_pre_sync (GnomePilotConduitStandardAbs *conduit,
					   GnomePilotDBInfo  *dbinfo)
{
	gint retval;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (dbinfo != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD_ABS (conduit), -1);

	gtk_signal_emit (GTK_OBJECT (conduit),
			 pilot_conduit_standard_abs_signals [PRE_SYNC],
			 dbinfo,
			 &retval);
	return retval;
}

/* Deletes a remote record from the pilot */
static void
standard_abs_delete_from_pilot(GnomePilotConduitStandardAbs *conduit,
			   int handle, 
			   int db, 
			   PilotRecord *remote) 
{
	int err;
	
	g_message("gpilotd: deleting record %ld from pilot",remote->ID);

	err = dlp_DeleteRecord(handle,db,0,remote->ID);
	if(err<=0) {
		g_warning("dlp_DeleteRecord returned %d",err);
	}
}

/* updates or adds a local record to the pilot. Note, that after calling this,
   the record has the dirty bit set.
   FIXME: Remember these id's and ignore during SyncRecord
   Use recordIdsToIgnore, add the recordId to this list, then let sync_record
   check the id of the record to sync as early as possible, and abort if match 
*/
static recordid_t
standard_abs_add_to_pilot (GnomePilotConduitStandardAbs *conduit,
			   int handle, 
			   int db, 
			   LocalRecord *local) 
{
	PilotRecord *remote;
	recordid_t assigned_id;
	int err;

	g_message("gpilotd: adding record to pilot");

	err = gnome_pilot_conduit_standard_abs_transmit (conduit, local, &remote);
	if(err < 0 || remote==NULL) {
	  g_warning(_("Conduit did not return a record"));
	}
	gnome_pilot_conduit_standard_abs_set_status (conduit, local, GnomePilotRecordNothing);

	err = dlp_WriteRecord (handle, db,
			       remote->secret?dlpRecAttrSecret:0,
			       remote->ID,
			       remote->category,
			       remote->record,
			       remote->length,
			       &assigned_id);
	if(err<=0) {
		g_warning("dlp_WriteRecord returned %d",err);
		return 0;
	}

	conduit->recordIdsToIgnore = g_slist_prepend(conduit->recordIdsToIgnore,
						     GINT_TO_POINTER(assigned_id));
	gnome_pilot_conduit_standard_abs_free_transmit (conduit, local, &remote);
	
	return assigned_id;
}

/*
  This is messy, but its my first attempt at implementing the
  algoritm of conduit.pdf page 39-40. And yes, I'll try to collapse
  the methods when I'm sure which are to do the same stuff
  I've marked which of the possibilities from page 39-40 each case
  correlates to. case e1 is where neither local nor remote is modified.
  Basically, given a pilot-side, pc-side or both, using the flags
  it will determine how to synchronize the two sides.
  If either remote or local is NULL, it will retrieve the record.

  Also check http://palm.3com.com/devzone/docs/30wincdk/conduitc.htm#627041

  direction decides which operations may be called
     SyncToRemote - only operations that alter remote (meaning AddToPilot, dlp...)
     SyncToLocal  - only operations that alter local (meaning gnome_pilot_conduit_...)

  Call with SyncToRemote|SyncToLocal to synchronize both ways

  Case 12, 13, 14, 15, 18, 19 and 20 are still missing, this basically
  renders archiving useless.

  eskil */

int
standard_abs_sync_record (GnomePilotConduitStandardAbs *conduit,
			  int handle,
			  int db,
			  LocalRecord *local,
			  PilotRecord *remote,
			  StandardAbsSyncDirection direction)
{
	gboolean free_remote;
	gboolean free_match;

	g_assert(conduit!=NULL);
	g_assert(!(local==NULL && remote==NULL));

	free_remote = FALSE;
	free_match = FALSE;

	if (local==NULL && remote!=NULL) {
		if(g_slist_find(conduit->recordIdsToIgnore,GINT_TO_POINTER(remote->ID))!=NULL) {
			g_message("gpilotd: this record has already been processed");
			return 0;
		}
       		gnome_pilot_conduit_standard_abs_match_record (conduit, &local, remote);
		free_match = TRUE;
	} else if (remote==NULL && local!=NULL) {
		int index;
		if(g_slist_find(conduit->recordIdsToIgnore,GINT_TO_POINTER(local->ID))!=NULL) {
			g_message("gpilotd: this record has already been processed");
			return 0;
		}

		g_message("gpilotd: retrieve %ld from pilot",local->ID);
		remote = g_new0(PilotRecord,1);
		remote->record = malloc(0xffff);
		dlp_ReadRecordById(handle,db,
				   local->ID,
				   remote->record,
				   &index,
				   &remote->length,
				   &remote->attr,
				   &remote->category);
		remote->ID = local->ID;
		standard_abs_compute_attr_field(remote);
		free_remote = TRUE;
	} else if (remote==NULL && local==NULL) {
		g_error("SyncRecord called with two NULL parameters");
		return 0;
	} 
		

	if (local) {
		/* local record exists */
		if(remote->archived) {
#if 0	
			switch(local->attr) {
			case GnomePilotRecordModified:
				if(remove->attr == GnomePilotRecordModified) {
					if (gnome_pilot_conduit_standard_abs_compare(conduit,local,remote) != 0) {
					 	/* CASE 14 */
					} else {
						/* CASE 15 */
					}
				} else {
					/* CASE 13 */
				}
				break;
			case GnomePilotRecordNothing:
				/* CASE 11 No Modify */
				g_message("gpilotd: sync_record: case 11 No Modify");
				remote->attr = GnomePilotRecordNothing;
				remote->archived = 0;
				if ( direction & SyncToLocal )
					gnome_pilot_conduit_standard_abs_archive_remote (conduit, 
											 NULL, 
											 remote);
				break;
			case GnomePilotRecordDeleted:
				/* CASE 12  */
				g_message("gpilotd: sync_record: case 12");
				remote->attr = 0;
				remote->archived = 0;
				if ( direction & SyncToLocal )
					gnome_pilot_conduit_standard_abs_archive_remote (conduit, local, remote);
				break;
			}
			
			switch(remote->attr) {
			case GnomePilotRecordModified:
				break;
			}
#endif
		} else {
			switch (remote->attr) {
			case GnomePilotRecordModified:
				switch(local->attr) {
				case GnomePilotRecordModified:
					if (gnome_pilot_conduit_standard_abs_compare (conduit, local, remote) != 0) {
						/* CASE 10 */
						g_message("gpilotd: sync_record: case 10");
						if ( direction & SyncToRemote ) {
							standard_abs_add_to_pilot (conduit, handle, db, local);
						}
						if ( direction & SyncToLocal ) {
							gnome_pilot_conduit_standard_abs_store_remote (conduit, remote);
						}
						dlp_AddSyncLogEntry (handle, LOG_CASE10);
					} else {
						/* CASE 9 */
						g_message("gpilotd: sync_record: case 9");
						if ( direction & SyncToLocal )
							gnome_pilot_conduit_standard_abs_set_status (conduit, local, GnomePilotRecordNothing);
					}
					break;
				case GnomePilotRecordNothing:
					/* CASE 7*/
					g_message("gpilotd: sync_record: case 7");
					if ( direction & SyncToLocal ) 
						gnome_pilot_conduit_standard_abs_store_remote (conduit, remote);
					break;
				case GnomePilotRecordDeleted:
					/* CASE 6 */
					g_message("gpilotd: sync_record: case 6");
					if ( direction & SyncToLocal )
						gnome_pilot_conduit_standard_abs_store_remote (conduit, remote);
					dlp_AddSyncLogEntry(handle, LOG_CASE6);
					break;
				default:
					g_warning("gpilotd: sync_record: Unhandled sync case (b) Remote.attr = %d, Local.attr = %d\n",
						remote->attr, local->attr);
					break;
				}
				break;
			case GnomePilotRecordDeleted:
				switch (local->attr) {
				case GnomePilotRecordModified:
					/* CASE 5 */
					g_message("gpilotd: sync_record: case 5");
					if ( direction & SyncToRemote ) {
						dlp_AddSyncLogEntry(handle, LOG_CASE5);
						standard_abs_add_to_pilot (conduit, handle, db, local);
					}
					break;
				case GnomePilotRecordDeleted:
#if 0
					/* CASE 17 */ /* THIS IS NOT CASE 17 */
					g_message("gpilotd: sync_record: case 17");
					if ( direction & SyncToLocal )
						gnome_pilot_conduit_standard_abs_archive_local (conduit, local);
					if ( direction & SyncToRemote )
						standard_abs_delete_from_pilot(conduit,handle,db,remote);
#endif
					break;
				case GnomePilotRecordNothing:
					/* CASE 3 */
					g_message("gpilotd: sync_record: case 3");
					/* can be collapsed with case GnomePilotRecordDeleted */
					if ( direction & SyncToRemote )
						standard_abs_delete_from_pilot(conduit,handle,db,remote);
					if ( direction & SyncToLocal )
						gnome_pilot_conduit_standard_abs_set_status (conduit, local, GnomePilotRecordDeleted);
					break;
				default:
					g_warning("gpilotd: sync_record: Unhandled sync case (a) Remote.attr = %d, local.attr = %d\n",
						remote->attr, local->attr);
					break;
				}
				break;
			case GnomePilotRecordNothing:
				if (local->archived) {
					/* CASE 16 */
					g_message("gpilotd: sync_record: case 16");
					if ( direction & SyncToLocal )
						gnome_pilot_conduit_standard_abs_archive_local (conduit, local);
					if ( direction & SyncToRemote )
						standard_abs_delete_from_pilot(conduit,handle,db,remote);
				} else
					switch (local->attr) {
					case GnomePilotRecordDeleted:
						/* CASE 4 */
						g_message("gpilotd: sync_record: case 4, deleting record %ld",remote->ID);
						if ( direction & SyncToRemote )
							standard_abs_delete_from_pilot(conduit,handle,db,remote);
						if ( direction & SyncToLocal )
							gnome_pilot_conduit_standard_abs_set_status (conduit, local, GnomePilotRecordDeleted);
						break;
					case GnomePilotRecordModified: {
						/* CASE 8 */
						g_message("gpilotd: sync_record: case 8");
						if ( direction & SyncToRemote ) 
							standard_abs_add_to_pilot (conduit, handle, db, local);
					}
					break;
					case GnomePilotRecordNothing:
						/* CASE e1 */
						/* g_message("gpilotd: sync_record: case e1"); */
						break;
					default:
						g_warning("gpilotd: sync_record: Unhandled sync case (c) Remote.attr = %d, Local.attr = %d\n",
							remote->attr, local->attr);
						break;
					}
				break;
			}
		}
		if (free_match)
			gnome_pilot_conduit_standard_abs_free_match (conduit, &local);
		if (free_remote) {
			free(remote->record);
			g_free(remote);
		}
	} else {
		/* no local record exists */
		if (remote->archived) {
			/* CASE 11 No Record */
			g_message("gpilotd: sync_record: case 11 No Record");
			remote->attr = GnomePilotRecordNothing;
			remote->archived = 0;
			if ( direction & SyncToLocal )
				gnome_pilot_conduit_standard_abs_archive_remote (conduit, 
										 NULL, 
										 remote);
		} else {
			/* CASE 1 */
			g_message("gpilotd: sync_record: case 1");
                        /* maybe it'd be nice if StoreRemote returned a localRecord ? */
			if ( direction & SyncToLocal ) {
				gnome_pilot_conduit_standard_abs_store_remote (conduit, 
									       remote); 
				gnome_pilot_conduit_standard_abs_match_record (conduit, 
									       &local, 
									       remote);
				if (local) {
					gnome_pilot_conduit_standard_abs_set_status (conduit, 
										     local, 
										     GnomePilotRecordNothing);
					gnome_pilot_conduit_standard_abs_free_match (conduit, &local);
				} else {
					g_warning(_("Error in conduit, newly added record could not be found"));
				}
			}
		}
	}
	/* CASE 2 handled by standard_abs_merge_to_remote */
	return 0;
}

gint 
standard_abs_check_locally_deleted_records(GnomePilotConduitStandardAbs *conduit, 
					   int handle, 
					   int db,
					   StandardAbsSyncDirection direction)
{	
	LocalRecord *local;
	local = NULL;

	g_assert(conduit!=NULL);
	/*
	  Vadim pointed this one out, I'll have to iterate over the locally deleted records
	  as well to delete them from the pilot. This isn't done in the merge calls, since they
	  merge, not synchronize.
	*/
	while(gnome_pilot_conduit_standard_abs_iterate_specific (conduit, &local, GnomePilotRecordDeleted, 0)) {
		g_message("gpilotd: locally deleted record...");
		standard_abs_sync_record(conduit, handle, db, local, NULL, direction);
	}
	return 0;
}


gint
standard_abs_merge_to_remote (GnomePilotConduitStandardAbs *conduit,
			      int handle, 
			      int db, 			 
			      StandardAbsSyncDirection direction) 
{
	LocalRecord *local;
	local = NULL;

	g_assert(conduit!=NULL);

	/* First iterate over new records, install them. 
	   After installing, set attr to Nothing */
	if ( direction & SyncToRemote ) 
		while(gnome_pilot_conduit_standard_abs_iterate_specific (conduit, &local, GnomePilotRecordNew, 0)) {
			recordid_t assigned_id;
			/* FIXME: remote may need to be deleted first */
			assigned_id = standard_abs_add_to_pilot (conduit, handle, db, local);
			gnome_pilot_conduit_standard_abs_set_pilot_id (conduit, 
								       local, 
								       assigned_id);
		}

	/* 
	   then iterate over modified records and sync them, this gets
	   the last records which FastSync and SlowSync didn't sync, since
	   they iterate over pilot-modified records
	*/
	while(gnome_pilot_conduit_standard_abs_iterate_specific (conduit, &local, GnomePilotRecordModified, 0)) {
		standard_abs_sync_record(conduit, handle, db, local, NULL, direction);
	}

	return 0;
}

gint
standard_abs_merge_to_local (GnomePilotConduitStandardAbs *conduit,
			     int handle, 
			     int db, 			     
			     StandardAbsSyncDirection direction) {
        int index;
	int retval;
	PilotRecord remote;
	
	g_assert(conduit!=NULL);

	index = 0;
	retval = 0;

	while (dlp_ReadRecordByIndex(handle,db,
				     index,
				     remote.record,
				     &remote.ID,
				     &remote.length,
				     &remote.attr,
				     &remote.category) >= 0) {
		standard_abs_compute_attr_field(&remote);
		if ( remote.attr == GnomePilotRecordNew ||
		     remote.attr == GnomePilotRecordModified)
			standard_abs_sync_record (conduit, handle, db, NULL, &remote, direction);
		index++;
	}


	return 0;
}

int
SlowSync (int handle, 
	  int db, 
	  GnomePilotConduitStandardAbs *conduit) 
{
        int index;
        int retval;
	PilotRecord remote;
	unsigned char buffer[0xffff];

	g_assert(conduit!=NULL);

	index = 0;
	retval = 0;
	remote.record = buffer;

	g_message("Performing Slow Synchronization");

	while (dlp_ReadRecordByIndex(handle,db,
				     index,
				     remote.record,
				     &remote.ID,
				     &remote.length,
				     &remote.attr,
				     &remote.category) >= 0) {
		standard_abs_compute_attr_field(&remote);
		standard_abs_sync_record (conduit, handle, db, NULL, &remote, SyncBothWays);
		index++;
	}

	return retval;
}

/* Perform a "fast" sync. This requires that both the remote (Pilot) and
   local (PC) have consistent, accurate, and sufficient modification flags.
   If this is not true, a slow sync should be used */
gint
FastSync (int handle, 
	  int db, 
	  GnomePilotConduitStandardAbs *conduit) 
{
        int index;
	int retval;
	unsigned char buffer[0xffff];
	PilotRecord remote;

      	g_assert(conduit!=NULL);

	index = 0;
	retval = 0;
	remote.record = buffer;

	g_message("Performing Fast Synchronization");

	while (dlp_ReadNextModifiedRec (handle, db,
					remote.record,
					&remote.ID,
					&index,
					&remote.length,
					&remote.attr,
					&remote.category) >= 0) {
		standard_abs_compute_attr_field(&remote);
		standard_abs_sync_record (conduit, handle, db, NULL, &remote, SyncBothWays);
	}

	return retval;
}

gboolean
gpilot_sync_pc_match(GnomePilotDBInfo *dbinfo)
{
	GnomePilotSyncStamp *stamp;

	stamp=(GnomePilotSyncStamp *)dbinfo->manager_data;

#if 0
	/* FIXME: these are for debug purposes */
	if(stamp->sync_PC_Id != dbinfo->pu->lastSyncPC) g_message("Cannot Fast Synchronize, SyncPC id failed");
	if(stamp->last_sync_date != dbinfo->pu->successfulSyncDate) g_message("Cannot Fast Synchronize, SyncDate failed");
#endif
/*
	if(stamp->sync_PC_Id == dbinfo->pu->lastSyncPC &&
	   stamp->last_sync_date==dbinfo->pu->successfulSyncDate) {
*/
	if(stamp->sync_PC_Id == dbinfo->pu->lastSyncPC) {
		return TRUE;
	}
	return FALSE;
}

void
standard_abs_open_db(GnomePilotConduitStandardAbs *conduit,
		     GnomePilotDBInfo *dbinfo)
{
	gint cd;
	gchar *name;
  
	g_assert(conduit!=NULL);
	g_assert(dbinfo!=NULL);

	name = g_strdup(gnome_pilot_conduit_standard_get_db_name (GNOME_PILOT_CONDUIT_STANDARD(conduit)));
	cd = 0;
	cd = (int)gtk_object_get_data(GTK_OBJECT(conduit),"db_open_mode");
	if(cd!=0) {
		g_message("gpilotd: open_db: opening with %d\n",cd);
		dlp_OpenDB (dbinfo->pilot_socket, 0, cd, 
			    name, 
			    &(dbinfo->db_handle));
	} else {
		dlp_OpenDB (dbinfo->pilot_socket, 0, dlpOpenReadWrite,
			    name,
			    &(dbinfo->db_handle));
	}

	g_free(name);
}

void
standard_abs_close_db_and_purge_local(GnomePilotConduitStandardAbs *conduit,
				      GnomePilotDBInfo *dbinfo)
{
	g_assert(conduit!=NULL);
	g_assert(dbinfo!=NULL);

	dlp_CleanUpDatabase (dbinfo->pilot_socket, dbinfo->db_handle);
	gnome_pilot_conduit_standard_abs_purge (conduit);

	dlp_ResetSyncFlags (dbinfo->pilot_socket, dbinfo->db_handle);
	dlp_CloseDB (dbinfo->pilot_socket, dbinfo->db_handle);
}
