#define _XOPEN_SOURCE
#include "config.h"
#include <sys/types.h>
#include <pwd.h>
#include <errno.h>
#include <sys/time.h>
#include <glib.h>
#include "manager.h"
#include <libgnome/libgnome.h>
#include <string.h>
#include <gpilot-userinfo.h>
#include <gpilot-structures.h>
#include <unistd.h>
#include <signal.h>
#include <pi-source.h>
#include <pi-dlp.h>
#include <sys/wait.h>
#include <pwd.h>

#define USE_NEW_IO
#include "orbit_daemon_glue.h"
#include <errno.h>

/* Grr, I needed a global variable.  Only used in local gpilotd */
GList *pending_children;
/* Set to true when the config should be reloaded */
gboolean reread_config;

/* FIXME: this is bad. After each sync, I immediately get notified of
   a event on the cradle just synced, this boolean is used to currently
   discard every 2nd sync. Which then means if you run two cradles,
   cradle one syncs and you have to press sync twice on cradle two... */
gboolean ugly_boolean;
#define USE_UGLY 1

/* #define USE_FORK */

gint device_equal_by_io(GPilotDevice *,GIOChannel *);
gboolean device_in(GIOChannel *,
		   GIOCondition ,
		   GPilotContext *);
gboolean device_err(GIOChannel *,
		    GIOCondition ,
		    GPilotContext *);
void monitor_channel(GPilotDevice *,GPilotContext *);
static void remove_pid_file(void);

static void 
pilot_set_baud_rate(GPilotDevice *device) 
{
	static gchar rate_buf[128];
	g_snprintf(rate_buf,128,"PILOTRATE=%d", device->speed);
	g_message("gpilotd: setting %s",rate_buf);
	putenv(rate_buf);
}

static int 
pilot_connect(GPilotDevice *device) 
{
	struct pi_sockaddr addr;
	int sd;
	int ret;
    
	pilot_set_baud_rate(device);

	if (!(sd = pi_socket(PI_AF_SLP, PI_SOCK_STREAM, PI_PF_PADP))) {
		g_warning("pi_socket: %s",strerror(errno));
		return -1;
	}
	
	addr.pi_family = PI_AF_SLP;
	strcpy(addr.pi_device,device->device_name);

	ret = pi_bind(sd, (struct sockaddr*)&addr, sizeof(addr));
	if(ret != 0) {
		g_warning(_("Unable to bind to pilot"));
		g_warning(_("This is a critical error, stopping gpilotd"));
		exit (-1);
	}

	ret = pi_listen(sd,1);
	if(ret != 0) {
		g_warning("pi_listen: %s", strerror(errno));
		return -2;
	}

	sd = pi_accept(sd, NULL,0);
	if(sd == -1) {
		g_warning("pi_accept: %s", strerror(errno));
		return -3;
	}
	return sd;
}

static void pilot_disconnect(int sd)
{
	dlp_EndOfSync(sd, 0);
	pi_close(sd);
}

static void write_sync_stamp(GPilotPilot *pilot,
			     int pfd,
			     struct PilotUser *pu,
			     guint32 last_sync_pc, 
			     time_t t)
{
	gchar prefix[256];

	pu->lastSyncPC=last_sync_pc;
	pu->lastSyncDate=t;

	g_snprintf(prefix,255,"/gnome-pilot.d/gpilotd/Pilot%d/",pilot->number);
	gnome_config_push_prefix(prefix);
	gnome_config_private_set_int("sync_date",t);
	gnome_config_pop_prefix();
	gnome_config_sync();

	dlp_WriteUserInfo(pfd,pu);
}

/** pilot lookup methods **/


/**************************/

/*
 * If there are events for the cradle, this executes them,
 * closes the connection and returns TRUE, otherwise false
 */
static gboolean 
do_cradle_events(int pfd,
		 GPilotContext *context,
		 struct PilotUser *pu,
		 GPilotPilot *pilot,
		 GPilotDevice *device) 
{
  GList *events,*it;

  /* elements in events freed by gpc_request_purge calls
     in orbed_notify_completion */
  events = gpc_queue_load_requests_for_cradle(device->device_name);

  g_message(_("Cradle %s has %d events"),device->device_name,g_list_length(events));

  it = events;

  while(it) {
    GPilotRequest *req;
    req = it->data;
    switch(req->type) {
    case GREQ_SET_USERINFO:
      g_message(_("Setting userinfo..."));
      g_snprintf(pu->username,127,"%s",req->parameters.set_userinfo.user_id);
      pu->userID = req->parameters.set_userinfo.pilot_id;
      dlp_WriteUserInfo(pfd,pu);
      orbed_notify_completion(&req);
      break;
    case GREQ_GET_USERINFO:
      g_message(_("Getting userinfo..."));
      orbed_notify_userinfo(*pu,&req);
      orbed_notify_completion(&req);
      break;
    case GREQ_NEW_USERINFO:
      /* FIXME: this is to set the new and return the old (or something) 
      g_message("getting & setting userinfo");
      g_snprintf(pu->username,127,"%s",req->parameters.set_userinfo.user_id);
      pu->userID = req->parameters.set_userinfo.pilot_id;
      dlp_WriteUserInfo(pfd,pu);
      orbed_notify_completion(&req);
      */
      break;
    default:
      g_warning("%s:%d: *** type = %d",__FILE__,__LINE__,req->type);
      g_assert_not_reached();
      break;
    }

    it = g_list_next(it);
  }

  if(g_list_length(events)>0) {
	  gchar *pilot_name;
	  pilot_name = pilot_name_from_id(pu->userID,context);
	  pilot_disconnect(pfd);
	  orbed_notify_disconnect(pilot_name);
	  g_free(pilot_name);
#ifdef USE_FORK
	  /* _exit(0); */
#else
	  monitor_channel(device,context);
	  return TRUE;
#endif
  }
  return FALSE;
}
/**************************/

static void 
do_sync(int pfd,   
	GPilotContext *context,
	struct PilotUser *pu,
	GPilotPilot *pilot, 
	GPilotDevice *device)
{
  GList *conduit_list, *backup_conduit_list, *file_conduit_list;
  GnomePilotSyncStamp stamp;
  gchar *pilot_name;

  pilot_name = pilot_name_from_id(pu->userID,context);

  gpilot_load_conduits(pilot,
		       &conduit_list, 
		       &backup_conduit_list,
		       &file_conduit_list);
  stamp.sync_PC_Id=context->sync_PC_Id;

  g_message(_("HotSync button pressed, synchronizing pilot"));
  
  /* Set a log entry in the pilot */
  {
	  gchar hostname[64];
	  gpilot_add_log_entry(pfd,"Synchronizing with GnomePilot v.%s\n",VERSION);
	  if (gethostname(hostname,63)==0)
		  gpilot_add_log_entry(pfd,"On host %s\n",hostname);
	  else
		  gpilot_add_log_entry(pfd,"On host %d\n",stamp.sync_PC_Id);
  }

  /* first, run the initial operations, such as single conduit runs,
     restores etc. If this returns True, continue with normal conduit running,
     if False, don't proceed */
  if (gpilot_initial_synchronize_operations(pfd,&stamp,pu,
					    conduit_list,
					    backup_conduit_list,
					    file_conduit_list,
					    context)) {

	  /* FIXME: We need to actually pass a different structure to these
	     functions containing the pu item and the pilot pointer so that
	     we can store the lastSyncPC and assorted fields in
	     ~/.gnome_private/gnome-pilot.d/gpilotd. We will also need some
	     lock mechanism when we start forking for that file :)*/
	  switch(pilot->sync_options.default_sync_action) {
	  case GnomePilotConduitSyncTypeSynchronize:
		  g_message(_("Synchronizing..."));
		  gpilot_synchronize(pfd,&stamp,pu,
				     conduit_list,
				     backup_conduit_list,
				     file_conduit_list,
				     context);
		  break;
	  case GnomePilotConduitSyncTypeCopyToPilot:
		  g_message(_("Copying to pilot..."));
		  gpilot_copy_to_pilot(pfd,&stamp,pu,
				       conduit_list,
				       backup_conduit_list,
				       file_conduit_list,
				       context);
		  break;
	  case GnomePilotConduitSyncTypeCopyFromPilot:
		  g_message(_("Copying from pilot..."));
		  gpilot_copy_from_pilot(pfd,&stamp,pu,
					 conduit_list,
					 backup_conduit_list,
					 context);
		  break;
	  case GnomePilotConduitSyncTypeMergeToPilot:
		  g_message(_("Merging to pilot..."));
		  gpilot_merge_to_pilot(pfd,&stamp,pu,
					conduit_list,
					backup_conduit_list,
					file_conduit_list,
					context);
		  break;
	  case GnomePilotConduitSyncTypeMergeFromPilot:
		  g_message(_("Merging from pilot..."));
		  gpilot_merge_from_pilot(pfd,&stamp,pu,
					  conduit_list,
					  backup_conduit_list,
					  context);
		  break;
	  case GnomePilotConduitSyncTypeNotSet:
	  case GnomePilotConduitSyncTypeCustom:
	  default:
		  g_message(_("Using conduit settings for sync..."));
		  gpilot_sync_default(pfd,&stamp,pu,
				      conduit_list,
				      backup_conduit_list,
				      file_conduit_list,
				      context);
		  break;
	  }
	  g_message(_("Synchronization ended")); 
	  gpilot_add_log_entry(pfd,"Synchronization completed");
  } else {
	  g_message(_("Synchronization ended early"));
	  gpilot_add_log_entry(pfd,"Synchronization terminated");
  }

  write_sync_stamp(pilot,pfd,pu,stamp.sync_PC_Id,time(NULL));
  pilot_disconnect(pfd);
  orbed_notify_disconnect(pilot_name);
  
  g_free(pilot_name);

  gpilot_unload_conduits(conduit_list);
  gpilot_unload_conduits(backup_conduit_list);
  gpilot_unload_conduits(file_conduit_list);
}

typedef struct {
  GPilotDevice *device;
  GList **devices;
  pid_t pending;
} pending_dev;

static void sync_foreach(GPilotDevice *device, GPilotContext *context)
{
  pid_t pid;
  GPilotPilot *pilot;
  int pfd;
  struct PilotUser pu;
  GList *tmp;
  void (*old_handler)(int);

#ifdef USE_FORK
  pid=fork();

  if(pid < 0) {
    g_warning("Unable to fork: %s",strerror(errno));
    return;
  }

  if(pid > 0) {
    pending_dev *p;
    p=(pending_dev *)g_malloc(sizeof(pending_dev));
    p->device=device;
    p->pending=pid;
    context->devices=g_list_remove(context->devices,device);
    p->devices=&(context->devices);

    /* Turn off sig_chld_handler to avoid race condition with GList */
    old_handler=signal(SIGCHLD,SIG_DFL);
    pending_children=g_list_append(pending_children,p);
    old_handler(SIGCHLD);
    signal(SIGCHLD,old_handler);
    return;
  }
#else
  pid = 0;
#endif
 
  /* signal(SIGHUP,SIG_DFL); */
  pfd=pilot_connect(device);

  if(pfd == -1) {
    g_warning(_("An error occured while listening for sync button press!"));
    /* _exit(1); */
  }
  if(pfd == -2) {
    g_warning(_("An error occured while listening for sync button press!"));
    /* _exit(1); */
  }

  if(dlp_ReadUserInfo(pfd,&pu) < 0) {
    g_warning(_("An error occured while getting the pilot's user data"));
    /* _exit(1); */
  }

  if(do_cradle_events(pfd,context,&pu,pilot,device)) return;

  /* validate pilot */
  tmp=g_list_find_custom(context->pilots, (gpointer)&pu.userID,
			 (GCompareFunc)match_pilot_userID);
  if(tmp == NULL) {
    g_warning(_("Unknown pilot, no userID/username match %ld"),pu.userID);
    /* FIXME: here, restoring one of the available pilots should be
       offered to the user. Of course with password prompt if the user
       has password set */
    pilot_disconnect(pfd);
    /* _exit(1); */
  } else {
    pilot=(GPilotPilot *)tmp->data;
    orbed_notify_connect(pilot->name,pu);

    if(pilot->passwd) {
      char *pwd;
      pwd = (char*)g_malloc(pu.passwordLength);
      strncpy(pwd,pu.password,pu.passwordLength);
      if(g_strcasecmp(pilot->passwd,(char*)crypt(pwd,pilot->passwd))) {
	g_warning(_("Unknown pilot, no userID/username/pwd match"));
	pilot_disconnect(pfd);
	orbed_notify_disconnect(pilot->name);
	/* _exit(1); */
      }
      g_free(pwd);
    }
    do_sync(pfd,context,&pu,pilot,device);
  }


#ifdef USE_FORK
  /* _exit(0); */
#else
  monitor_channel(device,context);
#endif
}

static gint match_pending_dev(pending_dev *d,pid_t *pid)
{
  if(d->pending==*pid) return 0;
  return -1;
}

gint device_equal_by_io(GPilotDevice *dev,GIOChannel *io) {
  return !(dev->io==io);
}

gboolean device_in(GIOChannel *io_channel,
		   GIOCondition condition,
		   GPilotContext *context) {
  GPilotDevice *device;
  GList *element;

  /* FIXME: After reengaging the watch on the fd, this is apparantly
   immediately called again. So every other time, I must ignore and retry.*/

#ifdef USE_UGLY
 if(!ugly_boolean) {
    ugly_boolean = TRUE;
    return TRUE;
  }
  ugly_boolean = FALSE;
#else
  g_warning("Beware, testing without ugly_boolean var");
#endif

  element = g_list_find_custom(context->devices,io_channel,(GCompareFunc)device_equal_by_io);

  if(element==NULL) {
    g_warning("cannot find device for active IO channel");
    return FALSE;
  }

  device = element->data;
  g_message(_("Woke on %s"),device->device_name);
  sync_foreach(device,context);
  return FALSE;
}

gboolean device_err(GIOChannel *io_channel,
		    GIOCondition condition,
		    GPilotContext *context) {
  return FALSE;
}

void monitor_channel(GPilotDevice *dev,GPilotContext *context) {
  g_message(_("Watching %s"),dev->device_name);

  g_io_add_watch(dev->io,G_IO_IN,(GIOFunc)device_in,(gpointer)context);
  g_io_add_watch(dev->io,G_IO_ERR,(GIOFunc)device_err,(gpointer)context);
}

static void sig_hup_handler(int dummy)
{
  signal(SIGHUP,sig_hup_handler);
  reread_config=TRUE;
}

static void sig_term_handler(int dummy) {
	g_message(_("Exiting (caught SIGTERM)..."));
	gpilotd_corba_quit();
	remove_pid_file();
	exit(1);
}

static void sig_int_handler(int dummy) {
	g_message(_("Exiting (caught SIGINT)..."));
	gpilotd_corba_quit();
	remove_pid_file();
	exit(1);
}

static void sig_chld_handler(int dummy)
{
  pid_t child;
  pending_dev *p;
  GList *node;
  while((child=waitpid(-1,NULL,WNOHANG)) > 0) {
    node=g_list_find_custom(pending_children,&child,
			    (GCompareFunc)match_pending_dev);
    if(!node) g_warning(_("What, child exited, but wasn't pending???"));
    p=(pending_dev *)node->data;
    pending_children=g_list_remove(pending_children,p);
    (*p->devices)=g_list_append((*p->devices),p->device);

#ifdef USE_UGLY
    ugly_boolean = FALSE;
#endif
    monitor_channel(p->device,orb_context->gpilotd_context);

    g_free(p);
  }

  signal(SIGCHLD,sig_chld_handler);
}

static void kill_foreach(pending_dev *p)
{
  kill(p->pending, SIGTERM);
}

/* Caller must turn off sig_chld_handler before calling this */
static void kill_all_pending_children(GPilotContext *context)
{
  g_list_foreach(pending_children,(GFunc)kill_foreach,NULL);
}

static void wait_for_sync_and_sync(GPilotContext *context) {
  void (*old_handler)(int);

  signal(SIGTERM,sig_term_handler);
  signal(SIGINT,sig_int_handler);

  signal(SIGCHLD,sig_chld_handler);
  signal(SIGHUP,sig_hup_handler);

  g_list_foreach(context->devices,(GFunc)monitor_channel,context);
#ifdef USE_UGLY
  ugly_boolean = TRUE;
#endif

  while(1) {
    if(reread_config) {
      g_message(_("Rereading configuration..."));
      old_handler=signal(SIGCHLD,SIG_DFL);
      kill_all_pending_children(context);
      /*
	FIXME: clean_context(context);
      */
      gpilot_context_init_user (context);
      old_handler(SIGCHLD);
      signal(SIGCHLD,old_handler);
      reread_config=FALSE;
      g_list_foreach(context->devices,(GFunc)monitor_channel,context);
    }
    g_main_iteration(TRUE);
  }
}

static pid_t check_for_pid()
{
  gchar *home_directory;
  gchar *pid_file;
  int fd;
  gchar the_pid[80];
  pid_t retval;
  home_directory=g_get_home_dir();
  if(home_directory) {
    pid_file=(gchar *)g_malloc(strlen(home_directory) + 
			       strlen("/.gpilotd.pid") + 1);
    strcpy(pid_file,home_directory);
    strcat(pid_file,"/.gpilotd.pid");
    if((fd=open(pid_file,O_RDONLY, 0644)) >= 0) {
      read(fd,the_pid,79);
      close(fd);
      retval=atol(the_pid);
    } else {
      retval=0;
    }
    g_free(pid_file);
    return retval;
  } else {
    g_warning(_("Unable to find home directory for uid %d"),geteuid());
    return 0;
  }
}

static void remove_pid_file() {
  gchar *home_directory;
  gchar *pid_file;
  int fd;
  home_directory=g_get_home_dir();
  if(home_directory) {
    pid_file=(gchar *)g_malloc(strlen(home_directory) + 
			       strlen("/.gpilotd.pid") + 1);
    strcpy(pid_file,home_directory);
    strcat(pid_file,"/.gpilotd.pid");
    if((fd=open(pid_file,O_RDONLY, 0644)) >= 0) {
      close(fd);
      unlink(pid_file);
    } 
    g_free(pid_file);
  }
}

static void write_local_pid_file()
{
  gchar *home_directory;
  gchar *pid_file;
  int fd;
  home_directory=g_get_home_dir();
  if(home_directory) {
    pid_file=(gchar *)g_malloc(strlen(home_directory) +
			       strlen("/.gpilotd.pid") + 1);
    strcpy(pid_file,home_directory);
    strcat(pid_file,"/.gpilotd.pid");
    if((fd=open(pid_file,O_RDWR | O_TRUNC | O_CREAT, 0644)) >= 0) {
      gchar pid[50]; /* If the pid is > 50 digits we have problems :) */
      if(sizeof(pid_t) == sizeof(int)) {
	g_snprintf(pid,50,"%d\n",getpid());
      } else { /* Assume pid_t is sizeof long */
	g_snprintf(pid,50,"%ld\n",(unsigned long)getpid());
      }
      write(fd,pid,strlen(pid));
      close(fd);
    } else {
      g_warning(_("Unable to open or create "
		"file %s with read/write privs"),pid_file);
    }

    g_free(pid_file);
  } else {
    g_warning(_("Unable to find home directory for uid %d"),geteuid());
  }
}

int 
main(int argc, char *argv[])
{
	GPilotContext *context;
	pid_t the_pid;
	
	the_pid=check_for_pid();

	if(the_pid) {
		int pid_status;
		/* g_warning("Already running, checking process %d...",the_pid); */
		waitpid(the_pid,&pid_status,WNOHANG);
		if(!WIFEXITED(pid_status)) {
			/* g_warning("%d is not running, starting...",the_pid); */
			remove_pid_file();
		} else {
			g_warning(_("GnomePilot Daemon is already running, process %d, exiting..."),the_pid);
			exit(1);
		}
	}
	
	/*gnome_init(PACKAGE,VERSION,argc,argv); */
	gpilotd_corba_init(&argc,argv,&context);
	
	write_local_pid_file();
	
	reread_config=FALSE;
	
	g_message("%s %s starting...",PACKAGE,VERSION);

	wait_for_sync_and_sync(context);
	
	g_free(context);
	gpilotd_corba_quit();
	
	return 0;
}







