/* Control applet ("capplet") for the gnome-pilot backup conduit,           */
/* based on                                                                 */
/* gpilotd control applet ('capplet') for use with the GNOME control center */


#include <pwd.h>
#include <sys/types.h>
#include <signal.h>
#include <gnome.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>

#include <config.h>
#include <capplet-widget.h>

#include <gpilotd/gpilotd-conduit-mgmt.h>
#include <gpilotd/gpilotd-app.h>
#include <gpilotd/gpilotd-app-dummy-callbacks.h>

#include "backup_conduit.h"

/* tell changes callbacks to ignore changes or not */
static gboolean ignore_changes = FALSE;

/* capplet widget */
static GtkWidget *capplet = NULL;

/* host/device/pilot configuration windows */
GtkWidget *cfgOptionsWindow = NULL;
GtkWidget *cfgStateWindow = NULL;
GtkWidget *dialogWindow = NULL;

gboolean activated,org_activation_state;
GnomePilotConduitMgmt *conduit;

ConduitCfg *origState = NULL;
ConduitCfg *curState = NULL;


static void doTrySettings(GtkWidget *widget, gpointer whatever);
static void doRevertSettings(GtkWidget *widget, gpointer whatever);
static void doSaveSettings(GtkWidget *widget, gpointer whatever);
static void doCancelSettings(GtkWidget *widget, gpointer whatever);

static void readOptionsCfg(GtkWidget *w, ConduitCfg *state);
static void setOptionsCfg(GtkWidget *w, ConduitCfg *state);
static void readStateCfg(GtkWidget *w, ConduitCfg *state,gboolean *);
static void setStateCfg(GtkWidget *w, ConduitCfg *state,gboolean);

gint pilotId;

CORBA_Environment ev;

static void 
setSettings(ConduitCfg* conduitCfg, gboolean active)
{
    if(active)
      gpilotd_conduit_mgmt_enable(conduit,pilotId,GnomePilotConduitSyncTypeCustom);
    else
      gpilotd_conduit_mgmt_disable(conduit,pilotId);

    save_configuration(conduitCfg);
}

static void
doTrySettings(GtkWidget *widget, gpointer whatever)
{
	readStateCfg(cfgStateWindow, curState,&activated);
	readOptionsCfg(cfgOptionsWindow, curState);
	setSettings(curState,activated);
}

static void
doSaveSettings(GtkWidget *widget, gpointer whatever)
{
	doTrySettings(widget, whatever);
}

static void
doCancelSettings(GtkWidget *widget, gpointer whatever)
{
        setSettings(origState,org_activation_state);
}

static void
doRevertSettings(GtkWidget *widget, gpointer whatever)
{
	activated = org_activation_state;
	copy_configuration(curState,origState);
	setOptionsCfg(cfgOptionsWindow, curState);
	setStateCfg(cfgStateWindow, curState,activated);
	setSettings(curState,activated);
}

static void
insert_dir_callback (GtkEditable    *editable, const gchar    *text,
		     gint len, gint *position, void *data)
{
	gint i;
	gchar *curname;

	curname = gtk_entry_get_text(GTK_ENTRY(editable));
	if (*curname == '\0' && len > 0) {
		if (isspace(text[0])) {
			gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
			return;
		}
	} else {
		for (i=0; i<len; i++) {
			if (isspace(text[i])) {
				gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), 
							     "insert_text");
				return;
			}
		}
	}
}
static void
statechange_cb(GtkEditable    *editable, const gchar    *text,
		     gint            length, gint           *position,
		     void *data)
{
	if (!ignore_changes)
		capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}

static void
clist_changed(GtkWidget *widget, gpointer data)
{
	if (!ignore_changes)
		capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}
	
void about_cb (GtkWidget *widget, gpointer data) {
	GtkWidget *about;
	const gchar *authors[] = {_("Eskil Heyn Olsen <deity@eskil.dk>"),NULL};
  
	about = gnome_about_new(_("Gpilotd backup conduit"), VERSION,
				_("(C) 1998 the Free Software Foundation"),
				authors,
				_("Configuration utility for the backup conduit.\n"
				  "The backup conduit is responsible for copying "
				  "files from the pilot to the local host, without "
				  "doing any conversion, whereby ensuring that a "
				  "proper restore can be done"),
				_("gnome-unknown.xpm"));
	gtk_widget_show (about);
  
	return;
}

static void toggled_cb(GtkWidget *widget, gpointer data) 
{
  if(!ignore_changes) {
	gtk_widget_set_sensitive(cfgOptionsWindow,GTK_TOGGLE_BUTTON(widget)->active);
	capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
  }
}

static void button_toggled_cb(GtkWidget *widget, gpointer data)
{
   if(!ignore_changes) {
	capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
   }
}
static GtkWidget
*createStateCfgWindow(void)
{
	GtkWidget *vbox, *table;
	GtkWidget *label;
	GtkWidget *button;

	vbox = gtk_vbox_new(FALSE, GNOME_PAD);

	table = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

	label = gtk_label_new(_("Enabled"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1,2);

	button = gtk_check_button_new();
	gtk_object_set_data(GTK_OBJECT(vbox), "conduit_on_off", button);
	gtk_signal_connect(GTK_OBJECT(button), "toggled",
			   GTK_SIGNAL_FUNC(toggled_cb),
			   NULL);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, 1,2);

	return vbox;
}

static void
setStateCfg(GtkWidget *cfg, ConduitCfg *state, gboolean active)
{
	GtkWidget *button;

	button = gtk_object_get_data(GTK_OBJECT(cfg), "conduit_on_off");

	g_assert(button!=NULL);

	ignore_changes = TRUE;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),active);
	gtk_widget_set_sensitive(cfgOptionsWindow,active);
	ignore_changes = FALSE;
}


static void
readStateCfg(GtkWidget *cfg, ConduitCfg *state, gboolean *active)
{
	GtkWidget *button;

	button  = gtk_object_get_data(GTK_OBJECT(cfg), "conduit_on_off");
  
	g_assert(button!=NULL);

	*active = GTK_TOGGLE_BUTTON(button)->active;
}

/* object data:                        */
/* "userid"   - entry box for pilot user id */
/* "username" - entry box for pilot user name */
static GtkWidget
*createCfgWindow(void)
{
	GtkWidget *vbox, *table;
	GtkWidget *entry, *label;
	GtkWidget *button;

	vbox = gtk_vbox_new(FALSE, GNOME_PAD);

	table = gtk_table_new(2, 3, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

	label = gtk_label_new(_("Backup directory"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
    
	entry = gtk_entry_new_with_max_length(128);
	gtk_object_set_data(GTK_OBJECT(vbox), "dir", entry);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 0, 1);
	gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
			   GTK_SIGNAL_FUNC(insert_dir_callback),
			   NULL);
	gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
				 GTK_SIGNAL_FUNC(statechange_cb),
				 NULL);
	gtk_signal_connect_after(GTK_OBJECT(entry), "delete_text",
				 GTK_SIGNAL_FUNC(statechange_cb),
				 NULL);

	label = gtk_label_new(_("Only backup changed bases"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1,2);

	button = gtk_check_button_new();
	gtk_object_set_data(GTK_OBJECT(vbox), "only_changed", button);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, 1,2);
	gtk_signal_connect(GTK_OBJECT(button), "toggled",
			   GTK_SIGNAL_FUNC(button_toggled_cb),
			   NULL);

	label = gtk_label_new(_("Remove local base if deleted on pilot"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2,3);

	button = gtk_check_button_new();
	gtk_object_set_data(GTK_OBJECT(vbox), "remove_local", button);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, 2,3);
	gtk_signal_connect(GTK_OBJECT(button), "toggled",
			   GTK_SIGNAL_FUNC(button_toggled_cb),
			   NULL);


	return vbox;
}

static void
setOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
	GtkWidget *dir,*updated_only,*remove_deleted;

	dir  = gtk_object_get_data(GTK_OBJECT(pilotcfg), "dir");
	updated_only = gtk_object_get_data(GTK_OBJECT(pilotcfg), "only_changed");
	remove_deleted = gtk_object_get_data(GTK_OBJECT(pilotcfg), "remove_local");

	g_assert(dir!=NULL);
	g_assert(updated_only!=NULL);
	g_assert(remove_deleted!=NULL);

	ignore_changes = TRUE;

	gtk_entry_set_text(GTK_ENTRY(dir), state->backup_dir);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(updated_only), state->updated_only);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(remove_deleted), state->remove_deleted);

	ignore_changes = FALSE;
}


static void
readOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
	GtkWidget *dir,*updated_only,*remove_deleted;

	dir  = gtk_object_get_data(GTK_OBJECT(pilotcfg), "dir");
	updated_only = gtk_object_get_data(GTK_OBJECT(pilotcfg), "only_changed");
	remove_deleted = gtk_object_get_data(GTK_OBJECT(pilotcfg), "remove_local");
  
	state->backup_dir = g_strdup(gtk_entry_get_text(GTK_ENTRY(dir)));
	state->updated_only = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(updated_only));
	state->remove_deleted = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(remove_deleted));
}

static void
pilot_capplet_setup(void)
{
	GtkWidget *frame, *table;

	capplet = capplet_widget_new();

	table = gtk_table_new(1, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), GNOME_PAD);
	gtk_container_add(GTK_CONTAINER(capplet), table); 

	frame = gtk_frame_new(_("Conduit state"));
	gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
	gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 0, 1);
	cfgStateWindow = createStateCfgWindow();
	gtk_container_add(GTK_CONTAINER(frame), cfgStateWindow);

	frame = gtk_frame_new(_("Backup options"));
	gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
	gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 1, 2);
	cfgOptionsWindow = createCfgWindow();
	gtk_container_add(GTK_CONTAINER(frame), cfgOptionsWindow);

	gtk_signal_connect(GTK_OBJECT(capplet), "try",
			   GTK_SIGNAL_FUNC(doTrySettings), NULL);
	gtk_signal_connect(GTK_OBJECT(capplet), "revert",
			   GTK_SIGNAL_FUNC(doRevertSettings), NULL);
	gtk_signal_connect(GTK_OBJECT(capplet), "ok",
			   GTK_SIGNAL_FUNC(doSaveSettings), NULL);
	gtk_signal_connect(GTK_OBJECT(capplet), "cancel",
			   GTK_SIGNAL_FUNC(doCancelSettings), NULL);
	gtk_signal_connect(GTK_OBJECT(capplet), "help",
			   GTK_SIGNAL_FUNC(about_cb), NULL);


	setStateCfg(cfgStateWindow, curState,activated);
	setOptionsCfg(cfgOptionsWindow, curState);

	gtk_widget_show_all(capplet);
}

void run_error_dialog(gchar *mesg,...) {
	char tmp[80];
	va_list ap;

	va_start(ap,mesg);
	vsnprintf(tmp,79,mesg,ap);
	dialogWindow = gnome_message_box_new(mesg,GNOME_MESSAGE_BOX_ERROR,GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
	va_end(ap);
}

gint get_pilot_id_from_gpilotd() {
	gint *pilots;
	int i;
  
	i=0;
	gpilotd_get_pilot_ids(&pilots);
	if(pilots) {
		while(pilots[i]!=-1) { g_message("pilot %d = \"%d\"",i,pilots[i]); i++; }
		if(i==0) {
			run_error_dialog(_("No pilot configured, please choose the\n'Pilot Link Properties' capplet first."));
			return -1;
		} else {
			if(i==1) 
				return pilots[0];
			else {
				g_message("too many pilots...");
				return pilots[0];
			}
		}
	} else {
		run_error_dialog(_("No pilot configured, please choose the\n'Pilot Link Properties' capplet first."));
		return -1;
	}  
}

int
main( int argc, char *argv[] )
{
	/* we're a capplet */
	gnome_capplet_init ("backup conduit control applet", NULL, argc, argv, 
			    NULL,
			    0, NULL);

	/* get pilot name from gpilotd */
	/* 1. initialize the gpilotd connection */
	if (gpilotd_init(&argc,argv)!=0) {
		run_error_dialog(_("Cannot initialze the GnomePilot Daemon"));
		g_error(_("Cannot initialze the GnomePilot Daemon"));
		return -1;
	}

	/* 2 connect to gpilotd */
	if (gpilotd_connect()!=0) {
		run_error_dialog(_("Cannot connect to the GnomePilot Daemon"));
		g_error(_("Cannot connect to the GnomePilot Daemon"));
		return -1;
	}
    
	pilotId = get_pilot_id_from_gpilotd();
	if(pilotId==-1) return -1;

	/* put all code to set things up in here */
	conduit = gpilotd_conduit_mgmt_new("backup_conduit",pilotId);

	load_configuration(&origState,pilotId);
	curState = dupe_configuration(origState);

	org_activation_state = activated = gpilotd_conduit_mgmt_is_enabled(conduit,pilotId);
    
	pilot_capplet_setup();

	/* done setting up, now run main loop */
	capplet_gtk_main();

	destroy_configuration(&origState);
	destroy_configuration(&curState);

	return 0;
}    
