#include <glib.h>
#include <gmodule.h>
#include <pi-socket.h>
#include <pi-sockaddr.h>
#include <pi-source.h>
#include <pi-sync.h>
#include <pi-dlp.h>

#include <libgnome/libgnome.h>

#include "orbit_daemon_glue.h"
#include "manager.h"
#include "queue_io.h"
#include "gnome-pilot-conduit-backup.h"
#include "gnome-pilot-conduit-file.h"
#include "gnome-pilot-conduit-standard.h"
#include "gpilot-gui.h"

#include <libgpilotdCM/gnome-pilot-conduit-management.h>
#include <libgpilotdCM/gnome-pilot-conduit-config.h>

#define PROGRESS_STEPPING 5
#define GPCM_LABEL "conduit_mgmt"
#define GPCC_LABEL "conduit_cfg"

/* File and Backup conduits are handled completely separately from the
   other conduits */
/*
typedef struct GnomePilotConduitManagerInfo {
	gchar *name;
	GModule *dlhandle;
	GnomePilotConduitDestroyFunc destroy_gpilot_conduit;

	gchar *config_name;
	GnomePilotConduitSyncType sync_type;
	GnomePilotConduitSyncType first_sync_type;
	gboolean slow;
} GnomePilotConduitManagerInfo;
*/
/* Typedefs. */
typedef struct {
	GnomePilotConduit *conduit;
	GPilotContext *context;
	struct PilotUser *pu;
	int pilot_socket;
        GList *conduit_list;
        GnomePilotSyncStamp *syncstamp;
} GnomePilotInstallCarrier;

typedef gint (*iterate_func)(GnomePilotConduitStandard *,GnomePilotDBInfo *);

/* Prototypes */
static void unload_foreach (GnomePilotConduit *conduit, gpointer unused);
static gint find_matching_conduit_compare_func (GnomePilotConduit *a,
						GnomePilotDBInfo *b);
static GnomePilotConduit *find_matching_conduit (GList *conlist, GnomePilotDBInfo *dbinfo);
static void backup_foreach (GnomePilotConduitBackup *conduit, GnomePilotDBInfo *info);
static gint iterate_dbs (gint pfd,
			 GnomePilotSyncStamp *stamp,
			 struct PilotUser *pu,
			 GList *conlist,
			 GList *bconlist,
			 iterate_func iter,
			 GPilotContext *context);
static gint conduit_synchronize (GnomePilotConduitStandard *conduit,
				 GnomePilotDBInfo *dbinfo);
static gint conduit_copy_to_pilot (GnomePilotConduitStandard *conduit,
				   GnomePilotDBInfo *dbinfo);
static gint conduit_copy_from_pilot (GnomePilotConduitStandard *conduit,
				     GnomePilotDBInfo *dbinfo);
static gint conduit_sync_default (GnomePilotConduitStandard *conduit,
			     GnomePilotDBInfo *dbinfo);
static void gpilot_manager_save_databases(GnomePilotDBInfo *dbinfo, GSList *databases);
/* This is defined in gpilotd.  But we need to use it... */
gchar *pilot_name_from_id(guint32 id,GPilotContext *context);

/**************************************************************************/
/* These are various match methods, used to do g_list_find's              */

static gint
find_matching_conduit_compare_func (GnomePilotConduit *conduit,
				    GnomePilotDBInfo *dbinfo)
{
	if (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit)) {
		if ((gnome_pilot_conduit_standard_get_creator_id (GNOME_PILOT_CONDUIT_STANDARD (conduit)) ==
		     PI_DBINFO (dbinfo)->creator) &&
		    (strcmp(gnome_pilot_conduit_standard_get_db_name (GNOME_PILOT_CONDUIT_STANDARD (conduit)),
			    PI_DBINFO(dbinfo)->name) == 0)) {
			return 0;
		}
	}
	return -1;
}

static GnomePilotConduit *
find_matching_conduit (GList *conlist,
		       GnomePilotDBInfo *dbinfo)
{
	GList *elem;
	elem = g_list_find_custom(conlist,
				  dbinfo,
				  (GCompareFunc) find_matching_conduit_compare_func);
	if (elem)
		return GNOME_PILOT_CONDUIT (elem->data);
	else
		return NULL;
}

static gint 
find_matching_conduit_compare_func_name(GnomePilotConduit *conduit,
					gchar *name) 
{
  if(GNOME_IS_PILOT_CONDUIT_STANDARD(conduit)) {
    if(strcmp(gnome_pilot_conduit_standard_get_db_name(GNOME_PILOT_CONDUIT_STANDARD(conduit)),
	      name)==0) {
      return 0;
    }
  }
  return -1;
}

static GnomePilotConduit*
find_matching_conduit_name(GList *conlist,gchar *name)
{
  GList *elem;
  elem=g_list_find_custom(conlist,name,
			  (GCompareFunc)find_matching_conduit_compare_func_name);
  if(elem)
    return GNOME_PILOT_CONDUIT(elem->data);
  else
    return NULL;
}

/**************************************************************************/
/* Here comes conduit signal handler, used to call methods from 
   orbit_daemon_glue                                                      */

static void 
conduit_progress_callback(GnomePilotConduit *conduit, 
			  gint total,
			  gint current,
			  gchar *pilot_id) {
	orbed_notify_conduit_progress(pilot_id,
				      conduit,
				      current,
				      total);
}

static void 
conduit_message_callback(GnomePilotConduit *conduit, 
			 gchar *message,
			 gchar *pilot_id) {
	orbed_notify_conduit_message(pilot_id,
				     conduit,
				     message);
}

static void 
conduit_error_callback(GnomePilotConduit *conduit, 
		       gchar *message,
			 gchar *pilot_id) {
	orbed_notify_conduit_error(pilot_id,
				   conduit,
				   message);
}

/* This toggles the callbacks for signal. Call with set=TRUE to 
   register the callbacks, and FALSE to remove them.
   Remember to match these two calls!
   NOTE: Not reentrant! lock when set, unlock when unset.
   could potientially be reentrant since the pilot-name
   would be unique even for multicradle sync
*/
static void 
set_callbacks(gboolean set,GnomePilotConduit *conduit, gchar *pilot_name) 
{
	static guint progress_h = 0,
		     error_h    = 0,
		     message_h  = 0;
	if (set==TRUE) {
		if (progress_h != 0) g_warning("Internal inconsistency, see %s:%d",
					       __FILE__,__LINE__);
		
		progress_h = gtk_signal_connect(GTK_OBJECT(conduit),"progress",
						(GtkSignalFunc)conduit_progress_callback,
						pilot_name);
		
		error_h = gtk_signal_connect(GTK_OBJECT(conduit),"error",
					     (GtkSignalFunc)conduit_error_callback,
					     pilot_name);
		
		message_h = gtk_signal_connect(GTK_OBJECT(conduit),"message",
					       (GtkSignalFunc)conduit_message_callback,
					       pilot_name);
		gnome_pilot_conduit_set_stepping(conduit,PROGRESS_STEPPING);
	} else {
		if (progress_h == 0) g_warning("Internal inconsistency, see %s:%d",
					       __FILE__,__LINE__);
		
		gtk_signal_disconnect(GTK_OBJECT(conduit),progress_h);
		gtk_signal_disconnect(GTK_OBJECT(conduit),error_h);
		gtk_signal_disconnect(GTK_OBJECT(conduit),message_h);
		progress_h = 0;
		error_h    = 0;
		message_h  = 0;
	}
}

/**************************************************************************/

static void
backup_foreach (GnomePilotConduitBackup *conduit, GnomePilotDBInfo *info)
{

	if (GNOME_IS_PILOT_CONDUIT_BACKUP (conduit) == FALSE) {
		g_error(_("non-backup conduit in backup conduit list"));
		return;
	}
	g_assert (info != NULL);
	
	set_callbacks(TRUE,GNOME_PILOT_CONDUIT(conduit),info->pilotInfo->name);
	gnome_pilot_conduit_backup_backup (conduit, info);
	set_callbacks(FALSE,GNOME_PILOT_CONDUIT(conduit),info->pilotInfo->name);
}

static gint
iterate_dbs (gint pfd,
	     GnomePilotSyncStamp *stamp,
	     struct PilotUser *pu,
	     GList *conlist,
	     GList *bconlist,
	     iterate_func iter,
	     GPilotContext *context)
{
	GnomePilotDBInfo dbinfo;
	GnomePilotConduit *conduit = NULL;
	GSList *db_list = NULL;
	int index = 0;
	int error = 0;

	index = 0;
	error = 0;
	dbinfo.pilot_socket = pfd;
	dbinfo.pilotInfo = gpilot_find_pilot_by_id(pu->userID,context->pilots);

	while(1) {
		if(dlp_OpenConduit(pfd) < 0) {
			g_error("Unable to open conduit!");
			error = -1;
			break;
		}
		/* load next dbinfo block */
		if(dlp_ReadDBList(pfd, 0, 0x80, index, PI_DBINFO (&dbinfo)) < 0) {
			/* is <0, there are no more databases, break
                           out so we can save the list */
			break;
		}
		index = PI_DBINFO (&dbinfo)->index + 1;
		db_list = g_slist_append(db_list,g_strdup(PI_DBINFO (&dbinfo)->name));
		/*
		g_message("DB: %20s: Exclude=%3s, Backup=%3s Flags=0x%8.8x Creator=0x%lx",
			   PI_DBINFO(&dbinfo)->name,
			   (PI_DBINFO(&dbinfo)->miscFlags & dlpDBMiscFlagExcludeFromSync)?"yes":"no",
			   (PI_DBINFO(&dbinfo)->flags & dlpDBFlagBackup)?"yes":"no",
			   PI_DBINFO(&dbinfo)->flags,
			   PI_DBINFO(&dbinfo)->creator); 
		*/
		/* check if the base is marked as being excluded from sync */
		if (!(PI_DBINFO(&dbinfo)->miscFlags & dlpDBMiscFlagExcludeFromSync)) {
		        gchar *pilot_name;

			dbinfo.pu = pu;
			dbinfo.pilot_socket = pfd;
			dbinfo.manager_data = (void *) stamp;
			conduit = find_matching_conduit (conlist,&dbinfo);
			if(!conduit) {
				/* FIXME: check if base is marked as to-be-backupped using
				   fields in dbinfo.
				   Or not, it doesn't look like dbinfo->flags are set when the
				   base is changed, nor is modifyDate > backupDate...
				 */
				g_list_foreach (bconlist,
						(GFunc) backup_foreach,
						&dbinfo);
				continue;
			}
			/* Now call the iter method on the conduit, after setting up
			   the signals. The signals aren't set before, since we 
			   want to pass dbinfo as the userdata, and we don't do it
			   in "iter", since it's a parameterized funtion */
      	                pilot_name = pilot_name_from_id(dbinfo.pu->userID,context);
			set_callbacks(TRUE,GNOME_PILOT_CONDUIT(conduit),pilot_name);
			error = iter (GNOME_PILOT_CONDUIT_STANDARD (conduit), &dbinfo);
			set_callbacks(FALSE,GNOME_PILOT_CONDUIT(conduit),pilot_name);
			g_free(pilot_name);
		} else {
			LOG("Base %s is to be ignored by sync",PI_DBINFO(&dbinfo)->name);
			continue;
		}
	}

	gpilot_manager_save_databases(&dbinfo,db_list);
	return error;
}

static gint
conduit_synchronize(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,_("Synchronizing %s\n"),
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeSynchronize);
	err = gnome_pilot_conduit_standard_synchronize (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_copy_to_pilot(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Copy to pilot %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeCopyToPilot);
	err = gnome_pilot_conduit_standard_copy_to_pilot (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_copy_from_pilot(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Copy from pilot %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeCopyFromPilot);
	err = gnome_pilot_conduit_standard_copy_from_pilot (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_merge_to_pilot(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Merge to pilot %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeMergeToPilot);
	err = gnome_pilot_conduit_standard_merge_to_pilot (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_merge_from_pilot(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Merge from pilot %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeMergeFromPilot);
	err = gnome_pilot_conduit_standard_merge_from_pilot (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_sync_default(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	GnomePilotConduitConfig *conduit_config;
	GnomePilotConduitManagement *manager;
	GnomePilotConduitSyncType action;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	g_return_val_if_fail (dbinfo!=NULL, -1);
	g_return_val_if_fail (dbinfo->pu!=NULL, -1);

	conduit_config = gtk_object_get_data (GTK_OBJECT (conduit), GPCC_LABEL);
	manager = gtk_object_get_data (GTK_OBJECT (conduit), GPCM_LABEL);
	g_return_val_if_fail(conduit_config!=NULL,-1);
	if (conduit_config->first_sync_type != GnomePilotConduitSyncTypeNotSet) {
		g_message(_("Conduit %s's first synchronization..."),
			  gnome_pilot_conduit_management_get_name(manager));
		gnome_pilot_conduit_config_remove_first_sync(conduit_config);
		action = conduit_config->first_sync_type;
		if(conduit_config->first_slow==TRUE) {
			gnome_pilot_conduit_standard_set_slow(conduit);
		}
	} else {
		action = conduit_config->sync_type;
	}
		
	switch (action) {
	case GnomePilotConduitSyncTypeSynchronize:
		return conduit_synchronize (conduit, dbinfo);
	case GnomePilotConduitSyncTypeCopyToPilot:
		return conduit_copy_to_pilot (conduit, dbinfo);
	case GnomePilotConduitSyncTypeCopyFromPilot:
		return conduit_copy_from_pilot (conduit, dbinfo);
	case GnomePilotConduitSyncTypeMergeToPilot:
		return conduit_merge_to_pilot (conduit, dbinfo);
	case GnomePilotConduitSyncTypeMergeFromPilot:
		return conduit_merge_from_pilot (conduit, dbinfo);
	default:
		g_warning (_("unknown syncing action (%d = %s).\n"),action,
			   gnome_pilot_conduit_sync_type_int_to_str(action));
		return -1;
	}
}

static void
install_db_foreach (GPilotRequest *req, GnomePilotInstallCarrier *carrier) {
	gchar *pilot_name;

	pilot_name = pilot_name_from_id(carrier->pu->userID,carrier->context);
	set_callbacks(TRUE,carrier->conduit,pilot_name);
	switch(req->type) {
	case GREQ_INSTALL:
		gnome_pilot_conduit_file_install_db (GNOME_PILOT_CONDUIT_FILE (carrier->conduit),
						     carrier->pilot_socket,
						     req->parameters.install.filename);
		break;
	case GREQ_RESTORE:
		gnome_pilot_conduit_file_restore_directory (GNOME_PILOT_CONDUIT_FILE (carrier->conduit),
							    carrier->pilot_socket,
							    req->parameters.restore.directory);
		break;
	default:
		g_warning (_("%s this request.type == %d is not a install/restore"), __PRETTY_FUNCTION__, req->type);
		break;
	}
	set_callbacks(FALSE,carrier->conduit,pilot_name);
	g_free(pilot_name);
	orbed_notify_completion (&req);
};

static void
install_foreach (GnomePilotConduit *conduit,
		 GnomePilotInstallCarrier *info)
{
	GList *request_list;

	g_return_if_fail (conduit != NULL);
	g_return_if_fail (info != NULL);
	g_return_if_fail (GNOME_IS_PILOT_CONDUIT_FILE (conduit));

	info->conduit = conduit;
	request_list = gpc_queue_load_requests(info->pu->userID, GREQ_INSTALL,FALSE);
	g_message ("Pilot has %d entries in file install queue", g_list_length (request_list));
	g_list_foreach (request_list, (GFunc)install_db_foreach, info);
	g_list_free (request_list);
}

static void
send_sysinfo_foreach(GPilotRequest *req, GnomePilotInstallCarrier *carrier)
{
	struct CardInfo card;
	char *pilot_name;

	pilot_name = pilot_name_from_id (carrier->pu->userID, carrier->context);
	dlp_ReadStorageInfo (carrier->pilot_socket,0,&card);
	orbed_notify_sysinfo (pilot_name,
			      card, 
			      &req);
	orbed_notify_completion (&req);
	g_free(pilot_name);
}

static void
do_restore_foreach(GPilotRequest *req, GnomePilotInstallCarrier *carrier)
{
	GList *iterator;
	gchar *pilot_name;
	
	g_assert(carrier!=NULL);

	pilot_name = pilot_name_from_id(carrier->pu->userID,carrier->context);
	/* FIXME: if carrier.conduit_list is NULL, the user hasn't
	  enabled any file conduits. The corba restore call should
	  throw an exception or something.  */
	iterator = carrier->conduit_list;
	while( iterator != NULL ) {
		GnomePilotConduitFile *conduit;
		conduit = GNOME_PILOT_CONDUIT_FILE(iterator->data);
		set_callbacks(TRUE,GNOME_PILOT_CONDUIT(conduit),pilot_name);
		gnome_pilot_conduit_file_restore_directory(conduit,carrier->pilot_socket,
							   req->parameters.restore.directory);
		iterator = iterator->next;
		set_callbacks(FALSE,GNOME_PILOT_CONDUIT(conduit),pilot_name);
	}

	g_free(pilot_name);
	orbed_notify_completion (&req);
}

static void
unload_foreach(GnomePilotConduit *conduit,
	       gpointer unused)
{
	GnomePilotConduitManagement *manager;
	g_return_if_fail (conduit != NULL);
	g_return_if_fail (GNOME_IS_PILOT_CONDUIT (conduit));
	
	manager = gtk_object_get_data (GTK_OBJECT (conduit), GPCM_LABEL);
	g_assert(manager!=NULL);
	/* FIXME: _destroy_conduit doesn't g_module_close... is this an issue ? 
	 note: the shlib loader in gnome_pilot_conduit_management now closes
	after loading the methods. */
	gnome_pilot_conduit_management_destroy_conduit(manager,&conduit);
	gnome_pilot_conduit_management_destroy(manager);
}


void
gpilot_load_conduits (GPilotPilot *pilot,
		      GList **clist,
		      GList **blist,
		      GList **flist)
{
	int i, cnt;
	gchar *prefix;
	GnomePilotConduit *conduit;
	guint32 pilotId;
	gchar **conduit_name;
	GnomePilotConduitManagement *manager;
	GnomePilotConduitConfig *conduit_config;

	pilotId = pilot->pilot_id;

	*clist = NULL;
	*blist = NULL;
	*flist = NULL;

	/* Read the conduit configuration */
	prefix = g_strdup_printf ("gnome-pilot.d/conduits%d/General/", pilot->pilot_id);
	gnome_config_push_prefix(prefix);
	gnome_config_get_vector("conduits",&cnt,&conduit_name);
	gnome_config_pop_prefix();
	g_free(prefix);
	g_message(_("Instantiating %d conduits..."),cnt);
	for(i = 0;i<cnt;i++) {
		gint err;
		manager = gnome_pilot_conduit_management_new(conduit_name[i],GNOME_PILOT_CONDUIT_MGMT_ID);
		if (manager == NULL) {
			g_message(_("Uknown conduit \"%s\" in configure!"),conduit_name[i]);			
			gpilot_gui_warning_dialog(_("Uknown conduit \"%s\" in configure!"),conduit_name[i]); 
			continue;
		}
		conduit_config = gnome_pilot_conduit_config_new(manager,pilot->pilot_id);
		gnome_pilot_conduit_config_load_config(conduit_config);
		err = gnome_pilot_conduit_management_instantiate_conduit(manager,pilotId,&conduit);
		if (err != GNOME_PILOT_CONDUIT_MGMT_OK || conduit == NULL) {
			g_message(_("Loading conduit \"%s\" failed!"),conduit_name[i]);
			gpilot_gui_warning_dialog(_("Loading conduit \"%s\" failed!\n"),conduit_name[i]);
			gnome_pilot_conduit_config_destroy(conduit_config);
			gnome_pilot_conduit_management_destroy(manager);
			continue;
		}

		gtk_object_set_data (GTK_OBJECT (conduit), GPCM_LABEL, manager);
		gtk_object_set_data (GTK_OBJECT (conduit), GPCC_LABEL, conduit_config);

		if (GNOME_IS_PILOT_CONDUIT_BACKUP (conduit)) {
			*blist = g_list_append (*blist, conduit);
		} else if (GNOME_IS_PILOT_CONDUIT_FILE (conduit)) {
			*flist = g_list_append (*flist, conduit);
		} else if (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit)) {
			*clist = g_list_append (*clist, conduit);
		} else {
			g_warning("Error in conduits definition file for pilot %d, %s is not a valid conduit",
				  pilot->number,
				  conduit_name[i]);
		}

		g_free(conduit_name[i]);
	}
	g_free(conduit_name);
}

void
gpilot_unload_conduits(GList *list)
{
	g_list_foreach (list, (GFunc) unload_foreach, NULL);
	g_list_free(list);
}

static void 
run_conduit_sync_foreach(GPilotRequest *req,
			 GnomePilotInstallCarrier *carrier)
{
	GnomePilotConduit *conduit;
	GList *clist,*blist,*flist;
	int err;

	g_return_if_fail(conduit!=NULL);
	g_return_if_fail(carrier!=NULL);

	err = 0;
	clist = NULL;
	blist = NULL;
	flist = NULL;
	conduit = NULL;

	conduit = find_matching_conduit_name(carrier->conduit_list,
					     req->parameters.conduit.name);
	if(conduit==NULL) {
		/* FIXME: remove the bloody request, or the user will get this
		   message every time */
		g_error(_("Conduit %s requested, but not found"),
			req->parameters.conduit.name);
	} else {
		if(GNOME_IS_PILOT_CONDUIT_BACKUP(conduit)) {
			blist = g_list_append(blist,conduit);
		} else if(GNOME_IS_PILOT_CONDUIT_STANDARD(conduit)) {
			clist = g_list_append(clist,conduit);
		} else if(GNOME_IS_PILOT_CONDUIT_FILE(conduit)) {
			flist = g_list_append(flist,conduit);
		} else {
			g_error(_("Conduit %s cannot be requested"),
				req->parameters.conduit.name);
		}

		if(flist) {
			g_list_foreach(flist,(GFunc)install_foreach,&carrier);
		} else {
	    
			err= iterate_dbs(carrier->pilot_socket,
					 carrier->syncstamp,
					 carrier->pu,
					 clist,
					 blist,
					 conduit_sync_default,
					 carrier->context); 
		}
	}
	orbed_notify_completion(&req);
}

gboolean gpilot_initial_synchronize_operations(int pilot_socket,
					       GnomePilotSyncStamp *stamp,
					       struct PilotUser *pu,
					       GList *conduit_list,
					       GList *backup_conduit_list,
					       GList *file_conduit_list,
					       GPilotContext *context)
{
	GList *request_list;
	gboolean do_normal_sync_stuff;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;

	/* If this variable is true, run all the normal conduit stuff in
	   conduit_list, backup_conduit_list and file_conduit_list in the end.
	   If false (getsysinfo, restore and/or single conduit run), dont
	   use the normal conduits */
	do_normal_sync_stuff = TRUE;

	/* elements in request_list freed by gpc_request_purge calls
           in send_sysinfo_foreach */
	request_list = gpc_queue_load_requests(pu->userID, GREQ_GET_SYSINFO,FALSE);
	g_message("Pilot has %d entries in get_sysinfo queue",g_list_length(request_list));
	if (g_list_length (request_list)) {
		g_list_foreach(request_list,(GFunc)send_sysinfo_foreach,&carrier);
	}

	/* elements in request_list freed by gpc_request_purge calls
           in do_restore_foreach */
	request_list = gpc_queue_load_requests(pu->userID, GREQ_RESTORE,FALSE);
	g_message("Pilot has %d entries in restore queue",g_list_length(request_list));
	if (g_list_length (request_list)) {
		do_normal_sync_stuff = FALSE;
		carrier.conduit_list = g_list_copy(file_conduit_list);
		g_list_foreach(request_list,(GFunc)do_restore_foreach,&carrier);
	}

	/* elements in request_list freed by gpc_request_purge calls
           in run_conduit_sync_foreach */
	request_list = gpc_queue_load_requests(pu->userID, GREQ_CONDUIT,FALSE);
	g_message("Pilot has %d entries in conduit queue",g_list_length(request_list));
	if (g_list_length (request_list)) {
		carrier.conduit_list = g_list_copy(conduit_list);
		carrier.conduit_list = g_list_concat(carrier.conduit_list,backup_conduit_list);
		carrier.conduit_list = g_list_concat(carrier.conduit_list,file_conduit_list);
		do_normal_sync_stuff = FALSE;
		g_list_foreach(request_list,(GFunc)run_conduit_sync_foreach,&carrier);
		g_list_free(carrier.conduit_list);
	}

	return do_normal_sync_stuff;
}

gint
gpilot_synchronize(int pilot_socket,
		   GnomePilotSyncStamp *stamp,
		   struct PilotUser *pu,
		   GList *conduit_list,
		   GList *backup_conduit_list,
		   GList *file_conduit_list,
		   GPilotContext *context)
{
	gint error;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;

	/* now run through all the file install requests */
	g_list_foreach(file_conduit_list,(GFunc)install_foreach,&carrier);
	/* and then iterate over all conduits */
	error = iterate_dbs(pilot_socket,
			    stamp,
			    pu,
			    conduit_list,
			    backup_conduit_list,
			    conduit_synchronize,
			    context);
	/* FIXME: empty queue for pilot here */

	return error;
}

gint 
gpilot_copy_to_pilot(int pilot_socket, 
		     GnomePilotSyncStamp *stamp,
		     struct PilotUser *pu,
		     GList *conduit_list,
		     GList *backup_conduit_list,
		     GList *file_conduit_list,
		     GPilotContext *context)
{
	gint error;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;

	g_list_foreach(file_conduit_list,(GFunc)install_foreach,&carrier);

	error = iterate_dbs(pilot_socket,stamp,pu,
			    conduit_list,
			    backup_conduit_list,
			    conduit_copy_to_pilot,
			    context);
	return error;
}

gint 
gpilot_copy_from_pilot(int pilot_socket, 
		       GnomePilotSyncStamp *stamp,
		       struct PilotUser *pu,
		       GList *conduit_list,
		       GList *backup_conduit_list,
		       GPilotContext *context)
{
	gint error;
	error = iterate_dbs(pilot_socket,stamp,pu,
			    conduit_list,
			    backup_conduit_list,
			    conduit_copy_from_pilot,
			    context);
	return error;
}

gint 
gpilot_merge_to_pilot(int pilot_socket, 
		      GnomePilotSyncStamp *stamp,
		      struct PilotUser *pu,
		      GList *conduit_list,
		      GList *backup_conduit_list, 
		      GList *file_conduit_list,
		      GPilotContext *context)
{
	gint error;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;

	g_list_foreach(file_conduit_list,(GFunc)install_foreach,&carrier);

	error = iterate_dbs(pilot_socket,stamp,pu,
			    conduit_list,
			    backup_conduit_list,
			    conduit_merge_to_pilot,
			    context);

	return error;
}

gint 
gpilot_merge_from_pilot(int pilot_socket, 
			GnomePilotSyncStamp *stamp,
			struct PilotUser *pu,
			GList *conduit_list,
			GList *backup_conduit_list,
			GPilotContext *context)
{
	gint error;
	error = iterate_dbs(pilot_socket,stamp,pu,conduit_list,backup_conduit_list,
			  conduit_merge_from_pilot,context);
	return error;
}

gint 
gpilot_sync_default(int pilot_socket, GnomePilotSyncStamp *stamp,
		    struct PilotUser *pu,
		    GList *conduit_list,
		    GList *backup_conduit_list,
		    GList *file_conduit_list,
		    GPilotContext *context)
{
	gint error;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;
	g_list_foreach(file_conduit_list,(GFunc)install_foreach,&carrier);

	error=iterate_dbs(pilot_socket,stamp,pu,conduit_list,backup_conduit_list,
			  conduit_sync_default,context);
	return error;
}

void 
gpilot_add_log_entry(int pilot_socket,
		     gchar *entry, ...) 
{
	gchar *e,*f;

	va_list ap;

	va_start(ap,entry);
	e = g_strdup_vprintf(entry,ap);
	f = g_strdup_printf("%s ",e);
	dlp_AddSyncLogEntry(pilot_socket,f);
	g_free (e);
	g_free (f);
}

/**
   save the list of databases this pilot has
 */
void 
gpilot_manager_save_databases(GnomePilotDBInfo *dbinfo, 
			      GSList *databases) {
	gchar *pfx;
	char **charlist;
	int cnt=0;
	GSList *ptr;

	charlist = g_new0(char*,g_slist_length(databases)+1);

	for (ptr = databases; ptr != NULL; ptr = ptr->next) {
		charlist[cnt] = ptr->data;
		cnt++;
	}

	pfx = g_strdup_printf("/gnome-pilot.d/PilotCache%d/Databases/",
			      dbinfo->pilotInfo->pilot_id);
	gnome_config_push_prefix(pfx);
	gnome_config_set_vector("databases",
				g_slist_length(databases),
				(const char**)charlist);
	gnome_config_pop_prefix();
	gnome_config_sync();
	/* FIXME: is this okay ? */
	g_slist_foreach(databases,(GFunc)g_free,NULL);
	g_free(pfx);
	g_free(charlist);
	g_slist_free(databases);
}
